#-------------------------------------------------------------------------------
#  
#  Defines the OMLink class of the Enable 'om' (Object Model) package. 
#
#  The OMLink class defines the base class for visual connections between 
#  two contacts (i.e. OMContact objects) contained within components (i.e.
#  OMComponent objects) belonging to a canvas (i.e. OMCanvas object).
#  A link represents a relationship between two elements of the underlying 
#  object model.
#  
#  Written by: David C. Morrill
#  
#  Date: 01/27/2005
#  
#  (c) Copyright 2005 by Enthought, Inc.
#  
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:  
#-------------------------------------------------------------------------------

from enthought.util.scipyx import array, transpose, putmask, amin, sqrt, add, isnan,\
                             PINF
                             
from om_traits        import StyleDelegate, ERGBAColor

from enthought.enable import Component

from enthought.traits.api import HasStrictTraits, Instance, Str, Property, true

#-------------------------------------------------------------------------------
#  'OMLinkStyle' class:  
#-------------------------------------------------------------------------------

class OMLinkStyle ( HasStrictTraits ):
    
    #---------------------------------------------------------------------------
    #  Trait definitions:  
    #---------------------------------------------------------------------------
   
    # Color of the link:
    color        = ERGBAColor( 'black' )
    
    # Show tooltip information?
    show_tooltip = true
    
    # Show status information?
    show_status  = true
    
# Create a default contact style:
default_link_style = OMLinkStyle()

#-------------------------------------------------------------------------------
#  'OMLink' class:
#-------------------------------------------------------------------------------

class OMLink ( Component ):
    
    #---------------------------------------------------------------------------
    #  Trait definitions:  
    #---------------------------------------------------------------------------
 
    # The position and size of the link:
    bounds       = Property
    
    # The style to use for drawing the contact:
    style        = Instance( OMLinkStyle, default_link_style )
   
    # Color of the link:
    color        = StyleDelegate
    
    # Show tooltip information?
    show_tooltip = StyleDelegate
    
    # Show status information?
    show_status  = StyleDelegate
    
    # Tooltip text:
    tooltip      = Str
    
    # Status text:
    status       = Str
    
    # Contacts the link is connected to:
    contact1     = Instance( 'enthought.enable.om.OMContact' )
    contact2     = Instance( 'enthought.enable.om.OMContact' )
    
#-- Property Implementations ---------------------------------------------------

    #---------------------------------------------------------------------------
    #  'bounds' property:  
    #---------------------------------------------------------------------------
    
    def _get_bounds ( self ):
        c1 = self.contact1
        c2 = self.contact2
        if (c1 is None) or (c2 is None):
            return ( 0, 0, 0, 0 )
        c1x, c1y = self._contact_center( c1 )
        c2x, c2y = self._contact_center( c2 )
        return ( float( min( c1x, c2x ) ) - 1.0, 
                 float( min( c1y, c2y ) ) - 1.0, 
                 float( abs( c1x - c2x ) ) + 2.0,
                 float( abs( c1y - c2y ) ) + 2.0 )
                 
    def _set_bounds ( self, bounds ):
        pass
        
#-- Event Handlers -------------------------------------------------------------

    #---------------------------------------------------------------------------
    #  Handles the 'contact1' or 'contact2' trait being changed:  
    #---------------------------------------------------------------------------

    def _contact1_changed ( self ):
        self.contact1.links.append( self )

    def _contact2_changed ( self ):
        self.contact2.links.append( self )

#-- Component overrides --------------------------------------------------------

    #---------------------------------------------------------------------------
    #  Draws the contact:  
    #---------------------------------------------------------------------------
                
    def _draw ( self, gc ):
        gc.save_state()
        
        x1, y1 = self._contact_center( self.contact1 )
        x2, y2 = self._contact_center( self.contact2 )
            
        gc.set_stroke_color( self.color_ )
        gc.set_line_width( 1 )
        gc.begin_path()
        gc.move_to( x1, y1 )
        gc.line_to( x2, y2 )
        gc.stroke_path()
        
        gc.restore_state()
        
    #---------------------------------------------------------------------------
    #  Returns the components at a specified (x,y) point:
    #---------------------------------------------------------------------------
       
    def _components_at ( self, x, y ):
        line = array( ( self._contact_center( self.contact1 ),
                        self._contact_center( self.contact2 ) ) )
        if self._distance_to( line, x, y ) <= 5.0:
            return [ self ]
        return []
        
#-- Private Helper Methods -----------------------------------------------------

    #---------------------------------------------------------------------------
    #  Returns the 'center' of a specified contact:  
    #---------------------------------------------------------------------------
    
    def _contact_center ( self, contact ):
        x, y, dx, dy = contact.bounds
        cx, cy       = contact.center
        return ( x + cx, y + cy )

    #---------------------------------------------------------------------------
    #  Calculates the distance from a given point (x,y) to the closest point
    #  on the line:
    #---------------------------------------------------------------------------
                
    def _distance_to ( self, xy, x, y ):
        """
        Calculates the distance from a given point (x,y) to the closest point
        on the line.
        """
        dpt  = xy - array( [ x, y ] )
        d10  = dpt[:-1]
        d20  = dpt[1:]
        d21  = xy[1:] - xy[:-1]
        t    = -(add.reduce( d21 * d10, 1 ) / add.reduce( d21 * d21, 1 ))
        dpt2 = add.reduce( dpt * dpt, 1 )
        temp = transpose( array( [ t, t ] ) ) * d21 + d10
        dt   = add.reduce( temp * temp, 1 )
        putmask( dt, t < 0.0, dpt2[:-1] )
        putmask( dt, t > 1.0, dpt2[1:] )
        try:
            result = sqrt( amin( dt ) )
        except:
            # Probably tried to take sqrt( +INF ):
            return PINF

        # If the result is 'NaN', map it to positive infinity:
        if isnan( result ):
            result = PINF

        return result

