#-------------------------------------------------------------------------------
#
#  Define the base Enable 'frame' component, from which all other Enable frame
#  based components are derived. A 'frame' is a component that acts as a 'frame'
#  or 'wrapper' around another component, giving it increased functionality
#  and/or improved appearance.
#
#  Note: The Frame class is not abstract, but provides only limited
#  functionality.
#
#  Written by: David C. Morrill
#
#  Date: 10/09/2003
#
#  (c) Copyright 2003 by Enthought, Inc.
#
#  Classes defined: Frame
#
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:
#-------------------------------------------------------------------------------

from enthought.traits.api import Trait
from enthought.traits.ui.api import Group, View, Include
from component        import Component
from controls         import Label
from base_container   import BaseContainer, default_container

#-------------------------------------------------------------------------------
#  'Frame' class:
#-------------------------------------------------------------------------------

class Frame ( Component, BaseContainer ):

    #---------------------------------------------------------------------------
    #  Trait definitions:
    #---------------------------------------------------------------------------

    component = Trait( None, Component )

    #---------------------------------------------------------------------------
    #  Trait view definitions:
    #---------------------------------------------------------------------------

    links = Group( '<links>', 'component' )

    #---------------------------------------------------------------------------
    #  Initialize the object:
    #---------------------------------------------------------------------------

    def __init__ ( self, component = None, **traits ):
        Component.__init__( self, **traits )
        if component is not None:
            if isinstance(component, basestring):
                component = Label( component )
            self.component = component

    #---------------------------------------------------------------------------
    #  Add a component to the container:
    #---------------------------------------------------------------------------

    def add ( self, *components ):
        if len( components ) > 0:
            self.component = components[-1]

    #---------------------------------------------------------------------------
    #  Remove a component from the container:
    #---------------------------------------------------------------------------

    def remove ( self, *components ):
        for component in components:
            if component is self.component:
                component.container = default_container
                self.component      = Component()
                break

    #---------------------------------------------------------------------------
    #  Handle the component being changed:
    #---------------------------------------------------------------------------

    def _component_changed ( self, old_component, new_component ):
        if old_component is not None:
            old_component.on_trait_change( self._component_bounds_modified,
                                          'bounds', remove = True )
            old_component.on_trait_change( self._component_min_size_modified,
                                          'min_width', remove = True )
            old_component.on_trait_change( self._component_min_size_modified,
                                          'min_height', remove = True )
        if new_component is None:
            self.component = Component()
            return
        new_component_container = new_component.container
        new_component_container.remove( new_component )
        new_component.container = self
        if ((new_component_container is default_container) or
            (self.container is not default_container)):
            self._bounds_modified( self.bounds )
        else:
            self.container = new_component_container
            self._component_bounds_modified( new_component.bounds )
        self._component_min_size_modified()
        new_component.on_trait_change( self._component_bounds_modified,
                                       'bounds' )
        new_component.on_trait_change( self._component_min_size_modified,
                                      'min_width' )
        new_component.on_trait_change( self._component_min_size_modified,
                                      'min_height' )

    #---------------------------------------------------------------------------
    #  Handle the bounds of the component being changed:
    #---------------------------------------------------------------------------

    def _bounds_changed ( self, old, new ):
        Component._bounds_changed( self, old, new )
        self._bounds_modified( new )

    #---------------------------------------------------------------------------
    #  Generate any additional components that contain a specified (x,y) point:
    #---------------------------------------------------------------------------

    def _components_at ( self, x, y ):
        return [ self ] + self.component.components_at( x, y )

    #---------------------------------------------------------------------------
    #  Draw the component in a specified graphics context:
    #---------------------------------------------------------------------------

    def _draw ( self, gc ):
        self._pre_draw( gc )
        self.component.draw( gc )
        self._post_draw( gc )

    #---------------------------------------------------------------------------
    #  Methods that 'should' be overridden by subclasses:
    #---------------------------------------------------------------------------

    #---------------------------------------------------------------------------
    #  Do any drawing that needs to be done before drawing the contained
    #  component:
    #---------------------------------------------------------------------------

    def _pre_draw ( self, gc ):
        pass

    #---------------------------------------------------------------------------
    #  Do any drawing that needs to be done after drawing the contained
    #  component:
    #---------------------------------------------------------------------------

    def _post_draw ( self, gc ):
        pass

    #---------------------------------------------------------------------------
    #  Handle the bounds of the component being modified:
    #---------------------------------------------------------------------------

    def _bounds_modified ( self, bounds ):
        if self.component is not None:
            self.component.bounds = bounds

    #---------------------------------------------------------------------------
    #  Handle the bounds of the component we are bound to being modified:
    #---------------------------------------------------------------------------

    def _component_bounds_modified ( self, bounds ):
        self.bounds = bounds

    #---------------------------------------------------------------------------
    #  Handle the minimum size of the component we are bound to being modified:
    #---------------------------------------------------------------------------

    def _component_min_size_modified ( self ):
        self.min_width  = self.component.min_width
        self.min_height = self.component.min_height

