#-----------------------------------------------------------------------------
#
#  Copyright (c) 2005, Enthought, Inc.
#  All rights reserved.
#
#  Author: Scott Swarts <swarts@enthought.com>
#
#-----------------------------------------------------------------------------

"""A dragged line segment
"""

# Enthought library imports
from enthought.enable import Line
from enthought.enable.enable_traits import Pointer
from enthought.traits.api import Event

##############################################################################
# class 'DragSegment'
##############################################################################

class DragSegment(Line):
    """A dragged line segment
    """

    ###########################################################################
    # Trait definitions.
    ###########################################################################

    # Override the vertex color so as to not draw it.
    vertex_color = (0.0, 0.0, 0.0, 0.0)

    # Event fired when the line is complete
    complete = Event

    #### Pointers. ####

    # Pointer for the complete state.
    complete_pointer = Pointer('cross')

    # Pointer for the drawing state.
    drawing_pointer = Pointer('cross')

    # Pointer for the normal state.
    normal_pointer = Pointer('cross')

    ###########################################################################
    # 'Component' interface.
    ###########################################################################

    #### 'complete' state #####################################################

    def complete_draw(self, gc):
        """ Draw the completed line. """

        self.line_dash = None

        self._draw(gc)
        self.redraw()

        return

    #### 'drawing' state ######################################################

    def drawing_draw(self, gc):

        self.line_dash = (4.0, 2.0)
        
        self._draw(gc)
        return

    def drawing_mouse_move(self, event):
        """ Handle the mouse moving in drawing state. """
        # Change the last point to the current event point
        self.points[-1] = (event.x, event.y)
        self.updated = self
        self.redraw()
        return

    def drawing_left_up(self, event):
        """ Handle the left mouse button coming up in the 'drawing' state. """

        self.event_state = 'complete'
        self.pointer = self.complete_pointer

        self.redraw()

        self.complete = True

        return

    #### 'normal' state #######################################################

    def normal_left_down(self, event):
        """ Handle the left button down in the 'normal' state. """

        # Set points the current segment, which is just the
        # current point twice.
        current_point = (event.x, event.y)
        self.points = [current_point, current_point]
        self.updated = self

        # Go into the drawing state
        self.event_state = 'drawing'
        self.pointer = self.drawing_pointer

        self.redraw()
        return

    def normal_mouse_move(self, event):

        self.pointer = self.normal_pointer

        return

    


#### EOF ######################################################################
