#-------------------------------------------------------------------------------
#
#  Define the base Enable 'component', from which all other Enable based
#  components are derived.
#
#  Note: The EnableComponent class is not abstract, but provides only limited
#  functionality.
#
#  Written by: David C. Morrill
#
#  Date: 09/22/2003
#
#  (c) Copyright 2003 by Enthought, Inc.
#
#  Classes defined: Component
#
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Imports:
#-------------------------------------------------------------------------------

import logging

from base             import xy_in_bounds, subclasses_of, gc_image_for, \
                             intersect_bounds, gc_temp
from base_container   import default_container, BaseContainer
from enable_traits    import simple_delegate, bounds_trait, \
                             ComponentMinSize, ComponentMaxSize, \
                             cursor_style_trait, red_color_trait
from enable_traits    import TimeInterval, Stretch
from events           import MouseHandler, drag_event_trait, key_event_trait, \
                             simple_event_trait
from enthought.traits.api import HasTraits, Trait, Property, Any, Disallow, Str, \
                             Bool
from enthought.traits.ui.api import Group, View


#-------------------------------------------------------------------------------
#  Module data:
#-------------------------------------------------------------------------------

enable_logger = None

#-------------------------------------------------------------------------------
#  'Component' class:
#-------------------------------------------------------------------------------

class Component ( MouseHandler ):

    #---------------------------------------------------------------------------
    #  Trait definitions:
    #---------------------------------------------------------------------------

    container      = Trait( default_container, BaseContainer,
                            desc  = 'the object containing this object' )
    bounds         = Trait( bounds_trait,
                            desc  = 'the location and size of the component' )
    cursor_style   = Trait( cursor_style_trait,
                            desc = 'the style of cursor drawn over the '
                                   'component' )
    cursor_color   = Trait( red_color_trait,
                            desc = 'the color of the cursor drawn over '
                                   'the component' )
    min_height     = ComponentMinSize( desc = 'the minimum height of the '
                                               'component (in pixels)' )
    min_width      = ComponentMinSize( desc = 'the minimum width of the '
                                              'component (in pixels)' )
    max_height     = ComponentMaxSize( desc = 'the maximum height of the '
                                              'component (in pixels)' )
    max_width      = ComponentMaxSize( desc = 'the maximum width of the '
                                              'component (in pixels)' )
    stretch_width  = Stretch( desc  = 'the relative horizontal stretchiness '
                                      'of the component' )
    stretch_height = Stretch( desc  = 'the relative vertical stretchiness '
                                      'of the component' )
    timer_interval = TimeInterval( desc  = 'the interval between timer events '
                                           '(in seconds)' )

    tooltip        = Str( desc = 'the text to display when the pointer hovers '
                                 'over the component' )

    window         = simple_delegate
    visible        = simple_delegate
    pointer        = simple_delegate

    accepts_focus  = Bool( False,
                           desc = 'whether the component accepts keyboard '
                                  'focus' )

    event_state    = Trait( 'normal',
                            desc = 'the current component event state' )

    # Events:
    pre_dropped_on = drag_event_trait
    dragged        = drag_event_trait
    dropped        = drag_event_trait
    drag_enter     = drag_event_trait
    drag_over      = drag_event_trait
    drag_leave     = drag_event_trait
    dropped_on     = drag_event_trait
    resized        = simple_event_trait
    key            = key_event_trait
    has_focus      = simple_event_trait
    timer          = simple_event_trait

    # Set up generic, default trait definitions:
    __             = Any
    _              = Disallow

    #---------------------------------------------------------------------------
    #  Traits view definition:
    #---------------------------------------------------------------------------
    # The default view for the component base class.
    # Subclasses should use Include(id='component_view') to include this.

    traits_view = View(
        Group( 'x', 'y', 'width', 'height', '_',
               'min_width', 'min_height',
               'stretch_width', 'stretch_height', '_',
               'visible',
               id = 'component' ),
        Group( 'container',
               id    = 'links',
               style = 'custom' )
    )

    colorchip_map = {}

    #---------------------------------------------------------------------------
    #  Initialize the object:
    #---------------------------------------------------------------------------

    def __init__ ( self, **traits ):
        self._drawable = True
        HasTraits.__init__( self, **traits )

    #---------------------------------------------------------------------------
    #  Property definitions:
    #---------------------------------------------------------------------------

    def __x_get ( self ):
        return self.bounds[0]

    def __x_set ( self, x ):
        ignore, y, width, height = self.bounds
        self.bounds = ( x, y, width, height )

    def __y_get ( self ):
        return self.bounds[1]

    def __y_set ( self, y ):
        x, ignore, width, height = self.bounds
        self.bounds = ( x, y, width, height )

    def __width_get ( self ):
        return self.bounds[2]

    def __width_set ( self, width ):
        x, y, ignore, height = self.bounds
        self.bounds = ( x, y, width, height )

    def __height_get ( self ):
        return self.bounds[3]

    def __height_set ( self, height ):
        x, y, width, ignore = self.bounds
        self.bounds = ( x, y, width, height )

    def __left_get ( self ):
        return self.bounds[0]

    def __bottom_get ( self ):
        return self.bounds[1]

    def __right_get ( self ):
        return self.bounds[0] + self.bounds[2]

    def __top_get ( self ):
        return self.bounds[1] + self.bounds[3]

    # Define 'x, y, width, height, left, right, bottom, top' properties:
    x      = Property( __x_get,      __x_set )
    y      = Property( __y_get,      __y_set )
    width  = Property( __width_get,  __width_set )
    height = Property( __height_get, __height_set )
    left   = Property( __left_get )
    right  = Property( __right_get )
    bottom = Property( __bottom_get )
    top    = Property( __top_get )

    #---------------------------------------------------------------------------
    #  Get/Set the location of the component:
    #---------------------------------------------------------------------------

    def location ( self, x = None, y = None ):
        if x is not None:
            if y is not None:
                self.bounds = ( x, y ) + self.bounds[2:]
            else:
                self.bounds = x + self.bounds[2:]
        else:
            return self.bounds[:2]

    #---------------------------------------------------------------------------
    #  Get/Set the dimensions of the component:
    #---------------------------------------------------------------------------

    def dimensions ( self, width = None, height = None ):
        if width is not None:
            if height is not None:
                self.bounds = self.bounds[:2] + ( width, height )
            else:
                self.bounds = self.bounds[:2] + width
        else:
            return self.bounds[2:]

    #---------------------------------------------------------------------------
    #  Handle the bounds of the component being changed:
    #---------------------------------------------------------------------------

    def _bounds_changed ( self, old, new ):
        self._invalidate_visible_bounds()
        self.redraw( old )
        self.redraw( new )

    #---------------------------------------------------------------------------
    #  Handle the minimum or maximum size changing:
    #---------------------------------------------------------------------------

    def _min_width_changed ( self, min_width ):
        if min_width > self.width:
            self.width = min_width

    def _min_height_changed ( self, min_height ):
        if min_height > self.height:
            self.height = min_height

    def _max_width_changed ( self, max_width ):
        if max_width < self.width:
            self.width = max_width

    def _max_height_changed ( self, max_height ):
        if max_height < self.height:
            self.height = max_height

    #---------------------------------------------------------------------------
    #  Handle the container of the component being changed:
    #---------------------------------------------------------------------------

    def _container_changed ( self, old, new ):
        self._invalidate_visible_bounds()
## FIXME : This seems like it would work, but sometimes the trait notifier
##         fails, so I'm removing it until I can discuss with Dave.
## Traceback (most recent call last):
##  File "C:\projects\proava2\branches\converge\src\lib\enthought\traits\trait_notifiers.py", line 206, in rebind_call_0
##    getattr( self.object(), self.name )()
##TypeError: 'NoneType' object is not callable
##
##         if new is not None:
##             new.on_trait_change( self._invalidate_visible_bounds, 'bounds')

##         if old is not None:
##             old.on_trait_change( self._invalidate_visible_bounds, 'bounds',
##                                  remove = True )
        self.redraw()

    #---------------------------------------------------------------------------
    #  Handle the visibility being changed:
    #---------------------------------------------------------------------------

    def _visible_changed ( self, old, new ):
        self.redraw()

    #---------------------------------------------------------------------------
    #  Handle the pointer shape being changed:
    #---------------------------------------------------------------------------

    def _pointer_changed ( self, pointer ):
        self.window._set_pointer( pointer )

    #---------------------------------------------------------------------------
    #  Handle the time interval being changed:
    #---------------------------------------------------------------------------

    def _timer_interval_changed ( self, interval ):
        self.window._set_timer_interval( self, interval )

    #---------------------------------------------------------------------------
    #  Handle the cursor style being changed:
    #---------------------------------------------------------------------------

    def _cursor_style_changed ( self ):
        self.window._on_mouse_move( None )

    #---------------------------------------------------------------------------
    #  Handle the cursor color being changed:
    #---------------------------------------------------------------------------

    def _cursor_color_changed ( self ):
        self.window._on_mouse_move( None )

    #---------------------------------------------------------------------------
    #  Default mouse event handlers:
    #---------------------------------------------------------------------------

    def _left_down_changed     ( self, event ):
        self._send_mouse_event( event, '_left_down' )

    def _left_up_changed       ( self, event ):
        self._send_mouse_event( event, '_left_up' )

    def _left_dclick_changed   ( self, event ):
        self._send_mouse_event( event, '_left_dclick' )

    def _right_down_changed    ( self, event ):
        self._send_mouse_event( event, '_right_down' )

    def _right_up_changed      ( self, event ):
        self._send_mouse_event( event, '_right_up' )

    def _right_dclick_changed  ( self, event ):
        self._send_mouse_event( event, '_right_dclick' )

    def _middle_down_changed   ( self, event ):
        self._send_mouse_event( event, '_middle_down' )

    def _middle_up_changed     ( self, event ):
        self._send_mouse_event( event, '_middle_up' )

    def _middle_dclick_changed ( self, event ):
        self._send_mouse_event( event, '_middle_dclick' )

    def _mouse_move_changed    ( self, event ):
        self._send_mouse_event( event, '_mouse_move' )

    def _mouse_wheel_changed   ( self, event ):
        self._send_mouse_event( event, '_mouse_wheel' )

    def _window_enter_changed  ( self, event ):
        self._send_mouse_event( event, '_window_enter' )

    def _window_leave_changed  ( self, event ):
        self._send_mouse_event( event, '_window_leave' )

    #---------------------------------------------------------------------------
    #  Handle other components being dropped on/dragged over the component:
    #---------------------------------------------------------------------------

    def _dropped_on_changed ( self, event ):
        self._send_drag_event( event, 'dropped_on_by_' )

    def _drag_enter_changed ( self, event ):
        self._send_drag_event( event, 'drag_enter_by_' )

    def _drag_over_changed ( self, event ):
        self._send_drag_event( event, 'drag_over_by_' )

    def _drag_leave_changed ( self, event ):
        self._send_drag_event( event, 'drag_leave_by_' )

    #---------------------------------------------------------------------------
    #  Handle being dropped on by a component that implements the
    #  IDroppedOnHandler interface:
    #---------------------------------------------------------------------------

    def dropped_on_by_idroppedonhandler ( self, component, event ):
        component.was_dropped_on( self, event )

    #---------------------------------------------------------------------------
    #  Handle drag events from a ColorChip:
    #---------------------------------------------------------------------------

    def drag_enter_by_colorchip ( self, colorchip, event ):
        event.handled = True
        item = self.colorchip_map.get( colorchip.item )
        if item is not None:
            self._save_color = getattr( self, item )
            setattr( self, item, colorchip.color )
            self.redraw()

    def drag_leave_by_colorchip ( self, colorchip, event ):
        event.handled = True
        if self._save_color is not None:
            setattr( self, self.colorchip_map[ colorchip.item ],
                     self._save_color )
            self.redraw()
            del self._save_color

    def dropped_on_by_colorchip ( self, colorchip, event ):
        self.drag_enter_by_colorchip( colorchip, event )
        del self._save_color

    #---------------------------------------------------------------------------
    #  Request that the component be redrawn:
    #---------------------------------------------------------------------------

    def redraw ( self, bounds = None ):
        if bounds is None:
            self._redraw_pending = True
            if self.container._redraw_pending:
                return
        self.window.redraw( self._visible_bounds( bounds ) )

    #---------------------------------------------------------------------------
    #  Determine if the component needs to be redrawn:
    #---------------------------------------------------------------------------

    def needs_redraw ( self ):
        return self.window._needs_redraw( self.bounds )

    #---------------------------------------------------------------------------
    #  Draw the component in a specified graphics context if it needs it:
    #---------------------------------------------------------------------------

    def draw ( self, gc ):
        try:
            if (self.visible and self._drawable and
                self.window._needs_redraw( self.bounds )):
                handler = getattr( self, self.event_state + '_draw', None )
                if handler is not None:
                    handler( gc )
                else:
                    self._draw( gc )
        except:
            global enable_logger
            if enable_logger is None:
                enable_logger = logging.getLogger( 'enthought.enable' )
            enable_logger.exception( 'Exception occurred in Enable component '
                                     'draw handler' )
        self._redraw_pending = False

    #---------------------------------------------------------------------------
    #  Return an image specified by name:
    #---------------------------------------------------------------------------

    def image_for ( self, image ):
        path   = ''
        prefix = image[:1]
        if prefix == '=':
            path  = self
            image = image[1:]
        elif prefix == '.':
            path  = None
            image = image[1:]
        return gc_image_for( image, path )

    #---------------------------------------------------------------------------
    #  Methods that may be overridden by subclasses:
    #---------------------------------------------------------------------------

    #---------------------------------------------------------------------------
    #  Return the components that contain a specified (x,y) point:
    #---------------------------------------------------------------------------

    def components_at ( self, x, y ):
        if (self.visible and self._drawable and
            xy_in_bounds( x, y, self.bounds )):
            return self._components_at( x, y )
        return []

    #---------------------------------------------------------------------------
    #  Determine whether a specified point is in the component:
    #---------------------------------------------------------------------------

    def xy_in_bounds ( self, x, y = None ):
        if y is None:
            # Handle the case where: ( x = mouse_event, y = None ):
            return xy_in_bounds( x.x, x.y, self.bounds )
        return xy_in_bounds( x, y, self.bounds )

    #---------------------------------------------------------------------------
    #  Methods that 'should' be overridden by subclasses:
    #---------------------------------------------------------------------------

    #---------------------------------------------------------------------------
    #  Draw the component in a specified graphics context:
    #---------------------------------------------------------------------------

    def _draw ( self, gc ):
        gc.save_state()
        gc.set_fill_color( ( 0.0, 0.0, 0.0, 1.0 ) )
        gc.begin_path()
        gc.rect( *self.bounds )
        gc.fill_path()
        gc.restore_state()

    #---------------------------------------------------------------------------
    #  Return the components that contain a specified (x,y) point:
    #---------------------------------------------------------------------------

    def _components_at ( self, x, y ):
        return [ self ]

    #---------------------------------------------------------------------------
    #  Compute the minimum size that the component can render itself in:
    #---------------------------------------------------------------------------

    def _min_size ( self, gc ):
        return ( self.min_width, self.min_height )

    #---------------------------------------------------------------------------
    #  Returns the visible bounds of the component:
    #---------------------------------------------------------------------------

    def _visible_bounds ( self, bounds = None ):
        if bounds is not None:
            container_bounds = self.container._visible_bounds()
            vb = intersect_bounds( bounds, container_bounds )

        else:
            # If possible, use the previously cached version of visible bounds.
            # The cached bounds are dependent on our own bounds and the visible
            # bounds of the container.  By reduction, our visible bounds are
            # dependent on the real bounds of the container.  Ergo, we must
            # invalidate the cache when our bounds change or the container's
            # bounds change.
            vb = self.__visible_bounds
            if vb is None:
                container_bounds = self.container._visible_bounds()
                self.__visible_bounds = vb = intersect_bounds( self.bounds,
                                                            container_bounds )

        return vb

    #---------------------------------------------------------------------------
    #  Clears the cached visible bounds:
    #---------------------------------------------------------------------------

    def _invalidate_visible_bounds( self ):
        self.__visible_bounds = None

    #---------------------------------------------------------------------------
    #  Returns the bounds that the component's cursor should be clipped against:
    #---------------------------------------------------------------------------

    def _cursor_bounds ( self ):
        return self.bounds

    #---------------------------------------------------------------------------
    #  Return a temporary Kiva graphics context used for non_drawing operations:
    #---------------------------------------------------------------------------

    def gc_temp ( self ):
        return gc_temp

    #---------------------------------------------------------------------------
    #  Private methods:
    #---------------------------------------------------------------------------

    #---------------------------------------------------------------------------
    #  Send a drag event based on the class name of the components dragged:
    #---------------------------------------------------------------------------

    def _send_drag_event ( self, event, prefix ):
        for component in event.components:
            for subclass in subclasses_of( component.__class__ ):
                method = getattr( self, prefix + subclass.__name__.lower(),
                                  None )
                if method is not None:
                    method( component, event )
                    break

    #---------------------------------------------------------------------------
    #  Send a mouse event based on the current event state of the component:
    #---------------------------------------------------------------------------

    def _send_mouse_event ( self, event, suffix ):
        handler       = getattr( self, self.event_state + suffix, None )
        if handler is not None:
            event.handled = True
            handler( event )

