#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <glib.h>
#include "edv_context.h"
#include "edv_id.h"
#include "edv_id_get.h"


/* UID */
GList *EDVUIDsList(edv_context_struct *ctx);
edv_uid_struct *EDVUIDMatchByUID(
	edv_context_struct *ctx,
	const gint user_id      
);
edv_uid_struct *EDVUIDMatchByGID(
	edv_context_struct *ctx,
	const gint group_id
);
gint EDVUIDNameToUID(
	edv_context_struct *ctx,
	const gchar *user_name  
);
gchar *EDVUIDGetNameFromUID(
	edv_context_struct *ctx,
	const gint user_id
);
gchar *EDVUIDGetNameFromGID(
	edv_context_struct *ctx,   
	const gint group_id
);

/* GID */
GList *EDVGIDsList(edv_context_struct *ctx);
gint EDVGIDNameToGID(
	edv_context_struct *ctx,
	const gchar *group_name
);
gchar *EDVGIDGetNameFromGID(     
	edv_context_struct *ctx,
	const gint group_id
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the UIDs list.
 *
 *	Returns a GList of edv_uid_struct * which must not be modified
 *	or deleted.
 */
GList *EDVUIDsList(edv_context_struct *ctx)
{
	if(ctx == NULL)
	    return(NULL);

	return(ctx->uids_list);
}

/*
 *	Gets the UID by user ID.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The user_id specifies the user ID.
 *
 *	Returns the UID or NULL on error.
 */
edv_uid_struct *EDVUIDMatchByUID(
	edv_context_struct *ctx,
	const gint user_id
)
{
	GList *glist;
	edv_uid_struct *uid;

	if(ctx == NULL)
	    return(NULL);

	for(glist = ctx->uids_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    uid = EDV_UID(glist->data);
	    if(uid == NULL)
		continue;

	    if(uid->user_id == user_id)
		return(uid);
	}

	return(NULL);
}

/*
 *	Gets the UID by group ID.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The group_id specifies the group ID.
 *
 *	Returns the UID or NULL on error.
 */
extern edv_uid_struct *EDVUIDMatchByGID(
	edv_context_struct *ctx,
	const gint group_id
)
{
	GList *glist;
	edv_uid_struct *uid;

	if(ctx == NULL)
	    return(NULL);

	for(glist = ctx->uids_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    uid = EDV_UID(glist->data);
	    if(uid == NULL)
		continue;

	    if(uid->group_id == group_id)
		return(uid);
	}

	return(NULL);
}

/*
 *	Gets the user ID from the user name.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The user_name specifies the string describing the user name.
 *
 *	Returns the user ID or negative on error.
 */
gint EDVUIDNameToUID(
	edv_context_struct *ctx,
	const gchar *user_name
)
{
	if((ctx == NULL) || STRISEMPTY(user_name))
	    return(-2);

	/* Match by number? */
	if(isdigit(*user_name))
	{
	    return(ATOI(user_name));
	}
	else
	{
	    GList *glist;
	    edv_uid_struct *uid;

	    for(glist = ctx->uids_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		uid = EDV_UID(glist->data);
		if(uid == NULL)
		    continue;

		if(uid->name == NULL)
		    continue;

		if(!strcmp((const char *)uid->name, (const char *)user_name))
		    return(uid->user_id);
	    }
	}

	return(-1);
}

/*
 *	Gets the user name or a string describing the user ID from the
 *	user ID.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The user_id specifies the user ID.
 *
 *	Returns a dynamically allocated string describing the user
 *	name or user ID or NULL on error.
 */
gchar *EDVUIDGetNameFromUID(
	edv_context_struct *ctx,
	const gint user_id
)
{
	edv_uid_struct *uid = EDVUIDMatchByUID(ctx, user_id);
	if((uid != NULL) ? (uid->name != NULL) : FALSE)
	{
	    return(STRDUP(uid->name));
	}
	else if(user_id < 0)
	{
	    return(STRDUP(""));
	}
	else
	{
	    return(g_strdup_printf(
		"%i",
		user_id
	    ));
	}
}

/*
 *	Gets the user name or a string describing the user ID from the
 *	group ID.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The group_id specifies the group ID.
 *
 *	Returns a dynamically allocated string describing the user
 *	name or user ID or NULL on error.
 */
gchar *EDVUIDGetNameFromGID(
	edv_context_struct *ctx,
	const gint group_id
)
{
	edv_uid_struct *uid = EDVUIDMatchByGID(ctx, group_id);
	if((uid != NULL) ? (uid->name != NULL) : FALSE)
	{
	    return(STRDUP(uid->name));
	}
	else if(group_id < 0)
	{
	    return(STRDUP(""));
	}
	else
	{
	    return(g_strdup_printf(
		"%i",
		group_id
	    ));
	}
}


/*
 *	Gets the GIDs list.
 *
 *	Returns a GList of edv_gid_struct * which must not be modified
 *	or deleted.
 */
GList *EDVGIDsList(edv_context_struct *ctx)
{
	if(ctx == NULL)
	    return(NULL);

	return(ctx->gids_list);
}

/*
 *	Gets the GID by group ID.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The group_id specifies the group ID.
 *
 *	Returns the GID or NULL on error.
 */
edv_gid_struct *EDVGIDMatchByGID(
	edv_context_struct *ctx,
	const gint group_id
)
{
	GList *glist;
	edv_gid_struct *gid;

	if(ctx == NULL)
	    return(NULL);

	for(glist = ctx->gids_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    gid = EDV_GID(glist->data);
	    if(gid == NULL)
		continue;

	    if(gid->group_id == group_id)
		return(gid);
	}

	return(NULL);
}

/*
 *	Gets the group ID from the group name.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The group_name specifies the string describing the group name.
 *
 *	Returns the group ID or negative on error.
 */
gint EDVGIDNameToGID(
	edv_context_struct *ctx,
	const gchar *group_name
)
{
	if((ctx == NULL) || STRISEMPTY(group_name))
	    return(-2);

	/* Match by number? */
	if(isdigit(*group_name))
	{
	    return(ATOI(group_name));
	}
	else
	{
	    GList *glist;
	    edv_gid_struct *gid;

	    for(glist = ctx->gids_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		gid = EDV_GID(glist->data);
		if(gid == NULL)
		    continue;

		if(gid->name == NULL)
		    continue;

		if(!strcmp((const char *)gid->name, (const char *)group_name))
		    return(gid->group_id);
	    }
	}

	return(-1);
}

/*
 *	Gets the group name or a string describing the group ID from
 *	the group ID.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The group_id specifies the group ID.
 *
 *	Returns a dynamically allocated string describing the
 *	group name or group ID or NULL on error.
 */
gchar *EDVGIDGetNameFromGID(
	edv_context_struct *ctx,
	const gint group_id
)
{
	edv_gid_struct *gid = EDVGIDMatchByGID(ctx, group_id);
	if((gid != NULL) ? (gid->name != NULL) : FALSE)
	{
	    return(STRDUP(gid->name));
	}
	else if(group_id < 0)
	{
	    return(STRDUP(""));
	}
	else
	{
	    return(g_strdup_printf(
		"%i",
		group_id
	    ));
	}
}
