#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/stat.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "url.h"
#include "edv_types.h"
#include "edv_obj.h"
#include "edv_recycled_obj.h"
#include "edv_archive_obj.h"
#include "edv_obj_op.h"
#include "edv_confirm.h"
#include "edv_recbin_stat.h"
#include "edv_recbin_recover.h"
#include "edv_archive_stat.h"
#include "edv_archive_extract.h"
#include "archive_options_dlg.h"
#include "imbr.h"
#include "imbr_dnd.h"
#include "endeavour2.h"
#include "edv_op.h"
#include "edv_cb.h"
#include "edv_utils.h"

#include "edv_utils_gtk.h"
#include "config.h"


static void EDVImbrDNDUpdateStatusBar(
	edv_imbr_struct *imbr,
	const gint gdk_action, const guint info,
	const gint total_src_objects,
	const gint total_objects_processed, const gint status
);

static void EDVImbrDragDataReceivedNexus(
	edv_core_struct *core, edv_imbr_struct *imbr,
	GdkDragContext *dc, guint info, GtkSelectionData *selection_data,
	edv_object_struct *obj			/* Can be NULL */
);

gint EDVImbrLocBarIconCrossingCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
);

void EDVImbrTListDNDSetIcon(
	edv_imbr_struct *imbr, gint thumb_num
);
void EDVImbrTListDragDataGetCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
void EDVImbrTListDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
void EDVImbrTListDragDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
);

void EDVImbrImgViewDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Updates the Image Browser's Status Bar to indicate the results
 *	of the DND operation.
 */
static void EDVImbrDNDUpdateStatusBar(
	edv_imbr_struct *imbr,
	const gint gdk_action, const guint info,
	const gint total_src_objects,
	const gint total_objects_processed, const gint status
)
{
	gchar *buf = NULL;
	edv_status_bar_struct *sb = (imbr != NULL) ?
	    imbr->status_bar : NULL;
	if(sb == NULL)
	    return;

	/* Format the status bar message */
	if((info == EDV_DND_INFO_TEXT_PLAIN) ||
	   (info == EDV_DND_INFO_TEXT_URI_LIST) ||
	   (info == EDV_DND_INFO_STRING)
	)
	{
	    switch(gdk_action)
	    {
	      case GDK_ACTION_COPY:
		if(status == -4)
		    buf = STRDUP(
			"Copy operation canceled"
		    );
		else if(total_objects_processed > 0)
		    buf = g_strdup_printf(
			"Coppied %i %s",
			total_objects_processed,
			(total_objects_processed == 1) ? "object" : "objects"
		    );
		else
		    buf = g_strdup_printf(
			"Unable to copy %s",
			(total_src_objects == 1) ? "object" : "objects"
		    );
		break;
	      case GDK_ACTION_MOVE:
		if(status == -4)
		    buf = STRDUP(
			"Move operation canceled"
		    );
		else if(total_objects_processed > 0)
		    buf = g_strdup_printf(
			"Moved %i %s",
			total_objects_processed,
			(total_objects_processed == 1) ? "object" : "objects"
		    );
		else
		    buf = g_strdup_printf(
			"Unable to move %s",
			(total_src_objects == 1) ? "object" : "objects"
		    );
		break;
	      case GDK_ACTION_LINK:
		if(status == -4)
		    buf = STRDUP(
			"Link operation canceled"
		    );
		else if(total_objects_processed > 0)
		    buf = g_strdup_printf(
			"Linked %i %s",
			total_objects_processed,
			(total_objects_processed == 1) ? "object" : "objects"
		    );
		else
		    buf = g_strdup_printf(
			"Unable to link %s",
			(total_src_objects == 1) ? "object" : "objects"
		    );
		break;
	    }
	}
	else if(info == EDV_DND_INFO_RECYCLED_OBJECT)
	{
	    switch(gdk_action)
	    {
	      case GDK_ACTION_COPY:
	      case GDK_ACTION_MOVE:
	      case GDK_ACTION_LINK:
		if(status == -4)
		    buf = STRDUP(
			"Recover operation canceled"
		    );
		else if(total_objects_processed > 0)
		    buf = g_strdup_printf(
			"Recovered %i %s",
			total_objects_processed,
			(total_objects_processed == 1) ? "object" : "objects"
		    );
		else
		    buf = g_strdup_printf(
			"Unable to recover %s",
			(total_src_objects == 1) ? "object" : "objects"
		    );
		break;
	    }
	}
	else if(info == EDV_DND_INFO_ARCHIVE_OBJECT)
	{
	    switch(gdk_action)
	    {
	      case GDK_ACTION_COPY:
	      case GDK_ACTION_MOVE:
	      case GDK_ACTION_LINK:
		if(status == -4)
		    buf = STRDUP(
			"Extract operation canceled"
		    );
		else if(total_objects_processed > 0)
		    buf = g_strdup_printf(
			"Extracted %i %s",
			total_objects_processed,
			(total_objects_processed == 1) ? "object" : "objects"
		    );
		else
		    buf = g_strdup_printf(
			"Unable to extract %s",
			(total_src_objects == 1) ? "object" : "objects"
		    );
		break;
	    }
	}

	/* Set the status bar message */
	EDVStatusBarMessage(sb, buf, FALSE);

	g_free(buf);
}


/*
 *	Image Browser drag data received nexus.
 *
 *	All inputs assumed valid.
 */
static void EDVImbrDragDataReceivedNexus(
	edv_core_struct *core, edv_imbr_struct *imbr,
	GdkDragContext *dc, guint info, GtkSelectionData *selection_data,
	edv_object_struct *obj			/* Can be NULL */
)
{
	gint initial_confirmation_result, nurls;
	gchar *parent_path;
	GList *urls_list;
	GtkWidget *toplevel = imbr->toplevel;
	const url_struct *url;

#define CLEANUP_RETURN	{		\
 g_free(parent_path);			\
					\
 if(urls_list != NULL) {		\
  g_list_foreach(			\
   urls_list, (GFunc)URLDelete, NULL	\
  );					\
  g_list_free(urls_list);		\
 }					\
					\
 return;				\
}

	/* Get copy of parent_path as the object's full path if the
	 * object is not NULL
	 */
	parent_path = (obj != NULL) ? STRDUP(obj->full_path) : NULL;

	/* If parent_path is not obtainable from the object then
	 * use the image browser's current location as the parent_path
	 */
	if(parent_path == NULL)
	    parent_path = STRDUP(EDVImbrCurrentLocation(imbr));

	/* Check if parent_path does not lead to a directory, if it does
	 * not lead to a directory then we need to make the parent_path's
	 * parent the directory to drop to
	 */
	if(!ISPATHDIR(parent_path))
	{
	    /* Droped on object does not lead to a directory, so change
	     * the parent_path to reflect its own parent so we know that
	     * it is a directory
	     */
	    gchar *s = g_dirname(parent_path);
	    g_free(parent_path);
	    parent_path = s;
	}

	/* Decode the DDE buffer into a list of URLs */
	urls_list = URLDecode(
	    (const guint8 *)selection_data->data,
	    selection_data->length
	);
	nurls = g_list_length(urls_list);

	/* External drag source checks go here, for example sources
	 * that have a url with a http or ftp protocol
	 */
	url = (urls_list != NULL) ? URL(urls_list->data) : NULL;
	if((url != NULL) ? !STRISEMPTY(url->protocol) : FALSE)
	{
	    const gchar *protocol = url->protocol;
	    if(!g_strcasecmp(protocol, "http") ||
	       !g_strcasecmp(protocol, "ftp") ||
	       !g_strcasecmp(protocol, "https")
	    )
	    {
		EDVInternetDownloadObject(
		    core, url, parent_path, toplevel
		);

		/* Do not continue with standard dnd handling */
		CLEANUP_RETURN;
	    }
	}


	/* Make initial user confirmation querying to proceed with
	 * this operation
	 */
	initial_confirmation_result = EDVConfirmDND(
	    core, (gint)dc->action, info,
	    toplevel, urls_list,
	    parent_path
	);
	/* User confirmed and dnd target type is a disk object? */
	if((initial_confirmation_result == CDIALOG_RESPONSE_YES) &&
	   ((info == EDV_DND_INFO_TEXT_PLAIN) ||
	    (info == EDV_DND_INFO_TEXT_URI_LIST) ||
	    (info == EDV_DND_INFO_STRING)
	   )
	)
	{
	    gint status = -1;
	    gboolean yes_to_all = FALSE;
	    gint objects_processed = 0;
	    gchar *new_path;
	    const gchar *path, *error_title, *error_msg;
	    GList *glist;
	    struct stat lstat_buf;

	    EDVImbrSetBusy(imbr, TRUE);
	    imbr->processing = TRUE;

	    /* Iterate through URL list */
	    for(glist = urls_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		url = URL(glist->data);
		if(url == NULL)
		    continue;

		path = url->path;
		if(STRISEMPTY(path))
		    continue;

		/* Handle by drag action type */
		status = -1;
		new_path = NULL;
		error_msg = NULL;
		switch((gint)dc->action)
		{
		  case GDK_ACTION_COPY:
		    status = EDVObjectOPCopy(
			core, path, parent_path,
			&new_path, toplevel,
			TRUE, TRUE,
			&yes_to_all
		    );
		    error_title = "Copy Object Error";
		    break;
		  case GDK_ACTION_MOVE:
		    status = EDVObjectOPMove(
			core, path, parent_path,
			&new_path, toplevel,
			TRUE, TRUE,
			&yes_to_all
		    );
		    error_title = "Move Object Error";
		    break;
		  case GDK_ACTION_LINK:
		    status = EDVObjectOPLink(
			core, parent_path, path,
			&new_path, toplevel,
			TRUE, TRUE,
			&yes_to_all
		    );
		    error_title = "Link Object Error";
		    break;
		  default:
		    error_title = "Error";
		    error_msg = "Unsupported drag operation";
		    break;
		}

		/* Check for errors */
		if(error_msg == NULL)
		    error_msg = EDVObjectOPGetError(core);
		if(!STRISEMPTY(error_msg))
		{
		    /* Report the error */
		    EDVPlaySoundError(core);
		    EDVMessageError(
			error_title,
			error_msg,
			NULL,
			toplevel
		    );
		}

		/* Get the source object's statistics for reporting */
		if(lstat((const char *)path, &lstat_buf))
		{
#ifdef ENOENT
		    /* Report the object being removed */
		    const gint error_code = (gint)errno;
		    if(error_code == ENOENT)
		    {
			imbr->processing = FALSE;
			EDVObjectRemovedEmit(core, path);
			imbr->processing = TRUE;
		    }
#endif
		}

		/* Report the new object being added? */
		if(new_path != NULL)
		{
		    if(!lstat((const char *)new_path, &lstat_buf))
		    {
			objects_processed++;

			imbr->processing = FALSE;
			EDVObjectAddedEmit(core, new_path, &lstat_buf);
			imbr->processing = TRUE;
		    }

		    g_free(new_path);
		}

		/* User aborted? */
		if(status == -4)
		    break;
	    }

	    /* Update the status bar message */
	    EDVImbrDNDUpdateStatusBar(
		imbr, (gint)dc->action, info,
		nurls, objects_processed, status
	    );

	    /* Unmap progress dialog, it may have been mapped if any
	     * operations occured in the above loop
	     */
	    ProgressDialogBreakQuery(TRUE);
	    ProgressDialogSetTransientFor(NULL);

	    /* Play the "completed" sound on success */
	    if(status == 0)
		EDVPlaySoundCompleted(core);

	    imbr->processing = FALSE;
	    EDVImbrSetBusy(imbr, FALSE);
	}
	/* User confirmed and dnd target type is a recycled object? */
	else if((initial_confirmation_result == CDIALOG_RESPONSE_YES) &&
		(info == EDV_DND_INFO_RECYCLED_OBJECT)
	)
	{
	    struct stat lstat_buf;
	    gboolean yes_to_all = FALSE;
	    gint status = -1, objects_recovered = 0;
	    guint index;
	    gchar *new_path;
	    const gchar *error_msg;
	    GList *glist;

	    EDVImbrSetBusy(imbr, TRUE);
	    imbr->processing = TRUE;

	    /* Iterate through URL list */
	    for(glist = urls_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {  
		url = URL(glist->data);
		if(url == NULL)
		    continue;

		index = (guint)ATOI(url->path);

		/* Handle by drag action type */
		status = -1;
		new_path = NULL;
		error_msg = NULL;
		switch((gint)dc->action)
		{
		  case GDK_ACTION_COPY:
		    error_msg = "Recycled objects may not be coppied";
		    break;
		  case GDK_ACTION_MOVE:
		    status = EDVRecBinRecover(
			core,
			index,		/* Recycled object to recover */
			parent_path,	/* Alternate recovery location */
			&new_path,	/* Recovered object path return */
			toplevel,
			TRUE, TRUE, &yes_to_all
		    );
		    break;
		  case GDK_ACTION_LINK:
		    error_msg = "Recycled objects may not be linked";
		    break;
		  default:
		    error_msg = "Unsupported drag operation";
		    break;
		}

		/* Check for errors */
		if(error_msg == NULL)
		    error_msg = EDVRecBinRecoverGetError(core);
		if(!STRISEMPTY(error_msg))
		{
		    /* Report the error */
		    EDVPlaySoundError(core);
		    EDVMessageError(
			"Recover Recycled Object Error",
			error_msg,
			NULL,
			toplevel
		    );
		}

		/* Object recovered? */
		if(new_path != NULL)
		{
		    if(!lstat((const char *)new_path, &lstat_buf))
		    {
			objects_recovered++;

			imbr->processing = FALSE;
			EDVRecycledObjectRemovedEmit(core, index);
			EDVObjectAddedEmit(core, new_path, &lstat_buf);
			imbr->processing = TRUE;
		    }

		    g_free(new_path);
		}

		/* User aborted? */
		if(status == -4)
		    break;
	    }

	    /* Update the status bar message */
	    EDVImbrDNDUpdateStatusBar(
		imbr, (gint)dc->action, info,
		nurls, objects_recovered, status
	    );

	    /* Unmap progress dialog, it may have been mapped if any
	     * operations occured in the above loop
	     */
	    ProgressDialogBreakQuery(TRUE);
	    ProgressDialogSetTransientFor(NULL);

	    /* Play the "completed" sound on success */
	    if(status == 0)
		EDVPlaySoundCompleted(core);

	    imbr->processing = FALSE;
	    EDVImbrSetBusy(imbr, FALSE);
	}
	/* User confirmed and dnd target type is an archive object? */
	else if((initial_confirmation_result == CDIALOG_RESPONSE_YES) &&
		(info == EDV_DND_INFO_ARCHIVE_OBJECT)
	)
	{
	    gboolean	yes_to_all = FALSE;
	    gboolean	preserve_directories = TRUE,
			preserve_timestamps = TRUE;
	    gint status, objects_extracted = 0;
	    const gchar *arch_path = NULL, *error_msg;
	    gchar *password = NULL;
	    GList *obj_list, *new_paths_list;

	    EDVImbrSetBusy(imbr, TRUE);
	    imbr->processing = TRUE;

	    /* Get location of the archive itself (the first object in
	     * the list)
	     */
	    if(nurls >= 1)
	    {
		url = URL(urls_list->data);
		arch_path = (url != NULL) ? url->path : NULL;
	    }

	    /* Query user for extract from archive options */
	    if(!EDVArchExtractOptsGetResponse(
		core, toplevel,
		arch_path,
		&password,
		&preserve_directories,
		&preserve_timestamps
	    ))
	    {
		/* User canceled */
		imbr->processing = FALSE;
		EDVImbrSetBusy(imbr, FALSE);
		g_free(password);  
		CLEANUP_RETURN;
	    }

	    /* Get the stats for all the objects found in the archive */
	    obj_list = NULL;
	    if(nurls >= 2)
	    {
		GList *glist, *paths_list = NULL;

		/* Add all the subsequent URL paths to the paths_list */
		for(glist = g_list_next(urls_list);	/* Skip the first URL */
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    url = URL(glist->data);
		    if(url == NULL)
			continue;

		    if(url->path == NULL)
			continue;

		    paths_list = g_list_append(
			paths_list,
			STRDUP(url->path)
		    );
		}
		if(paths_list != NULL)   
		{
		    obj_list = EDVArchStatList(
			core,
			arch_path,
			paths_list,
			NULL,			/* No filter */
			password,		/* No password */
			NULL, NULL
		    );
		    g_list_foreach(paths_list, (GFunc)g_free, NULL);
		    g_list_free(paths_list);
		}
	    }

	    /* Extract the object(s) from the archive */
	    new_paths_list = NULL;
	    status = EDVArchExtract(
		core, arch_path,
		obj_list,
		FALSE,				/* Not extract all */
		parent_path, &new_paths_list,
		password,
		toplevel, TRUE, TRUE, &yes_to_all,
		preserve_directories,
		preserve_timestamps
	    );

	    /* Check for errors */
	    error_msg = EDVArchExtractGetError(core);
	    if(!STRISEMPTY(error_msg))
	    {
		/* Report the error */
		EDVPlaySoundError(core);
		EDVMessageError(
		    "Extract Object Error",
		    error_msg,
		    NULL,
		    toplevel
		);
	    }

	    /* Report the extracted objects */
	    if(new_paths_list != NULL)
	    {
		const gchar *path;
		struct stat lstat_buf;
		GList *glist;

		imbr->processing = FALSE;

		for(glist = new_paths_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    path = (const gchar *)glist->data;
		    if(STRISEMPTY(path))
			continue;

		    if(lstat((const char *)path, &lstat_buf))
			continue;

		    EDVObjectAddedEmit(core, path, &lstat_buf);
		    objects_extracted++;
		}

		imbr->processing = TRUE;

		/* Delete the list of the extracted object paths */
		g_list_foreach(new_paths_list, (GFunc)g_free, NULL);
		g_list_free(new_paths_list);
	    }

	    /* Update the status bar message */
	    EDVImbrDNDUpdateStatusBar(
		imbr, (gint)dc->action, info,
		g_list_length(obj_list), objects_extracted, status
	    );

	    /* Unmap the progress dialog */
	    ProgressDialogBreakQuery(TRUE);
	    ProgressDialogSetTransientFor(NULL);

	    g_free(password);

	    /* Delete the list of archive object stats */
	    if(obj_list != NULL)
	    {
		g_list_foreach(obj_list, (GFunc)EDVArchObjectDelete, NULL);
		g_list_free(obj_list);
	    }

	    /* Play the "completed" sound on success */
	    if(status == 0)
		EDVPlaySoundCompleted(core);

	    imbr->processing = FALSE;
	    EDVImbrSetBusy(imbr, FALSE);
	}

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}


/*
 *	Image Browser Location Bar icon "enter_notify_event" or
 *	"leave_notify_event" signal callback.
 */
gint EDVImbrLocBarIconCrossingCB(
	GtkWidget *widget, GdkEventCrossing *crossing, gpointer data
)
{
	gint status = FALSE;
	GdkBitmap *mask;
	GdkPixmap *pixmap;
	GtkWidget *w;
	edv_status_bar_struct *sb;
	edv_imbr_struct *imbr = EDV_IMBR(data);
	if((crossing == NULL) || (imbr == NULL))
	    return(status);

	w = imbr->location_icon_pm;
	sb = imbr->status_bar;
	if((w == NULL) || (sb == NULL))
	    return(status);

	switch((gint)crossing->type)
	{
	  case GDK_ENTER_NOTIFY:
	    EDVStatusBarMessage(
		sb,
#if defined(PROG_LANGUAGE_SPANISH)
"Arrastre esto crear una conexin a esta ubicacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Traner ceci pour crer un lien  cet emplacement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Schleppen sie dies, eine verbindung zu diesem ort zu schaffen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Trascinare questo per creare una maglia a questa posizione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Sleep dit om een schakel aan deze plaats te creren"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Arraste isto criar um elo a esta localidade"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Slep dette skape et ledd til denne plasseringen"
#else
"Drag this to create a link to this location"
#endif
		, FALSE
	    );
	    gtk_pixmap_get(GTK_PIXMAP(w), &pixmap, &mask);
	    if(pixmap != NULL)
	    {
		gint width = 15, height = 15;
		gdk_window_get_size(pixmap, &width, &height);
		GUIDNDSetDragIcon(
		    pixmap, mask,
		    width / 2, height / 2
		);
	    }
	    status = TRUE;
	    break;

	  case GDK_LEAVE_NOTIFY:
	    EDVStatusBarMessage(sb, NULL, FALSE);
	    status = TRUE;
	    break;
	}

	return(status);
}


/*
 *	Sets the DND icon based on the Image Browser Thumb List's
 *	selected item.
 */
void EDVImbrTListDNDSetIcon(
	edv_imbr_struct *imbr, gint thumb_num
)
{
	tlist_thumb_struct *thumb_ptr;
	tlist_struct *tlist = (imbr != NULL) ? imbr->tlist : NULL;
	if(tlist == NULL)
	    return;

	/* Get pointer to thumb structure */
	if((thumb_num >= 0) && (thumb_num < tlist->total_thumbs))
	    thumb_ptr = tlist->thumb[thumb_num];
	else
	    thumb_ptr = NULL;
	if(thumb_ptr != NULL)
	{
	    GdkPixmap *pixmap = thumb_ptr->pixmap;
	    GdkBitmap *mask = thumb_ptr->mask;
	    if(pixmap != NULL)
	    {
		gint width = 15, height = 15;
		gdk_window_get_size(pixmap, &width, &height);
		GUIDNDSetDragIcon(
		    pixmap, mask,
		    width / 2, height /2
		);
	    }
	}
}

/*
 *	Image Browser Thumbs List "drag_data_get" signal callback.
 */
void EDVImbrTListDragDataGetCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gboolean data_sent = FALSE;
	gint thumb_num;
	edv_core_struct *core;
	GList *glist, *urls_list = NULL;
	url_struct *url;
	tlist_struct *tlist;
	edv_object_struct *obj;
	edv_imbr_struct *imbr = EDV_IMBR(data);
	if((dc == NULL) || (imbr == NULL))
	    return;

	if(imbr->processing)
	    return;

	tlist = imbr->tlist;
	core = imbr->core;

	EDVImbrSyncData(imbr);

	/* Generate a list of URLs from the selected thumbs */
	for(glist = tlist->selection;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    thumb_num = (gint)glist->data;
	    obj = EDV_OBJECT(TListGetThumbData(tlist, thumb_num));
	    if(obj == NULL)
		continue;

	    url = URLNew();
	    url->path = STRDUP(obj->full_path);
	    urls_list = g_list_append(urls_list, url);
	}

	/* Encode DDE buffer from the URL list */
	if(urls_list != NULL)
	{
	    gint buf_len;
	    guint8 *buf = URLEncode(urls_list, &buf_len);
	    if(buf != NULL)
	    {
		/* Send out DND data buffer */ 
		gtk_selection_data_set(
		    selection_data,
		    GDK_SELECTION_TYPE_STRING,
		    8,			/* Bits Per Character */
		    buf,		/* Data */
		    buf_len		/* Length */
		);
		data_sent = TRUE;
		g_free(buf);
	    }
	}

	/* Delete the URLs list */
	if(urls_list != NULL)
	{
	    g_list_foreach(urls_list, (GFunc)URLDelete, NULL);
	    g_list_free(urls_list);
	}

	/* If failed to send out data then respond with error */
	if(!data_sent)
	{
	    const gchar *s = "Error";
	    gtk_selection_data_set(
		selection_data,
		GDK_SELECTION_TYPE_STRING,
		8,			/* Bits Per Character */
		s,			/* Data */
		STRLEN(s)		/* Length */
	    );
	    data_sent = TRUE;
	}
}

/*
 *	Image Browser Thumbs List "drag_data_received" signal callback.
 */
void EDVImbrTListDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gint thumb_num = -1;
	tlist_struct *tlist;
	edv_object_struct *obj;
	edv_core_struct *core;
	edv_imbr_struct *imbr = EDV_IMBR(data);
	if((dc == NULL) || (imbr == NULL))
	    return;

	tlist = imbr->tlist;
	core = imbr->core;
	if((tlist == NULL) || (core == NULL))
	    return;

	/* Check if received data is not empty */
	if((selection_data != NULL) ? (selection_data->length <= 0) : TRUE)
	    return;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, imbr->toplevel))
	    return;

	EDVImbrSyncData(imbr);

	/* Find the thumb that this drop occured on, it is okay if no
	 * thumb is matched in which case thumb_num is -1 and object
	 * is NULL
	 *
	 * The drop can still continue using the current location
	 * instead of a droped on thumb since we know the drop occured
	 * on the thumbs list atleast
	 */
	if(!TListGetSelection(tlist, x, y, &thumb_num, NULL, NULL))
	    thumb_num = -1;
	obj = EDV_OBJECT(TListGetThumbData(tlist, thumb_num));

	/* Handle received drag data */
	EDVImbrDragDataReceivedNexus(
	    core, imbr, dc, info, selection_data, obj
	);
}

/*
 *	Image browser thumbs list DND "drag_data_delete" signal callback.
 */
void EDVImbrTListDragDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{

}


/*
 *	Image Browser ImgView "drag_data_received" signal callback.
 */
void EDVImbrImgViewDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	edv_core_struct *core;
	edv_imbr_struct *imbr = EDV_IMBR(data);
	if((dc == NULL) || (imbr == NULL))
	    return;

	core = imbr->core;
	if(core == NULL)
	    return;

	/* Check if received data is not empty */
	if((selection_data != NULL) ? (selection_data->length <= 0) : TRUE)
	    return;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, imbr->toplevel))
	    return;

	EDVImbrSyncData(imbr);

	/* Handle received drag data */
	EDVImbrDragDataReceivedNexus(
	    core, imbr, dc, info, selection_data,
	    NULL		/* No object, suggest use current location */
	);
}
