#ifndef _USE_BSD
/* For wait3() */
# define _USE_BSD
#endif
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <ctype.h>
#include <errno.h>
#include <signal.h>
#include <sched.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <fnmatch.h>
#include <glib.h>
#include <unistd.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"

#include "edv_types.h"
#include "edv_utils.h"
#include "config.h"


extern char **environ;


/* String Substitution */
gchar *EDVStrSub(
	const gchar *fmt, const gchar *token, const gchar *val  
);
void EDVStrEvaluateTokenHexToChar(gchar *s);

/* Type */
edv_object_type EDVStatModeToObjectType(const mode_t m);
mode_t EDVObjectTypeToStatType(const edv_object_type type);

/* Type Name */
const gchar *EDVGetObjectTypeName(const edv_object_type type);
const gchar *EDVGetObjectTypeNameLower(const edv_object_type type);

/* Sizes */
const gchar *EDVSizeStrDelim(const gulong i);
const gchar *EDVSizeStrDelimChar(const gulong i, const gchar delim_char);
const gchar *EDVSizeStrFormat(
	const gulong size,
	const edv_size_format size_format,
	const gulong block_size,
	const gchar delim_char,
	const gboolean allow_unit_conversion
);

/* Permissions */
edv_permission_flags EDVStatModeToEDVPermissions(const mode_t m);
mode_t EDVEDVPermissionsToStatMode(const edv_permission_flags permissions);
gchar *EDVGetPermissionsString(const edv_permission_flags permissions);

/* Paths */
void EDVSimplifyPath(gchar *path);
gboolean EDVIsParentPath(const gchar *parent, const gchar *child);
static gboolean EDVIsExtensionIterate(
	const gchar *name, const gint name_len,
	const gchar *ext, const gint ext_len
);
gboolean EDVIsExtension(const gchar *name, const gchar *ext);
gchar *EDVShortenPath(const gchar *path, const gint max);
gchar *EDVEvaluatePath(const gchar *parent, const gchar *path);
gchar *EDVCompletePath(
	gchar *path,
	edv_complete_path_status *status
);
gchar *EDVPlotRelativePath(
	const gchar *start_path, const gchar *dest_path
);
gchar *EDVGetCWD(void);
gint EDVSetCWD(const gchar *path);
guint EDVGetUMask(void);
void EDVSetUMask(const guint m);
gchar *EDVTmpName(const gchar *dir);
gchar *EDVWhich(const gchar *name);
gchar *EDVGetLinkTarget(const gchar *path);
gchar *EDVGetLinkTargetFull(const gchar *path);
gboolean EDVIsLinkInfinatelyRecursive(const gchar *path);

/* Device Numbers */
void EDVGetDeviceNumbers(const gint rdev, gint *major_rtn, gint *minor_rtn);
gint EDVFormatDeviceNumbers(const gint major, const gint minor);

/* Processes */
gboolean EDVProcessIsRunning(const gint pid);

static void EDVSystemWaitCB(gint s);
static gpointer EDVSystemAddSignal(gint signum, void (*handler)(gint));
gint EDVSystem(const gchar *cmd);
gint EDVSystemShell(
	const gchar *cmd,
	const gchar *shell, const gchar *shell_args
);
gint EDVSystemBlock(const gchar *cmd, gint *status);
gint EDVSystemShellBlock(
	const gchar *cmd,
	const gchar *shell, const gchar *shell_args,
	gint *status
); 

/* GList File IO */
GList *EDVOpenFileGList(const gchar *path, const gint max_lines);
void EDVSaveFileGList(const gchar *path, GList *lines_list);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


static gchar *G_STRCAT(gchar *s, const gchar *s2)
{
	if(s != NULL) {
	    if(s2 != NULL) {
		gchar *sr = g_strconcat(s, s2, NULL);
		g_free(s);
		s = sr;
	    }
	} else {
	    if(s2 != NULL)
		s = STRDUP(s2);
	    else
		s = STRDUP("");
	}
	return(s);
}


/*
 *	Substitutes a format string containing tokens for values.
 *
 *	The fmt specifies the format string.
 *
 *	The token specifies the token string.
 *
 *	The val specifies the value string to replace each token
 *	string encountered in the format string.
 *
 *	Returns a dynamically allocated string describing the
 *	substituted format string or NULL on error.
 */
gchar *EDVStrSub(
	const gchar *fmt, const gchar *token, const gchar *val  
)
{
	return((gchar *)strsub(
	    (const char *)fmt,
	    (const char *)token,
	    (const char *)val
	));
}

/*
 *	Evaluates all occurances of "%HH" by replacing them with
 *	a single character described by HH in hexidecimal value.
 *
 *	The s specifies the string to replace.
 */
void EDVStrEvaluateTokenHexToChar(gchar *s)
{
	if(s == NULL)
	    return;

	/* Iterate through the string and evaluate all tokens */
	while(*s != '\0')
	{
	    if(*s == '%')
	    {
		guint i;
		gchar *s1, *s2, hex_str[3];

		/* Get the two characters after the token character
		 * and store them in hex_str
		 *
		 * Break if the end of the string
		 * is encountered
		 */
		s++;
		if(*s == '\0')
		    break;

		hex_str[0] = *s;

		s++;
		if(*s == '\0')
		    break;

		hex_str[1] = *s;
		hex_str[2] = '\0';

		/* Convert the hex value described in hex_str into
		 * its intended character value
		 */
		sscanf(hex_str, "%x", &i);

		/* Seek back to the token start position */
		s -= 2;

		/* Replace the token with the intended character */
		*s = (gchar)i;

		/* Delete the two characters after the token start
		 * position
		 */
		for(s1 = s + 1,
		    s2 = s + 3;
		    *s2 != '\0';
		    s1++,
		    s2++
		)
		    *s1 = *s2;

		*s1 = '\0';
	    }
	    s++;
	}
}


/*
 *	Converts the stat() mode_t type to edv_object_type type.
 *
 *	The m specifies the stat() mode_t type value. Only the
 *	bit pertaining to the type in m is checked, all other bits
 *	are ignored.
 *
 *	Returns the edv_object_type type value or
 *	EDV_OBJECT_TYPE_UNKNOWN on failed match.
 */
edv_object_type EDVStatModeToObjectType(const mode_t m)
{
#ifdef S_ISREG
        if(S_ISREG(m))
#else
# warning "S_ISREG was not #defined, there will be no way to check for a regular file"
        if(TRUE)
#endif
            return(EDV_OBJECT_TYPE_FILE);
#ifdef S_ISDIR
        else if(S_ISDIR(m))
            return(EDV_OBJECT_TYPE_DIRECTORY);
#endif
#ifdef S_ISLNK
        else if(S_ISLNK(m))
            return(EDV_OBJECT_TYPE_LINK);
#endif
#ifdef S_ISCHR
        else if(S_ISCHR(m))
            return(EDV_OBJECT_TYPE_DEVICE_CHARACTER);
#endif
#ifdef S_ISBLK
        else if(S_ISBLK(m))
            return(EDV_OBJECT_TYPE_DEVICE_BLOCK);
#endif
#ifdef S_ISFIFO
        else if(S_ISFIFO(m))
            return(EDV_OBJECT_TYPE_FIFO);
#endif
#ifdef S_ISSOCK
        else if(S_ISSOCK(m))
            return(EDV_OBJECT_TYPE_SOCKET);
#endif
        else
            return(EDV_OBJECT_TYPE_FILE);
}

/*
 *	Converts the edv_object_type type to stat() mode_t type.
 *
 *	The type specifies the edv_object_type which must be one of
 *	EDV_OBJECT_TYPE_*.
 *
 *	Returns the mode_t type value or 0 on failed match.
 */
mode_t EDVObjectTypeToStatType(const edv_object_type type)
{
        switch(type)
        {
          case EDV_OBJECT_TYPE_UNKNOWN:
            break;
          case EDV_OBJECT_TYPE_FILE:
#ifdef S_IFREG
            return(S_IFREG);
#else
# warning "S_IFREG was not #defined, there will be no way to describe a regular file"
#endif
            break;
          case EDV_OBJECT_TYPE_DIRECTORY:
#ifdef S_IFDIR
            return(S_IFDIR);
#endif
            break;
          case EDV_OBJECT_TYPE_LINK:
#ifdef S_IFLNK
            return(S_IFLNK);
#endif
            break;
          case EDV_OBJECT_TYPE_FIFO:
#if defined(S_IFFIFO)
            return(S_IFFIFO);
#elif defined(S_IFIFO)
            return(S_IFIFO);
#endif
            break;
          case EDV_OBJECT_TYPE_DEVICE_BLOCK:
#ifdef S_IFBLK
            return(S_IFBLK);
#endif
            break;
          case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
#ifdef S_IFCHR
            return(S_IFCHR);
#endif
            break;
          case EDV_OBJECT_TYPE_SOCKET:
#ifdef S_IFSOCK
            return(S_IFSOCK);
#endif
            break;
          case EDV_OBJECT_TYPE_ERROR:
            break;
        }

        return(0);
}

/*
 *	Converts the edv_object_type type to a string name.
 *
 *	The type specifies the edv_object_type which must be one of
 *	EDV_OBJECT_TYPE_*.
 *
 *	Returns a statically allocated string describing the type
 *	name.
 */
const gchar *EDVGetObjectTypeName(const edv_object_type type)
{
        switch(type)
        {
          case EDV_OBJECT_TYPE_UNKNOWN:
            return("Unknown");
            break;
          case EDV_OBJECT_TYPE_FILE:
            return("File");
            break;
          case EDV_OBJECT_TYPE_DIRECTORY:
            return("Directory");
            break;
          case EDV_OBJECT_TYPE_LINK:
            return("Link");
            break;
          case EDV_OBJECT_TYPE_FIFO:
            return("FIFO Pipe");
            break;
          case EDV_OBJECT_TYPE_DEVICE_BLOCK:
            return("Block Device");
            break;
          case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
            return("Character Device");
            break;
          case EDV_OBJECT_TYPE_SOCKET:
            return("Socket");
            break;
          case EDV_OBJECT_TYPE_ERROR:
            return("Error");
            break;
        }

        return("Unknown");
}

/*
 *	Converts the edv_object_type type to a string name in
 *	lowercase.
 *
 *	The type specifies the edv_object_type which must be one of
 *	EDV_OBJECT_TYPE_*.
 *
 *	Returns a statically allocated string describing the type
 *	name in lowercase.
 */
const gchar *EDVGetObjectTypeNameLower(const edv_object_type type)
{
        switch(type)
        {
          case EDV_OBJECT_TYPE_UNKNOWN:
            return("unknown");
            break;
          case EDV_OBJECT_TYPE_FILE:
            return("file");
            break;
          case EDV_OBJECT_TYPE_DIRECTORY:
            return("directory");
            break;
          case EDV_OBJECT_TYPE_LINK:
            return("link");
            break;
          case EDV_OBJECT_TYPE_FIFO:
            return("FIFO pipe");
            break;
          case EDV_OBJECT_TYPE_DEVICE_BLOCK:
            return("block device");
            break;
          case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
            return("character device");
            break;
          case EDV_OBJECT_TYPE_SOCKET:
            return("socket");
            break;
          case EDV_OBJECT_TYPE_ERROR:
            return("error");
            break;
        }

        return("unknown");
}


/*
 *	Creates a string describing the value with ',' deliminators.
 *
 *	The i specifies the value.
 *
 *	Returns a statically allocated string describing the
 *	value with ',' deliminators or NULL on error.
 */
const gchar *EDVSizeStrDelim(const gulong i)
{
	return(EDVSizeStrDelimChar(i, ','));
}


const gchar *EDVSizeStrDelimChar(const gulong i, const gchar delim_char)
{
#define RTN_STR_LEN	80
	static gchar rtn_s[RTN_STR_LEN];

	/* 3 digits or less? (no commas needed) */
	if(i < 1000l)
	{
	    g_snprintf(rtn_s, sizeof(rtn_s), "%ld", i);
	}
	else
	{
	    gint delim_counter, src_s_len;
	    gchar	src_s[sizeof(rtn_s)], *src_ptr,
			*rtn_s_ptr = rtn_s,
			*rtn_s_end = rtn_s_ptr + sizeof(rtn_s);

	    /* Generate the source string describing the size without
	     * any deliminators
	     */
	    g_snprintf(src_s, sizeof(src_s), "%ld", i);

	    /* Calculate the length of the source string */
	    src_s_len = STRLEN(src_s);

	    /* Initialize the comma counter */
	    delim_counter = src_s_len % 3;
	    if(delim_counter <= 0)
		delim_counter = 3;

	    /* Iterate through the source string converting/copying
	     * to the return string and adding deliminators
	     */
	    src_ptr = src_s;
	    while((*src_ptr != '\0') && (rtn_s_ptr < rtn_s_end))
	    {
		/* At the position to put in a deliminator? */
		if(delim_counter <= 0)
		{
		    *rtn_s_ptr++ = delim_char;
		    delim_counter = 3;	/* Reset the deliminator counter */
		}

		if(rtn_s_ptr >= rtn_s_end)
		    break;

		*rtn_s_ptr++ = *src_ptr++;
		delim_counter--;
	    }

	    /* Null terminate the return string */
	    if(rtn_s_ptr < rtn_s_end)
		*rtn_s_ptr = '\0';
	    else
		*(rtn_s_end - 1) = '\0';
	}

	return(rtn_s);
#undef RTN_STR_LEN
}

/*
 *	Formats a string describing the size with the specified
 *	format.
 *
 *	The size specifies the size in bytes.
 *
 *	The size_format specifies the format.
 *
 *	The block_size specifies the size of each block in bytes.
 *
 *	The delim_char specifies the character to use as the
 *	deliminator.
 *
 *	If allow_unit_conversion is TRUE then the returned string
 *	will describe the size in units other than bytes if size_format
 *	specifies a size format other than bytes. Otherwise FALSE
 *	forces the returned string to describe the size in bytes.
 *
 *	Returns a statically allocated string.
 */
const gchar *EDVSizeStrFormat(
	const gulong size,
	const edv_size_format size_format,
	const gulong block_size,
	const gchar delim_char,
	const gboolean allow_unit_conversion
)
{
#define RTN_STR_LEN	80
	static gchar rtn_s[RTN_STR_LEN];

	/* Human readable */
	if((size_format == EDV_SIZE_FORMAT_HUMAN_READABLE) &&
	   allow_unit_conversion
	)
	{
	    gulong _block_size = block_size;
	    if(_block_size <= 0l)
		_block_size = 1024l;

	    /* Gigabytes */
	    if((size / _block_size / _block_size / _block_size) >= 1l)
	    {
		const gulong converted_size = size / _block_size /
		    _block_size / _block_size;
		if(converted_size >= 10l)
		    g_snprintf(
			rtn_s, sizeof(rtn_s),
			"%ldG",
			converted_size
		    );
		else
		    g_snprintf(
			rtn_s, sizeof(rtn_s),
			"%.1fG",
			(gfloat)size / (gfloat)_block_size /
			    (gfloat)_block_size / (gfloat)_block_size
		    );
	    }
	    /* Megabytes */
	    else if((size / _block_size / _block_size) >= 1l)
	    {
		const gulong converted_size = size / _block_size /
		    _block_size;
		if(converted_size >= 10l)
		    g_snprintf(
			rtn_s, sizeof(rtn_s),
			"%ldM",
			converted_size
		    );
		else
		    g_snprintf(
			rtn_s, sizeof(rtn_s),
			"%.1fM",
			(gfloat)size / (gfloat)_block_size /
			    (gfloat)_block_size 
		    );
	    }
	    /* Kilobytes */
	    else if((size / _block_size) >= 1l)
	    {
		const gulong converted_size = size / _block_size;
		if(converted_size >= 10l)
		    g_snprintf(
			rtn_s, sizeof(rtn_s),
			"%ldK",
			converted_size
		    );
		else
		    g_snprintf(
			rtn_s, sizeof(rtn_s),
			"%.1fK",
			(gfloat)size / (gfloat)_block_size
		    );
	    }
	    /* Bytes */
	    else
	    {
		g_snprintf(
		    rtn_s, sizeof(rtn_s),
		    "%ld",
		    size
		);
	    }
	}
	/* Blocks */
	else if((size_format == EDV_SIZE_FORMAT_BLOCKS) &&
		allow_unit_conversion
	)
	{
	    gulong _block_size = block_size;
	    if(_block_size <= 0l)
		_block_size = 1024l;

	    if((size / _block_size) >= 10l)
		g_snprintf(
		    rtn_s, sizeof(rtn_s),
		    "%sK",
		    EDVSizeStrDelim(size / _block_size)
		);
	    else
		g_snprintf(
		    rtn_s, sizeof(rtn_s),
		    "%.1fK",
		    (gfloat)size / (gfloat)_block_size
		);
	}
	/* Exact Deliminated */
	else if(size_format == EDV_SIZE_FORMAT_DELIMINATED)
	{
	    strcpy(
		(char *)rtn_s,
		(const char *)EDVSizeStrDelimChar(size, delim_char)
	    );
	}
	/* Exact */
	else    /* EDV_SIZE_FORMAT_RAW */
	{
	    g_snprintf(
		rtn_s, sizeof(rtn_s),
		"%ld",
		size
	    );
	}

	return(rtn_s);
#undef RTN_STR_LEN
}


/*
 *	Converts the stat() mode_t permissions to edv_permission_flags
 *	permissions.
 *
 *	The m specifies the stat() mode_t permissions.
 *
 *	Returns the edv_permission_flags permissions.
 */
edv_permission_flags EDVStatModeToEDVPermissions(const mode_t m)
{
        edv_permission_flags p = 0x00000000;

        if(m & S_IXUSR)
            p |= EDV_PERMISSION_UEXECUTE;
        if(m & S_IRUSR)
            p |= EDV_PERMISSION_UREAD;
        if(m & S_IWUSR)
            p |= EDV_PERMISSION_UWRITE;

        if(m & S_IXGRP)
            p |= EDV_PERMISSION_GEXECUTE;
        if(m & S_IRGRP)
            p |= EDV_PERMISSION_GREAD;
        if(m & S_IWGRP)
            p |= EDV_PERMISSION_GWRITE;

        if(m & S_IXOTH)
            p |= EDV_PERMISSION_AEXECUTE;
        if(m & S_IROTH)
            p |= EDV_PERMISSION_AREAD;
        if(m & S_IWOTH)
            p |= EDV_PERMISSION_AWRITE;

        if(m & S_ISUID)
            p |= EDV_PERMISSION_SETUID;
        if(m & S_ISGID)
            p |= EDV_PERMISSION_SETGID;
        if(m & S_ISVTX)
            p |= EDV_PERMISSION_STICKY;

        return(p);
}

/*
 *	Converts the edv_permission_flags permissions to stat() mode_t
 *	permissions.
 *
 *	The permissions specifies the edv_permission_flags, which can
 *	be any of EDV_PERMISSION_*.
 *
 *	Returns the stat() mode_t permissions.
 */
mode_t EDVEDVPermissionsToStatMode(const edv_permission_flags permissions)
{
        mode_t m = 0;

        if(permissions & EDV_PERMISSION_UEXECUTE)
            m |= S_IXUSR;
        if(permissions & EDV_PERMISSION_UREAD)
            m |= S_IRUSR;
        if(permissions & EDV_PERMISSION_UWRITE)
            m |= S_IWUSR;

        if(permissions & EDV_PERMISSION_GEXECUTE)
            m |= S_IXGRP;
        if(permissions & EDV_PERMISSION_GREAD)
            m |= S_IRGRP;
        if(permissions & EDV_PERMISSION_GWRITE)
            m |= S_IWGRP;

        if(permissions & EDV_PERMISSION_AEXECUTE)
            m |= S_IXOTH;
        if(permissions & EDV_PERMISSION_AREAD)
            m |= S_IROTH;
        if(permissions & EDV_PERMISSION_AWRITE)
            m |= S_IWOTH;

        if(permissions & EDV_PERMISSION_SETUID)
            m |= S_ISUID;
        if(permissions & EDV_PERMISSION_SETGID)
            m |= S_ISGID;
        if(permissions & EDV_PERMISSION_STICKY)
            m |= S_ISVTX;

        return(m);
}

/*
 *	Converts the edv_permission_flags permissions to a string
 *	description.
 *
 *	The permissions specifies the edv_permission_flags, which can
 *	be any of EDV_PERMISSION_*.
 *
 *	Returns a dynamically allocated string describing the
 *	permissions.
 */
gchar *EDVGetPermissionsString(const edv_permission_flags permissions)
{
        return(g_strdup_printf(
            "%c%c%c%c%c%c%c%c%c",
            (permissions & EDV_PERMISSION_UREAD) ? 'r' : '-',
            (permissions & EDV_PERMISSION_UWRITE) ? 'w' : '-',
            (permissions & EDV_PERMISSION_SETUID) ?
                'S' :
                ((permissions & EDV_PERMISSION_UEXECUTE) ? 'x' : '-'),
            (permissions & EDV_PERMISSION_GREAD) ? 'r' : '-',
            (permissions & EDV_PERMISSION_GWRITE) ? 'w' : '-',
            (permissions & EDV_PERMISSION_SETGID) ?
                'G' :
                ((permissions & EDV_PERMISSION_GEXECUTE) ? 'x' : '-'),
            (permissions & EDV_PERMISSION_AREAD) ? 'r' : '-',
            (permissions & EDV_PERMISSION_AWRITE) ? 'w' : '-',
            (permissions & EDV_PERMISSION_STICKY) ?
                'T' :
                ((permissions & EDV_PERMISSION_AEXECUTE) ? 'x' : '-')
        ));
}


/*
 *	Simplifies the path.
 *
 *	Removes any tailing path deliminators and reduces any
 *	occurances of "/.." or "/.".
 *
 *	The path specifies the string describing the path to simplify.
 */
void EDVSimplifyPath(gchar *path)
{
	gboolean is_absolute;

	if(STRISEMPTY(path))
	    return;

	is_absolute = g_path_is_absolute(path);

	/* Remove tailing deliminator characters if any */
	StripPath(path);

	/* Reduce all occurances of "/.." in the path */
	SimplifyPath(path);

	/* If the path was originally an absolute path and now its
	 * become an empty string then set path as toplevel
	 */
	if(is_absolute && (*path == '\0'))
	    strcpy((char *)path, G_DIR_SEPARATOR_S);
}

/*
 *	Checks if the parent path is a parent or grand parent of the
 *	child path.
 *
 *	The parent specifies the string describing the parent path.
 *	The parent path must be an absolute path that does not contain
 *	any occurances of "/.." or "/.".
 *
 *	The child specifies the string describing the child path.
 *	The child path must be an absolute path that does not contain
 *	any occurances of "/.." or "/.".
 *
 *	Returns TRUE if the parent path is a parent or grand parent
 *	of the child path.
 */
gboolean EDVIsParentPath(const gchar *parent, const gchar *child)
{
	gboolean status;
	gint len;
	gchar	*lparent,
		*lchild,
		delim_str[2];

#define CLEANUP_RETURN(_v_)	{	\
 g_free(lparent);			\
 g_free(lchild);			\
					\
 return(_v_);				\
}
	
	if(STRISEMPTY(parent) || STRISEMPTY(child))
	    return(FALSE);

	/* Both paths must be absolute */
	if(!g_path_is_absolute(parent) || !g_path_is_absolute(child))
	    return(FALSE);

	/* Make coppies of the parent and child paths */
	lparent = STRDUP(parent);
	lchild = STRDUP(child);

	/* Need to tack on a tailing G_DIR_SEPARATOR character to the
	 * parent path for proper prefix matching below
	 */
	delim_str[0] = G_DIR_SEPARATOR;
	delim_str[1] = '\0';

	len = STRLEN(lparent);
	if(len > 0)
	{
	    if(lparent[len - 1] != G_DIR_SEPARATOR)
		lparent = G_STRCAT(lparent, delim_str);
	    if(lparent == NULL)
	    {
		CLEANUP_RETURN(FALSE);
	    }
	}

	len = STRLEN(lchild);
	if(len > 0)
	{
	    if(lchild[len - 1] != G_DIR_SEPARATOR)
		lchild = G_STRCAT(lchild, delim_str);
	    if(lchild == NULL)
	    {
		CLEANUP_RETURN(FALSE);
	    }
	}

	/* Check if the lparent is a prefix of lchild, if it is then
	 * it suggest that it is a parent or grand parent of the child
	 * path
	 */
	if(strpfx((const char *)lchild, (const char *)lparent))
	    status = TRUE;
	else
	    status = FALSE;

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Called by EDVIsExtension() to check if ext matches the path.
 *
 *	The name specifies the name without the path and may not be
 *	NULL.
 *
 *	The ext expecifies one extension (no spaces) with wildcards
 *	allowed and may not be NULL.
 */
static gboolean EDVIsExtensionIterate(
	const gchar *name, const gint name_len,
	const gchar *ext, const gint ext_len
)
{
	/* Extension starts with a '.' deliminator? */
	if(*ext == '.')
	{
	    const gchar *name_ext = name + (name_len - ext_len);

	    /* Check if ext is a suffix of name */
	    if(name_len < ext_len)
		return(FALSE);

	    return((g_strcasecmp(name_ext, ext)) ? FALSE : TRUE);
	}
	/* Not an extension, so use fnmatch() */
	else
	{
	    return((fnmatch(ext, name, 0) == 0) ? TRUE : FALSE);
	}
}

/*
 *	Checks if the name's extension matches one in the extensions
 *	list.
 *
 *	The name specifies the name without the path portion, for
 *	example:
 *
 *		"file.txt"
 *
 *	The ext specifies the extensions list, a space-separated list
 *	of extensions with wilcards allowed, for example:
 *
 *		".txt .doc *rc Makefile*"
 *
 */
gboolean EDVIsExtension(const gchar *name, const gchar *ext)
{
	const gchar *ss, *ext_ptr;
	gint name_len, ext_len;
	gchar *st;
	gchar cur_ext[NAME_MAX];

	if(STRISEMPTY(name) || STRISEMPTY(ext))
	    return(FALSE);

	name_len = STRLEN(name);

	/* Set ext_ptr to start of extensions list string, seeking past
	 * any initial spaces
	 */
	ext_ptr = ext;
	while(ISBLANK(*ext_ptr))
	    ext_ptr++;

	/* Iterate through each word in the extensions list */
	while(*ext_ptr != '\0')
	{
	    /* Copy this word in the extensions list string to
	     * cur_ext and calculate this extension's length as
	     * ext_len
	     */
	    for(ext_len = 0,
		st = cur_ext,
		ss = ext_ptr;
		(ext_len < (gint)(sizeof(cur_ext) - 1)) &&
		!ISBLANK(*ss) && (*ss != '\0');
	        ext_len++
	    )
		*st++ = *ss++;
	    *st = '\0';

	    /* Check this extension word matches */
	    if(EDVIsExtensionIterate(name, name_len, cur_ext, ext_len))
		return(TRUE);

	    /* At this point ss should be at the end of the word (at
	     * the first space) or at the end of the extensions list
	     * string
	     */
	    ext_ptr = ss;
	    while(ISBLANK(*ext_ptr))
		ext_ptr++;
	}

	return(FALSE);
}

/*
 *	Shortens a path string if it exceeds a maximum number of
 *	characters.
 *
 *	The path specifies the string describing the path to shorten
 *	as needed.
 *
 *	The max specifies the maximum number of characters allowed.
 *
 *	Returns a dynamically allocated copy of the path that is no
 *	longer than max characters. A "..." will be prefixed to the
 *	returned string contents if it was longer than max
 *	characters.
 */
gchar *EDVShortenPath(const gchar *path, const gint max)
{
	gint len;

	if(path == NULL)
	    return(NULL);

	len = STRLEN(path);
	if((len > max) && (max > 3))
	{
	    /* Need to shorten string */
	    const gint i = len - max + 3;

	    return(g_strdup_printf(
		"...%s", &path[i]
	    ));
	}
	else
	{
	    return(STRDUP(path));
	}
}

/*
 *	Evaluates/coppies a path.
 *
 *	The parent specifies the string describing the parent
 *	path or current location. It is used only if path is not
 *	an absolute path. If parent is NULL or not an absolute path
 *	then the toplevel directory will be used as the parent path
 *	instead.
 *
 *	The path specifies the string describing the path to be
 *	evaluated.
 *
 *	The path will be evaulated as follows:
 *
 *	Checks if the path has a "file://" prefix and, if it does,
 *	removes it from the path.
 *
 *	Checks if the path is "." and, if it does, replaces the
 *	path with the parent path.
 *
 *	Checks if the path is ".." and, if it does, replaces the
 *	path with the parent of the parent path.
 *
 *	Checks if the path has a "~" prefix and, if it does,
 *	substitutes it with the home directory.
 *
 *	Postfixed to the parent path if path is not an absolute path.
 *
 *	Simplified, all occurances of ".." evaluated.
 *
 *	Tailing directory deliminators removed.
 *
 *	Returns a dynamically allocated string describing the
 *	evaluated path or NULL on error.
 */
gchar *EDVEvaluatePath(const gchar *parent, const gchar *path)
{
	gchar *eval_path;

	if(path == NULL)
	{
	    errno = EINVAL;
	    return(NULL);
	}

	if(parent == NULL)
#if defined(_WIN32)
	    parent = "\\";
#else
	    parent = "/";
#endif
	else if(!g_path_is_absolute(parent))
#if defined(_WIN32)
	    parent = "\\";
#else
	    parent = "/";
#endif

	/* Remove the "file://" prefix as needed */
	if(strpfx((const char *)path, "file://"))
	{
	    const gint pfx_len = STRLEN("file://");
	    eval_path = STRDUP(path + pfx_len);
	}
	/* Current directory? */
	else if(!strcmp((const char *)path, "."))
	{
	    eval_path = STRDUP(parent);
	}
	/* Parent directory? */
	else if(!strcmp((const char *)path, ".."))
	{
	    eval_path = g_dirname(parent);
	}
	/* Home directory prefix? */
	else if(*path == '~')
	{
	    /* Prefix the value from the HOME environment variable to
	     * the input path and generate the new path
	     */
	    const gchar	*home = g_getenv(ENV_VAR_NAME_HOME),
			*sub_path = path + 1;

	    /* If home directory environment variable value was not set
	     * then assume toplevel
	     */
	    if(home == NULL)
#if defined(_WIN32)
		home = "\\";
#else
		home = "/";
#endif

	    /* Seek sub_path, which is now after the prefix, past
	     * any deliminator characters
	     */
	    while(*sub_path == G_DIR_SEPARATOR)
		sub_path++;

	    /* Create the evaluated absolute path with the home dir
	     * prefixed
	     */
	    eval_path = g_strconcat(
		home,
		G_DIR_SEPARATOR_S,
		sub_path,
		NULL
	    );
	}
	else
	{
	    /* Create the evaluated absolute path */
	    eval_path = g_path_is_absolute(path) ?
		STRDUP(path) :
		STRDUP(PrefixPaths(parent, path));
	}
	if(eval_path == NULL)
	{
	    errno = ENOMEM;
	    return(NULL);
	}

	/* Begin simplifying the generated path */

	/* Simplify path, reducing occurances of "../" */
	SimplifyPath(eval_path);

	/* Remove tailing deliminators */
	StripPath(eval_path);

	return(eval_path);
}

/*
 *	Completes the path.
 *
 *	The path specifies the string describing the path to complete,
 *	this string may be modified by this function. The path must
 *	be an absolute path. In the case of an empty path, the path
 *	will be returned describing toplevel.
 *
 *	If status is not NULL then *status will be set to one of
 *	EDV_COMPLETE_PATH_*.
 *
 *	Returns the reallocated path on success or NULL on error.
 */
gchar *EDVCompletePath(
	gchar *path,
	edv_complete_path_status *status
)
{
	int status2;

	path = (gchar *)CompletePath((char *)path, &status2);
	if(status != NULL)
	{
	    switch(status2)
	    {
	      case COMPLETE_PATH_SUCCESS:
		*status = EDV_COMPLETE_PATH_SUCCESS;
		break;
	      case COMPLETE_PATH_NONE:
		*status = EDV_COMPLETE_PATH_NONE;
		break;
	      case COMPLETE_PATH_AMBIGUOUS:
		*status = EDV_COMPLETE_PATH_AMBIGUOUS;
		break;
	      case COMPLETE_PATH_PARTIAL:
		*status = EDV_COMPLETE_PATH_PARTIAL;
		break;
	    }
	}

	return(path);
}

/*
 *	Generates a relative path from a starting path to a
 *	destination path.
 *
 *	The start_path specifies the starting path. The start_path
 *	must be an absolute path.
 *
 *	The dest_path specifies the destination path. The dest_path
 *	must be an absolute path.
 *
 *	Returns a dynamically allocated string describing the
 *	relative path or NULL on error.
 */
gchar *EDVPlotRelativePath(
	const gchar *start_path, const gchar *dest_path
)
{
	gint i, tar_deliminators = 0;
	gchar *parent_path, *rel_path;
	const gchar *src_s, *tar_s, *tar_s2;

	if(STRISEMPTY(start_path) || STRISEMPTY(dest_path))
	{
	    errno = EINVAL;
	    return(NULL);
	}

	/* Get the path to the link's parent */
	parent_path = g_dirname(start_path);
	if(parent_path == NULL)
	    parent_path = STRDUP(start_path);
	if(parent_path == NULL)
	{
	    errno = ENOMEM;
	    return(NULL);
	}

	rel_path = NULL;

	/* If the destination and the starting path parents are the
	 * same paths then just return "."
	 */
	if(!strcmp((const char *)dest_path, (const char *)parent_path))
	{
	    rel_path = STRDUP(".");
	    g_free(parent_path);
	    return(rel_path);
	}

	/* Set the starting destination and parent path positions past
	 * their toplevel characters
	 */
	src_s = (gchar *)strchr((char *)dest_path, G_DIR_SEPARATOR);
	if(src_s != NULL)
	    src_s++;
	else
	    src_s = dest_path + 1;

	tar_s = (gchar *)strchr((char *)parent_path, G_DIR_SEPARATOR);
	if(tar_s != NULL)
	    tar_s++;
	else
	    tar_s = parent_path + 1;

	/* Seek src_s and tar_s to the first character of which they
	 * differ
	 */
	while((*src_s != '\0') && (*tar_s != '\0'))
	{
	    if(*src_s != *tar_s)
		break;

	    src_s++;
	    tar_s++;
	}
	/* Deliminator at source position where difference was
	 * encountered? If so then we need to decrement deliminator
	 * count by one
	 *
	 * This will get added up to or past 0 further below
	 */
	if(*src_s == G_DIR_SEPARATOR)
	    tar_deliminators--;

	/* Seek source position backwards to last deliminator, but keep
	 * it one position ahead of the last deliminator
	 */
	while(src_s > dest_path)
	{
	    if(*src_s == G_DIR_SEPARATOR)
	    {
		src_s++;
		break;
	    }

	    src_s--;
	}
	/* If source position seeked all the way back to the beginning
	 * then increment it one past the first character to skip the
	 * toplevel character and thus keep source position infront of
	 * the last deliminator
	 */
	if(src_s <= dest_path)
	    src_s = dest_path + 1;


	/* Count deliminators in target path from where target position
	 * differed from the corresponding source position
	 */
	tar_deliminators++;
	tar_s2 = tar_s;
	while(*tar_s2 != '\0')
	{
	    if(*tar_s2 == G_DIR_SEPARATOR)
		tar_deliminators++;

	    tar_s2++;
	}

	/* Special check, if the target happens to be just toplevel
	 * then do not count any deliminators
	 */
	if(!strcmp((const char *)parent_path, "/"))
	    tar_deliminators = 0;

	/* Begin generating new path */
	g_free(rel_path);
	rel_path = STRDUP("");
	for(i = 0; i < tar_deliminators; i++)
	    rel_path = G_STRCAT(rel_path, "../");

	rel_path = G_STRCAT(rel_path, src_s);
	if(rel_path == NULL)
	{
	    g_free(parent_path);
	    errno = ENOMEM;
	    return(rel_path);
	}

	/* If new path was generated as an empty string (perhaps if the
	 * source and target directories were the same), then just set
	 * the new path to "." to indicate current directory
	 */
	if(*rel_path == '\0')
	{
	    rel_path = G_STRCAT(rel_path, ".");
	    if(rel_path == NULL)
	    {
		g_free(parent_path);
		errno = ENOMEM;
		return(rel_path);
	    }
	}

	/* Remove any tailing path deliminators */
	StripPath((char *)rel_path);

	g_free(parent_path);

	return(rel_path);
}


/*
 *	Gets the current working directory.
 *
 *	Returns a dynamically allocated string describing the current
 *	working directory, the calling function must delete this
 *	string.
 */
gchar *EDVGetCWD(void)
{
	return(STRDUP(g_get_current_dir()));
}

/*
 *	Sets the current working directory.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint EDVSetCWD(const gchar *path)
{
	if(STRISEMPTY(path))
	{
	    errno = EINVAL;
	    return(-2);
	}

	return(chdir(path));
}

/*
 *	Returns the current umask.
 */
guint EDVGetUMask(void)
{
	const guint m = (guint)umask(0);
	umask((mode_t)m);
	return(m);
}

/*
 *	Sets the umask.
 */
void EDVSetUMask(const guint m)
{
	umask((mode_t)m);
}

/*
 *	Returns a dynamically allocated string describing a unique file
 *	name in the specified dir that is gauranteed to not exist.
 */
gchar *EDVTmpName(const gchar *dir)
{
	gint fd;
	gchar *path;

	if(STRISEMPTY(dir))
	    dir = g_getenv(ENV_VAR_NAME_TMPDIR);
	if(STRISEMPTY(dir))
#if defined(P_tmpdir)
	    dir = P_tmpdir;
#elif defined(_WIN32)
	    dir = "C:\\TEMP";
#else
	    dir = "/tmp";
#endif

	/* Create template path */
	path = STRDUP(PrefixPaths(
	    dir,
	    PROG_NAME "XXXXXX"
	));
	if(path == NULL)
	{
	    errno = ENOMEM;
	    return(NULL);
	}

	/* Create tempory file and modify the template path */
	fd = (gint)mkstemp((char *)path);
	if(fd > -1)
	    close((int)fd);

	return(path);
}

/*
 *	Returns a dynamically allocated string path describing the
 *	completed absolute path to the program found from the PATH
 *	environment variable for the program specified name or NULL if
 *	there is no match.
 */
gchar *EDVWhich(const gchar *name)
{
	struct stat stat_buf;
	gint i;
	const gchar *path_list;
	gchar *s, *matched_path, **pathv;

	if(STRISEMPTY(name))
	    return(NULL);

	/* Specified name already has an absolute path to it? */
	if(g_path_is_absolute(name))
	    return(STRDUP(name));

	/* Get the value of the path environment */
	path_list = g_getenv(ENV_VAR_NAME_PATH);
	if(path_list == NULL)
	    return(NULL);

	/* Break up the path environment into individual paths */
	pathv = g_strsplit(path_list, G_SEARCHPATH_SEPARATOR_S, -1);
	if(pathv == NULL)
	    return(NULL);

	/* Check each individual path location for the specified name */
	matched_path = NULL;
	for(i = 0; pathv[i] != NULL; i++);
	for(i--; i >= 0; i--)
	{
	    s = g_strconcat(
		pathv[i],
		G_DIR_SEPARATOR_S,
		name,
		NULL
	    );
	    if(s == NULL)
		continue;

	    if(!stat((const char *)s, &stat_buf))
	    {
#if defined(S_IXUSR) && defined(S_IXGRP) && defined(S_IXOTH)
		const mode_t m = stat_buf.st_mode;
		if((m & S_IXUSR) ||
		   (m & S_IXGRP) ||
		   (m & S_IXOTH)
		)
		{
		    matched_path = s;
		    break;
		}
#endif
	    }

	    g_free(s);
	}

	g_strfreev(pathv);

	return(matched_path);
}

/*
 *	Gets the link's target value.
 *
 *	The path specifies the full path to the link.
 *
 *	Returns a dynamically allocated string containing the link's
 *	target value or NULL on error. The global errno will be
 *	set if an error has occured.
 */
gchar *EDVGetLinkTarget(const gchar *path)
{
	struct stat lstat_buf;
	gint len, bytes_read;
	gchar *v;

	if(STRISEMPTY(path))
	{
	    errno = EINVAL;
	    return(NULL);
	}

	if(lstat((const char *)path, &lstat_buf))
	    return(NULL);

#ifdef S_ISLNK
	if(!S_ISLNK(lstat_buf.st_mode))
#else
	if(TRUE)
#endif
	{
	    errno = EINVAL;
	    return(NULL);
	}

	len = (gint)lstat_buf.st_size;
	v = (gchar *)g_malloc((len + 1) * sizeof(gchar));
	if(v == NULL)
	    return(NULL);

	if(len > 0)
	{
	    bytes_read = readlink(
		(const char *)path,		/* Link */
		v,				/* Value return */
		(size_t)len			/* Value allocation */
	    );
	    if(bytes_read != len)
	    {
		const gint error_code = (gint)errno;
		g_free(v);
		errno = (int)error_code;
		return(NULL);
	    }
	}

	v[len] = '\0';

	return(v);
}

/*
 *	Same as EDVGetLinkTarget() except that it appends the parent
 *	of the path to the link's target value if the link's
 *	target value is a relative path.
 */
gchar *EDVGetLinkTargetFull(const gchar *path)
{
	gchar	*parent,
		*full_path,
		*v = EDVGetLinkTarget(path);
	if(v == NULL)
	    return(NULL);

	if(g_path_is_absolute(v))
	    return(v);

	parent = g_dirname(path);
	if(parent == NULL)
	    return(v); 

	full_path = g_strconcat(
	    parent,
	    G_DIR_SEPARATOR_S,
	    v,
	    NULL
	);

	g_free(parent);
	g_free(v);

	SimplifyPath(full_path);

	return(full_path);
}

/*
 *	Checks if the link specified by path has a target that may
 *	possibly be infinately recursive.
 */
gboolean EDVIsLinkInfinatelyRecursive(const gchar *path)
{
	struct stat stat_buf;
	gboolean status = FALSE;
	gint len;
	gchar *s, *parent, *cur_path;
	gchar link_value[PATH_MAX + NAME_MAX + 1];

	if(STRISEMPTY(path))
	    return(status);

	/* Is the link's target not valid? */
	if(stat((const char *)path, &stat_buf))
	    return(status);

#ifdef S_ISDIR
	/* Is the link's target not a directory? */
	if(!S_ISDIR(stat_buf.st_mode))
	    return(status);
#endif

	/* Record the parent of the starting link */
	parent = g_dirname(path);
	if(parent == NULL)
	    return(status);

	/* Begin going through each link to reach the target */
	cur_path = STRDUP(path);
	while(cur_path != NULL)
	{
	    /* Get the link's target value, if we get an error it
	     * means we have reached the end and cur_path is the final
	     * target
	     */
	    len = (gint)readlink(cur_path, link_value, sizeof(link_value));
	    if(len <= 0)
		break;
	    if(len >= (gint)sizeof(link_value))
		len = MAX((gint)(sizeof(link_value) - 1), 0);
	    link_value[len] = '\0';

	    /* Get this link's parent */
	    s = g_dirname(cur_path);
	    if(s == NULL)
		break;

	    /* Get the link's target as the current path */
	    g_free(cur_path);
	    cur_path = EDVEvaluatePath(s, link_value);
	    g_free(s);
	}

	if(cur_path != NULL)
	{
	    /* Is the target a grand parent of the starting link? */
	    if(strpfx(parent, cur_path))
		status = TRUE;
	}

	g_free(cur_path);
	g_free(parent);

	return(status);
}


/*
 *	Gets the device major and minor numbers from the device value.
 *
 *	The rdev specifies the device value which should come from
 *	*stat()'s struct stat (dev_t)st_rdev value.
 *
 *	The major_rtn specifies the device major number return value.
 *
 *	The minor_rtn specifies the device minor number return value.
 */
void EDVGetDeviceNumbers(const gint rdev, gint *major_rtn, gint *minor_rtn)
{
	if(major_rtn != NULL)
	    *major_rtn = (gint)(((guint32)rdev >> 8) & 0x000000ff);
	if(minor_rtn != NULL)
	    *minor_rtn = (gint)((guint32)rdev & 0x000000ff);
}

/*
 *	Combines the device major and minor numbers into a single
 *	device value.
 *
 *	The major specifies the device major number.
 *
 *	The minor specifies the device minor number.
 *
 *	Returns the device value which confirms to *stat()'s struct
 *	stat (dev_t)st_rdev value.
 */
gint EDVFormatDeviceNumbers(const gint major, const gint minor)
{
	return((gint)(
	    (((guint32)major & 0x000000ff) << 8) |
	    ((guint32)minor & 0x000000ff)
	));
}


/*
 *	Checks if the process speecified by pid is still running.
 */
gboolean EDVProcessIsRunning(const gint pid)
{
#if defined(__linux__)
	gchar path[PATH_MAX + NAME_MAX];
	g_snprintf(path, sizeof(path), "/proc/%i", pid);
	return(access(path, F_OK) ? FALSE : TRUE);
#else
	struct sched_param sp;

	if(pid <= 0)
	{
	    errno = EINVAL;
	    return(FALSE);
	}

	if(sched_getparam(
	    (pid_t)pid,
	    &sp
	) == 0)
	    return(TRUE);
	else
	    return(FALSE);
#endif   
}


/*
 *	SIGCHLD signal callback.
 *
 *	Used by EDVSystem*().
 */
static void EDVSystemWaitCB(gint s)
{
	gint status;
	while(wait3((int *)&status, WNOHANG, NULL) > 0);
}

/*
 *	Adds a signal using the BSD style signal(2).
 *
 *	Returns the old signal action handler of type
 *	void (*handler)(gint).
 */
static gpointer EDVSystemAddSignal(gint signum, void (*handler)(gint))
{
	struct sigaction act, old_act;

	act.sa_handler = handler;
	act.sa_flags = 0;

	if(sigaction((int)signum, &act, &old_act) == -1)
	    return((gpointer)SIG_ERR);
	else
	    return((gpointer)old_act.sa_handler);
}

/*
 *	Executes the command in the background.
 *
 *	The cmd specifies the command. The command must contain
 *	an absolute path to an executable and (optionally) any
 *	subsequent arguments. Shell tokens are accepted, the shell
 *	defaults to /bin/sh.
 *
 *	Returns the process ID or negative on error.
 */
gint EDVSystem(const gchar *cmd)
{
	return(EDVSystemShell(cmd, NULL, NULL));
}

/*
 *	Executes the command in the background using a particular
 *	shell.
 *
 *	The cmd specifies the command. The command must contain
 *	an absolute path to an executable and (optionally) any
 *	subsequent arguments. Shell tokens are accepted.
 *
 *	The shell specifies the full path to the shell. If shell
 *	is NULL then the shell specified by the environment
 *	variable SHELL will be used. If no environment variable is
 *	specified then "/bin/sh" will be used.
 *
 *	The shell_args specifies the shell's arguments. If shell_args
 *	is NULL then "-c" will be used.
 *
 *	Returns the process ID or negative on error.
 */     
gint EDVSystemShell(
	const gchar *cmd, 
	const gchar *shell, const gchar *shell_args
)
{
	gint pid;

	if(STRISEMPTY(shell))
	    shell = g_getenv(ENV_VAR_NAME_SHELL);
	else if(!g_path_is_absolute(shell))
	    shell = g_getenv(ENV_VAR_NAME_SHELL);
	if(shell == NULL)
	    shell = "/bin/sh";

	if(STRISEMPTY(shell_args))
	    shell_args = "-c";

	if(STRISEMPTY(cmd))
	{
	    errno = EINVAL;
	    return(-2);
	}

#ifdef SIGCHLD
	EDVSystemAddSignal(SIGCHLD, EDVSystemWaitCB);
#endif

	/* Fork */
	pid = (gint)fork();
	if(pid == -1)
	    return(-3);

	/* Are we the child? */
	if(pid == 0)
	{
	    /* We are the child, execute the command */
	    gchar *argv[4];
	    argv[0] = (gchar *)g_basename(shell);
	    argv[1] = (gchar *)shell_args;
	    argv[2] = (gchar *)cmd;
	    argv[3] = NULL;
	    execve((const char *)shell, (char **)argv, environ);
	    exit(0);
	}
	else
	{
	    /* We are the parent */
	}

	return(pid);
}

/*
 *	Executes the command and waits for it to finish.
 *
 *	If status is not NULL then the status of the command
 *	be returned.
 *
 *	Returns the process ID or -1 on error.
 */
gint EDVSystemBlock(const gchar *cmd, gint *status)
{
	return(EDVSystemShellBlock(cmd, NULL, NULL, status));
}

/*
 *	Executes the command and blocks until it has exited.
 *
 *	The cmd specifies the command. The command must contain
 *	an absolute path to an executable and (optionally) any
 *	subsequent arguments. Shell tokens are accepted.
 *
 *	The shell specifies the full path to the shell. If shell
 *	is NULL then "/bin/sh" will be used.
 *
 *	The shell specifies the full path to the shell. If shell
 *	is NULL then the shell specified by the environment
 *	variable SHELL will be used. If no environment variable is
 *	specified then "/bin/sh" will be used.
 *
 *	If status is not NULL then the return value of the executed
 *	process will be set to *status.
 *
 *	Returns the process ID or negative on error.
 */
gint EDVSystemShellBlock(
	const gchar *cmd,
	const gchar *shell, const gchar *shell_args,
	gint *status
)
{
	gint pid;

	if(status != NULL)
	    *status = 0;

	if(STRISEMPTY(shell))
	    shell = g_getenv(ENV_VAR_NAME_SHELL);
	else if(!g_path_is_absolute(shell))
	    shell = g_getenv(ENV_VAR_NAME_SHELL);
	if(shell == NULL)
	    shell = "/bin/sh";

	if(STRISEMPTY(shell_args))
	    shell_args = "-c";

	if(STRISEMPTY(cmd))
	{
	    errno = EINVAL;
	    return(-2);
	}

	/* Fork */
	pid = (gint)fork();
	if(pid == -1)
	    return(-3);

	/* Are we the child? */
	if(pid == 0)
	{
	    /* We are the child, execute the command */
	    gchar *argv[4];
	    argv[0] = (gchar *)g_basename(shell);
	    argv[1] = (gchar *)shell_args;
	    argv[2] = (gchar *)cmd;
	    argv[3] = NULL;
	    execve((const char *)shell, (char **)argv, environ);
	    exit(0);
	}
	else
	{
	    /* We are the parent, wait for child to exit */
	    const gint wait_status = (gint)waitpid(
		(int)pid, (int *)status, 0
	    );
	    if(wait_status == -1)
	    {
		/* Interrupted? */
		if(errno == EINTR)
		{

		}
	    }
	}

	return(pid);
}


/*
 *	Opens a GList of gchar * strings from a file.
 *
 *	The path specifies the full path to the file to open from.
 *
 *	If max_lines is positive then no more than max_lines will be
 *	read from the file.
 *
 *	Returns a dynamically allocated GList of gchar * strings.
 */
GList *EDVOpenFileGList(const gchar *path, const gint max_lines)
{
	FILE *fp;
	gint lines_read;
	gchar *s;
	GList *lines_list;

	if(STRISEMPTY(path))
	    return(NULL);

	/* Open the file for reading */
	fp = fopen((const char *)path, "rb");
	if(fp == NULL)
	    return(NULL);

	/* Read each line from the file */
	lines_read = 0;
	lines_list = NULL;
	s = (gchar *)FGetStringLiteral(fp);
	while(s != NULL)
	{
	    lines_read++;

	    /* Append this line to the list */
	    lines_list = g_list_append(lines_list, s);

	    /* Is max_lines specified and number of lines read has
	     * reached max_lines?
	     */
	    if((max_lines > 0) && (lines_read >= max_lines))
		break;

	    /* Get the next line */
	    s = (gchar *)FGetStringLiteral(fp);
	}

	/* Close the file */
	fclose(fp);

	return(lines_list);
}

/*
 *	Saves the GList of gchar * strings to a file.
 * 
 *	The path specifies the full path to the file to save to.
 *
 *	The lines_list specifies a GList of gchar * strings to
 *	write to the file.
 */
void EDVSaveFileGList(const gchar *path, GList *lines_list)
{
	FILE *fp;
	gint len;
	const gchar *s;
	GList *glist;

	if(STRISEMPTY(path))
	    return;

	/* Open the file for writing */
	fp = fopen((const char *)path, "wb");
	if(fp == NULL)
	    return;

	/* Write each line to the file */
	for(glist = lines_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    /* Get this line */
	    s = (const gchar *)glist->data;
	    if(s == NULL)
		continue;

	    /* Write this line */
	    len = STRLEN(s);
	    if(fwrite(s, sizeof(gchar), len, fp) < len)
		break;

	    if(fputc('\n', fp) == EOF)
		break;
	}

	/* Close the file */
	fclose(fp);
}

