/*
 *	Prints a list of all Endeavour2 configuration parameters or
 *	a specific configuration parameter.
 */

#include <glib.h>
#include <endeavour2.h>

static void print_help(const gchar *prog_name)
{
	g_print(
"Usage: %s [parameter]\n",
	    prog_name
	);
}

static void match_print_param(edv_context_struct *ctx, const gchar *param)
{
	switch(EDVGetType(ctx, param))
	{
	  case EDV_GET_TYPE_NONE:
	    g_printerr("%s: No such parameter.\n", param);
	    break;

          case EDV_GET_TYPE_INT8:
          case EDV_GET_TYPE_UINT8:
          case EDV_GET_TYPE_INT16:
          case EDV_GET_TYPE_UINT16:
          case EDV_GET_TYPE_INT32:
          case EDV_GET_TYPE_UINT32:
	    g_print(
"%s = %i\n",
		param, EDVGetI(ctx, param)
	    );
	    break;
          case EDV_GET_TYPE_INT64:
	    g_print(
"%s = %ld\n",
		param, EDVGetL(ctx, param)
	    );
	    break;
          case EDV_GET_TYPE_UINT64:
	    g_print(
"%s = %ld\n",
		param, EDVGetUL(ctx, param)
	    );
	    break;
          case EDV_GET_TYPE_FLOAT:
	    g_print(
"%s = %f\n",
		param, EDVGetF(ctx, param)
	    );
	    break;
          case EDV_GET_TYPE_DOUBLE:
	    g_print(
"%s = %f\n",
		param, EDVGetD(ctx, param)
	    );
	    break;
          case EDV_GET_TYPE_STRING:
	    g_print(
"%s = \"%s\"\n",
		param, EDVGetS(ctx, param)
	    );
	    break;
          case EDV_GET_TYPE_INTLIST:
	    g_print(
"%s = (intlist)\n",
		param
	    );
	    break;
          case EDV_GET_TYPE_COLOR:
	    g_print(
"%s = (color)\n",
		param
	    );
	    break;
          case EDV_GET_TYPE_ACCELKEY_LIST:
	    g_print(
"%s = (accelkey list)\n",
		param
	    );
	    break;
          case EDV_GET_TYPE_STYLE:
	    g_print(
"%s = (style)\n",
		param
	    );
	    break;
          case EDV_GET_TYPE_MENU:
	    g_print(
"%s = (menu)\n",
		param
	    );
	    break;
	}
}

int main(int argc, char *argv[])
{
        /* Initialize the Endeavour2 Context */
	edv_context_struct *ctx = EDVContextNew();
	EDVContextLoadConfigurationFile(ctx, NULL);

	/* Specific configuration parameter? */
	if(argc > 1)
	{
	    const gchar *arg = argv[1];

	    /* Help */
	    if(!g_strcasecmp(arg, "--help") ||
	       !g_strcasecmp(arg, "-help") ||
	       !g_strcasecmp(arg, "--h") ||
	       !g_strcasecmp(arg, "-h") ||
	       !g_strcasecmp(arg, "-?")
	    )
	    {
		print_help(argv[0]);
	    }
	    else
	    {
		/* Print the specified configuration parameter */
		match_print_param(ctx, arg);
	    }
	}
	/* Print all the configuration parameters */
	else
	{
	    gint i, total = 0;
	    gchar **list = EDVGetConfigurationParameters(ctx, &total);
	    if(list != NULL)
	    {
		for(i = 0; i < total; i++)
		{
		    match_print_param(ctx, list[i]);
		    g_free(list[i]);
		}
		g_free(list);

		g_print(
"%i %s found\n",
		    i,
		    (i == 1) ? "parameter" : "parameters"
		);
	    }
	    else
	    {
		g_printerr(
"Unable to obtain the parameters list.\n"
		);
	    }
	}

        /* Flush any pending Endeavour2 operations */
	EDVContextSync(ctx);

        /* Shutdown the Endeavour2 context */
	EDVContextDelete(ctx);

	return(0);
}
