#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#ifdef HAVE_LIBTAR
# ifdef HAVE_LIBZ
#  include <zlib.h>
# endif
# ifdef HAVE_LIBBZ2
#  include <bzlib.h>
# endif
# include <fcntl.h>
# include <time.h>
# include <utime.h>
# include <libtar.h>
#else
# include <signal.h>
# include <unistd.h>
#endif

#ifdef HAVE_LIBTAR
# include "../include/disk.h"
# include "cdialog.h"
# include "progressdialog.h"
# include "edv_types.h"
# include "edv_archive_obj.h"
# include "edv_archive_extract.h"
# include "edv_archive_extract_tar.h"
# include "endeavour2.h"
# include "edv_utils.h"
# include "edv_utils_gtk.h"
#else
# include "../include/fio.h"
# include "../include/disk.h"
# include "../include/prochandle.h"
# include "progressdialog.h"
# include "cfg.h"
# include "edv_types.h"
# include "edv_archive_obj.h"
# include "edv_archive_extract.h"
# include "edv_archive_extract_tar.h"
# include "endeavour2.h"
# include "edv_utils.h"
# include "edv_utils_gtk.h"
# include "edv_cfg_list.h"
#endif
#include "config.h"


#ifdef HAVE_LIBTAR
# ifdef HAVE_LIBZ
static int EDVArchExtractLibTarOpenLibZCB(const char *path, int oflags, int mode);
# endif
# ifdef HAVE_LIBBZ2
static int EDVArchExtractLibTarOpenLibBZ2CB(const char *path, int oflags, int mode);
static ssize_t EDVArchExtractLibTarReadLibBZ2CB(int fd, void *buf, size_t buf_len);
static ssize_t EDVArchExtractLibTarWriteLibBZ2CB(int fd, const void *buf, size_t buf_len);
static int EDVArchExtractLibTarCloseLibBZ2CB(int fd);
# endif
static edv_object_type EDVArchExtractLibTarGetType(TAR *tar);
static gchar *EDVArchExtractLibTarGetPath(TAR *tar);
static gint EDVArchExtractLibTarNext(TAR *tar);
#endif  /* HAVE_LIBTAR */

/* Extract From Archive */
gint EDVArchExtractTapeArchive(
	edv_core_struct *core,
	const gchar *arch_path,
	GList *objs_list, const gint nobjs,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps,
	const gboolean is_compress_compressed,
	const gboolean is_gzip_compressed,
	const gboolean is_bzip2_compressed
);


#if defined(HAVE_LIBTAR) && defined(HAVE_LIBBZ2)
typedef struct {
	int             fd;
	BZFILE          *bz2d;
} bz2_cb_data_struct;
#define BZ2_CB_DATA(p)                  ((bz2_cb_data_struct *)(p))
#endif


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define UNLINK(p)	(((p) != NULL) ? (gint)unlink((const char *)(p)) : -1)
#define INTERRUPT(i)	(((i) > 0) ? (gint)kill((int)(i), SIGINT) : -1)

#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))


#ifdef HAVE_LIBTAR
# ifdef HAVE_LIBZ
/*
 *	libtar GZip open callback.
 */
static int EDVArchExtractLibTarOpenLibZCB(const char *path, int oflags, int mode)
{
	const char *libz_oflags;
	int fd;
	gzFile zd;

	switch(oflags & O_ACCMODE)
	{
	  case O_WRONLY:
	    libz_oflags = "wb";
	    break;
	  case O_RDONLY:
	    libz_oflags = "rb";
	    break;
	  default:
	  case O_RDWR:
	    errno = EINVAL;
	    return(-1);
	    break;
	}

	fd = open(path, oflags, mode);
	if(fd < 0)
	    return(-1);

	if((oflags & O_CREAT) && fchmod(fd, mode))
	    return(-1);

	zd = gzdopen(fd, libz_oflags);
	if(zd == NULL)
	{
	    errno = ENOMEM;
	    return(-1);
	}

	return((int)zd);
}
# endif	/* HAVE_LIBZ */

# ifdef HAVE_LIBBZ2
/*
 *	libtar BZip2 open callback.
 */
static int EDVArchExtractLibTarOpenLibBZ2CB(const char *path, int oflags, int mode)
{
	const char *libbz2_oflags;
	bz2_cb_data_struct *d;

	switch(oflags & O_ACCMODE)
	{
	  case O_WRONLY:
	    libbz2_oflags = "wb";
	    break;
	  case O_RDONLY:
	    libbz2_oflags = "rb";
	    break;
	  default:
	  case O_RDWR:
	    errno = EINVAL;
	    return(-1);
	    break;
	}

	d = BZ2_CB_DATA(g_malloc0(
	    sizeof(bz2_cb_data_struct)
	));
	if(d == NULL)
	    return(-1);

	d->fd = open(path, oflags, mode);
	if(d->fd < 0)
	{
	    g_free(d);
	    return(-1);
	}

	if(oflags & O_CREAT)
	{
	    if(fchmod(d->fd, mode) != 0)
	    {
		const gint error_code = (gint)errno;
		close(d->fd);
		g_free(d);
		errno = (int)error_code;
		return(-1);
	    }
	}

	d->bz2d = BZ2_bzdopen(d->fd, libbz2_oflags);
	if(d->bz2d == NULL)
	{
	    close(d->fd);
	    g_free(d);
	    errno = ENOMEM;
	    return(-1);
	}

	/* Return the libbz2 callback data as the descriptor */
	return((int)d);
}

/*
 *      libtar BZip2 read callback.
 */
static ssize_t EDVArchExtractLibTarReadLibBZ2CB(int fd, void *buf, size_t buf_len)
{
	bz2_cb_data_struct *d = BZ2_CB_DATA(fd);
	return(BZ2_bzread(d->bz2d, buf, buf_len));
}

/*
 *      libtar BZip2 write callback.
 */
static ssize_t EDVArchExtractLibTarWriteLibBZ2CB(int fd, const void *buf, size_t buf_len)
{
	bz2_cb_data_struct *d = BZ2_CB_DATA(fd);
	return(BZ2_bzwrite(d->bz2d, (void *)buf, buf_len));
}

/*
 *      libtar BZip2 close callback.
 */
static int EDVArchExtractLibTarCloseLibBZ2CB(int fd)
{
	bz2_cb_data_struct *d = BZ2_CB_DATA(fd);
	BZ2_bzclose(d->bz2d);
	g_free(d);
	return(0);
}
# endif /* HAVE_LIBBZ2 */

/*
 *	Gets the Tape Archive's current object's type.
 */
static edv_object_type EDVArchExtractLibTarGetType(TAR *tar)
{
	if(TH_ISREG(tar))
	    return(EDV_OBJECT_TYPE_FILE);
	else if(TH_ISDIR(tar))
	    return(EDV_OBJECT_TYPE_DIRECTORY);
	else if(TH_ISLNK(tar) || TH_ISSYM(tar))
	    return(EDV_OBJECT_TYPE_LINK);
	else if(TH_ISBLK(tar))
	    return(EDV_OBJECT_TYPE_DEVICE_BLOCK);
	else if(TH_ISCHR(tar))
	    return(EDV_OBJECT_TYPE_DEVICE_CHARACTER);
	else if(TH_ISFIFO(tar))
	    return(EDV_OBJECT_TYPE_FIFO);
	else
	    return(EDV_OBJECT_TYPE_UNKNOWN);
}

/*
 *      Gets the Tape Archive's current object's path.
 */
static gchar *EDVArchExtractLibTarGetPath(TAR *tar)
{
	struct tar_header *tar_obj = &tar->th_buf;
	gchar *path;

	if(tar_obj->gnu_longname != NULL)
	    path = g_strdup(tar_obj->gnu_longname);
	else if(tar_obj->prefix[0] != '\0')
	    path = g_strdup_printf(
		"%.155s/%.100s",
		tar_obj->prefix,
		tar_obj->name
	    );
	else
	    path = g_strdup_printf(
		"%.100s",
		tar_obj->name
	    );
	if(path == NULL)
	    return(NULL);

	/* Remove any tailing deliminators */
	StripPath((char *)path);

	return(path);
}

/*
 *      Seeks to the next object in the Tape Archive.
 */
static gint EDVArchExtractLibTarNext(TAR *tar)
{
	/* No need to seek to the next object if the current object
	 * is a file
	 */
	if(!TH_ISREG(tar))
	    return(0);

	/* Seek past this file */
	if(tar_skip_regfile(tar) != 0)
	    return(-1);

	return(0);
}
#endif  /* HAVE_LIBTAR */


/*
 *	Extract object from a Tape Archive.
 *
 *	Inputs assumed valid.
 */
gint EDVArchExtractTapeArchive(
	edv_core_struct *core,
	const gchar *arch_path,
	GList *objs_list, const gint nobjs,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps,
	const gboolean is_compress_compressed,
	const gboolean is_gzip_compressed,
	const gboolean is_bzip2_compressed
)
{
#ifdef HAVE_LIBTAR
#ifdef HAVE_LIBZ
	tartype_t tar_io_z_cb = {
	    (openfunc_t)EDVArchExtractLibTarOpenLibZCB,
	    (closefunc_t)gzclose,
	    (readfunc_t)gzread,
	    (writefunc_t)gzwrite
	};
#endif
#ifdef HAVE_LIBBZ2
	tartype_t tar_io_bz2_cb = {
	    (openfunc_t)EDVArchExtractLibTarOpenLibBZ2CB,
	    (closefunc_t)EDVArchExtractLibTarCloseLibBZ2CB,
	    (readfunc_t)EDVArchExtractLibTarReadLibBZ2CB,
	    (writefunc_t)EDVArchExtractLibTarWriteLibBZ2CB
	};
#endif
	tartype_t *tar_io_cbs;
	TAR *tar = NULL;
	gint status;
	gulong cur_size, total_size, cur_time;

	/* Select the IO callbacks based on the compression format */
	if(is_compress_compressed)
	{
	    tar_io_cbs = NULL;
	}
	else if(is_gzip_compressed)
	{
#ifdef HAVE_LIBZ
	    tar_io_cbs = &tar_io_z_cb;
#else
	    tar_io_cbs = NULL;
#endif
	}
	else if(is_bzip2_compressed)
	{
#ifdef HAVE_LIBBZ2
	    tar_io_cbs = &tar_io_bz2_cb;
#else
	    tar_io_cbs = NULL;
#endif
	}
	else
	{
	    tar_io_cbs = NULL;
	}

	/* Open the Tape Archive for reading */
	if(tar_open(
	    &tar,
	    (char *)arch_path,
	    tar_io_cbs,
	    O_RDONLY, 0,
	    TAR_GNU | TAR_NOOVERWRITE
	) == -1)
	{
	    const gint error_code = (gint)errno;
	    gchar *msg = g_strdup_printf(
"Unable to open the Tape Archive for reading:\n\
\n\
    %s\n\
\n\
%s.",
		arch_path, g_strerror(error_code)
	    );
	    EDVArchExtractCopyErrorMessage(core, msg);
	    g_free(msg);
	    return(-1);
	}
	if(tar == NULL)
	{
	    const gint error_code = (gint)errno;
	    gchar *msg = g_strdup_printf(
"Unable to open the Tape Archive for reading:\n\
\n\
    %s\n\
\n\
%s.",
		arch_path, g_strerror(error_code)
	    );
	    EDVArchExtractCopyErrorMessage(core, msg);
	    g_free(msg);
	    return(-1);
	}

	/* Calculate the total size of all the objects */
	total_size = 0l;
	if(objs_list != NULL)
	{
	    GList *glist;
	    edv_archive_object_struct *obj;
	    for(glist = objs_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		obj = EDV_ARCHIVE_OBJECT(glist->data);
		if(obj == NULL)
		    continue;

		total_size += obj->size;
	    }
	}

	/* Iterate through the list of objects to extract and
	 * extract each one
	 */
	status = 0;
	cur_size = 0l;
	cur_time = (gulong)time(NULL);
	if(TRUE)
	{
	    gint i;
	    gchar *parent, *tar_path, *src_path;
	    edv_object_type type;

	    /* Begin reading each object in the Tape Archive */
	    while((i = th_read(tar)) == 0)
	    {
		/* Get the path of this object in the archive */
		src_path = EDVArchExtractLibTarGetPath(tar);
		if(src_path == NULL)
		{
		    /* Unable to get the path of this object in the
		     * archive, try to seek to the next object in
		     * the archive
		     */
		    if(EDVArchExtractLibTarNext(tar))
		    {
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"An error occured while reading from the Tape Archive:\n\
\n\
    %s\n\
\n\
%s.",
			    arch_path,
			    g_strerror(error_code)
			);
			EDVArchExtractCopyErrorMessage(core, msg);
			g_free(msg);
			status = -1;
			return(status);
		    }
		    continue;
		}

		/* If not extracting all the objects then check if
		 * this object is in the list of objects to extract
		 */
		if(!extract_all)
		{
		    GList *glist;
		    edv_archive_object_struct *obj;

		    for(glist = objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			obj = EDV_ARCHIVE_OBJECT(glist->data);
			if(obj == NULL)
			    continue;

			if(obj->full_path == NULL)
			    continue;

			/* Should this object be extracted? */
			if(!strcmp(src_path, obj->full_path))
			    break;
		    }
		    /* Do not extract this object? */
		    if(glist == NULL)
		    {
			/* Seek to the next object in the archive */
			if(EDVArchExtractLibTarNext(tar))
			{
			    const gint error_code = (gint)errno;
			    gchar *msg = g_strdup_printf(
"An error occured while reading the object:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
				src_path,
				arch_path,
				g_strerror(error_code)
			    );
			    EDVArchExtractCopyErrorMessage(core, msg);
			    g_free(msg);
			    g_free(src_path);
			    status = -1;
			    return(status);
			}
			g_free(src_path);
			continue;
		    }
		}

		/* Format the path of the object to extract */
		if(preserve_directories)
		    tar_path = STRDUP(PrefixPaths(
			(const char *)dest_path,
			(const char *)src_path
		    ));
		else
		    tar_path = STRDUP(PrefixPaths(
			(const char *)dest_path,
			(const char *)g_basename(src_path)
		    ));
		if(STRISEMPTY(tar_path))
		{
		    g_free(src_path);
		    g_free(tar_path);
		    core->archive_last_error =
"Unable to generate the extracted object's path.";
		    status = -1;
		    break;
		}

		EDVSimplifyPath(tar_path);

		/* Get this object's type */
		type = EDVArchExtractLibTarGetType(tar);

		/* Update the progress dialog message to display
		 * the current object being extracted
		 */
		if(show_progress)
		{
		    gchar	*p1 = EDVShortenPath(
			src_path,
			EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
		    ),
				*p2 = EDVShortenPath(
			tar_path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
		    ),
				*msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Extraer:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Extraire:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Extrahieren:\n\
\n\
    %s\n\
\n\
Zu:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Estrarre:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Onttrekken:\n\
\n\
    %s\n\
\n\
Te:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Extrair:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Trekking Ut:\n\
\n\
    %s\n\
\n\
Til:\n\
\n\
    %s\n"
#else
"Extracting:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n"
#endif
			, p1, p2
		    );
		    EDVArchExtractMapProgressDialog(
			msg,
			(total_size > 0l) ?
			    ((gfloat)cur_size / (gfloat)total_size) : -1.0f,
			toplevel, FALSE
		    );
		}

#define QUERY_CONTINUE_EXTRACT	{			\
							\
 /* Need to query the user? */				\
 if(interactive && !(*yes_to_all)) {			\
  gint response;					\
  gchar *msg = g_strdup_printf(				\
"An error occured while extracting the object:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
Continue extracting subsequent objects?",		\
   src_path,						\
   tar_path						\
  );							\
  EDVPlaySoundError(core);				\
  CDialogSetTransientFor(toplevel);			\
  response = CDialogGetResponse(			\
   "Extract Failed",					\
   msg,							\
   NULL,						\
   CDIALOG_ICON_ERROR,					\
   CDIALOG_BTNFLAG_YES |				\
    CDIALOG_BTNFLAG_YES_TO_ALL |			\
   CDIALOG_BTNFLAG_NO,					\
   CDIALOG_BTNFLAG_YES					\
  );							\
  g_free(msg);						\
  CDialogSetTransientFor(NULL);				\
							\
  /* Stop extracting? */				\
  if((response == CDIALOG_RESPONSE_NO) ||		\
     (response == CDIALOG_RESPONSE_CANCEL) ||		\
     (response == CDIALOG_RESPONSE_NOT_AVAILABLE)	\
  )							\
  {							\
   g_free(src_path);					\
   g_free(tar_path);					\
   break;						\
  }							\
							\
  if(response == CDIALOG_RESPONSE_YES_TO_ALL)		\
   *yes_to_all = TRUE;					\
 }							\
							\
 g_free(src_path);					\
 g_free(tar_path);					\
 continue;						\
}

		/* Need to create the parent directory(ies)? */
		parent = g_dirname(tar_path);
		if(parent != NULL)
		{
		    struct stat lstat_buf;
		    const gint	stat_status = (gint)lstat((const char *)parent, &lstat_buf),
				error_code = (gint)errno;
		    if((stat_status != 0) && (error_code == ENOENT))
		    {
			/* Create each parent directory(ies) and add
			 * them to the list of extracted objects
			 */
			if(EDVArchExtractRMkDir(parent, new_paths_list_rtn))
			{
			    const gint error_code = (gint)errno;
			    gchar *msg = g_strdup_printf(
"Unable to create the parent directories to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
				tar_path,
				arch_path,
				g_strerror(error_code)
			    );
			    EDVArchExtractCopyErrorMessage(core, msg);
			    g_free(msg);
			    g_free(parent);
			    g_free(src_path);
			    g_free(tar_path);
			    status = -1;
			    break;
			}
		    }
		    g_free(parent);
		}

		/* Report the progress? */
		if(show_progress && ProgressDialogIsQuery())
		{
		    ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			(total_size > 0l) ?
			    ((gfloat)cur_size / (gfloat)total_size) : -1.0f,
			EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		    );
		    if(ProgressDialogStopCount() > 0)
		    {
			g_free(src_path);
			g_free(tar_path);
			status = -4;
			break;
		    }
		}

		/* Count and report this object as being extracted
		 *
		 * (Actual extraction will occure below)
		 */
		cur_size += (gulong)th_get_size(tar);

		if(new_paths_list_rtn != NULL)
		    *new_paths_list_rtn = g_list_append(
			*new_paths_list_rtn, STRDUP(tar_path)
		    );

		/* Remove the target object in case it already exists */
		if(EDVArchExtractRemove(tar_path))
		{
		    const gint error_code = (gint)errno;
		    gchar *msg = g_strdup_printf(
"Unable to remove the existing object:\n\
\n\
    %s\n\
\n\
To extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
			tar_path,
			src_path,
			arch_path,
			g_strerror(error_code)
		    );
		    EDVArchExtractCopyErrorMessage(core, msg);
		    g_free(msg);
		    status = -1;
		    QUERY_CONTINUE_EXTRACT
		}

		/* Extract this object */
		if(tar_extract_file(
		    tar,
		    tar_path
		) != 0)
		{
		    /* Extract error */
		    const gint error_code = (gint)errno;
		    gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
			tar_path,
			arch_path,
			g_strerror(error_code)
		    );
		    EDVArchExtractCopyErrorMessage(core, msg);
		    g_free(msg);
		    status = -1;
		    QUERY_CONTINUE_EXTRACT
		}

		/* Report the progress? */
		if(show_progress && ProgressDialogIsQuery())
		{
		    ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			(total_size > 0l) ?
			    ((gfloat)cur_size / (gfloat)total_size) : -1.0f,
			EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		    );
		    if(ProgressDialogStopCount() > 0)
		    {
			g_free(src_path);
			g_free(tar_path);
			status = -4;
			break;
		    }
		}

		/* Do not preserve timestamps? */
		if(!preserve_timestamps &&
		   (type != EDV_OBJECT_TYPE_LINK)
		)
		{
		    /* Set the current time for this object */
		    struct utimbuf utime_buf;
		    utime_buf.actime = (time_t)cur_time;
		    utime_buf.modtime = (time_t)cur_time;
		    utime((const char *)tar_path, &utime_buf);
		}

		/* Report the final progress for this object? */
		if(show_progress && ProgressDialogIsQuery())
		{
		    ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			(total_size > 0l) ?
			    ((gfloat)cur_size / (gfloat)total_size) : -1.0f,
			EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		    );
		    if(ProgressDialogStopCount() > 0)
		    {
			g_free(src_path);
			g_free(tar_path);
			status = -4;
			break;
		    }
		}

		g_free(tar_path);
		g_free(src_path);

#undef QUERY_CONTINUE_EXTRACT
	    }
	}

	/* Close the Tape Archive */
	if(tar_close(tar) != 0)
	{
	    if((status == 0) || (status == -4))
	    {
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"Unable to close the Tape Archive:\n\
\n\
    %s\n\
\n\
%s.",
		    arch_path, g_strerror(error_code)
		);
		EDVArchExtractCopyErrorMessage(core, msg);
		g_free(msg);
		status = -1;
	    }
	}

	return(status);
#else
	const gchar *prog_tar = CFGItemListGetValueS(
	    core->cfg_list, EDV_CFG_PARM_PROG_TAR
	);
	const gchar *prog_bunzip2 = CFGItemListGetValueS(
	    core->cfg_list, EDV_CFG_PARM_PROG_BUNZIP2
	);
	FILE *fp;
	gint status, p, nobjs_extracted;
	gchar	*cmd = NULL,
		*pwd = NULL,
		*stdout_path = NULL,
		*stderr_path = NULL;
	const gchar *src_path;
	GList *glist;
	edv_archive_object_struct *obj;

#define CLEANUP_RETURN(_v_)	{		\
 g_free(cmd);					\
 g_free(stdout_path);				\
 g_free(stderr_path);				\
						\
 /* Restore the previous working dir */		\
 if(pwd != NULL) {				\
  EDVSetCWD(pwd);				\
  g_free(pwd);					\
 }						\
						\
 return(_v_);					\
}

	/* Record previous working dir and set new working dir */
	pwd = EDVGetCWD();
	if(EDVSetCWD(dest_path))
	{
	    core->archive_last_error =
"Unable to change working directory to the destination directory.";
	    CLEANUP_RETURN(-1);
	}

	/* Format extract object from archive command */
	if(is_compress_compressed)
	    cmd = g_strdup_printf(
		"\"%s\" -Z -x%s -v -f \"%s\"",
		prog_tar,
		preserve_timestamps ? "" : " -m",
		arch_path
	    );
	else if(is_gzip_compressed)
	    cmd = g_strdup_printf(
		"\"%s\" -z -x%s -v -f \"%s\"",
		prog_tar,
		preserve_timestamps ? "" : " -m",
		arch_path
	    );
	else if(is_bzip2_compressed)
	    cmd = g_strdup_printf(
		"\"%s\" \"--use-compress-program=%s\" -x%s -v -f \"%s\"",
		prog_tar,
		prog_bunzip2,
		preserve_timestamps ? "" : " -m",
		arch_path
	    );
	else
	    cmd = g_strdup_printf(
		"\"%s\" -x%s -v -f \"%s\"",
		prog_tar,
		preserve_timestamps ? "" : " -m",
		arch_path
	    );
	if(cmd == NULL)
	{
	    core->archive_last_error = "Unable to generate the extract command.";
	    CLEANUP_RETURN(-1);
	}
	/* Append the objects to extract to the command string
	 * only if not extracting all the objects
	 */
	if(!extract_all)
	{
	    gchar *s;

	    for(glist = objs_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		obj = EDV_ARCHIVE_OBJECT(glist->data);
		if(obj == NULL)
		    continue;

		src_path = obj->full_path;
		if(STRISEMPTY(src_path))
		    continue;

		if(obj->type == EDV_OBJECT_TYPE_DIRECTORY)
		{
		    const gint len = STRLEN(src_path);

		    s = g_strconcat(cmd, " \"", src_path, NULL);
		    g_free(cmd);
		    cmd = s;

		    /* If directory does not have a tailing
		     * deliminator then we must append one or else
		     * it will not get matched
		     */
		    if(len > 1)
		    {
			if(src_path[len - 1] != G_DIR_SEPARATOR)
			{
			    s = g_strconcat(cmd, G_DIR_SEPARATOR_S, NULL);
			    g_free(cmd);
			    cmd = s;
			}
		    }
		    s = g_strconcat(cmd, "\"", NULL);
		    g_free(cmd);
		    cmd = s;
		}
		else
		{
		    s = g_strconcat(cmd, " \"", src_path, "\"", NULL);
		    g_free(cmd);
		    cmd = s;
		}
	    }
	}
	if(cmd == NULL)
	{
	    core->archive_last_error = "Unable to generate the extract command.";
	    CLEANUP_RETURN(-1);
	}


	/* Generate the output file paths */
	stdout_path = EDVTmpName(NULL);
	stderr_path = EDVTmpName(NULL);

	/* Begin extracting */

	status = 0;
	nobjs_extracted = 0;

	/* Execute the extract objects from archive command */
	p = (gint)ExecOE(
	    (const char *)cmd,
	    (const char *)stdout_path,
	    (const char *)stderr_path
	);
	if(p <= 0)
	{
	    core->archive_last_error = "Unable to execute the extract command.";
	    CLEANUP_RETURN(-1);
	}

	g_free(cmd);
	cmd = NULL;

	/* Open the output file for reading */
	fp = fopen((const char *)stdout_path, "rb");
	if(fp != NULL)
	{
	    gboolean need_break = FALSE;
	    gint	buf_pos = 0,
			line_count = 0;
	    gfloat  progress = 0.0f;
	    gchar buf[10000];

	    /* Begin reading the output file */
	    while(TRUE)
	    {
		/* Update progress? */
		if(show_progress && ProgressDialogIsQuery())
		{
		    ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		    );
		    if(ProgressDialogStopCount() > 0)
		    {
			INTERRUPT(p);
			p = 0;
			status = -4;
			break;
		    }
		}

		/* Check if there is new data to be read from the output
		 * file
		 */
		if(FPending(fp))
		{
		    gint c;
		    gboolean got_complete_line = FALSE;

		    /* Copy all available data from the current output
		     * file position to its end to the line buffer buf
		     */
		    while(TRUE)
		    {
			c = (gint)fgetc(fp);
			if((int)c == EOF)
			{
			    clearerr(fp);
			    break;
			}

			if(ISCR(c))
			{
			    got_complete_line = TRUE;
			    line_count++;

			    if(buf_pos < sizeof(buf))
				buf[buf_pos] = '\0';
			    else
				buf[sizeof(buf) - 1] = '\0';
			    buf_pos = 0;

			    break;
			}

			if(buf_pos < sizeof(buf))
			{
			    buf[buf_pos] = c;
			    buf_pos++;
			}
		    }
		    if(got_complete_line)
		    {
			gchar *extracted_path = STRDUP(PrefixPaths(dest_path, buf));
			StripPath(extracted_path);

			/* Append this path to the list of paths
			 * extracted from the archive
			 */
			if(new_paths_list_rtn != NULL)
			    *new_paths_list_rtn = g_list_append(
				*new_paths_list_rtn, STRDUP(extracted_path)
			    );

			/* Update the progress dialog's label? */
			if(show_progress && !STRISEMPTY(extracted_path))
			{
			    gchar	*p1 = EDVShortenPath(
				buf, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
			    ),
					*p2 = EDVShortenPath(
				extracted_path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
			    ),
					*msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Extraer:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Extraire:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Extrahieren:\n\
\n\
    %s\n\
\n\
Zu:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Estrarre:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Onttrekken:\n\
\n\
    %s\n\
\n\
Te:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Extrair:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Trekking Ut:\n\
\n\
    %s\n\
\n\
Til:\n\
\n\
    %s\n"
#else
"Extracting:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n"
#endif
				, p1, p2
			    );
			    EDVArchExtractMapProgressDialog(
				msg, progress, toplevel, FALSE
			    );
			    g_free(msg);
			    g_free(p1);
			    g_free(p2);
			}

			nobjs_extracted++;
			progress = (nobjs > 0) ?
			    ((gfloat)nobjs_extracted / (gfloat)nobjs) : 0.0f;

			g_free(extracted_path);

			continue;
		    }
		}

		if(need_break)
		    break;

		/* Check if the extract process has exited, if it has
		 * then we set need_break to TRUE. Which will be
		 * tested on the next loop if there is still no more
		 * data to be read
		 */
		if(!ExecProcessExists(p))
		    need_break = TRUE;

		usleep(8000);
	    }

	    fclose(fp);
	}

	/* Remove the output files */
	UNLINK(stdout_path);
	UNLINK(stderr_path);

	/* Report the final progress? */
	if(show_progress && (status == 0) &&
	   ProgressDialogIsQuery()
	)
	{
	    ProgressDialogUpdate(
		NULL, NULL, NULL, NULL,
		1.0f, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
		status = -4;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
#endif	/* !HAVE_LIBTAR */
}
