#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <unistd.h>

#include "../include/disk.h"
#include "guiutils.h"
#include "conmsg.h"

#include "images/icon_info_32x32.xpm"
#include "images/icon_warning_32x32.xpm"

#include "images/icon_ok_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_clear_20x20.xpm"


typedef struct _ConMsgDlg		ConMsgDlg;
#define CON_MSG_DLG(p)			((ConMsgDlg *)(p))
typedef struct _ConMsgCore		ConMsgCore;
#define CON_MSG_CORE(p)			((ConMsgCore *)(p))


/* Callbacks */
static gint ConMsgDlgEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static gint ConMsgDlgTextEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void ConMsgDlgOKCB(GtkWidget *widget, gpointer data);
static void ConMsgDlgNoMoreMessagesCB(GtkWidget *widget, gpointer data);
static void ConMsgDlgClearCB(GtkWidget *widget, gpointer data);

/* Console Message Dialog */
static ConMsgDlg *ConMsgDlgNew(
	ConMsgCore *core,
	const gchar *title, guint8 **icon_data,
	const gchar *font_name,
	const gint columns,
	const gint lines
);
static void ConMsgDlgMsgAppend(ConMsgDlg *d, const gchar *s);
static void ConMsgDlgUpdateWidgets(ConMsgDlg *d);
static void ConMsgDlgDelete(ConMsgDlg *d);

static gboolean ConMsgGetMsg(
	ConMsgCore *core,
	const gint fd, ConMsgDlg **d_ptr,
	const gchar *title, guint8 **icon_data
);
static gint ConMsgCheckTimeoutCB(gpointer data);
static gint ConMsgBuffersReset(void);

gint ConMsgInit(
	const gchar *name,	/* Program name */
	const gchar *font_name,
	const gint columns,	/* Can be 0 to indicate use env var $COLUMNS */
	const gint lines,	/* Can be 0 to indicate use env var $LINES */
	const gboolean show_stdout,
	const gboolean show_stderr
);
gint ConMsgReset(
	const gchar *name,	/* Program name */
	const gchar *font_name,
	const gint columns,	/* Can be 0 to indicate use env var $COLUMNS */
	const gint lines,	/* Can be 0 to indicate use env var $LINES */
	const gboolean show_stdout,
	const gboolean show_stderr
);
void ConMsgShutdown(void);

void ConMsgStdOutDlgMap(void);
void ConMsgStdErrDlgMap(void);


/*
 *	Message Dialog:
 */
struct _ConMsgDlg {

	GtkWidget	*toplevel;
	GtkAccelGroup	*accelgrp;
	gint		freeze_count;
	GdkFont		*font;
	ConMsgCore	*core;

	GtkWidget	*text,
			*ok_btn,
			*no_more_messages_btn,
			*clear_btn;

};

/*
 *	Core:
 */
struct _ConMsgCore {

	gchar		*prog_name,
			*out_dlg_title,
			*err_dlg_title,
			*font_name;
	gint		terminal_columns,
			terminal_lines;
	gboolean	out_show,
			err_show;
	gint		out_rfd,
			out_wfd,
			err_rfd,
			err_wfd;
	guint		msg_check_toid;
	ConMsgDlg	*out_dlg,
			*err_dlg;

};

static ConMsgCore	*con_msg_core = NULL;


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)

#define UNLINK(p)	(STRISEMPTY(p) ? -1 : (gint)unlink((const char *)(p)))
#define CLOSE(i)	(((i) > -1) ? (gint)close((int)i) : -1)


/*
 *	Toplevel GtkWindow event signal callback.
 */
static gint ConMsgDlgEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	ConMsgDlg *d = CON_MSG_DLG(data);
	if((widget == NULL) || (event == NULL) || (d == NULL))
	    return(status);

	if(d->freeze_count > 0)
	    return(status);

	switch((gint)event->type)
	{
	  case GDK_DELETE:
	    ConMsgDlgOKCB(widget, data);
	    status = TRUE;
	    break;
	}

	return(status);
}

/*
 *	GtkText event signal callback.
 */
static gint ConMsgDlgTextEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	GdkEventButton *button;
	GtkText *text;
	ConMsgDlg *d = CON_MSG_DLG(data);
	if((widget == NULL) || (event == NULL) || (d == NULL))
	    return(status);

	if(d->freeze_count > 0)
	    return(status);

	text = GTK_TEXT(widget);

	switch((gint)event->type)
	{
	  case GDK_BUTTON_PRESS:
	    button = (GdkEventButton *)event;
	    switch(button->button)
	    {
	      case GDK_BUTTON4:
		/* Scroll up */
		if(text->vadj != NULL)
		{
		    GtkAdjustment *adj = text->vadj;
		    const gfloat inc = MAX(
			(0.25f * adj->page_size),
			adj->step_increment  
		    );
		    gfloat v = adj->value - inc;
		    if(v > (adj->upper - adj->page_size))
			v = adj->upper - adj->page_size;
		    if(v < adj->lower)
			v = adj->lower;
		    gtk_adjustment_set_value(adj, v);
		}     
		/* Need to mark the GtkText button as 0 or else it will
		 * keep marking
		 */
		text->button = 0;
		gtk_grab_remove(widget);
		gtk_signal_emit_stop_by_name(
		    GTK_OBJECT(widget), "button_press_event"
		);
		status = TRUE;
		break;

	      case GDK_BUTTON5:
		/* Scroll down */
		if(text->vadj != NULL)
		{
		    GtkAdjustment *adj = text->vadj;
		    const gfloat inc = MAX(
			(0.25f * adj->page_size),
			adj->step_increment
		    );
		    gfloat v = adj->value + inc;
		    if(v > (adj->upper - adj->page_size))
			v = adj->upper - adj->page_size;
		    if(v < adj->lower)
			v = adj->lower;
		    gtk_adjustment_set_value(adj, v);
		}
		/* Need to mark the GtkText button as 0 or else it will
		 * keep marking
		 */
		text->button = 0;
		gtk_grab_remove(widget);
		gtk_signal_emit_stop_by_name(
		    GTK_OBJECT(widget), "button_press_event"
		);
		status = TRUE;
		break;
	    }
	    break;

	  case GDK_BUTTON_RELEASE:
	    button = (GdkEventButton *)event;
	    switch(button->button)
	    {
	      case GDK_BUTTON4:
		/* Need to mark the GtkText button as 0 or else it will
		 * keep marking
		 */
		text->button = 0;
		gtk_grab_remove(widget);
		gtk_signal_emit_stop_by_name(
		    GTK_OBJECT(widget), "button_release_event"
		);
		status = TRUE;
		break;
	      case GDK_BUTTON5:
		/* Need to mark the GtkText button as 0 or else it will
		 * keep marking
		 */
		text->button = 0;
		gtk_grab_remove(widget);
		gtk_signal_emit_stop_by_name(
		    GTK_OBJECT(widget), "button_release_event"
		);
		status = TRUE;
		break;
	    }
	    break;
	}

	return(status);
}

/*
 *	OK callback.
 */
static void ConMsgDlgOKCB(GtkWidget *widget, gpointer data)
{
	ConMsgCore *core;
	ConMsgDlg *d = CON_MSG_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	core = d->core;

	/* Reset the global variable that references this Message
	 * Dialog and delete this Message Dialog
	 */
	if(d == core->out_dlg)
	    core->out_dlg = NULL;
	else if(d == core->err_dlg)
	    core->err_dlg = NULL;

	ConMsgDlgDelete(d);
}

/*
 *	No More Messages callback.
 */
static void ConMsgDlgNoMoreMessagesCB(GtkWidget *widget, gpointer data)
{                          
	ConMsgCore *core;
	ConMsgDlg *d = CON_MSG_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	core = d->core;

	/* Reset the global variable that references this Message Dialog
	 * and delete this Message Dialog
	 */
	if(d == core->out_dlg)
	{
	    core->out_show = FALSE;
	    core->out_dlg = NULL;
	}
	else if(d == core->err_dlg)
	{
	    core->err_show = FALSE;
	    core->err_dlg = NULL;
	}

	ConMsgDlgDelete(d);
}

/*
 *	Clear callback.
 */
static void ConMsgDlgClearCB(GtkWidget *widget, gpointer data)
{
	GtkText *text;
	ConMsgDlg *d = CON_MSG_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	text = GTK_TEXT(d->text);

	d->freeze_count++;

	while(gtk_events_pending() > 0)
	    gtk_main_iteration();

	gtk_text_freeze(text);
	gtk_text_set_point(text, 0);
	gtk_text_forward_delete(
	    text,
	    gtk_text_get_length(text)
	);
	gtk_text_thaw(text);

	while(gtk_events_pending() > 0)
	    gtk_main_iteration();

	ConMsgDlgUpdateWidgets(d);

	d->freeze_count--;
}


/*
 *	Creates a new Message Dialog.
 */
static ConMsgDlg *ConMsgDlgNew(
	ConMsgCore *core,
	const gchar *title, guint8 **icon_data,
	const gchar *font_name,
	const gint columns,
	const gint lines
)
{
	const gint	border_major = 5,
			border_minor = 2;
	gint		_columns = columns,
			_lines = lines,
			font_width, font_height;
	GdkFont *font;
	GdkWindow *window;
	GtkAccelGroup *accelgrp;
	GtkWidget *w, *parent, *parent2, *toplevel, *sb;
	GtkText *text;
	ConMsgDlg *d;

	if(core == NULL)
	    return(NULL);

	if(_columns <= 0)
	    _columns = 80;
	if(_lines <= 0)
	    _lines = 20;

	/* Get the default font (must be a fixed-width font) and
	 * get its size
	 */
	if(STRISEMPTY(font_name))
	    font = gdk_font_load(
"-adobe-courier-medium-r-normal-*-12-*-*-*-m-*-iso8859-1"
	    );
	else
	    font = gdk_font_load(font_name);
	if(font == NULL)
	    font = gdk_font_load("6x12");
	if(font != NULL)
	{
	    GdkTextBounds b;
	    gdk_text_bounds(font, "X", 1, &b);
	    font_width = b.width;
	    font_height = font->ascent + font->descent;
	}
	else
	{
	    font_width = 6;
	    font_height = 12;
	}

	/* Create the Message Dialog */
	d = CON_MSG_DLG(g_malloc0(sizeof(ConMsgDlg))); 
	if(d == NULL)  
	    return(NULL);

	d->toplevel = toplevel = gtk_window_new(GTK_WINDOW_DIALOG);
	d->accelgrp = accelgrp = gtk_accel_group_new();
	d->freeze_count = 0;
	d->font = font;
	d->core = core;

	d->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_widget_set_name(w, CON_MSG_TOPLEVEL_WIDGET_NAME);
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, TRUE);
	gtk_window_set_position(GTK_WINDOW(w), GTK_WIN_POS_MOUSE);
	if(title != NULL)
	    gtk_window_set_title(GTK_WINDOW(w), title);
	if(!STRISEMPTY(core->prog_name))
	    gtk_window_set_wmclass(
		GTK_WINDOW(w), "dialog", core->prog_name
	    );
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_MENU | GDK_DECOR_TITLE |
		GDK_DECOR_MINIMIZE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
	    );
	    GUISetWMIcon(window, icon_data);
	}
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(ConMsgDlgEventCB), d
	);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	parent = w;

	/* Main GtkVBox */
	w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;

	/* Text GtkTable */
	w = gtk_table_new(2, 2, FALSE);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_table_set_row_spacing(GTK_TABLE(w), 0, border_minor);
	gtk_table_set_col_spacing(GTK_TABLE(w), 0, border_minor);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;
	/* Text */
	d->text = w = gtk_text_new(NULL, NULL);
	text = GTK_TEXT(w);
	gtk_widget_set_name(w, CON_MSG_TEXT_WIDGET_NAME);
	gtk_widget_set_usize(
	    w,
	    MAX(
		(font_width * _columns) + (2 * border_major),
		300
	    ),
	    MAX(
		(font_height * _lines) + (2 * border_major),
		100
	    )
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(ConMsgDlgTextEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(ConMsgDlgTextEventCB), d
	);
	gtk_text_set_editable(text, FALSE);
	gtk_text_set_word_wrap(text, TRUE);
	gtk_table_attach(
	    GTK_TABLE(parent2), w,
	    0, 1, 0, 1,
	    GTK_EXPAND | GTK_SHRINK | GTK_FILL,
	    GTK_EXPAND | GTK_SHRINK | GTK_FILL,
	    0, 0
	);
	GUIEditableEndowPopupMenu(w, GUI_EDITABLE_POPUP_MENU_READ_ONLY);
	gtk_widget_show(w);
	/* Vertical GtkScrollBar */
	sb = gtk_vscrollbar_new(GTK_TEXT(w)->vadj);
	gtk_table_attach(   
	    GTK_TABLE(parent2), sb,
	    1, 2, 0, 1,
	    GTK_FILL,
	    GTK_EXPAND | GTK_SHRINK | GTK_FILL,
	    0, 0
	);
	gtk_widget_show(sb);


	/* GtkSeparator */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Buttons GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* OK button */
	d->ok_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_ok_20x20_xpm, "OK", NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(ConMsgDlgOKCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_o, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_o);
	gtk_widget_show(w);

	/* No More Messages button */
	d->no_more_messages_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_cancel_20x20_xpm, "No More Messages", NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    -1, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(ConMsgDlgNoMoreMessagesCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_n, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_n);
	gtk_widget_show(w);

	/* Clear button */
	d->clear_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_clear_20x20_xpm, "Clear", NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(  
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(ConMsgDlgClearCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_l, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_l);
	gtk_widget_show(w);


	/* Map the dialog */
	gtk_widget_show_raise(d->toplevel);

	/* Have the OK button grab focus */
	gtk_widget_grab_focus(d->ok_btn);
	gtk_widget_grab_default(d->ok_btn);

	d->freeze_count--;

	return(d);
}

/*
 *	Appends the message to the Message Dialog.
 */
static void ConMsgDlgMsgAppend(ConMsgDlg *d, const gchar *s)
{
	GtkAdjustment *adj;
	GtkWidget *w;
	GtkEditable *editable;
	GtkText *text;

	if((d == NULL) || (s == NULL))
	    return;

	w = d->text;
	editable = GTK_EDITABLE(w);
	text = GTK_TEXT(w);

	gtk_text_freeze(text);
	gtk_text_insert(
	    text, d->font, NULL, NULL, s, -1
	);
	gtk_text_thaw(text);

	adj = text->vadj;
	if(adj != NULL)
	    gtk_adjustment_set_value(
		adj,
		MAX(adj->upper - adj->page_size, adj->lower)
	    );
}

/*
 *	Updates the Message Dialog's widget values.
 */
static void ConMsgDlgUpdateWidgets(ConMsgDlg *d)
{
	GtkEditable *editable;
	GtkText *text;

	if(d == NULL)
	    return;

	editable = GTK_EDITABLE(d->text);
	text = GTK_TEXT(editable);

}

/*
 *	Deletes the Message Dialog.
 */
static void ConMsgDlgDelete(ConMsgDlg *d)
{
	if(d == NULL)
	    return;

	gtk_widget_hide(d->toplevel);

	d->freeze_count++;

	GTK_WIDGET_DESTROY(d->toplevel);
	GTK_ACCEL_GROUP_UNREF(d->accelgrp);
	GDK_FONT_UNREF(d->font);

	d->freeze_count--;

	g_free(d);
}


/*
 *	Checks the buffer descriptor for data and processes it.
 *
 *	If the pointer to the Message Dialog is not NULL then the
 *	message will be displayed on that Message Dialog (creating the
 *	Message Dialog as needed).
 *
 *	Returns TRUE if data was processed.
 */
static gboolean ConMsgGetMsg(
	ConMsgCore *core,
	const gint fd, ConMsgDlg **d_ptr,
	const gchar *title, guint8 **icon_data
)
{
	struct stat stat_buf;
	gint buf_len, bytes_read, total_bytes_read = 0;
	gchar *buf;
 
	/* Check if the specified descriptor is valid and get its
	 * statistics
	 */
	if((fd > -1) ? fstat(fd, &stat_buf) : TRUE)
	    return(FALSE);

	/* Get block size for optimul reading and allocate the read
	 * buffer
	 */
#if defined(_WIN32)
	buf_len = 1024;
#else
	buf_len = (gint)stat_buf.st_blksize;
#endif
	if(buf_len <= 0)
	    buf_len = 1024;
	buf = (gchar *)g_malloc(buf_len * sizeof(gchar));
	if(buf == NULL)
	    return(FALSE);

	/* Begin reading the message buffer */
	while(TRUE)
	{
	    /* Read data from the message buffer */
	    bytes_read = (gint)read((int)fd, buf, buf_len);
	    if(bytes_read <= 0)
		break;

	    /* Read data */
	    total_bytes_read += bytes_read;

	    /* Null terminate message buffer */
	    if(bytes_read < buf_len)
		buf[bytes_read] = '\0';
	    else
		buf[buf_len - 1] = '\0';

	    /* Create new Message Dialog as needed */
	    if(d_ptr != NULL)
	    {
		ConMsgDlg *d = *d_ptr;
		if(d == NULL)
		    *d_ptr = d = ConMsgDlgNew(
			core,
			title, icon_data,
			core->font_name,
			core->terminal_columns, core->terminal_lines
		    );
		ConMsgDlgMsgAppend(d, buf);
	    }
	}

	/* Delete read buffer */
	g_free(buf);

	/* Map and raise the Message Dialog if there was data read */
	if((total_bytes_read > 0) && (d_ptr != NULL))
	{
	    ConMsgDlg *d = *d_ptr;
	    if(d != NULL)
		gtk_widget_show_raise(d->toplevel);
	}

	/* Return TRUE if there was data read */
	return((total_bytes_read > 0) ? TRUE : FALSE);
}

/*
 *	Console message monitor timeout callback.
 */
static gint ConMsgCheckTimeoutCB(gpointer data)
{
	ConMsgCore *core = CON_MSG_CORE(data);
	if(core == NULL)
	    return(FALSE);

	/* Check for data on stdout and stderr, if there is data then
	 * read it and display it on the Message Dialogs
	 */
	ConMsgGetMsg(
	    core,
	    core->out_rfd,
	    core->out_show ? &core->out_dlg : NULL,
	    core->out_dlg_title,
	    (guint8 **)icon_info_32x32_xpm
	);
	ConMsgGetMsg(
	    core,
	    core->err_rfd,
	    core->err_show ? &core->err_dlg : NULL,
	    core->err_dlg_title,
	    (guint8 **)icon_warning_32x32_xpm
	);

	return(TRUE);
}

/*
 *	Recreates the message buffers.
 */
static gint ConMsgBuffersReset(void)
{
	ConMsgCore *core = con_msg_core;
	if(core == NULL)
	    return(-1);

	/* Close existing message buffers */
	CLOSE(core->out_rfd);
	core->out_rfd = -1;
	CLOSE(core->out_wfd);
	core->out_wfd = -1;
	CLOSE(core->err_rfd);
	core->err_rfd = -1;
	CLOSE(core->err_wfd);
	core->err_wfd = -1;

	/* Create new message buffers */
	if(core->out_show)
	{
	    gint fdpair[2];
	    if(!pipe(fdpair))
	    {
		core->out_rfd = fdpair[0];
		core->out_wfd = fdpair[1];
		dup2(core->out_wfd, fileno(stdout));
		fcntl(core->out_rfd, F_SETFL, O_NONBLOCK);
	    }
	}
	if(core->err_show)
	{
	    gint fdpair[2];
	    if(!pipe(fdpair))
	    {
		core->err_rfd = fdpair[0];
		core->err_wfd = fdpair[1];
		dup2(core->err_wfd, fileno(stderr));
		fcntl(core->err_rfd, F_SETFL, O_NONBLOCK);
	    }
	}

	return(0);
}


/*
 *	Initializes the Console Message Display System.
 */
gint ConMsgInit(
	const gchar *name,	/* Program name */
	const gchar *font_name,
	const gint columns,	/* Can be 0 to indicate use env var $COLUMNS */
	const gint lines,	/* Can be 0 to indicate use env var $LINES */
	const gboolean show_stdout,
	const gboolean show_stderr
)
{
	ConMsgCore *core;

	/* Already initialized? */
	if(con_msg_core != NULL)
	    return(0);

	/* Create a new Console Message Core */
	con_msg_core = core = CON_MSG_CORE(g_malloc0(sizeof(ConMsgCore)));
	if(core == NULL)
	    return(-3);

	core->prog_name = STRDUP(name);
	core->font_name = STRDUP(font_name);

	/* Set the dialog titles */
	if(STRISEMPTY(core->prog_name))
	    core->out_dlg_title = STRDUP("Message");
	else
	    core->out_dlg_title = g_strconcat(
		core->prog_name,
		" - Message",
		NULL
	    );

	if(STRISEMPTY(core->prog_name))
	    core->err_dlg_title = STRDUP("Error Message");
	else
	    core->err_dlg_title = g_strconcat(
		core->prog_name,
		" - Error Message",
		NULL
	    );

	/* Set the terminal geometry */
	if(columns > 0)
	{
	    core->terminal_columns = columns;
	}
	else
	{
	    const gchar *s = g_getenv("COLUMNS");
	    if(s != NULL)
		core->terminal_columns = ATOI(s);
	}
	if(lines > 0)
	{
	    core->terminal_lines = lines;
	}
	else
	{
	    const gchar *s = g_getenv("LINES");
	    if(s != NULL)
		core->terminal_lines = ATOI(s);
	}

	/* Set show stdout & stderr options */
	core->out_show = show_stdout;
	core->err_show = show_stderr;

	/* Reset the buffer descriptors */
	core->out_rfd = -1;
	core->out_wfd = -1;
	core->err_rfd = -1;
	core->err_wfd = -1;

	/* Reset the Message Dialogs */
	core->out_dlg = NULL;
	core->err_dlg = NULL;

	/* Create message buffers for the first time */
	ConMsgBuffersReset();

	/* Set the console message check timeout callback */
	core->msg_check_toid = gtk_timeout_add(
	    1000l,
	    ConMsgCheckTimeoutCB, core
	);

	return(0);
}

/*
 *	Resets the Console Message Display System.
 */
gint ConMsgReset(
	const gchar *name,	/* Program name */
	const gchar *font_name,
	const gint columns,	/* Can be 0 to indicate use env var $COLUMNS */
	const gint lines,	/* Can be 0 to indicate use env var $LINES */
	const gboolean show_stdout,
	const gboolean show_stderr
)
{
	ConMsgCore *core = con_msg_core;
	if(core == NULL)
	    return(-1);

	/* Set the program name */
	g_free(core->prog_name);
	core->prog_name = STRDUP(name);

	/* Set the font name */
	g_free(core->font_name);
	core->font_name = STRDUP(font_name);

	/* Set the dialog titles */
	g_free(core->out_dlg_title);
	if(STRISEMPTY(core->prog_name))
	    core->out_dlg_title = STRDUP("Message");
	else
	    core->out_dlg_title = g_strconcat(
		core->prog_name,
		" - Message",
		NULL
	    );
	g_free(core->err_dlg_title);
	if(STRISEMPTY(core->prog_name))
	    core->err_dlg_title = STRDUP("Error Message");
	else
	    core->err_dlg_title = g_strconcat(
		core->prog_name,
		" - Error Message",
		NULL
	    );

	/* Set terminal geometry */
	if(columns > 0)
	{
	    core->terminal_columns = columns;
	}
	else
	{   
	    const gchar *s = g_getenv("COLUMNS");
	    if(s != NULL)
		core->terminal_columns = ATOI(s);
	}
	if(lines > 0)
	{
	    core->terminal_lines = lines;
	}
	else
	{   
	    const gchar *s = g_getenv("LINES");
	    if(s != NULL)
		core->terminal_lines = ATOI(s);
	}

	/* Set the show stdout & stderr options */
	core->out_show = show_stdout;
	core->err_show = show_stderr;

	/* Delete the Message Dialogs so that they are created next
	 * time with the reset values
	 */
	ConMsgDlgDelete(core->out_dlg);
	core->out_dlg = NULL;
	ConMsgDlgDelete(core->err_dlg);
	core->err_dlg = NULL;

	/* Recreate the message buffers */
	ConMsgBuffersReset();

	/* Reset the console message check timeout callback */
	GTK_TIMEOUT_REMOVE(core->msg_check_toid);
	core->msg_check_toid = gtk_timeout_add(
	    1000l,
	    ConMsgCheckTimeoutCB, core
	);

	return(0);
}

/*
 *	Shuts down the Console Message Display System.
 */
void ConMsgShutdown(void)
{
	ConMsgCore *core = con_msg_core;
	if(core == NULL)
	    return;

	con_msg_core = NULL;

	/* Wait for all the Message Dialogs to be unmapped */
	while(gtk_events_pending() > 0)
	    gtk_main_iteration();
	while((core->out_dlg != NULL) || (core->err_dlg != NULL))
	{
	    if(gtk_events_pending() > 0)
		gtk_main_iteration();
	}

	/* Remove the console message check timeout callback */
	GTK_TIMEOUT_REMOVE(core->msg_check_toid);
	core->msg_check_toid = 0;

	/* Delete the Message Dialogs */
	ConMsgDlgDelete(core->out_dlg);
	core->out_dlg = NULL;
	ConMsgDlgDelete(core->err_dlg);
	core->err_dlg = NULL;

	/* Close the message buffers */
	CLOSE(core->out_rfd);
	CLOSE(core->out_wfd);
	CLOSE(core->err_rfd);
	CLOSE(core->err_wfd);

	g_free(core->prog_name);
	g_free(core->font_name);
	g_free(core->out_dlg_title);
	g_free(core->err_dlg_title);
	g_free(core);
}


/*
 *	Maps the StdOut Message Dialog.
 */
void ConMsgStdOutDlgMap(void)
{
	ConMsgDlg *d;
	ConMsgCore *core = con_msg_core;
	if(core == NULL)
	    return;

	d = core->out_dlg;
	if(d == NULL)
	    core->out_dlg = d = ConMsgDlgNew(
		core,
		core->out_dlg_title, (guint8 **)icon_info_32x32_xpm,
		core->font_name,
		core->terminal_columns, core->terminal_lines
	    );
	if(d == NULL)
	    return;

	gtk_widget_show_raise(d->toplevel);

	/* Have the OK button grab focus */
	gtk_widget_grab_focus(d->ok_btn);
	gtk_widget_grab_default(d->ok_btn);
}

/*
 *	Maps the StdErr Message Dialog.
 */
void ConMsgStdErrDlgMap(void)
{
	ConMsgDlg *d;
	ConMsgCore *core = con_msg_core;
	if(core == NULL)
	    return;

	d = core->err_dlg;
	if(d == NULL)          
	    core->err_dlg = d = ConMsgDlgNew(
		core,
		core->err_dlg_title, (guint8 **)icon_warning_32x32_xpm,
		core->font_name,
		core->terminal_columns, core->terminal_lines
	    );
	if(d == NULL)
	    return;

	gtk_widget_show_raise(d->toplevel);

	/* Have the OK button grab focus */
	gtk_widget_grab_focus(d->ok_btn);
	gtk_widget_grab_default(d->ok_btn);
}
