#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <gtk/gtk.h>
#include <unistd.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"
#include "pdialog.h"
#include "fb.h"
#include "progressdialog.h"
#include "toolbar.h"

#include "cfg.h"
#include "edv_types.h"
#include "edv_confirm.h"
#include "edv_obj.h"
#include "edv_archive_obj.h"
#include "edv_archive_stat.h"
#include "edv_archive_add.h"
#include "edv_archive_extract.h"
#include "edv_archive_delete.h"
#include "edv_archive_chk.h"
#include "edv_archive_fix.h"
#include "edv_find_bar.h"
#include "edv_status_bar.h"
#include "archive_options_dlg.h"
#include "archive_info_dlg.h"
#include "archiver.h"
#include "archiver_cb.h"
#include "archiver_op_cb.h"
#include "archiver_contents_list.h"
#include "endeavour2.h"
#include "edv_cb.h"
#include "edv_help.h"
#include "edv_op.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_wildcards_32x32.xpm"


void EDVArchiverOPCB(
	toolbar_item_struct *item, gint id, gpointer data  
);
void EDVArchiverOPEnterCB(
	toolbar_item_struct *item, gint id, gpointer data  
);
void EDVArchiverOPLeaveCB(
	toolbar_item_struct *item, gint id, gpointer data
);

void EDVArchiverOPSyncDisks(edv_archiver_struct *archiver);
void EDVArchiverOPWriteProtect(edv_archiver_struct *archiver);
void EDVArchiverOPDeleteMethodRecycle(edv_archiver_struct *archiver);
void EDVArchiverOPDeleteMethodPurge(edv_archiver_struct *archiver);

void EDVArchiverOPNew(edv_archiver_struct *archiver);
void EDVArchiverOPOpen(edv_archiver_struct *archiver);

void EDVArchiverOPClose(edv_archiver_struct *archiver);
void EDVArchiverOPExit(edv_archiver_struct *archiver);

static GList *EDVArchiverGetSelectedObjects(
	edv_archiver_struct *archiver
);
void EDVArchiverOPAdd(edv_archiver_struct *archiver);
void EDVArchiverOPExtract(edv_archiver_struct *archiver);
void EDVArchiverOPExtractAll(edv_archiver_struct *archiver);
void EDVArchiverOPDelete(edv_archiver_struct *archiver);
void EDVArchiverOPCheck(edv_archiver_struct *archiver);
void EDVArchiverOPFix(edv_archiver_struct *archiver);
void EDVArchiverOPSelectAll(edv_archiver_struct *archiver);
void EDVArchiverOPUnselectAll(edv_archiver_struct *archiver);
void EDVArchiverOPInvertSelection(edv_archiver_struct *archiver);
void EDVArchiverOPProperties(edv_archiver_struct *archiver);

void EDVArchiverOPRefresh(edv_archiver_struct *archiver);
void EDVArchiverOPRefreshAll(edv_archiver_struct *archiver);
void EDVArchiverOPCommentAndStatistics(edv_archiver_struct *archiver);

void EDVArchiverContentsFilter(edv_archiver_struct *archiver);
void EDVArchiverMIMETypes(edv_archiver_struct *archiver);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Operation ID callback nexus.
 *
 *	The data must be a edv_archiver_opid_struct *.
 */
void EDVArchiverOPCB(
	toolbar_item_struct *item, gint id, gpointer data  
)
{
	GtkWidget *toplevel;
	edv_archiver_struct *archiver;
	edv_core_struct *core;
	edv_archiver_opid_struct *opid = EDV_ARCHIVER_OPID(data);
	if(opid == NULL)
	    return;

	archiver = opid->archiver;
	if(archiver == NULL)
	    return;

	if(archiver->processing || (archiver->freeze_count > 0))
	    return;

	toplevel = archiver->toplevel;
	core = archiver->core;

	archiver->freeze_count++;

	/* Handle by operation id code */
	switch(opid->op)
	{
	  case EDV_ARCHIVER_OP_NONE:
	  case EDV_ARCHIVER_OP_SEPARATOR:
	    break;

	  case EDV_ARCHIVER_OP_NEW:
	    EDVArchiverOPNew(archiver);
	    break;

	  case EDV_ARCHIVER_OP_OPEN:
	    EDVArchiverOPOpen(archiver);
	    break;


	  case EDV_ARCHIVER_OP_CLOSE:
	    EDVArchiverOPClose(archiver);
	    break;

	  case EDV_ARCHIVER_OP_EXIT:
	    EDVArchiverOPExit(archiver);
	    break;


	  case EDV_ARCHIVER_OP_ADD:
	    EDVArchiverOPAdd(archiver);
	    break;

	  case EDV_ARCHIVER_OP_EXTRACT:
	    EDVArchiverOPExtract(archiver);
	    break;

	  case EDV_ARCHIVER_OP_EXTRACT_ALL:
	    EDVArchiverOPExtractAll(archiver);
	    break;

	  case EDV_ARCHIVER_OP_DELETE:
	    EDVArchiverOPDelete(archiver);
	    break;

	  case EDV_ARCHIVER_OP_CHECK:
	    EDVArchiverOPCheck(archiver);
	    break;

	  case EDV_ARCHIVER_OP_FIX:
	    EDVArchiverOPFix(archiver);
	    break;

	  case EDV_ARCHIVER_OP_SELECT_ALL:
	    EDVArchiverOPSelectAll(archiver);
	    break;

	  case EDV_ARCHIVER_OP_UNSELECT_ALL:
	    EDVArchiverOPUnselectAll(archiver);
	    break;

	  case EDV_ARCHIVER_OP_INVERT_SELECTION:
	    EDVArchiverOPInvertSelection(archiver);
	    break;

	  case EDV_ARCHIVER_OP_FIND:
	    EDVMapArchiverFindWin(core, archiver);
	    break;

	  case EDV_ARCHIVER_OP_PROPERTIES:
	    EDVArchiverOPProperties(archiver);
	    break;


	  case EDV_ARCHIVER_OP_HISTORY:
	    EDVMapHistoryListWin(core, toplevel);
	    break;

	  case EDV_ARCHIVER_OP_SYNC_DISKS:
	    EDVArchiverOPSyncDisks(archiver);
	    break;

	  case EDV_ARCHIVER_OP_WRITE_PROTECT:
	    EDVArchiverOPWriteProtect(archiver);
	    break;

	  case EDV_ARCHIVER_OP_DELETE_METHOD_RECYCLE:
	    EDVArchiverOPDeleteMethodRecycle(archiver);
	    break;

	  case EDV_ARCHIVER_OP_DELETE_METHOD_PURGE:
	    EDVArchiverOPDeleteMethodPurge(archiver);
	    break;

	  case EDV_ARCHIVER_OP_RUN:
	    EDVMapRunDialogCommand(
		core,
		NULL,
		NULL,
		toplevel
	    );
	    break;

	  case EDV_ARCHIVER_OP_RUN_TERMINAL:
	    if(core != NULL)
	    {
		gchar	*location = STRDUP(EDVArchiverCurrentLocation(archiver)),
			*wd = g_dirname(location);
		EDVRunTerminal(core, NULL, wd, toplevel);
		g_free(wd);
		g_free(location);
	    }
	    break;


	  case EDV_ARCHIVER_OP_REFRESH:
	    EDVArchiverOPRefresh(archiver);
	    break;

	  case EDV_ARCHIVER_OP_REFRESH_ALL:
	    EDVArchiverOPRefreshAll(archiver);
	    break;

	  case EDV_ARCHIVER_OP_COMMENT_AND_STATISTICS:
	    EDVArchiverOPCommentAndStatistics(archiver);
	    break;

	  case EDV_ARCHIVER_OP_SHOW_TOOL_BAR:
	    if(core != NULL)
	    {
		gboolean state = !CFGItemListGetValueI(
		    core->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_TOOL_BAR
		);
		CFGItemListSetValueI(
		    core->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_TOOL_BAR,
		    state, FALSE
		);
		EDVReconfiguredEmit(core);
	    }
	    break;

	  case EDV_ARCHIVER_OP_SHOW_LOCATION_BAR:
	    if(core != NULL)
	    {
		gboolean state = !CFGItemListGetValueI(
		    core->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_LOCATION_BAR
		);
		CFGItemListSetValueI(
		    core->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_LOCATION_BAR,
		    state, FALSE
		);
		EDVReconfiguredEmit(core);
	    }
	    break;

	  case EDV_ARCHIVER_OP_SHOW_FIND_BAR:
	    if(core != NULL)
	    {
		gboolean state = !CFGItemListGetValueI(
		    core->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_FIND_BAR
		);
		CFGItemListSetValueI(
		    core->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_FIND_BAR,
		    state, FALSE
		);
		EDVReconfiguredEmit(core);
	    }
	    break;

	  case EDV_ARCHIVER_OP_SHOW_STATUS_BAR:
	    if(core != NULL)
	    {
		gboolean state = !CFGItemListGetValueI(
		    core->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_STATUS_BAR
		);
		CFGItemListSetValueI(
		    core->cfg_list, EDV_CFG_PARM_ARCHIVER_SHOW_STATUS_BAR,
		    state, FALSE
		);
		EDVReconfiguredEmit(core);
	    }
	    break;


	  case EDV_ARCHIVER_OP_CONTENTS_LIST_FILTER:
	    EDVArchiverContentsFilter(archiver);
	    break;


	  case EDV_ARCHIVER_OP_MIME_TYPES:
	    EDVArchiverMIMETypes(archiver);
	    break;


	  case EDV_ARCHIVER_OP_NEW_BROWSER:
	    EDVNewBrowser(core, NULL);
	    break;

	  case EDV_ARCHIVER_OP_NEW_IMBR:
	    EDVNewImbr(core, NULL);
	    break;

	  case EDV_ARCHIVER_OP_NEW_ARCHIVER:
	    EDVNewArchiver(core, NULL, NULL);
	    break;

	  case EDV_ARCHIVER_OP_RECYCLE_BIN:
	    EDVMapRecBin(core);
	    break;


	  case EDV_ARCHIVER_OP_OPTIONS:
	    EDVMapOptionsWin(core, toplevel);
	    break;

	  case EDV_ARCHIVER_OP_CUSTOMIZE:
	    EDVMapCustomizeWin(core, toplevel);
	    break;


	  case EDV_ARCHIVER_OP_HELP_ABOUT:
	    EDVAbout(core, toplevel);
	    break;
	  case EDV_ARCHIVER_OP_HELP_CONTENTS:
	    EDVHelp(core, "Contents", toplevel);
	    break;
	  case EDV_ARCHIVER_OP_HELP_FILE_BROWSER:
	    EDVHelp(core, "File Browser", toplevel);
	    break;
	  case EDV_ARCHIVER_OP_HELP_IMAGE_BROWSER:
	    EDVHelp(core, "Image Browser", toplevel);
	    break;
	  case EDV_ARCHIVER_OP_HELP_ARCHIVER:
	    EDVHelp(core, "Archiver", toplevel);
	    break;
	  case EDV_ARCHIVER_OP_HELP_RECYCLE_BIN:
	    EDVHelp(core, "Recycle Bin", toplevel);
	    break;
	  case EDV_ARCHIVER_OP_HELP_KEYS_LIST:
	    EDVHelp(core, "Keys List", toplevel);
	    break;
	  case EDV_ARCHIVER_OP_HELP_COMMON_OPERATIONS:
	    EDVHelp(core, "Common Operations", toplevel);
	    break;
	}

	archiver->freeze_count--;
}

/*
 *	Operation ID enter notify callback nexus.
 *
 *	The data must be a edv_archiver_opid_struct *.
 */
void EDVArchiverOPEnterCB(
	toolbar_item_struct *item, gint id, gpointer data  
)
{
	const gchar *tooltip;
	edv_archiver_opid_struct *opid = EDV_ARCHIVER_OPID(data);
	edv_archiver_struct *archiver = (opid != NULL) ? opid->archiver : NULL;
	if(archiver == NULL)
	    return;

	tooltip = opid->tooltip;
	if(!STRISEMPTY(tooltip))
	    EDVStatusBarMessage(archiver->status_bar, tooltip, FALSE);
}

/*
 *	Operation ID leave notify callback nexus.
 */
void EDVArchiverOPLeaveCB(
	toolbar_item_struct *item, gint id, gpointer data  
)
{
	edv_archiver_opid_struct *opid = EDV_ARCHIVER_OPID(data);
	edv_archiver_struct *archiver = (opid != NULL) ? opid->archiver : NULL;
	if(archiver == NULL)
	    return;

	EDVStatusBarMessage(archiver->status_bar, NULL, FALSE);
}


/*
 *	Sync Disks.
 */
void EDVArchiverOPSyncDisks(edv_archiver_struct *archiver)
{
	edv_status_bar_struct *sb;

	if(archiver == NULL)
	    return;

	EDVArchiverSetBusy(archiver, TRUE);

	sb = archiver->status_bar;

	EDVStatusBarMessage(sb, "Syncing disks...", TRUE);

	/* Sync disks */
	EDVSyncDisks(archiver->core);

	EDVStatusBarMessage(sb, "Disk sync done", FALSE);
	EDVStatusBarProgress(sb, 0.0f, FALSE);

	EDVArchiverSetBusy(archiver, FALSE);
}

/*
 *      Write Protect toggle.
 */
void EDVArchiverOPWriteProtect(edv_archiver_struct *archiver)
{
	gboolean write_protect;
	cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(archiver == NULL)
	    return;

	EDVArchiverSetBusy(archiver, TRUE);

	core = archiver->core;
	cfg_list = core->cfg_list;

	/* Get the current write protect state */
	write_protect = (gboolean)CFGItemListGetValueI(
	    cfg_list, EDV_CFG_PARM_WRITE_PROTECT
	);

	/* Toggle write protect */
	write_protect = !write_protect;

	/* Set new write protect state */
	CFGItemListSetValueI(
	    cfg_list, EDV_CFG_PARM_WRITE_PROTECT,
	    write_protect, FALSE
	);

	/* Notify about the write protect change */
	EDVWriteProtectChangedEmit(core, write_protect);

	EDVArchiverSetBusy(archiver, FALSE);
}

/*
 *	Delete Method: Recycle.
 */
void EDVArchiverOPDeleteMethodRecycle(edv_archiver_struct *archiver)
{
	cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(archiver == NULL)
	    return;

	EDVArchiverSetBusy(archiver, TRUE);

	core = archiver->core;
	cfg_list = core->cfg_list;

	EDV_SET_I(
	    EDV_CFG_PARM_DELETE_METHOD,
	    EDV_DELETE_METHOD_RECYCLE
	);
	EDVReconfiguredEmit(core);

	EDVArchiverSetBusy(archiver, FALSE);
}

/*
 *      Delete Method: Purge.
 */                          
void EDVArchiverOPDeleteMethodPurge(edv_archiver_struct *archiver)
{
	cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(archiver == NULL)
	    return;

	EDVArchiverSetBusy(archiver, TRUE);

	core = archiver->core;
	cfg_list = core->cfg_list;

	EDV_SET_I(
	    EDV_CFG_PARM_DELETE_METHOD,
	    EDV_DELETE_METHOD_PURGE
	);
	EDVReconfiguredEmit(core);

	EDVArchiverSetBusy(archiver, FALSE);
}


/*
 *	New Archive callback.
 */
void EDVArchiverOPNew(edv_archiver_struct *archiver)
{
	const gchar *ext_list[] = EDV_ARCHIVER_ARCHIVE_FILE_EXTENSIONS;
	gboolean response, created_archive = FALSE;
	gint i, nftypes = 0, npaths = 0;
	gchar **paths_list = NULL;
	GtkWidget *toplevel;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	edv_core_struct *core;

	if((archiver == NULL) || FileBrowserIsQuery())
	    return;

	EDVArchiverSetBusy(archiver, TRUE);

	toplevel = archiver->toplevel;
	core = archiver->core;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	{
	    EDVArchiverSetBusy(archiver, FALSE);
	    return;
	}

	/* Create the file types list */
	for(i = 0; ext_list[i] != NULL; i += 2)
	    FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		ext_list[i], ext_list[i + 1]
	    );
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);

	/* Query user for new archive */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
	    "Create Archive",
	    "Create", "Cancel",
	    NULL,			/* Use the last location */
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(response)
	{
	    gchar *new_path = STRDUP((npaths > 0) ?
		paths_list[0] : NULL
	    );
	    if(!STRISEMPTY(new_path))
	    {
		struct stat stat_buf;
		guint m;
		gint fd;

		/* Get the name */
		const gchar *name = g_basename(new_path);
		if(name == NULL)
		    name = new_path;

		/* Name does not contain an extension? */
		if((strchr((const char *)name, '.') == NULL) &&
		   (ftype_rtn != NULL)
		)
		{
		    /* Append the extension to new_path */
		    gchar *ext = STRDUP(ftype_rtn->ext);
		    if(ext != NULL)
		    {
			gchar *s = (gchar *)strpbrk((char *)ext, " \t,");
			if(s != NULL)
			    *s = '\0';

			/* Update the new path */
			s = g_strconcat(
			    new_path,
			    (*ext != '.') ? "." : "",
			    ext,
			    NULL
			);
			g_free(ext);
			g_free(new_path);
			new_path = s;
			if(new_path == NULL)
			{
			    FileBrowserDeleteTypeList(ftypes_list, nftypes);
			    EDVArchiverSetBusy(archiver, FALSE);
			    return;
			}

			/* Reget the pointer to the new_path's name */
			name = g_basename(new_path);
			if(name == NULL)
			    name = new_path;
		    }
		}

		/* Check the extension to see if it is a valid archive */

		/* If the new archive exists then confirm overwrite */
		if(!stat((const char *)new_path, &stat_buf))
		{
		    /* Confirm overwrite */
		    gint response;
		    gchar *msg = g_strdup_printf(
"Overwrite existing file:\n\
\n\
    %s",
			new_path
		    );
		    EDVPlaySoundWarning(core);
		    CDialogSetTransientFor(toplevel);
		    response = CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Confirme Escriba Para Reemplazar"
#elif defined(PROG_LANGUAGE_FRENCH)
"Confirmer craser"
#elif defined(PROG_LANGUAGE_GERMAN)
"Besttigen Sie berschreibt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Confermare Sovrascrivere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bevestiig Beschrijft"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Confirme Overwrite"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Bekreft Overskriver"
#else
"Confirm Overwrite"
#endif
			, msg,
			NULL,
			CDIALOG_ICON_WARNING,
			CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
			CDIALOG_BTNFLAG_NO
		    );
		    g_free(msg);
		    CDialogSetTransientFor(NULL);
		    if(response == CDIALOG_RESPONSE_YES)
		    {
			/* Remove the existing file */
			if(unlink((const char *)new_path))
			{
			    const gint error_code = (gint)errno;
			    gchar *msg = g_strdup_printf(
"Unable to remove the existing file:\n\
\n\
    %s\n\
\n\
%s.",
				new_path, g_strerror(error_code)
			    );
			    EDVPlaySoundError(core);
			    EDVMessageError(
				"New Archive Failed",
				msg,
				NULL,
				toplevel
			    );
			    g_free(msg);
			    FileBrowserDeleteTypeList(ftypes_list, nftypes);
			    g_free(new_path);
			    EDVArchiverSetBusy(archiver, FALSE);
			    return;
			}

			/* Notify about the existing file being removed */
			EDVObjectRemovedEmit(core, new_path);
		    }
		    else
		    {
			/* Do not overwrite */
			FileBrowserDeleteTypeList(ftypes_list, nftypes);
			g_free(new_path);
			EDVArchiverSetBusy(archiver, FALSE);
			return;
		    }
		}

		/* Create the new archive */
		m = EDVGetUMask();
		fd = (gint)open(
		    (const char *)new_path,
		    O_WRONLY | O_CREAT | O_EXCL,
		    (mode_t)(~m) &
			(S_IRUSR | S_IWUSR |
			 S_IRGRP | S_IWGRP |
			 S_IROTH | S_IWOTH)
		    );
		if(fd > -1)
		{
		    /* Get the new archive's stats */
		    fstat(fd, &stat_buf);

		    /* Do not put anything in to the archive */
		    close((int)fd);

		    created_archive = TRUE;

		    /* Open the new archive */
		    EDVArchiverOpenArchive(
			archiver, new_path,
			EDVArchiverCurrentPassword(archiver)
		    );

		    /* Notify about the new archive being created */
		    EDVObjectAddedEmit(
			core, new_path, &stat_buf
		    );
		}
		else
		{
		    /* Unable to create a new archive */
		    const gint error_code = (gint)errno;
		    gchar *msg = g_strdup_printf(
"Unable to create the new archive:\n\
\n\
    %s\n\
\n\
%s.",
			new_path, g_strerror(error_code)
		    );
		    EDVPlaySoundError(core);
		    EDVMessageError(
			"New Archive Failed",
			msg,
			NULL,
			toplevel
		    );
		    g_free(msg);
		}

		EDVArchiverUpdateMenus(archiver);
	    }

	    g_free(new_path);
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	/* Reset the file selector due to changes */
	FileBrowserReset();

	/* If a new archive was successfully created, then prompt to
	 * add objects
	 */
	if(created_archive)
	{
	    EDVArchiverOPAdd(archiver);
	}

	EDVArchiverSetBusy(archiver, FALSE);
}

/*
 *	Open archive callback.
 */
void EDVArchiverOPOpen(edv_archiver_struct *archiver)
{
	const gchar *ext_list[] = EDV_ARCHIVER_ARCHIVE_FILE_EXTENSIONS;
	gboolean response;
	gint nftypes = 0, npaths = 0;
	gchar **paths_list = NULL;
	GtkWidget *toplevel;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	edv_core_struct *core;

	if((archiver == NULL) || FileBrowserIsQuery())
	    return;

	EDVArchiverSetBusy(archiver, TRUE);

	toplevel = archiver->toplevel;
	core = archiver->core;

	/* Create the file types list */
	if(ext_list != NULL)
	{
	    gint i;
	    gchar *s, *s2;

	    s = STRDUP("");
	    for(i = 0; ext_list[i] != NULL; i += 2)
	    {
		s2 = g_strconcat(s, ext_list[i], NULL);
		g_free(s);
		s = s2;

		if(ext_list[i + 2] != NULL)
		{
		    s2 = g_strconcat(s, " ", NULL);
		    g_free(s);
		    s = s2;
		}
	    }
	    FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		s, "All Archives"
	    );
	    g_free(s);

	    for(i = 0; ext_list[i] != NULL; i += 2)
		FileBrowserTypeListNew(
		    &ftypes_list, &nftypes,
		    ext_list[i], ext_list[i + 1]
		);
	}
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);


	/* Query user for archive to open */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
	    "Open Archive",
	    "Open", "Cancel",
	    NULL,		/* Startup path */
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(response)
	{
	    const gchar *new_path = (npaths > 0) ?
		paths_list[0] : NULL;
	    if(!STRISEMPTY(new_path))
	    {
		/* Open the archive */
		EDVArchiverOpenArchive(
		    archiver,
		    new_path,
		    EDVArchiverCurrentPassword(archiver)
		);
	    }
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	EDVArchiverSetBusy(archiver, FALSE);
}


/*
 *      Close.
 */
void EDVArchiverOPClose(edv_archiver_struct *archiver)
{
	if(archiver == NULL)
	    return;

	EDVArchiverSyncConfiguration(archiver);
	EDVArchiverUnmap(archiver);
}

/*
 *	Close All Windows.
 */
void EDVArchiverOPExit(edv_archiver_struct *archiver)
{
	edv_core_struct *core;

	if(archiver == NULL)
	    return;

	core = archiver->core;

	EDVArchiverSyncConfiguration(archiver);
	EDVArchiverUnmap(archiver);

	/* Schedual a new pending operation on the core to close all
	 * the windows
	 */
	core->pending_flags |= EDV_CORE_PENDING_CLOSE_ALL_WINDOWS;
}


/*
 *	Returns a list of selected objects in the archive.
 *
 *	The calling function must delete the returned list and each
 *	object.
 */
static GList *EDVArchiverGetSelectedObjects(
	edv_archiver_struct *archiver
)
{
	GList *glist, *obj_list = NULL;
	GtkCList *clist;
	edv_archive_object_struct *obj;

	if(archiver == NULL)
	    return(obj_list);

	clist = (GtkCList *)archiver->contents_clist;
	if(clist == NULL)
	    return(obj_list);

	/* Generate a list of archive object stat structures from the
	 * selected rows on the contents clist. Each structure must
	 * be coppied
	 */
	glist = clist->selection;
	while(glist != NULL)
	{
	    obj = EDV_ARCHIVE_OBJECT(gtk_clist_get_row_data(
		clist, (gint)glist->data
	    ));
	    if(obj != NULL)
		obj_list = g_list_append(
		    obj_list, EDVArchObjectCopy(obj)
		);

	    glist = g_list_next(glist);
	}

	return(obj_list);
}

/*
 *	Add Objects To Archive.
 */
void EDVArchiverOPAdd(edv_archiver_struct *archiver)
{
	gboolean	yes_to_all = FALSE,
			recurse = TRUE,
			archive_existed = FALSE;
	gint compression = 50;		/* 0 to 100 */
	gboolean dereference_links = FALSE;

	gboolean response;
	gint	status, nobjs, objects_added = 0,
		nftypes = 0, npaths = 0;
	gchar	*arch_path = NULL,
		*password = NULL,
		*add_dir = NULL,
		**paths_list = NULL;
	const gchar *error_msg;
	GList *src_paths_list = NULL, *new_paths_list = NULL;
	GtkWidget *toplevel;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	edv_core_struct *core;

	if((archiver == NULL) || FileBrowserIsQuery())
	    return;

	EDVArchiverSetBusy(archiver, TRUE);

	toplevel = archiver->toplevel;
	core = archiver->core;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	{
	    EDVArchiverSetBusy(archiver, FALSE);
	    return;
	}

#define CLEANUP_RETURN	{				\
 if(new_paths_list != NULL) {				\
  g_list_foreach(					\
   new_paths_list, (GFunc)g_free, NULL			\
  );							\
  g_list_free(new_paths_list);				\
 }							\
							\
 if(src_paths_list != NULL) {				\
  g_list_foreach(					\
   src_paths_list, (GFunc)g_free, NULL			\
  );							\
  g_list_free(src_paths_list);				\
 }							\
							\
 g_free(arch_path);					\
 g_free(password);					\
 g_free(add_dir);					\
							\
 /* Delete the file types list */			\
 FileBrowserDeleteTypeList(ftypes_list, nftypes);	\
							\
 return;						\
}

	/* Get the current archive path */
	arch_path = STRDUP(EDVArchiverCurrentLocation(archiver));
	if(STRISEMPTY(arch_path))
	{
	    /* No current archive specified */
	    EDVPlaySoundWarning(core);
	    EDVMessageWarning(
		"Add Object Failed",
"The current archive location was not specified.\n\
\n\
You must either go to File->New to create a new\n\
archive or go to File->Open to open an existing\n\
archive in order to add objects to it.",
		NULL,
		toplevel
	    );
	    EDVArchiverSetBusy(archiver, FALSE);
	    CLEANUP_RETURN;
	}

	/* Generate the suggested add location based on the parent
	 * directory of the current archive
	 */
	add_dir = g_dirname(arch_path);

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);

	/* Query the user for the objects to add */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
	    "Add Objects To Archive",
	    "Add", "Cancel",
	    add_dir,
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* User canceled? */
	if(!response)
	{
	    EDVArchiverSetBusy(archiver, FALSE);
	    CLEANUP_RETURN;
	}

	/* Get the list of objects to add */
	if(npaths > 0)
	{
	    gint i;
	    const gchar *s;
	    gchar *path;

	    for(i = 0; i < npaths; i++)
	    {
		s = paths_list[i]; 
		if(STRISEMPTY(s))
		    continue;

		path = STRDUP(s);
		if(path == NULL)
		    continue;

		/* Path from file browser may have tailing
		 * deliminators, get rid of them
		 */
		StripPath(path);

		src_paths_list = g_list_append(src_paths_list, path);
	    }
	}
	/* No objects to add? */
	if(src_paths_list == NULL)
	{
	    EDVArchiverSetBusy(archiver, FALSE);
	    CLEANUP_RETURN;
	}

	nobjs = g_list_length(src_paths_list);

	/* Query the user for the add to archive options */
	password = STRDUP(EDVArchiverCurrentPassword(archiver));
	if(!EDVArchAddOptsGetResponse(
	    core, toplevel,
	    arch_path,
	    &password,
	    &recurse, &compression, &dereference_links
	))
	{
	    /* User canceled */
	    EDVArchiverSetBusy(archiver, FALSE);
	    CLEANUP_RETURN;
	}

	/* Record if the archive exists */
	if(arch_path != NULL)
	{
	    struct stat stat_buf;
	    if(!stat((const char *)arch_path, &stat_buf))
		archive_existed = TRUE;
	}

	status = 0;
	if(TRUE)
	{
	    /* Add the selected object(s) to the archive */
	    status = EDVArchAdd(
		core, arch_path,
		src_paths_list,		/* List of objects to add to the
					 * archive */
		&new_paths_list,	/* Return list of objects added to
					 * the archive */
		password,
		toplevel, TRUE, TRUE, &yes_to_all,
		recurse, compression, dereference_links
	    );

	    /* Check for errors */
	    error_msg = EDVArchAddGetError(core);
	    if(!STRISEMPTY(error_msg))
	    {
		/* Report the error */
		EDVPlaySoundError(core);
		EDVMessageError(
		    "Add Object Error",
		    error_msg,
		    NULL,
		    toplevel
		);
	    }

	    objects_added = g_list_length(new_paths_list);

	    /* Report that the archive has been modified, this will
	     * cause the Archiver to reload the archive listing
	     */
	    if(arch_path != NULL)
	    {
		struct stat lstat_buf;
		if(!lstat((const char *)arch_path, &lstat_buf))
		{
		    if(archive_existed)
			EDVObjectModifiedEmit(
			    core, arch_path, arch_path, &lstat_buf
			);
		    else
			EDVObjectAddedEmit(
			    core, arch_path, &lstat_buf
			);
		}
	    }
	}

	/* Unmap the progress dialog */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Play the completed sound on success */
	if(status == 0)
	    EDVPlaySoundCompleted(core);

	/* Update the status bar */
	if(TRUE)
	{
	    gchar *msg;
	    if(status == -4)
		msg = g_strdup_printf(
		    "Add operation canceled"
		);
	    else if(objects_added > 0)
		msg = g_strdup_printf(
		    "Added %i %s",
		    objects_added,
		    (objects_added == 1) ? "object" : "objects"
		);
	    else
		msg = g_strdup_printf(
		    "Unable to add %s",
		    (nobjs == 1) ? "object" : "objects"
		);
	    EDVStatusBarMessage(archiver->status_bar, msg, FALSE);
	    g_free(msg);
	}

	/* Reset the file selector due to changes */
	FileBrowserReset();

	EDVArchiverSetBusy(archiver, FALSE);

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}

/*
 *	Extract Objects From Archive.
 */
void EDVArchiverOPExtract(edv_archiver_struct *archiver)
{
	gboolean	yes_to_all = FALSE,
			preserve_directories = TRUE,
			preserve_timestamps = TRUE;
	gboolean response;
	gint	status,
		nftypes = 0, npaths = 0,
		nobjs, objects_extracted = 0;
	gchar	*arch_path = NULL,
		*password = NULL,
		*dest_path = NULL,
		**paths_list = NULL;
	GList *obj_list = NULL;
	GtkWidget *toplevel;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	edv_core_struct *core;

	if((archiver == NULL) || FileBrowserIsQuery())
	    return;

	EDVArchiverSetBusy(archiver, TRUE);

	toplevel = archiver->toplevel;
	core = archiver->core;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	{
	    EDVArchiverSetBusy(archiver, FALSE);
	    return;
	}

#define CLEANUP_RETURN	{				\
 if(obj_list != NULL) {					\
  g_list_foreach(					\
   obj_list, (GFunc)EDVArchObjectDelete, NULL		\
  );							\
  g_list_free(obj_list);				\
 }							\
							\
 g_free(arch_path);					\
 g_free(password);					\
 g_free(dest_path);					\
							\
 /* Delete the file types list */			\
 FileBrowserDeleteTypeList(ftypes_list, nftypes);	\
							\
 return;						\
}

	/* Get the current archive path */
	arch_path = STRDUP(EDVArchiverCurrentLocation(archiver));
	if(STRISEMPTY(arch_path))
	{
	    /* No current archive specified */
	    EDVPlaySoundWarning(core);
	    EDVMessageWarning(
		"Extract Object Failed",
"The current archive location was not specified.\n\
\n\
You must go to File->Open to open an existing\n\
archive in order to extract objects from it.",
		NULL,
		toplevel
	    );
	    EDVArchiverSetBusy(archiver, FALSE);
	    CLEANUP_RETURN;
	}

	/* Get the list of selected objects */
	obj_list = EDVArchiverGetSelectedObjects(archiver);
	if(obj_list == NULL)
	{
	    EDVArchiverSetBusy(archiver, FALSE);
	    CLEANUP_RETURN;
	}

	nobjs = g_list_length(obj_list);

	/* Get the suggested extract location based on the parent
	 * directory of the current archive
	 */
	dest_path = g_dirname(arch_path);

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);

	/* Query user for extract destination */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
	    "Select Extract Destination",
	    "Select", "Cancel",
	    dest_path,
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* User canceled? */
	if(!response)
	{
	    EDVArchiverSetBusy(archiver, FALSE);
	    CLEANUP_RETURN;
	}

	/* Get the new destination */
	g_free(dest_path);
	dest_path = NULL;
	if(npaths > 0)
	{
	    const gchar *path = paths_list[0];
	    if(!STRISEMPTY(path))
	    {
		g_free(dest_path);
		dest_path = STRDUP(path);

		/* Path from file browser may have tailing
		 * deliminators, so we need to get rid of them
		 */
		StripPath(dest_path);
	    }
	}
	/* Invalid extract destination? */
	if(dest_path == NULL)
	{
	    EDVPlaySoundWarning(core);
	    EDVMessageWarning(
		"Invalid Extract Destination",
		"The selected extract destination directory is invalid",
		NULL,
		toplevel
	    );
	    EDVArchiverSetBusy(archiver, FALSE);
	    CLEANUP_RETURN;
	}

	/* Query the user for extract from archive options */
	password = STRDUP(EDVArchiverCurrentPassword(archiver));
	if(!EDVArchExtractOptsGetResponse(
	    core, toplevel,
	    arch_path,
	    &password,
	    &preserve_directories,
	    &preserve_timestamps
	))
	{
	    /* User canceled */
	    EDVArchiverSetBusy(archiver, FALSE);
	    CLEANUP_RETURN;
	}

	status = 0;
	if(TRUE)
	{
	    const gchar *error_msg;
	    GList *new_paths_list = NULL;

	    /* Extract the objects from the archive */
	    status = EDVArchExtract(
		core, arch_path,
		obj_list,
		FALSE,				/* Not extract all */
		dest_path, &new_paths_list,
		password,
		toplevel, TRUE, TRUE, &yes_to_all,
		preserve_directories,
		preserve_timestamps
	    );

	    /* Check for errors */
	    error_msg = EDVArchExtractGetError(core);
	    if(!STRISEMPTY(error_msg))
	    {
		/* Report the error */
		EDVPlaySoundError(core);
		EDVMessageError(
		    "Extract Object Error",
		    error_msg,
		    NULL,
		    toplevel
		);
	    }

	    /* Notify about the extracted objects */
	    if(new_paths_list != NULL)
	    {
		struct stat lstat_buf;
		const gchar *path;
		GList *glist;

		for(glist = new_paths_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    path = (gchar *)glist->data;
		    if(path == NULL)
			continue;

		    if(lstat((const char *)path, &lstat_buf))
			continue;

		    EDVObjectAddedEmit(core, path, &lstat_buf);
		    objects_extracted++;
		}

		/* Delete the list of extracted object paths */
		g_list_foreach(new_paths_list, (GFunc)g_free, NULL);
		g_list_free(new_paths_list);
	    }
	}

	/* Unmap the progress dialog */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Play the completed sound on success */
	if(status == 0)                      
	    EDVPlaySoundCompleted(core);

	/* Update the status bar */
	if(TRUE)
	{
	    gchar *msg;
	    if(status == -4)
		msg = STRDUP(
		    "Extract operation canceled"
		);
	    else if(objects_extracted > 0)
		msg = g_strdup_printf(
		    "Extracted %i %s",
		    objects_extracted,
		    (objects_extracted == 1) ? "object" : "objects"
		);
	    else
		msg = g_strdup_printf(
		    "Unable to extract %s",
		    (nobjs == 1) ? "object" : "objects"
		);
	    EDVStatusBarMessage(archiver->status_bar, msg, FALSE);
	    g_free(msg);
	}

	/* Reset the file selector due to changes */
	FileBrowserReset();

	EDVArchiverSetBusy(archiver, FALSE);

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}

/*
 *	Extract All Objects From Archive.
 */
void EDVArchiverOPExtractAll(edv_archiver_struct *archiver)
{
	gboolean	response,
			yes_to_all = FALSE,
			preserve_directories = TRUE,
			preserve_timestamps = TRUE;
	gint	status,
		nftypes = 0, npaths = 0,
		nobjs, objects_extracted = 0;
	gchar	*arch_path = NULL,
		*password = NULL,
		*dest_path = NULL,
		**paths_list = NULL;
	GList *obj_list = NULL;
	GtkWidget *toplevel;
	GtkCList *clist;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	edv_core_struct *core;

	if((archiver == NULL) || FileBrowserIsQuery())
	    return;

	EDVArchiverSetBusy(archiver, TRUE);

	toplevel = archiver->toplevel;
	clist = GTK_CLIST(archiver->contents_clist);
	core = archiver->core;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	{
	    EDVArchiverSetBusy(archiver, FALSE);
	    return;
	}

#define CLEANUP_RETURN	{				\
 if(obj_list != NULL) {					\
  g_list_foreach(					\
   obj_list, (GFunc)EDVArchObjectDelete, NULL		\
  );							\
  g_list_free(obj_list);				\
 }							\
							\
 g_free(arch_path);					\
 g_free(password);					\
 g_free(dest_path);					\
							\
 /* Delete the file types list */			\
 FileBrowserDeleteTypeList(ftypes_list, nftypes);	\
							\
 return;						\
}

	/* Get the current archive path */
	arch_path = STRDUP(EDVArchiverCurrentLocation(archiver));
	if(STRISEMPTY(arch_path))
	{
	    /* No current archive specified */
	    EDVPlaySoundWarning(core);
	    EDVMessageWarning(
		"Extract Object Failed",
"The current archive location was not specified.\n\
\n\
You must go to File->Open to open an existing\n\
archive in order to extract objects from it.",
		NULL,
		toplevel
	    );
	    EDVArchiverSetBusy(archiver, FALSE);
	    CLEANUP_RETURN;
	}

	/* Get the list of all the objects in the archive */
	if(clist->rows > 0)
	{
	    const gint nrows = clist->rows;
	    gint i;
	    edv_archive_object_struct *obj;

	    for(i = 0; i < nrows; i++)
	    {
		obj = EDV_ARCHIVE_OBJECT(
		    gtk_clist_get_row_data(clist, i)
		);
		if(obj == NULL)
		    continue;

		obj_list = g_list_append(
		    obj_list, EDVArchObjectCopy(obj)
		);
	    }
	}
	if(obj_list == NULL)
	{
	    EDVArchiverSetBusy(archiver, FALSE);
	    CLEANUP_RETURN;
	}

	nobjs = g_list_length(obj_list);

	/* Get the suggested extract location based the on parent
	 * directory of the current archive
	 */
	dest_path = g_dirname(arch_path);

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);

	/* Query user for extract destination */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
	    "Select Extract Destination",
	    "Select", "Cancel",
	    dest_path,
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* User canceled? */
	if(!response)
	{
	    EDVArchiverSetBusy(archiver, FALSE);
	    CLEANUP_RETURN;
	}

	/* Get the new destination */
	g_free(dest_path);
	dest_path = NULL;
	if(npaths > 0)
	{
	    const gchar *path = paths_list[0];
	    if(!STRISEMPTY(path))
	    {
		g_free(dest_path);
		dest_path = STRDUP(path);

		/* Path from file browser may have tailing
		 * deliminators, so we need to get rid of them
		 */
		StripPath(dest_path);
	    }
	}
	/* Invalid extract destination? */
	if(dest_path == NULL)
	{
	    EDVPlaySoundWarning(core);
	    EDVMessageWarning(
		"Invalid Extract Destination",
		"The selected extract destination directory is invalid",
		NULL,
		toplevel
	    );
	    EDVArchiverSetBusy(archiver, FALSE);
	    CLEANUP_RETURN;
	}

	/* Query user for extract from archive options */
	password = STRDUP(EDVArchiverCurrentPassword(archiver));
	if(!EDVArchExtractOptsGetResponse(
	    core, toplevel,
	    arch_path,
	    &password,
	    &preserve_directories,
	    &preserve_timestamps
	))
	{
	    /* User canceled */
	    EDVArchiverSetBusy(archiver, FALSE);
	    CLEANUP_RETURN;
	}


	status = 0;
	if(TRUE)
	{
	    const gchar *error_msg;
	    GList *new_paths_list = NULL;

	    /* Extract all the objects from the archive */
	    status = EDVArchExtract(
		core, arch_path,
		obj_list,
		TRUE,				/* Extract all */
		dest_path, &new_paths_list,
		password,
		toplevel, TRUE, TRUE, &yes_to_all,
		preserve_directories,
		preserve_timestamps
	    );

	    /* Check for errors */
	    error_msg = EDVArchExtractGetError(core);
	    if(!STRISEMPTY(error_msg))
	    {
		/* Report the error */
		EDVPlaySoundError(core);
		EDVMessageError(
		    "Extract Object Error",
		    error_msg,
		    NULL,
		    toplevel
		);
	    }

	    /* Notify about the extracted objects */
	    if(new_paths_list != NULL)
	    {
		struct stat lstat_buf;
		const gchar *path;
		GList *glist;

		for(glist = new_paths_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    path = (const gchar *)glist->data;
		    if(path == NULL)
			continue;

		    if(lstat((const char *)path, &lstat_buf))
			continue;

		    EDVObjectAddedEmit(core, path, &lstat_buf);
		    objects_extracted++;
		}

		/* Delete the list of extracted object paths */
		g_list_foreach(new_paths_list, (GFunc)g_free, NULL);
		g_list_free(new_paths_list);
	    }
	}

	/* Unmap the progress dialog */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Play the completed sound on success */
	if(status == 0)
	    EDVPlaySoundCompleted(core);

	/* Update the status bar */
	if(TRUE)
	{
	    gchar *msg;
	    if(status == -4)
		msg = STRDUP(
		    "Extract operation canceled"
		);
	    else if(objects_extracted > 0)
		msg = g_strdup_printf(
		    "Extracted %i %s",
		    objects_extracted,
		    (objects_extracted == 1) ? "object" : "objects"
		);
	    else
		msg = g_strdup_printf(
		    "Unable to extract %s",
		    (nobjs == 1) ? "object" : "objects"
		);
	    EDVStatusBarMessage(archiver->status_bar, msg, FALSE);
	    g_free(msg);
	}

	/* Reset the file selector due to changes */
	FileBrowserReset();

	EDVArchiverSetBusy(archiver, FALSE);

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}

/*
 *	Delete Object From Archive.
 */
void EDVArchiverOPDelete(edv_archiver_struct *archiver)
{
	gboolean yes_to_all = FALSE;
	gint status, nobjs, objects_deleted = 0;
	gchar *arch_path = NULL;
	const gchar *src_path, *error_msg;
	GList *obj_list = NULL;
	GtkWidget *toplevel;
	edv_archive_object_struct *obj;
	edv_core_struct *core;

	if(archiver == NULL)
	    return;

	EDVArchiverSetBusy(archiver, TRUE);

	toplevel = archiver->toplevel;
	core = archiver->core;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	{
	    EDVArchiverSetBusy(archiver, FALSE);
	    return;
	}

#define CLEANUP_RETURN	{			\
 if(obj_list != NULL) {				\
  g_list_foreach(				\
   obj_list, (GFunc)EDVArchObjectDelete, NULL	\
  );						\
  g_list_free(obj_list);			\
 }						\
						\
 g_free(arch_path);				\
						\
 return;					\
}

	/* Get the current archive's path */
	arch_path = STRDUP(EDVArchiverCurrentLocation(archiver));
	if(arch_path == NULL)
	{
	    EDVArchiverSetBusy(archiver, FALSE);
	    CLEANUP_RETURN;
	}

	/* Get the list of selected objects */
	obj_list = EDVArchiverGetSelectedObjects(archiver);
	if(obj_list == NULL)
	{
	    EDVArchiverSetBusy(archiver, FALSE);
	    CLEANUP_RETURN;
	}

	/* Get the path of the first object only if it is the only
	 * object in the list
	 */
	nobjs = g_list_length(obj_list);
	if(nobjs == 1)
	{
	    obj = EDV_ARCHIVE_OBJECT(obj_list->data);
	    src_path = (obj != NULL) ? obj->full_path : NULL;
	}
	else
	    src_path = NULL;

	/* Confirm delete */
	status = EDVConfirmArchiveDelete(
	    core, toplevel, src_path, nobjs
	);
	switch(status)
	{
	  case CDIALOG_RESPONSE_YES_TO_ALL:
	    yes_to_all = TRUE;
	  case CDIALOG_RESPONSE_YES:
	    break;
	  default:
	    EDVArchiverSetBusy(archiver, FALSE);
	    CLEANUP_RETURN;
	    break;
	}

	/* Iterate through selected objects in the archive */
	status = EDVArchDelete(
	    core,
	    arch_path,			/* Archive */
	    obj_list,			/* Objects to delete from the archive */
	    EDVArchiverCurrentPassword(archiver),
	    toplevel, TRUE, TRUE, &yes_to_all
	);

	/* Check for errors */
	error_msg = EDVArchDeleteGetError(core);
	if(!STRISEMPTY(error_msg))
	{
	    /* Report the error */
	    EDVPlaySoundError(core);
	    EDVMessageError(
		"Delete Object From Archive Error",
		error_msg,
		NULL,
		toplevel
	    );
	}

	/* Was the archive object successfully delected? */
	if(status == 0)
	{
	    /* Report object in archive being removed */
	    GList *glist;
	    for(glist = obj_list; glist != NULL; glist = g_list_next(glist))
	    {
		obj = EDV_ARCHIVE_OBJECT(glist->data);
		if(obj == NULL)
		    continue;

#if 0
/* Note that this may not be good since some archive formats have 
 * archive programs that do not report which objects were deleted
 * so it may be more accurate to skip this and just have the
 * EDVObjectModifiedEmit() below cause this archiver to reload
 */
		EDVArchiveObjectRemovedEmit(
		    core, arch_path, obj->full_path
		);
#endif
		objects_deleted++;
	    }
	}

	/* Unmap the progress dialog */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Play the completed sound on success */
	if(status == 0)
	    EDVPlaySoundCompleted(core);
	
	/* Report that the archive has been modified, this will
	 * cause the archiver to reload the archive listing
	 */
	if(arch_path != NULL)
	{
	    struct stat lstat_buf;
	    if(lstat((const char *)arch_path, &lstat_buf))
	    {
		const gint error_code = (gint)errno;
#ifdef ENOENT
		if(error_code == ENOENT)
		    EDVObjectRemovedEmit(core, arch_path);
#endif
	    }
	    else
	    {
		EDVObjectModifiedEmit(core, arch_path, arch_path, &lstat_buf);
	    }
	}

	/* Update the status bar */
	if(TRUE)
	{
	    gchar *msg;
	    if(status == -4)
		msg = STRDUP(
		    "Delete operation canceled"
		);
	    else if(objects_deleted > 0)
		msg = g_strdup_printf(
		    "Deleted %i %s",
		    objects_deleted,
		    (objects_deleted == 1) ? "object" : "objects"
		);
	    else
		msg = g_strdup_printf(
		    "Unable to delete %s",
		    (nobjs == 1) ? "object" : "objects"
		);
	    EDVStatusBarMessage(archiver->status_bar, msg, FALSE);
	    g_free(msg);
	}

	EDVArchiverSetBusy(archiver, FALSE);

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}

/*
 *	Check Archive.
 */
void EDVArchiverOPCheck(edv_archiver_struct *archiver)
{
	gboolean yes_to_all = FALSE;
	gint status;
	gchar *arch_path;
	const gchar *error_msg;
	GtkWidget *toplevel;
	edv_core_struct *core;

	if(archiver == NULL)
	    return;

	EDVArchiverSetBusy(archiver, TRUE);

	toplevel = archiver->toplevel;
	core = archiver->core;

	/* Get the current archive's path */
	arch_path = STRDUP(EDVArchiverCurrentLocation(archiver));
	if(arch_path == NULL)
	{
	    EDVArchiverSetBusy(archiver, FALSE);
	    return;
	}

	/* Check the archive */
	status = EDVArchChk(
	    core,
	    arch_path,
	    EDVArchiverCurrentPassword(archiver),
	    toplevel,
	    TRUE, TRUE, &yes_to_all
	);

	/* Unmap the progress dialog which may have been mapped
	 * during the above operation
	 */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Check for errors */
	error_msg = EDVArchChkGetError(core);
	if(!STRISEMPTY(error_msg))
	{
	    gchar *msg = g_strdup_printf(
"The following problem was found:\n\
\n\
%s\n\
\n\
You may attempt to try and fix this problem by going to\n\
Edit->Fix.",
		error_msg
	    );
	    EDVPlaySoundWarning(core);
	    EDVMessageWarning(
		"Check Results",
		msg,
		NULL,
		toplevel
	    );
	    g_free(msg);
	}
	else if(status != -4)
	{
	    gchar *msg = g_strdup_printf(
"There were no problems found in the archive\n\
\"%s\".",
		g_basename(arch_path)
	    );
	    EDVPlaySoundCompleted(core);
	    EDVMessageInfo(
		"Check Results",
		msg,
		NULL,
		toplevel
	    );
	    g_free(msg);
	}

	/* Update the status bar */
	if(TRUE)
	{
	    gchar *msg;
	    if(status == -4)
		msg = STRDUP(
		    "Check operation canceled"
		);
	    else
		msg = STRDUP(
		    "Checked archive"
		);
	    EDVStatusBarMessage(archiver->status_bar, msg, FALSE);
	    g_free(msg);
	}

	g_free(arch_path);

	EDVArchiverSetBusy(archiver, FALSE);
}

/*
 *	Fix Archive.
 */
void EDVArchiverOPFix(edv_archiver_struct *archiver)
{
	gboolean yes_to_all = FALSE;
	gint status;
	gchar *arch_path;
	const gchar *error_msg;
	GtkWidget *toplevel;
	edv_core_struct *core;

	if(archiver == NULL)
	    return;

	EDVArchiverSetBusy(archiver, TRUE);

	toplevel = archiver->toplevel;
	core = archiver->core;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	{
	    EDVArchiverSetBusy(archiver, FALSE);
	    return;
	}

	/* Get the current archive's path */
	arch_path = STRDUP(EDVArchiverCurrentLocation(archiver));
	if(arch_path == NULL)
	{
	    EDVArchiverSetBusy(archiver, FALSE);
	    return;
	}

	/* Confirm fix */
	if(TRUE)
	{
	    gint response;
	    gchar *msg = g_strdup_printf(
"Fix archive \"%s\"?",
		g_basename(arch_path)
	    );
	    EDVPlaySoundQuestion(core);
	    CDialogSetTransientFor(toplevel);
	    response = CDialogGetResponse(
		"Confirm Fix",
		msg,
		NULL,
		CDIALOG_ICON_QUESTION,
		CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
		CDIALOG_BTNFLAG_YES
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(msg);
	    if(response != CDIALOG_RESPONSE_YES)
	    {
		g_free(arch_path);
		EDVArchiverSetBusy(archiver, FALSE);
		return;
	    }
	}

	/* Fix the archive */
	status = EDVArchFix(
	    core,
	    arch_path,
	    EDVArchiverCurrentPassword(archiver),
	    toplevel,
	    TRUE, TRUE, &yes_to_all
	);

	/* Unmap the progress dialog which may have been mapped
	 * during the above operation
	 */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Get the fix results */
	error_msg = EDVArchFixGetError(core);
	if(!STRISEMPTY(error_msg))
	{
	    EDVPlaySoundError(core);
	    EDVMessageObjectOPError(
		"Fix Error",
		error_msg,
		arch_path,
		toplevel
	    );
	}
	else if(status != -4)
	{
	    EDVPlaySoundCompleted(core);
	}

	if(TRUE)
	{
	    /* Report that the archive has been modified */
	    struct stat lstat_buf;
	    if(!lstat((const char *)arch_path, &lstat_buf))
		EDVObjectModifiedEmit(
		    core, arch_path, arch_path, &lstat_buf
		);
	}

	/* Update the status bar */
	if(TRUE)
	{
	    gchar *msg;
	    if(status == -4)
		msg = STRDUP(
		    "Fix operation canceled"
		);
	    else if(status == 0)
		msg = STRDUP(
		    "Fixed archive"
		);
	    else
		msg = STRDUP(
		    "Unable to fix archive"
		);
	    EDVStatusBarMessage(archiver->status_bar, msg, FALSE);
	    g_free(msg);
	}

	g_free(arch_path);

	EDVArchiverSetBusy(archiver, FALSE);
}

/*
 *	Select All.
 */
void EDVArchiverOPSelectAll(edv_archiver_struct *archiver)
{
	edv_core_struct *core;
	GtkCList *clist;

	if(archiver == NULL)
	    return;

	EDVArchiverSetBusy(archiver, TRUE);

	clist = GTK_CLIST(archiver->contents_clist);
	core = archiver->core;

	/* Select all */
	gtk_clist_freeze(clist);
	gtk_clist_select_all(clist);
	gtk_clist_thaw(clist);

	/* Assume highest row index as the last selected row */
	archiver->contents_clist_selected_row = clist->rows - 1;

	EDVStatusBarMessage(
	    archiver->status_bar, "All objects selected", FALSE
	);

	EDVArchiverSetBusy(archiver, FALSE);
}

/*
 *	Unselect All.
 */
void EDVArchiverOPUnselectAll(edv_archiver_struct *archiver)
{
	edv_core_struct *core;
	GtkCList *clist;

	if(archiver == NULL)
	    return;

	EDVArchiverSetBusy(archiver, TRUE);

	clist = GTK_CLIST(archiver->contents_clist);
	core = archiver->core;

	/* Unselect all */
	gtk_clist_freeze(clist);
	gtk_clist_unselect_all(clist);
	gtk_clist_thaw(clist);

	/* Mark contents clist's row as unselected */
	archiver->contents_clist_selected_row = -1;

	EDVStatusBarMessage(
	    archiver->status_bar, "All objects unselected", FALSE
	);

	EDVArchiverSetBusy(archiver, FALSE);
}

/*
 *      Invert Selection.
 */
void EDVArchiverOPInvertSelection(edv_archiver_struct *archiver)
{
	edv_core_struct *core;
	GtkCList *clist;
	GList *glist, *selection;
	gint row, total_rows;

	if(archiver == NULL)
	    return;

	EDVArchiverSetBusy(archiver, TRUE);

	clist = GTK_CLIST(archiver->contents_clist);
	core = archiver->core;

	/* Get a copy of selected rows list */
	selection = (clist->selection != NULL) ?
	    g_list_copy(clist->selection) : NULL;

	/* Invert the selection */
	gtk_clist_freeze(clist);
	for(row = 0, total_rows = clist->rows;
	    row < total_rows;
	    row++
	)
	{
	    for(glist = selection;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		if(row == (gint)glist->data)
		{
		    gtk_clist_unselect_row(clist, row, 0);
		    break;
		}
	    }
	    /* Row not selected? */
	    if(glist == NULL)
		gtk_clist_select_row(clist, row, 0);
	}
	gtk_clist_thaw(clist);

	g_list_free(selection);

	EDVStatusBarMessage(
	    archiver->status_bar, "Selection inverted", FALSE
	);

	EDVArchiverSetBusy(archiver, FALSE);
}

/*
 *	Archive Properties.
 */
void EDVArchiverOPProperties(edv_archiver_struct *archiver)
{
	struct stat lstat_buf;
	gchar *arch_path;
	GtkWidget *toplevel;
	edv_object_struct *obj;
	edv_core_struct *core;

	if(archiver == NULL)
	    return;

	EDVArchiverSetBusy(archiver, TRUE);

	toplevel = archiver->toplevel;
	core = archiver->core;

	/* Get the current archive */
	arch_path = STRDUP(EDVArchiverCurrentLocation(archiver));
	if(arch_path == NULL)
	{
	    EDVArchiverSetBusy(archiver, FALSE);
	    return;
	}

	obj = EDVObjectNew();
	if(obj == NULL)
	{
	    EDVArchiverSetBusy(archiver, FALSE);
	    return;
	}

	EDVObjectSetPath(obj, arch_path);
	if(lstat((const char *)arch_path, &lstat_buf))
	{
	    const gint error_code = (gint)errno;
	    EDVObjectPropSet(
		obj,
		EDV_OBJECT_PROP_NAME_ERROR,
		g_strerror(error_code),
		TRUE
	    );
	}
	else
	{
	    EDVObjectSetStat(obj, &lstat_buf);
	}
	EDVObjectUpdateLinkFlags(obj);

	/* Create a new Properties Dialog */
	EDVNewPropertiesDialog(core, obj, toplevel);

	EDVObjectDelete(obj);

	g_free(arch_path);

	EDVArchiverSetBusy(archiver, FALSE);
}


/*
 *	Refresh.
 */
void EDVArchiverOPRefresh(edv_archiver_struct *archiver)
{
	GtkWidget *toplevel;
	GtkCList *clist;
	const cfg_item_struct *cfg_list;
	edv_status_bar_struct *sb;
	edv_core_struct *core;

	if(archiver == NULL)
	    return;

	toplevel = archiver->toplevel;
	core = archiver->core;
	cfg_list = core->cfg_list;
	sb = archiver->status_bar;

	EDVArchiverSetBusy(archiver, TRUE);
	GUIBlockInput(toplevel, TRUE);

	/* Update the Contents GtkList */
	clist = GTK_CLIST(archiver->contents_clist);
	if(clist != NULL)
	{
	    /* Record the last scroll position */
	    const gfloat	last_x = GTK_ADJUSTMENT_GET_VALUE(clist->hadjustment),
				last_y = GTK_ADJUSTMENT_GET_VALUE(clist->vadjustment);

	    gtk_clist_freeze(clist);

	    /* Reget the listing */
	    EDVArchiverContentsGetListing(
		archiver,
		EDV_GET_B(EDV_CFG_PARM_LISTS_ANIMATE_UPDATES)
	    );

	    gtk_clist_thaw(clist);

	    /* Scroll back to the original position */
	    EDVScrollCListToPosition(clist, last_x, last_y);
	}

	EDVArchiverUpdateMenus(archiver);
	EDVStatusBarMessage(sb, "Refreshed contents listing", FALSE);

	GUIBlockInput(toplevel, FALSE);
	EDVArchiverSetBusy(archiver, FALSE);
}

/*
 *      Refresh All.
 */
void EDVArchiverOPRefreshAll(edv_archiver_struct *archiver)
{
	if(archiver == NULL)
	    return;

	EDVArchiverSetBusy(archiver, TRUE);

	/* Refresh Archiver */
	EDVArchiverOPRefresh(archiver);

	EDVArchiverSetBusy(archiver, FALSE);
}

/*
 *	View Comment & Statistics.
 */
void EDVArchiverOPCommentAndStatistics(edv_archiver_struct *archiver)
{
	gchar *arch_path;
	edv_archive_info_dlg_struct *d;

	if(archiver == NULL)
	    return;

	EDVArchiverSetBusy(archiver, TRUE);

	arch_path = STRDUP(EDVArchiverCurrentLocation(archiver));
	if(arch_path == NULL)
	{
	    EDVArchiverSetBusy(archiver, FALSE);
	    return;
	}

	/* Delete the previous Archive Info Dialog (if any) */
	d = archiver->arch_info_dlg;
	if(d != NULL)
	{
	    EDVArchiveInfoDelete(d);
	    archiver->arch_info_dlg = d = NULL;
	}

	/* Create a new Archive Info Dialog, load the current archive
	 * information, and map the Archive Info Dialog
	 */
	archiver->arch_info_dlg = d = EDVArchiveInfoNew(
	    archiver->core,
	    arch_path,
	    EDVArchiverCurrentPassword(archiver),
	    archiver->toplevel
	);
	if(d != NULL)
	{
	    EDVArchiveInfoMap(d);
	    EDVArchiveInfoSetHasChanges(d, FALSE);
	    EDVArchiveInfoUpdateMenus(d);
	}

	g_free(arch_path);

	EDVArchiverSetBusy(archiver, FALSE);
}


/*
 *	Sets the contents list filter.
 */
void EDVArchiverContentsFilter(edv_archiver_struct *archiver)
{
	gchar **strv;
	gint strc;
	GtkWidget *toplevel;

	if((archiver == NULL) || PDialogIsQuery())
	    return;

	EDVArchiverSetBusy(archiver, TRUE);

	toplevel = archiver->toplevel;

	PDialogDeleteAllPrompts();
	PDialogSetTransientFor(toplevel);
	PDialogAddPrompt(NULL, "Filter:", archiver->contents_list_filter);
	PDialogSetSize(320, -1);
	strv = PDialogGetResponseIconData(
	    "Set Filter",
	    NULL, NULL,
	    (guint8 **)icon_wildcards_32x32_xpm,
	    "Set", "Cancel",
	    PDIALOG_BTNFLAG_SUBMIT | PDIALOG_BTNFLAG_CANCEL,
	    PDIALOG_BTNFLAG_SUBMIT,
	    &strc
	);
	PDialogSetTransientFor(NULL);
	if((strv != NULL) && (strc > 0))  
	{                                 
	    if(strc > 0)
	    {
		g_free(archiver->contents_list_filter);
		archiver->contents_list_filter = STRDUP(strv[0]);
	    }

	    EDVArchiverOPRefresh(archiver);
	}

	PDialogDeleteAllPrompts();

	EDVArchiverSetBusy(archiver, FALSE);
}

/*
 *	MIME Types.
 */
void EDVArchiverMIMETypes(edv_archiver_struct *archiver)
{
	gchar *type_str = NULL;
	gint i;
	GtkWidget *toplevel;
	GtkCList *clist;
	edv_core_struct *core;
	if(archiver == NULL)
	    return;

	EDVArchiverSetBusy(archiver, TRUE);

	toplevel = archiver->toplevel;
	clist = (GtkCList *)archiver->contents_clist;
	core = archiver->core;

	i = EDVCListGetSelectedLast(clist, NULL);
	if(i > -1)
	{
	    edv_archive_object_struct *obj = EDV_ARCHIVE_OBJECT(
		gtk_clist_get_row_data(clist, i)
	    );
	    if(obj != NULL)
		EDVMatchObjectTypeString(
		    core->mimetype, core->total_mimetypes,
		    obj->type,
		    obj->permissions,
		    obj->full_path,
		    &type_str
	    );
	}

	EDVMapMIMETypesListWin(core, type_str, toplevel);

	EDVArchiverSetBusy(archiver, FALSE);
}
