# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core.components import input_provider
from elisa.core import input_event
from elisa.core.tests.elisa_test_case import ElisaTestCase

def create_input_event(data, component_source):
    source, typ, value = data
    source = getattr(input_event.EventSource, source)
    value = getattr(input_event.EventValue, value)
    typ = getattr(input_event.EventType, typ)
    evt = input_event.InputEvent(input_event.EventSource.MOUSE, typ, value)
    evt.component_source = component_source
    return evt


EVTS = [ ('MOUSE', 'MOUSE_DOWN', 'MOUSE_LEFT'),
         ('MOUSE', 'MOUSE_UP',   'MOUSE_LEFT')
         ]


class MousePollInputProvider(input_provider.PollInputProvider):
    name = 'mouse'

    def create_input_event(self, data):
        return create_input_event(data, self)

    def get_input_events(self):
        return [self.create_input_event(data) for data in EVTS]

class MousePushInputProvider(input_provider.PushInputProvider):
    name = 'push_mouse'

    def create_input_event(self, data):
        return create_input_event(data, self)

    def bind(self):
        pass

    def unbind(self):
        pass

    def trigger_some_events(self):
        for evt in [self.create_input_event(data) for data in EVTS]:
            self.input_manager.process_event(evt)

class FakeManager:

    up_ok = False
    down_ok = False

    def process_event(self, event):
        if event.type == input_event.EventType.MOUSE_UP and \
           event.value == input_event.EventValue.MOUSE_LEFT:
            self.up_ok = True

        if event.type == input_event.EventType.MOUSE_DOWN and \
             event.value == input_event.EventValue.MOUSE_LEFT:
            self.down_ok = True

class TestInputProvider(ElisaTestCase):
    """
    """

    def test_poll_provider(self):
        """ check poll provider

        Poll some events from a provider and check their value/type.
        """
        c = MousePollInputProvider()
        events = c.get_input_events()
        self.failUnless(events)

        self.assertEquals(len(events), 2)
        down, up = events

        self.assertEquals(up.type, input_event.EventType.MOUSE_UP)
        self.assertEquals(up.value, input_event.EventValue.MOUSE_LEFT)

        self.assertEquals(down.type, input_event.EventType.MOUSE_DOWN)
        self.assertEquals(down.value, input_event.EventValue.MOUSE_LEFT)

    def test_push_provider(self):
        """ check push_input_provider calls its manager on event trigger

        Attach a fake manager to the component, trigger some events
        and check some bool attributes of the manager.
        """
        c = MousePushInputProvider()
        m = FakeManager()
        c.input_manager = m

        c.trigger_some_events()
        self.assertEquals(m.up_ok, True)
        self.assertEquals(m.down_ok, True)
