# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

"""
GstMetadataProvider component class
"""


__maintainer__ = 'Florian Boucault <florian@fluendo.com>'
__maintainer2__ = 'Philippe Normand <philippe@fluendo.com>'

try:
    import gnomevfs
except ImportError:
    gnomevfs = None

from elisa.base_components.metadata_provider import MetadataProvider
import time, os
import gst
from elisa.core.media_uri import MediaUri

from twisted.internet import threads

class GstMetadata(MetadataProvider):

    _supported_schemes = [ "http", "file",
                           "smb", "ftp",
                           "sftp", "nfs",
                           "ssh",
                         ]

    def __init__(self):
        MetadataProvider.__init__(self)
        self._mp3_pipe = None
        self._ogg_pipe = None
        self._tags = {}
        self._lasterror = None
        self._rebuild_pipeline = False

    def _get_type(self, uri):
        dummy, ext = os.path.splitext(uri.path)
        ext = ext[1:]
        if ext not in ('mp3', 'ogg'):
            ext = 'unknown'
        return ext

    def _pad_added(self, element, pad, pipeline):
        if not pad.is_linked():
            fakesink = pipeline.get_by_name('sink')
            try:
                pad.link(fakesink.get_pad('sink'))
            except gst.LinkError:
                # We ignore this error
                pass

    def _get_pipeline(self, uri):
        ext = self._get_type(uri)
        self._tags = {}

        scheme = uri.scheme
        if scheme == 'file':
            src = 'filesrc'
        else:
            src = 'gnomevfssrc'

        if ext == 'mp3':
            if self._mp3_pipe == None or self._rebuild_pipeline:
                self._mp3_pipe = gst.parse_launch('%s name=source ! id3demux name=demux ! fakesink name=sink' % src)
                demux = self._mp3_pipe.get_by_name('demux')
                if demux:
                    demux.connect ('pad-added', self._pad_added, self._mp3_pipe)

            return self._mp3_pipe
        elif ext == 'ogg':
            if self._ogg_pipe == None or self._rebuild_pipeline:
                self._ogg_pipe = gst.parse_launch('%s name=source ! oggdemux name=demux ! vorbisdec name=sink ! fakesink' % src)
                demux = self._ogg_pipe.get_by_name('demux')
                demux.connect ('pad-added', self._pad_added, self._ogg_pipe)
            return self._ogg_pipe
        return None


    def _get_tags(self, uri):
        scheme = uri.scheme
        #scheme, path = filename.split('://')

        if scheme == 'file':
            filename = uri.path

        elif gnomevfs:
            filename = gnomevfs.escape_host_and_path_string(str(uri))


        # build or get existing pipeline
        pipeline = self._get_pipeline(uri)
        if not pipeline:
            return {}

        #stop bus flushing
        bus = pipeline.get_bus()
        bus.set_flushing(False)

        #if pipeline is not rebuilded, put to READY state
        if not self._rebuild_pipeline:
            if not self._change_state(pipeline, gst.STATE_READY):
                self._clean(pipeline)
                return {}

        #set filename and put to paused state
        #message will processed and tags will found if they are in first buffer (most cases)
        source = pipeline.get_by_name('source')
        source.set_property('location', filename.encode('utf-8'))
        if not self._change_state(pipeline, gst.STATE_PAUSED):
            self._clean(pipeline)
            return {}

        #flushing and empty the bus
        #bus.set_flushing(True)
        #self._process_bus(pipeline)

        self._clean(pipeline)

        return self._tags

    def _clean(self, pipeline):
        #clean if we want to rebuild pipeline each time
        if self._rebuild_pipeline:
            self._change_state(pipeline, gst.STATE_NULL)


    def _change_state(self, pipeline, state):
        r = self._change_state_no_bus(pipeline, state)
        self._process_bus(pipeline)
        return r

    def _change_state_no_bus(self, pipeline, state):
        pipeline.set_state(state)
        max_step = 10
        step = 0
        current_states = pipeline.get_state(50*gst.MSECOND)
        while gst.STATE_CHANGE_ASYNC == current_states[0]:
            step +=1
            time.sleep(0.01)
            if step > max_step:
                return False
            current_states = pipeline.get_state(50*gst.MSECOND)

        if current_states == (gst.STATE_CHANGE_SUCCESS, state,
                              gst.STATE_VOID_PENDING):
            return True

        return False

    def _process_bus(self, pipeline):
        bus = pipeline.get_bus()
        while True:
            time.sleep(0.01)
            msg = bus.pop()
            if not msg or msg.type==gst.MESSAGE_EOS:
                return True
            elif msg.type==gst.MESSAGE_TAG:
                self._tags = {}
                parsed_tags = dict(msg.parse_tag())
                for key, value in parsed_tags.iteritems():
                    if type(value) == str:
                        value = value.decode('utf-8')
                    self._tags[key] = value
                if pipeline != self._ogg_pipe:
                    self._tags['audio-codec'] = u''
            elif msg.type==gst.MESSAGE_ERROR:
                self._lasterror = msg.parse_error()
                gerror, debug = self._lasterror
                return False

        return True


    def get_rank(self):
        rank = 128
        return rank

    def able_to_handle(self, metadata):
        content_type = metadata.get('content-type', '')
        uri = metadata.get('uri')
        able = uri is not None and uri.scheme in self._supported_schemes and \
               content_type in ('audio', 'video')
        return able

    def get_metadata(self,metadata):
        d = threads.deferToThread(self._get_metadata,metadata)
        return d


    def _get_metadata(self, metadata):
        tags = self._get_tags(metadata['uri'])

        if 'track-number' in tags:
            tags['track'] = tags['track-number']
        if 'title' in tags:
            tags['song'] = tags['title']
        if 'extended-comment' in tags:
            del tags['extended-comment']
            
        for key, value in tags.items():
            if value != '':
                if metadata.has_key(key):
                    if metadata[key] == None:
                        metadata[key] = value
                    else:
                        self.debug("No write the key %s with the value %s",
                                   key, value)
                else:
                    metadata[key] = value
        return metadata
