# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

"""
HalServiceProvider component class
"""


__maintainer__ = 'Philippe Normand <philippe@fluendo.com>'
__maintainer2__ = 'Benjamin Kampmann <benjamin@fluendo.com>'


from elisa.base_components.service_provider import ServiceProvider
from elisa.core.bus.bus_message import ComponentsLoaded, DeviceAction
from elisa.core import common
from elisa.core import component
from elisa.core.player import PlayerPlaying, PlayerStopping
import os, time
from twisted.internet import threads

from elisa.extern.translation import gettexter, N_
T_ = gettexter('elisa-hal-service')

import dbus
if getattr(dbus, 'version', (0,0,0)) >= (0,41,0):
    import dbus.glib
from dbus.exceptions import DBusException

try:
    import gconf
except ImportError:
    gconf = None

class NotMountable(Exception):
    pass

def get_dbus_error_message(exception):
    """
    Retrieve the message of the DBus error. Usually something like
    'org.freedesktop....'

    @param exception: the DBus exception to analyze
    @type exception:  L{dbus.DBusException}
    @rtype:           string
    """
    if dbus.version < (0, 82, 0):
        msg = exception.message
    else:
        msg = exception._dbus_error_name
    return msg


class HalService(ServiceProvider):
    """
    This class implements HAL support
    """

    def __init__(self):
        ServiceProvider.__init__(self)
        self._hotplugged_devices = {}

    def initialize(self):
        ServiceProvider.initialize(self)
        bus = common.application.bus
        bus.register(self._components_loaded, ComponentsLoaded)
        bus.register(self._player_started, PlayerPlaying)
        bus.register(self._player_stopped, PlayerStopping)
        bus.register(self._device_action, DeviceAction)

        self.bus = dbus.SystemBus()
        self.ses = dbus.SessionBus()
        try:
            self.hal_manager = self.bus.get_object('org.freedesktop.Hal',
                                                   '/org/freedesktop/Hal/Manager')
        except DBusException, error:
            self.warning(error)
            msg = "HAL is not installed or might not be running."
            raise component.InitializeFailure(msg)
        
        self.hal_iface = dbus.Interface(self.hal_manager,
                                        'org.freedesktop.Hal.Manager')

        # TODO: remove this when dbus_service is checked in
        self.pm_iface = None
        self._pm_cookie = None

        try:
            # Even if there is also a org.gnome.PowerManager, the
            # ScreenSaver is for our use the better one!
            self.pm = self.ses.get_object('org.gnome.ScreenSaver',
                                          '/org/gnome/Screensaver')
            self.pm_iface = dbus.Interface(self.pm, "org.gnome.ScreenSaver")
        except Exception, e:
            self.warning("Could not connect to the Gnome ScreenSaver: %s" % e)
        # / TODO

    def _device_action(self, msg, sender):
        if msg.action == DeviceAction.ActionType.EJECT:
            for udi, infos in self._hotplugged_devices.iteritems():
                (name, fstype, mount_point) = infos
                if fstype == msg.fstype and mount_point == msg.mount_point:
                    device = self.get_device_with_udi(udi)
                    self.eject_device(device)
                    break
                
    # TODO: remove this when dbus_service is checked in
    def _player_started(self, msg, sender):
        if self.pm_iface:
            if not self._pm_cookie:
                self.debug("Asking for Inhibition of PowerManager")
                self._pm_cookie = self.pm_iface.Inhibit('elisa', 'playing media') 

    def _player_stopped(self, msg, sender):
        if self.pm_iface:
            if self._pm_cookie:
                self.debug("Asking for UnInhibition of PowerManager")
                self.pm_iface.UnInhibit(self._pm_cookie)
                self._pm_cookie = None
    # / TODO
    
    def _components_loaded(self, msg, sender):
        self.detect_coldplugged()
        self.hal_iface.connect_to_signal('DeviceAdded',
                                         self.device_added_callback)
        self.hal_iface.connect_to_signal('DeviceRemoved',
                                         self.device_removed_callback)

    def start(self):
        self.stop_volume_manager_monitoring()

    def stop(self):
        self.restore_volume_manager_monitoring()
        self._player_stopped('','')

    def detect_coldplugged(self):
        udis = self.hal_iface.FindDeviceByCapability('volume')

        for udi in udis:
            # get a hal object for the volume referenced by udi
            volume = self.get_device_with_udi(udi)
            parent_uri = volume.GetProperty('info.parent')
            parent = self.get_device_with_udi(parent_uri)

            if (parent.GetProperty('storage.removable') \
                or parent.GetProperty('storage.removable.media_available')) \
                and (parent.GetProperty('storage.hotpluggable') or
                volume.GetProperty('volume.is_disc')):
                # if its parent is removable and hotpluggable or the volume
                # is a disc (DVD or CD) then the volume has been coldplugged
                self.device_added_callback(udi)

    def stop_volume_manager_monitoring(self):
        """
        Neutralize some of the volume_manager monitoring settings so
        that the user won't see rhythmbox pop up when an iPod is
        inserted (for example).
        """
        if gconf:
            client = gconf.client_get_default()
            path = '/desktop/gnome/volume_manager'
            autoplay_cda = client.get_bool('%s/autoplay_cda' % path)
            autoplay_dvd = client.get_bool('%s/autoplay_dvd' % path)
            autobrowse = client.get_bool('%s/autobrowse' % path)
            self.volume_manager_config = {'autoplay_cda': autoplay_cda,
                                          'autoplay_dvd': autoplay_dvd,
                                          'autobrowse': autobrowse}
            for prop in self.volume_manager_config.keys():
                client.set_bool('%s/%s' % (path, prop), False)

    def restore_volume_manager_monitoring(self):
        """
        Restore the volume_manager gconf settings
        """
        if gconf:
            client = gconf.client_get_default()
            path = '/desktop/gnome/volume_manager'
            for prop, value in self.volume_manager_config.iteritems():
                client.set_bool('%s/%s' % (path, prop), value)

    def get_device_with_udi(self, udi):
        obj = self.bus.get_object('org.freedesktop.Hal', udi)
        if obj != None:
            device = dbus.Interface(obj, 'org.freedesktop.Hal.Device')
            return device
        return None

    def parent_is_ipod(self, device):
        parent_udi = device.GetProperty(u'info.parent')
        parent = self.get_device_with_udi(parent_udi)
        if not parent.PropertyExists(u'portable_audio_player.type'):
            return False
        return parent.GetProperty(u'portable_audio_player.type').lower() == 'ipod'
        # Is it intelligent to try to get the portable_audio_player.output_formats
        # too for accessing in audio or video?

    def eject_device(self, device):
        name = device.GetProperty(u'info.product')
        self.debug("ejecting: %r", name)
        interface = 'org.freedesktop.Hal.Device.Storage'
        method = device.get_dbus_method('Eject', dbus_interface=interface)

        try:
            method([])
        except ValueError:
            # not exposed on HAL DBus API
            
            block_device = device.GetProperty(u'block.device')
            # TODO: find something more cross-platform
            threads.deferToThread(os.system, "eject %s" % block_device)
                

    def mount_device(self, device):
        ignore = False
        
        if device.PropertyExists(u'volume.ignore'):
            ignore = device.GetProperty(u'volume.ignore')

        if ignore:
            self.debug("volume.ignore property set on %r, can't mount it", device)
            return
        
        if not device.PropertyExists(u'org.freedesktop.Hal.Device.Volume.method_names'):
            raise NotMountable()

        if not 'Mount' in device.GetProperty(u'org.freedesktop.Hal.Device.Volume.method_names'):
            raise NotMountable()

        interface = 'org.freedesktop.Hal.Device.Volume'
        method = device.get_dbus_method('Mount', dbus_interface=interface)

        name = device.GetProperty(u'info.product')
        self.debug("mounting: %r ", name)
        
        try:
            # Let's HAL decide, what the mount point should be
            method('', device.GetProperty(u'volume.fstype'),[])
        except DBusException, exc:
            already_mounted = 'org.freedesktop.Hal.Device.Volume.AlreadyMounted'
            unavailable = 'org.freedesktop.Hal.Device.Volume.MountPointNotAvailable'
            permission_denied = 'org.freedesktop.Hal.Device.Volume.PermissionDenied'
            
            msg = get_dbus_error_message(exc)
            
            if msg.startswith(already_mounted):
                self.info("Already mounted")
            elif msg.startswith(permission_denied):
                idx = msg.index(permission_denied) + len(permission_denied) + 2
                device_error = msg[idx:]
                self.info("Permission denied: %s", device_error)
            elif msg.startswith(unavailable):
                return None
            else:
                raise

        return device.GetProperty(u'volume.mount_point')

    def device_added_callback(self, udi):
        mount_point_prop = u'volume.label'

        if udi not in self._hotplugged_devices:
            device = self.get_device_with_udi(udi)
            if device.QueryCapability("volume"):

                try:
                    name = str(device.GetProperty('info.product'))
                except dbus.DBusException:
                    name = ''
                if self.parent_is_ipod(device):
                    fstype = 'ipod'
                    try:
                        point = self.mount_device(device)
                        # we support audio-only iPod for now
                        self.new_volume_cb(udi, name, fstype, point,
                                           ['audio', ])
                    except NotMountable, exception:
                        pass
                elif device.PropertyExists(u'volume.disc.is_videodvd'):
                    # No VideoDVD-Access-Plugin available for now
                    if device.GetProperty(u'volume.disc.is_videodvd'):
                        if not name:
                            name = T_(N_('DVD'))
                        self.new_volume_cb(udi, name, 'dvd', '', ['video',])
                elif device.PropertyExists(u'volume.disc.has_audio'):
                    # Checking for Audio-CDs
                    if device.GetProperty(u'volume.disc.has_audio'):
                        if not name:
                            name = T_(N_('Audio CD'))
                        fstype = 'cdda'
                        # FIXME: We are currently not supporting different
                        # devices. So the name is not sent!
                        self.new_volume_cb(udi, name, fstype,'', ['audio',])
                elif device.PropertyExists(u'volume.is_partition'):
                    if device.GetProperty(u'volume.is_partition'):
                        fstype = 'file'
                        if not name:
                            name = T_(N_('File device'))
                        try:
                            point = self.mount_device(device)
                            self.new_volume_cb(udi,name, fstype, point,
                                               ['audio', 'video', 'image'])
                        except NotMountable, exc:
                            pass

    def device_removed_callback(self, uid):
        if uid in self._hotplugged_devices:
            name, fstype, mount_point = self._hotplugged_devices[uid]
            del self._hotplugged_devices[uid]
            if self.del_volume_cb:
                if mount_point:
                    self.del_volume_cb(name, fstype, mount_point)
                if fstype == 'cdda':
                    # FIXME: add support for AudioCD removal
                    self.del_volume_cb(T_(N_('Audio CD')),'cdda','')
                elif fstype == 'dvd':
                    self.del_volume_cb(T_(N_('DVD')), 'dvd', '')
                    
    def new_volume_cb(self, uid, name, fstype, mount_point, media_types=None):
        """
        Called when a new volume has been detected

        @param name:            Name of the device
        @type name:             string
        @param fstype:          Filesystem type
        @type fstype:           string
        @param mount_point:     Mount point
        @type mount_point:      string
        """
        self._hotplugged_devices[uid] = ( name, fstype, mount_point )
        self.debug("New volume found %r at %r", name, mount_point)

        mount_point = "%s://%s" % (fstype, mount_point)
        msg = DeviceAction(DeviceAction.ActionType.LOCATION_ADDED,
                           name, fstype, mount_point, media_types)
        common.application.bus.send_message(msg)

    def del_volume_cb(self, name, fstype, mount_point):
        """
        Called when volume has been removed

        @param name:            Name of the device
        @type name:             string
        @param fstype:          Filesystem type
        @type fstype:           string
        @param mount_point:     Mount point
        @type mount_point:      string
        """
        self.debug("Volume unmounted %s" % name)
        mount_point = "%s://%s" % (fstype, mount_point)
        msg = DeviceAction(DeviceAction.ActionType.LOCATION_REMOVED,
                           name, fstype, mount_point)
        common.application.bus.send_message(msg)
