# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

# Based on work by Jan Jokela <janjokela@gmail.com>

__maintainer__ = 'Philippe Normand <philippe@fluendo.com>'

from elisa.base_components.media_provider import MediaProvider
from elisa.core import media_uri, component, common, db_backend, log
from elisa.core.utils import misc
from elisa.core.bus import bus_message
from elisa.core.observers.dict import DictObservable

from twisted.internet import threads

import os, sys, threading

DEFAULT_DB_PATH = os.path.join(os.path.expanduser('~'),
                               '.gnome2', 'f-spot', 'photos.db')

class FspotDB(log.Loggable):
    """ Retrieves tags and videos from the F-spot sqlite database
    """
    log_category = 'fspot_db'
    
    def __init__(self, db_path=DEFAULT_DB_PATH):
        """ Attempt a connection to the database
        """
        log.Loggable.__init__(self)

        if not os.path.exists(db_path):
            raise db_backend.DBBackendError("FSpot DB not found at %r" % db_path)
        
        options = {'db_backend': 'sqlite', 'database': db_path}
        self._backend = db_backend.DBBackend(**options)
        self._version = None

    def get_version(self):
        if self._version is None:
            rows = self._backend.sql_execute("select data from meta where name=?",
                                             "F-Spot Database Version")
            if rows:
                self._version = int(rows[0].data)
            else:
                self._version = 0
                
        return self._version

    def get_tag_with_id(self, tag_id):
        tag = None
        query = "SELECT id, name, category_id as parent_id from tags where id=?"
        rows = self._backend.sql_execute(query, tag_id)
        if len(rows):
            tag = rows[0]
        return tag

    def get_tags(self, tag_id=None):
        """ Return a list of tags [(tag id, tag name), (..)]
        @rtype: list
        """
        if tag_id == None:
            query = "SELECT id, name FROM tags WHERE category_id=0 ORDER BY name"
            params = ()
        else:
            query = "SELECT id, name FROM tags WHERE category_id=? ORDER BY name"
            params = (tag_id,)
        rows = self._backend.sql_execute(query, *params)
        return rows
      
    def get_photos(self, tag):
        """ Return a list of photos with the given tag
            [(diectory_path, filename), (..)]

        @param tag: the tag id 
        @type tag:  int
        @rtype:     list
        """
        if self.get_version() < 8:
            query = "SELECT id, directory_path, name FROM photos, photo_tags WHERE "\
                    "photos.id=photo_tags.photo_id AND photo_tags.tag_id=? "\
                    "ORDER BY photos.time"
        else:
            query = "SELECT id, uri FROM photos, photo_tags WHERE "\
                    "photos.id=photo_tags.photo_id AND photo_tags.tag_id=? "\
                    "ORDER BY photos.time"
            
        rows = self._backend.sql_execute(query, tag)
        return rows


class FspotMedia(MediaProvider):
    """ The Fspot MediaProvider component
    """

    default_config = {'db_path': DEFAULT_DB_PATH}
    config_doc = {'db_path': 'absolute path to f-spot photos.db file'}

    def initialize(self):
        self._db_lock = threading.Lock()
        self._db_connections = {}
        self._cache = {}
        db_path = self.config.get('db_path', DEFAULT_DB_PATH)

        if os.path.dirname(db_path) in ('','..'):
            app_config = common.application.config
            conf_path = app_config.get_config_dir()
            self.debug("Using %r to search F-Spot DB", conf_path)
            db_path = os.path.join(conf_path, db_path)
            
        self._db_path = db_path
        
        # Some rudimentary database checks
        try:
            db = FspotDB(self._db_path)
        # if not, we can't initialize our component
        except db_backend.DBBackendError, error:
            raise component.InitializeFailure(self.name, str(error))
        else:
            version = db.get_version()
            if version == 0:
                msg = "%r is not F-Spot DB" % self._db_path
                raise component.InitializeFailure(self.name, msg)
            
            self.debug("Found a F-Spot DB with version %s at %r", version,
                       self._db_path)
            
        # Lets make sure the user doesn't need to manually add Fspot
        # to the image activity locations
        uri = "fspot:///"
        action_type = bus_message.MediaLocation.ActionType.LOCATION_ADDED
        msg = bus_message.ForeignApplication(action_type, "F-Spot", 'fspot',
                                             uri, media_types=['image',])
        common.application.bus.send_message(msg)

    def db__get(self):
        db = None

        try:
            self._db_lock.acquire()
            current_thread = threading.currentThread()
            thread_id = id(current_thread)
            db = self._db_connections.get(thread_id)
            if not db:
                db = FspotDB(self._db_path)
                self._db_connections[thread_id] = db
        finally:
            self._db_lock.release()

        return db

    def scannable_uri_schemes__get(self):
        return []

    def supported_uri_schemes__get(self):
        return { 'fspot': 0 }

    def get_real_uri(self, uri):
        # our photos are files, so let's convert our uri's
        # to 'file' scheme
        if uri.host == '.':
            dirname = os.path.dirname(self._db_path)
        else:
            dirname = uri.host
        real_uri = media_uri.MediaUri("file://%s%s" % (dirname,uri.path))
        self.debug("%r is real URI of %r", real_uri, uri)
        return real_uri

    def blocking_get_media_type(self, uri):
        """
        @param uri: the URI to analyze
        @type uri:  L{elisa.core.media_uri.MediaUri}
        @rtype:     dict
        """
        file_type = ''
        mime_type = ''
        if not self.blocking_is_directory(uri):
            # ok, it's a file, let's ask for a mime and file type
            mime_type, file_type = misc.get_media_infos_from_mime(uri)
            if None in (mime_type, file_type):
                self.warning("Unknown media: %r", uri)
        else:   
            # directories don't have mime_types
            file_type = 'directory'
        return {'file_type': file_type, 'mime_type': mime_type}

    def blocking_is_directory(self, uri):
        """
        return True if a directory

        @param uri: the URI to analyze
        @type uri:  L{elisa.core.media_uri.MediaUri}
        @rtype:     bool
        """
        if uri.path == '/':
            # yup, the sheme root dir is a .. dir 
            return True
        elif uri.get_param('id'): 
            # this means it's a tag, we can assume tags are dirs
            return True
        else: return False
        
    def has_children_with_types(self, uri, media_types):
        return threads.deferToThread(self.blocking_has_children_with_types,
                                     uri, media_types)

    def blocking_has_children_with_types(self, uri, media_types):
        """
        @param uri:         the URI to scan
        @type uri:          L{elisa.core.media_uri.MediaUri}
        @param media_types: the media_types to look for on the directory
        @type media_types:  list of strings
        @rtype:             bool
        """
        has_children = False
        if 'image' in media_types or 'directory' in media_types:
            db = self.db
            if self.blocking_is_directory(uri): 
                if 'directory' in media_types:
                    category_id = int(uri.get_param('id','-1'))
                    if category_id != -1:
                        has_children = len(db.get_tags(category_id)) > 0
                        if not has_children and 'image' in media_types:
                            has_children = len(db.get_photos(category_id)) > 0
                    else:
                        has_children = True
                elif uri.path == '/':
                    tags = db.get_tags()
                    if tags != []:
                        has_children = True
                elif len(uri.path):
                    photos = db.get_photos(int(uri.get_param('id')))
                    if photos != []:
                        has_children = True
        return has_children
    
    def get_direct_children(self, uri, l):
        d = threads.deferToThread(self.blocking_get_direct_children, uri, l)
        return d

    def blocking_get_direct_children(self, uri, list_of_children):
        db = self.db
        # if the uri path is /, we have to retrieve the tags from the fspot
        # database
        if uri.path == '/':
            # add to the children list a MediaUri representing each tag
            tags = db.get_tags()
            for tag in tags:
                t = media_uri.MediaUri("fspot:///%s" % tag.name)
                t.set_param('id', int(tag.id))
                t.set_param('tag_id', -1)
                list_of_children.append((t, {}))
        
        elif len(uri.path):
            # it's a tag, lets get our child tags and child photos

            # strip leading /
            path = uri.path[1:]

            tag_id = int(uri.get_param('id'))

            # add child tags
            tags = db.get_tags(tag_id)
            for tag in tags:
                t = media_uri.MediaUri("fspot:///%s" % tag.name)
                t.set_param('id', int(tag.id))
                t.set_param('tag_id', tag_id)
                list_of_children.append((t, {}))

            # add to the children list a MediaUri representing each video
            photos = db.get_photos(tag_id)
            for photo in photos:
                if photo.has_key('uri'):
                    p = media_uri.MediaUri(photo.uri)
                    p.scheme = 'fspot'
                else:
                    photo_path = os.path.join(photo.directory_path, photo.name)
                    p = media_uri.MediaUri("fspot://%s" % photo_path)
                    # we want the photo filename
                    label = photo.name.decode("utf-8")
                    p.label = label
                p.fragment = photo.id
                p.set_param('tag_id', tag_id)
                d = DictObservable({'default_image': self.get_real_uri(p)})
                list_of_children.append((p, d))
                
        return list_of_children

    def blocking_next_location(self, from_uri, root=None):
        next_uri = None

        def from_cache(uri, node_id):
            parent = None
            for dir_uri in self._cache.keys():
                if dir_uri.get_param('id', None) == node_id:
                    parent = dir_uri
                    break
            if not parent:
                # cache wasn't filled, let's fill it a bit more
                tag = self.db.get_tag_with_id(node_id)
                u = 'fspot:///%s?id=%s&tag_id=%s' % (tag.name, node_id,
                                                     tag.parent_id)
                parent = media_uri.MediaUri(u)
                children = self.blocking_get_direct_children(parent, [])
                self._cache[parent] = [ c[0] for c in children ]
                
            return parent

        def get_parent(uri):
            parent = None
            parent_id = uri.get_param('tag_id', None)
            if parent_id is not None:
                if int(parent_id) == -1:
                    # parent is root
                    parent = media_uri.MediaUri('fspot:///')
                else:
                    parent = from_cache(uri, parent_id)
            self.debug("Parent of %r : %r", uri, parent)
            return parent

        def find_from_parent(uri, root):
            next_uri = None
            parent = get_parent(uri)
            if parent == root:
                parent = uri
                items = self._cache.get(root, [])
            else:
                parent_parent = get_parent(parent)
                if not parent_parent:
                    parent_parent = root
                self.debug("Parent of %r : %r", parent, parent_parent)
                items = []
                for item, children in self._cache.iteritems():
                    if str(item) == str(parent_parent):
                        items = children
                        break
                
            if parent in items:
                index = items.index(parent)
                if index < len(items) -1:
                    next_uri = items[index+1]
            return next_uri
        
        if self.blocking_is_directory(from_uri):
            children = self.blocking_get_direct_children(from_uri, [])
            cached_items = [ child[0] for child in children ]
            self._cache[from_uri] = cached_items
            if cached_items:
                next_uri = cached_items[0]
            else:
                next_uri = find_from_parent(from_uri, root)
        else:
            for dir_uri, items in self._cache.copy().iteritems():
                if from_uri in items:
                    index = items.index(from_uri)
                    if index < len(items)-1:
                        next_uri = items[index+1]
                    else:
                        next_uri = find_from_parent(from_uri, root)
                                    
            if next_uri and self.blocking_is_directory(next_uri):
                next_uri = self.blocking_next_location(next_uri, root=root)
            
        self.info("Next of %r : %r", from_uri,next_uri)
        return next_uri
        
    def blocking_open(self, uri, mode='r'):
        # we can't open directories
        if self.blocking_is_directory(uri):
            return None

        # lets convert the uri from our scheme to a file scheme
        # and ask the media_manager to provide us a capable
        # component
        uri = self.get_real_uri(uri)
        media_manager = common.application.media_manager
        if media_manager.enabled:
            media = media_manager.open(uri, mode, block)
        else:
            media = None
        return media
