# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


__maintainer__ = 'Lionel Martin <lionel@fluendo.com>'


from elisa.core.component import ComponentError
from elisa.base_components.view import View
from elisa.core import common
from elisa.core.observers.list import ListObserver
from elisa.base_components.controller import Controller


class ListView(View, list, ListObserver):

    supported_controllers = ('base:list_controller')
    default_associations = (
    ('base:list_controller','base:list_view'),
    )

    def __eq__(self, other):
        # necessary because otherwise list comparison would be used and this
        # only compares the content of the lists
        return id(self) == id(other)

    def __repr__(self):
        if hasattr(self, "controller"):
            return "<%s: %s items; connected to %s>" \
                   % (self.__class__.__name__,
                      len(self),
                      self.controller)
        else:
            return "<%s: %s items>" % (self.__class__.__name__,
                                       len(self))


 

    def inserted(self, elements, position):
        current_position = position
        for child_controller in elements:
            self.create_child_view(child_controller, current_position)
            current_position += 1


    def removed(self, elements, position):
        self.debug("%s got REMOVED for %s of %s elements" % (self, position,
                                                            len(elements)))

        for elem in elements:
            if position >= len(self):
                break
            self.debug("%s performs a remove of %s" % (self, self[position]))
            self.remove(self[position])


    def create_child_view(self, controller, position = 0):
        """
        @raise ComponentError : when view cannot be created
        """

        if isinstance(controller, Controller):
            pr = common.application.plugin_registry
            view_path_list = self.get_view_paths(controller.path)
            if len(view_path_list) == 0:
                self.warning("can't create view for controller %r (with view %r)" % (controller.path, self.path))
                raise ComponentError(controller.path)
            else:
                view = pr.create_component(view_path_list[0])
                view.parent = self
                view.frontend = self.frontend
                self.child_view_creating(view, controller, position)
                view.controller = controller
                self.insert(position, view)
                self.child_view_created(view, position)
                return view
        else:
            self.warning("can't create view for controller %r (with view %r)" % (controller.path, self.path))
            raise TypeError


    def child_view_created(self, view, position = 0):
        """
        called when the view is created
        """
        pass


    def child_view_creating(self, view, controller, position = 0):
        """
        called when the view is created
        before the controller is set to the view
        """
        pass


    def controller_changed(self):
        if self.controller == None:
            # FIXME: might be useful to remove all the children
            return
        
        current_position = 0
        for child_controller in self.controller:
            self.create_child_view(child_controller, current_position)
            current_position += 1

    def frontend_changed(self, previous_frontend, new_frontend):
        for child_view in self:
            child_view.frontend = new_frontend

    def append(self, element):
        if isinstance(element, View):
            element.parent = self
            element.frontend = self.frontend
        list.append(self, element)


    def extend(self, elements):
        for element in elements:
            if isinstance(element, View):
                element.parent = self
                element.frontend = self.frontend
        list.extend(self, elements)


    def insert(self, position, element):
        if isinstance(element, View):
            element.parent = self
            element.frontend = self.frontend
        list.insert(self, position, element)


    def pop(self, position=-1):
        element = list.pop(self, position)
        if isinstance(element, View):
            element.parent = None
            element.frontend = None


    def remove(self, element):
        list.remove(self, element)
        if isinstance(element, View):
            element.controller = None
            element.parent = None
            element.frontend = None

    def __setitem__(self, key, value):
        previous_value = self[key]
        if isinstance(value, View):
            value.parent = self
            value.frontend = self.frontend
        list.__setitem__(self, key, value)
        if isinstance(previous_value, View):
            previous_value.parent = None
            value.frontend = None


    def __delitem__(self, key):
        element = self[key]
        list.__delitem__(self, key)
        if isinstance(element, View):
            element.parent = None
            element.frontend = None

    def __setslice__(self, i, j, sequence):
        removed_elements = self[i:j]
        for element in sequence:
            if isinstance(element, View):
                element.parent = self
                element.frontend = self.frontend

        list.__setslice__(self, i, j, sequence)

        for element in removed_elements:
            if isinstance(element, View):
                element.parent = None
                element.frontend = None


    def __delslice__(self, i, j):
        removed_elements = self[i:j]
        list.__delslice__(self, i, j)
        for element in removed_elements:
            if isinstance(element, View):
                element.parent = None
                element.frontend = None
