# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

__maintainer__ = 'Benjamin Kampmann <benjamin@fluendo.com>'

from elisa.base_components.player_engine import PlayerEngine
from elisa.core.player import PlayerLoading, \
                              PlayerPlaying, \
                              PlayerPausing, \
                              PlayerStopping, \
                              PlayerError, \
                              PlayerBuffering, \
                              NewBaseTime, \
                              NewClock, \
                              STATES

from elisa.core.media_uri import MediaUri

import pygst
pygst.require('0.10')
import gst

import urllib

class PlaybinEngine(PlayerEngine):
    """ 
    This class implements a player using playbin to access the uris
    """

    uri_schemes = {'http' : 150, 'smb' : 150, 'file': 150, 'ftp' : 150,
                    'sftp' : 150, 'cdda' : 150, 'mms': 150}

    def __init__(self):
        PlayerEngine.__init__(self)
        self.log_category = 'playbin_engine'

        self._pipeline = None
        self._current_uri = None

        self._audiosink = None
        self._videosink = None 

        # Using for better Probing
        self._apad = None
        self._vpad = None

        self._probe_vid = None
        self._probe_aid = None

        # Used for faster access of Sink-Changing
        self._audiobin = None
        self._audiopad = None

        self._videobin = None
        self._videopad = None

        self._subtitle_uri = None
        self.visualization_name = ''
        self.visualization_enabled = True
        self._probe_id = None
#        self._create_pipeline()

        # A gst.STATE
        self._loading = False
        self._state = STATES.STOPPED

    # FIXME: make elisa able to handle methods for uri_schemes__get also!
    def _uri_schemes__get(self):
        schemes = {}
        reg = gst.registry_get_default()
        plugins = reg.get_plugin_list()
        try:
            for plugin in plugins:
                features = reg.get_feature_list_by_plugin(plugin.get_name())
                for feature in features:
                    if isinstance(feature, gst.ElementFactory) and \
                           feature.get_uri_type():
                        protocols = feature.get_uri_protocols()
                        for protocol in protocols:
                            schemes[str(protocol)] = 150
        except AttributeError:
            # FIXME: URI-Schemes bug..
            # There is a problem in getting the uri_protocols of factories.
            # depending on error #385841. 
            # The problem is fixed in the CVS only.
            schemes = {'http' : 150, 'smb' : 150, 'file': 150, 'ftp' : 150,
                    'sftp' : 150, 'cdda' : 150,}
                    
            self.warning("Could not find out all uris which are supported by"
                            " playbin. Using statically file, smb, ftp,"
                            " sftp, cdda and http only! \n If you have"
                            " problems with playing this uri-schemes, you"
                            " might install the latest gstreamer and python-"
                            "bindings from cvs.")
        return schemes


    def audio_sink__get(self):
        return self._audiosink

    def audio_sink__set(self, sink):

        if sink == None and self._pipeline:
            self._pipeline.set_state(gst.STATE_NULL)
            del self._pipeline
            self._pipeline = None

        self._audiosink = sink

        if self._pipeline:
            self._pipeline.set_property('audio-sink', self._audiosink)

        if self._pipeline:
            self._pipeline.set_property('audio-sink', self._audiosink)


    def video_sink__get(self):
        return self._videosink

    def video_sink__set(self, sink):
        """
        Set the videosink to sink.
        If the sink none and there was a video-sink before, destroy the
        pipeline as well and make a new one!
        """
        if sink == None:
            self._pipeline.set_state(gst.STATE_NULL)
            del self._pipeline
            self._pipeline = None

        self._videosink = sink
        if self._pipeline:
            self._pipeline.set_property('video-sink', self._video_sink)


    def uri__set(self, uri):
        if not self._pipeline:
            self._create_pipeline()

        self._current_uri = uri
        self._pipeline.set_state(gst.STATE_READY)
        self._state = STATES.STOPPED
        states = self._pipeline.get_state(3000 * gst.MSECOND)

        if states[1] == gst.STATE_READY:
           
            # quote the uri before passing it to playbin
            quoted_uri = MediaUri(uri)
            quoted_uri.path = urllib.quote(uri.path.encode('utf8'))
    
            self.debug("Loading %r", str(quoted_uri))
    
            self._pipeline.set_property('uri', str(quoted_uri))
    
            self.debug("Ready to play %r", str(quoted_uri))

    def volume__set(self, volume):
        self._pipeline.set_property('volume', volume)

    def volume__get(self):
        return self._pipeline.get_property('volume')


    # Internal methods

    def _create_pipeline(self):
        self._pipeline = gst.element_factory_make('playbin', 'playbin')
        pbus = self._pipeline.get_bus()
        pbus.connect("message", self._bus_message_cb)
        pbus.add_signal_watch()

        self._pipeline.set_property('video-sink', self._videosink)
        self._pipeline.set_property('audio-sink', self._audiosink)
        self._pipeline.set_state(gst.STATE_READY)

    def _bus_message_cb(self, bus, message):
        # FIXME: handle audio/video codec not found
        if message.type == gst.MESSAGE_EOS:
            self._on_eos()
        elif message.type == gst.MESSAGE_BUFFERING:
            # Message_Buffering only exists in gstreamer >= 0.10.11
            try:
                percent = message.parse_buffering()
                if percent < 100:
                    self.pause()
                else:
                    self.play()

                self._send_msg(PlayerBuffering(percent))
            except AttributeError:
                self.info("You won't get a buffering message. This is only"
                          " working with the python bindings >= 0.10.8!")
        elif message.type == gst.MESSAGE_ERROR:
            err, msg = message.parse_error()
            self.warning("Gstreamer %s:%s" % (err, msg))
            self._send_msg(PlayerError((err, msg)))
            self._pipeline.set_state(gst.STATE_READY)

        elif message.type == gst.MESSAGE_STATE_CHANGED:
            old_state, new_state, pending = message.parse_state_changed()
            if new_state == gst.STATE_PLAYING:
                # We have a new Basetime
                base_time = self._pipeline.get_base_time()
                self._send_msg(NewBaseTime(base_time))
                if self._state != STATES.PLAYING:
                    self._state = STATES.PLAYING
                    self._send_msg(PlayerPlaying())
            elif new_state == gst.STATE_NULL or \
                        new_state == gst.STATE_READY:
                self._state = STATES.STOPPED
            elif new_state == gst.STATE_PAUSED:
                self._state = STATES.PAUSED    

        elif message.type == gst.MESSAGE_NEW_CLOCK:
            # Clock was updated
            clock = message.parse_new_clock()
            self._send_msg(NewClock(clock))

    def _on_eos(self):
        if self.state != STATES.STOPPED:
            self._state = STATES.STOPPED
            self._send_msg(PlayerStopping())
        self.info("End of stream reached")

