# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

"""
LocalMediaProvider component class
"""


__maintainer__ = 'Florian Boucault <florian@fluendo.com>'
__maintainer2__ = 'Philippe Normand <philippe@fluendo.com>'

from elisa.base_components.media_provider import MediaProvider, UriNotMonitorable
from elisa.base_components.media_provider import NotifyEvent
from elisa.core.media_uri import MediaUri
from elisa.core.media_file import MediaFile
from elisa.core.utils import misc
from elisa.core import common
from elisa.extern import path, natural_sort

try:
    from elisa.extern.coherence import inotify
except ImportError:
    inotify = None

import os
import shutil
import re

from twisted.internet import defer, threads


class LocalMedia(MediaProvider):
    """
    This class implements local filesystem support
    """

    hidden_file_pattern = re.compile("\..*")
    hidden_fnmatch_pattern = '[!.]*'

    def __init__(self):
        MediaProvider.__init__(self)
        self._paths = {}

    def supported_uri_schemes__get(self):
        return { 'file': 0 }

    def blocking_get_media_type(self, uri):
        file_type = ''
        mime_type = ''
        if not self.blocking_is_directory(uri):
            mime_type, file_type = misc.get_media_infos_from_mime(uri)
            if None in (mime_type, file_type):
                self.warning("Unknown media: %r", uri)
        else:
            file_type = 'directory'

        return {'file_type': file_type, 'mime_type': mime_type}

    def blocking_is_directory(self, uri):
        return os.path.isdir(uri.path)


    def has_children_with_types(self, uri, media_types):
        d = threads.deferToThread(self.blocking_has_children_with_types,
                                  uri, media_types)
        return d

    def blocking_has_children_with_types(self, uri, media_types):
        has_children = False
        if self.blocking_is_directory(uri):
            try:
                contents = os.listdir(uri.path)
            except OSError, error:
                self.info("Could not list directory %r: %s", path, error)
            else:
                while contents and not has_children:
                    filename = contents.pop()
                    try:
                        child_uri = uri.join(filename)
                    except UnicodeDecodeError:
                        ### give a warning here?
                        continue
                    media_type = self.blocking_get_media_type(child_uri)
                    if media_types:
                        has_children = media_type['file_type'] in media_types
                    else:
                        has_children = True
        return has_children

    def blocking_get_direct_children(self, uri, list_of_children):

        if self.blocking_is_directory(uri):
            try:
                contents = os.listdir(uri.path)
            except OSError, error:
                self.info("Could not list directory %r: %s" % (uri.path,
                                                               error))
            else:

                # sort directory contents. folders, then files alphanumm sorted
                folders = []
                files = []
                for fname in contents:
                    try:
                        full_path = os.path.join(uri.path, fname)
                    except UnicodeDecodeError:
                        self.warning("The file %r is encoded wrong. We can"
                                     " not handle this encoding. Please"
                                     " rename the file in the folder %s!"  %
                                     (fname, uri.path))
                        continue
                    if os.path.isdir(full_path):
                        folders.append(fname)
                    else:
                        files.append(fname)

                key = lambda item: item.lower()
                folders = natural_sort.obj_natsorted(folders,
                                                     natural_sort.natcasecmp,
                                                     key)
                files = natural_sort.obj_natsorted(files,
                                                   natural_sort.natcasecmp,
                                                   key)
                contents = folders + files

                for fname in contents:
                    if uri.path.endswith('/'):
                        full_path = uri.path + fname
                    else:
                        full_path = "%s/%s" % (uri.path, fname)
                    if self.hidden_file_pattern.match(fname) or \
                           not os.path.exists(full_path):
                        continue
                    parts = { 'scheme': 'file', 'path': full_path }
                    child_uri = MediaUri(parts)
                    list_of_children.append((child_uri, {}))

        return list_of_children


    def get_direct_children(self, uri, l):
        d = threads.deferToThread(self.blocking_get_direct_children, uri, l)
        return d


    def blocking_open(self, uri, mode='r'):
        ret = None
        if not self.blocking_is_directory(uri):
            try:
                handle = open(uri.path, mode)
            except Exception, exc:
                self.info("Could not open %s : %s" % (uri, exc))
            else:
                ret = MediaFile(self, handle)
        return ret

    def next_location(self, uri, root=None):
        return threads.deferToThread(self.blocking_next_location, uri,
                                   root=root)

    def blocking_next_location(self, uri, root=None):
        next_uri = None

        local = uri.path

        if os.path.isdir(local):
            dir_path = local
        else:
            dir_path = os.path.dirname(local)

        root_path = None
        walk_gen = None
        if root:
            root_path = root.path
            walk_gen = self._paths.get(root_path)
        else:
            for walker_root, walker in self._paths.iteritems():
                if local.startswith(walker_root):
                    walk_gen = walker
                    root_path = walker_root
                    break

        if not root_path:
            root_path = dir_path

        end = False
        if not walk_gen:
            p = path.path(root_path)
            walk_gen = p.walk(self.hidden_fnmatch_pattern, errors='ignore')
            self._paths[root_path] = walk_gen
            if root_path != local:
                try:
                    n = walk_gen.next()
                except StopIteration:
                    end = True
                else:
                    while n != local:
                        try:
                            n = walk_gen.next()
                        except StopIteration:
                            del self._paths[root_path]
                            end = True
                            break

        if not end:
            try:
                next_file = walk_gen.next()
            except StopIteration:
                del self._paths[root_path]
                next_file = None

            if next_file:
                parts = {'scheme': 'file', 'path': next_file}
                next_uri = MediaUri(parts)

        self.debug("Next URI of %s: %s", uri, next_uri)
        return next_uri

    def previous_location(self, uri):
        return threads.deferToThread(self.blocking_previous_location, uri)

    def blocking_previous_location(self, uri):
        if uri == None:
            return None

        if os.path.isdir(uri.path):
            dir_path = uri.path
        else:
            dir_path = os.path.dirname(uri.path)

        p = path.path(dir_path)
        walk_gen = p.walk(self.hidden_fnmatch_pattern, errors='ignore')

        p = dir_path
        if dir_path != uri.path:
            while True:
                try:
                    n = walk_gen.next()
                    if n == uri.path:
                        break
                    else:
                        p = n
                except StopIteration:
                    return None

        if p == uri.path:
            return None
        if p:
            parts = {'scheme': 'file', 'path': p}
            return MediaUri(parts)
        return None

    def monitor_uri(self, uri, callback, *extra_args):

        def inotify_cb(iwp, filename, mask, parameter):
            media_uri = MediaUri(unicode("file://%s/%s" % (iwp.path, filename)))
            event = self._inotify_event_type(mask)
            return callback(media_uri, {}, event,uri, *extra_args)

        if self.uri_is_monitorable(uri):
            try:
                inotify_instance = inotify.INotify()
            except SystemError, error:
                self.warning(error)
                raise UriNotMonitorable(uri)
            else:
                path = uri.path.encode('utf-8')

                self.info("Starting to monitor %r", path)
                try:
                    inotify_instance.watch(path, auto_add = False, \
                                           recursive=False, \
                                           mask = inotify.IN_CREATE \
                                           | inotify.IN_DELETE \
                                           | inotify.IN_ATTRIB \
                                           | inotify.IN_MOVED_TO \
                                           | inotify.IN_MOVED_FROM, \
                                           callbacks=(inotify_cb, None))
                except IOError, error:
                    raise UriNotMonitorable(uri)
        else:
            raise UriNotMonitorable(uri)

    def _inotify_event_type(self, mask):
        if mask & inotify.IN_CREATE or mask & inotify.IN_MOVED_TO:
            return NotifyEvent.ADDED
        elif mask & inotify.IN_DELETE or mask & inotify.IN_UNMOUNT or mask & inotify.IN_MOVED_FROM:
            return NotifyEvent.REMOVED
        elif mask & inotify.IN_ATTRIB:
            return NotifyEvent.MODIFIED
        else:
            # FIXME: exception should be raised
            inotify_instance = inotify.INotify()
            human = inotify_instance.flag_to_human(mask)
            self.info("Unexpected inotify event: %s" % human)
            return NotifyEvent.REMOVED


    def unmonitor_uri(self, uri):
        self.debug("Stopping %r monitoring", uri)
        inotify.INotify().ignore(uri.path)


    def uri_is_monitorable(self, uri):
        monitorable = False
        home = os.path.expanduser('~')
        if inotify != None:
            inotify_instance = inotify.INotify()
            try:
                inotify_instance.watch(home)
            except UriNotMonitorable:
                monitorable = False
            else:
                monitorable = True
                inotify_instance.ignore(home)
        return monitorable

    def uri_is_monitored(self, uri):
        return inotify and inotify.INotify().is_watched(uri.path)

    def copy(self, orig_uri, dest_uri, recursive=False):
        try:
            if recursive:
                shutil.copytree(orig_uri.path, dest_uri.path)
            else:
                shutil.copy(orig_uri.path, dest_uri.path)
        except Exception, e:
            raise e

    def move(self, orig_uri, dest_uri):
        try:
            shutil.move(orig_uri.path, dest_uri.path)
        except shutil.Error, e:
            raise e


    def delete(self, uri, recursive=False):
        try:
            if recursive:
                shutil.rmtree(uri.path)
            else:
                os.remove(uri.path)
        except shutil.Error, e:
            raise e

