# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


__maintainer__ = 'Lionel Martin <lionel@fluendo.com>'
__maintainer2__ = 'Florian Boucault <florian@fluendo.com>'

from elisa.core.component import ComponentError
from elisa.base_components.controller import Controller
from elisa.base_components.model import Model
from elisa.core.observers.list import ListObservable, ListObserver
from elisa.core import common

class ListController(Controller, ListObservable, ListObserver):
    """
    DOCME

    @ivar current_index:  index of the currently selected item in the list
    @type current_index:  int
    """

    supported_models = ('base:list_model')

    default_associations = (
    ('base:list_model','base:list_controller'),
    )

    def __init__(self):
        Controller.__init__(self)
        ListObservable.__init__(self)
        ListObserver.__init__(self)
        
        self._current_index = -1

    def __repr__(self):
        return "<%s current_index=%r %s items>" % (self.__class__.__name__,
                                                   self._current_index,
                                                   len(self))
        
    def current_index__set(self, index):
        if index >= 0 and index < len(self):
            self.debug("current_index set to %d" % index)
            self._current_index = index

    def current_index__get(self):
        # FIXME: this does not seem right
        if self._current_index < 0:
            return 0
        return self._current_index


    def inserted(self, elements, position):
        """
        DOCME

        called when a child_model is inserted in controller model
        """            
        current_position = position
        for child_model in elements:
            self._add_child_controller_in_list(child_model, current_position)
            current_position += 1
        
        if self.current_index >= position:
            self.current_index += len(elements)

    def removed(self, elements, position):
        """
        DOCME

        called when a child_model is removed in controller model
        """
        self.debug("%s got a REMOVED at %s for %s elements" % (self, position,
                                                                len(elements)))
        end = position+len(elements)
        # current index needs updating
        if self.current_index >= position:
            # current index is set to the end of the remaining children
            if end >= len(self):
                self.current_index = position - 1
            else:
                # current index still points to the same item
                if self.current_index >= end:
                    self.current_index -= len(elements)
                else:
                # current index is set to the child after the removed ones
                    self.current_index = position

        for elem in elements:
            if position >= len(self):
                break
            to_del = self[position]
            self.debug("%s performs a remove of %s" % (self, to_del))
            self.remove(to_del)

    def loadmore(self):
        """
        DOCME

        ask to the activity to load items to the model
        """
        if self.model.activity != None:
            self.model.activity.loadmore(self.model)


    def _add_child_controller_in_list(self, model, position):
        """
        @raise ComponentError: When the controller cannot be created
        """
        controller = None
        if isinstance(model, Model):
            pr = common.application.plugin_registry
            controller_path_list = self.get_controller_paths(model.path)
            if len(controller_path_list) == 0:
                self.warning("Cannot create controller for model %r (with controller %r)" % (model.path, self.path))
                raise ComponentError(model.path)
            else:
                controller = pr.create_component(controller_path_list[0])
                controller.parent = self
                controller.backend = self.backend
                controller.model = model
                self.insert(position, controller)

    def model_changed(self, old_model, new_model):
        """
        DOCME
        """
        Controller.model_changed(self, old_model, new_model)

        if new_model == None:
            # FIXME: might be useful to remove all the children
            return

        # create child controller for existing model
        current_position = 0
        for child_model in self.model:
            self._add_child_controller_in_list(child_model, current_position)
            current_position += 1


    def backend__set(self, value):
        Controller.backend__set(self, value)
        for child_controller in self:
            if isinstance(child_controller, Controller):
                child_controller.backend=value


    def append(self, element):
        if isinstance(element, Controller):
            element.parent = self
            element.backend = self.backend
        ListObservable.append(self, element)


    def extend(self, elements):
        for element in elements:
            if isinstance(element, Controller):
                element.parent = self
                element.backend = self.backend
        ListObservable.extend(self, elements)


    def insert(self, position, element):
        if isinstance(element, Controller):
            element.parent = self
            element.backend = self.backend
        ListObservable.insert(self, position, element)


    def pop(self, position=-1):
        element = ListObservable.pop(self, position)
        if isinstance(element, Controller):
            element.parent = None
            element.backend = None


    def remove(self, element):
        if isinstance(element, Controller):
            element.model = None
            element.parent = None
            element.backend = None

            if isinstance(element, ListController):
                self.debug("%s saying %s to remove the children" % (self,
                                                            element))
                self._remove_children(element)

        ListObservable.remove(self, element)

    def _remove_children(self, parent):
        # Does anyone know why for child in parent does not work correctly?
        while len(parent):
            child = parent[0]
            self.debug("%s:%s should remove %s" %  (self, parent, child))
            parent.remove(child)

    def __setitem__(self, key, value):
        previous_value = self[key]
        if isinstance(value, Controller):
            value.parent = self
            value.backend = self.backend
        ListObservable.__setitem__(self, key, value)
        if isinstance(previous_value, Controller):
            previous_value.parent = None
            previous_value.backend = None


    def __delitem__(self, key):
        if len(self)>0 and key >=0 and key < len(self):
            element = self[key]
            ListObservable.__delitem__(self, key)
            if isinstance(element, Controller):
                element.parent = None
                element.backend = None
                if isinstance(element, ListController):
                    self._remove_children(element)


    def __setslice__(self, i, j, sequence):
        removed_elements = self[i:j]
        for element in sequence:
            if isinstance(element, Controller):
                element.parent = self
                element.backend = self.backend

        ListObservable.__setslice__(self, i, j, sequence)

        for element in removed_elements:
            if isinstance(element, Controller):
                element.parent = None
                element.backend = None
                if isinstance(element, ListController):
                    self._remove_children(element)


    def __delslice__(self, i, j):
        removed_elements = self[i:j]
        ListObservable.__delslice__(self, i, j)
        for element in removed_elements:
            if isinstance(element, Controller):
                element.parent = None
                element.backend = None
                if isinstance(element, ListController):
                    self._remove_children(element)
