# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

"""
List with observable functionality
"""


from observable import Observable
from observer import Observer
from elisa.core.utils.threadsafe_list import ThreadsafeList

class ListObservable(Observable, ThreadsafeList):
    """
    This Class is implementing a List in L{Observable}. It has all
    functionality a normal L{list} also has but informs all its observers
    about the changes.

    Attention: If you want to add your L{Observer} to this L{Observable},
    please be sure that it is at least implementing all methods of
    L{ListObserver}
    """

    # FIXME: this class should be made thread-safe

    def __init__(self, *args, **kw):
        ThreadsafeList.__init__(self, *args, **kw)
        Observable.__init__(self)

    def __hash__(self):
        return Observable.__hash__(self)

    def __repr__(self):
        return "<ListObservable %r>" % ThreadsafeList.__repr__(self)

    def __eq__(self, other):
        return id(self) == id(other)

    def append(self, element):
        """
        Append element to the list. Put the element to the end of the list.
        Inform the observers about it by doing L{ListObserver.inserted}.
        @param element: the element to insert
        """
        position = len(self)
        ThreadsafeList.append(self, element)
        for observer in self._observers:
            observer.inserted([element], position)

    def extend(self, elements):
        """
        Extend the list with these elements to. Put the elements to the end of
        the list. Inform the observers about it by doing  L{ListObserver.inserted}.
        @param elements: the elements to insert
        @type elements: list
        """       
        position = len(self)
        ThreadsafeList.extend(self, elements)
        for observer in self._observers:
            observer.inserted(elements, position)

    def insert(self, position, element):
        """
        Insert this element to the list at the certain position. Inform the
        observers about it by doing  L{ListObserver.inserted}.
        @param position: the position the element should be added at
        @type position: positive int

        @param element: the element to insert
        """
        ThreadsafeList.insert(self, position, element)
        for observer in self._observers:
            observer.inserted([element], position)

    def pop(self, position=-1):
        """
        Return the element at position and remove it from the list. Inform the
        observers about it by doing L{ListObserver.removed}.

        If no position is specified the last element is returned and removed.

        @param position: the position of the element
        @type position: positive int

        @return: the element at this position
        """ 
        ret = ThreadsafeList.pop(self, position)
        for observer in self._observers:
            observer.removed([ret], position)
        return ret

    def remove(self, element):
        """
        Remove the element from the list. Inform the observers about it by
        doing L{ListObserver.removed}.
        """ 
        position = self.index(element)
        for observer in self._observers:
            observer.removed([element], position)
        ThreadsafeList.remove(self, element)

    def reverse(self):
        """
        Reverse the list. Inform the observers after about it by doing
        L{ListObserver.dirtied}.
        """
        ThreadsafeList.reverse(self)
        for observer in self._observers:
            observer.dirtied()

    def sort(self, *args, **kwargs):
        """
        See L{ThreadsafeList.sort}. After that inform the observers about it
        by doing L{ListObserver.dirtied}.
        """
        ThreadsafeList.sort(self, *args, **kwargs)
        for observer in self._observers:
           observer.dirtied()

    def __iadd__(self, elements):
        """
        Add the elements to the end of the list. Inform the observers about it
        by doing L{ListObserver.inserted} afterwards.

        @param elements: the elements to add
        @type elements: list
        """
        position = len(self)
        ret = ThreadsafeList.__iadd__(self, elements)
        for observer in self._observers:
            observer.inserted(elements, position)
        return ret

    def __imul__(self, mult):
        """
        Multiplice this to the list. Inform the observers about it
        by doing L{ListObserver.inserted} afterwards.

        @param mult: to multiplice it with
        """ 
        position = len(self)
        elements = self[0:len(self)]
        elements *= mult - 1
        for observer in self._observers:
            observer.inserted(elements, position)
        return ThreadsafeList.__imul__(self, mult)

    def __setitem__(self, key, value):
        """
        Set the item at key to value. Inform the observers about it
        by doing L{ListObserver.modified} afterwards.

        @param key: the place in the list
        @type key: positive int

        @param value: the value to set it to
        """ 
        ThreadsafeList.__setitem__(self, key, value)
        for observer in self._observers:
           observer.modified(key, value)

    def __delitem__(self, key):
        """
        Delete the item at postion key. Inform the observers about it by doing
        L{ListObserver.removed} before.
        @param key : the position of the element to delete
        @type key: positiv int
        """
        element = self[key]
        for observer in self._observers:
            observer.removed([element], key)
        ThreadsafeList.__delitem__(self, key)

    def __setslice__(self, i, j, sequence):
        """
        Set the element from the position i until j to sequence. Inform the
        observers about it by doing a L{ListObserver.removed} on the old
        elements in the list and then a L{ListObserver.inserted} after setting
        it in the list.

        @param i: the position of first element to replace
        @type i: positive int

        @param j: the position of the last element to replace
        @type j: positive int

        @param sequence: the elements to put there
        @type sequence: list
        """
        removed_elements = self[i:j]
        ThreadsafeList.__setslice__(self, i, j, sequence)
        for observer in self._observers:
            observer.removed(removed_elements, i)
            observer.inserted(sequence, i)

    def __delslice__(self, i, j):
        """
        Delete all elements from i to j. Inform the observers about it by
        doing L{ListObserver.removed} before.
        @param i: the position of the first element to delete
        @type i: positive int
        @param j: the position of the last element to delete
        @type j: positive int
        """
        removed_elements = self[i:j]
        for observer in self._observers:
            observer.removed(removed_elements, i)
        ThreadsafeList.__delslice__(self, i, j)

class ListObserver(Observer):

    def inserted(self, elements, position):
        """
        This is triggered by the L{ListObservable}, when a new elements are inserted.
        @param elements: the elements, that has been inserted
        @type elements: list
        @param position: the start position, where the elements have been
                        inserted at
        @type position: positive int
        """
        raise NotImplementedError("method inserted not implemented by %r" % type(self))

    def removed(self, elements, position):
        """
        This is triggered by the L{ListObservable}, when it'll to remove some elements.
        @param elements: the elements, that will be removed
        @type elements: list
        @param position: the start position, where the elements have been
                        removed at
        @type position: positive int
        """
        raise NotImplementedError("method removed not implemented by %r" % type(self))

    def modified(self, key, value):
        """
        This is triggered by the L{ListObservable}, when an element has been
        modified.
        @param key: the position of the modified element
        @type key: positive integer
        @param value: the value this element was replaced by
        """
        raise NotImplementedError("method modified not implemented by %r" % type(self))

    def dirtied(self):
        """
        This is triggered by the L{ListObservable}, when the list has totally
        changed. For example after doing a reverse or a sort.
        If you need the whole list, you better reread it from the
        L{ListObservable} here!
        """
        raise NotImplementedError("method dirtied not implemented by %r" % type(self))

if __name__ == "__main__":

    l = ListObservable()
    o = ListObserver()

    o.observe(l)

    print l
    l.append(3)
    print l
    l.extend([44, 4, 78, 18])
    print l
    l.insert(0, 15)
    print l
    l.pop(0)
    print l
    l.remove(3)
    print l
    l.reverse()
    print l
    l.sort()
    print l
    l += [12, 29]
    print l
    l *= 4
    print l
    l[2] = 555
    print l
    del l[2]
    print l
    l[0:3] = []
    print l
    del l[:]
    print l
