# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

"""
Database powered media sources caching
"""


__maintainer__ = 'Philippe Normand <philippe@fluendo.com>'


from elisa.core import log
from elisa.core import common
from elisa.core import media_uri
from elisa.extern import enum
from elisa.core.utils import classinit, signal, threadsafe_list

import thread, threading, time
from twisted.internet import threads, defer, reactor
from twisted.internet import protocol
from cPickle import dump, load
import os

MediaEvents = enum.Enum('NONE', 'ADDED', 'REMOVED', 'UPDATED')

SAVED_STATE_FILE = os.path.expanduser('~/.elisa/media_scanner.state')

class MediaScanner(log.Loggable):
    """
    The MediaScanner handles a list of media sources declared in the
    Application's config by Activities (audio, video, pictures, ...),
    is able to scan them (via the MediaManager's MediaProviders),
    retrieve metadata of the found medias using the MetadataManager
    and store everything in a database.

    The scanner maintains the database up-to-date by periodically
    rescanning media sources. Media sources can also be monitored if
    related MediaProviders support monitoring.

    @ivar _queue:            a queue where pending media sources are stored
                             during media scanning
    @type _queue:            list
    @ivar _media_manager:    the MediaManager which has created the MediaScanner
    @type _media_manager:    L{elisa.core.media_manager.MediaManager}
    @ivar _metadata_manager: the MetadataManager responsible to fetch metadata
                             information for medias
    @type _metadata_manager: L{elisa.core.metadata_manager.MetadataManager}
    """

    # Allows property fget/fset/fdel/doc overriding
    __metaclass__ = classinit.ClassInitMeta
    __classinit__ = classinit.build_properties

    source_updated = signal.Signal('media-source-updated', media_uri.MediaUri)

    def __init__(self, media_manager, metadata_manager, config=None):
        """ The media_scanner requires access to the media_manager to
        access its media_provider components when it scans media
        sources.

        @param media_manager:    the MediaManager which has created the
                                 MediaScanner
        @type media_manager:     L{elisa.core.media_manager.MediaManager}
        @param metadata_manager: the MetadataManager responsible to fecth
                                 metadata information from medias
        @type metadata_manager:  L{elisa.core.metadata_manager.MetadataManager}

        DOCME: config
        """
        log.Loggable.__init__(self)
        self.debug("Creating")
        
        if config:
            self._config = config
        elif common.application:
            self._config = common.application.config.get_section('media_scanner',
                                                                 {})
        else:
            self._config = {}

        self._auto_resume = False
        self.delayed_start = 10
        self._current_scan = ()
        self._queue = threadsafe_list.ThreadsafeList()
        self._media_manager = media_manager
        self._metadata_manager = metadata_manager

        self._update_intervals = dict(fivemin=60*5,
                                      hour = 60*60, day = 60* 60 * 24,
                                      week = 60 * 60 * 24 * 7)
        self._main_loop_lock = threading.Lock()
        self._main_loop_running = False

        self._fivemin_locations = self._get_option('fivemin_location_updates',[])
        self._hourly_locations = self._get_option('hourly_location_updates',[])
        self._daily_locations = self._get_option('daily_location_updates',[])
        self._weekly_locations = self._get_option('weekly_location_updates',[])
        self._unmonitored_locations = self._get_option('unmonitored_locations',[])
        self._ignored_locations = [ unicode(media_uri.MediaUri(l))
                                    for l in self._get_option('ignored_locations',[])]

        self._source_updates = {}
        self._source_callbacks = {} # mapping source ids to callbacks
        self._delayed_calls = {}

    def _get_option(self, name, default=None):
        return self._config.get(name, default)

    def _set_option(self, name, value):
        self._config[name] = value

    def save_config(self):
        """
        DOCME
        """
        self.debug("Saving config")
        self._set_option('fivemin_location_updates', self._fivemin_locations)
        self._set_option('hourly_location_updates', self._hourly_locations)
        self._set_option('daily_location_updates',self._daily_locations)
        self._set_option('weekly_location_updates',self._weekly_locations)
        self._set_option('unmonitored_locations',self._unmonitored_locations)

        common.application.config.set_section('media_scanner', self._config)

    def enabled__get(self):
        """
        """
        return int(self._get_option('enabled', '0'))

    def get_metadata(self, metadata):
        """
        DOCME: this method should probably not be public, should it?
        """
        return self._metadata_manager.get_metadata(metadata)


    def start(self, auto_resume=False):
        """ Start the sources scanning loop. The queue
        will then be parsed and when each source to scan will
        be processed.

        DOCME: auto_resume
        """
        if self.enabled:
            self.debug("Starting")
            self._auto_resume = auto_resume
            self._load_state()
            self._main_loop_running = True
            self._start_thread()
            self._schedule_updates()

    def stop(self):
        """ Stop the sources scanning loop. Any source being processed
        stops to be so.
        """
        self.info("Stopping")
        self._main_loop_lock.acquire()
        if self._main_loop_running:
            self._save_state()
        self._main_loop_running = False
        self._main_loop_lock.release()

        for label, call in self._delayed_calls.iteritems():
            call.cancel()

        self._delayed_calls = {}
        self.save_config()

    def add_source(self, source, media_types, callback=None):
        """
        Add a new source in database. If it's already there, just mark it as
        available again (and start monitoring it).

        @param source_uri: The location of the source
        @param callback:   Function called by the MediaScanner when it needs to
                           process files and directories
        @type callback:    callable(media_uri, event_type, file_type, media_types)
        """
        if self.enabled:
            self._source_callbacks[source.id] = (callback, media_types)
            self._start_source_update(source)

    def remove_source(self, source_uri):
        """
        Mark a source as unavailable in the database and stop monitoring it.

        @param source_uri: The location of the source
        @type source_uri:  L{elisa.core.media_uri.MediaUri}
        """
        # TODO: cancel an eventual update of the source
        pass

    def update_source(self, source_uri):
        """
        Schedule a new scan of the source located at given uri. The
        scan may not start just after calling this method.

        @param source_uri: The location of the source
        @type source_uri:  L{elisa.core.media_uri.MediaUri}
        """
        if self.enabled:
            source = self._media_manager.get_source_for_uri(source_uri)
            if source:
                self._start_source_update(source)

    def add(self, source, uri):
        """
        DOCME
        """
        is_directory = self._media_manager.blocking_is_directory(uri)
        if not is_directory:
            func = self._process_media_file

            callback, media_types = None, []
            source_cb = self._source_callbacks.get(source.id)
            if source_cb:
                callback, media_types = source_cb

            func(source, media_types, callback, uri)

    def remove(self, source, uri):
        db = self._media_manager.media_db
        media = db.get_media_information(uri, extended=False)
        if media:
            db.del_media_node(media)
        
        
    def _start_source_update(self, source):
        if source.uri not in self._ignored_locations:

            callback, media_types = None, []
            source_cb = self._source_callbacks.get(source.id)
            if source_cb:
                callback, media_types = source_cb
            
            uri = media_uri.MediaUri(source.uri)
            if self._media_manager.is_scannable(uri):
                if source.uri in self._source_updates.keys():
                    self.warning("Source %r is already being updated",
                                 source.uri)
                else:
                    self.info("Starting update of %r", source.uri)
                    self._media_manager.media_db.prepare_source_for_update(source)

                    self._enqueue((source, media_types, callback, None))
                        
                    if not self._is_running():
                        self._start_thread()
        else:
            self.info("Ignoring update of %r", source.uri)

    def _start_thread(self):
        self.log("Starting thread")
        try:
            self.log("Acquiring main loop lock")
            self._main_loop_lock.acquire()
            running = self._main_loop_running
            if not running:
                self._main_loop_running = True
                self.debug("Starting a new thread ... dum dee dum")
                #reactor.callInThread(self._process_queue)
                thread.start_new_thread(self._process_queue,())
        finally:
            self.log("Releasing main loop lock")
            self._main_loop_lock.release()
        
    def _stop_thread(self):
        self.debug("Ending thread")
        try:
            self._main_loop_lock.acquire()
            self._main_loop_running = False
        finally:
            self._main_loop_lock.release()
        
    def _is_running(self):
        running = False
        try:
            self._main_loop_lock.acquire()
            running = self._main_loop_running
        finally:
            self._main_loop_lock.release()
        return running

    def _save_state(self):
        current = self._current_scan
        if current and self._auto_resume:
            source_uri = current[0]
            media_type = current[1]
            uri = current[2]
            if uri is not None:
                self.debug("Saving scan state %r", current)
                f = open(SAVED_STATE_FILE, 'w')
                dump((source_uri, media_type, unicode(uri)), f)
                f.close()

    def _load_state(self):
        item = None
        if os.path.exists(SAVED_STATE_FILE) and self._auto_resume:
            f = open(SAVED_STATE_FILE,'r')
            
            try:
                item = load(f)
            except EOFError:
                f.close()
            if item:
                source_uri, media_type, uri = item
                source = self._media_manager.get_source_for_uri(source_uri)
                if source:
                    self.debug("Resuming scan at %r", item)
                    uri = media_uri.MediaUri(uri)
                    item = (source, media_type, None, uri)
                    self._enqueue(item)

    def _enqueue(self, item):
        self.log("enqueing %r", item)
        self._queue.insert(0, item)
        self.log("%r items in queue", len(self._queue))
                 
    def _dequeue(self):
        try:
            item = self._queue.pop(0)
        except IndexError:
            item = None
        return item

    def _process_queue(self):
        self.debug("Delayed start in %s seconds", self.delayed_start)
        while self.delayed_start > 0:
            if not self._is_running():
                self.debug("Am running, dude..")
                break
            time.sleep(1.)
            self.delayed_start -= 1
            self.debug("Starting in %s seconds", self.delayed_start)
            
        while 1:
            if not self._is_running():
                break

            item = self._dequeue()
            if item:

                source = item[0]
                current_uri = item[-1]
                
                if not current_uri:
                    # first time we're processing this source
                    self._source_updates[source.uri] = time.time()

                try:
                    next_location = self._process_source(*item)
                except Exception, exc:
                    self.warning("Source parsing failed: %s" % exc)
                    # FIXME: this is wrong(tm)
                    raise
                    
                self._current_scan = (source.uri, item[1], next_location)
                
                if next_location:
                    new_item = list(item)
                    new_item[-1] = next_location
                    self._enqueue(new_item)
                else:
                    self._source_updated(source)
            else:
                self._stop_thread()
                break
            
            # keeps CPU available when queue empty
            # FIXME: please do not keep a busy loop like that
            # better doing what's done in deferred_action to avoid thta
            # must be >=10, for low hard disk (phlin pc for example :) )
            time.sleep(0.10)

        self._stop_thread()
        return None
    
    def _schedule_updates(self):
        # schedule some events
        for label, interval in self._update_intervals.iteritems():
            self._schedule(interval, label)

    def _schedule(self, interval, label):
        call = reactor.callLater(interval, self._scheduled_update, label)
        self._delayed_calls[label] = call

    def _scheduled_update(self, interval_label):
        interval = self._update_intervals.get(interval_label)
        self._schedule(interval, interval_label)
        interval_locations = {'fivemin': self._fivemin_locations,
                              'hour': self._hourly_locations,
                              'day': self._daily_locations,
                              'week': self._weekly_locations}
        sources = interval_locations.get(interval_label)
        if sources:
            msg = "Launching scheduled update of the %s: %s" % (interval_label,
                                                                sources)
            self.info(msg)
            for source_uri in sources:
                self.update_source(media_uri.MediaUri(source_uri))

    def _process_source(self, source, media_types, callback, current_uri=None):
        self.debug("Processing source %r", source.uri)
        source_uri = media_uri.MediaUri(source.uri)

        if not current_uri:
            current_uri = source_uri

        is_directory = self._media_manager.blocking_is_directory(current_uri)
        if not is_directory:
            self._process_media_file(source, media_types, callback,
                                     current_uri)
        next_location = self._media_manager.blocking_next_location(current_uri,
                                                                   root=source_uri)
        return next_location


    def _source_updated(self, source):
        """
        hide un-updated medias for the source in db
        eventually schedule a new update of the source
        """
        t1 = time.time()
        t0 = self._source_updates.get(source.uri)
        media_manager = self._media_manager
        if t0:
            del self._source_updates[source.uri]

            delta = t1 - t0
            count = media_manager.media_db.get_files_count_for_source(source)
            if count:
                speed = "(%s s/file)"  % (delta / count,)
            else:
                speed = ""
            msg = 'Parse of %s took %s seconds %s' % (source.uri, delta, speed)
        else:
            msg = 'Finished parsing %r' % source.uri
            
        self.info(msg)

        # scan un-updated medias of the source and hide them
        rows = media_manager.media_db.hide_un_updated_medias_for_source(source)

        callback, media_types = None, []
        source_cb = self._source_callbacks.get(source.id)
        if source_cb:
            callback, media_types = source_cb
        
        if callback:
            for media in rows:
                callback(media.uri, media.typ, media.format,
                         MediaEvents.REMOVED)

        self.source_updated.emit(media_uri.MediaUri(source.uri))

    def _process_media_file(self, source, media_types, callback, uri):
        self.debug("File processing %s for %s", uri, media_types)

        media_type = self._media_manager.blocking_get_media_type(uri)
        uri_media_type = media_type.get('file_type')
        if uri_media_type:
            if callback:
                callback(uri, 'file', uri_media_type, MediaEvents.NONE)

            if not media_types or uri_media_type in media_types:
                return self._try_insert(source, uri, 'file', uri_media_type)

    def _try_insert(self, source, uri, typ, media_type, **extra):
        self.debug("Try insert %r", uri)
        media = self._media_manager.get_media_information(uri,
                                                          extended=False,
                                                          media_type=media_type)

        callback, media_types = None, []
        source_cb = self._source_callbacks.get(source.id)
        if source_cb:
            callback, media_types = source_cb

        extra.update({'format': media_type})

        if not media:
            metadf = self.get_metadata({'uri': uri,
                                        'content-type': media_type,
                                        'artist' : None, 'album': None,
                                        'track': None, 'song': None})
            metadf.addCallback(self._do_callback_insert,
                               source, uri, media_type, typ, callback,
                               **extra)
        else:
            self.debug('Already in db: %s', uri)
            self._media_manager.media_db.update_media(media, updated=1)

    def _do_callback_insert(self, mdict, source, uri, media_type, typ,
                            callback, **extra):

        if mdict.has_key('content-type'):
            # The db only supports audio currently
            if mdict['content-type'] == 'audio':
                ndict = {}
                for key in ('artist','album', 'song', 'track'):
                    if key in mdict:
                        ndict[key] = mdict[key]
		extra.update({'metadata' : ndict})

        source_uri = source.uri
        if not source_uri.endswith('/'):
            source_uri += '/'

        parent = int(source.id)
        db = self._media_manager.media_db
        result = db.add_media(uri, uri.label, parent, typ, **extra)
        if result:
            event_type = MediaEvents.ADDED
        else:
            event_type = MediaEvents.UPDATED

        if callback:
            callback(uri, typ, media_type, event_type)
