# Elisa - Home multimedia server
# Copyright (C) 2006,2007 Fluendo, S.A. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

__maintainer__ = 'Philippe Normand <philippe@fluendo.com>'
__maintainer2__ = 'Florian Boucault <florian@fluendo.com>'


import threading

from elisa.core import log
from elisa.core.bus import bus_message
from elisa.core.utils.threadsafe_list import ThreadsafeList

from twisted.internet import reactor

class Bus(log.Loggable):
    """
    Python objects can register callbacks with the bus and be called when
    L{elisa.core.bus.bus_message.Message}s are sent by other objects.

    Here is a simple example::

        bus = Bus()

        def my_cb(message, sender):
            print 'Got message %r from %r' % (message, sender)

        bus.register(my_cb)

        bus.send_message(Message())


    Messages dispatching empties the message queue and call the registered
    callbacks. Messages filtering is also supported, just pass a Message
    type class to bus.register(), like this::

        bus.register(my_cb, Message)

    You can filter on multiple Message types by supplying a list to
    bus.register()::

        bus.register(my_cb, (FooMessage, DataMessage))

    @ivar queue:     list of messages not yet processed
    @type queue:     list
    @ivar callbacks: registered callbacks
    @type callbacks: dict, keys are callable objects and values are
                     Message types lists
    """

    def __init__(self):
        """Bus constructor. Initialize the Message queue and
        the callbacks dictionary.
        """
        log.Loggable.__init__(self)
        self.debug("Creating")

        # dispatch speed
        self._dispatch_speed = 1 / 10.

        # list of (message, sender) tuples
        self._queue = ThreadsafeList()

        # callbacks dictionary mapping callables to Message type lists
        self._callbacks = {}

        # threading lock for operations on self._callbacks
        self._callbacks_lock = threading.Lock()

        # handle on the message dispatching loop
        self._loop = None

    def start(self):
        """
        Start message dispatching: once started, messages sent over the bus
        are guaranteed to be dispatched.
        """
        self.info("Starting")
        # FIXME: using callLater (== polling) seems useless here: when
        # send_message gets called, the message could be immediately dispatched
        self._loop = reactor.callLater(self._dispatch_speed, self._dispatch)

    def stop(self):
        """
        Stop message dispatching: messages sent over the bus will not be
        dispatched automatically anymore.
        """
        self.info("Stopping")
        if self._loop != None and self._loop.active():
            self._loop.cancel()

    def send_message(self, message, sender=None):
        """Send a message over the bus.

        MT safe.

        @param message: the message to send
        @type message:  L{elisa.core.bus.bus_message.Message}
        @param sender:  the sender object. None by default. Will be passed to
                        receiver callbacks.
        @type sender:   object
        """
        assert isinstance(message, bus_message.Message), message
        self.debug("Sending message %r", message)

        self._queue.insert(0, (message, sender))

    def register(self, callback, *message_types):
        """
        Register a new callback with the bus. The given callback will be
        called when a message of one of the given types is dispatched on the
        bus.

        MT safe.

        @param callback:      the callback to register
        @type callback:       callable
        @param message_types: Message types to filter on
        @type message_types:  type or list of types
        """
        if not message_types:
            message_types = (bus_message.Message,)

        self.debug("Registering callback %r for %r message types",
                   callback, message_types)

        self._callbacks_lock.acquire()
        self._callbacks[callback] = message_types
        self._callbacks_lock.release()

    def unregister(self, callback):
        """Unregister a callback from the bus.

        MT safe.

        @param callback: the callback to register
        @type callback:  callable
        """
        self._callbacks_lock.acquire()
        if callback in self._callbacks:
            del self._callbacks[callback]
        self._callbacks_lock.release()

    def _dispatch_messages(self):
        """Dispatch messages to registered callbacks and empty the
        message queue.
        """

        while len(self._queue) > 0:
            dispatched = False
            message, sender = self._queue.pop()
            # broadcast to all registered entities
            for callback, mfilter in self._callbacks.iteritems():
                if isinstance(message, mfilter):
                    try:
                        cb_name = "%s.%s" % (callback.im_class.__name__,
                                             callback.__name__)
                    except AttributeError:
                        cb_name = callback.__name__
                    self.debug("Dispatching message %r to %r", message, cb_name)
                    callback(message, sender)
                    dispatched = True

            if not dispatched:
                self.debug("Undispatched message: %r", message)

    def _dispatch(self):
        self._dispatch_messages()
        self._loop = reactor.callLater(self._dispatch_speed, self._dispatch)


def bus_listener(bus, *message_types):
    """ Utility decorator to simply register a function or method on
    the message bus.
    """
    def decorator(func):
        bus.register(func, *message_types)

    return decorator

if __name__ == '__main__':
    log.init()

    class DataMessage(bus_message.Message):
        def __init__(self, data):
            bus_message.Message.__init__(self)
            self.data = data

    bus = Bus()

    @bus_listener(bus)
    def on_message(message, sender):
        print 'Got Message: %r' % message

    for i in range(5):
        data = '%s. Hello you' % i
        bus.send_message(DataMessage(data))

    bus._dispatch_messages()
