/*
***************************************************************************
*
* Author: Teunis van Beelen
*
* Copyright (C) 2010, 2011 Teunis van Beelen
*
* teuniz@gmail.com
*
***************************************************************************
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation version 2 of the License.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License along
* with this program; if not, write to the Free Software Foundation, Inc.,
* 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*
***************************************************************************
*
* This version of GPL is at http://www.gnu.org/licenses/old-licenses/gpl-2.0.txt
*
***************************************************************************
*/



#include "signalcurve.h"


SignalCurve::SignalCurve(QWidget *parent) : QWidget(parent)
{
  setAttribute(Qt::WA_OpaquePaintEvent);

  SignalColor = Qt::blue;
  tracewidth = 0;
  BackgroundColor = Qt::gray;
  RasterColor = Qt::darkGray;
  BorderColor = Qt::lightGray;
  RulerColor = Qt::black;
  TextColor = Qt::black;
  crosshair_1_color = Qt::red;

  Marker1Pen.setStyle(Qt::DashLine);
  Marker1Pen.setColor(Qt::yellow);

  dbuf = NULL;
  fbuf = NULL;
  ibuf = NULL;
  bordersize = 60;
  drawHruler = 1;
  drawVruler = 1;
  h_ruler_startvalue = 0.0;
  h_ruler_endvalue = 100.0;
  h_ruler_precision = 2;
  drawcurve_before_raster = 0;
  h_label[0] = 0;
  v_label[0] = 0;
  upperlabel1[0] = 0;
  upperlabel2[0] = 0;
  lowerlabel[0] = 0;
  max_value = 100.0;
  min_value = -100.0;
  extra_button = 0;
  extra_button_txt[0] = 0;
  use_move_events = 0;
  crosshair_1_active = 0;
  crosshair_1_moving = 0;
  crosshair_1_value = 0.0;
  crosshair_1_value_2 = 0.0;
  crosshair_1_x_position = 0;
  marker_1_position = 0.5;
  fillsurface = 0;

  cursorEnabled = TRUE;
  printEnabled = TRUE;
  dashBoardEnabled = TRUE;
  updates_enabled = TRUE;
  Marker1Enabled = FALSE;

  spectrum_color = NULL;

  old_w = 10000;
}


void SignalCurve::clear()
{
  dbuf = NULL;
  fbuf = NULL;

  bufsize = 0;

  use_move_events = 0;
  crosshair_1_active = 0;
  crosshair_1_moving = 0;
  crosshair_1_value = 0.0;
  crosshair_1_value_2 = 0.0;
  crosshair_1_x_position = 0;

  update();
}


void SignalCurve::mousePressEvent(QMouseEvent *event)
{
  int m_x,
      m_y;

  setFocus(Qt::MouseFocusReason);

  w = width();
  h = height();

  m_x = event->x() - bordersize;
  m_y = event->y() - bordersize;

  if(m_x < 0 ||
     m_x > (w - (bordersize * 2)) ||
     m_y < 0 ||
     m_y > (h - (bordersize * 2)))
  {
    return;
  }

  if(event->button()==Qt::LeftButton)
  {
    if(printEnabled == TRUE)
    {
      if((m_y<21)&&(m_y>3)&&(m_x>((w - (bordersize * 2)) - 43))&&(m_x<((w - (bordersize * 2)) - 3)))
      {
        exec_sidemenu();

        return;
      }
    }

    if(dashBoardEnabled == TRUE)
    {
      if((m_y<61)&&(m_y>43)&&(m_x>((w - (bordersize * 2)) - 43))&&(m_x<((w - (bordersize * 2)) - 3)))
      {
        emit dashBoardClicked();

        return;
      }
    }

    if(cursorEnabled == TRUE)
    {
      if((m_y<41)&&(m_y>23)&&(m_x>((w - (bordersize * 2)) - 43))&&(m_x<((w - (bordersize * 2)) - 3)))
      {
        if(crosshair_1_active)
        {
          crosshair_1_active = 0;
          crosshair_1_moving = 0;
          use_move_events = 0;
          setMouseTracking(FALSE);
        }
        else
        {
          crosshair_1_active = 1;
          if(!crosshair_1_x_position)
          {
            crosshair_1_value = 0.0;
            crosshair_1_x_position = (w - (bordersize * 2)) / 2;
            crosshair_1_y_position = (h - (bordersize * 2)) / 2;
            mouse_old_x = crosshair_1_x_position;
            mouse_old_y = crosshair_1_y_position;
          }
        }

        update();

        return;
      }
    }

    if(crosshair_1_active)
    {
      if((m_y<(crosshair_1_y_position + 15))&&(m_y>(crosshair_1_y_position - 25))&&(m_x>crosshair_1_x_position)&&(m_x<(crosshair_1_x_position + 65)))
      {
        crosshair_1_moving = 1;
        use_move_events = 1;
        setMouseTracking(TRUE);
        mouse_old_x = m_x;
        mouse_old_y = m_y;
      }

      if(m_x>(crosshair_1_x_position-10)&&(m_x<(crosshair_1_x_position + 10)))
      {
        crosshair_1_moving = 1;
        use_move_events = 1;
        setMouseTracking(TRUE);
        mouse_old_x = m_x;
        mouse_old_y = m_y;
      }
    }
  }
}


void SignalCurve::mouseReleaseEvent(QMouseEvent *)
{
  crosshair_1_moving = 0;
  use_move_events = 0;
  setMouseTracking(FALSE);
}


void SignalCurve::mouseMoveEvent(QMouseEvent *event)
{
  if(!use_move_events)
  {
    return;
  }

  mouse_x = event->x() - bordersize;
  mouse_y = event->y() - bordersize;

  if(crosshair_1_moving)
  {
    crosshair_1_x_position += (mouse_x - mouse_old_x);
    mouse_old_x = mouse_x;
    if(crosshair_1_x_position<2)
    {
      crosshair_1_x_position = 2;
    }
    if(crosshair_1_x_position>(w-(bordersize * 2) -40))
    {
      crosshair_1_x_position = w-(bordersize * 2) -40;
    }

    crosshair_1_y_position += (mouse_y - mouse_old_y);
    mouse_old_y = mouse_y;
    if(crosshair_1_y_position<25)
    {
      crosshair_1_y_position = 25;
    }
    if(crosshair_1_y_position>(h-(bordersize * 2) -25))
    {
      crosshair_1_y_position = h-(bordersize * 2) -25;
    }
  }

  update();
}


void SignalCurve::resizeEvent(QResizeEvent *event)
{
  if(crosshair_1_active)
  {
    crosshair_1_x_position *= ((double)(width() - (bordersize * 2)) / (double)(old_w - (bordersize * 2)));
  }

  QWidget::resizeEvent(event);
}


void SignalCurve::exec_sidemenu()
{
  sidemenu = new QDialog(this);

  if(extra_button)
  {
    sidemenu->setMinimumSize(QSize(120, 160));
    sidemenu->setMaximumSize(QSize(120, 160));
  }
  else
  {
    sidemenu->setMinimumSize(QSize(120, 130));
    sidemenu->setMaximumSize(QSize(120, 130));
  }
  sidemenu->setWindowTitle("Print");
  sidemenu->setModal(TRUE);
  sidemenu->setAttribute(Qt::WA_DeleteOnClose, TRUE);

  sidemenuButton1 = new QPushButton(sidemenu);
  sidemenuButton1->setGeometry(10, 10, 100, 20);
  sidemenuButton1->setText("to printer");

  sidemenuButton2 = new QPushButton(sidemenu);
  sidemenuButton2->setGeometry(10, 40, 100, 20);
  sidemenuButton2->setText("to Postscript");

  sidemenuButton3 = new QPushButton(sidemenu);
  sidemenuButton3->setGeometry(10, 70, 100, 20);
  sidemenuButton3->setText("to PDF");

  sidemenuButton4 = new QPushButton(sidemenu);
  sidemenuButton4->setGeometry(10, 100, 100, 20);
  sidemenuButton4->setText("to Image");

  if(extra_button)
  {
    sidemenuButton5 = new QPushButton(sidemenu);
    sidemenuButton5->setGeometry(10, 130, 100, 20);
    sidemenuButton5->setText(extra_button_txt);
  }

  QObject::connect(sidemenuButton1, SIGNAL(clicked()), this, SLOT(print_to_printer()));
  QObject::connect(sidemenuButton2, SIGNAL(clicked()), this, SLOT(print_to_postscript()));
  QObject::connect(sidemenuButton3, SIGNAL(clicked()), this, SLOT(print_to_pdf()));
  QObject::connect(sidemenuButton4, SIGNAL(clicked()), this, SLOT(print_to_image()));
  if(extra_button)
  {
    QObject::connect(sidemenuButton5, SIGNAL(clicked()), this, SLOT(send_button_event()));
  }

  sidemenu->exec();
}


void SignalCurve::send_button_event()
{
  emit extra_button_clicked();

  sidemenu->close();
}


void SignalCurve::create_button(const char *txt)
{
  extra_button = 1;
  strncpy(extra_button_txt, txt, 16);
  extra_button_txt[15] = 0;
}


void SignalCurve::backup_colors_for_printing(void)
{
  backup_color_1 = SignalColor;
  SignalColor = Qt::black;
  backup_color_2 = RasterColor;
  RasterColor = Qt::black;
  backup_color_3 = BorderColor;
  BorderColor = Qt::black;
  backup_color_4 = RulerColor;
  RulerColor = Qt::black;
  backup_color_5 = TextColor;
  TextColor = Qt::black;
  backup_color_6 = Marker1Pen.color();
  Marker1Pen.setColor(Qt::black);
}


void SignalCurve::restore_colors_after_printing(void)
{
  SignalColor = backup_color_1;
  RasterColor = backup_color_2;
  BorderColor = backup_color_3;
  RulerColor = backup_color_4;
  TextColor = backup_color_5;
  Marker1Pen.setColor(backup_color_6);
}


void SignalCurve::print_to_postscript()
{
  char path[1024];

  strcpy(path, "curve.ps");

  strcpy(path, QFileDialog::getSaveFileName(0, "Print to PostScript", path, "PostScript files (*.ps *.PS)").toLatin1().data());

  if(!strcmp(path, ""))
  {
    sidemenu->close();

    return;
  }

  QPrinter printer(QPrinter::HighResolution);

  printer.setOutputFormat(QPrinter::PostScriptFormat);
  printer.setOutputFileName(path);
  printer.setPageSize(QPrinter::A4);
  printer.setOrientation(QPrinter::Landscape);

  backup_colors_for_printing();

  QPainter paint(&printer);

  drawWidget_to_printer(&paint, printer.pageRect().width(), printer.pageRect().height());

  restore_colors_after_printing();

  sidemenu->close();
}


void SignalCurve::print_to_pdf()
{
  char path[1024];

  strcpy(path, "curve.pdf");

  strcpy(path, QFileDialog::getSaveFileName(0, "Print to PDF", path, "PDF files (*.pdf *.PDF)").toLatin1().data());

  if(!strcmp(path, ""))
  {
    sidemenu->close();

    return;
  }

  QPrinter printer(QPrinter::HighResolution);

  printer.setOutputFormat(QPrinter::PdfFormat);
  printer.setOutputFileName(path);
  printer.setPageSize(QPrinter::A4);
  printer.setOrientation(QPrinter::Landscape);

  backup_colors_for_printing();

  QPainter paint(&printer);

  drawWidget_to_printer(&paint, printer.pageRect().width(), printer.pageRect().height());

  restore_colors_after_printing();

  sidemenu->close();
}


void SignalCurve::print_to_image()
{
  char path[1024];

  strcpy(path, "curve.png");

  strcpy(path, QFileDialog::getSaveFileName(0, "Print to Image", path, "PNG files (*.png *.PNG)").toLatin1().data());

  if(!strcmp(path, ""))
  {
    sidemenu->close();

    return;
  }

  QPixmap pixmap(width(), height());

  QPainter paint(&pixmap);

  drawWidget(&paint, width(), height());

  pixmap.save(path, "PNG", 90);

  sidemenu->close();
}


void SignalCurve::print_to_printer()
{
  QPrinter printer(QPrinter::HighResolution);

  printer.setOutputFormat(QPrinter::NativeFormat);
  printer.setPageSize(QPrinter::A4);
  printer.setOrientation(QPrinter::Landscape);

  QPrintDialog printerdialog(&printer, this);
  printerdialog.setWindowTitle("Print");

  if(!(printerdialog.exec()==QDialog::Accepted))
  {
    sidemenu->close();

    return;
  }

  backup_colors_for_printing();

  QPainter paint(&printer);

  drawWidget_to_printer(&paint, printer.pageRect().width(), printer.pageRect().height());

  restore_colors_after_printing();

  sidemenu->close();
}


void SignalCurve::drawWidget_to_printer(QPainter *painter, int w, int h)
{
  int i, j,
      precision,
      bordersize_backup=0,
      p_w,
      p_divisor,
      p_range,
      p_multiplier,
      p_ruler_startvalue,
      p_ruler_endvalue,
      p_tmp,
      p_h,
      p2_divisor,
      p2_range,
      p2_multiplier,
      p2_ruler_startvalue,
      p2_ruler_endvalue,
      p2_tmp;

  char str[128];

  double v_sens,
         offset,
         h_step,
         value,
         p_factor,
         p_pixels_per_unit,
         p2_pixels_per_unit;

  QString q_str;

  QFont font;


  p_factor = (double)w / width();

  bordersize_backup = bordersize;
  bordersize *= p_factor;

  font.setFamily("Arial");
  font.setPixelSize((int)((double)w / 104.0));
  painter->setFont(font);

  if((w < ((bordersize * 2) + 5)) || (h < ((bordersize * 2) + 5)))
  {
    bordersize = bordersize_backup;

    return;
  }

/////////////////////////////////// draw the rulers ///////////////////////////////////////////

  p_w = w - bordersize - bordersize;

  p_multiplier = 1;

  while((h_ruler_endvalue * p_multiplier) < 10000.0)
  {
    p_multiplier *= 10;

    if(p_multiplier > 10000000)
    {
      break;
    }
  }

  p_ruler_startvalue = h_ruler_startvalue * p_multiplier;

  p_ruler_endvalue = h_ruler_endvalue * p_multiplier;

  p_range = p_ruler_endvalue - p_ruler_startvalue;

  p_pixels_per_unit = (double)p_w / (double)p_range;

  p_divisor = 1;

  while((p_range / p_divisor) > 10)
  {
    p_divisor *= 2;

    if((p_range / p_divisor) <= 10)
    {
      break;
    }

    p_divisor /= 2;

    p_divisor *= 5;

    if((p_range / p_divisor) <= 10)
    {
      break;
    }

    p_divisor *= 2;
  }

  if(drawHruler && (bordersize > (19 * p_factor)))
  {
    painter->setPen(RulerColor);

    painter->drawLine(bordersize, h - bordersize + (5 * p_factor), w - bordersize, h - bordersize + (5 * p_factor));

    h_ruler_precision = 0;

    if((h_ruler_endvalue < 10.0) && (h_ruler_endvalue > -10.0) && (h_ruler_startvalue < 10.0) && (h_ruler_startvalue > -10.0))
    {
      h_ruler_precision = 1;

      if((h_ruler_endvalue < 1.0) && (h_ruler_endvalue > -1.0) && (h_ruler_startvalue < 1.0) && (h_ruler_startvalue > -1.0))
      {
        h_ruler_precision = 2;

        if((h_ruler_endvalue < 0.1) && (h_ruler_endvalue > -0.1) && (h_ruler_startvalue < 0.1) && (h_ruler_startvalue > -0.1))
        {
          h_ruler_precision = 3;

          if((h_ruler_endvalue < 0.01) && (h_ruler_endvalue > -0.01) && (h_ruler_startvalue < 0.01) && (h_ruler_startvalue > -0.01))
          {
            h_ruler_precision = 4;
          }
        }
      }
    }

    for(i = (p_ruler_startvalue / p_divisor) * p_divisor; i <= p_ruler_endvalue; i += p_divisor)
    {
      if(i < p_ruler_startvalue)
      {
        continue;
      }

      q_str.setNum((double)i / (double)p_multiplier, 'f', h_ruler_precision);

      p_tmp = (double)(i - p_ruler_startvalue) * p_pixels_per_unit;

      painter->drawText(bordersize + p_tmp - (30 * p_factor),  h - bordersize + (18 * p_factor), 60 * p_factor, 16 * p_factor, Qt::AlignCenter | Qt::TextSingleLine, q_str);

      painter->drawLine(bordersize + p_tmp, h - bordersize + (5 * p_factor), bordersize + p_tmp, h - bordersize + ((5 + 10) * p_factor));
    }

    painter->drawText(w - bordersize + (20 * p_factor),  h - bordersize + (18 * p_factor), 40 * p_factor, 16 * p_factor, Qt::AlignLeft | Qt::AlignVCenter | Qt::TextSingleLine, h_label);
  }

  p_h = h - bordersize - bordersize;

  p2_multiplier = 1;

  while((max_value * p2_multiplier) < 10000.0)
  {
    p2_multiplier *= 10;

    if(p2_multiplier > 10000000)
    {
      break;
    }
  }

  p2_ruler_startvalue = min_value * p2_multiplier;

  p2_ruler_endvalue = max_value * p2_multiplier;

  p2_range = p2_ruler_endvalue - p2_ruler_startvalue;

  p2_pixels_per_unit = (double)p_h / (double)p2_range;

  p2_divisor = 1;

  while((p2_range / p2_divisor) > 10)
  {
    p2_divisor *= 2;

    if((p2_range / p2_divisor) <= 10)
    {
      break;
    }

    p2_divisor /= 2;

    p2_divisor *= 5;

    if((p2_range / p2_divisor) <= 10)
    {
      break;
    }

    p2_divisor *= 2;
  }

  if(drawVruler && (bordersize > (29 * p_factor)))
  {
    painter->setPen(RulerColor);

    painter->drawLine(bordersize - (5 * p_factor), bordersize, bordersize - (5 * p_factor), h - bordersize);

    precision = 0;

    if((max_value < 10.0) && (max_value > -10.0) && (min_value < 10.0) && (min_value > -10.0))
    {
      precision = 1;

      if((max_value < 1.0) && (max_value > -1.0) && (min_value < 1.0) && (min_value > -1.0))
      {
        precision = 2;

        if((max_value < 0.1) && (max_value > -0.1) && (min_value < 0.1) && (min_value > -0.1))
        {
          precision = 3;

          if((max_value < 0.01) && (max_value > -0.01) && (min_value < 0.01) && (min_value > -0.01))
          {
            precision = 4;
          }
        }
      }
    }

    for(i = (p2_ruler_startvalue / p2_divisor) * p2_divisor; i <= p2_ruler_endvalue; i += p2_divisor)
    {
      if(i < p2_ruler_startvalue)
      {
        continue;
      }

      q_str.setNum((double)i / (double)p2_multiplier, 'f', precision);

      p2_tmp = (double)(i - p2_ruler_startvalue) * p2_pixels_per_unit;

      painter->drawText((3 * p_factor), h - bordersize - p2_tmp - (8 * p_factor), (40 * p_factor), (16 * p_factor), Qt::AlignRight | Qt::AlignVCenter | Qt::TextSingleLine, q_str);

      painter->drawLine(bordersize - (5 * p_factor), h - bordersize - p2_tmp, bordersize - (15 * p_factor), h - bordersize - p2_tmp);
    }
  }

/////////////////////////////////// draw the labels ///////////////////////////////////////////

  painter->setPen(TextColor);
  painter->setFont(font);

  if(v_label[0] != 0)
  {
    painter->drawText(8 * p_factor, 30 * p_factor, 80 * p_factor, 16 * p_factor, Qt::AlignCenter | Qt::TextSingleLine, v_label);
  }

  if(upperlabel1[0] != 0)
  {
    painter->drawText(w / 2 - (150 * p_factor), 10 * p_factor, 300 * p_factor, 16 * p_factor, Qt::AlignCenter | Qt::TextSingleLine, upperlabel1);
  }

  if(upperlabel2[0] != 0)
  {
    painter->drawText(w / 2 - (150 * p_factor), 30 * p_factor, 300 * p_factor, 16 * p_factor, Qt::AlignCenter | Qt::TextSingleLine, upperlabel2);
  }

  if(lowerlabel[0] != 0)
  {
    painter->drawText(w / 2 - (80 * p_factor), h - (20 * p_factor), 160 * p_factor, 16 * p_factor, Qt::AlignCenter | Qt::TextSingleLine, lowerlabel);
  }

/////////////////////////////////// translate coordinates, draw and fill a rectangle ///////////////////////////////////////////

  painter->translate(QPoint(bordersize, bordersize));

  w -= (bordersize * 2);

  h -= (bordersize * 2);

  painter->setClipping(TRUE);
  painter->setClipRegion(QRegion(0, 0, w, h), Qt::ReplaceClip);

/////////////////////////////////// draw the rasters ///////////////////////////////////////////

  painter->setPen(RasterColor);

  painter->drawRect (0, 0, w - 1, h - 1);

  for(i = (p_ruler_startvalue / p_divisor) * p_divisor; i <= p_ruler_endvalue; i += p_divisor)
  {
    if(i < p_ruler_startvalue)
    {
      continue;
    }

    p_tmp = (double)(i - p_ruler_startvalue) * p_pixels_per_unit;

    painter->drawLine(p_tmp, 0, p_tmp, h);
  }

  for(i = (p2_ruler_startvalue / p2_divisor) * p2_divisor; i <= p2_ruler_endvalue; i += p2_divisor)
  {
    if(i < p2_ruler_startvalue)
    {
      continue;
    }

    p2_tmp = (double)(i - p2_ruler_startvalue) * p2_pixels_per_unit;

    painter->drawLine(0, h - p2_tmp, w, h - p2_tmp);
  }

/////////////////////////////////// draw the curve ///////////////////////////////////////////

  if((dbuf == NULL) && (ibuf == NULL) && (fbuf == NULL)) return;

  if(max_value == min_value)  return;

  if(bufsize < 2)  return;

  offset = (-(max_value));

  v_sens = (-(h / (max_value - min_value)));

  h_step = (double)w / (double)bufsize;

  painter->setPen(QPen(QBrush(SignalColor, Qt::SolidPattern), tracewidth, Qt::SolidLine, Qt::SquareCap, Qt::BevelJoin));

  if(dbuf)
  {
    for(i = 0; i < bufsize; i++)
    {
      if(fillsurface)
      {
        if(bufsize < w)
        {
          for(j = 0; j < h_step; j++)
          {
            painter->drawLine((i * h_step) + j, (dbuf[i] + offset) * v_sens, (i * h_step) + j, h);
          }
        }
        else
        {
          painter->drawLine(i * h_step, (dbuf[i] + offset) * v_sens, i * h_step, h);
        }
      }
      else
      {
        if(bufsize < (w / 2))
        {
          painter->drawLine(i * h_step, (dbuf[i] + offset) * v_sens, (i + 1) * h_step, (dbuf[i] + offset) * v_sens);
          if(i)
          {
            painter->drawLine(i * h_step, (dbuf[i - 1] + offset) * v_sens, i * h_step, (dbuf[i] + offset) * v_sens);
          }
        }
        else
        {
          if(i < (bufsize - 1))
          {
            {
              painter->drawLine(i * h_step, (dbuf[i] + offset) * v_sens, (i + 1) * h_step, (dbuf[i + 1] + offset) * v_sens);
            }
          }
        }
      }

      if(crosshair_1_active)
      {
        if(i==((int)(((double)crosshair_1_x_position * p_factor) / h_step)))
        {
          crosshair_1_y_value = (dbuf[i] + offset) * v_sens;
          crosshair_1_value = dbuf[i];
          value = (h_ruler_endvalue - h_ruler_startvalue) / bufsize;
          crosshair_1_value_2 = (i * value) + (0.5 * value) + h_ruler_startvalue;
        }
      }
    }
  }

  if(ibuf)
  {
    for(i = 0; i < bufsize; i++)
    {
      if(fillsurface)
      {
        if(bufsize < w)
        {
          for(j = 0; j < h_step; j++)
          {
            painter->drawLine((i * h_step) + j, ((double)(ibuf[i]) + offset) * v_sens, (i * h_step) + j, h);
          }
        }
        else
        {
          painter->drawLine(i * h_step, ((double)(ibuf[i]) + offset) * v_sens, i * h_step, h);
        }
      }
      else
      {
        if(bufsize < (w / 2))
        {
          painter->drawLine(i * h_step, ((double)(ibuf[i]) + offset) * v_sens, (i + 1) * h_step, ((double)(ibuf[i]) + offset) * v_sens);
          if(i)
          {
            painter->drawLine(i * h_step, ((double)(ibuf[i - 1]) + offset) * v_sens, i * h_step, ((double)(ibuf[i]) + offset) * v_sens);
          }
        }
        else
        {
          if(i < (bufsize - 1))
          {
            {
              painter->drawLine(i * h_step, ((double)(ibuf[i]) + offset) * v_sens, (i + 1) * h_step, ((double)(ibuf[i + 1]) + offset) * v_sens);
            }
          }
        }
      }

      if(crosshair_1_active)
      {
        if(i==((int)(((double)crosshair_1_x_position * p_factor) / h_step)))
        {
          crosshair_1_y_value = ((double)(ibuf[i]) + offset) * v_sens;
          crosshair_1_value = (double)(ibuf[i]);
          value = (h_ruler_endvalue - h_ruler_startvalue) / bufsize;
          crosshair_1_value_2 = (i * value) + (0.5 * value) + h_ruler_startvalue;
        }
      }
    }
  }

  if(fbuf)
  {
    for(i = 0; i < bufsize; i++)
    {
      if(fillsurface)
      {
        if(bufsize < w)
        {
          for(j = 0; j < h_step; j++)
          {
            painter->drawLine((i * h_step) + j, (fbuf[i] + offset) * v_sens, (i * h_step) + j, h);
          }
        }
        else
        {
          painter->drawLine(i * h_step, (fbuf[i] + offset) * v_sens, i * h_step, h);
        }
      }
      else
      {
        if(bufsize < (w / 2))
        {
          painter->drawLine(i * h_step, (fbuf[i] + offset) * v_sens, (i + 1) * h_step, (fbuf[i] + offset) * v_sens);
          if(i)
          {
            painter->drawLine(i * h_step, (fbuf[i - 1] + offset) * v_sens, i * h_step, (fbuf[i] + offset) * v_sens);
          }
        }
        else
        {
          if(i < (bufsize - 1))
          {
            {
              painter->drawLine(i * h_step, (fbuf[i] + offset) * v_sens, (i + 1) * h_step, (fbuf[i + 1] + offset) * v_sens);
            }
          }
        }
      }

      if(crosshair_1_active)
      {
        if(i==((int)(((double)crosshair_1_x_position * p_factor) / h_step)))
        {
          crosshair_1_y_value = (fbuf[i] + offset) * v_sens;
          crosshair_1_value = fbuf[i];
          value = (h_ruler_endvalue - h_ruler_startvalue) / bufsize;
          crosshair_1_value_2 = (i * value) + (0.5 * value) + h_ruler_startvalue;
        }
      }
    }
  }

/////////////////////////////////// draw the marker ///////////////////////////////////////////

  if(Marker1Enabled == TRUE)
  {
    painter->setPen(Marker1Pen);

    painter->drawLine(w * marker_1_position, 0, w * marker_1_position, h);
  }

/////////////////////////////////// draw the cursor ///////////////////////////////////////////

  painter->setPen(TextColor);

  if(crosshair_1_active)
  {
    QPainterPath path;
    path.moveTo(crosshair_1_x_position * p_factor, crosshair_1_y_value);
    path.lineTo((crosshair_1_x_position - 4) * p_factor, crosshair_1_y_value - (9 * p_factor));
    path.lineTo((crosshair_1_x_position + 4) * p_factor, crosshair_1_y_value - (9 * p_factor));
    path.lineTo(crosshair_1_x_position * p_factor, crosshair_1_y_value);
    painter->fillPath(path, QBrush(Qt::black));

    snprintf(str, 128, "%f", crosshair_1_value);
    painter->drawText((crosshair_1_x_position + 8) * p_factor, (crosshair_1_y_position - 10) * p_factor, str);
    snprintf(str, 128, "%f %s", crosshair_1_value_2, h_label);
    painter->drawText((crosshair_1_x_position + 8) * p_factor, (crosshair_1_y_position + 10) * p_factor, str);
  }

  bordersize = bordersize_backup;
}


void SignalCurve::setUpdatesEnabled(bool enabled)
{
  updates_enabled = enabled;
}


void SignalCurve::paintEvent(QPaintEvent *)
{
  if(updates_enabled == TRUE)
  {
    QPainter paint(this);

    drawWidget(&paint, width(), height());

    old_w = width();
  }
}


void SignalCurve::drawWidget(QPainter *painter, int w, int h)
{
  int i, j,
      precision,
      p_w,
      p_divisor,
      p_range,
      p_multiplier,
      p_ruler_startvalue,
      p_ruler_endvalue,
      p_tmp,
      p_h,
      p2_divisor,
      p2_range,
      p2_multiplier,
      p2_ruler_startvalue,
      p2_ruler_endvalue,
      p2_tmp;

  double v_sens,
         offset,
         h_step,
         value,
         pixelsPerUnit,
         max_colorbar_value,
         p_pixels_per_unit,
         p2_pixels_per_unit;

  char str[128];

  QString q_str;


  painter->setFont(QFont("Arial", 8));

  painter->fillRect(0, 0, w, h, BorderColor);

  if((w < ((bordersize * 2) + 5)) || (h < ((bordersize * 2) + 5)))
  {
    return;
  }

/////////////////////////////////// draw the rulers ///////////////////////////////////////////

  p_w = w - bordersize - bordersize;

  p_multiplier = 1;

  while((h_ruler_endvalue * p_multiplier) < 10000.0)
  {
    p_multiplier *= 10;

    if(p_multiplier > 10000000)
    {
      break;
    }
  }

  p_ruler_startvalue = h_ruler_startvalue * p_multiplier;

  p_ruler_endvalue = h_ruler_endvalue * p_multiplier;

  p_range = p_ruler_endvalue - p_ruler_startvalue;

  p_pixels_per_unit = (double)p_w / (double)p_range;

  p_divisor = 1;

  while((p_range / p_divisor) > 10)
  {
    p_divisor *= 2;

    if((p_range / p_divisor) <= 10)
    {
      break;
    }

    p_divisor /= 2;

    p_divisor *= 5;

    if((p_range / p_divisor) <= 10)
    {
      break;
    }

    p_divisor *= 2;
  }

//   printf("p_multiplier is %i\n"
//         "p_ruler_startvalue is %i\n"
//         "p_ruler_endvalue is %i\n"
//         "p_range is %i\n"
//         "p_divisor is %i\n"
//         "p_pixels_per_unit is %f\n\n",
//         p_multiplier,
//         p_ruler_startvalue,
//         p_ruler_endvalue,
//         p_range,
//         p_divisor,
//         p_pixels_per_unit);

  if(drawHruler && (bordersize > 19))
  {
    painter->setPen(RulerColor);

    painter->drawLine(bordersize, h - bordersize + 5, w - bordersize, h - bordersize + 5);

    h_ruler_precision = 0;

    if((h_ruler_endvalue < 10.0) && (h_ruler_endvalue > -10.0) && (h_ruler_startvalue < 10.0) && (h_ruler_startvalue > -10.0))
    {
      h_ruler_precision = 1;

      if((h_ruler_endvalue < 1.0) && (h_ruler_endvalue > -1.0) && (h_ruler_startvalue < 1.0) && (h_ruler_startvalue > -1.0))
      {
        h_ruler_precision = 2;

        if((h_ruler_endvalue < 0.1) && (h_ruler_endvalue > -0.1) && (h_ruler_startvalue < 0.1) && (h_ruler_startvalue > -0.1))
        {
          h_ruler_precision = 3;

          if((h_ruler_endvalue < 0.01) && (h_ruler_endvalue > -0.01) && (h_ruler_startvalue < 0.01) && (h_ruler_startvalue > -0.01))
          {
            h_ruler_precision = 4;
          }
        }
      }
    }

    for(i = (p_ruler_startvalue / p_divisor) * p_divisor; i <= p_ruler_endvalue; i += p_divisor)
    {
      if(i < p_ruler_startvalue)
      {
        continue;
      }

      q_str.setNum((double)i / (double)p_multiplier, 'f', h_ruler_precision);

      p_tmp = (double)(i - p_ruler_startvalue) * p_pixels_per_unit;

      painter->drawText(bordersize + p_tmp - 30,  h - bordersize + 18, 60, 16, Qt::AlignCenter | Qt::TextSingleLine, q_str);

      painter->drawLine(bordersize + p_tmp, h - bordersize + 5, bordersize + p_tmp, h - bordersize + 5 + 10);
    }

    painter->drawText(w - bordersize + 20,  h - bordersize + 18, 40, 16, Qt::AlignLeft | Qt::AlignVCenter | Qt::TextSingleLine, h_label);
  }

  p_h = h - bordersize - bordersize;

  p2_multiplier = 1;

  while((max_value * p2_multiplier) < 10000.0)
  {
    p2_multiplier *= 10;

    if(p2_multiplier > 10000000)
    {
      break;
    }
  }

  p2_ruler_startvalue = min_value * p2_multiplier;

  p2_ruler_endvalue = max_value * p2_multiplier;

  p2_range = p2_ruler_endvalue - p2_ruler_startvalue;

  p2_pixels_per_unit = (double)p_h / (double)p2_range;

  p2_divisor = 1;

  while((p2_range / p2_divisor) > 10)
  {
    p2_divisor *= 2;

    if((p2_range / p2_divisor) <= 10)
    {
      break;
    }

    p2_divisor /= 2;

    p2_divisor *= 5;

    if((p2_range / p2_divisor) <= 10)
    {
      break;
    }

    p2_divisor *= 2;
  }

//   printf("p2_multiplier is %i\n"
//         "p2_ruler_startvalue is %i\n"
//         "p2_ruler_endvalue is %i\n"
//         "p2_range is %i\n"
//         "p2_divisor is %i\n"
//         "p2_pixels_per_unit is %f\n"
//         "max_value is %f\n"
//         "min_value is %f\n\n",
//         p2_multiplier,
//         p2_ruler_startvalue,
//         p2_ruler_endvalue,
//         p2_range,
//         p2_divisor,
//         p2_pixels_per_unit,
//         max_value,
//         min_value);

  if(drawVruler && (bordersize > 29))
  {
    painter->setPen(RulerColor);

    painter->drawLine(bordersize - 5, bordersize, bordersize - 5, h - bordersize);

    precision = 0;

    if((max_value < 10.0) && (max_value > -10.0) && (min_value < 10.0) && (min_value > -10.0))
    {
      precision = 1;

      if((max_value < 1.0) && (max_value > -1.0) && (min_value < 1.0) && (min_value > -1.0))
      {
        precision = 2;

        if((max_value < 0.1) && (max_value > -0.1) && (min_value < 0.1) && (min_value > -0.1))
        {
          precision = 3;

          if((max_value < 0.01) && (max_value > -0.01) && (min_value < 0.01) && (min_value > -0.01))
          {
            precision = 4;
          }
        }
      }
    }

    for(i = (p2_ruler_startvalue / p2_divisor) * p2_divisor; i <= p2_ruler_endvalue; i += p2_divisor)
    {
      if(i < p2_ruler_startvalue)
      {
        continue;
      }

      q_str.setNum((double)i / (double)p2_multiplier, 'f', precision);

      p2_tmp = (double)(i - p2_ruler_startvalue) * p2_pixels_per_unit;

      painter->drawText(3, h - bordersize - p2_tmp - 8, 40, 16, Qt::AlignRight | Qt::AlignVCenter | Qt::TextSingleLine, q_str);

      painter->drawLine(bordersize - 5, h - bordersize - p2_tmp, bordersize - 5 - 10, h - bordersize - p2_tmp);
    }
  }

/////////////////////////////////// draw the labels ///////////////////////////////////////////

  painter->setPen(TextColor);
  painter->setFont(QFont("Arial", 8));

  if(v_label[0] != 0)
  {
    painter->drawText(8, 30, 80, 16, Qt::AlignCenter | Qt::TextSingleLine, v_label);
  }

  if(upperlabel1[0] != 0)
  {
    painter->drawText(w / 2 - 150, 20, 300, 16, Qt::AlignCenter | Qt::TextSingleLine, upperlabel1);
  }

  if(lowerlabel[0] != 0)
  {
    painter->drawText(w / 2 - 80, h - 20, 160, 16, Qt::AlignCenter | Qt::TextSingleLine, lowerlabel);
  }

/////////////////////////////////// translate coordinates, draw and fill a rectangle ///////////////////////////////////////////

  painter->translate(QPoint(bordersize, bordersize));

  w -= (bordersize * 2);

  h -= (bordersize * 2);

  painter->fillRect(0, 0, w, h, BackgroundColor);

  painter->setClipping(TRUE);
  painter->setClipRegion(QRegion(0, 0, w, h), Qt::ReplaceClip);

/////////////////////////////////// draw the colorbars /////////////////////////////////////////

  if(spectrum_color != NULL)
  {
    max_colorbar_value = 0.0;

    for(i=0; i < spectrum_color->items; i++)
    {
      if(spectrum_color->value[i] > max_colorbar_value)
      {
        max_colorbar_value = spectrum_color->value[i];
      }
    }

    max_colorbar_value *= 1.05;

    pixelsPerUnit = (double)w / (h_ruler_endvalue - h_ruler_startvalue);

    if((spectrum_color->freq[0] > h_ruler_startvalue) && (spectrum_color->items > 1))
    {
      painter->fillRect(0,
                        (max_colorbar_value - spectrum_color->value[0]) * ((double)h / max_colorbar_value),
                        (spectrum_color->freq[0] - h_ruler_startvalue) * pixelsPerUnit,
                        h,
                        (Qt::GlobalColor)spectrum_color->color[0]);

      painter->setPen(Qt::lightGray);
      painter->drawText(10, 20, spectrum_color->label[0]);
    }

    for(i=1; i < spectrum_color->items; i++)
    {
      if(spectrum_color->freq[i] > h_ruler_startvalue)
      {
        painter->fillRect((spectrum_color->freq[i-1] - h_ruler_startvalue) * pixelsPerUnit,
                          (max_colorbar_value - spectrum_color->value[i]) * ((double)h / max_colorbar_value),
                          (spectrum_color->freq[i] - spectrum_color->freq[i-1]) * pixelsPerUnit,
                          h,
                          (Qt::GlobalColor)spectrum_color->color[i]);

        painter->setPen(Qt::lightGray);
        painter->drawText((spectrum_color->freq[i-1] - h_ruler_startvalue) * pixelsPerUnit + 10, 20, spectrum_color->label[i]);
      }
    }
  }

/////////////////////////////////// draw the curve ///////////////////////////////////////////

  if(drawcurve_before_raster)
  {
    if((dbuf == NULL) && (ibuf == NULL) && (fbuf == NULL)) return;

    if(max_value == min_value)  return;

    if(bufsize < 2)  return;

    offset = (-(max_value));

    v_sens = (-(h / (max_value - min_value)));

    h_step = (double)w / (double)bufsize;

    painter->setPen(QPen(QBrush(SignalColor, Qt::SolidPattern), tracewidth, Qt::SolidLine, Qt::SquareCap, Qt::BevelJoin));

    if(dbuf)
    {
      for(i = 0; i < bufsize; i++)
      {
        if(fillsurface)
        {
          if(bufsize < w)
          {
            for(j = 0; j < h_step; j++)
            {
              painter->drawLine((i * h_step) + j, (dbuf[i] + offset) * v_sens, (i * h_step) + j, h);
            }
          }
          else
          {
            painter->drawLine(i * h_step, (dbuf[i] + offset) * v_sens, i * h_step, h);
          }
        }
        else
        {
          if(bufsize < (w / 2))
          {
            painter->drawLine(i * h_step, (dbuf[i] + offset) * v_sens, (i + 1) * h_step, (dbuf[i] + offset) * v_sens);
            if(i)
            {
              painter->drawLine(i * h_step, (dbuf[i - 1] + offset) * v_sens, i * h_step, (dbuf[i] + offset) * v_sens);
            }
          }
          else
          {
            if(i < (bufsize - 1))
            {
              {
                painter->drawLine(i * h_step, (dbuf[i] + offset) * v_sens, (i + 1) * h_step, (dbuf[i + 1] + offset) * v_sens);
              }
            }
          }
        }

        if(crosshair_1_active)
        {
          if(i==((int)(((double)crosshair_1_x_position) / h_step)))
          {
            crosshair_1_y_value = (dbuf[i] + offset) * v_sens;
            crosshair_1_value = dbuf[i];
            value = (h_ruler_endvalue - h_ruler_startvalue) / bufsize;
            crosshair_1_value_2 = (i * value) + h_ruler_startvalue;
          }
        }
      }
    }

    if(ibuf)
    {
      for(i = 0; i < bufsize; i++)
      {
        if(fillsurface)
        {
          if(bufsize < w)
          {
            for(j = 0; j < h_step; j++)
            {
              painter->drawLine((i * h_step) + j, ((double)(ibuf[i]) + offset) * v_sens, (i * h_step) + j, h);
            }
          }
          else
          {
            painter->drawLine(i * h_step, ((double)(ibuf[i]) + offset) * v_sens, i * h_step, h);
          }
        }
        else
        {
          if(bufsize < (w / 2))
          {
            painter->drawLine(i * h_step, ((double)(ibuf[i]) + offset) * v_sens, (i + 1) * h_step, ((double)(ibuf[i]) + offset) * v_sens);
            if(i)
            {
              painter->drawLine(i * h_step, ((double)(ibuf[i - 1]) + offset) * v_sens, i * h_step, ((double)(ibuf[i]) + offset) * v_sens);
            }
          }
          else
          {
            if(i < (bufsize - 1))
            {
              {
                painter->drawLine(i * h_step, ((double)(ibuf[i]) + offset) * v_sens, (i + 1) * h_step, ((double)(ibuf[i + 1]) + offset) * v_sens);
              }
            }
          }
        }

        if(crosshair_1_active)
        {
          if(i==((int)(((double)crosshair_1_x_position) / h_step)))
          {
            crosshair_1_y_value = ((double)(ibuf[i]) + offset) * v_sens;
            crosshair_1_value = (double)(ibuf[i]);
            value = (h_ruler_endvalue - h_ruler_startvalue) / bufsize;
            crosshair_1_value_2 = (i * value) + h_ruler_startvalue;
          }
        }
      }
    }

    if(fbuf)
    {
      for(i = 0; i < bufsize; i++)
      {
        if(fillsurface)
        {
          if(bufsize < w)
          {
            for(j = 0; j < h_step; j++)
            {
              painter->drawLine((i * h_step) + j, (fbuf[i] + offset) * v_sens, (i * h_step) + j, h);
            }
          }
          else
          {
            painter->drawLine(i * h_step, (fbuf[i] + offset) * v_sens, i * h_step, h);
          }
        }
        else
        {
          if(bufsize < (w / 2))
          {
            painter->drawLine(i * h_step, (fbuf[i] + offset) * v_sens, (i + 1) * h_step, (fbuf[i] + offset) * v_sens);
            if(i)
            {
              painter->drawLine(i * h_step, (fbuf[i - 1] + offset) * v_sens, i * h_step, (fbuf[i] + offset) * v_sens);
            }
          }
          else
          {
            if(i < (bufsize - 1))
            {
              {
                painter->drawLine(i * h_step, (fbuf[i] + offset) * v_sens, (i + 1) * h_step, (fbuf[i + 1] + offset) * v_sens);
              }
            }
          }
        }

        if(crosshair_1_active)
        {
          if(i==((int)(((double)crosshair_1_x_position) / h_step)))
          {
            crosshair_1_y_value = (fbuf[i] + offset) * v_sens;
            crosshair_1_value = fbuf[i];
            value = (h_ruler_endvalue - h_ruler_startvalue) / bufsize;
            crosshair_1_value_2 = (i * value) + h_ruler_startvalue;
          }
        }
      }
    }
  }

/////////////////////////////////// draw the rasters ///////////////////////////////////////////

  painter->setPen(RasterColor);

  painter->drawRect (0, 0, w - 1, h - 1);

  for(i = (p_ruler_startvalue / p_divisor) * p_divisor; i <= p_ruler_endvalue; i += p_divisor)
  {
    if(i < p_ruler_startvalue)
    {
      continue;
    }

    p_tmp = (double)(i - p_ruler_startvalue) * p_pixels_per_unit;

    painter->drawLine(p_tmp, 0, p_tmp, h);
  }

  for(i = (p2_ruler_startvalue / p2_divisor) * p2_divisor; i <= p2_ruler_endvalue; i += p2_divisor)
  {
    if(i < p2_ruler_startvalue)
    {
      continue;
    }

    p2_tmp = (double)(i - p2_ruler_startvalue) * p2_pixels_per_unit;

    painter->drawLine(0, h - p2_tmp, w, h - p2_tmp);
  }

/////////////////////////////////// draw the curve ///////////////////////////////////////////

  if(!drawcurve_before_raster)
  {
    if((dbuf == NULL) && (ibuf == NULL) && (fbuf == NULL)) return;

    if(max_value == min_value)  return;

    if(bufsize < 2)  return;

    offset = (-(max_value));

    v_sens = (-(h / (max_value - min_value)));

    h_step = (double)w / (double)bufsize;

    painter->setPen(QPen(QBrush(SignalColor, Qt::SolidPattern), tracewidth, Qt::SolidLine, Qt::SquareCap, Qt::BevelJoin));

    if(dbuf)
    {
      for(i = 0; i < bufsize; i++)
      {
        if(fillsurface)
        {
          if(bufsize < w)
          {
            for(j = 0; j < h_step; j++)
            {
              painter->drawLine((i * h_step) + j, (dbuf[i] + offset) * v_sens, (i * h_step) + j, h);
            }
          }
          else
          {
            painter->drawLine(i * h_step, (dbuf[i] + offset) * v_sens, i * h_step, h);
          }
        }
        else
        {
          if(bufsize < (w / 2))
          {
            painter->drawLine(i * h_step, (dbuf[i] + offset) * v_sens, (i + 1) * h_step, (dbuf[i] + offset) * v_sens);
            if(i)
            {
              painter->drawLine(i * h_step, (dbuf[i - 1] + offset) * v_sens, i * h_step, (dbuf[i] + offset) * v_sens);
            }
          }
          else
          {
            if(i < (bufsize - 1))
            {
              painter->drawLine(i * h_step, (dbuf[i] + offset) * v_sens, (i + 1) * h_step, (dbuf[i + 1] + offset) * v_sens);
            }
          }
        }

        if(crosshair_1_active)
        {
          if(i==((int)(((double)crosshair_1_x_position) / h_step)))
          {
            crosshair_1_y_value = (dbuf[i] + offset) * v_sens;
            crosshair_1_value = dbuf[i];
            value = (h_ruler_endvalue - h_ruler_startvalue) / bufsize;
            crosshair_1_value_2 = (i * value) + h_ruler_startvalue;
          }
        }
      }
    }

    if(ibuf)
    {
      for(i = 0; i < bufsize; i++)
      {
        if(fillsurface)
        {
          if(bufsize < w)
          {
            for(j = 0; j < h_step; j++)
            {
              painter->drawLine((i * h_step) + j, ((double)(ibuf[i]) + offset) * v_sens, (i * h_step) + j, h);
            }
          }
          else
          {
            painter->drawLine(i * h_step, ((double)(ibuf[i]) + offset) * v_sens, i * h_step, h);
          }
        }
        else
        {
          if(bufsize < (w / 2))
          {
            painter->drawLine(i * h_step, ((double)(ibuf[i]) + offset) * v_sens, (i + 1) * h_step, ((double)(ibuf[i]) + offset) * v_sens);
            if(i)
            {
              painter->drawLine(i * h_step, ((double)(ibuf[i - 1]) + offset) * v_sens, i * h_step, ((double)(ibuf[i]) + offset) * v_sens);
            }
          }
          else
          {
            if(i < (bufsize - 1))
            {
              {
                painter->drawLine(i * h_step, ((double)(ibuf[i]) + offset) * v_sens, (i + 1) * h_step, ((double)(ibuf[i + 1]) + offset) * v_sens);
              }
            }
          }
        }

        if(crosshair_1_active)
        {
          if(i==((int)(((double)crosshair_1_x_position) / h_step)))
          {
            crosshair_1_y_value = ((double)(ibuf[i]) + offset) * v_sens;
            crosshair_1_value = (double)(ibuf[i]);
            value = (h_ruler_endvalue - h_ruler_startvalue) / bufsize;
            crosshair_1_value_2 = (i * value) + h_ruler_startvalue;
          }
        }
      }
    }

    if(fbuf)
    {
      for(i = 0; i < bufsize; i++)
      {
        if(fillsurface)
        {
          if(bufsize < w)
          {
            for(j = 0; j < h_step; j++)
            {
              painter->drawLine((i * h_step) + j, (fbuf[i] + offset) * v_sens, (i * h_step) + j, h);
            }
          }
          else
          {
            painter->drawLine(i * h_step, (fbuf[i] + offset) * v_sens, i * h_step, h);
          }
        }
        else
        {
          if(bufsize < (w / 2))
          {
            painter->drawLine(i * h_step, (fbuf[i] + offset) * v_sens, (i + 1) * h_step, (fbuf[i] + offset) * v_sens);
            if(i)
            {
              painter->drawLine(i * h_step, (fbuf[i - 1] + offset) * v_sens, i * h_step, (fbuf[i] + offset) * v_sens);
            }
          }
          else
          {
            if(i < (bufsize - 1))
            {
              {
                painter->drawLine(i * h_step, (fbuf[i] + offset) * v_sens, (i + 1) * h_step, (fbuf[i + 1] + offset) * v_sens);
              }
            }
          }
        }

        if(crosshair_1_active)
        {
          if(i==((int)(((double)crosshair_1_x_position) / h_step)))
          {
            crosshair_1_y_value = (fbuf[i] + offset) * v_sens;
            crosshair_1_value = fbuf[i];
            value = (h_ruler_endvalue - h_ruler_startvalue) / bufsize;
            crosshair_1_value_2 = (i * value) + h_ruler_startvalue;
          }
        }
      }
    }
  }

/////////////////////////////////// draw the marker ///////////////////////////////////////////

  if(Marker1Enabled == TRUE)
  {
    painter->setPen(Marker1Pen);

    painter->drawLine(w * marker_1_position, 0, w * marker_1_position, h);
  }

/////////////////////////////////// draw the cursor ///////////////////////////////////////////

  if(crosshair_1_active)
  {
    painter->setPen((Qt::GlobalColor)crosshair_1_color);

    QPainterPath path;
    path.moveTo(crosshair_1_x_position, crosshair_1_y_value);
    path.lineTo(crosshair_1_x_position - 4, crosshair_1_y_value - 10);
    path.lineTo(crosshair_1_x_position + 5, crosshair_1_y_value - 10);
    path.lineTo(crosshair_1_x_position, crosshair_1_y_value);
    painter->fillPath(path, QBrush((Qt::GlobalColor)crosshair_1_color));

    painter->setFont(QFont("Arial", 10));
    painter->fillRect(crosshair_1_x_position + 6, crosshair_1_y_position - 23, 60, 16, BackgroundColor);
    snprintf(str, 128, "%f", crosshair_1_value);
    painter->drawText(crosshair_1_x_position + 8, crosshair_1_y_position - 10, str);
    painter->fillRect(crosshair_1_x_position + 6, crosshair_1_y_position - 3, 60, 16, BackgroundColor);
    snprintf(str, 128, "%f %s", crosshair_1_value_2, h_label);
    painter->drawText(crosshair_1_x_position + 8, crosshair_1_y_position + 10, str);
  }

/////////////////////////////////// draw the buttons ///////////////////////////////////////////

  painter->setPen(Qt::black);
  painter->setFont(QFont("Arial", 8));
  if(printEnabled == TRUE)
  {
    painter->fillRect(w - 43, 3, 40, 18, Qt::gray);
    painter->drawText(w - 38, 16, "print");
  }
  if(cursorEnabled == TRUE)
  {
    painter->fillRect(w - 43, 23, 40, 18, Qt::gray);
    painter->drawText(w - 38, 36, "cursor");
  }
  if(dashBoardEnabled == TRUE)
  {
    painter->fillRect(w - 43, 43, 40, 18, Qt::gray);
    painter->drawText(w - 38, 56, "ctls");
  }
}


void SignalCurve::drawCurve(double *samplebuf, int size, double max_val, double min_val)
{
  dbuf = samplebuf;

  ibuf = NULL;

  fbuf = NULL;

  bufsize = size;

  max_value = max_val;

  min_value = min_val;

  update();
}


void SignalCurve::drawCurve(int *samplebuf, int size, double max_val, double min_val)
{
  ibuf = samplebuf;

  dbuf = NULL;

  fbuf = NULL;

  bufsize = size;

  max_value = max_val;

  min_value = min_val;

  update();
}


void SignalCurve::drawCurve(float *samplebuf, int size, double max_val, double min_val)
{
  fbuf = samplebuf;

  dbuf = NULL;

  ibuf = NULL;

  bufsize = size;

  max_value = max_val;

  min_value = min_val;

  update();
}


void SignalCurve::setFillSurfaceEnabled(bool enabled)
{
  if(enabled == TRUE)
  {
    fillsurface = 1;
  }
  else
  {
    fillsurface = 0;
  }

  update();
}


void SignalCurve::setMarker1Position(double pos)
{
  marker_1_position = pos;

  if(marker_1_position > 1.01)  marker_1_position = 1.01;

  if(marker_1_position < 0.0001)  marker_1_position = 0.0001;

  update();
}


void SignalCurve::setMarker1Enabled(bool on)
{
  Marker1Enabled = on;
  update();
}


void SignalCurve::setMarker1Color(QColor color)
{
  Marker1Pen.setColor(color);
  update();
}


void SignalCurve::setH_RulerValues(double start, double end)
{
  h_ruler_startvalue = start;
  h_ruler_endvalue = end;

  update();
}


void SignalCurve::setSignalColor(QColor newColor)
{
  SignalColor = newColor;
  update();
}


void SignalCurve::setTraceWidth(int width)
{
  tracewidth = width;
  if(tracewidth < 0)  tracewidth = 0;
  update();
}


void SignalCurve::setBackgroundColor(QColor newColor)
{
  BackgroundColor = newColor;
  update();
}


void SignalCurve::setRasterColor(QColor newColor)
{
  RasterColor = newColor;
  update();
}


void SignalCurve::setBorderColor(QColor newColor)
{
  BorderColor = newColor;
  update();
}


void SignalCurve::setTextColor(QColor newColor)
{
  TextColor = newColor;
  update();
}


void SignalCurve::setBorderSize(int newsize)
{
  bordersize = newsize;
  if(bordersize < 0)  bordersize = 0;
  update();
}


void SignalCurve::setH_label(const char *str)
{
  strncpy(h_label, str, 32);
  h_label[31] = 0;
  update();
}


void SignalCurve::setV_label(const char *str)
{
  strncpy(v_label, str, 15);
  v_label[15] = 0;
  update();
}


void SignalCurve::setUpperLabel1(const char *str)
{
  strncpy(upperlabel1, str, 64);
  upperlabel1[63] = 0;
  update();
}


void SignalCurve::setUpperLabel2(const char *str)
{
  strncpy(upperlabel2, str, 64);
  upperlabel2[63] = 0;
  update();
}


void SignalCurve::setLowerLabel(const char *str)
{
  strncpy(lowerlabel, str, 64);
  lowerlabel[63] = 0;
  update();
}


void SignalCurve::setCursorEnabled(bool value)
{
  cursorEnabled = value;
}


void SignalCurve::setPrintEnabled(bool value)
{
  printEnabled = value;
}


void SignalCurve::setDashBoardEnabled(bool value)
{
  dashBoardEnabled = value;
}


void SignalCurve::enableSpectrumColors(struct spectrum_markersblock *spectr_col)
{
  spectrum_color = spectr_col;
  update();
}


void SignalCurve::disableSpectrumColors()
{
  spectrum_color = NULL;
  update();
}


void SignalCurve::setV_rulerEnabled(bool value)
{
  if(value == TRUE)
  {
    drawVruler = 1;
  }
  else
  {
    drawVruler = 0;
  }
}






