<?php
/* Copyright (C) 2011		Dimitri Mouillard	<dmouillard@teclib.com>
 * Copyright (C) 2012-2015	Laurent Destailleur	<eldy@users.sourceforge.net>
 * Copyright (C) 2012-2016	Regis Houssin		<regis.houssin@capnetworks.com>
 * Copyright (C) 2013		Juanjo Menent		<jmenent@2byte.es>
 * Copyright (C) 2014		Ferran Marcet		<fmarcet@2byte.es>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, orwrite
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 *   	\file       htdocs/holiday/card.php
 *		\ingroup    holiday
 *		\brief      Form and file creation of paid holiday.
 */

require('../main.inc.php');
require_once DOL_DOCUMENT_ROOT.'/core/class/html.form.class.php';
require_once DOL_DOCUMENT_ROOT.'/user/class/usergroup.class.php';
require_once DOL_DOCUMENT_ROOT.'/core/class/html.formfile.class.php';
require_once DOL_DOCUMENT_ROOT.'/core/class/CMailFile.class.php';
require_once DOL_DOCUMENT_ROOT.'/core/class/html.formmail.class.php';
require_once DOL_DOCUMENT_ROOT.'/core/lib/date.lib.php';
require_once DOL_DOCUMENT_ROOT.'/core/lib/holiday.lib.php';
require_once DOL_DOCUMENT_ROOT.'/holiday/common.inc.php';

// Get parameters
$myparam = GETPOST("myparam");
$action=GETPOST('action', 'alpha');
$id=GETPOST('id', 'int');
$userid = GETPOST('userid')?GETPOST('userid'):$user->id;

// Protection if external user
if ($user->societe_id > 0) accessforbidden();

$now=dol_now();


/*
 * Actions
 */

// If create a request
if ($action == 'create')
{
	$cp = new Holiday($db);

    // If no right to create a request
    if (($userid == $user->id && empty($user->rights->holiday->write)) || ($userid != $user->id && empty($user->rights->holiday->write_all)))
    {
    	$error++;
    	setEventMessages($langs->trans('CantCreateCP'), null, 'errors');
    	$action='request';
    }

    if (! $error)
    {
    	$db->begin();

	    $date_debut = dol_mktime(0, 0, 0, GETPOST('date_debut_month'), GETPOST('date_debut_day'), GETPOST('date_debut_year'));
	    $date_fin = dol_mktime(0, 0, 0, GETPOST('date_fin_month'), GETPOST('date_fin_day'), GETPOST('date_fin_year'));
	    $date_debut_gmt = dol_mktime(0, 0, 0, GETPOST('date_debut_month'), GETPOST('date_debut_day'), GETPOST('date_debut_year'), 1);
	    $date_fin_gmt = dol_mktime(0, 0, 0, GETPOST('date_fin_month'), GETPOST('date_fin_day'), GETPOST('date_fin_year'), 1);
	    $starthalfday=GETPOST('starthalfday');
	    $endhalfday=GETPOST('endhalfday');
	    $type=GETPOST('type');
	    $halfday=0;
	    if ($starthalfday == 'afternoon' && $endhalfday == 'morning') $halfday=2;
	    else if ($starthalfday == 'afternoon') $halfday=-1;
	    else if ($endhalfday == 'morning') $halfday=1;

	    $valideur = GETPOST('valideur');
	    $description = trim(GETPOST('description'));
	    $userID = GETPOST('userID');

    	// If no type
	    if ($type <= 0)
	    {
	        setEventMessages($langs->trans("ErrorFieldRequired", $langs->transnoentitiesnoconv("Type")), null, 'errors');
	        $error++;
	        $action='create';
	    }

	    // If no start date
	    if (empty($date_debut))
	    {
	        setEventMessages($langs->trans("NoDateDebut"), null, 'errors');
	        $error++;
	        $action='create';
	    }
	    // If no end date
	    if (empty($date_fin))
	    {
	        setEventMessages($langs->trans("NoDateFin"), null, 'errors');
	        $error++;
	        $action='create';
	    }
	    // If start date after end date
	    if ($date_debut > $date_fin)
	    {
	        setEventMessages($langs->trans("ErrorEndDateCP"), null, 'errors');
	        $error++;
	        $action='create';
	    }

	    // Check if there is already holiday for this period
	    $verifCP = $cp->verifDateHolidayCP($userID, $date_debut, $date_fin, $halfday);
	    if (! $verifCP)
	    {
	        setEventMessages($langs->trans("alreadyCPexist"), null, 'errors');
	        $error++;
	        $action='create';
	    }

	    // If there is no Business Days within request
	    $nbopenedday=num_open_day($date_debut_gmt, $date_fin_gmt, 0, 1, $halfday);
	    if($nbopenedday < 0.5)
	    {
	        setEventMessages($langs->trans("ErrorDureeCP"), null, 'errors');
	        $error++;
	        $action='create';
	    }

	    // If no validator designated
	    if ($valideur < 1)
	    {
	        setEventMessages($langs->transnoentitiesnoconv('InvalidValidatorCP'), null, 'errors');
	        $error++;
	    }

	    $result = 0;

	    $result = 0;

	    if (! $error)
	    {
    	    $cp->fk_user = $userid;
    	    $cp->description = $description;
    	    $cp->date_debut = $date_debut;
    	    $cp->date_fin = $date_fin;
    	    $cp->fk_validator = $valideur;
    		$cp->halfday = $halfday;
    		$cp->fk_type = $type;

    		$result = $cp->create($user);
	    }

	    // If no SQL error we redirect to the request card
	    if (! $error && $result > 0)
	    {
			$db->commit();

	    	header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id);
	        exit;
	    }
	    else
		{
	    	$db->rollback();
	    }
    }
}

if ($action == 'update')
{
	$date_debut = dol_mktime(0, 0, 0, GETPOST('date_debut_month'), GETPOST('date_debut_day'), GETPOST('date_debut_year'));
	$date_fin = dol_mktime(0, 0, 0, GETPOST('date_fin_month'), GETPOST('date_fin_day'), GETPOST('date_fin_year'));
	$date_debut_gmt = dol_mktime(0, 0, 0, GETPOST('date_debut_month'), GETPOST('date_debut_day'), GETPOST('date_debut_year'), 1);
	$date_fin_gmt = dol_mktime(0, 0, 0, GETPOST('date_fin_month'), GETPOST('date_fin_day'), GETPOST('date_fin_year'), 1);
	$starthalfday=GETPOST('starthalfday');
	$endhalfday=GETPOST('endhalfday');
	$halfday=0;
	if ($starthalfday == 'afternoon' && $endhalfday == 'morning') $halfday=2;
	else if ($starthalfday == 'afternoon') $halfday=-1;
	else if ($endhalfday == 'morning') $halfday=1;

    // If no right to modify a request
    if (! $user->rights->holiday->write)
    {
        header('Location: '.$_SERVER["PHP_SELF"].'?action=request&error=CantUpdate');
        exit;
    }

    $cp = new Holiday($db);
    $cp->fetch($id);

	$canedit=(($user->id == $cp->fk_user && $user->rights->holiday->write) || ($user->id != $cp->fk_user && $user->rights->holiday->write_all));

	// If under validation
    if ($cp->statut == 1)
    {
        // If this is the requestor or has read/write rights
        if ($canedit)
        {
            $valideur = $_POST['valideur'];
            $description = trim($_POST['description']);

            // If no start date
            if (empty($_POST['date_debut_'])) {
                header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id.'&action=edit&error=nodatedebut');
                exit;
            }

            // If no end date
            if (empty($_POST['date_fin_'])) {
                header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id.'&action=edit&error=nodatefin');
                exit;
            }

            // If start date after end date
            if ($date_debut > $date_fin) {
                header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id.'&action=edit&error=datefin');
                exit;
            }

            // If no validator designated
            if ($valideur < 1) {
                header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id.'&action=edit&error=Valideur');
                exit;
            }

            // If there is no Business Days within request
            $nbopenedday=num_open_day($date_debut_gmt, $date_fin_gmt, 0, 1, $halfday);
            if ($nbopenedday < 0.5)
            {
                header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id.'&action=edit&error=DureeHoliday');
                exit;
            }

            $cp->description = $description;
            $cp->date_debut = $date_debut;
            $cp->date_fin = $date_fin;
            $cp->fk_validator = $valideur;
			$cp->halfday = $halfday;

			// Update
			$verif = $cp->update($user->id);
            if ($verif > 0)
            {
                header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id);
                exit;
            }
            else
           {
                // Otherwise we display the request form with the SQL error message
                header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id.'&action=edit&error=SQL_Create&msg='.$cp->error);
                exit;
            }
        }
    } else {
        header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id);
        exit;
    }
}

// If delete of request
if ($action == 'confirm_delete' && GETPOST('confirm') == 'yes' && $user->rights->holiday->delete)
{
	$error=0;

	$db->begin();

	$cp = new Holiday($db);
	$cp->fetch($id);

	$canedit=(($user->id == $cp->fk_user && $user->rights->holiday->write) || ($user->id != $cp->fk_user && $user->rights->holiday->write_all));

    // If this is a rough draft
	if ($cp->statut == 1 || $cp->statut == 3)
	{
		// Si l'utilisateur à le droit de lire cette demande, il peut la supprimer
		if ($canedit)
		{
			$result=$cp->delete($cp->id);
		}
		else
		{
			$error = $langs->trans('ErrorCantDeleteCP');
		}
	}

	if (! $error)
	{
		$db->commit();
		header('Location: list.php');
		exit;
	}
	else
	{
		$db->rollback();
	}
}

// Si envoi de la demande
if ($action == 'confirm_send')
{
    $cp = new Holiday($db);
    $cp->fetch($id);

    $canedit=(($user->id == $cp->fk_user && $user->rights->holiday->write) || ($user->id != $cp->fk_user && $user->rights->holiday->write_all));

    // Si brouillon et créateur
    if($cp->statut == 1 && $canedit)
    {
        $cp->statut = 2;

        $verif = $cp->update($user->id);

        // Si pas d'erreur SQL on redirige vers la fiche de la demande
        if ($verif > 0)
        {
            // To
            $destinataire = new User($db);
            $destinataire->fetch($cp->fk_validator);
            $emailTo = $destinataire->email;

            if (!$emailTo)
            {
                dol_syslog("Expected validator has no email, so we redirect directly to finished page without sending email");
                header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id);
                exit;
            }

            // From
            $expediteur = new User($db);
            $expediteur->fetch($cp->fk_user);
            $emailFrom = $expediteur->email;

            // Subject
			$societeName = $conf->global->MAIN_INFO_SOCIETE_NOM;
            if (! empty($conf->global->MAIN_APPLICATION_TITLE)) $societeName = $conf->global->MAIN_APPLICATION_TITLE;

            $subject = $societeName." - ".$langs->transnoentitiesnoconv("HolidaysToValidate");

            // Content
            $message = $langs->transnoentitiesnoconv("Hello")." ".$destinataire->firstname.",\n";
            $message.= "\n";
            $message.= $langs->transnoentities("HolidaysToValidateBody")."\n";

            $delayForRequest = $cp->getConfCP('delayForRequest');
            //$delayForRequest = $delayForRequest * (60*60*24);

            $nextMonth = dol_time_plus_duree($now, $delayForRequest, 'd');

            // Si l'option pour avertir le valideur en cas de délai trop court
            if($cp->getConfCP('AlertValidatorDelay'))
            {
                if($cp->date_debut < $nextMonth)
                {
                    $message.= "\n";
                    $message.= $langs->transnoentities("HolidaysToValidateDelay",$cp->getConfCP('delayForRequest'))."\n";
                }
            }

            // Si l'option pour avertir le valideur en cas de solde inférieur à la demande
            if ($cp->getConfCP('AlertValidatorSolde'))
            {
            	$nbopenedday=num_open_day($cp->date_debut_gmt,$cp->date_fin_gmt,0,1,$cp->halfday);
                if ($nbopenedday > $cp->getCPforUser($cp->fk_user, $cp->fk_type))
                {
                    $message.= "\n";
                    $message.= $langs->transnoentities("HolidaysToValidateAlertSolde")."\n";
                }
            }

            $message.= "\n";
            $message.= "- ".$langs->transnoentitiesnoconv("Name")." : ".dolGetFirstLastname($expediteur->firstname, $expediteur->lastname)."\n";
            $message.= "- ".$langs->transnoentitiesnoconv("Period")." : ".dol_print_date($cp->date_debut,'day')." ".$langs->transnoentitiesnoconv("To")." ".dol_print_date($cp->date_fin,'day')."\n";
            $message.= "- ".$langs->transnoentitiesnoconv("Link")." : ".$dolibarr_main_url_root."/holiday/card.php?id=".$cp->id."\n\n";
            $message.= "\n";

            $mail = new CMailFile($subject,$emailTo,$emailFrom,$message);

            // Envoi du mail
            $result=$mail->sendfile();

            if (!$result)
            {
                header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id.'&error=mail&error_content='.$mail->error);
                exit;
            }
            header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id);
            exit;
        }
        else
        {
            // Sinon on affiche le formulaire de demande avec le message d'erreur SQL
            header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id.'&error=SQL_Create&msg='.$cp->error);
            exit;
        }
    }
}


// Si Validation de la demande
if ($action == 'confirm_valid')
{
    $cp = new Holiday($db);
    $cp->fetch($id);

    // Si statut en attente de validation et valideur = utilisateur
    if ($cp->statut == 2 && $user->id == $cp->fk_validator)
    {
        $cp->date_valid = dol_now();
        $cp->fk_user_valid = $user->id;
        $cp->statut = 3;

        $verif = $cp->update($user->id);

        // Si pas d'erreur SQL on redirige vers la fiche de la demande
        if ($verif > 0)
        {
            // Calculcate number of days consummed
            $nbopenedday=num_open_day($cp->date_debut_gmt,$cp->date_fin_gmt,0,1,$cp->halfday);

            $soldeActuel = $cp->getCpforUser($cp->fk_user, $cp->fk_type);
            $newSolde = $soldeActuel - ($nbopenedday * $cp->getConfCP('nbHolidayDeducted'));

            // On ajoute la modification dans le LOG
            $cp->addLogCP($user->id, $cp->fk_user, $langs->transnoentitiesnoconv("Holidays"), $newSolde, $cp->fk_type);

            // Mise à jour du solde
            $cp->updateSoldeCP($cp->fk_user, $newSolde, $cp->fk_type);

            // To
            $destinataire = new User($db);
            $destinataire->fetch($cp->fk_user);
            $emailTo = $destinataire->email;

            if (!$emailTo)
            {
                dol_syslog("User that request leave has no email, so we redirect directly to finished page without sending email");
                header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id);
                exit;
            }

            // From
            $expediteur = new User($db);
            $expediteur->fetch($cp->fk_validator);
            $emailFrom = $expediteur->email;

            // Subject
			$societeName = $conf->global->MAIN_INFO_SOCIETE_NOM;
            if (! empty($conf->global->MAIN_APPLICATION_TITLE)) $societeName = $conf->global->MAIN_APPLICATION_TITLE;

            $subject = $societeName." - ".$langs->transnoentitiesnoconv("HolidaysValidated");

            // Content
            $message = $langs->transnoentitiesnoconv("Hello")." ".$destinataire->firstname.",\n";
            $message.= "\n";
            $message.=  $langs->transnoentities("HolidaysValidatedBody", dol_print_date($cp->date_debut,'day'),dol_print_date($cp->date_fin,'day'))."\n";

            $message.= "- ".$langs->transnoentitiesnoconv("ValidatedBy")." : ".dolGetFirstLastname($expediteur->firstname, $expediteur->lastname)."\n";

            $message.= "- ".$langs->transnoentitiesnoconv("Link")." : ".$dolibarr_main_url_root."/holiday/card.php?id=".$cp->id."\n\n";
            $message.= "\n";

            $mail = new CMailFile($subject,$emailTo,$emailFrom,$message);

            // Envoi du mail
            $result=$mail->sendfile();

            if(!$result) {
                header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id.'&error=mail&error_content='.$mail->error);
                exit;
            }

            header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id);
            exit;
        } else {
            // Sinon on affiche le formulaire de demande avec le message d'erreur SQL
            header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id.'&error=SQL_Create&msg='.$cp->error);
            exit;
        }

    }

}

if ($action == 'confirm_refuse')
{
    if (! empty($_POST['detail_refuse']))
    {
        $cp = new Holiday($db);
        $cp->fetch($id);

        // Si statut en attente de validation et valideur = utilisateur
        if ($cp->statut == 2 && $user->id == $cp->fk_validator)
        {
            $cp->date_refuse = date('Y-m-d H:i:s', time());
            $cp->fk_user_refuse = $user->id;
            $cp->statut = 5;
            $cp->detail_refuse = $_POST['detail_refuse'];

            $verif = $cp->update($user->id);

            // Si pas d'erreur SQL on redirige vers la fiche de la demande
            if ($verif > 0)
            {
                // To
                $destinataire = new User($db);
                $destinataire->fetch($cp->fk_user);
                $emailTo = $destinataire->email;

                if (!$emailTo)
                {
                    header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id);
                    exit;
                }

                // From
                $expediteur = new User($db);
                $expediteur->fetch($cp->fk_validator);
                $emailFrom = $expediteur->email;

	            // Subject
				$societeName = $conf->global->MAIN_INFO_SOCIETE_NOM;
	            if (! empty($conf->global->MAIN_APPLICATION_TITLE)) $societeName = $conf->global->MAIN_APPLICATION_TITLE;

	            $subject = $societeName." - ".$langs->transnoentitiesnoconv("HolidaysRefused");

                // Content
            	$message = $langs->transnoentitiesnoconv("Hello")." ".$destinataire->firstname.",\n";
	            $message.= "\n";
                $message.= $langs->transnoentities("HolidaysRefusedBody", dol_print_date($cp->date_debut,'day'), dol_print_date($cp->date_fin,'day'))."\n";
                $message.= GETPOST('detail_refuse','alpha')."\n\n";

	            $message.= "- ".$langs->transnoentitiesnoconv("ModifiedBy")." : ".dolGetFirstLastname($expediteur->firstname, $expediteur->lastname)."\n";

	            $message.= "- ".$langs->transnoentitiesnoconv("Link")." : ".$dolibarr_main_url_root."/holiday/card.php?id=".$cp->id."\n\n";
                $message.= "\n";

                $mail = new CMailFile($subject,$emailTo,$emailFrom,$message);

                // Envoi du mail
                $result=$mail->sendfile();

                if(!$result) {
                    header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id.'&error=mail&error_content='.$mail->error);
                    exit;
                }

                header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id);
                exit;
            } else {
                // Sinon on affiche le formulaire de demande avec le message d'erreur SQL
                header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id.'&error=SQL_Create&msg='.$cp->error);
                exit;
            }

        }

    } else {
        header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id.'&error=NoMotifRefuse');
        exit;
    }
}

// Si Validation de la demande
if ($action == 'confirm_cancel' && GETPOST('confirm') == 'yes')
{
    $cp = new Holiday($db);
    $cp->fetch($id);

    // Si statut en attente de validation et valideur = utilisateur
    if (($cp->statut == 2 || $cp->statut == 3) && ($user->id == $cp->fk_validator || $user->id == $cp->fk_user))
    {
    	$db->begin();

    	$oldstatus = $cp->statut;
        $cp->date_cancel = dol_now();
        $cp->fk_user_cancel = $user->id;
        $cp->statut = 4;

        $result = $cp->update($user->id);

        if ($result >= 0 && $oldstatus == 3)	// holiday was already validated, status 3, so we must increase back sold
        {
        	// Calculcate number of days consummed
        	$nbopenedday=num_open_day($cp->date_debut_gmt,$cp->date_fin_gmt,0,1,$cp->halfday);

        	$soldeActuel = $cp->getCpforUser($cp->fk_user, $cp->fk_type);
        	$newSolde = $soldeActuel + ($nbopenedday * $cp->getConfCP('nbHolidayDeducted'));

        	// On ajoute la modification dans le LOG
        	$result1=$cp->addLogCP($user->id, $cp->fk_user, $langs->transnoentitiesnoconv("HolidaysCancelation"), $newSolde, $cp->fk_type);

        	// Mise à jour du solde
        	$result2=$cp->updateSoldeCP($cp->fk_user, $newSolde, $cp->fk_type);

        	if ($result1 < 0 || $result2 < 0)
        	{
        		$error = $langs->trans('ErrorCantDeleteCP');
        	}
        }

        if (! $error)
        {
        	$db->commit();
        }
        else
        {
        	$db->rollback();
        }

        // Si pas d'erreur SQL on redirige vers la fiche de la demande
        if (! $error && $result > 0)
        {
            // To
            $destinataire = new User($db);
            $destinataire->fetch($cp->fk_user);
            $emailTo = $destinataire->email;

            if (!$emailTo)
            {
                header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id);
                exit;
            }

            // From
            $expediteur = new User($db);
            $expediteur->fetch($cp->fk_user_cancel);
            $emailFrom = $expediteur->email;

            // Subject
			$societeName = $conf->global->MAIN_INFO_SOCIETE_NOM;
            if (! empty($conf->global->MAIN_APPLICATION_TITLE)) $societeName = $conf->global->MAIN_APPLICATION_TITLE;

            $subject = $societeName." - ".$langs->transnoentitiesnoconv("HolidaysCanceled");

            // Content
           	$message = $langs->transnoentitiesnoconv("Hello")." ".$destinataire->firstname.",\n";
            $message.= "\n";

            $message.= $langs->transnoentities("HolidaysCanceledBody", dol_print_date($cp->date_debut,'day'), dol_print_date($cp->date_fin,'day'))."\n";
            $message.= "- ".$langs->transnoentitiesnoconv("ModifiedBy")." : ".dolGetFirstLastname($expediteur->firstname, $expediteur->lastname)."\n";

            $message.= "- ".$langs->transnoentitiesnoconv("Link")." : ".$dolibarr_main_url_root."/holiday/card.php?id=".$cp->id."\n\n";
            $message.= "\n";

            $mail = new CMailFile($subject,$emailTo,$emailFrom,$message);

            // Envoi du mail
            $result=$mail->sendfile();

            if(!$result)
            {
                header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id.'&error=mail&error_content='.$mail->error);
                exit;
            }

            header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id);
            exit;
        }
        else
        {
            // Sinon on affiche le formulaire de demande avec le message d'erreur SQL
            header('Location: '.$_SERVER["PHP_SELF"].'?id='.$cp->id.'&error=SQL_Create&msg='.$cp->error);
            exit;
        }

    }

}



/*
 * View
 */

$form = new Form($db);
$cp = new Holiday($db);

$listhalfday=array('morning'=>$langs->trans("Morning"),"afternoon"=>$langs->trans("Afternoon"));

llxHeader('', $langs->trans('CPTitreMenu'));

if (empty($id) || $action == 'add' || $action == 'request' || $action == 'create')
{
    // Si l'utilisateur n'a pas le droit de faire une demande
    if (($userid == $user->id && empty($user->rights->holiday->write)) || ($userid != $user->id && empty($user->rights->holiday->write_all)))
    {
        $errors[]=$langs->trans('CantCreateCP');
    }
    else
    {
        // Formulaire de demande de congés payés
        print load_fiche_titre($langs->trans('MenuAddCP'), '', 'title_hrm.png');

        // Si il y a une erreur
        if (GETPOST('error')) {

            switch(GETPOST('error')) {
                case 'datefin' :
                    $errors[] = $langs->trans('ErrorEndDateCP');
                    break;
                case 'SQL_Create' :
                    $errors[] = $langs->trans('ErrorSQLCreateCP').' <b>'.htmlentities($_GET['msg']).'</b>';
                    break;
                case 'CantCreate' :
                    $errors[] = $langs->trans('CantCreateCP');
                    break;
                case 'Valideur' :
                    $errors[] = $langs->trans('InvalidValidatorCP');
                    break;
                case 'nodatedebut' :
                    $errors[] = $langs->trans('NoDateDebut');
                    break;
                case 'nodatefin' :
                    $errors[] = $langs->trans('NoDateFin');
                    break;
                case 'DureeHoliday' :
                    $errors[] = $langs->trans('ErrorDureeCP');
                    break;
                case 'alreadyCP' :
                    $errors[] = $langs->trans('alreadyCPexist');
                    break;
            }

	        setEventMessages($errors, null, 'errors');
        }


        $delayForRequest = $cp->getConfCP('delayForRequest');
        //$delayForRequest = $delayForRequest * (60*60*24);

        $nextMonth = dol_time_plus_duree($now, $delayForRequest, 'd');

		print '<script type="text/javascript">
	    function valider()
	    {
    	    if(document.demandeCP.date_debut_.value != "")
    	    {
	           	if(document.demandeCP.date_fin_.value != "")
	           	{
	               if(document.demandeCP.valideur.value != "-1") {
	                 return true;
	               }
	               else {
	                 alert("'.dol_escape_js($langs->transnoentities('InvalidValidatorCP')).'");
	                 return false;
	               }
	            }
	            else
	            {
	              alert("'.dol_escape_js($langs->transnoentities('NoDateFin')).'");
	              return false;
	            }
	        }
	        else
	        {
	           alert("'.dol_escape_js($langs->transnoentities('NoDateDebut')).'");
	           return false;
	        }
       	}
       </script>'."\n";

        // Formulaire de demande
        print '<form method="POST" action="'.$_SERVER['PHP_SELF'].'" onsubmit="return valider()" name="demandeCP">'."\n";
        print '<input type="hidden" name="action" value="create" />'."\n";
        print '<input type="hidden" name="userID" value="'.$userid.'" />'."\n";

        dol_fiche_head();

        $out='';
        $typeleaves=$cp->getTypes(1,1);
    	foreach($typeleaves as $key => $val)
		{
			$nb_type = $cp->getCPforUser($user->id, $val['rowid']);
			$nb_holiday += $nb_type;
			$out .= ' - '.$val['label'].': <strong>'.($nb_type?price2num($nb_type):0).'</strong><br>';
		}
        print $langs->trans('SoldeCPUser', round($nb_holiday,5)).'<br>';
		print $out;

        dol_fiche_end();


        dol_fiche_head();

        //print '<span>'.$langs->trans('DelayToRequestCP',$cp->getConfCP('delayForRequest')).'</span><br /><br />';

        print '<table class="border" width="100%">';
        print '<tbody>';

        // User
        print '<tr>';
        print '<td class="titlefield fieldrequired">'.$langs->trans("User").'</td>';
        print '<td>';
        if (empty($user->rights->holiday->write_all))
        {
        	print $form->select_dolusers($userid, 'useridbis', 0, '', 1, '', '', 0, 0, 0, '', 0, '', 'maxwidth300');
        	print '<input type="hidden" name="userid" value="'.$userid.'">';
        }
        else print $form->select_dolusers(GETPOST('userid')?GETPOST('userid'):$user->id,'userid',0,'',0);
        print '</td>';
        print '</tr>';

        // Type
        print '<tr>';
        print '<td class="fieldrequired">'.$langs->trans("Type").'</td>';
        print '<td>';
        $typeleaves=$cp->getTypes(1,-1);
        $arraytypeleaves=array();
        foreach($typeleaves as $key => $val)
        {
        	$labeltoshow = $val['label'];
        	$labeltoshow .= ($val['delay'] > 0 ? ' ('.$langs->trans("NoticePeriod").': '.$val['delay'].' '.$langs->trans("days").')':'');
			$arraytypeleaves[$val['rowid']]=$labeltoshow;
        }
        print $form->selectarray('type', $arraytypeleaves, (GETPOST('type')?GETPOST('type'):''), 1);
        if ($user->admin) print info_admin($langs->trans("YouCanChangeValuesForThisListFromDictionarySetup"),1);
        print '</td>';
        print '</tr>';

        // Date start
        print '<tr>';
        print '<td class="fieldrequired">'.$langs->trans("DateDebCP").' ('.$langs->trans("FirstDayOfHoliday").')</td>';
        print '<td>';
        // Si la demande ne vient pas de l'agenda
        if (! GETPOST('date_debut_')) {
            $form->select_date(-1, 'date_debut_', 0, 0, 0, '', 1, 1);
        } else {
            $tmpdate = dol_mktime(0, 0, 0, GETPOST('date_debut_month'), GETPOST('date_debut_day'), GETPOST('date_debut_year'));
            $form->select_date($tmpdate, 'date_debut_', 0, 0, 0, '', 1, 1);
        }
        print ' &nbsp; &nbsp; ';
        print $form->selectarray('starthalfday', $listhalfday, (GETPOST('starthalfday')?GETPOST('starthalfday'):'morning'));
        print '</td>';
        print '</tr>';

        // Date end
        print '<tr>';
        print '<td class="fieldrequired">'.$langs->trans("DateFinCP").' ('.$langs->trans("LastDayOfHoliday").')</td>';
        print '<td>';
        // Si la demande ne vient pas de l'agenda
        if (! GETPOST('date_fin_')) {
            $form->select_date(-1,'date_fin_', 0, 0, 0, '', 1, 1);
        } else {
            $tmpdate = dol_mktime(0, 0, 0, GETPOST('date_fin_month'), GETPOST('date_fin_day'), GETPOST('date_fin_year'));
            $form->select_date($tmpdate,'date_fin_', 0, 0, 0, '', 1, 1);
        }
        print ' &nbsp; &nbsp; ';
        print $form->selectarray('endhalfday', $listhalfday, (GETPOST('endhalfday')?GETPOST('endhalfday'):'afternoon'));
        print '</td>';
        print '</tr>';

        // Approved by
        print '<tr>';
        print '<td class="fieldrequired">'.$langs->trans("ReviewedByCP").'</td>';
        print '<td>';
        print $form->select_dolusers((GETPOST('valideur')>0?GETPOST('valideur'):$user->fk_user), "valideur", 1, ($user->admin ? '' : array($user->id)), 0, '', 0, 0, 0, 0, '', 0, '', '', 1);	// By default, hierarchical parent
        print '</td>';
        print '</tr>';

        // Description
        print '<tr>';
        print '<td>'.$langs->trans("DescCP").'</td>';
        print '<td>';
        print '<textarea name="description" class="flat" rows="'.ROWS_3.'" cols="70">'.GETPOST('description').'</textarea>';
        print '</td>';
        print '</tr>';

        print '</tbody>';
        print '</table>';

        dol_fiche_end();

        print '<div class="center">';
        print '<input type="submit" value="'.$langs->trans("SendRequestCP").'" name="bouton" class="button">';
        print '&nbsp; &nbsp; ';
        print '<input type="button" value="'.$langs->trans("Cancel").'" class="button" onclick="history.go(-1)">';
        print '</div>';

        print '</from>'."\n";
    }
}
else
{
    if ($error)
    {
        print '<div class="tabBar">';
        print $error;
        print '<br><br><input type="button" value="'.$langs->trans("ReturnCP").'" class="button" onclick="history.go(-1)" />';
        print '</div>';
    }
    else
    {
        // Affichage de la fiche d'une demande de congés payés
        if ($id > 0)
        {
            $cp->fetch($id);

			$canedit=(($user->id == $cp->fk_user && $user->rights->holiday->write) || ($user->id != $cp->fk_user && $user->rights->holiday->write_all));

            $valideur = new User($db);
            $valideur->fetch($cp->fk_validator);

            $userRequest = new User($db);
            $userRequest->fetch($cp->fk_user);

            //print load_fiche_titre($langs->trans('TitreRequestCP'));

            // Si il y a une erreur
            if (GETPOST('error'))
            {
                switch(GETPOST('error'))
                {
                    case 'datefin' :
                        $errors[] = $langs->transnoentitiesnoconv('ErrorEndDateCP');
                        break;
                    case 'SQL_Create' :
                        $errors[] = $langs->transnoentitiesnoconv('ErrorSQLCreateCP').' '.$_GET['msg'];
                        break;
                    case 'CantCreate' :
                        $errors[] = $langs->transnoentitiesnoconv('CantCreateCP');
                        break;
                    case 'Valideur' :
                        $errors[] = $langs->transnoentitiesnoconv('InvalidValidatorCP');
                        break;
                    case 'nodatedebut' :
                        $errors[] = $langs->transnoentitiesnoconv('NoDateDebut');
                        break;
                    case 'nodatefin' :
                        $errors[] = $langs->transnoentitiesnoconv('NoDateFin');
                        break;
                    case 'DureeHoliday' :
                        $errors[] = $langs->transnoentitiesnoconv('ErrorDureeCP');
                        break;
                    case 'NoMotifRefuse' :
                        $errors[] = $langs->transnoentitiesnoconv('NoMotifRefuseCP');
                        break;
                    case 'mail' :
                        $errors[] = $langs->transnoentitiesnoconv('ErrorMailNotSend')."\n".$_GET['error_content'];
                        break;
                }

	            setEventMessages($errors, null, 'errors');
            }

            // On vérifie si l'utilisateur à le droit de lire cette demande
            if ($canedit)
            {
                if ($action == 'delete')
                {
                    if ($user->rights->holiday->delete)
                    {
                        print $form->formconfirm($_SERVER["PHP_SELF"]."?id=".$cp->id,$langs->trans("TitleDeleteCP"),$langs->trans("ConfirmDeleteCP"),"confirm_delete", '', 0, 1);
                    }
                }

                // Si envoi en validation
                if ($action == 'sendToValidate' && $cp->statut == 1)
                {
                    print $form->formconfirm($_SERVER["PHP_SELF"]."?id=".$cp->id,$langs->trans("TitleToValidCP"),$langs->trans("ConfirmToValidCP"),"confirm_send", '', 1, 1);
                }

                // Si validation de la demande
                if ($action == 'valid')
                {
                    print $form->formconfirm($_SERVER["PHP_SELF"]."?id=".$cp->id,$langs->trans("TitleValidCP"),$langs->trans("ConfirmValidCP"),"confirm_valid", '', 1, 1);
                }

                // Si refus de la demande
                if ($action == 'refuse')
                {
                    $array_input = array(array('type'=>"text",'label'=> $langs->trans('DetailRefusCP'),'name'=>"detail_refuse",'size'=>"50",'value'=>""));
                    print $form->formconfirm($_SERVER["PHP_SELF"]."?id=".$cp->id."&action=confirm_refuse", $langs->trans("TitleRefuseCP"), $langs->trans('ConfirmRefuseCP'), "confirm_refuse", $array_input, 1, 0);
                }

                // Si annulation de la demande
                if ($action == 'cancel')
                {
                    print $form->formconfirm($_SERVER["PHP_SELF"]."?id=".$cp->id,$langs->trans("TitleCancelCP"),$langs->trans("ConfirmCancelCP"),"confirm_cancel", '', 1, 1);
                }

                $head=holiday_prepare_head($cp);


                if ($action == 'edit' && $cp->statut == 1)
                {
                    $edit = true;
                    print '<form method="post" action="'.$_SERVER['PHP_SELF'].'?id='.$cp->id.'">'."\n";
                    print '<input type="hidden" name="action" value="update"/>'."\n";
                    print '<input type="hidden" name="id" value="'.$cp->id.'" />'."\n";
                }

                dol_fiche_head($head,'card',$langs->trans("CPTitreMenu"),0,'holiday');

                print '<table class="border" width="100%">';
                print '<tbody>';

                $linkback='<a href="'.DOL_URL_ROOT.'/holiday/list.php">'.$langs->trans("BackToList").'</a>';

                print '<tr>';
                print '<td class="titlefield">'.$langs->trans("Ref").'</td>';
                print '<td>';
                print $form->showrefnav($cp, 'id', $linkback, 1, 'rowid', 'ref');
                print '</td>';
                print '</tr>';

                print '<td>'.$langs->trans("User").'</td>';
        		print '<td>';
        		print $userRequest->getNomUrl(1, 'leave');
        		print '</td></tr>';

		        // Type
		        print '<tr>';
		        print '<td>'.$langs->trans("Type").'</td>';
		        print '<td>';
		        $typeleaves=$cp->getTypes(1,-1);
		        print $typeleaves[$cp->fk_type]['label'];
		        print '</td>';
		        print '</tr>';

			    $starthalfday=($cp->halfday == -1 || $cp->halfday == 2)?'afternoon':'morning';
			    $endhalfday=($cp->halfday == 1 || $cp->halfday == 2)?'morning':'afternoon';

                if(!$edit)
                {
                    print '<tr>';
                    print '<td>'.$langs->trans('DateDebCP').' ('.$langs->trans("FirstDayOfHoliday").')</td>';
                    print '<td>'.dol_print_date($cp->date_debut,'day');
			        print ' &nbsp; &nbsp; ';
                    print $langs->trans($listhalfday[$starthalfday]);
                    print '</td>';
                    print '</tr>';
                }
                else
                {
                    print '<tr>';
                    print '<td>'.$langs->trans('DateDebCP').' ('.$langs->trans("FirstDayOfHoliday").')</td>';
                    print '<td>';
                    $form->select_date($cp->date_debut,'date_debut_');
			        print ' &nbsp; &nbsp; ';
        			print $form->selectarray('starthalfday', $listhalfday, (GETPOST('starthalfday')?GETPOST('starthalfday'):$starthalfday));
                    print '</td>';
                    print '</tr>';
                }

                if (!$edit)
                {
                    print '<tr>';
                    print '<td>'.$langs->trans('DateFinCP').' ('.$langs->trans("LastDayOfHoliday").')</td>';
                    print '<td>'.dol_print_date($cp->date_fin,'day');
                    print ' &nbsp; &nbsp; ';
                    print $langs->trans($listhalfday[$endhalfday]);
                    print '</td>';
                    print '</tr>';
                }
                else
                {
                    print '<tr>';
                    print '<td>'.$langs->trans('DateFinCP').' ('.$langs->trans("LastDayOfHoliday").')</td>';
                    print '<td>';
                    $form->select_date($cp->date_fin,'date_fin_');
			        print ' &nbsp; &nbsp; ';
        			print $form->selectarray('endhalfday', $listhalfday, (GETPOST('endhalfday')?GETPOST('endhalfday'):$endhalfday));
                    print '</td>';
                    print '</tr>';
                }
                print '<tr>';
                print '<td>'.$langs->trans('NbUseDaysCP').'</td>';
                print '<td>'.num_open_day($cp->date_debut_gmt, $cp->date_fin_gmt, 0, 1, $cp->halfday).'</td>';
                print '</tr>';

                // Status
                print '<tr>';
                print '<td>'.$langs->trans('StatutCP').'</td>';
                print '<td>'.$cp->getLibStatut(2).'</td>';
                print '</tr>';
                if ($cp->statut == 5)
                {
                	print '<tr>';
                	print '<td>'.$langs->trans('DetailRefusCP').'</td>';
                	print '<td>'.$cp->detail_refuse.'</td>';
                	print '</tr>';
                }

                // Description
                if (!$edit)
                {
                    print '<tr>';
                    print '<td>'.$langs->trans('DescCP').'</td>';
                    print '<td>'.nl2br($cp->description).'</td>';
                    print '</tr>';
                }
                else
                {
                    print '<tr>';
                    print '<td>'.$langs->trans('DescCP').'</td>';
                    print '<td><textarea name="description" class="flat" rows="'.ROWS_3.'" cols="70">'.$cp->description.'</textarea></td>';
                    print '</tr>';
                }

                print '</tbody>';
                print '</table>'."\n";

                print '<br><br>';

				// Info workflow
                print '<table class="border" width="50%">'."\n";
                print '<tbody>';
                print '<tr class="liste_titre">';
                print '<td colspan="2">'.$langs->trans("InfosWorkflowCP").'</td>';
                print '</tr>';

                if (! empty($cp->fk_user_create))
                {
                	$userCreate=new User($db);
                	$userCreate->fetch($cp->fk_user_create);
	                print '<tr>';
	                print '<td>'.$langs->trans('RequestByCP').'</td>';
	                print '<td>'.$userCreate->getNomUrl(1).'</td>';
	                print '</tr>';
                }

                if (!$edit) {
                    print '<tr>';
                    print '<td width="50%">'.$langs->trans('ReviewedByCP').'</td>';
                    print '<td>'.$valideur->getNomUrl(1).'</td>';
                    print '</tr>';
                } else {
                    print '<tr>';
                    print '<td width="50%">'.$langs->trans('ReviewedByCP').'</td>';
                    print '<td>';
        			print $form->select_dolusers($cp->fk_user, "valideur", 1, ($user->admin ? '' : array($user->id)));	// By default, hierarchical parent
                    print '</td>';
                    print '</tr>';
                }

                print '<tr>';
                print '<td>'.$langs->trans('DateCreateCP').'</td>';
                print '<td>'.dol_print_date($cp->date_create,'dayhour').'</td>';
                print '</tr>';
                if ($cp->statut == 3) {
                    print '<tr>';
                    print '<td>'.$langs->trans('DateValidCP').'</td>';
                    print '<td>'.dol_print_date($cp->date_valid,'dayhour').'</td>';
                    print '</tr>';
                }
                if ($cp->statut == 4) {
                    print '<tr>';
                    print '<td>'.$langs->trans('DateCancelCP').'</td>';
                    print '<td>'.dol_print_date($cp->date_cancel,'dayhour').'</td>';
                    print '</tr>';
                }
                if ($cp->statut == 5) {
                    print '<tr>';
                    print '<td>'.$langs->trans('DateRefusCP').'</td>';
                    print '<td>'.dol_print_date($cp->date_refuse,'dayhour').'</td>';
                    print '</tr>';
                }
                print '</tbody>';
                print '</table>';

                dol_fiche_end();

                if ($action == 'edit' && $cp->statut == 1)
                {
                    print '<div align="center">';
                    if ($canedit && $cp->statut == 1)
                    {
                        print '<input type="submit" value="'.$langs->trans("Save").'" class="button">';
                    }
                    print '</div>';

                    print '</form>';
                }


                if (! $edit)
                {
		            print '<div class="tabsAction">';

                    // Boutons d'actions
                    if ($canedit && $cp->statut == 1)
                    {
                        print '<a href="'.$_SERVER["PHP_SELF"].'?id='.$cp->id.'&action=edit" class="butAction">'.$langs->trans("EditCP").'</a>';
                    }
                    if ($canedit && $cp->statut == 1)
                    {
                        print '<a href="'.$_SERVER["PHP_SELF"].'?id='.$cp->id.'&action=sendToValidate" class="butAction">'.$langs->trans("Validate").'</a>';
                    }
                    if ($user->rights->holiday->delete && $cp->statut == 1)	// If draft
                    {
                    	print '<a href="'.$_SERVER["PHP_SELF"].'?id='.$cp->id.'&action=delete" class="butActionDelete">'.$langs->trans("DeleteCP").'</a>';
                    }

                    if ($user->id == $cp->fk_validator && $cp->statut == 2)
                    {
                        print '<a href="'.$_SERVER["PHP_SELF"].'?id='.$cp->id.'&action=valid" class="butAction">'.$langs->trans("Approve").'</a>';
                        print '<a href="'.$_SERVER["PHP_SELF"].'?id='.$cp->id.'&action=refuse" class="butAction">'.$langs->trans("ActionRefuseCP").'</a>';
                    }

                    if (($user->id == $cp->fk_validator || $user->id == $cp->fk_user) && ($cp->statut == 2 || $cp->statut == 3))	// Status validated or approved
                    {
                    	if (($cp->date_debut > dol_now()) || $user->admin) print '<a href="'.$_SERVER["PHP_SELF"].'?id='.$cp->id.'&action=cancel" class="butAction">'.$langs->trans("ActionCancelCP").'</a>';
                    	else print '<a href="#" class="butActionRefused" title="'.$langs->trans("HolidayStarted").'">'.$langs->trans("ActionCancelCP").'</a>';
                    }

                    print '</div>';
                }

            } else {
                print '<div class="tabBar">';
                print $langs->trans('ErrorUserViewCP');
                print '<br /><br /><input type="button" value="'.$langs->trans("ReturnCP").'" class="button" onclick="history.go(-1)" />';
                print '</div>';
            }

        } else {
            print '<div class="tabBar">';
            print $langs->trans('ErrorIDFicheCP');
            print '<br /><br /><input type="button" value="'.$langs->trans("ReturnCP").'" class="button" onclick="history.go(-1)" />';
            print '</div>';
        }

    }

}

// End of page
llxFooter();

if (is_object($db)) $db->close();
