using namespace std;

/**
   This file contains explanations and test code for all of the features of
   yagol. If you want to see what you can do with yagol, you've come to the
   right place. Scroll forward and enjoy.
**/

#ifndef Test_h
#include "Test.h"
#endif

#ifndef Yagol_h
#include "Yagol.h"
#endif

#ifndef RCFile_h
#include "RCFile.h"
#endif

#ifndef std_iostream
#define std_iostream
#include <iostream>
#endif

#ifndef std_string
#define std_string
#include <string>
#endif

#ifndef std_vector
#define std_vector
#include <vector>
#endif

#ifndef yagol_unistd_h
#define yagol_unistd_h
#include <unistd.h>
#endif

#ifdef HAVE_SSTREAM
    #ifndef yagol_sstream
    #define yagol_sstream
    #include <sstream>
    #endif
#else
    #ifndef yagol_strstream
    #define yagol_strstream
    #include <strstream>
    #endif
#endif

#ifndef std_fstream
#define std_fstream
#include <fstream>
#endif

#ifndef yagol_stdlib_h
#define yagol_stdlib_h
#include <stdlib.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>

class FeatureTest : public Test {
public:
    FeatureTest();

    virtual ~FeatureTest();

    void run();

    void testMultiWithoutEndTag();

    void testMultiNestedOptionLike();

    void testMultiStart();

    void testDescription();

    void testHowTo();

    void testConfigFileValue();

    void testConfigFileSequence();

    void testEnvironmentVariable();

    void testRangedOption();

    void testSizeValidation();

    void testRangedGroupOption();

    void testInputValidation();

    void testInputFile();

    void testOutputFile();

    void testNamespaces();

    void testShortNames();

    void testLinks();

    void testConfigOnlyOptions();
    
protected:

    void testConfigLine(const string& fname, const string& line);

};


FeatureTest::FeatureTest() : Test(true)
{
}

FeatureTest::~FeatureTest() 
{
}

void FeatureTest::run() 
{
    testMultiWithoutEndTag();
    testMultiNestedOptionLike();
    testMultiStart();
    testDescription();
    testHowTo();
    testConfigFileValue();
    testConfigFileSequence();
    testEnvironmentVariable();
    testRangedOption();
    testSizeValidation();
    testRangedGroupOption();
    testInputValidation();
    testInputFile();
    testOutputFile();
    testNamespaces();
    testShortNames();
    testLinks();
    testConfigOnlyOptions();
}



ostream& operator<<(ostream& os, const vector<string>& vec)
{
    os << "(";
    vector<string>::const_iterator stop = vec.end();
    vector<string>::const_iterator it = vec.begin();
    while (it != stop) {
        if (it != vec.begin()) {
            os << ", ";
        }
        os << *it;
        ++it;
    }
    os << ")";
    return os;
}


/**
   The Big Idea

   yagol, stands for "yagol: Another Getopt Library". The intent is to offer a
   command-line syntax and processing that are more intuitive for the user and
   for the programmer, and to handle the most common actions that rely on
   options. Its syntax is similar to the POSIX standard, but is not an exact
   match. Automatic conversions are done for all primitive C++ data types and
   for strings.

   Usage
   
   Single-value options are straightforward and predictable:
   
   --size 24      An option that takes an integer
   --user Fred    A string option
   --pi 3.14      A real number (float or double) option
   
   Note that arguments are separated by whitespace from the option tag itself.
   That is, rather than "--size=24", one would write "--size 24".
     
   For boolean options, the syntax is slightly different:
     
   --read         A boolean option, equivalent to POSIX "--read true"
   --noread       A boolean option, equivalent to POSIX "--read false"
   
   Note that the boolean options are the only option types that do not take a
   following argument. There is no concept of an "argument optional" option.

   File Options are Special
   
   Functionality is extended for options that correspond to files. An input
   file is verified that it exists and is readable. An output file must either
   not exist, or must be writable. If these conditions do not obtain, an
   exception is thrown.
   
   More than One Value per Option
   
   Deviating from the POSIX standard, an option can be tied to multiple values
   without repeatedly specifying the option tag. For yagol, an option that is
   tied to multiple values has the same beginning tag as single-value options.
   However, the arguments following the option tag are "consumed" until either
   "--end-of-<tag>" is found, or to the end of the argument list. For example,
   the two following examples are equivalent in their population of the "input"
   option:
   
   --input One.h Two.cpp Three.txt --end-of-input --another-option
     
   --input One.h Two.cpp Three.txt
**/

void FeatureTest::testMultiWithoutEndTag() 
{
    vector<string> inputOne;
    char* cmdLineOne[] = { "app", "--input", "One.h", "Two.cpp", "Three.txt", "--end-of-input", "--another-option" };

    vector<string> inputTwo;
    char* cmdLineTwo[] = { "app", "--input", "One.h", "Two.cpp", "Three.txt" };

    yagol::AppOptionSet optsOne("app");
    optsOne.addOption("input", "input to this app", &inputOne);
    // not handling exceptions
    optsOne.process(sizeof(cmdLineOne) / sizeof(cmdLineOne[0]), cmdLineOne);
    
    yagol::AppOptionSet optsTwo("app");
    optsTwo.addOption("input", "input to this app", &inputTwo);
    // not handling exceptions
    optsTwo.process(sizeof(cmdLineTwo) / sizeof(cmdLineTwo[0]), cmdLineTwo);

    TESTEQ(inputTwo.size(), 3UL, "size of input two");
    TESTEQ(inputOne[1], string("Two.cpp"), "value of input one at index 1");
    TESTEQ(inputOne, inputTwo, "comparison of vectors");
}

/**
   Note that arguments added to the multiple-value option are examined for usage
   of another option "within" the multiple-value option. For example, each of
   the following two examples results in an exception, if such checking is set
   to "strict":
          
   --input One.h Two.cpp Three.txt --another-option --end-of-input
   
   --input One.h Two.cpp Three.txt --another-option

   If checking is not set to "strict", the value at index 3 of the input option
   will be "--another-option".
**/

void FeatureTest::testMultiStart() 
{
    vector<string> inputOne;
    char* cmdLineOne[] = { "app", "--input", "One.h", "Two.cpp", "Three.txt", "--another-option", "--end-of-input" };

    vector<string> inputTwo;
    char* cmdLineTwo[] = { "app", "--input", "One.h", "Two.cpp", "Three.txt", "--another-option" };

    {
        // should not get exceptions here
        
        yagol::AppOptionSet optsOne("app");
        optsOne.addOption("input", "input to this app", &inputOne);
        // should not get an exception
        try {
            optsOne.process(sizeof(cmdLineOne) / sizeof(cmdLineOne[0]), cmdLineOne);
        }
        catch (const yagol::Exception& e) {
            FAIL("unexpected exception");
        }
        TESTEQ(inputOne[3], string("--another-option"), "option-like argument should be consumed into the collection");
    
        yagol::AppOptionSet optsTwo("app");
        optsTwo.addOption("input", "input to this app", &inputTwo);
        // should not get an exception
        try {
            optsTwo.process(sizeof(cmdLineTwo) / sizeof(cmdLineTwo[0]), cmdLineTwo);
        }
        catch (const yagol::Exception& e) {
            FAIL("unexpected exception");
        }
        TESTEQ(inputTwo[3], string("--another-option"), "option-like argument should be consumed into the collection");
    }

    // these should result in exceptions
    {
        yagol::AppOptionSet optsOne(true, false, "app");
        optsOne.addOption("input", "input to this app", &inputOne);
        // should get an exception
        try {
            optsOne.process(sizeof(cmdLineOne) / sizeof(cmdLineOne[0]), cmdLineOne);
            FAIL("exception expected");
        }
        catch (const yagol::Exception& e) {
        }
        TESTEQ(inputOne[3], string("--another-option"), "option-like argument should be consumed into the collection");
    
        yagol::AppOptionSet optsTwo(true, false, "app");
        optsTwo.addOption("input", "input to this app", &inputTwo);
        // should not get an exception
        try {
            optsTwo.process(sizeof(cmdLineTwo) / sizeof(cmdLineTwo[0]), cmdLineTwo);
            FAIL("exception expected");
        }
        catch (const yagol::Exception& e) {
        }
        TESTEQ(inputTwo[3], string("--another-option"), "option-like argument should be consumed into the collection");
    }
}


/**
   The multiple-value option is predominantly useful for wildcards. Compare the
   following two examples:
   
   % myprog --headers *.h --end-of-headers --postscript --sources *.cpp
   
   % myprog --headers Alpha.h --headers Bravo.h --headers Charlie.h \
            --headers Delta.h --headers Echo.h (etc.) \
            --postscript --sources *.cpp
    
   The default beginning of a list of multiple values can be defined by either
   "--tag" or "--start-of-tag", so that these are equivalent:
   
   --input One.h Two.cpp Three.txt --end-of-input
   
   --start-of-input One.h Two.cpp Three.txt --end-of-input
**/

void FeatureTest::testMultiNestedOptionLike() 
{
    vector<string> inputOne;
    char* cmdLineOne[] = { "app", "--input", "One.h", "Two.cpp", "Three.txt", "--end-of-input" };

    vector<string> inputTwo;
    char* cmdLineTwo[] = { "app", "--start-of-input", "One.h", "Two.cpp", "Three.txt", "--end-of-input" };

    yagol::AppOptionSet optsOne("app");
    optsOne.addOption("input", "input to this app", &inputOne);
    // not handling exceptions
    optsOne.process(sizeof(cmdLineOne) / sizeof(cmdLineOne[0]), cmdLineOne);
    
    yagol::AppOptionSet optsTwo("app");
    optsTwo.addOption("input", "input to this app", &inputTwo);
    // not handling exceptions
    optsTwo.process(sizeof(cmdLineTwo) / sizeof(cmdLineTwo[0]), cmdLineTwo);

    TESTEQ(inputTwo.size(), 3UL, "size of input two");
    TESTEQ(inputOne[1], string("Two.cpp"), "value of input one at index 1");
    TESTEQ(inputOne, inputTwo, "equality of vectors");
}
   

/**
   At present, multiple-value options are tied only to STL vectors. Additional
   collection types are expected to be added in forthcoming releases.
   
   
   Read This before Programming with yagol for the First Time

   The Basics
   
   The basic concept is that the client's variables are "tied" to options, and
   are used as both input and output: input for displaying default values of
   options, and output for after the option is processed.
     
   Example code for this:
   
   bool writeOutput(false);
   options.addOption("write", "Whether to write the output", &writeOutput);
   
   This is essentially saying, "for the option 'write', use the variable
   'writeOutput', which has the default value of 'false'". The description of
   this option is "Whether to write the output". Thus, the usage statement for
   this option would be:
     
   --write     Whether to write the output (default = false)
**/

void FeatureTest::testDescription()
{
    yagol::AppOptionSet options("app");
    bool writeOutput(false);
    options.addOption("write", "Whether to write the output", &writeOutput);

    vector<yagol::Option*> opts = options.findByName("write");
    TESTEQ(opts.size(), 1UL, "number of options matching name");

    yagol::Option* opt = opts[0];
    TESTNE(opt, (yagol::Option*)NULL, "existence of option");

    string name("write");
    string shortDesc("Whether to write the output");
    string fullDesc("Whether to write the output (default = false)");
    string n = opt->name();
    string d = opt->description();

#ifdef HAVE_SSTREAM
    ostringstream os;
#else
    ostrstream os;
#endif
    opt->printDescription(os);
    string st = os.str();
    
    TESTEQ(n, name, "option name");
    TESTEQ(d, shortDesc, "option description (short)");
    TESTEQ(st, fullDesc, "option description (full)");
}

/**
   If the option is encountered, in either the list of command line arguments
   or in a name/value map (presumably from a configuration file), then the
   variable "tied" to the option will be appropriately set. For example, the
   command line:
   
   % myprog --write
   
   Would result in the variable writeOutput being set to true.

   How to Do It in Four Easy Steps
   
   The steps involved in usage of yagol are:
   
   1) Create the yagolOptionList or yagolAppOptionSet object
   2) Add the tied variables/options
   3) Add the names of the configuration files, if any
   4) Invoke the process method with the command line arguments, and handle
   exceptions thrown

   Here's an example:
**/

static void showOptions(const string& user, int size, bool readOnly, 
                        const vector<int>& dimensions,
                        const vector<char*>& unused)
{
    cout << "options:" << endl;
    cout << "user: '"     << user      << "'" << endl;
    cout << "size: "      << size             << endl;
    cout << "readOnly: " << readOnly        << endl;
    cout << "dimensions: ";
    for (vector<int>::const_iterator it = dimensions.begin(); it != dimensions.end(); ++it) {
        cout << *it << " ";
    }
    cout << endl;
    cout << "unused arguments: ";
    for (vector<char*>::const_iterator it = unused.begin(); it != unused.end(); ++it) {
        cout << *it << " ";
    }
    cout << endl;
}

void FeatureTest::testHowTo()
{
    string user("jrandom");
    int size(17);
    bool readOnly(false);
    vector<int> dimensions;
    dimensions.push_back(0);
    dimensions.push_back(10);
    dimensions.push_back(0);
    dimensions.push_back(20);
    
    yagol::AppOptionSet options(true, true, "myapp", "My application that does something cool.");

    options.addOption("user",                 "The user of the software",       &user);
    options.addOption("size",                 "The size of the something",      &size);
    options.addOption("readOnly",             "Whether to read, and not write", &readOnly);
    options.addOption("something.dimensions", "The geometrical dimensions",     &dimensions);

    // unused arguments
    vector<char*> unused;

    cout << "before processing the argument list (default values)..." << endl;
    showOptions(user, size, readOnly, dimensions, unused);

    // process the command line

    char* argv[] = { "app", "--user", "fred", "--size", "15", "--noread", "--dimensions", "7", "82", "10", "101" };
    int argc = sizeof(argv) / sizeof(argv[0]);

    try {
        options.process(argc, argv);
    }
    catch (const yagol::Exception& e) {
        // Something bad happened. Dump the usage statement:
        options.writeUsage(cout);
    }

    unused = options.getUnprocessedArgs();

    cout << "after processing the options:" << endl;
    showOptions(user, size, readOnly, dimensions, unused);

    // continue on our merry way...
}


/**
   Configuration Files
   
   Configuration files are handled similarly to options. A sequence of files
   provided by the programmer is used, if the file exists. The order is such
   that the value as set in later directories will override any previously set
   values. Due to my Linux/Unix bias, these are called RC files in the code, but
   Windows programmers, have no fear: they are the same as .ini files.
   
   For example, the developer could use the following three files, in the given
   order of precedence:

   yagol::AppOptionSet opts("app");
   opts.addConfigFile("/etc/myapp.conf");
   opts.addConfigFile("~/.myapprc");
   opts.addConfigFile("./myapp.ini");
   
   Thus, any value set in ~/.myapprc will supercede those in /etc/myapp.conf,
   and likewise for ./myapp.ini superceding any previously set values. RC files
   are processed before the command line arguments, so any command line argument
   will supercede those in a configuration file.
**/

static void createConfigFile(yagol::AppOptionSet* const opts, const string& fname, const string& line)
{
    opts->addConfigFile(fname);
    string cfgFile = opts->expandFileName(fname);
    ofstream out(cfgFile.c_str());
    out << line << endl;
}

static void removeConfigFile(const yagol::AppOptionSet& opts, const string& fname)
{
    string cfgFile = opts.expandFileName(fname);
    unlink(cfgFile.c_str());
}

void FeatureTest::testConfigFileSequence()
{
    yagol::AppOptionSet opts("app");

    string conf("/tmp/myapp.conf");
    string rc("~/.myapprc");
    string ini("./myapp.ini");
    
    createConfigFile(&opts, conf, "    name=John");
    createConfigFile(&opts, rc,   "    name=Paul");
    createConfigFile(&opts, ini,  "    name=George");

    string name("Ringo");
    opts.addOption("name", "my favorite Beatle", &name);

    opts.process(0, NULL);

    TESTEQ("George", name, "last name set in the configuration files");

    // remove all those files
    removeConfigFile(opts, conf);
    removeConfigFile(opts, rc);
    removeConfigFile(opts, ini);
}

/**
   At run time, tildes ("~") are replaced by the value of the $HOME environment
   variable. I should probably also look for HOMEDRIVE and HOMEPATH, but I
   haven't implemented that yet (unless I already have, and just forgot to
   update this comment).

   RC files are simple name/value pairs separated by the equals character ('='),
   where the values may continue to the following line by using a backslash as
   the last character on the line. Comments are preceded by the pound character
   ('#') and run to the end of the line. The whitespace before and after a names
   is removed, and the leading whitespace of values is removed.
   
   For example all of the following result in the association of the value "my dog
   has fleas" with the name "statement":
   
	statement=my dog has fleas
	    statement = my dog has fleas
	statement    = \
	    my \
dog \
has fleas

**/

void FeatureTest::testConfigLine(const string& fname, const string& line)
{
    ofstream out(fname.c_str());
    out << line << endl;
    RCFile rcfile(fname);
    string value = rcfile.get("statement");
    TESTEQ("my dog has fleas", value, "value of normal configuration");
}

void FeatureTest::testConfigFileValue()
{
    testConfigLine("/tmp/valueTestOne.rc",  "	statement=my dog has fleas");
    testConfigLine("/tmp/valueTestTwo.rc",  "	    statement = my dog has fleas");
    string line = "	statement    = \\\n";
    line += "	    my \\\n";
    line += "dog \\\n";
    line += "has fleas";
    testConfigLine("/tmp/valueTestThree.rc", line);
}

/**
   Exploiting the Environment

   Environment variables can also be added to be checked for options.
   Environment variables have greater precedence than RC files, and lower
   precedence than user-specified options.

   For example:
   
   yagol::AppOptionSet opts("myapp");
   opts.addEnvironmentVariable("MYAPPOPTS");

   The value of the given environment variable is split on whitespace and
   processed the same as a set of command-line arguments. For example:
   
   % export MYAPPOPTS="--height 14 --depth 21 --width 7"
   % myapp --depth 105

   will result in the following values being set:
   height = 14
   depth = 105 (overridden on the command line)
   width = 7
**/

void FeatureTest::testEnvironmentVariable()
{
    yagol::AppOptionSet opts("myapp");
    opts.addEnvironmentVariable("MYAPPOPTS");
    int height = 317;
    int depth = 812;
    int width = 215;
    opts.addOption("height", "height of the thing", &height);
    opts.addOption("depth",  "depth of the thing", &depth);
    opts.addOption("width",  "width of the thing", &width);

    setenv("MYAPPOPTS", "--height 14 --depth 21 --width 7", 1);
    
    char* argv[] = { "myapp", "--depth", "105" };
    int argc = sizeof(argv) / sizeof(argv[0]);

    opts.process(argc, argv);

    TESTEQ(height,  14, "value of height, set from the environment variable");
    TESTEQ(depth,  105, "value of depth, set from the command line");
    TESTEQ(width,    7, "value of width, set from the environment variable");
}

/**
   Validation of Input, or How to Use Callbacks for Fun and Profit

   The value of the option can be optionally checked, via callbacks tied to the
   option. For example, an integer value can be bound to a certain range. Also,
   validation can be used with multiple options, so that, for example, certain
   options or specific values cannot be used together.
   
   An example of range validation:

   yagol::AppOptionSet opts("myapp", "my application");
   int size = 100;            // default value
   opts.addOptionRanged("size", "the size of the thing", 10, 500, &size);
**/

void FeatureTest::testRangedOption()
{
    yagol::AppOptionSet opts(true, false, "myapp", "my application");
    // no help on error ........../////
    int size = 100;            // default value

    // range is inclusive
    opts.addOptionRanged("size", "the size of the thing", 10, 500, &size);

    {
        // within range
        char* argv[] = { "myapp", "--size", "105" };
        int argc = sizeof(argv) / sizeof(argv[0]);

        try {
            opts.process(argc, argv);
        }
        catch (const yagol::Exception& e) {
            FAIL("unexpected exception for valid input in range");
        }
    }

    {
        // edge case, bottom of range
        char* argv[] = { "myapp", "--size", "9" };
        int argc = sizeof(argv) / sizeof(argv[0]);
        
        try {
            opts.process(argc, argv);
            FAIL("exception expected for input below range");
        }
        catch (const yagol::BelowRangeException<int>& e) {
            // that's good
        }
        catch (const yagol::Exception& e) {
            // that's bad
            FAIL("unexpected exception for input below range");
        }
    }

    {
        // edge case, top of range
        char* argv[] = { "myapp", "--size", "501" };
        int argc = sizeof(argv) / sizeof(argv[0]);
        
        try {
            opts.process(argc, argv);
            FAIL("exception expected for input above range");
        }
        catch (const yagol::AboveRangeException<int>& e) {
            // that's good
        }
        catch (const yagol::Exception& e) {
            // that's bad
            FAIL("unexpected exception for input above range");
        }
    }
    
}


/**
   An example of an option more specifically validated:
   
	class SizeValidator : public yagol::ArgCallback< int > {
            // extraneous code omitted ...

	    virtual string description() const {
	        return "size should be even";
	    }

            virtual bool onProcessed(const int& sz) {
                if ((sz % 2) != 0) {
                    throw yagol::Exception(description());
                }
                return true;
            }
	};

	yagol::AppOptionSet opts("myapp", "my application");
	int size = 100;            // default value
        SizeValidator* sv = new SizeValidator(&size);
        opts.addOptionCb("size", "the size", sv);
**/

class SizeValidator : public yagol::ArgCallback< int > {
public:
    SizeValidator(int* const i) : yagol::ArgCallback<int>(i) {}
    virtual ~SizeValidator() {}

    virtual string description() const {
        return "size should be even";
    }
    
    virtual bool onProcessed(const int& sz) {
        if ((sz % 2) != 0) {
            throw yagol::Exception(description());
        }
        return true;
    }
};


void FeatureTest::testSizeValidation()
{
    yagol::AppOptionSet opts(~yagol::AppOptionSet::HELP_ON_ERROR, string("myapp"), "my application");
    int size = 100;            // default value
    SizeValidator* sv = new SizeValidator(&size);
    opts.addOptionCb("size", "the size", sv);

    {
        // good size
        char* argv[] = { "myapp", "--size", "104" };
        int argc = sizeof(argv) / sizeof(argv[0]);

        try {
            opts.process(argc, argv);
        }
        catch (const yagol::Exception& e) {
            FAIL("unexpected exception for valid input");
        }
    }

    {
        // bad size
        char* argv[] = { "myapp", "--size", "9" };
        int argc = sizeof(argv) / sizeof(argv[0]);
        
        try {
            opts.process(argc, argv);
            FAIL("exception expected for bad input");
        }
        catch (const yagol::Exception& e) {
            // that's good
        }
    }
}


/**
   Note that the validator will be called both when the option is read (each
   time, if in multiple locations), and after the list of options is processed.
   All options validators will be called after all of the options are read, not
   only the options that were actually found.

   An example of an option validated for a group of values:

   yagol::AppOptionSet opts("myapp", "my application");
   int sizes[] = { 5, 75, 100, 500, 625 };
   int size = 100;            // default value
   opts.addOptionRanged("size", "the size of the thing", &size, sizes, 5);
**/

void FeatureTest::testRangedGroupOption()
{
    yagol::AppOptionSet opts(~yagol::AppOptionSet::HELP_ON_ERROR, string("myapp"), "my application");
    int sizes[] = { 5, 75, 100, 500, 625 };
    int nSizes = sizeof(sizes) / sizeof(sizes[0]);
    int size = 100;             // default value
    opts.addOptionRanged("size", "the size of the thing", sizes, nSizes, &size);

    {
        // good size
        char* argv[] = { "myapp", "--size", "100" };
        int argc = sizeof(argv) / sizeof(argv[0]);

        try {
            opts.process(argc, argv);
        }
        catch (const yagol::Exception& e) {
            FAIL("unexpected exception for valid input");
        }
    }

    {
        // bad size
        char* argv[] = { "myapp", "--size", "9" };
        int argc = sizeof(argv) / sizeof(argv[0]);
        
        try {
            opts.process(argc, argv);
            FAIL("exception expected for bad input");
        }
        catch (const yagol::InvalidValueException<int>& e) {
            // that's good
        }
    }
}


/**
   An example of multiple options:

	class UserReadonlyValidator : public yagol::Processor {
	    virtual string description() {
	        return "root should always run as read-only";
	    }

	    virtual bool onComplete(const yagol::OptionList& opts) {
                yagol::Option* uo = opts.findFirstByName("user");
                yagol::Option* ro = opts.findFirstByName("readonly");
                yagol::OptionGeneric<string>* userOpt = 
                    dynamic_cast< yagol::OptionGeneric<string>* >(uo);
                yagol::OptionGeneric<bool>* rdOpt = 
                    dynamic_cast< yagol::OptionGeneric<bool>* >(ro);
                if (userOpt && rdOpt) {
                    string userName = userOpt->value();
                    bool isReadOnly = rdOpt->value();
                    if (userName == "root" && !isReadOnly) {
                        throw yagol::Exception("root should always run as read-only");
                        // throw yagol::InvalidValueCombination("root should always run as read-only");
                    }
                }
	    }
	};

	yagol::AppOptionSet opts("myapp", "my application");
	string user;
	bool readonly = true;
	opts.addOption("user", "the user", &user);
	opts.addOption("readonly", "whether to run in read-only mode", &readonly);
	opts.addValidator(new UserReadOnlyValidator());

   In this case, the validator will be called only after the entire list of
   options is processed.
**/

class UserReadOnlyValidator : public yagol::Processor {
public:
    UserReadOnlyValidator() {}
    virtual ~UserReadOnlyValidator() {}

    virtual string description() const {
        return "size should be even";
    }
    
    virtual void onComplete(const yagol::OptionList& opts) {
        yagol::Option* uo = opts.findFirstByName("user");
        yagol::Option* ro = opts.findFirstByName("readonly");
        yagol::OptionGeneric<string>* userOpt = 
            dynamic_cast< yagol::OptionGeneric<string>* >(uo);
        yagol::OptionGeneric<bool>* rdOpt = 
            dynamic_cast< yagol::OptionGeneric<bool>* >(ro);
        if (userOpt && rdOpt) {
            string userName = userOpt->value();
            bool isReadOnly = rdOpt->value();
            if (userName == "root" && !isReadOnly) {
                throw yagol::Exception("root should always run as read-only");
            }
        }
    }
};

void FeatureTest::testInputValidation()
{
    yagol::AppOptionSet opts(~yagol::AppOptionSet::HELP_ON_ERROR, string("myapp"), "my application");
    string user;
    bool readonly = true;
    opts.addOption("user", "the user", &user);
    opts.addOption("readonly", "whether to run in read-only mode", &readonly);
    opts.addProcessor(new UserReadOnlyValidator());

    {
        // no conflict between user and readonly
        char* argv[] = { "myapp", "--user", "fred", "--noreadonly" };
        int argc = sizeof(argv) / sizeof(argv[0]);

        try {
            opts.process(argc, argv);
        }
        catch (const yagol::Exception& e) {
            FAIL("unexpected exception for valid input");
        }
    }

    {
        // root running as read only (which is OK)
        char* argv[] = { "myapp", "--user", "root", "--readonly" };
        int argc = sizeof(argv) / sizeof(argv[0]);

        try {
            opts.process(argc, argv);
        }
        catch (const yagol::Exception& e) {
            FAIL("unexpected exception for valid root/readonly combination");
        }
    }

    {
        // root NOT running as read only
        char* argv[] = { "myapp", "--user", "root", "--noreadonly" };
        int argc = sizeof(argv) / sizeof(argv[0]);

        try {
            opts.process(argc, argv);
            FAIL("exception expected for bad input");
        }
        catch (const yagol::Exception& e) {
            // that's OK
        }
    }
}


/**
   Option Validation for File Permissions
   
   Option validation is provided for file permissions, specifically for file
   existence, and read and write permissions. That is, one may specify an option
   as being tied to a file that must be writable or readable.

   yagol::AppOptionSet opts("myapp", "my application");
   string logFile;
   string dataFile;
   opts.addOption(new OptionOutputFile("log", "the log file", &logFile));
   opts.addOption(new OptionInputFile("data", "the data file", &dataFile));
   
   In this case, the log file must either not exist, or must be writable. The
   data file must exist and be readable.
**/

void FeatureTest::testInputFile() 
{
    yagol::AppOptionSet opts(~yagol::AppOptionSet::HELP_ON_ERROR, string("myapp"), "my application");
    string dataFile("default.dat");
    opts.addOption(new yagol::OptionInputFile("data", "the data file", &dataFile));
                                  
    // test for a file that exists
    {
        char* argv[] = { "myapp", "--data", "Makefile" };
        int argc = sizeof(argv) / sizeof(argv[0]);

        try {
            opts.process(argc, argv);
        }
        catch (const yagol::Exception& e) {
            FAIL("caught exception for valid, existing file: " + e.message());
        }
    }

    // test for a file that does not exist
    {
        char* argv[] = { "myapp", "--data", "nosuchfile.exe" };
        int argc = sizeof(argv) / sizeof(argv[0]);

        try {
            opts.process(argc, argv);
            FAIL("expected exception for missing file");
        }
        catch (const yagol::NoSuchFileException& e) {
            // expected
        }
        catch (const yagol::Exception& e) {
            FAIL("wrong exception thrown--expecting No Such File Exception");
        }
    }

    // test for a file that cannot be read
    {
        // a nonreadable file:
        char* nrname = "/tmp/nonreadable.txt";
        ofstream nr(nrname, ios::out);
        chmod(nrname, 0000);
        char* argv[] = { "myapp", "--data", nrname };
        int argc = sizeof(argv) / sizeof(argv[0]);

        try {
            opts.process(argc, argv);
            FAIL("expected exception for valid, existing unreadable file");
        }
        catch (const yagol::FileNotReadableException& e) {
            // expected
        }
        catch (const yagol::Exception& e) {
            cout << "got exception: " << e << endl;
            FAIL("wrong exception thrown--expecting File Not Readable Exception");
        }
        unlink(nrname);
    }

    // test for a non-file (e.g., a directory)
    {
        char* argv[] = { "myapp", "--data", "/tmp" };
        int argc = sizeof(argv) / sizeof(argv[0]);

        try {
            opts.process(argc, argv);
            FAIL("expected exception for directory");
        }
        catch (const yagol::NotAFileException& e) {
            // expected
        }
        catch (const yagol::Exception& e) {
            FAIL("wrong exception thrown--expecting Not a File Exception");
        }
    }
}


void FeatureTest::testOutputFile() 
{
    yagol::AppOptionSet opts(~yagol::AppOptionSet::HELP_ON_ERROR, string("myapp"), "my application");
    string logFile("default.log");
    opts.addOption(new yagol::OptionOutputFile("log", "the log file", &logFile));
                                  
    // test for a file that exists, and is writeable
    {
        char* name = "/tmp/writable.txt";
        ofstream nw(name, ios::out);
        char* argv[] = { "myapp", "--log", name };
        int argc = sizeof(argv) / sizeof(argv[0]);

        try {
            opts.process(argc, argv);
        }
        catch (const yagol::Exception& e) {
            FAIL("caught exception for valid, writable file: " + e.message());
        }
        unlink(name);
    }

    // test for a file that does not exist
    {
        char* name = "/tmp/nosuchfile.exe";
        char* argv[] = { "myapp", "--log", name };
        int argc = sizeof(argv) / sizeof(argv[0]);

        try {
            opts.process(argc, argv);
        }
        catch (const yagol::Exception& e) {
            FAIL("exception thrown for nonfile: " + e.message());
        }
    }

    // test for a file that cannot be written to
    {
        char* name = "/tmp/nonwritable.txt";
        ofstream  nw(name, ios::out);
        chmod(name, 0444);
        char* argv[] = { "myapp", "--log", name };
        int argc = sizeof(argv) / sizeof(argv[0]);

        try {
            opts.process(argc, argv);
            FAIL("expected exception for nonwritable file");
        }
        catch (const yagol::FileNotWritableException& e) {
            // expected
        }
        catch (const yagol::Exception& e) {
            cout << "got exception: " << e << endl;
            FAIL("wrong exception thrown--expecting File Not Writable Exception");
        }
        unlink(name);
    }

    // test for a non-file (e.g., a directory)
    {
        char* argv[] = { "myapp", "--log", "/tmp" };
        int argc = sizeof(argv) / sizeof(argv[0]);

        try {
            opts.process(argc, argv);
            FAIL("expected exception for directory");
        }
        catch (const yagol::NotAFileException& e) {
            // expected
        }
        catch (const yagol::Exception& e) {
            FAIL("wrong exception thrown--expecting Not a File Exception");
        }
    }
}


/**
   Namespaces

   Options can be defined within namespaces. This functionality can be used with
   RC files to create virtual "packages" of options, that is, options delimited
   by dots.

   For example:
   
   yagol::AppOptionSet opts("myapp");
   string user;
   string password;
   bool enable = true;
   opts.addOption("security.user", "the user", &user);
   opts.addOption("security.password", "the password", &password);
   opts.addOption("security.enable", "whether to log in the user and password", &enable);
   
   bool color = true;
   string resolution("1024x768");
   opts.addOption("display.color", "whether to show colors", &color);
   opts.addOption("display.resolution", "the resolution to display", &resolution);
   
   // rc file:
   security.user = fred
   security.password = Joshua
   security.enable = true
   display.color = false
   display.resolution = 80x24
   
   However, on the command line the namespace usage can be optionally not used,
   e.g.:
   
   % myapp --user fred --password Joshua --noenable --color --resolution 100x40
**/

void FeatureTest::testNamespaces() 
{
    yagol::AppOptionSet opts(~yagol::AppOptionSet::HELP_ON_ERROR, string("myapp"), "my application");

    string user;
    string password;
    bool enable = true;
    opts.addOption("security.user", "the user", &user);
    opts.addOption("security.password", "the password", &password);
    opts.addOption("security.enable", "whether to log in the user and password", &enable);
   
    bool color = true;
    string resolution("1024x768");
    opts.addOption("display.color", "whether to show colors", &color);
    opts.addOption("display.resolution", "the resolution to display", &resolution);

    string rcFile("/tmp/myapprc");
   
    ofstream out(rcFile.c_str());

    out << "security.user = fred" << endl;
    out << "security.password = Joshua" << endl;
    out << "security.enable = true" << endl;
    out << "display.color = false" << endl;
    out << "display.resolution = 80x24" << endl;
   
    opts.addConfigFile(rcFile);

    {
        char* argv[] = { 
            "myapp", 
            "--noenable",
            "--color",
            "--resolution", "25x40"
        };
        int argc = sizeof(argv) / sizeof(argv[0]);

        try {
            opts.process(argc, argv);
            TESTEQ(user, "fred", "value of user");
            TESTEQ(password, "Joshua", "value of password");
            TESTEQ(enable, false, "value of enable");
            TESTEQ(color, true, "value of color");
            TESTEQ(resolution, "25x40", "value of resolution");
        }
        catch (const yagol::Exception& e) {
            FAIL("caught exception: " + e.message());
        }
    }
    
    {
        char* argv[] = { 
            "myapp", 
            "--security.user", "barney",
            "--password", "Betty",
            "--noenable",
            "--display.resolution", "100x40"
        };
        int argc = sizeof(argv) / sizeof(argv[0]);

        try {
            opts.process(argc, argv);
            TESTEQ(user, "barney", "value of user");
            TESTEQ(password, "Betty", "value of password");
            TESTEQ(enable, false, "value of enable");
            TESTEQ(color, false, "value of color");
            TESTEQ(resolution, "100x40", "value of resolution");
        }
        catch (const yagol::Exception& e) {
            FAIL("caught exception: " + e.message());
        }
    }

}
   
   
/**
   Shortening of Option Names: Help for the Indolent
   
   Options can be used by only the "unique" part of their name, that is, only by
   enough characters that distinguish them from another option.
   
   For example:
   
   yagol::AppOptionSet opts("myapp");
   int size = 4;
   int length = 10;
   string color = "blue";
   bool console = false;
   
   opts.addOption("size", "the size", &size);
   opts.addOption("length", "the length", &length);
   opts.addOption("color", "the color", &color);
   opts.addOption("console", "whether to write to the console", &console);
   
   The following is valid:
   
   % myapp --s 200 --l 75 --col red --con
   
   But the following will result in an error:
   
   % myapp --s 200 --l 75 --co red --co
   
   Error: option "co" ambiguous: could match either "color" or "console".
**/

void FeatureTest::testShortNames() 
{
    yagol::AppOptionSet opts(~yagol::AppOptionSet::HELP_ON_ERROR, string("myapp"), "my application");
    
    int size = 4;
    int length = 10;
    string color = "blue";
    bool console = false;
    
    opts.addOption("size", "the size", &size);
    opts.addOption("length", "the length", &length);
    opts.addOption("color", "the color", &color);
    opts.addOption("console", "whether to write to the console", &console);
   
    // valid: myapp --s 200 --l 75 --col red --con
    {
        char* argv[] = { 
            "myapp",
            "--s", "200", 
            "--l", "75",
            "--col", "red",
            "--con"
        };
        int argc = sizeof(argv) / sizeof(argv[0]);

        try {
            opts.process(argc, argv);
            TESTEQ(size, 200, "value of size");
            TESTEQ(length, 75, "value of length");
            TESTEQ(color, "red", "value of color");
            TESTEQ(console, true, "value of console");
        }
        catch (const yagol::Exception& e) {
            FAIL("caught exception: " + e.message());
        }
    }

    // valid: myapp --s 200 --l 75 --co red --co
    {
        char* argv[] = { 
            "myapp",
            "--s", "200", 
            "--l", "75",
            "--co", "red",
            "--co"
        };
        int argc = sizeof(argv) / sizeof(argv[0]);

        try {
            opts.process(argc, argv);
            FAIL("expected 'ambiguous argument' exception");
        }
        catch (const yagol::Exception& e) {
            // that's good. I meant to do that.
        }
    }
}   


/**
   Option Inheritance
   
   Options can be chained to inherit other values if they are not set. In this
   way any number of options can be set "at once". For example:
   
   yagol::AppOptionSet opts("report");
   bool runTimeErrors = true;
   bool compileTimeErrors = false;
   bool docErrors = false;
   bool spellingErrors = false;
   bool allErrors = true;

   opts.addOption("runtime", "whether to report run time errors", &runTimeErrors);
   opts.addOption("compile", "whether to report compile errors", &compileErrors);
   opts.addOption("documentation", "whether to report documentation errors", &docErrors);
   opts.addOption("spelling", "whether to report spelling errors", &spellingErrors);
   opts.addOption("all", "whether to report all errors", &allErrors);
   opts.addLink("runtime", "all");
   opts.addLink("compile", "all");
   // no link from documentation to all
   opts.addLink("spelling", "all");
   
   Given the arguments:
   
   % myapp --all
   
   The values of runTimeErrors, compileTimeErrors and spellingErrors will also
   be set to "true".
   
   This feature carries over into the RC file:

   all = true
   
   Note however that if the options is explicitly set, the "linked" value will
   not be used. For example, in both of the following cases, the value for
   "spellingErrors" will be "false":

   % myapp --all --nospelling
   
   all = true
   spelling = false
**/

void FeatureTest::testLinks() 
{
    yagol::AppOptionSet opts(~yagol::AppOptionSet::HELP_ON_ERROR, string("myapp"), "my application");

    bool runTimeErrors     = true;
    bool compileTimeErrors = false;
    bool docErrors         = false;
    bool spellingErrors    = false;
    bool allErrors         = true;

    opts.addOption("runtime",       "whether to report run time errors",      &runTimeErrors);
    opts.addOption("compile",       "whether to report compile errors",       &compileTimeErrors);
    opts.addOption("documentation", "whether to report documentation errors", &docErrors);
    opts.addOption("spelling",      "whether to report spelling errors",      &spellingErrors);
    opts.addOption("all",           "whether to report all errors",           &allErrors);

    opts.addLink("runtime",  "all");
    opts.addLink("compile",  "all");
    // no link from documentation to all
    opts.addLink("spelling", "all");
   
    // myapp --all
    {
        char* argv[] = { "myapp", "--all" };
        int argc = sizeof(argv) / sizeof(argv[0]);
        try {
            opts.process(argc, argv);
            TESTEQ(runTimeErrors,     true, "value of runTimeErrors");
            TESTEQ(compileTimeErrors, true, "value of compileTimeErrors");
            TESTEQ(spellingErrors,    true, "value of spellingErrors");
        }
        catch (const yagol::Exception& e) {
            FAIL("caught exception: " + e.message());
        }
    }

    // RC file with "all = true"
    {
        char* rcFile = "/tmp/myapprc";
        ofstream out(rcFile);
        out << "all = true" << endl;
        opts.addConfigFile(rcFile);

        char* argv[] = { "myapp" };
        int argc = sizeof(argv) / sizeof(argv[0]);

        try {
            opts.process(argc, argv);
            TESTEQ(runTimeErrors,     true, "value of runTimeErrors");
            TESTEQ(compileTimeErrors, true, "value of compileTimeErrors");
            TESTEQ(spellingErrors,    true, "value of spellingErrors");
        }
        catch (const yagol::Exception& e) {
            FAIL("caught exception: " + e.message());
        }
    }

    // myapp --all --nospelling
    {
        char* argv[] = { "myapp", "--nospelling" };
        int   argc   = sizeof(argv) / sizeof(argv[0]);

        try {
            opts.process(argc, argv);
            TESTEQ(runTimeErrors, true, "value of runTimeErrors");
            TESTEQ(compileTimeErrors, true, "value of compileTimeErrors");
            TESTEQ(spellingErrors, false, "value of spellingErrors");
        }
        catch (const yagol::Exception& e) {
            FAIL("caught exception: " + e.message());
        }
    }

    // RC file with "all = true; spelling = false"
    {
        char* rcFile = "/tmp/myapprc";
        ofstream out(rcFile);
        out << "all = true" << endl;
        out << "spelling = false" << endl;
        opts.addConfigFile(rcFile);

        char* argv[] = { "myapp" };
        int   argc   = sizeof(argv) / sizeof(argv[0]);

        try {
            opts.process(argc, argv);
            TESTEQ(runTimeErrors,     true,  "value of runTimeErrors");
            TESTEQ(compileTimeErrors, true,  "value of compileTimeErrors");
            TESTEQ(spellingErrors,    false, "value of spellingErrors");
        }
        catch (const yagol::Exception& e) {
            FAIL("caught exception: " + e.message());
        }
    }
}   


/**
   Based on my own experiences with using this library, there are times when an
   option is appropriate for a configuration file, but not the command line, so
   there are special options that provide the ability to be "hidden" from the
   command line, yet are read from configuration files, if any. Like regular
   options, these config-only options also have a default value and a
   description.
**/

void FeatureTest::testConfigOnlyOptions()
{
    yagol::AppOptionSet opts("myapp");

    string rc("/tmp/.myapprc");
    
    createConfigFile(&opts, rc, "    multi.part.option.name=Fred");

    string name("Barney");
    yagol::Option* nmopt = opts.addOption("multi.part.option.name", "the star of the 'Stones", &name);
    
    string pet("Dino");
    opts.addOption("pet", "his pet", &pet);
    nmopt->setConfigOnly(true);

    opts.process(0, NULL);

    // Actually, we should grab that in a file and check it:
    opts.writeUsage(cout);

    // ditto.
    opts.writeConfigUsage(cout);

    TESTEQ("Fred", name, "name set in the configuration file");

    removeConfigFile(opts, rc);
}


/**
   Future Features, or How I Intend to Waste Even More of My Copious Free Time:

   The following features may be added in the future, pending feedback from
   users (and my own experience) that leads me to believe that they will be
   useful.


   Enumerations
   
   I am considering adding enumerations, so that one could write:
   
   enum ColorType { BLUE, RED, GREEN };
   ColorType types[] = { BLUE, RED, GREEN };
   char* colors[] = { "blue", "red", "green" };
   
   yagol::AppOptionSet display("display");
   ColorType color = "blue";
   report.addOption("color", "the color", &color, types, colors, 3);

   Users would of course see the strings, but the values would be converted to
   enumerations for internal usage.


   Aliases
   
   Multiple option names (tags) can refer to the same option. For example:
   
   yagol::AppOptionSet report("report");
   int duration = 4;
   report.addOption("duration", "the duration", &size);
   report.addAlias("length", "duration");
   report.addAlias("time", "duration");
   
   Thus, all of the following have the same effect:
   
   % myapp --duration 10
   % myapp --length 10
   % myapp --time 10
   
**/

int main(int argc, char** argv)
{
    FeatureTest t;
    t.run();
    cout << "features test results: " << t << endl;
    return t.nerrors();
}
