/**
 * An extension of Example.t.cpp, using callbacks instead of simple tied
 * objects.
 */

#ifndef Yagol_h
#include "Yagol.h"
#endif

#ifndef std_vector
#define std_vector
#include <vector>
#endif

using namespace std;

/**
 * Writes all of the options to stdout.
 */
static void showOptions(const string& user, int size, bool readOnly, 
                        const vector<int>& dimensions,
                        const vector<char*>& unused)
{
    cout << "options:" << endl;
    cout << "\tuser: '"     << user      << "'" << endl;
    cout << "\tsize: "      << size      << endl;
    cout << "\tread-only: " << readOnly  << endl;
    cout << "\tdimensions: ";
    for (vector<int>::const_iterator it = dimensions.begin(); it != dimensions.end(); ++it) {
        cout << *it << " ";
    }
    cout << endl;
    cout << "\tunused arguments: ";
    for (vector<char*>::const_iterator it = unused.begin(); it != unused.end(); ++it) {
        cout << *it << " ";
    }
    cout << endl;
}

class UserOptionCallback : public yagol::ArgCallback<string>
{
public:
    UserOptionCallback(string* user) : yagol::ArgCallback<string>(user) {}

    virtual ~UserOptionCallback() {}

    // can't override with "const string&", since onProcessed expects
    // pass-by-value.
    virtual bool onProcessed(const string& who) {
        if (who == "fred") {
            // I don't like fred; he can't be the user
            throw yagol::Exception("'fred' is an invalid user");
            // return value not needed, since we're throwing an exception:
        }
        else {
            return true;
        }
    }
};

class SizeOptionCallback : public yagol::ArgCallback<int>
{
public:
    SizeOptionCallback(int* size) : yagol::ArgCallback<int>(size) {}

    virtual ~SizeOptionCallback() {}

    virtual bool onProcessed(const int& sz) {
        if (sz <= 0) {
            // here's a warning:
            cerr << "WARNING: size should be a positive number" << endl;
            // don't return true; continue processing the arguments
        }
        return true;
    }

    // not doing anything special with onComplete, so don't override it

};

class ReadOnlyOptionCallback : public yagol::ArgCallback<bool>
{
public:
    ReadOnlyOptionCallback(bool* readonly, string* user) :
            yagol::ArgCallback<bool>(readonly), readonly_(readonly), user_(user)
    {}

    virtual ~ReadOnlyOptionCallback() {}

    // nothing special with onProcessed...

    // validate that 'root' is running as read-only:
    virtual void onComplete(const yagol::OptionList& opts) {
        if (*user_ == "root" && !readonly_) {
            string errmsg = "root should always run as read-only";
            throw yagol::Exception(errmsg);
        }
    }

private:
    bool readonly_;
    string* user_;
};

class DimensionsOptionCallback : public yagol::ArgCallback<vector<int> >
{
public:
    DimensionsOptionCallback(vector<int>* dims) : 
            yagol::ArgCallback<vector<int> >(dims) {}
    
    virtual ~DimensionsOptionCallback() {}

    // check the number of dimensions once processed
    virtual bool onProcessed(const vector<int>& dims) {
        if (dims.size() < 2 || dims.size() > 4) {
            string errmsg = "there must be between two and four dimensions";
            throw yagol::Exception(errmsg);
        }
        else if (find(dims.begin(), dims.end(), 4) == dims.end()) {
            cout << "WARNING: one of the dimensions should be 4" << endl;
            // but continue processing anyway, and don't treat it as an error
        }
        return true;
    }

};

// to check the user:
//     % ExampleCB.t --user fred

// to check the read-only permissions against the user:
//     % ExampleCB.t --user root --noreadOnly

// to check the number of dimensions:
//     % ExampleCB.t --dimensions 1 2 3 4 5
//     % ExampleCB.t --dimensions 1

// to validate the dimensions
//     % ExampleCB.t --dimensions 1 2 3 5

int main(int argc, char** argv)
{
    // list o' options
    yagol::AppOptionSet options(true, true, "myapp", "My application that does something cool.");

    string user("jrandom");
    int size(17);
    bool readOnly(false);
    vector<int> dimensions;
    dimensions.push_back(0);
    dimensions.push_back(10);
    dimensions.push_back(4);
    dimensions.push_back(20);

    UserOptionCallback usercb(&user);
    SizeOptionCallback sizecb(&size);
    ReadOnlyOptionCallback readOnlycb(&readOnly, &user);
    DimensionsOptionCallback dimcb(&dimensions);
    
    options.addOptionCb("user",       "The user of the software",       &usercb);
    options.addOptionCb("size",       "The size of the something",      &sizecb);
    options.addOptionCb("read-only",  "Whether to read, and not write", &readOnlycb);
    options.addOptionCb("dimensions", "The geometrical dimensions",     &dimcb);

    options.addEnvironmentVariable("MYAPPOPTS");

    // unused arguments
    vector<char*> unused;

    cout << "before processing the argument list (default values)..." << endl;
    showOptions(user, size, readOnly, dimensions, unused);

    // process the command line
    try {
        options.process(argc, argv);
    }
    catch (const yagol::Exception& e) {
        // Something bad happened. Dump the usage statement:
        options.writeUsage(cout);
    }

    unused = options.getUnprocessedArgs();

    if (options.hadError()) {
        cout << "there was an error processing the options" << endl;
        return 1;
    }
    else {
        cout << "after processing the options:" << endl;
        showOptions(user, size, readOnly, dimensions, unused);
        // continue on our merry way...
        return 0;
    }
}
