#ifndef Platform_h
#define Platform_h

#ifndef std_string
#define std_string
#include <string>
#endif

#ifndef std_vector
#define std_vector
#include <vector>
#endif

using namespace std;

namespace doctorj 
{
    class FileProcessor 
    {
    public:
        FileProcessor(vector<string>* const matches);

        virtual ~FileProcessor();

        virtual void processDirectory(const string& dirname);

        virtual void handle(const string& fname) = 0;

    protected:
        vector<string>* matches_;
    };


    class FileFilter : public FileProcessor 
    {
    public:
        FileFilter(vector<string>* const matches, const string& ext = string(""));

        virtual ~FileFilter();

        void addExtension(const string& ext);

        virtual void handle(const string& fname);
    
    protected:
        /**
         * Returns whether the extension matches the filter criteria.
         */
        bool matches(const string& fname) const;

        vector<string> extns_;
    };


    class FileFinder : public FileFilter
    {
    public:
        FileFinder(vector<string>* const matches, const string& ext = string(""));

        virtual ~FileFinder();

        virtual void handle(const string& fname);
    };


    /**
     * This class encapsulates common functionality across multiple platforms,
     * e.g., Unix and DOS/Windows. 
     */
    class Platform
    {
    public:
        enum Type { FILE, DIRECTORY, OTHER };
        
        /**
         * Acts like Unix find, taking the extension as a filter. Thus,
         *
         *     find(".", "java", ...)
         * 
         * is the same as
         *
         *     % find . -name '*.java' -type f -print
         */
        static void findFiles(const string& dir, 
                              const string& ext, 
                              vector<string>* const files);
        
        /**
         *  Passes back the files in the given directory.
         */
        static void getFiles(const string& dir, 
                             vector<string>* const files);
        /**
         *  Passes back the files in the given directory that match the
         *  extension.
         */
        static void getFiles(const string& dir, 
                             const string& ext,
                             vector<string>* const files);

        /**
         *  Passes back the files in the given directory that match either
         *  extension.
         */
        static void getFiles(const string& dir, 
                             const string& ext0,
                             const string& ext1,
                             vector<string>* const files);

        /**
         *  Passes back the files and directories in the given directory.
         */
        static void getEntries(const string& dir, 
                               vector<string>* const entries);
        
        static bool isDirectory(const string& name);

        static bool isRegularFile(const string& name);

        static bool isLink(const string& name);

        static bool isReadableFile(const string& name);

        /**
         * Passes back the value for the environment variable. The return value
         * indicates whether the environment variable was found, since it could
         * be set, but to an empty string. If a value string is not passed in,
         * then only the return value is valid. Ugh, this is a lame description.
         */
        static bool getEnv(const string& name, string* const value = NULL);

        /**
         * Returns the current directory.
         */
        static string getCurrentDirectory();

        /**
         * Renames the given file.
         */
        static void renameFile(const string& from, const string& to);

        /**
         * mkdir -p equivalent. Returns whether the path could be created.
         */
        static bool ensureDirectoryExists(const string& path);

    };

}

#endif //! Platform_h
