#ifndef File_h
#include "File.h"
#endif

#ifndef StringEdit_h
#include "StringEdit.h"
#endif

#ifndef UtilDebug_h
#include "UtilDebug.h"
#endif

#ifndef Platform_h
#include "Platform.h"
#endif

#ifndef std_algorithm
#define std_algorithm
#include <algorithm>
#endif

#ifndef std_fstream
#define std_fstream
#include <fstream>
#endif

#ifndef std_stdio_h
#define std_stdio_h
#include <stdio.h>
#endif

#ifndef std_math_h
#define std_math_h
#include <math.h>
#endif

using namespace std;
using namespace doctorj;

char File::CR = '\r';
char File::LF = '\n';

class Output
{
public:
    virtual ~Output();

    virtual void showLineNumber(int lnum) const;

    virtual void show(const string& str, int start, int end, int startLine) const;
};

Output::~Output()
{
}

void Output::showLineNumber(int lnum) const
{
    // right align the line numbers.
    double ll  = log10(double(lnum));
    int    len = 1 + int(floor(ll));
        
    int spaces = 6 - len;
    for (int i = 0; i < spaces; ++i) {
        cout << " ";
    }

    cout << lnum << ". " << flush;
}

void Output::show(const string& str, int start, int end, int startLine) const
{
    //     cout << "show(" << str << ", " << start << ", " <<  end << ", " << startLine << ")" << endl;
//     DEBUG_UTIL(cout << "FILE   show(" << str << ", " << start << ", " <<  end << ", " << startLine << ")" << endl);

//     DEBUG_UTIL(cout << "    indices: ");
//     for (int i = 0; i < str.length(); i += 10) {
//         DEBUG_UTIL(cout << ((i / 10) % 10) << "         ");
//     }
//     DEBUG_UTIL(cout << endl);
//     DEBUG_UTIL(cout << "             ");
//     for (int i = 0; i < str.length(); ++i) {
//         DEBUG_UTIL(cout << i % 10);
//     }
//     DEBUG_UTIL(cout << endl);
//     DEBUG_UTIL(cout << "    string: '");
//     for (int i = 0; i < str.length(); ++i) {
//         if (str[i] == '\n') {
//             DEBUG_UTIL(cout << '|');
//         }
//         else {
//             DEBUG_UTIL(cout << str[i]);
//         }
//     }
//     cout << endl;

    DEBUG_UTIL(cout << "FILE   str[start] = '" << str[start] << "'" << endl);
    
    while (start >= 0 && str[start] != '\n') {
        DEBUG_UTIL(cout << "FILE   start decrementing" << endl);
        --start;
    }

    DEBUG_UTIL(cout << "FILE   str[end] = '" << str[end] << "'" << endl);
    
    while (end < str.length() && str[end] != '\n') {
        //         cout << "advancing end " << end << " because str[" << end << "] = '" << str[end] << "'" << endl;
        DEBUG_UTIL(cout << "FILE   end advancing" << endl);
        ++end;
    }

    // why was I doing this?:
    //--end;

    //     cout << "show: corrected: (start " << start << ", end " <<  end << ", startLine " << startLine << ")" << endl;
    DEBUG_UTIL(cout << "FILE   corrected: (start " << start << ", end " <<  end << ", startLine " << startLine << ")" << endl);

    if (start >= end) {
        DEBUG_UTIL(cout << "FILE   start >= end)" << endl);
        
        showLineNumber(startLine);
        cout << str << endl;
    }
    else {
        DEBUG_UTIL(cout << "FILE   start < end)" << endl);        

        for (int i = start, line = startLine; i < end; ++i) {
            //cout << "i = " << i << "; end = " << end << endl;
            if (str[i] == '\n' || i == start) {
                if (i != start) {
                    cout << endl;
                }
                showLineNumber(line);
                ++line;
            }
            else {
                cout << str[i];
            }
        }
    }

    cout << endl;
}


File::File(const string& fname) throw(FileException)
        : inputPosition_(0), name_(fname), changed_(false)
{
    // Read the entire file in first, so that the scanner doesn't have to return
    // full lines to it. However, the scanner _does_ need to tell the file when
    // it has progressed to the next line.
    readFile();
    init(text_);
}

File::~File()
{
    delete [] text_;
}

string File::name() const
{
    return name_;
}
        
int File::getInput(char* buf, size_t maxSize)
{
    size_t n = std::min(maxSize, static_cast<size_t>(end_ - inputPosition_));
    if (n > 0) {
        memcpy(buf, inputPosition_, n);
        inputPosition_ += n;
    }
    return n;
}

/**
 * Returns the number of characters in the file.
 */
int File::length() const
{
    // char defaults to unsigned for some architectures/OSes, so this would
    // never satisfy the != EOF (which is -1) test.
    signed char ch;
    
    ifstream f(name_.c_str());
    int count = 0;
    while ((ch = f.get()) != EOF) {
        count++;
        if (ch == CR) {
            // maybe Dos, if it's CRLF
            char nch = f.get();
            if (nch != LF) {
                f.unget();
            }
        }
    }
    return count;
}

void File::readFile()
{
    // We have to to read a character at a time, and twiddle with the end of
    // line sequences.

    ifstream f(name_.c_str());
    if (!f.is_open()) {
        throw FileException();
    }

    // see comment in File::length()
    signed char ch;

    // may need to come up with better string/memory management later.
    int len = length();
    inputPosition_ = text_ = new char[1 + len];
    end_ = text_ + len;
    string str;

    char* pos = text_;
    bool savenext = true;
    vector<char*> positions;

    while ((ch = f.get()) != EOF) {
        if (savenext) {
            positions_.push_back(pos);
        }
        if (ch == LF) {
            // save the position with the next character
            savenext = true;

            // same as *pos = LF:
            *pos = ch;
        }
        else if (ch == CR) {
            // save the position with the next character
            savenext = true;

            // maybe Dos, if it's CRLF
            char nch = f.get();
            if (nch != LF) {
                // No LF following, so it's Mac. Push the character back into
                // the stream for next time.
                f.unget();
            }
            *pos = LF;
        }
        else {
            savenext = false;
            *pos = ch;
        }
        
        ++pos;
    }
    
    if (savenext) {
        positions_.push_back(pos);
    }

    // necessary to do this?
    *pos = '\0';
}

char* File::head() const
{
    return text_;
}

int File::lineOf(char* position) const
{
    // walk backward through the list of positions--this probably isn't the best
    // algorithm, but it works.
    int sz = positions_.size();
    for (int ln = sz - 1; ln >= 0; --ln) {
        char* const pos = positions_[ln];
        if (position >= pos) {
            // add one, because our vector is 0-indexed, and lines are 1-indexed
            return ln + 1;
        }
    }

    cout << "ERROR: lineOf(" << (void*)(position) << "): not found" << endl;
    return -1;
}

int File::columnOf(char* position) const
{
    int ln = lineOf(position);
    if (ln == -1) {
        cout << "ERROR: columnOf(" << (void*)(position) << "): not found" << endl;
        return -1;
    }
    else {
        // account for 0 index vs. 1 index:
        return position - positions_[ln - 1];
    }
    
}

string File::fullLine(char* position) const
{
    int ln = lineOf(position);
    return fullLine(ln);
}

char* File::end() const
{
    return end_;
}

char* File::startPosition(unsigned int line) const
{
    if (line > positions_.size()) {
        cout << "ERROR: startPosition(" << line << "): out of range of " << positions_.size() << endl;
        return NULL;
    }
    else {
        return positions_[line - 1];
    }
}

string File::fullLine(int line) const
{
    if (line < 0) {
        cout << "ERROR: fullLine(" << line << "): no such line" << endl;
        return "";
    }
    else {
        int   idx   = line - 1;
        char* start = positions_[idx];
        char* end   = ((unsigned int)idx == positions_.size() - 1) ? end_ : positions_[idx + 1] - 1;
        string str(start, end - start);
        return str;
    }
}

void File::addEdit(StringEdit* const edit)
{
    changed_ = true;
    StringEditor::add(edit);
    execute();
}
        
void File::write(ostream& os)
{
    execute();
    os << str();
}

void File::write()
{
    ofstream of(name_.c_str());
    write(of);
    of.close();
}

bool File::changed() const
{
    return changed_;
}

void File::showString(const string& str, char* const start, char* const end)
{
    int idxstart = index(start);
    int idxend = index(end);
    int startLine = lineOf(start);

    DEBUG_UTIL(cout << "FILE   start = " << (void*)start << endl);
    DEBUG_UTIL(cout << "FILE   index of start = " << idxstart << endl);
    DEBUG_UTIL(cout << "FILE   end = " << (void*)end << endl);
    DEBUG_UTIL(cout << "FILE   index of end = " << idxend << endl);
    DEBUG_UTIL(cout << "FILE   start line = " << startLine << endl);

    Output o;
    o.show(str, idxstart, idxend, startLine);
}

void File::showLine(const string& str, int startLine)
{
    Output o;
    o.showLineNumber(startLine);
    cout << str << endl;
}

void File::showCurrent(char* const start, char* const end)
{
    int idxstart = start - text_;
    int idxend = end - text_;
    int startLine = lineOf(start);

    DEBUG_UTIL(cout << "FILE   start = " << (void*)start << endl);
    DEBUG_UTIL(cout << "FILE   index of start = " << idxstart << endl);
    DEBUG_UTIL(cout << "FILE   end = " << (void*)end << endl);
    DEBUG_UTIL(cout << "FILE   index of end = " << idxend << endl);
    DEBUG_UTIL(cout << "FILE   start line = " << startLine << endl);

    string current = originalStr();
    Output o;
    o.show(current, idxstart, idxend, startLine);
}

void File::showPreview(StringEdit* const edit, char* const start, char* const end)
{
    string prev = preview(*edit);

    int idxstart = index(start);
    int idxend = idxstart + end - start;
    int startLine = lineOf(start);

    DEBUG_UTIL(cout << "FILE   start = " << (void*)start << endl);
    DEBUG_UTIL(cout << "FILE   index of start = " << idxstart << endl);
    DEBUG_UTIL(cout << "FILE   end = " << (void*)end << endl);
    DEBUG_UTIL(cout << "FILE   index of end = " << idxend << endl);
    DEBUG_UTIL(cout << "FILE   start line = " << startLine << endl);

    Output o;
    o.show(prev, idxstart, idxend, startLine);
}

void File::showPreview(StringEditDelete* const del, char* const start, char* const end)
{
    string prev = preview(*del);

    int idxstart = index(start);
    if (*start != '\n' && prev[idxstart] == '\n') {
        --idxstart;
    }
    
    int idxend = idxstart + end - start;
    int startLine = lineOf(start);

    DEBUG_UTIL(cout << "FILE   start = " << (void*)start << endl);
    DEBUG_UTIL(cout << "FILE   index of start = " << idxstart << endl);
    DEBUG_UTIL(cout << "FILE   end = " << (void*)end << endl);
    DEBUG_UTIL(cout << "FILE   index of end = " << idxend << endl);
    DEBUG_UTIL(cout << "FILE   start line = " << startLine << endl);

    Output o;
    o.show(prev, idxstart, idxend, startLine);
}


string FileName::extension(const string& fname)
{
    int pos = fname.rfind(".");
    if (pos > -1) {
        return fname.substr(pos + 1);
    }
    else {
        return "";
    }
}

string FileName::basename(const string& fname)
{
    int pos = fname.rfind(".");
    if (pos > -1) {
        return fname.substr(0, pos);
    }
    else {
        return fname;
    }
}

FileName::FileName(const string& str) : str_(str)
{
}

FileName::~FileName()
{
}

string FileName::extension() const
{
    return extension(str_);
}

string FileName::basename() const
{
    return basename(str_);
}


OutFile::OutFile(const string& name, const string& contents, const string& eoln) :
        name_(name),
     contents_(contents),
     eoln_(eoln)
{
}
        
OutFile::~OutFile()
{
}

void OutFile::write()
{
    DEBUG_UTIL(cout << "OUTFL  ::write()" << endl);
    DEBUG_UTIL(cout << "OUTFL  name_ = " << name_ << endl);
    DEBUG_UTIL(cout << "OUTFL  eoln_ = " << eoln_ << endl);

    int slpos = name_.rfind('/');
    DEBUG_UTIL(cout << "OUTFL  slpos = " << slpos << endl);
    if (slpos != string::npos) {
        string dir = name_.substr(0, slpos);
        DEBUG_UTIL(cout << "OUTFL  dir = '" << dir << "'" << endl);
        Platform::ensureDirectoryExists(dir);
    }
    
    ofstream of(name_.c_str(), ios::out);
    DEBUG_UTIL(cout << "OUTFL  of.is_open ? " << of.is_open() << endl);
    int len = contents_.length();
    for (int i = 0; i < len; ++i) {
        char ch = contents_[i];
        if (ch == '\n') {
            DEBUG_UTIL(cout << "OUTFL  substituting eoln" << endl);
            of << eoln_;
        }
        else {
            of << ch;
        }
    }
    of << flush;
    of.close();

    DEBUG_UTIL(cout << "OUTFL  done writing" << endl);
}
