#ifndef Java_h
#include "Java.h"
#endif

#ifndef StringUtilities_h
#include "StringUtilities.h"
#endif

#ifndef Platform_h
#include "Platform.h"
#endif

#ifndef File_h
#include "File.h"
#endif

#ifndef FileByteStream_h
#include "FileByteStream.h"
#endif

#ifndef Bc_h
#include "Bc.h"
#endif

#ifndef ByteArray_h
#include "ByteArray.h"
#endif

#ifndef ZipFile_h
#include "ZipFile.h"
#endif

#ifndef iterext_h
#include "iterext.h"
#endif

#ifndef DeleteObject_h
#include "DeleteObject.h"
#endif

#ifndef Parser_h
#include "Parser.h"
#endif

#ifndef std_iostream
#define std_iostream
#include <iostream>
#endif

#ifndef std_algorithm
#define std_algorithm
#include <algorithm>
#endif

#ifndef std_algorithm
#define std_algorithm
#include <algorithm>
#endif

#ifndef std_functional
#define std_functional
#include <functional>
#endif

#ifndef std_iterator
#define std_iterator
#include <iterator>
#endif

using namespace std;
using namespace doctorj;

#define TRY_CAST(What, ClassType) \
    { \
        ClassType* ct = dynamic_cast<ClassType*>(What); \
        if (ct) { \
            return ct->getModifierList(); \
        } \
    }


#define TRY_CAST_BLOCK(What, ClassType, Block) \
    { \
        ClassType* ct = dynamic_cast<ClassType*>(What); \
        if (ct) \
            Block \
    }

/**
 * This is repeated in StatisticsAnalyzer.cpp; these should be merged.
 */
#define LIST_CONTAINS(node, countMethod, accessor, baseType, classType) \
    int count = node->countMethod(); \
    for (int i = 0; i < count; ++i) { \
        baseType* x = node->accessor(i); \
        if (dynamic_cast<classType*>(x)) { \
            return true; \
        } \
    } \
    return false;

REGISTER_DEBUG_OPTION('j', "Java code (decompiling and encapsulation)");

JavaArgumentType::JavaArgumentType()
{
}

JavaArgumentType::~JavaArgumentType()
{
}

bool JavaArgumentType::matches(AstFormalParameter* const fp) const
{
    // cout << "JavaArgumentType::matches(" << fp->text() << "): NOT IMPLEMENTED!" << endl;
    return false;
}


JavaArgumentTypeList::JavaArgumentTypeList()
{
}

JavaArgumentTypeList::~JavaArgumentTypeList()
{
    vector<JavaArgumentType*>::iterator stop = arguments_.end();
    vector<JavaArgumentType*>::iterator it = arguments_.begin();
    while (it != stop) {
        JavaArgumentType* argType = *it;
        delete argType;
        ++it;
    }
}

bool JavaArgumentTypeList::matches(AstMethodDeclarator* const md) const
{
    // cout << "checking argument type list against " << md->text() << endl;
    AstMethodDeclaratorWArgs* mdargs = dynamic_cast<AstMethodDeclaratorWArgs*>(md);
    int ct = count();
    if (mdargs) {
        // cout << "that has arguments..." << endl;
        AstFormalParameterList* fpl = mdargs->getFormalParameterList();
        int fpcount = fpl->getFormalParameterCount();
        if (ct == fpcount) {
            for (int i = 0; i < ct; ++i) {
                AstFormalParameter* fp = fpl->getFormalParameter(i);
                JavaArgumentType* argType = arguments_[i];
                if (argType->matches(fp)) {
                    // cout << "MATCH between arguments at index " << i << endl;
                }
                else {
                    // cout << "MISMATCH between arguments at index " << i << endl;
                    return false;
                }
            }
            return true;
        }
        else {
            return false;
        }
    }
    else {
        // cout << "that has NO arguments, and our count is " << ct << "..." << endl;
        // there are no arguments in the method declarator
        return ct == 0;
    }
}

void JavaArgumentTypeList::add(JavaArgumentType* const argType)
{
    arguments_.push_back(argType);
}

int JavaArgumentTypeList::count() const
{
    return arguments_.size();
}


JavaClass::JavaClass(AstClassDeclaration* const cd) : cd_(cd)
{
}

JavaClass::~JavaClass()
{
}

vector<JavaMethod> JavaClass::getMethods() const
{
    vector<JavaMethod> methods;
    AstClassBody* cb = cd_->getClassBody();
    if (cb) {
        AstClassBodyDecls* cbd = dynamic_cast<AstClassBodyDecls*>(cb);
        if (cbd) {
            AstClassBodyDeclarationList* cbdl = cbd->getClassBodyDeclarationList();
            if (cbdl) {
                int count = cbdl->getClassBodyDeclarationCount();
                for (int i = 0; i < count; ++i) {
                    AstItem* decl = cbdl->getClassBodyDeclaration(i);
                    AstMethodDeclaration* md = dynamic_cast<AstMethodDeclaration*>(decl);
                    if (md) {
                        JavaMethod method(md);
                        methods.push_back(method);
                    }
                }
            }
        }
    }
    return methods;
}


static string fullNameToAbbrev(const string& name)
{
    return name.substr(name.rfind('.') + 1);
}

/**
 * Returns whether the given class file matches for the package directory (of
 * the form "java/lang/reflect").
 */
static bool matchesPackage(const string& pkgDir, const string& classFile)
{
    JAVALOGF("considering class file '%s' as part of '%s", classFile.c_str(), pkgDir.c_str());
    string cfDots = classFile.substr(0, classFile.length() - 6); // -6 for ".class"
    StringUtilities::subst(&cfDots, "/", ".");
    StringUtilities::subst(&cfDots, "$", ".");
    int pos = cfDots.rfind('.');
    string cls(cfDots.substr(pos + 1, cfDots.length() - pos - 1)); // -6 for ".class"
    size_t where = cls.find_first_not_of("0123456789");
    if (where == string::npos) {
        JAVALOG("skipping anonymous inner class");
    }
    else {
        string pkg = classFile.substr(0, pos);
        bool b = pkg == pkgDir;
        return b;
    }
    return false;
}

struct stripdir : public unary_function<string, string> {
    string leading_;
    stripdir(const string& leading) : leading_(leading) {}
    string operator()(string dir) const { 
        int pos = dir.find(leading_);
        string sd = pos >= 0 ? dir.substr(leading_.length()) : dir;
        return sd;
    }
};  


JavaClassLoader* JavaClassLoader::instance_ = NULL;

JavaClassLoader* JavaClassLoader::get(const string& classPath)
{
    if (!instance_) {
        instance_ = new JavaClassLoader(classPath);
    }
    return instance_;
}

JavaClassLoader::JavaClassLoader(const string& classPath) : classPath_(classPath)
{
}

JavaClassLoader::~JavaClassLoader()
{
}

bool JavaClassLoader::getTypesInPackage(const string& pkgName, vector<string>* const types)
{
    JAVALOGF("pkgName: %s", pkgName.c_str());
    bool wasRead = readClassPath(pkgName, types);
    JAVALOGF("returning %d", wasRead);
    return wasRead;
}

bool JavaClassLoader::readClassFile(const string& cfname, const string& pkgName, vector<string>* const types)
{
    JAVALOGF("reading class file '%s' for %s", cfname.c_str(), pkgName.c_str());
    bool                    wasFound(false);
    FileByteStreamBigEndian bs(cfname);
    BcClassFileReader       rdr(&bs);
    BcTypeDeclaration*      td(rdr.getType());
    if ((td->access() & BC_ACCESS_PUBLIC) != 0) {
        // this class is public and is in this package
        types->push_back(fullNameToAbbrev(td->name()));
        wasFound = true;
    }
    else {
        JAVALOG("not public access");
    }
    delete td;
    return wasFound;
}

bool JavaClassLoader::readClassFiles(const vector<string>& classes, const string& pkgName, vector<string>* const types)
{
    bool wasFound = false;
    EACHC(vector<string>, classes, cit) {
        string cname = *cit;
        if (matchesPackage(pkgName, cname)) {
            wasFound = readClassFile(cname, pkgName, types) || wasFound;
        }
    }
    return wasFound;
}

bool JavaClassLoader::readArchiveFile(const string& archive, const string& pkgName, vector<string>* const types)
{
    // they can be looking for either:
    //     java.lang     --> matches java/lang/Foo.class      (plain old class)
    //     java.lang.Bar --> matches java/lang/Bar$Baz.class  (inner class)
    // but we won't match anonymous inner types:
    //     java.lang.Bar --> not for java/lang/Bar$1.class    (first anon. inner type in Bar)
    bool wasFound = false;
    string pkgDir = pkgName;
    StringUtilities::subst(&pkgDir, ".", "/");
    ZipFile zf(archive);
    vector<string> names = zf.entryNames();

    int nNames = names.size();
    for (int i = 0; i < nNames; ++i) {
        JAVALOGF("processing '%s'", names[i].c_str());
        if (StringUtilities::endsWith(names[i], ".class")) {
            if (matchesPackage(pkgDir, names[i])) {
                char* bytes  = NULL;
                int   nBytes = zf.getBytes(names[i], &bytes);
                JAVALOGF("%d\t%s\n", nBytes, names[i].c_str());
                ByteArrayBigEndian ba(bytes, nBytes);
                BcClassFileReader  rdr(&ba);
                BcTypeDeclaration* td(rdr.getType());
                if ((td->access() & BC_ACCESS_PUBLIC) != 0) {
                    //  this class is public and is in this package
                    types->push_back(fullNameToAbbrev(td->name()));
                    JAVALOGF("archive: %s; found type for %s", archive.c_str(), pkgName.c_str());
                    wasFound = true;
                }
                else {
                    JAVALOG("not public access");
                }
                delete td;
            }
            else {
                JAVALOG("not a match");
            }
        }
        else {
            JAVALOGF("%s: not a class file", names[i].c_str());
        }
    }

    return wasFound;
}

bool JavaClassLoader::readClassPath(const string& pkgName, vector<string>* const types)
{
    // foreach file:
    //     if .zip or .jar file
    //         uncompress file
    //         foreach .class in file
    //             decompile
    //             read the public type name
    //             get the package name
    //             packages[pkgName].add(typeName)
    //     else if .class:
    //         decompile
    //         read the public type name
    //         get the package name
    //         packages[pkgName].add(typeName)

    bool   wasFound = false;
    string classDir = pkgName;
    StringUtilities::subst(&classDir, ".", "/"); // foo.bar.baz => foo/bar/baz
    
    if (classPath_.length() > 0) {
        vector<string> cpath = StringUtilities::split(classPath_, ":");
        for (unsigned int i = 0; i < cpath.size(); ++i) {
            string cp = cpath[i];
            JAVALOGF("checking class path: %s", cp.c_str());
            if (Platform::isDirectory(cp)) {
                // cout << "    classpath segment is a directory" << endl;
                // read each zip and jar file in the directory, and the class files
                vector<string> classes;
                Platform::getFiles(cp + "/" + classDir, "class", &classes);
                string leading(cp + "/");
                EACHC(vector<string>, classes, cit) {
                    string cname = *cit;
                    JAVALOGF("considering decompiling '%s'", cname.c_str());
                    int pos = cname.find(leading);
                    string shortName = pos >= 0 ? cname.substr(leading.length()) : cname;
                    JAVALOGF("short name = '%s'", shortName.c_str());

                    if (matchesPackage(classDir, shortName)) {
                        wasFound = readClassFile(cname, pkgName, types) || wasFound;
                    }
                }

                // wasFound = readClassFiles(classes, classDir, types) || wasFound;

                vector<string> archives;
                Platform::getFiles(cp, "jar", "zip", &archives);
                EACH(vector<string>, archives, ait) {
                    string archive = *ait;
                    wasFound = readArchiveFile(archive, pkgName, types) || wasFound;
                }
            }
            else if (Platform::isRegularFile(cp) || Platform::isLink(cp)) {
                string ext = FileName::extension(cp);
                if (ext == "zip" || ext == "jar") {
                    readArchiveFile(cp, pkgName, types);
                }
                else {
                    // what else could it be?
                    JAVALOGF("unknown type in classpath: %s", cp.c_str());
                }
            }
        }
    }
    else {
        JAVALOG("no class path");
    }
    return wasFound;
}


static bool hasTrueExpression(AstItem* const it)
{
    AstPrimaryParenExpression* ppe = dynamic_cast<AstPrimaryParenExpression*>(it);
    if (ppe) {
        return hasTrueExpression(ppe->getExpression());
    }
    else {
        return dynamic_cast<AstTrueLiteral*>(it);
    }
}

static bool hasFalseExpression(AstItem* const it)
{
    AstPrimaryParenExpression* ppe = dynamic_cast<AstPrimaryParenExpression*>(it);
    if (ppe) {
        return hasFalseExpression(ppe->getExpression());
    }
    else {
        return dynamic_cast<AstFalseLiteral*>(it);
    }
}

bool JavaEqualityExpression::hasTrueLiteralInExpression(AstEqualityExpression* const ee)
{
    AstItem* lhs = ee->getLhsExpression();
    AstItem* rhs = ee->getRhsExpression();

    return hasTrueExpression(lhs) || hasTrueExpression(rhs);
}

bool JavaEqualityExpression::hasFalseLiteralInExpression(AstEqualityExpression* const ee)
{
    AstItem* lhs = ee->getLhsExpression();
    AstItem* rhs = ee->getRhsExpression();

    return hasFalseExpression(lhs) || hasFalseExpression(rhs);
}


JavaArgumentTypeList JavaGrammar::getArgumentTypes(AstMethodInvocationSuperArgs* const mi)
{
    JavaArgumentTypeList at;
    //!!! not implemented

    // 
    
    return at;
}

JavaArgumentTypeList JavaGrammar::getArgumentTypes(AstMethodInvocationSuperNoArgs* const mi)
{
    // empty list.
    JavaArgumentTypeList at;
    return at;
}

bool JavaGrammar::isPrimitiveType(AstItem* const varType)
{
    string type = varType->text();
    return (type == "boolean" ||
            type == "byte"    ||
            type == "char"    ||
            type == "double"  ||
            type == "float"   ||
            type == "int"     ||
            type == "long"    ||
            type == "short");
}
   

JavaInterface::JavaInterface(AstInterfaceDeclaration* const id) : id_(id)
{
}

JavaInterface::~JavaInterface()
{
}

vector<JavaMethod> JavaInterface::getMethods() const
{
    vector<JavaMethod> methods;
    if (AstInterfaceBody* ib = id_->getInterfaceBody()) {
        if (AstInterfaceBodyDecls* ibd = dynamic_cast<AstInterfaceBodyDecls*>(ib)) {
            if (AstInterfaceMemberDeclarationList* imdl = ibd->getInterfaceMemberDeclarationList()) {
                int count = imdl->getInterfaceMemberDeclarationCount();
                for (int i = 0; i < count; ++i) {
                    AstItem* decl = imdl->getInterfaceMemberDeclaration(i);
                    // cout << "checking declaration: " << decl->type() << ": " << decl->text() << endl;
                    if (AstAbstractMethodDeclaration* amd = dynamic_cast<AstAbstractMethodDeclaration*>(decl)) {
                        // cout << "well, now I know that is an abstract method declaration..." << endl;
                        JavaMethod method(amd->getMethodHeader());
                        methods.push_back(method);
                    }
                }
            }
        }
    }
    return methods;
}

static vector<string> toVector(const string& s)
{
    vector<string> v;
    v.push_back(s);
    return v;
}

JavaType* JavaItem::getContainingType(AstItem* const item)
{
    // cout << "seeking containing type for " << item->text() << endl;
    AstItem* p = item->parent();
    while (p) {
        // cout << "    considering parent: " << p->type() << endl;
        AstClassDeclaration* cd = dynamic_cast<AstClassDeclaration*>(p);
        if (cd) {
            // cout << "got a class type..." << endl;
            return new JavaClass(cd);
        }
        else {
            AstInterfaceDeclaration* id = dynamic_cast<AstInterfaceDeclaration*>(p);
            if (id) {
                // cout << "got an interface type..." << endl;
                return new JavaInterface(id);
            }
            else {
                p = p->parent();
            }
        }
    }
    return NULL;
}

vector<string> JavaItem::namesOf(AstItem* const item)
{
    AstItem* it = item;

    // I'm not sure that I like doing it this way...
    TRY_CAST_BLOCK(it, AstClassMemberDeclaration, {
        return toVector(ct->getIdentifier()->text());
    });
    
    TRY_CAST_BLOCK(it, AstFieldDeclaration, {
        it = ct->getVariableDeclaratorList();
    });
    
    TRY_CAST_BLOCK(it, AstVariableDeclaratorList, {
        vector<string> v;
        int count = ct->getVariableDeclaratorCount();
        for (int i = 0; i < count; ++i) {
            AstVariableDeclarator* vd = ct->getVariableDeclarator(i);
            v.push_back(vd->name());
        }
        return v;
    });
    
    TRY_CAST_BLOCK(it, AstInterfaceDeclaration, {
        return toVector(ct->getIdentifier()->text());
    });
    
    TRY_CAST_BLOCK(it, AstMethodHeader, {
        return toVector(ct->name());
    });

    TRY_CAST_BLOCK(it, AstMethodDeclaration, {
        return toVector(ct->getMethodHeader()->name());
    });

    TRY_CAST_BLOCK(it, AstAbstractMethodDeclaration, {
        return toVector(ct->getMethodHeader()->name());
    });

    TRY_CAST_BLOCK(it, AstConstructorDeclarator, {
        return toVector(ct->getIdentifier()->text());
    });

    TRY_CAST_BLOCK(it, AstConstructorDeclaration, {
        return toVector(ct->getConstructorDeclarator()->getIdentifier()->text());
    });

    TRY_CAST_BLOCK(it, AstStaticInitializer, {
        return toVector("<clinit>");
    });

    return toVector("<<unknown name for " + it->type() + ">>");
}

AstItem* JavaItem::primaryItem(AstItem* const item)
{
    if (AstClassMemberDeclaration* cmd = dynamic_cast<AstClassMemberDeclaration*>(item)) {
        return cmd->getIdentifier();
    }
    else if (AstFieldDeclaration* fd = dynamic_cast<AstFieldDeclaration*>(item)) {
        AstVariableDeclaratorList* vdl = fd->getVariableDeclaratorList();
        return vdl;
    }
    else if (AstInterfaceDeclaration* id = dynamic_cast<AstInterfaceDeclaration*>(item)) {
        return id;
    }
    else if (AstMethodHeader* mh = dynamic_cast<AstMethodHeader*>(item)) {
        return mh->getMethodDeclarator();
    }
    else if (AstAbstractMethodDeclaration* md = dynamic_cast<AstAbstractMethodDeclaration*>(item)) {
        return md->getMethodHeader();
    }
    else if (AstConstructorDeclaration* cd = dynamic_cast<AstConstructorDeclaration*>(item)) {
        return cd->getConstructorDeclarator();
    }
    else if (AstMethodDeclaration* md = dynamic_cast<AstMethodDeclaration*>(item)) {
        return md->getMethodHeader();
    }
    else {
        return item;
    }
}

AstModifierList* JavaItem::getModifierList(AstItem* const item)
{
    // Oh, I wish C++ were better with multiple inheritance...
    TRY_CAST(item, AstClassDeclarationModsBaseless);
    TRY_CAST(item, AstClassDeclarationModsExtends);
    TRY_CAST(item, AstClassDeclarationModsExtendsImplements);
    TRY_CAST(item, AstClassDeclarationModsImplements);
    TRY_CAST(item, AstClassMemberDeclarationModsExtendsImplements);
    TRY_CAST(item, AstClassMemberDeclarationModsExtendsNoImplements);
    TRY_CAST(item, AstClassMemberDeclarationModsNoExtendsImplements);
    TRY_CAST(item, AstClassMemberDeclarationModsNoExtendsNoImplements);
    TRY_CAST(item, AstConstructorDeclarationModsNoThrows);
    TRY_CAST(item, AstConstructorDeclarationModsThrows);
    TRY_CAST(item, AstFieldDeclarationMods);
    TRY_CAST(item, AstInterfaceDeclarationModsExtends);
    TRY_CAST(item, AstInterfaceDeclarationModsNoExtends);
    TRY_CAST(item, AstMethodHeaderTypeModsNoThrows);
    TRY_CAST(item, AstMethodHeaderTypeModsThrows);
    TRY_CAST(item, AstMethodHeaderVoidModsNoThrows);
    TRY_CAST(item, AstMethodHeaderVoidModsThrows);
    
    TRY_CAST_BLOCK(item, AstMethodDeclaration, {
        return getModifierList(ct->getMethodHeader());
    });

    TRY_CAST_BLOCK(item, AstAbstractMethodDeclaration, {
        return getModifierList(ct->getMethodHeader());
    });

    return NULL;
}

string JavaItem::typeOf(AstItem* const item)
{
    string t = StringUtilities::toLower(item->type());
    if (StringUtilities::startsWith(t, "class")) {
        AstModifierList* mods = getModifierList(item);
        if (mods && JavaModifierList::getAbstractModifier(mods)) {
            return "abstractclass";
        }
        else {
            return "class";
        }
    }
    else if (StringUtilities::startsWith(t, "constructor")) {
        return "ctor";
    }
    else if (StringUtilities::startsWith(t, "field") ||
             StringUtilities::startsWith(t, "variable")) {
        return "field";
    }
    else if (StringUtilities::startsWith(t, "interface")) {
        return "interface";
    }
    else if (StringUtilities::startsWith(t, "method")) {
        return "method";
    }
    else if (StringUtilities::startsWith(t, "semicolon")) {
        return "semicolon";
    }
    else if (StringUtilities::startsWith(t, "staticinitializer")) {
        return "staticinitializer";
    }
    else if (StringUtilities::startsWith(t, "abstractmethod")) {
        return "abstractmethod";
    }
    else if (StringUtilities::startsWith(t, "block")) {
        return "block";
    }
    else {
        cerr << "type '" << t << "' not understood as type declaration" << endl;
        return "<<unknown>>";
    }
}

int JavaItem::lineCount(AstItem* const i)
{
    return i->endLine() - i->line() + 1;
}

int JavaItem::lineCount(AstItem* const from, AstItem* const to)
{
    return to->endLine() - from->line() + 1;
}

bool JavaItem::isEmptyStatement(AstItem* const statement)
{
    if (AstSemicolon* sc = dynamic_cast<AstSemicolon*>(statement)) {
        AstNoncode* lnc = sc->leadingNoncode();
        return lnc == NULL || JavaNoncode::getFirstComment(lnc) == NULL;
    }
    else {
        return false;
    }
}

AstNoncode* JavaItem::getTrailingNoncode(AstItem* const item) 
{
    AstNoncode* nc = item->trailingNoncode();
    AstItem* i = item;
    while (i && !nc) {
        i = i->parent();
        nc = i->trailingNoncode();
    }
    return nc;
}


JavaLoader* JavaLoader::instance_ = NULL;

JavaLoader* JavaLoader::get(AstItem* const item, const string& sourcePath, const string& classPath)
{
    if (!instance_) {
        instance_ = new JavaLoader(item, sourcePath, classPath);
    }
    return instance_;
}

JavaLoader::JavaLoader(AstItem* const item, const string& sourcePath, const string& classPath) : 
     sourceLoader_(item, sourcePath), classLoader_(classPath)
{
}

JavaLoader::~JavaLoader()
{
}

bool JavaLoader::getTypesInPackage(const string& pkgName, vector<string>* const types)
{
    JAVALOG("");

    // either from source ...
    bool wasRead = sourceLoader_.getTypesInPackage(pkgName, types);

    // ... or from class/archive files
    wasRead = classLoader_.getTypesInPackage(pkgName, types) || wasRead;
    
    return wasRead;
}

static bool containsAbstract(AstModifierList* const mods) 
{
    LIST_CONTAINS(mods, getModifierCount, getModifier, AstLeaf, AstAbstract);
}

JavaMethod::JavaMethod(AstMethodHeader* const header) : header_(header)
{
}

JavaMethod::JavaMethod(AstMethodDeclaration* const method) : header_(method->getMethodHeader())
{
}

JavaMethod::~JavaMethod()
{
}

bool JavaMethod::isMatch(const string& name, const JavaArgumentTypeList& argTypes) const
{
    string methName = getName();
    if (name == methName) {
        AstMethodDeclarator* mdec = header_->getMethodDeclarator();
        bool argsMatch = argTypes.matches(mdec);
        if (argsMatch) {
            return true;
        }
    }
    else {
        // the names don't match
    }
    return false;
}

AstItem* JavaMethod::getReturnType() const
{
    TRY_CAST_BLOCK(header_, AstMethodHeaderTypeNoModsNoThrows, {
        return ct->getType();
    });

    TRY_CAST_BLOCK(header_, AstMethodHeaderTypeNoModsThrows, {
        return ct->getType();
    });

    TRY_CAST_BLOCK(header_, AstMethodHeaderVoidNoModsNoThrows, {
        return ct->getVoid();
    });

    TRY_CAST_BLOCK(header_, AstMethodHeaderVoidNoModsThrows, {
        return ct->getVoid();
    });

    TRY_CAST_BLOCK(header_, AstMethodHeaderTypeModsNoThrows, {
        return ct->getType();
    });

    TRY_CAST_BLOCK(header_, AstMethodHeaderTypeModsThrows, {
        return ct->getType();
    });

    TRY_CAST_BLOCK(header_, AstMethodHeaderVoidModsNoThrows, {
        return ct->getVoid();
    });
    
    TRY_CAST_BLOCK(header_, AstMethodHeaderVoidModsThrows, {
        return ct->getVoid();
    });

    return NULL;
}

bool JavaMethod::isAbstract() const
{
    TRY_CAST_BLOCK(header_, AstMethodHeaderTypeNoModsNoThrows, {
        return false;
    });

    TRY_CAST_BLOCK(header_, AstMethodHeaderTypeNoModsThrows, {
        return false;
    });

    TRY_CAST_BLOCK(header_, AstMethodHeaderVoidNoModsNoThrows, {
        return false;
    });

    TRY_CAST_BLOCK(header_, AstMethodHeaderVoidNoModsThrows, {
        return false;
    });

    TRY_CAST_BLOCK(header_, AstMethodHeaderTypeModsNoThrows, {
        return containsAbstract(ct->getModifierList());
    });

    TRY_CAST_BLOCK(header_, AstMethodHeaderTypeModsThrows, {
        return containsAbstract(ct->getModifierList());
    });

    TRY_CAST_BLOCK(header_, AstMethodHeaderVoidModsNoThrows, {
        return containsAbstract(ct->getModifierList());
    });
    
    TRY_CAST_BLOCK(header_, AstMethodHeaderVoidModsThrows, {
        return containsAbstract(ct->getModifierList());
    });

    return false;
}

string JavaMethod::getName() const
{
    return header_->getMethodDeclarator()->getIdentifier()->text();
}


using namespace doctorj;

static AstModifierList* getModifierList(AstInterfaceDeclaration* const id)
{
    if (dynamic_cast<AstInterfaceDeclarationModsExtends*>(id)) {
        return ((AstInterfaceDeclarationModsExtends*)id)->getModifierList();
    }
    else if (dynamic_cast<AstInterfaceDeclarationModsNoExtends*>(id)) {
        return ((AstInterfaceDeclarationModsNoExtends*)id)->getModifierList();
    }
    else {
        return NULL;
    }
}    

static AstModifierList* getModifierList(AstMethodHeader* const header)
{
    if (dynamic_cast<AstMethodHeaderTypeModsNoThrows*>(header)) {
        return ((AstMethodHeaderTypeModsNoThrows*)header)->getModifierList();
    }
    else if (dynamic_cast<AstMethodHeaderTypeModsThrows*>(header)) {
        return ((AstMethodHeaderTypeModsThrows*)header)->getModifierList();
    }
    else if (dynamic_cast<AstMethodHeaderVoidModsNoThrows*>(header)) {
        return ((AstMethodHeaderVoidModsNoThrows*)header)->getModifierList();
    }
    else if (dynamic_cast<AstMethodHeaderVoidModsThrows*>(header)) {
        return ((AstMethodHeaderVoidModsThrows*)header)->getModifierList();
    }
    else {
        return NULL;
    }
}

static AstModifierList* getModifierList(AstClassDeclaration* const cls)
{
    return cls->getModifierList();
}

JavaModifierList::JavaModifierList(AstMethodDeclaration* const method)
{
    AstMethodHeader* header = method->getMethodHeader();
    modifiers_              = getModifierList(header);
}

JavaModifierList::JavaModifierList(AstClassDeclaration* const cls)
{
    modifiers_ = getModifierList(cls);
}

JavaModifierList::JavaModifierList(AstInterfaceDeclaration* const id)
{
    modifiers_ = getModifierList(id);
}

JavaModifierList::JavaModifierList(AstFieldDeclarationMods* const field) : modifiers_(field->getModifierList())
{
}

JavaModifierList::JavaModifierList(AstModifierList* const modifiers) : modifiers_(modifiers)
{
}

JavaModifierList::~JavaModifierList()
{
}

bool JavaModifierList::containsStatic() const
{
    return getStaticModifier(modifiers_) != NULL;
}

bool JavaModifierList::containsFinal() const
{
    return getFinalModifier(modifiers_) != NULL;
}

AstLeaf* JavaModifierList::getAccess() const
{
    return getAccess(modifiers_);
}

string JavaModifierList::getAccessString() const
{
    return accessOf(modifiers_);
}

string JavaModifierList::accessOf(AstModifierList* const mods)
{
    if (mods) {
        AstLeaf* al = getAccess(mods);
        if (al) {
            return al->text();
        }
    }
    return "package";
}

string JavaModifierList::finalityOf(AstModifierList* const mods)
{
    return mods && getFinalModifier(mods) ? "final" : "nonfinal";
}

string JavaModifierList::ownershipOf(AstModifierList* const mods)
{
    return mods && getStaticModifier(mods) ? "static" : "instance";
}

AstLeaf* JavaModifierList::getAccess(AstModifierList* const mods)
{
    if (mods) {
        EACHC(vector<AstLeaf*>, mods->getModifiers(), mit) {
            AstLeaf* mod = *mit;
            if (mod) {
                if (dynamic_cast<AstPublic*>(mod) || 
                    dynamic_cast<AstProtected*>(mod) ||
                    dynamic_cast<AstPrivate*>(mod)) {
                    return mod;
                }
            }
        }
    }
    return NULL;
}

AstStatic* JavaModifierList::getStaticModifier(AstModifierList* const mods)
{
    return mods ? JavaItem::firstOfType<AstStatic*>(mods->getModifiers()) : NULL;
}

AstAbstract* JavaModifierList::getAbstractModifier(AstModifierList* const mods)
{
    return mods ? JavaItem::firstOfType<AstAbstract*>(mods->getModifiers()) : NULL;
}

AstFinal* JavaModifierList::getFinalModifier(AstModifierList* const mods)
{
    return mods ? JavaItem::firstOfType<AstFinal*>(mods->getModifiers()) : NULL;
}


template < class Type >
static Type* seekFirst(AstNoncode* const nc)
{
    list<AstItem*>&          nccomps = nc->components();
    list<AstItem*>::iterator stop    = nccomps.end();
    
    for (list<AstItem*>::iterator it = nccomps.begin(); it != stop; ++it) {
        if (AstItem* item = *it) {
            Type* cmt = dynamic_cast<Type*>(item);
            if (cmt) {
                return cmt;
            }
        }
    }
    return NULL;
}

AstComment* JavaNoncode::getFirstComment(AstNoncode* const nc)
{          
    return seekFirst<AstComment>(nc);
}

AstJavadocComment* JavaNoncode::getFirstJavadocComment(AstNoncode* const nc)
{          
    return seekFirst<AstJavadocComment>(nc);
}


JavaPackage::JavaPackage(const string& pkgName) : pkgName_(pkgName)
{
}

JavaPackage::~JavaPackage()
{
}

string JavaPackage::getName() const
{
    return pkgName_;
}

void JavaPackage::addType(const string& type)
{
    types_.push_back(type);
}
        
void JavaPackage::getTypes(vector<string>* const types) const
{
    *types = types_;
}


JavaParameter::JavaParameter()
{
}

JavaParameter::JavaParameter(AstFormalParameter* const fp) : fp_(fp)
{
}

JavaParameter::~JavaParameter()
{
}

bool JavaParameter::isMatch(AstFormalParameter* const fp, AstItem* const item)
{
//     AstVariableDeclaratorId* vdid = fp->getVariableDeclaratorId();
//     AstIdentifier*           id   = vdid->getIdentifier();
    return isMatch(fp, item->text());
}

bool JavaParameter::isMatch(AstFormalParameter* const param, const string& name)
{
    return param->name() == name;
}

string JavaParameter::getType() const
{
    AstItem* type = fp_->getType();
    return type->text();
}

bool JavaParameter::isFinal() const
{
    return fp_->getFinal() != NULL;
}

string JavaParameter::getName() const
{
    return fp_->name();
}


JavaParameterList::JavaParameterList()
{
}

JavaParameterList::~JavaParameterList()
{
}

AstFormalParameter* JavaParameterList::getMatching(AstFormalParameterList* const params, AstItem* const item)
{
    return getMatching(params, item->text());
}

AstFormalParameter* JavaParameterList::getMatching(AstFormalParameterList* const params, const string& name)
{
    int nParams = params ? params->getFormalParameterCount() : 0;
    for (int i = 0; i < nParams; ++i) {
        AstFormalParameter* param = params->getFormalParameter(i);
        if (JavaParameter::isMatch(param, name)) {
            return param;
        }
        else {
            // not a matching parameter
        }
    }
    return NULL;
}


JavaSourceLoader* JavaSourceLoader::instance_ = NULL;

extern Parser parser;

extern AstCompilationUnit* compUnit;

JavaSourceLoader* JavaSourceLoader::get(AstItem* const item, const string& sourcePath)
{
    if (!instance_) {
        instance_ = new JavaSourceLoader(item, sourcePath);
    }
    return instance_;
}

JavaSourceLoader::JavaSourceLoader(AstItem* const item, const string& sourcePath) : 
        project_(NULL), packages_(NULL), sourcePath_(sourcePath)

{
    AstItem* p = item;
    while (p && !project_) {
        project_ = dynamic_cast<AstProject*>(p);
        p = p->parent();
    }
}

JavaSourceLoader::~JavaSourceLoader()
{
    if (packages_) {
        for_each(packages_->begin(), packages_->end(), DeleteObject());
        delete packages_;
    }
}

bool JavaSourceLoader::getTypesInPackage(const string& pkgName, vector<string>* const types)
{
    // cout << "JavaSourceLoader::getTypesInPackage(" << pkgName << ", types" << ")" << endl;

    // I don't think that we can cache much here. Maybe only the class names
    // within the archive files.

    bool wasRead = false;

    JAVALOGF("reading for '%s'", pkgName.c_str());

    // maybe we've parsed it in the set of input files...
    int cuc = project_ ? project_->getCompilationUnitCount() : 0;
    for (int i = 0; i < cuc; ++i) {
        AstCompilationUnit* cu = project_->getCompilationUnit(i);
        wasRead = readCompilationUnit(cu, pkgName, types) || wasRead;
    }

    // there could be other (non-analyzed) source files
    wasRead = readSourcePath(pkgName, types) || wasRead;

    JAVALOGF("returning %d", wasRead);

    return wasRead;
}

bool JavaSourceLoader::getPublicType(AstTypeDeclarationList* const typelist, vector<string>* const types)
{
    for (int i = 0, count = typelist->getTypeDeclarationCount(); i < count; ++i) {
        AstItem* type = typelist->getTypeDeclaration(i);
        // the type can be:
        //     class_declaration
        //     interface_declaration
        //     SEMICOLON
                
        // we care only about public classes and interfaces, so they
        // must have modifiers (for "public").
        if (AstClassDeclaration* cd = dynamic_cast<AstClassDeclaration*>(type)) {
            JavaModifierList ml(cd);
            if (ml.getAccessString() == "public") {
                AstIdentifier* cid = cd->getIdentifier();
                string typeName = cid->text();
                JAVALOGF("adding public class %s", typeName.c_str());
                types->push_back(typeName);
                // because there cannot be any more public types in this compilation unit
                return true;
            }
        }
        else {
            // an interface?
            if (AstInterfaceDeclaration* id = dynamic_cast<AstInterfaceDeclaration*>(type)) {
                JavaModifierList ml(id);
                if (ml.getAccessString() == "public") {
                    AstIdentifier* iid = id->getIdentifier();
                    string typeName = iid->text();
                    JAVALOGF("adding public interface %s", typeName.c_str());
                    types->push_back(typeName);
                    // because there cannot be any more public types in this compilation unit
                    return true;
                }
            }
            else {
                // neither an interface nor a class. Must be a semicolon.
            }
        }
    }
    return false;
}

bool JavaSourceLoader::readCompilationUnits(const string& pkgName, vector<string>* const types)
{
    bool wasFound = false;
    int  cuc      = project_ ? project_->getCompilationUnitCount() : 0;
    for (int i = 0; i < cuc; ++i) {
        AstCompilationUnit* cu = project_->getCompilationUnit(i);
        wasFound = readCompilationUnit(cu, pkgName, types) || wasFound;
    }
    JAVALOGF("pkgName: %s; #types: %d; found? %d", pkgName.c_str(), types->size(), wasFound);
    return wasFound;
}


bool JavaSourceLoader::readCompilationUnit(AstCompilationUnit* const cu, const string& pkgName, vector<string>* const types)
{
    // we only care about compilation units with both a package declaration
    // and a list of type declarations

    AstPackageDeclaration*  pkg      = cu->getPackageDeclaration();
    AstTypeDeclarationList* typelist = cu->getTypeDeclarationList();

    if (pkg && pkg->getName()->text() == pkgName && typelist) {
        return getPublicType(typelist, types);
    }
    else {
        return false;
    }
}

bool JavaSourceLoader::readSourceFile(const string& sourceFile, const string& pkgName, vector<string>* const types)
{
    // cout << "parsing " << sourceFile << endl;
    parser.parse(compUnit, sourceFile);
    bool wasFound = false;
    if (compUnit) {
        JAVALOG("got a compilation unit");
        wasFound = readCompilationUnit(compUnit, pkgName, types) || wasFound;
        delete compUnit;
    }
    else {
        cerr << "doctorj: no compilation unit; error during parsing" << endl;
    }
    return wasFound;
}

bool JavaSourceLoader::readSourcePath(const string& pkgName, vector<string>* const types)
{
    string pkgDir = pkgName;
    StringUtilities::subst(&pkgDir, ".", "/"); // foo.bar.baz => foo/bar/baz
    bool wasFound = false;
    
    if (sourcePath_.length() > 0) {
        vector<string> spath = StringUtilities::split(sourcePath_, ":");
        for (unsigned int i = 0; i < spath.size(); ++i) {
            string sp = spath[i];
            // cout << "    checking source path: " << sp << endl;
            if (Platform::isDirectory(sp)) {
                string dir = sp + "/" + pkgDir;
                vector<string> javas;
                Platform::getFiles(dir, "java", &javas);
                // foreach .java file:
                //     if not compiled:
                //         compile it
                //         get the public type from the compilation unit
                //         get the package name
                //         packages[pkgName].add(typeName)
                //         delete the compilation unit
                EACH(vector<string>, javas, jit) {
                    string jname = *jit;
                    wasFound = readSourceFile(jname, pkgName, types) || wasFound;
                }
            }
            else if (FileName::extension(sp) == "java") {
                wasFound = readSourceFile(sp, pkgName, types) || wasFound;
            }
            else {
                // huh?
                // cout << "unknown type in sourcepath: " << sp << endl;
            }
        }
    }
    else {
        // cout << "no source path" << endl;
    }

    return wasFound;
}


AstSwitchLabelDefault* JavaSwitchStatement::getFirstSwitchLabelDefault(AstSwitchLabelList* const sll)
{
    int nLabels = sll->getSwitchLabelCount();
    for (int i = 0; i < nLabels; ++i) {
        AstSwitchLabel* sl = sll->getSwitchLabel(i);
        AstSwitchLabelDefault* sld = dynamic_cast<AstSwitchLabelDefault*>(sl);
        if (sld) {
            return sld;
        }
    }
    return NULL;
}

AstSwitchLabelDefault* JavaSwitchStatement::getFirstSwitchLabelDefault(AstSwitchBlockStatementGroup* const sbsg)
{
    AstSwitchLabelList*    sll = sbsg->getSwitchLabelList();
    AstSwitchLabelDefault* sld = getFirstSwitchLabelDefault(sll);
    return sld;
}

AstSwitchLabelDefault* JavaSwitchStatement::getFirstSwitchLabelDefault(AstSwitchBlockStatementGroupList* const sbsgl)
{
    int sbsglc = sbsgl->getSwitchBlockStatementGroupCount();
    for (int si = 0; si < sbsglc; ++si) {
        AstSwitchBlockStatementGroup* sbsg = sbsgl->getSwitchBlockStatementGroup(si);
        AstSwitchLabelDefault*        sld  = getFirstSwitchLabelDefault(sbsg);
        if (sld) {
            return sld;
        }
    }
    return NULL;
}

AstSwitchLabelDefault* JavaSwitchStatement::getFirstSwitchLabelDefault(AstSwitchStatement* const ss)
{
    AstSwitchBlock* sb = ss->getSwitchBlock();
    if (AstSwitchBlockStatementGroupList* sbsgl = sb->getSwitchBlockStatementGroupList()) {
        if (AstSwitchLabelDefault* sldef = getFirstSwitchLabelDefault(sbsgl)) {
            return sldef;
        }
    }
    
    if (AstSwitchLabelList* sll = sb->getSwitchLabelList()) {
        return getFirstSwitchLabelDefault(sll);
    }
    else {
        return NULL;
    }
}

#ifdef DEBUGGING
static void spew(const string& name, AstItem* const item, int indent = 0)
{
    while (--indent >= 0) {
        cout << "    ";
    }
    cout << name << " = " << item->type() << " (" << item->text() << ")" << endl;
}
#endif

static AstLocalVariableDeclaration* getVariableDeclaration(AstForStatement* const fs)
{
    if (AstForStatementVarsConditionNoUpdate* f = dynamic_cast<AstForStatementVarsConditionNoUpdate*>(fs)) {
        return f->getLocalVariableDeclaration();
    }
    else if (AstForStatementVarsConditionUpdate* f = dynamic_cast<AstForStatementVarsConditionUpdate*>(fs)) {
        return f->getLocalVariableDeclaration();
    }
    else if (AstForStatementVarsNoConditionNoUpdate* f = dynamic_cast<AstForStatementVarsNoConditionNoUpdate*>(fs)) {
        return f->getLocalVariableDeclaration();
    }
    else if (AstForStatementVarsNoConditionUpdate * f = dynamic_cast<AstForStatementVarsNoConditionUpdate *>(fs)) {
        return f->getLocalVariableDeclaration();
    }
    else {
        return NULL;
    }
}

static AstItem* findMatchingVariable(AstItem* const item, AstVariableDeclaratorList* const vars)
{
    for (int vi = 0, nVars = vars->getVariableDeclaratorCount(); vi < nVars; ++vi) {
        AstVariableDeclarator* vd = vars->getVariableDeclarator(vi);
        // DEBUG_JAVA_CODE(cout << "                    considering var dec " << vd->type() << " (" << vd->text() << ")" << endl);
        AstVariableDeclaratorId* vdid = vd->getVariableDeclaratorId();
        // DEBUG_JAVA_CODE(cout << "                        considering var dec id " << vdid->type() << " (" << vdid->text() << ")" << endl);
        AstIdentifier* id = vdid->getIdentifier();
        // DEBUG_JAVA_CODE(cout << "                            considering var id " << id->type() << " (" << id->text() << ")" << endl);
        if (id->text() == item->text()) {
            // DEBUG_JAVA_CODE(cout << "                                got a match!" << endl);
            return vd;
        }
    }
    return NULL;
}


static AstItem* findMatchingVariable(AstItem* const item, AstLocalVariableDeclaration* const lvd)
{
    AstVariableDeclaratorList* vars = lvd->getVariableDeclaratorList();
    return findMatchingVariable(item, vars);
}

JavaSymbol::JavaSymbol()
{
}

JavaSymbol::~JavaSymbol()
{
}

// static bool isParameterMatch(AstFormalParameter* const fp, AstItem* const item)
// {
//     AstVariableDeclaratorId* vdid = fp->getVariableDeclaratorId();
//     AstIdentifier*           id   = vdid->getIdentifier();
//     return id->text() == item->text();
// }

static AstFormalParameter* getMatchingParameter(AstFormalParameterList* const params, AstItem* const item)
{
    int nParams = params->getFormalParameterCount();
    for (int i = 0; i < nParams; ++i) {
        // DEBUG_JAVA_CODE(cout << "            testing parameter #" << i << endl);
        AstFormalParameter* param = params->getFormalParameter(i);
        if (JavaParameter::isMatch(param, item)) {
            // DEBUG_JAVA_CODE(cout << "            catch clause parameter matches; returning " << (void*)param << endl);
            return param;
        }
        else {
            // DEBUG_JAVA_CODE(cout << "            not a matching catch clause parameter" << endl);
        }
    }
    return NULL;
}

static AstLocalVariableDeclaration* findAsLocalVariable(AstBlockStatementList* const block, AstItem* const statement, AstName* name)
{
    AstItem* stmt = statement;
    while (stmt) {
        // DEBUG_JAVA_CODE(cout << "            testing statement: " << stmt->type() << " (" << stmt->text() << ")" << endl);
        if (AstLocalVariableDeclarationStatement* vds = dynamic_cast<AstLocalVariableDeclarationStatement*>(stmt)) {
            // DEBUG_JAVA_CODE(cout << "                we've got variables!" << endl);
            AstLocalVariableDeclaration* lvd = vds->getLocalVariableDeclaration();
            if (AstItem* var = findMatchingVariable(name, lvd)) {
                return lvd;
            }
            else {
                // DEBUG_JAVA_CODE(cout << "                no variables matched" << endl);
            }
        }
        else {
            // DEBUG_JAVA_CODE(cout << "                not a variable declaration statement" << endl);
        }
        stmt = block->findPrevious(stmt);
    }
    return NULL;
}

static AstFormalParameter* findAsCatchClauseParameter(AstCatchClause* const cc, AstName* const name)
{
    AstFormalParameter* ccparam = cc->getFormalParameter();
    if (JavaParameter::isMatch(ccparam, name)) {
        return ccparam;
    }
    else {
        // DEBUG_JAVA_CODE(cout << "            not a matching catch clause parameter" << endl);
        return NULL;
    }
}

static AstLocalVariableDeclaration* findAsForStatementVariable(AstForStatement* const fs, AstName* const name)
{
    if (AstLocalVariableDeclaration* lvd = getVariableDeclaration(fs)) {
        if (AstItem* var = findMatchingVariable(name, lvd)) {
            return lvd;
        }
        else {
            // DEBUG_JAVA_CODE(cout << "not a matching variable" << endl);
            return NULL;
        }
    }
    else {
        // DEBUG_JAVA_CODE(cout << "for statement has no variables" << endl);
        return NULL;
    }
}

static AstFormalParameter* findAsConstructorParameter(AstConstructorDeclaration* const ctor, AstName* const name)
{
    AstConstructorDeclarator* ctordtor = ctor->getConstructorDeclarator();
    AstFormalParameterList*   params  = ctordtor->getFormalParameterList();
    // OK, this is ugly, but at least the syntax is interesting:
    if (AstFormalParameter* param = params ? getMatchingParameter(params, name) : NULL) {
        // DEBUG_JAVA_CODE(cout << "            returning constructor parameter " << (void*)param << endl);
        return param;
    }
    else {
        // DEBUG_JAVA_CODE(cout << "no matching ctor parameter found" << endl);
        return NULL;
    }
}

static AstFormalParameter* findAsMethodParameter(AstMethodDeclaration* const method, AstName* const name)
{
    AstMethodHeader*        header     = method->getMethodHeader();
    AstMethodDeclarator*    methoddtor = header->getMethodDeclarator();
    AstFormalParameterList* params     = methoddtor->getFormalParameterList();
    
    if (AstFormalParameter* param = params ? getMatchingParameter(params, name) : NULL) {
        // DEBUG_JAVA_CODE(cout << "            returning method parameter " << (void*)param << endl);
        return param;
    }
    else {
        // DEBUG_JAVA_CODE(cout << "no matching method parameter found" << endl);
        return NULL;
    }
}

static AstFieldDeclaration* findAsFieldInCurrentClass(AstClassDeclaration* const cd, AstName* const name)
{
    AstClassBody* classbody = cd->getClassBody();
    // DEBUG_JAVA_CODE(spew("class body", classbody));

    AstClassBodyDeclarationList* cbdl = classbody->getClassBodyDeclarationList();
    // DEBUG_JAVA_CODE(spew("class declaration list", cbdl));

    int nDecls = cbdl->getClassBodyDeclarationCount();
    for (int i = 0; i < nDecls; ++i) {
        AstItem* flditem = cbdl->getClassBodyDeclaration(i);
        // DEBUG_JAVA_CODE(spew("field item", flditem));

        if (AstFieldDeclaration* fielddec = dynamic_cast<AstFieldDeclaration*>(flditem)) {
            // DEBUG_JAVA_CODE(spew("field decl", fielddec, 1));

            AstVariableDeclaratorList* fldvars = fielddec->getVariableDeclaratorList();
            // DEBUG_JAVA_CODE(spew("field var list", fldvars, 2));

            AstItem* var = findMatchingVariable(name, fldvars);
            if (var) {
                return fielddec;
            }
            else {
                // DEBUG_JAVA_CODE(cout << "no matching variable in list" << endl);
            }
        }
        else {
            // DEBUG_JAVA_CODE(cout << "not a field declaration" << endl);
        }
    }

    // DEBUG_JAVA_CODE(cout << "not found as a field in this class" << endl);
    return NULL;
}

static AstFieldDeclaration* findAsFieldInSuperClass(AstClassDeclaration* const cd, AstName* const name)
{
    return NULL;
}

static AstFieldDeclaration* findAsFieldInOtherClass(AstClassDeclaration* const cd, AstName* const name)
{
    return NULL;
}

static AstFieldDeclaration* findAsField(AstClassDeclaration* const cd, AstName* const name)
{
    // Could be a field in this class, or in a superclass, or else it is
    // a field in another class or interface.

    if (AstFieldDeclaration* cfield = findAsFieldInCurrentClass(cd, name)) {
        return cfield;
    }
    else if (AstFieldDeclaration* scfield = findAsFieldInSuperClass(cd, name)) {
        return scfield;
    }
    else if (AstFieldDeclaration* ofield = findAsFieldInOtherClass(cd, name)) {
        return ofield;
    }
    else {
        // DEBUG_JAVA_CODE(cout << "not found as a field" << endl);
        return NULL;
    }
}

// -------------------------------------------------------
// interface

static AstFieldDeclaration* findAsFieldInCurrentInterface(AstInterfaceDeclaration* const id, AstName* const name)
{
    AstInterfaceBody* ibody = id->getInterfaceBody();
    // DEBUG_JAVA_CODE(spew("interface body", ibody));

    AstInterfaceMemberDeclarationList* imdl = ibody->getInterfaceMemberDeclarationList();
    // DEBUG_JAVA_CODE(spew("interface member declaration list", imdl));

    int nDecls = imdl->getInterfaceMemberDeclarationCount();
    for (int i = 0; i < nDecls; ++i) {
        AstItem* flditem = imdl->getInterfaceMemberDeclaration(i);
        // DEBUG_JAVA_CODE(spew("field item", flditem));

        if (AstFieldDeclaration* fielddec = dynamic_cast<AstFieldDeclaration*>(flditem)) {
            // DEBUG_JAVA_CODE(spew("field decl", fielddec, 1));

            AstVariableDeclaratorList* fldvars = fielddec->getVariableDeclaratorList();
            // DEBUG_JAVA_CODE(spew("field var list", fldvars, 2));

            AstItem* var = findMatchingVariable(name, fldvars);
            if (var) {
                return fielddec;
            }
            else {
                // DEBUG_JAVA_CODE(cout << "no matching variable in list" << endl);
            }
        }
        else {
            // DEBUG_JAVA_CODE(cout << "not a field declaration" << endl);
        }
    }

    // DEBUG_JAVA_CODE(cout << "not found as a field in this interface" << endl);
    return NULL;
}

static AstFieldDeclaration* findAsFieldInSuperInterface(AstInterfaceDeclaration* const id, AstName* const name)
{
    return NULL;
}

static AstFieldDeclaration* findAsFieldInOtherInterface(AstInterfaceDeclaration* const id, AstName* const name)
{
    return NULL;
}

static AstFieldDeclaration* findAsField(AstInterfaceDeclaration* const id, AstName* const name)
{
    // Could be a field in this interface, or in a superinterface, or else it is
    // a field in another interface or interface.

    if (AstFieldDeclaration* cfield = findAsFieldInCurrentInterface(id, name)) {
        return cfield;
    }
    else if (AstFieldDeclaration* scfield = findAsFieldInSuperInterface(id, name)) {
        return scfield;
    }
    else if (AstFieldDeclaration* ofield = findAsFieldInOtherInterface(id, name)) {
        return ofield;
    }
    else {
        // DEBUG_JAVA_CODE(cout << "not found as a field" << endl);
        return NULL;
    }
}

AstItem* JavaSymbol::findVariableDeclaration(AstName* const name)
{
    // DEBUG_JAVA_CODE(cout << "seeking variable declaration for " << name->type() << " (" << name->text() << ")" << endl);
    // this could be a:
    //  - local variable, which is in:
    //     - local variable declaration statement
    //     - for statement
    //  - formal parameter, which is in:
    //     - catch clause
    //     - constructor declarator
    //     - method declarator
    //  - field
    //     - current class
    //     - superclass
    //     - different class (if format "Other.field")
    
    AstItem* prev = name;       // so we know where we came from
    AstItem* context = name->parent();
    while (context) {
        // DEBUG_JAVA_CODE(cout << "    considering context " << context->type() << " (" << context->text() << ")" << endl);

        if (AstBlockStatementList* block = dynamic_cast<AstBlockStatementList*>(context)) {
            // DEBUG_JAVA_CODE(cout << "        in a block" << endl);
            if (AstLocalVariableDeclaration* var = findAsLocalVariable(block, prev, name)) {
                return var;
            }
            else {
                // DEBUG_JAVA_CODE(cout << "not a local variable match" << endl);
            }
        }
        else if (AstCatchClause* cc = dynamic_cast<AstCatchClause*>(context)) {
            // DEBUG_JAVA_CODE(cout << "        in a catch clause" << endl);
            if (AstFormalParameter* ccp = findAsCatchClauseParameter(cc, name)) {
                return ccp;
            }
            else {
                // DEBUG_JAVA_CODE(cout << "            not a matching catch clause parameter" << endl);
            }
        }
        else if (AstForStatement* fs = dynamic_cast<AstForStatement*>(context)) {
            // DEBUG_JAVA_CODE(cout << "        in a for statement" << endl);
            if (AstLocalVariableDeclaration* var = findAsForStatementVariable(fs, name)) {
                return var;
            }
            else {
                // DEBUG_JAVA_CODE(cout << "not a matching for statement variable" << endl);
            }
        }
        else if (AstConstructorDeclaration* ctor = dynamic_cast<AstConstructorDeclaration*>(context)) {
            // DEBUG_JAVA_CODE(cout << "        in constructor" << endl);
            if (AstFormalParameter* param = findAsConstructorParameter(ctor, name)) {
                // DEBUG_JAVA_CODE(cout << "            returning as constructor parameter " << (void*)param << endl);
                return param;
            }
            else {
                // DEBUG_JAVA_CODE(cout << "not a matching constructor parameter" << endl);
            }
        }
        else if (AstMethodDeclaration* method = dynamic_cast<AstMethodDeclaration*>(context)) {
            // DEBUG_JAVA_CODE(cout << "        in method" << endl);
            if (AstFormalParameter* param = findAsMethodParameter(method, name)) {
                return param;
            }
            else {
                // DEBUG_JAVA_CODE(cout << "not a matching method parameter" << endl);
            }
        }
        else if (AstClassDeclaration* cd = dynamic_cast<AstClassDeclaration*>(context)) {
            // DEBUG_JAVA_CODE(cout << "        in class declaration" << endl);
            if (AstFieldDeclaration* field = findAsField(cd, name)) {
                return field;
            }
            else {
                // DEBUG_JAVA_CODE(cout << "not a matching field" << endl);
            }
        }
        else if (AstInterfaceDeclaration* iface = dynamic_cast<AstInterfaceDeclaration*>(context)) {
            // Yes, we can be in an interface, if the variable refers to another
            // static field.
            // DEBUG_JAVA_CODE(cout << "        in interface" << endl);
            if (AstFieldDeclaration* field = findAsField(iface, name)) {
                return field;
            }
            else {
                // DEBUG_JAVA_CODE(cout << "not a matching field" << endl);
            }
        }
        else {
            // DEBUG_JAVA_CODE(cout << "        in something else" << endl);
        }
                
        prev = context;
        context = context->parent();
    }
    return NULL;
}

AstItem* JavaSymbol::getVariableType(AstName* const name)
{
    AstItem* decl = findVariableDeclaration(name);
    if (decl) {
        if (AstFieldDeclaration* field = dynamic_cast<AstFieldDeclaration*>(decl)) {
            return field->getType();
        }
        else if (AstFormalParameter* param = dynamic_cast<AstFormalParameter*>(decl)) {
            return param->getType();
        }
        else if (AstLocalVariableDeclaration* lvd = dynamic_cast<AstLocalVariableDeclaration*>(decl)) {
            return lvd->getType();
        }
        else {
            return NULL;
        }
    }
    else {
        return NULL;
    }

}


#define PROCESS_SYMBOL_TABLE(item) \
    symbolTable_ = new JavaSymbolTable(item, symbolTable_); \
    traverse(item); \
    symbolTable_ = symbolTable_->getParentTable();

JavaSymbolReader::JavaSymbolReader() : symbolTable_(NULL)
{
}

JavaSymbolReader::~JavaSymbolReader()
{
}

JavaSymbolTable* JavaSymbolReader::processCompilationUnit(AstCompilationUnit* const ac)
{
    // cout << "JSYMBRDR  ::processCompilationUnit(AstCompilationUnit*)" << endl;
    symbolTable_ = new JavaSymbolTable(ac);
    traverse(ac);
    // cout << "JSYMBRDR  done traversing. returning " << symbolTable_ << endl;
    return symbolTable_;
}

void JavaSymbolReader::process(AstAbstractMethodDeclaration* const aa)
{
    symbolTable_->addDeclaration(aa);
    traverse(aa);
}

void JavaSymbolReader::process(AstBlock* const ab)
{
    PROCESS_SYMBOL_TABLE(ab);
}

void JavaSymbolReader::process(AstBlockNoStatements* const ab)
{
    process((AstBlock*)ab);
}

void JavaSymbolReader::process(AstBlockWStatements* const ab)
{
    process((AstBlock*)ab);
}

void JavaSymbolReader::process(AstCatchClause* const ac)
{
    PROCESS_SYMBOL_TABLE(ac);
}

void JavaSymbolReader::process(AstClassDeclaration* const ac)
{
    PROCESS_SYMBOL_TABLE(ac);
}

void JavaSymbolReader::process(AstClassDeclarationModsBaseless* const ac)
{
    process((AstClassDeclaration*)ac);
}

void JavaSymbolReader::process(AstClassDeclarationModsExtends* const ac)
{
    process((AstClassDeclaration*)ac);
}

void JavaSymbolReader::process(AstClassDeclarationModsExtendsImplements* const ac)
{
    process((AstClassDeclaration*)ac);
}

void JavaSymbolReader::process(AstClassDeclarationModsImplements* const ac)
{
    process((AstClassDeclaration*)ac);
}

void JavaSymbolReader::process(AstClassDeclarationNoModsBaseless* const ac)
{
    process((AstClassDeclaration*)ac);
}

void JavaSymbolReader::process(AstClassDeclarationNoModsExtends* const ac)
{
    process((AstClassDeclaration*)ac);
}

void JavaSymbolReader::process(AstClassDeclarationNoModsExtendsImplements* const ac)
{
    process((AstClassDeclaration*)ac);
}

void JavaSymbolReader::process(AstClassDeclarationNoModsImplements* const ac)
{
    process((AstClassDeclaration*)ac);
}

void JavaSymbolReader::process(AstClassMemberDeclaration* const ac)
{
    symbolTable_->addDeclaration(ac);
    traverse(ac);
}

void JavaSymbolReader::process(AstClassMemberDeclarationModsExtendsImplements* const ac)
{
    process((AstClassMemberDeclaration*)ac);
}

void JavaSymbolReader::process(AstClassMemberDeclarationModsExtendsNoImplements* const ac)
{
    process((AstClassMemberDeclaration*)ac);
}

void JavaSymbolReader::process(AstClassMemberDeclarationModsNoExtendsImplements* const ac)
{
    process((AstClassMemberDeclaration*)ac);
}

void JavaSymbolReader::process(AstClassMemberDeclarationModsNoExtendsNoImplements* const ac)
{
    process((AstClassMemberDeclaration*)ac);
}

void JavaSymbolReader::process(AstClassMemberDeclarationNoModsExtendsImplements* const ac)
{
    process((AstClassMemberDeclaration*)ac);
}

void JavaSymbolReader::process(AstClassMemberDeclarationNoModsExtendsNoImplements* const ac)
{
    process((AstClassMemberDeclaration*)ac);
}

void JavaSymbolReader::process(AstClassMemberDeclarationNoModsNoExtendsImplements* const ac)
{
    process((AstClassMemberDeclaration*)ac);
}

void JavaSymbolReader::process(AstClassMemberDeclarationNoModsNoExtendsNoImplements* const ac)
{
    process((AstClassMemberDeclaration*)ac);
}

void JavaSymbolReader::process(AstConstructorDeclaration* const ac)
{
    symbolTable_->addDeclaration(ac);
    PROCESS_SYMBOL_TABLE(ac);
}

void JavaSymbolReader::process(AstConstructorDeclarationModsNoThrows* const ac)
{
    process((AstConstructorDeclaration*)ac);
}

void JavaSymbolReader::process(AstConstructorDeclarationModsThrows* const ac)
{
    process((AstConstructorDeclaration*)ac);
}

void JavaSymbolReader::process(AstConstructorDeclarationNoModsNoThrows* const ac)
{
    process((AstConstructorDeclaration*)ac);
}

void JavaSymbolReader::process(AstConstructorDeclarationNoModsThrows* const ac)
{
    process((AstConstructorDeclaration*)ac);
}

void JavaSymbolReader::process(AstFor* const af)
{
    PROCESS_SYMBOL_TABLE(af);
}

void JavaSymbolReader::process(AstForStatement* const af)
{
    PROCESS_SYMBOL_TABLE(af);
}

void JavaSymbolReader::process(AstForStatementNoInitConditionNoUpdate* const af)
{
    process((AstForStatement*)af);
}

void JavaSymbolReader::process(AstForStatementNoInitConditionUpdate* const af)
{
    process((AstForStatement*)af);
}

void JavaSymbolReader::process(AstForStatementNoInitNoConditionNoUpdate* const af)
{
    process((AstForStatement*)af);
}

void JavaSymbolReader::process(AstForStatementNoInitNoConditionUpdate* const af)
{
    process((AstForStatement*)af);
}

void JavaSymbolReader::process(AstForStatementStmtsConditionNoUpdate* const af)
{
    process((AstForStatement*)af);
}

void JavaSymbolReader::process(AstForStatementStmtsConditionUpdate* const af)
{
    process((AstForStatement*)af);
}

void JavaSymbolReader::process(AstForStatementStmtsNoConditionNoUpdate* const af)
{
    process((AstForStatement*)af);
}

void JavaSymbolReader::process(AstForStatementStmtsNoConditionUpdate* const af)
{
    process((AstForStatement*)af);
}

void JavaSymbolReader::process(AstForStatementVarsConditionNoUpdate* const af)
{
    process((AstForStatement*)af);
}

void JavaSymbolReader::process(AstForStatementVarsConditionUpdate* const af)
{
    process((AstForStatement*)af);
}

void JavaSymbolReader::process(AstForStatementVarsNoConditionNoUpdate* const af)
{
    process((AstForStatement*)af);
}

void JavaSymbolReader::process(AstForStatementVarsNoConditionUpdate* const af)
{
    process((AstForStatement*)af);
}

void JavaSymbolReader::process(AstFormalParameter* const af)
{
    symbolTable_->addDeclaration(af);
    traverse(af);
}

void JavaSymbolReader::process(AstFormalParameterFinal* const af)
{
    process((AstFormalParameter*)af);
}

void JavaSymbolReader::process(AstFormalParameterNonFinal* const af)
{
    process((AstFormalParameter*)af);
}

void JavaSymbolReader::process(AstInterfaceDeclaration* const ai)
{
    PROCESS_SYMBOL_TABLE(ai);
}

void JavaSymbolReader::process(AstInterfaceDeclarationModsExtends* const ai)
{
    PROCESS_SYMBOL_TABLE(ai);
}

void JavaSymbolReader::process(AstInterfaceDeclarationModsNoExtends* const ai)
{
    PROCESS_SYMBOL_TABLE(ai);
}

void JavaSymbolReader::process(AstInterfaceDeclarationNoModsExtends* const ai)
{
    PROCESS_SYMBOL_TABLE(ai);
}

void JavaSymbolReader::process(AstInterfaceDeclarationNoModsNoExtends* const ai)
{
    PROCESS_SYMBOL_TABLE(ai);
}

void JavaSymbolReader::process(AstLocalVariableDeclaration* const al)
{
    symbolTable_->addDeclaration(al);
    traverse(al);
}

void JavaSymbolReader::process(AstLocalVariableDeclarationFinal* const al)
{
    process((AstLocalVariableDeclaration*)al);
}

void JavaSymbolReader::process(AstLocalVariableDeclarationNonFinal* const al)
{
    process((AstLocalVariableDeclaration*)al);
}

void JavaSymbolReader::process(AstMethodDeclaration* const am)
{
    symbolTable_->addDeclaration(am);
    PROCESS_SYMBOL_TABLE(am);
}

void JavaSymbolReader::process(AstMethodDeclarationBlock* const am)
{
    process((AstMethodDeclaration*)am);
}

void JavaSymbolReader::process(AstMethodDeclarationSemicolon* const am)
{
    process((AstMethodDeclaration*)am);
}

void JavaSymbolReader::process(AstTryStatementCatches* const at)
{
    PROCESS_SYMBOL_TABLE(at);
}

void JavaSymbolReader::process(AstTryStatementCatchesFinally* const at)
{
    PROCESS_SYMBOL_TABLE(at);
}

void JavaSymbolReader::process(AstTryStatementFinally* const at)
{
    PROCESS_SYMBOL_TABLE(at);
}


static void indent(ostream& os, int depth)
{
     for (int i = 0; i < depth; ++i) {
         os << "    ";
     }
}

JavaSymbolTable::JavaSymbolTable(AstItem* const item, JavaSymbolTable* const parentTable)
        : item_(item), parentTable_(parentTable)
{
    // cout << "JSYMTABL  ::ctor(" << item->type() << ", " << parentTable << ")" << endl;
    if (parentTable_) {
        // cout << "adding to parentTable_ " << parentTable_ << endl;
        parentTable_->subTables_.push_back(this);
    }
    else {
        // cout << "no parentTable" << endl;
    }
}

JavaSymbolTable::~JavaSymbolTable()
{
    vector<JavaSymbolTable*>::iterator it   = subTables_.begin();
    vector<JavaSymbolTable*>::iterator stop = subTables_.end();
    while (it != stop) {
        JavaSymbolTable* st = *it;
        delete st;
        ++it;
    }
}

AstItem* JavaSymbolTable::getItem() const
{
    return item_;
}
        
vector<JavaSymbolTable*> JavaSymbolTable::getSubTables() const
{
    return subTables_;
}

JavaSymbolTable* JavaSymbolTable::getParentTable() const
{
    return parentTable_;
}

JavaSymbolTable* JavaSymbolTable::getChildTable(AstItem* const item) const
{
    vector<JavaSymbolTable*>::const_iterator it   = subTables_.begin();
    vector<JavaSymbolTable*>::const_iterator stop = subTables_.end();
    while (it != stop) {
        JavaSymbolTable* st = *it;
        if (st->getItem() == item) {
            return st;
        }
        ++it;
    }
    return NULL;
}

static void printItem(AstItem* const item, int nRefs, ostream& os)
{
    if (AstMethodDeclarator* md = dynamic_cast<AstMethodDeclarator*>(item)) {
        os << "method '" << md->text() << "' at " << item->line();
    }
    else if (AstLocalVariableDeclaration* vd = dynamic_cast<AstLocalVariableDeclaration*>(item)) {
        JavaVariable jv(vd);
        os << "variable(s) of type '" << jv.getType() << "': ";
        vector<string> names = jv.getNames();
        copy(names.begin(), names.end(), ostream_iterator<string>(os, " "));
        os << "[" << item->line() << "]";
    }
    else if (AstVariableDeclaratorId* vdid = dynamic_cast<AstVariableDeclaratorId*>(item)) {
        os << "variable '" << vdid->name();
    }
    else if (AstFormalParameter* fp = dynamic_cast<AstFormalParameter*>(item)) {
        JavaParameter jp(fp);
        os << "param '" << jp.getName() << "' of type '" << jp.getType() << "'";
    }
    else if (AstConstructorDeclaration* cd = dynamic_cast<AstConstructorDeclaration*>(item)) {
        AstConstructorDeclarator* decl = cd->getConstructorDeclarator();
        os << "constructor '" << decl->text() << "'";
    }
    else if (AstClassBody* cd = dynamic_cast<AstClassBody*>(item)) {
        os << "class body";
    }
    else if (AstConstructorDeclarator* decl = dynamic_cast<AstConstructorDeclarator*>(item)) {
        os << "constructor '" << decl->text() << "'";
    }
    else if (AstClassDeclaration* cd = dynamic_cast<AstClassDeclaration*>(item)) {
        AstIdentifier* id = cd->getIdentifier();
        os << "class '" << id->text() << "'";
    }
    else if (AstBlock* bl = dynamic_cast<AstBlock*>(item)) {
        os << "block";
    }
    else {
        os << "generic type " << item->type() << " [line " << item->line() << "]";
    }

    if (nRefs != -1) {
        os << "; #references: " << nRefs;
    }

    os << endl;
}

void JavaSymbolTable::printAll(ostream& os, int depth /* = 0 */) const
{
//     indent(os, depth);
//     cout << "-------------------------------------------------------" << endl;

    indent(os, depth);
    printItem(item_, -1, os);

    indent(os, depth + 1);
    cout << "types (" << types_.size() << ")" << endl;
    {
        map<AstItem*, int>::const_iterator it   = types_.begin();
        map<AstItem*, int>::const_iterator stop = types_.end();
        while (it != stop) {
            AstItem* item = it->first;
            int num = it->second;
            indent(os, depth + 2);
            printItem(item, num, os);
            ++it;
        }
    }

    indent(os, depth + 1);
    cout << "methods (" << methods_.size() << ")" << endl;
    {
        map<AstItem*, int>::const_iterator it   = methods_.begin();
        map<AstItem*, int>::const_iterator stop = methods_.end();
        while (it != stop) {
            AstItem* item = it->first;
            int num = it->second;
            indent(os, depth + 2);
            printItem(item, num, os);
            ++it;
        }
    }

    indent(os, depth + 1);
    cout << "variables (" << variables_.size() << ")" << endl;
    {
        map<AstItem*, int>::const_iterator it   = variables_.begin();
        map<AstItem*, int>::const_iterator stop = variables_.end();
        while (it != stop) {
            AstItem* item = it->first;
            int num = it->second;
            indent(os, depth + 2);
            printItem(item, num, os);
            ++it;
        }
    }

    indent(os, depth + 1);
    cout << "subtables (" << subTables_.size() << ")" << endl;
    {
        vector<JavaSymbolTable*>::const_iterator it   = subTables_.begin();
        vector<JavaSymbolTable*>::const_iterator stop = subTables_.end();
        while (it != stop) {
            JavaSymbolTable* st = *it;
            st->printAll(os, depth + 2);
            ++it;
        }
    }

    if (depth == 0) {
        cout << endl;
    }
}

void JavaSymbolTable::addDeclaration(AstItem* const item)
{
    cout << "JavaSymbolTable -- addDeclaration(" << item->text() << " (" << item->type() << "))" << endl;
    // references_[item] = 0;
}

void JavaSymbolTable::addVariableDeclaration(AstVariableDeclaratorId* const id)
{
    // eventually we'll have these split up by type (variables, methods, types).
    cout << "JavaSymbolTable -- adding variable declaration for " << id->text() << " (" << id->type() << ")" << endl;
    variables_[id] = 0;
}

void JavaSymbolTable::addMethodDeclaration(AstItem* const item)
{
    // eventually we'll have these split up by type (methods, methods, types).
    cout << "JavaSymbolTable -- adding method declaration for " << item->text() << " (" << item->type() << ")" << endl;
    methods_[item] = 0;
}

void JavaSymbolTable::addTypeDeclaration(AstItem* const item)
{
    cout << "JavaSymbolTable -- adding type declaration for " << item->text() << " (" << item->type() << ")" << endl;
    types_[item] = 0;
}


void JavaSymbolTable::addReference(AstClassMemberDeclaration* const cmd)
{
    // _[cmd] = 0;
}

void addReference(AstItem* const item)
{
    // Do an intelligent conversion of that item to whatever we've got in the
    // symbol table. That is, addReference("foo()") should add to the "foo"
    // method, not the foo type or field.

    cout << "JavaSymbolTable::addReference(" << item->type() << " [" << item->line() << "])" << endl;
}

vector<AstItem*> JavaSymbolTable::findUnreferenced() const
{
    vector<AstItem*> unref;
//     map<AstItem*, int>::const_iterator it   = references_.begin();
//     map<AstItem*, int>::const_iterator stop = references_.end();
//     while (it != stop) {
//         if (it->second == 0) {
//             unref.push_back(it->first);
//         }
//         ++it;
//     }
    return unref;
}

AstItem* JavaSymbolTable::lookup(AstName* const name)
{
    cout << "JavaSymbolTable::lookup(AstName* " << name->text() << ")" << endl;
    cout << "item = ";
    printItem(item_, -1, cout);

    cout << "all = ";
    printAll(cout);

    string nmstr = name->text();
    int dotPos = nmstr.rfind(".");
    if (dotPos != -1) {
        nmstr = nmstr.substr(dotPos + 1);
    }

    cout << "JavaSymbolTable::lookup -- name = '" << nmstr << "'" << endl;

    map<AstItem*, int>::iterator it   = variables_.begin();
    map<AstItem*, int>::iterator stop = variables_.end();
    while (it != stop) {
        AstItem* item = it->first;
        string istr = item->text();
        cout << "JavaSymbolTable::lookup -- comparing " << istr << " of type " << item->type() << " to " << nmstr << endl;
        if (item->text() == nmstr) {
            variables_[item] = it->second + 1;
            return item;
        }
        ++it;
    }
    
    return parentTable_ ? parentTable_->lookup(name) : NULL;    
}


JavaType::JavaType()
{
}

JavaType::~JavaType()
{
}

bool JavaType::hasLocalMethod(const string& name, const JavaArgumentTypeList& argTypes) const
{
    vector<JavaMethod> methods = getMethods();
    EACH(vector<JavaMethod>, methods, mit) {
        JavaMethod method = *mit;
        if (method.isMatch(name, argTypes)) {
            // cout << "got a match" << endl;
            return true;
        }
    }
    return false;
}


JavaVariable::JavaVariable(AstLocalVariableDeclaration* const lvd) : lvd_(lvd)
{
}
        
JavaVariable::~JavaVariable()
{
}

string JavaVariable::getType() const
{
    AstItem* type = lvd_->getType();
    return type->text();
}

bool JavaVariable::isFinal() const
{
    return lvd_->getFinal() != NULL;
}

vector<string> JavaVariable::getNames() const
{
    vector<string> names;
    AstVariableDeclaratorList* vars = lvd_->getVariableDeclaratorList();
    int count = vars->getVariableDeclaratorCount();
    for (int i = 0; i < count; ++i) {
        AstVariableDeclarator* vd = vars->getVariableDeclarator(i);
        AstVariableDeclaratorId* id = vd->getVariableDeclaratorId();
        string nm = id->name();
        names.push_back(nm);
    }
    return names;
}
