#ifndef AST_h
#define AST_h

#ifndef std_vector
#define std_vector
#include <vector>
#endif

#ifndef std_list
#define std_list
#include <list>
#endif

#ifndef std_string
#define std_string
#include <string>
#endif


namespace doctorj
{
    class AstAbstract;
    class AstAbstractMethodDeclaration;
    class AstAdditiveExpression;
    class AstAdditiveExpressionMinus;
    class AstAdditiveExpressionPlus;
    class AstAnd;
    class AstAndExpression;
    class AstAndand;
    class AstAndeq;
    class AstArrayAccess;
    class AstArrayAccessName;
    class AstArrayAccessPrimary;
    class AstArrayCreationExpression;
    class AstArrayCreationExpressionNameDimsInit;
    class AstArrayCreationExpressionNameExprDims;
    class AstArrayCreationExpressionNameExprNoDims;
    class AstArrayCreationExpressionPrimitiveDimsInit;
    class AstArrayCreationExpressionPrimitiveExprDims;
    class AstArrayCreationExpressionPrimitiveExprNoDims;
    class AstArrayInitializer;
    class AstArrayInitializerComma;
    class AstArrayInitializerEmpty;
    class AstArrayInitializerVars;
    class AstArrayInitializerVarsComma;
    class AstArrayNameType;
    class AstArrayPrimitiveType;
    class AstArrayType;
    class AstArrayTypeDotClass;
    class AstAssert;
    class AstAssertStatement;
    class AstAssertStatementOneExpression;
    class AstAssertStatementTwoExpressions;
    class AstAssignment;
    class AstAssignmentArray;
    class AstAssignmentField;
    class AstAssignmentName;
    class AstAssignmentOperator;
    class AstBlock;
    class AstBlockNoStatements;
    class AstBlockStatementList;
    class AstBlockWStatements;
    class AstBoolean;
    class AstBooleanLiteral;
    class AstBreak;
    class AstBreakStatement;
    class AstBreakStatementId;
    class AstBreakStatementNoId;
    class AstByte;
    class AstCComment;
    class AstCase;
    class AstCastExpression;
    class AstCastExpressionExpr;
    class AstCastExpressionNameDims;
    class AstCastExpressionPrimitiveDims;
    class AstCastExpressionPrimitiveNoDims;
    class AstCatch;
    class AstCatchClause;
    class AstCatchClauseList;
    class AstChar;
    class AstCharacterLiteral;
    class AstClass;
    class AstClassBody;
    class AstClassBodyDeclarationList;
    class AstClassBodyDecls;
    class AstClassBodyEmpty;
    class AstClassDeclaration;
    class AstClassDeclarationModsBaseless;
    class AstClassDeclarationModsExtends;
    class AstClassDeclarationModsExtendsImplements;
    class AstClassDeclarationModsImplements;
    class AstClassDeclarationNoModsBaseless;
    class AstClassDeclarationNoModsExtends;
    class AstClassDeclarationNoModsExtendsImplements;
    class AstClassDeclarationNoModsImplements;
    class AstClassInstanceCreationExpression;
    class AstClassInstanceCreationExpressionNameExprBody;
    class AstClassInstanceCreationExpressionNameExprNoBody;
    class AstClassInstanceCreationExpressionNameIdExprBody;
    class AstClassInstanceCreationExpressionNameIdExprNoBody;
    class AstClassInstanceCreationExpressionNameIdNoExprBody;
    class AstClassInstanceCreationExpressionNameIdNoExprNoBody;
    class AstClassInstanceCreationExpressionNameNoExprBody;
    class AstClassInstanceCreationExpressionNameNoExprNoBody;
    class AstClassInstanceCreationExpressionPrimaryExprBody;
    class AstClassInstanceCreationExpressionPrimaryExprNoBody;
    class AstClassInstanceCreationExpressionPrimaryNoExprBody;
    class AstClassInstanceCreationExpressionPrimaryNoExprNoBody;
    class AstClassMemberDeclaration;
    class AstClassMemberDeclarationModsExtendsImplements;
    class AstClassMemberDeclarationModsExtendsNoImplements;
    class AstClassMemberDeclarationModsNoExtendsImplements;
    class AstClassMemberDeclarationModsNoExtendsNoImplements;
    class AstClassMemberDeclarationNoModsExtendsImplements;
    class AstClassMemberDeclarationNoModsExtendsNoImplements;
    class AstClassMemberDeclarationNoModsNoExtendsImplements;
    class AstClassMemberDeclarationNoModsNoExtendsNoImplements;
    class AstColon;
    class AstComma;
    class AstComment;
    class AstComp;
    class AstCompilationUnit;
    class AstCompilationUnitEmpty;
    class AstCompilationUnitImp;
    class AstCompilationUnitImpTypes;
    class AstCompilationUnitPkg;
    class AstCompilationUnitPkgImp;
    class AstCompilationUnitPkgImpTypes;
    class AstCompilationUnitPkgTypes;
    class AstCompilationUnitTypes;
    class AstConditionalAndExpression;
    class AstConditionalArithExpression;
    class AstConditionalOrExpression;
    class AstConst;
    class AstConstructorBody;
    class AstConstructorBodyBlock;
    class AstConstructorBodyEmpty;
    class AstConstructorBodyExplicit;
    class AstConstructorBodyExplicitBlock;
    class AstConstructorDeclaration;
    class AstConstructorDeclarationModsNoThrows;
    class AstConstructorDeclarationModsThrows;
    class AstConstructorDeclarationNoModsNoThrows;
    class AstConstructorDeclarationNoModsThrows;
    class AstConstructorDeclarator;
    class AstConstructorDeclaratorEmpty;
    class AstConstructorDeclaratorParams;
    class AstContinue;
    class AstContinueStatement;
    class AstContinueStatementId;
    class AstContinueStatementNoId;
    class AstCppComment;
    class AstDefault;
    class AstDimension;
    class AstDimensionExpression;
    class AstDimensionExpressionList;
    class AstDimensionList;
    class AstDiv;
    class AstDiveq;
    class AstDo;
    class AstDoStatement;
    class AstDot;
    class AstDouble;
    class AstElse;
    class AstEq;
    class AstEqeq;
    class AstEqualityExpression;
    class AstEqualityExpressionEq;
    class AstEqualityExpressionNe;
    class AstExplicitConstructorInvocation;
    class AstExplicitConstructorInvocationPrimarySuper;
    class AstExplicitConstructorInvocationPrimarySuperExpressions;
    class AstExplicitConstructorInvocationPrimaryThis;
    class AstExplicitConstructorInvocationPrimaryThisExpressions;
    class AstExplicitConstructorInvocationSuper;
    class AstExplicitConstructorInvocationSuperExpressions;
    class AstExplicitConstructorInvocationThis;
    class AstExplicitConstructorInvocationThisExpressions;
    class AstExpressionList;
    class AstExpressionStatement;
    class AstExtends;
    class AstExtendsInterfaceList;
    class AstFalseLiteral;
    class AstFieldAccess;
    class AstFieldAccessNameSuper;
    class AstFieldAccessPrimary;
    class AstFieldAccessSuper;
    class AstFieldDeclaration;
    class AstFieldDeclarationMods;
    class AstFieldDeclarationNoMods;
    class AstFinal;
    class AstFinally;
    class AstFinallyBlock;
    class AstFloat;
    class AstFloatingPointLiteral;
    class AstFloatingPointType;
    class AstFor;
    class AstForStatement;
    class AstForStatementNoInitConditionNoUpdate;
    class AstForStatementNoInitConditionUpdate;
    class AstForStatementNoInitNoConditionNoUpdate;
    class AstForStatementNoInitNoConditionUpdate;
    class AstForStatementStmtsConditionNoUpdate;
    class AstForStatementStmtsConditionUpdate;
    class AstForStatementStmtsNoConditionNoUpdate;
    class AstForStatementStmtsNoConditionUpdate;
    class AstForStatementVarsConditionNoUpdate;
    class AstForStatementVarsConditionUpdate;
    class AstForStatementVarsNoConditionNoUpdate;
    class AstForStatementVarsNoConditionUpdate;
    class AstFormalParameter;
    class AstFormalParameterFinal;
    class AstFormalParameterList;
    class AstFormalParameterNonFinal;
    class AstGoto;
    class AstGt;
    class AstGteq;
    class AstIdentifier;
    class AstIf;
    class AstIfThenElseStatement;
    class AstIfThenStatement;
    class AstImplements;
    class AstImplementsInterfaceList;
    class AstImport;
    class AstImportDeclaration;
    class AstImportDeclarationList;
    class AstImportDeclarationOnDemand;
    class AstImportDeclarationSingle;
    class AstInstanceof;
    class AstInt;
    class AstIntegerLiteral;
    class AstIntegralType;
    class AstInterface;
    class AstInterfaceBody;
    class AstInterfaceBodyDecls;
    class AstInterfaceBodyNoDecls;
    class AstInterfaceDeclaration;
    class AstInterfaceDeclarationModsExtends;
    class AstInterfaceDeclarationModsNoExtends;
    class AstInterfaceDeclarationNoModsExtends;
    class AstInterfaceDeclarationNoModsNoExtends;
    class AstInterfaceMemberDeclarationList;
    class AstItem;
    class AstJavadocComment;
    class AstLabeledStatement;
    class AstLbrace;
    class AstLbracket;
    class AstLeaf;
    class AstLiteral;
    class AstLocalVariableDeclaration;
    class AstLocalVariableDeclarationFinal;
    class AstLocalVariableDeclarationNonFinal;
    class AstLocalVariableDeclarationStatement;
    class AstLong;
    class AstLparen;
    class AstLshift;
    class AstLshifteq;
    class AstLt;
    class AstLteq;
    class AstMethodDeclaration;
    class AstMethodDeclarationBlock;
    class AstMethodDeclarationSemicolon;
    class AstMethodDeclarator;
    class AstMethodDeclaratorNoArgs;
    class AstMethodDeclaratorWArgs;
    class AstMethodHeader;
    class AstMethodHeaderTypeModsNoThrows;
    class AstMethodHeaderTypeModsThrows;
    class AstMethodHeaderTypeNoModsNoThrows;
    class AstMethodHeaderTypeNoModsThrows;
    class AstMethodHeaderVoidModsNoThrows;
    class AstMethodHeaderVoidModsThrows;
    class AstMethodHeaderVoidNoModsNoThrows;
    class AstMethodHeaderVoidNoModsThrows;
    class AstMethodInvocation;
    class AstMethodInvocationNameArgs;
    class AstMethodInvocationNameNoArgs;
    class AstMethodInvocationNameSuperArgs;
    class AstMethodInvocationNameSuperNoArgs;
    class AstMethodInvocationPrimaryArgs;
    class AstMethodInvocationPrimaryNoArgs;
    class AstMethodInvocationSuperArgs;
    class AstMethodInvocationSuperNoArgs;
    class AstMinus;
    class AstMinuseq;
    class AstMinusminus;
    class AstMod;
    class AstModeq;
    class AstModifierList;
    class AstMult;
    class AstMulteq;
    class AstMultiplicativeExpression;
    class AstMultiplicativeExpressionDiv;
    class AstMultiplicativeExpressionMod;
    class AstMultiplicativeExpressionMult;
    class AstName;
    class AstNameDotClass;
    class AstNameDotThis;
    class AstNameList;
    class AstNative;
    class AstNew;
    class AstNode;
    class AstNoncode;
    class AstNot;
    class AstNoteq;
    class AstNullLiteral;
    class AstNumericType;
    class AstOr;
    class AstOrExpression;
    class AstOreq;
    class AstOror;
    class AstPackage;
    class AstPackageDeclaration;
    class AstPlus;
    class AstPluseq;
    class AstPlusplus;
    class AstPostdecrementExpression;
    class AstPostincrementExpression;
    class AstPredecrementExpression;
    class AstPreincrementExpression;
    class AstPrimaryParenExpression;
    class AstPrimitiveDotClass;
    class AstPrimitiveType;
    class AstPrivate;
    class AstProject;
    class AstProtected;
    class AstPublic;
    class AstQuestion;
    class AstRbrace;
    class AstRbracket;
    class AstRelationalExpression;
    class AstRelationalExpressionGt;
    class AstRelationalExpressionGteq;
    class AstRelationalExpressionInstanceof;
    class AstRelationalExpressionLt;
    class AstRelationalExpressionLteq;
    class AstReturn;
    class AstReturnStatement;
    class AstReturnStatementEmpty;
    class AstReturnStatementExpr;
    class AstRparen;
    class AstRshift;
    class AstRshifteq;
    class AstSemicolon;
    class AstShiftExpression;
    class AstShiftExpressionLshift;
    class AstShiftExpressionRshift;
    class AstShiftExpressionUrshift;
    class AstShort;
    class AstStatementExpressionList;
    class AstStatic;
    class AstStaticInitializer;
    class AstStrictfp;
    class AstStringLeaf;
    class AstStringLiteral;
    class AstSuper;
    class AstSwitch;
    class AstSwitchBlock;
    class AstSwitchBlockEmpty;
    class AstSwitchBlockGroups;
    class AstSwitchBlockGroupsLabels;
    class AstSwitchBlockLabels;
    class AstSwitchBlockStatementGroup;
    class AstSwitchBlockStatementGroupList;
    class AstSwitchLabel;
    class AstSwitchLabelCase;
    class AstSwitchLabelDefault;
    class AstSwitchLabelList;
    class AstSwitchStatement;
    class AstSynchronized;
    class AstSynchronizedStatement;
    class AstTaggedComment;
    class AstTaggedDescribedComment;
    class AstThis;
    class AstThrow;
    class AstThrowStatement;
    class AstThrows;
    class AstThrowsNameList;
    class AstTokenClass;
    class AstTransient;
    class AstTrueLiteral;
    class AstTry;
    class AstTryStatement;
    class AstTryStatementCatches;
    class AstTryStatementCatchesFinally;
    class AstTryStatementFinally;
    class AstTypeDeclarationList;
    class AstUnaryExpression;
    class AstUnaryExpressionComp;
    class AstUnaryExpressionMinus;
    class AstUnaryExpressionNot;
    class AstUnaryExpressionPlus;
    class AstUrshift;
    class AstUrshifteq;
    class AstValue;
    class AstVariableDeclarator;
    class AstVariableDeclaratorAssign;
    class AstVariableDeclaratorId;
    class AstVariableDeclaratorIdDimensions;
    class AstVariableDeclaratorIdNoDimensions;
    class AstVariableDeclaratorList;
    class AstVariableDeclaratorNoAssign;
    class AstVariableInitializerList;
    class AstVoid;
    class AstVoidDotClass;
    class AstVolatile;
    class AstWhile;
    class AstWhileStatement;
    class AstXor;
    class AstXorExpression;
    class AstXoreq;
}



using namespace std;

namespace doctorj
{
    class Processor;
    class File;
    
    /**
     * Ugh, having to refer to subclasses from the base class.
     */

    /**
     * A single object in the AST.
     */
    class AstItem
    {
    public:
        /**
         * An item from a file. Note that the <code>srcfile</code> object is
         * shared, and will not be deleted by an object (item/leaf/node) in the
         * AST. Note that if the srcfile is NULL (not provided) then this item
         * is not from a file, and therefore encapsulates items that span
         * multiple files. I hate to refer to subclasses from a superclass, but
         * I will: see <code>AstProject</code>.
         */
        AstItem(File* const srcfile);
    
        /**
         * Destructor. Deletes associated objects (items).
         */
        virtual ~AstItem();

        /**
         * This method must be defined <em>by every subclass</em>. 
         * The implementation should be <code>proc->process(this)</code>.
         */
        virtual void getProcessed(Processor* const proc) = 0;

        /**
         * Processes this item, then all of its components.
         */
        virtual void process(const std::vector<Processor*>& processors);

        /**
         * The previous sibling item of this item.
         */
        virtual AstItem* previous() const;

        /**
         * The next sibling item of this item.
         */
        virtual AstItem* next() const;

        /**
         * The parent of this item.
         */
        virtual AstNode* parent() const;
             
        /**
         * Sets the parent of this item.
         */
        virtual void setParent(AstNode* const pt);
        /**
         * The non code (whitespace, comments) preceding this item. See
         * <code>leaf</code>.
         */
        virtual AstNoncode* leadingNoncode() const = 0;

        /**
         * The non code (whitespace, comments) after this item. See
         * <code>leaf</code>.
         */
        virtual AstNoncode* trailingNoncode() const = 0;

        virtual char* position() const = 0;
        
        virtual File* sourceFile() const;

        virtual std::string type() const = 0;

        virtual int line() const;
    
        virtual int column() const;

        virtual char* endPosition() const = 0;

        virtual int endLine() const;

        virtual int endColumn() const;

        virtual std::string text() const;
        
    private:
        File* srcfile_;
        AstNode* parent_;
    };

    
    class AstLeaf : public AstItem 
    {
    public:
        /**
         * A leaf represents a single textual item, at the given position in the
         * file.
         */
        AstLeaf(char* position, File* const srcfile);

        /**
         * Destructor. Does not delete the reference to <code>srcfile</code>.
         */
        virtual ~AstLeaf();

        virtual char* position() const;

        /**
         * The noncode preceding this leaf.
         */
        virtual AstNoncode* leadingNoncode() const;
        
        /**
         * Sets the noncode preceding this leaf.
         */
        virtual void setLeadingNoncode(AstNoncode* const nc);

        /**
         * The noncode after this leaf.
         */
        virtual AstNoncode* trailingNoncode() const;

        virtual int length() const = 0;    

        virtual char* endPosition() const;

    private:
        char* position_;
        AstNoncode* leadingNoncode_;
    };


    /**
     * A node is the same thing as what it is in a tree (duh, because this is an
     * abstract syntax <em>tree</em>), an object with components.
     */
    class AstNode : public AstItem 
    {
    public:
        /**
         * For all constructors, see comments for <code>AstItem</code>.
         */
        AstNode(File* const srcfile = NULL);

        AstNode(AstItem* const it0,
                File* const srcfile = NULL);

        AstNode(AstItem* const it0, 
                AstItem* const it1,
                File* const srcfile = NULL);

        AstNode(AstItem* const it0, 
                AstItem* const it1,
                AstItem* const it2,
                File* const srcfile = NULL);

        AstNode(AstItem* const it0, 
                AstItem* const it1,
                AstItem* const it2,
                AstItem* const it3,
                File* const srcfile = NULL);

        AstNode(AstItem* const it0, 
                AstItem* const it1,
                AstItem* const it2,
                AstItem* const it3,
                AstItem* const it4,
                File* const srcfile = NULL);

        AstNode(AstItem* const it0, 
                AstItem* const it1,
                AstItem* const it2,
                AstItem* const it3,
                AstItem* const it4,
                AstItem* const it5,
                File* const srcfile = NULL);

        AstNode(AstItem* const it0, 
                AstItem* const it1,
                AstItem* const it2,
                AstItem* const it3,
                AstItem* const it4,
                AstItem* const it5,
                AstItem* const it6,
                File* const srcfile = NULL);

        AstNode(AstItem* const it0, 
                AstItem* const it1,
                AstItem* const it2,
                AstItem* const it3,
                AstItem* const it4,
                AstItem* const it5,
                AstItem* const it6,
                AstItem* const it7,
                File* const srcfile = NULL);

        AstNode(AstItem* const it0, 
                AstItem* const it1,
                AstItem* const it2,
                AstItem* const it3,
                AstItem* const it4,
                AstItem* const it5,
                AstItem* const it6,
                AstItem* const it7,
                AstItem* const it8,
                File* const srcfile = NULL);

        AstNode(AstItem* const it0, 
                AstItem* const it1,
                AstItem* const it2,
                AstItem* const it3,
                AstItem* const it4,
                AstItem* const it5,
                AstItem* const it6,
                AstItem* const it7,
                AstItem* const it8,
                AstItem* const it9,
                File* const srcfile = NULL);
    
        /**
         * Destructor. Deletes each component AstItem.
         */
        virtual ~AstNode();
    
        virtual AstItem* findPrevious(const AstItem* const it) const;
        
        virtual AstItem* findNext(const AstItem* const it) const;

        /**
         * The noncode preceding the first component AstItem.
         */
        virtual AstNoncode* leadingNoncode() const;
        
        /**
         * The noncode after the last component AstItem.
         */
        virtual AstNoncode* trailingNoncode() const;

        virtual char* position() const;

        /**
         * This method must be defined <em>by every subclass</em>. 
         * The implementation should be <code>proc->process(this)</code>.
         *
         * For an AstNode, this method calls getProcessed on each of its components,
         * but not <strong>this</strong> itself. Thus, this method should be
         * called by all subclasses, unless there is a compelling reason not to
         * be invoked for the component items.
         */
        virtual void getProcessed(Processor* const proc) = 0;

        virtual const list<AstItem*>& components() const;

        virtual list<AstItem*>& components();

        /**
         * Adds the given item to the list, after the other, or at the end of the
         * list.
         */
        virtual void append(AstItem* const item, AstItem* const prev = NULL);
        
        /**
         * Inserts the item, before <code>next</code>, or at the head of the list if
         * <code>next</code> is NULL.
         */
        virtual void prepend(AstItem* const item, AstItem* const next = NULL);

        list<AstItem*>::iterator findPosition(AstItem* const item);

        virtual char* endPosition() const;

    private:
        list<AstItem*> components_;
    };


    /**
     * "abstract"
     */
    class AstAbstract : public AstLeaf 
    {
    public:
        AstAbstract(char* position, File* const srcfile);

        virtual ~AstAbstract();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    class AstAbstractMethodDeclaration : public AstNode 
    {
    public:
        AstAbstractMethodDeclaration(AstMethodHeader* const mh, AstSemicolon* const s, File* const srcfile);

        virtual ~AstAbstractMethodDeclaration();

        virtual void getProcessed(Processor* const proc);

        AstMethodHeader* getMethodHeader() const;
        
        AstSemicolon* getSemicolon() const;

        virtual string type() const;
        
    private:
        AstMethodHeader* methodHeader_;
        AstSemicolon* semicolon_;
    };


    class AstAdditiveExpression : public AstNode 
    {
    public:
        AstAdditiveExpression(AstItem* const lhex, AstItem* const rhex, File* const srcfile);

        virtual ~AstAdditiveExpression();

        AstItem* getLhsExpression() const;

        AstItem* getRhsExpression() const;

    private:
        AstItem* lhsExpression_;
        AstItem* rhsExpression_;
    };


    class AstAdditiveExpressionMinus : public AstAdditiveExpression 
    {
    public:
        AstAdditiveExpressionMinus(AstItem* const lhex,
                                   AstMinus* const mi,
                                   AstItem* const rhex,
                                   File* const srcfile);

        virtual ~AstAdditiveExpressionMinus();

        AstMinus* getMinus() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstMinus* minus_;
    };


    class AstAdditiveExpressionPlus : public AstAdditiveExpression 
    {
    public:
        AstAdditiveExpressionPlus(AstItem* const lhex,
                                  AstPlus* const pl,
                                  AstItem* const rhex,
                                  File* const srcfile);

        virtual ~AstAdditiveExpressionPlus();

        AstPlus* getPlus() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstPlus* plus_;
    };


    /**
     * "&"
     */
    class AstAnd : public AstLeaf 
    {
    public:
        AstAnd(char* position, File* const srcfile);

        virtual ~AstAnd();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    class AstAndExpression : public AstNode 
    {
    public:
        AstAndExpression(AstItem* const le, 
                         AstAnd* const a,
                         AstItem* const re,
                         File* const srcfile);

        virtual ~AstAndExpression();

        virtual void getProcessed(Processor* const proc);

        AstItem* getLhsExpression() const;
        
        AstAnd* getAnd() const;

        AstItem* getRhsExpression() const;

        virtual string type() const;
        
    private:
        AstItem* lhsExpression_;
        AstAnd* and_;
        AstItem* rhsExpression_;
    };


    /**
     * "&&"
     */
    class AstAndand : public AstLeaf 
    {
    public:
        AstAndand(char* position, File* const srcfile);

        virtual ~AstAndand();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    class AstArrayAccess : public AstNode 
    {
    public:
        AstArrayAccess(AstDimensionExpression* const diex, File* const srcfile);

        virtual ~AstArrayAccess();

        AstDimensionExpression* getDimensionExpression() const;

    private:
        AstDimensionExpression* dimensionExpression_;
    };


    class AstArrayAccessName : public AstArrayAccess 
    {
    public:
        AstArrayAccessName(AstName* const na, AstDimensionExpression* const diex, File* const srcfile);

        virtual ~AstArrayAccessName();

        AstName* getName() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstName* name_;
    };


    class AstArrayAccessPrimary : public AstArrayAccess 
    {
    public:
        AstArrayAccessPrimary(AstItem* const pr, AstDimensionExpression* const diex, File* const srcfile);

        virtual ~AstArrayAccessPrimary();

        AstItem* getPrimary() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstItem* primary_;
    };


    class AstArrayCreationExpression : public AstNode 
    {
    public:
        AstArrayCreationExpression(AstNew* const ne, File* const srcfile);

        virtual ~AstArrayCreationExpression();

        AstNew* getNew() const;

    private:
        AstNew* new_;
    };


    class AstArrayCreationExpressionNameDimsInit : public AstArrayCreationExpression 
    {
    public:
        AstArrayCreationExpressionNameDimsInit(AstNew* const ne,
                                               AstName* const na,
                                               AstDimensionList* const dili,
                                               AstArrayInitializer* const arin,
                                               File* const srcfile);

        virtual ~AstArrayCreationExpressionNameDimsInit();

        AstName* getName() const;

        AstDimensionList* getDimensionList() const;

        AstArrayInitializer* getArrayInitializer() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstName* name_;
        AstDimensionList* dimensionList_;
        AstArrayInitializer* arrayInitializer_;
    };


    class AstArrayCreationExpressionNameExprDims : public AstArrayCreationExpression 
    {
    public:
        AstArrayCreationExpressionNameExprDims(AstNew* const ne,
                                               AstName* const na,
                                               AstDimensionExpressionList* const diexli,
                                               AstDimensionList* const dili,
                                               File* const srcfile);

        virtual ~AstArrayCreationExpressionNameExprDims();

        AstName* getName() const;

        AstDimensionExpressionList* getDimensionExpressionList() const;

        AstDimensionList* getDimensionList() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstName* name_;
        AstDimensionExpressionList* dimensionExpressionList_;
        AstDimensionList* dimensionList_;
    };


    class AstArrayCreationExpressionNameExprNoDims : public AstArrayCreationExpression 
    {
    public:
        AstArrayCreationExpressionNameExprNoDims(AstNew* const ne,
                                                 AstName* const na,
                                                 AstDimensionExpressionList* const diexli,
                                                 File* const srcfile);

        virtual ~AstArrayCreationExpressionNameExprNoDims();

        AstName* getName() const;

        AstDimensionExpressionList* getDimensionExpressionList() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstName* name_;
        AstDimensionExpressionList* dimensionExpressionList_;
    };


    class AstArrayCreationExpressionPrimitiveDimsInit : public AstArrayCreationExpression 
    {
    public:
        AstArrayCreationExpressionPrimitiveDimsInit(AstNew* const ne,
                                                    AstPrimitiveType* const prty,
                                                    AstDimensionList* const dili,
                                                    AstArrayInitializer* const arin,
                                                    File* const srcfile);

        virtual ~AstArrayCreationExpressionPrimitiveDimsInit();

        AstPrimitiveType* getPrimitiveType() const;

        AstDimensionList* getDimensionList() const;

        AstArrayInitializer* getArrayInitializer() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstPrimitiveType* primitiveType_;
        AstDimensionList* dimensionList_;
        AstArrayInitializer* arrayInitializer_;
    };


    class AstArrayCreationExpressionPrimitiveExprDims : public AstArrayCreationExpression 
    {
    public:
        AstArrayCreationExpressionPrimitiveExprDims(AstNew* const ne,
                                                    AstPrimitiveType* const prty,
                                                    AstDimensionExpressionList* const diexli,
                                                    AstDimensionList* const dili,
                                                    File* const srcfile);

        virtual ~AstArrayCreationExpressionPrimitiveExprDims();

        AstPrimitiveType* getPrimitiveType() const;

        AstDimensionExpressionList* getDimensionExpressionList() const;

        AstDimensionList* getDimensionList() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstPrimitiveType* primitiveType_;
        AstDimensionExpressionList* dimensionExpressionList_;
        AstDimensionList* dimensionList_;
    };


    class AstArrayCreationExpressionPrimitiveExprNoDims : public AstArrayCreationExpression 
    {
    public:
        AstArrayCreationExpressionPrimitiveExprNoDims(AstNew* const ne,
                                                      AstPrimitiveType* const prty,
                                                      AstDimensionExpressionList* const diexli,
                                                      File* const srcfile);

        virtual ~AstArrayCreationExpressionPrimitiveExprNoDims();

        AstPrimitiveType* getPrimitiveType() const;

        AstDimensionExpressionList* getDimensionExpressionList() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstPrimitiveType* primitiveType_;
        AstDimensionExpressionList* dimensionExpressionList_;
    };

    
    /**
     * array_initializer:
     *           LBRACE variable_initializer_list COMMA RBRACE
     *         | LBRACE variable_initializer_list RBRACE
     *         | LBRACE COMMA RBRACE
     *         | LBRACE RBRACE
     *         ;
     */
    class AstArrayInitializer : public AstNode 
    {
    public:
        virtual ~AstArrayInitializer();

        AstLbrace* getLbrace() const;
        
        AstVariableInitializerList* getVariableInitializerList() const;

        AstComma* getComma() const;

        AstRbrace* getRbrace() const;

    protected:
        AstArrayInitializer(AstLbrace* const l, 
                            AstVariableInitializerList* const vil,
                            AstComma* const c,
                            AstRbrace* const r,
                            File* const srcfile);
        
    private:
        AstLbrace* lbrace_;
        AstVariableInitializerList* variableInitializerList_;
        AstComma* comma_;
        AstRbrace* rbrace_;
    };
    

    class AstArrayInitializerComma : public AstArrayInitializer 
    {
    public:
        AstArrayInitializerComma(AstLbrace* const l, 
                                 AstComma* const c,
                                 AstRbrace* const r,
                                 File* const srcfile);

        virtual ~AstArrayInitializerComma();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };

    
    class AstArrayInitializerEmpty : public AstArrayInitializer 
    {
    public:
        AstArrayInitializerEmpty(AstLbrace* const l, AstRbrace* const r, File* const srcfile);

        virtual ~AstArrayInitializerEmpty();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };
    

    class AstArrayInitializerVars : public AstArrayInitializer 
    {
    public:
        AstArrayInitializerVars(AstLbrace* const l, 
                                AstVariableInitializerList* const vil,
                                AstRbrace* const r,
                                File* const srcfile);

        virtual ~AstArrayInitializerVars();

        virtual void getProcessed(Processor* const proc);
        

        virtual string type() const;
    };

    
    class AstArrayInitializerVarsComma : public AstArrayInitializer 
    {        
    public:
        AstArrayInitializerVarsComma(AstLbrace* const l, 
                                     AstVariableInitializerList* const vil,
                                     AstComma* const c,
                                     AstRbrace* const r,
                                     File* const srcfile);

        virtual ~AstArrayInitializerVarsComma();

        virtual void getProcessed(Processor* const proc);
        
        virtual string type() const;
    };
    

    class AstArrayType : public AstNode 
    {
    public:
        AstArrayType(AstItem* const type, AstDimensionList* const dl, File* const srcfile);

        virtual ~AstArrayType();

        virtual void getProcessed(Processor* const proc);

        AstItem* getType() const;
        
        AstDimensionList* getDimensionList() const;

        virtual string type() const;
        
    private:
        AstItem* type_;
        AstDimensionList* dimensionList_;
    };

    
    class AstArrayNameType : public AstArrayType 
    {
    public:
        AstArrayNameType(AstName* const n, AstDimensionList* const dl, File* const srcfile);

        virtual ~AstArrayNameType();

        virtual void getProcessed(Processor* const proc);

        AstName* getName() const;

        virtual string type() const;
        
    private:
        AstName* name_;
    };
    

    class AstArrayPrimitiveType : public AstArrayType 
    {
    public:
        AstArrayPrimitiveType(AstPrimitiveType* const pt, 
                              AstDimensionList* const dl,
                              File* const srcfile);

        virtual ~AstArrayPrimitiveType();

        virtual void getProcessed(Processor* const proc);

        AstPrimitiveType* getPrimitiveType() const;

        virtual string type() const;
        
    private:
        AstPrimitiveType* primitiveType_;
    };
    

    class AstArrayTypeDotClass : public AstNode 
    {
    public:
        AstArrayTypeDotClass(AstArrayType* const at, 
                             AstDot* const d,
                             AstClass* const c,
                             File* const srcfile);

        virtual ~AstArrayTypeDotClass();

        virtual void getProcessed(Processor* const proc);

        AstArrayType* getArrayType() const;
        
        AstDot* getDot() const;

        AstClass* getClass() const;

        virtual string type() const;
        
    private:
        AstArrayType* arrayType_;
        AstDot* dot_;
        AstClass* class_;
    };


    /**
     * A token without an associated value, whose type is ASSERT.
     */
    class AstAssert : public AstLeaf 
    {
    public:
        AstAssert(char* position, File* const srcfile);

        virtual ~AstAssert();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * assert_statement:
     *           ASSERT expression SEMICOLON
     *         | ASSERT expression COLON expression SEMICOLON
     *         ;
     */
    class AstAssertStatement : public AstNode 
    {
    public:
        /**
         * Destructor deletes each child item.
         */
        virtual ~AstAssertStatement();

        AstAssert* getAssert() const;

        AstItem* getExpressionOne() const;

        AstColon* getColon() const;

        AstItem* getExpressionTwo() const;

        AstSemicolon* getSemicolon() const;

    protected:
        AstAssertStatement(AstAssert* const as,
                           AstItem* const exon,
                           AstColon* const co,
                           AstItem* const extw,
                           AstSemicolon* const se,
                           File* const srcfile);

    private:
        AstAssert* assert_;
        AstItem* expressionOne_;
        AstColon* colon_;
        AstItem* expressionTwo_;
        AstSemicolon* semicolon_;
    };


    class AstAssertStatementOneExpression : public AstAssertStatement 
    {
    public:
        AstAssertStatementOneExpression(AstAssert* const as,
                                        AstItem* const ex,
                                        AstSemicolon* const se,
                                        File* const srcfile);

        /**
         * Destructor deletes each child item.
         */
        virtual ~AstAssertStatementOneExpression();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstAssertStatementTwoExpressions : public AstAssertStatement 
    {
    public:
        AstAssertStatementTwoExpressions(AstAssert* const as,
                                         AstItem* const exon,
                                         AstColon* const co,
                                         AstItem* const extw,
                                         AstSemicolon* const se,
                                         File* const srcfile);

        /**
         * Destructor deletes each child item.
         */
        virtual ~AstAssertStatementTwoExpressions();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstAssignment : public AstNode 
    {
    public:
        AstAssignment(AstAssignmentOperator* const asop, AstItem* const ex, File* const srcfile);

        virtual ~AstAssignment();

        AstAssignmentOperator* getAssignmentOperator() const;

        AstItem* getExpression() const;

    private:
        AstAssignmentOperator* assignmentOperator_;
        AstItem* expression_;
    };


    class AstAssignmentArray : public AstAssignment 
    {
    public:
        AstAssignmentArray(AstArrayAccess* const arac,
                           AstAssignmentOperator* const asop,
                           AstItem* const ex,
                           File* const srcfile);

        virtual ~AstAssignmentArray();

        AstArrayAccess* getArrayAccess() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstArrayAccess* arrayAccess_;
    };


    class AstAssignmentField : public AstAssignment 
    {
    public:
        AstAssignmentField(AstFieldAccess* const fiac,
                           AstAssignmentOperator* const asop,
                           AstItem* const ex,
                           File* const srcfile);

        virtual ~AstAssignmentField();

        AstFieldAccess* getFieldAccess() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstFieldAccess* fieldAccess_;
    };


    class AstAssignmentName : public AstAssignment 
    {
    public:
        AstAssignmentName(AstName* const na,
                          AstAssignmentOperator* const asop,
                          AstItem* const ex,
                          File* const srcfile);

        virtual ~AstAssignmentName();

        AstName* getName() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstName* name_;
    };


    class AstAssignmentOperator : public AstLeaf 
    {
    public:
        /**
         * A leaf represents a single textual item, at the given line and at the
         * given positions in the file.
         */
        AstAssignmentOperator(char* position, File* const srcfile);

        /**
         * Destructor. Does not delete the reference to <code>srcfile</code>.
         */
        virtual ~AstAssignmentOperator();
    };


    /**
     * "&="
     */
    class AstAndeq : public AstAssignmentOperator 
    {
    public:
        AstAndeq(char* position, File* const srcfile);

        virtual ~AstAndeq();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    /**
     * block:
     *           LBRACE block_statement_list RBRACE
     *         | LBRACE RBRACE
     *         ;
     */
    class AstBlock : public AstNode 
    {
    public:
        virtual ~AstBlock();

        AstLbrace* getLbrace() const;
        
        AstBlockStatementList* getBlockStatementList() const;
        
        AstRbrace* getRbrace() const;

    protected:
        AstBlock(AstLbrace* const l, AstBlockStatementList* const bsl, AstRbrace* const r, File* const srcfile);
        
    private:
        AstLbrace* lbrace_;
        AstBlockStatementList* blockStatementList_;
        AstRbrace* rbrace_;
    };

    
    class AstBlockNoStatements : public AstBlock 
    {
    public:
        AstBlockNoStatements(AstLbrace* const l, AstRbrace* const r, File* const srcfile);
        
        virtual ~AstBlockNoStatements();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };

    
    class AstBlockStatementList : public AstNode 
    {
    public:
        AstBlockStatementList(AstItem* const bs, File* const srcfile);

        virtual ~AstBlockStatementList();

        virtual void getProcessed(Processor* const proc);

        AstItem* getBlockStatement(int index) const;

        int getBlockStatementCount() const;

        virtual string type() const;

        virtual void add(AstItem* const bs);
        
    private:
        vector<AstItem*> blockStatements_;
    };

    
    class AstBlockWStatements : public AstBlock 
    {
    public:
        AstBlockWStatements(AstLbrace* const l, 
                            AstBlockStatementList* const bsl,
                            AstRbrace* const r,
                            File* const srcfile);

        virtual ~AstBlockWStatements();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;    
    };


    class AstPrimitiveType : public AstLeaf 
    {
    public:
        AstPrimitiveType(char* position, File* const srcfile);

        virtual ~AstPrimitiveType();
    };


    /**
     * "boolean"
     */
    class AstBoolean : public AstPrimitiveType 
    {
    public:
        AstBoolean(char* position, File* const srcfile);

        virtual ~AstBoolean();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    class AstValue : public AstLeaf 
    {
    public:
        AstValue(const string& text, char* position, File* const srcfile);

        virtual ~AstValue();

        virtual string text() const;

        virtual int length() const;

    private:
        string text_;
    };


    class AstLiteral : public AstValue
    {
    public:
        AstLiteral(const string& text, char* position, File* const srcfile);

        virtual ~AstLiteral();
    };


    /**
     * A value token, whose type is BOOLEAN_LITERAL.
     */
    class AstBooleanLiteral : public AstLiteral
    {
    public:
        AstBooleanLiteral(const string& text, char* position, File* const srcfile);

        virtual ~AstBooleanLiteral();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);
    };


    /**
     * "break"
     */
    class AstBreak : public AstLeaf 
    {
    public:
        AstBreak(char* position, File* const srcfile);

        virtual ~AstBreak();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    /**
     * break_statement:
     *           BREAK SEMICOLON
     *         | BREAK IDENTIFIER SEMICOLON
     *         ;
     */
    class AstBreakStatement : public AstNode 
    {
    public:
        virtual ~AstBreakStatement();

        virtual void getProcessed(Processor* const proc);

        AstBreak* getBreak() const;

        AstIdentifier* getIdentifier() const;
        
        AstSemicolon* getSemicolon() const;

        virtual string type() const;

    protected:
        AstBreakStatement(AstBreak* const b, 
                          AstIdentifier* const i,
                          AstSemicolon* const s,
                          File* const srcfile);
        
    private:
        AstBreak* break_;
        AstIdentifier* identifier_;
        AstSemicolon* semicolon_;
    };

    
    class AstBreakStatementId : public AstBreakStatement 
    {
    public:
        AstBreakStatementId(AstBreak* const b, 
                            AstIdentifier* const i,
                            AstSemicolon* const s,
                            File* const srcfile);

        virtual ~AstBreakStatementId();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };

    
    class AstBreakStatementNoId : public AstBreakStatement 
    {
    public:
        AstBreakStatementNoId(AstBreak* const b, AstSemicolon* const s, File* const srcfile);

        virtual ~AstBreakStatementNoId();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstNumericType : public AstPrimitiveType
    {
    public:
        AstNumericType(char* position, File* const srcfile);

        virtual ~AstNumericType();
    };


    class AstIntegralType : public AstNumericType 
    {
    public:
        AstIntegralType(char* position, File* const srcfile);

        virtual ~AstIntegralType();
    };


    /**
     * "byte"
     */
    class AstByte : public AstIntegralType 
    {
    public:
        AstByte(char* position, File* const srcfile);

        virtual ~AstByte();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    class AstComment : public AstNode 
    {
    public:
        AstComment(char* start, char* end, File* const srcfile);

        virtual ~AstComment();

        virtual string fulltext() const;

        virtual string type() const = 0;

        virtual char* position() const;

        virtual char* endPosition() const;

    protected:
        char* start_;

        char* end_;    };


    class AstCComment : public AstComment 
    {
    public:
        AstCComment(char* start, char* end, File* const srcfile);

        virtual ~AstCComment();

        virtual string type() const;

        /**
         * Invokes process on the Processor for this specific type of
         * AstNode/AstItem/AstLeaf.
         */
        virtual void getProcessed(Processor* const p);
    };


    /**
     * "case"
     */
    class AstCase : public AstLeaf 
    {
    public:
        AstCase(char* position, File* const srcfile);

        virtual ~AstCase();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    class AstCastExpression : public AstNode 
    {
    public:
        AstCastExpression(AstLparen* const lp, AstRparen* const rp, AstItem* const unex, File* const srcfile);

        virtual ~AstCastExpression();

        AstLparen* getLparen() const;

        AstRparen* getRparen() const;

        AstItem* getUnaryExpression() const;

    private:
        AstLparen* lparen_;
        AstRparen* rparen_;
        AstItem* unaryExpression_;
    };


    class AstCastExpressionExpr : public AstCastExpression 
    {
    public:
        AstCastExpressionExpr(AstLparen* const lp,
                              AstItem* const ex,
                              AstRparen* const rp,
                              AstItem* const unex,
                              File* const srcfile);

        virtual ~AstCastExpressionExpr();

        AstItem* getExpression() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstItem* expression_;
    };


    class AstCastExpressionNameDims : public AstCastExpression 
    {
    public:
        AstCastExpressionNameDims(AstLparen* const lp,
                                  AstName* const na,
                                  AstDimensionList* const dili,
                                  AstRparen* const rp,
                                  AstItem* const unex,
                                  File* const srcfile);

        virtual ~AstCastExpressionNameDims();

        AstName* getName() const;

        AstDimensionList* getDimensionList() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstName* name_;
        AstDimensionList* dimensionList_;
    };


    class AstCastExpressionPrimitiveDims : public AstCastExpression 
    {
    public:
        AstCastExpressionPrimitiveDims(AstLparen* const lp,
                                       AstPrimitiveType* const prty,
                                       AstDimensionList* const dili,
                                       AstRparen* const rp,
                                       AstItem* const unex,
                                       File* const srcfile);

        virtual ~AstCastExpressionPrimitiveDims();

        AstPrimitiveType* getPrimitiveType() const;

        AstDimensionList* getDimensionList() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstPrimitiveType* primitiveType_;
        AstDimensionList* dimensionList_;
    };


    class AstCastExpressionPrimitiveNoDims : public AstCastExpression 
    {
    public:
        AstCastExpressionPrimitiveNoDims(AstLparen* const lp,
                                         AstPrimitiveType* const prty,
                                         AstRparen* const rp,
                                         AstItem* const unex,
                                         File* const srcfile);

        virtual ~AstCastExpressionPrimitiveNoDims();

        AstPrimitiveType* getPrimitiveType() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstPrimitiveType* primitiveType_;
    };


    /**
     * "catch"
     */
    class AstCatch : public AstLeaf 
    {
    public:
        AstCatch(char* position, File* const srcfile);

        virtual ~AstCatch();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    class AstCatchClause : public AstNode 
    {        
    public:
        AstCatchClause(AstCatch* const c, 
                       AstLparen* const l,
                       AstFormalParameter* const fp,
                       AstRparen* const r,
                       AstBlock* const b,
                       File* const srcfile);
        
        virtual ~AstCatchClause();

        virtual void getProcessed(Processor* const proc);

        AstCatch* getCatch() const;
        
        AstLparen* getLparen() const;

        AstFormalParameter* getFormalParameter() const;

        AstRparen* getRparen() const;

        AstBlock* getBlock() const;

        virtual string type() const;
        
    private:
        AstCatch* catch_;
        AstLparen* lparen_;
        AstFormalParameter* formalParameter_;
        AstRparen* rparen_;
        AstBlock* block_;
    };

    
    class AstCatchClauseList : public AstNode 
    {
    public:
        AstCatchClauseList(AstCatchClause* const cc, File* const srcfile);

        virtual ~AstCatchClauseList();

        virtual void getProcessed(Processor* const proc);

        AstCatchClause* getCatchClause(int index) const;

        int getCatchClauseCount() const;

        virtual string type() const;

        virtual void add(AstCatchClause* const cc);
        
    private:
        vector<AstCatchClause*> catchClauses_;
    };


    /**
     * "char"
     */
    class AstChar : public AstIntegralType 
    {
    public:
        AstChar(char* position, File* const srcfile);

        virtual ~AstChar();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * A value token, whose type is CHARACTER_LITERAL.
     */
    class AstCharacterLiteral : public AstLiteral
    {
    public:
        AstCharacterLiteral(const string& text, char* position, File* const srcfile);

        virtual ~AstCharacterLiteral();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);
    };


    /**
     * "class"
     */
    class AstClass : public AstLeaf 
    {
    public:
        AstClass(char* position, File* const srcfile);

        virtual ~AstClass();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * class_body:
     *           LBRACE class_body_declaration_list RBRACE
     *         | LBRACE RBRACE
     *         ;
     */
    class AstClassBody : public AstNode 
    {
    public:
        virtual ~AstClassBody();

        AstLbrace* getLbrace() const;

        AstClassBodyDeclarationList* getClassBodyDeclarationList() const;

        AstRbrace* getRbrace() const;

    protected:
        AstClassBody(AstLbrace* const lb,
                     AstClassBodyDeclarationList* const clbodeli,
                     AstRbrace* const rb,
                     File* const srcfile);

    private:
        AstLbrace* lbrace_;
        AstClassBodyDeclarationList* classBodyDeclarationList_;
        AstRbrace* rbrace_;
    };

    
    class AstClassBodyDeclarationList : public AstNode 
    {
    public:
        AstClassBodyDeclarationList(AstItem* const cbd, File* const srcfile);

        virtual ~AstClassBodyDeclarationList();

        virtual void getProcessed(Processor* const proc);

        AstItem* getClassBodyDeclaration(int index) const;

        int getClassBodyDeclarationCount() const;

        virtual void add(AstItem* const cbd);

        virtual string type() const;
        
    private:
        vector<AstItem*> classBodyDeclarations_;
    };


    class AstClassBodyDecls : public AstClassBody 
    {
    public:
        AstClassBodyDecls(AstLbrace* const lb,
                          AstClassBodyDeclarationList* const clbodeli,
                          AstRbrace* const rb,
                          File* const srcfile);

        virtual ~AstClassBodyDecls();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstClassBodyEmpty : public AstClassBody 
    {
    public:
        AstClassBodyEmpty(AstLbrace* const lb, AstRbrace* const rb, File* const srcfile);

        virtual ~AstClassBodyEmpty();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };

    
    /**
     * class_declaration:
     *           CLASS IDENTIFIER class_body
     *         | CLASS IDENTIFIER implements_interface_list class_body
     *         | CLASS IDENTIFIER EXTENDS name class_body
     *         | CLASS IDENTIFIER EXTENDS name implements_interface_list class_body
     *         | modifier_list CLASS IDENTIFIER class_body
     *         | modifier_list CLASS IDENTIFIER implements_interface_list class_body
     *         | modifier_list CLASS IDENTIFIER EXTENDS name class_body
     *         | modifier_list CLASS IDENTIFIER EXTENDS name implements_interface_list class_body
     *         ;
     */
    class AstClassDeclaration : public AstNode 
    {
    public:
        virtual ~AstClassDeclaration();

        AstModifierList* getModifierList() const;

        AstClass* getClass() const;

        AstIdentifier* getIdentifier() const;

        AstExtends* getExtends() const;

        AstName* getName() const;

        AstImplementsInterfaceList* getImplementsInterfaceList() const;

        AstClassBody* getClassBody() const;

    protected:
        AstClassDeclaration(AstClass* const cl,
                            AstIdentifier* const id,
                            AstClassBody* const clbo,
                            File* const srcfile);
        
        AstClassDeclaration(AstModifierList* const moli,
                            AstClass* const cl,
                            AstIdentifier* const id,
                            AstExtends* const ex,
                            AstName* const na,
                            AstImplementsInterfaceList* const iminli,
                            AstClassBody* const clbo,
                            File* const srcfile);
        
    private:
        AstModifierList* modifierList_;
        AstClass* class_;
        AstIdentifier* identifier_;
        AstExtends* extends_;
        AstName* name_;
        AstImplementsInterfaceList* implementsInterfaceList_;
        AstClassBody* classBody_;
    };


    class AstClassDeclarationModsBaseless : public AstClassDeclaration 
    {
    public:
        AstClassDeclarationModsBaseless(AstModifierList* const moli,
                                        AstClass* const cl,
                                        AstIdentifier* const id,
                                        AstClassBody* const clbo,
                                        File* const srcfile);

        virtual ~AstClassDeclarationModsBaseless();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstClassDeclarationModsExtends : public AstClassDeclaration 
    {
    public:
        AstClassDeclarationModsExtends(AstModifierList* const moli,
                                       AstClass* const cl,
                                       AstIdentifier* const id,
                                       AstExtends* const ex,
                                       AstName* const na,
                                       AstClassBody* const clbo,
                                       File* const srcfile);

        virtual ~AstClassDeclarationModsExtends();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstClassDeclarationModsExtendsImplements : public AstClassDeclaration 
    {
    public:
        AstClassDeclarationModsExtendsImplements(AstModifierList* const moli,
                                                 AstClass* const cl,
                                                 AstIdentifier* const id,
                                                 AstExtends* const ex,
                                                 AstName* const na,
                                                 AstImplementsInterfaceList* const iminli,
                                                 AstClassBody* const clbo,
                                                 File* const srcfile);

        virtual ~AstClassDeclarationModsExtendsImplements();
        
        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstClassDeclarationModsImplements : public AstClassDeclaration 
    {
    public:
        AstClassDeclarationModsImplements(AstModifierList* const moli,
                                          AstClass* const cl,
                                          AstIdentifier* const id,
                                          AstImplementsInterfaceList* const iminli,
                                          AstClassBody* const clbo,
                                          File* const srcfile);

        virtual ~AstClassDeclarationModsImplements();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstClassDeclarationNoModsBaseless : public AstClassDeclaration 
    {
    public:
        AstClassDeclarationNoModsBaseless(AstClass* const cl,
                                          AstIdentifier* const id,
                                          AstClassBody* const clbo,
                                          File* const srcfile);

        virtual ~AstClassDeclarationNoModsBaseless();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstClassDeclarationNoModsExtends : public AstClassDeclaration 
    {
    public:
        AstClassDeclarationNoModsExtends(AstClass* const cl,
                                         AstIdentifier* const id,
                                         AstExtends* const ex,
                                         AstName* const na,
                                         AstClassBody* const clbo,
                                         File* const srcfile);

        virtual ~AstClassDeclarationNoModsExtends();

        AstExtends* getExtends() const;

        AstName* getName() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstExtends* extends_;
        AstName* name_;
    };


    class AstClassDeclarationNoModsExtendsImplements : public AstClassDeclaration 
    {
    public:
        AstClassDeclarationNoModsExtendsImplements(AstClass* const cl,
                                                   AstIdentifier* const id,
                                                   AstExtends* const ex,
                                                   AstName* const na,
                                                   AstImplementsInterfaceList* const iminli,
                                                   AstClassBody* const clbo,
                                                   File* const srcfile);

        virtual ~AstClassDeclarationNoModsExtendsImplements();

        AstExtends* getExtends() const;

        AstName* getName() const;

        AstImplementsInterfaceList* getImplementsInterfaceList() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstExtends* extends_;
        AstName* name_;
        AstImplementsInterfaceList* implementsInterfaceList_;
    };


    class AstClassDeclarationNoModsImplements : public AstClassDeclaration 
    {
    public:
        AstClassDeclarationNoModsImplements(AstClass* const cl,
                                            AstIdentifier* const id,
                                            AstImplementsInterfaceList* const iminli,
                                            AstClassBody* const clbo,
                                            File* const srcfile);

        virtual ~AstClassDeclarationNoModsImplements();

        AstImplementsInterfaceList* getImplementsInterfaceList() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstImplementsInterfaceList* implementsInterfaceList_;
    };


    /**
     * class_instance_creation_expression:
     *           NEW name LPAREN RPAREN
     *         | NEW name LPAREN expression_list RPAREN
     *         | NEW name LPAREN RPAREN class_body
     *         | NEW name LPAREN expression_list RPAREN class_body
     *         | primary DOT NEW IDENTIFIER LPAREN RPAREN
     *         | primary DOT NEW IDENTIFIER LPAREN expression_list RPAREN
     *         | primary DOT NEW IDENTIFIER LPAREN RPAREN class_body
     *         | primary DOT NEW IDENTIFIER LPAREN expression_list RPAREN class_body
     *         | name DOT NEW IDENTIFIER LPAREN RPAREN
     *         | name DOT NEW IDENTIFIER LPAREN expression_list RPAREN
     *         | name DOT NEW IDENTIFIER LPAREN RPAREN class_body
     *         | name DOT NEW IDENTIFIER LPAREN expression_list RPAREN class_body
     *         ;
     */
    class AstClassInstanceCreationExpression : public AstNode
    {
    public:
        virtual ~AstClassInstanceCreationExpression();

        AstNew* getNew() const;

        AstLparen* getLparen() const;

        AstExpressionList* getExpressionList() const;

        AstRparen* getRparen() const;

        AstClassBody* getClassBody() const;

    protected:
        AstClassInstanceCreationExpression(AstNew* const ne,
                                           AstLparen* const lp,
                                           AstExpressionList* const exli,
                                           AstRparen* const rp,
                                           AstClassBody* const clbo,
                                           File* const srcfile);
        
    private:
        AstNew* new_;
        AstLparen* lparen_;
        AstExpressionList* expressionList_;
        AstRparen* rparen_;
        AstClassBody* classBody_;
    };


    class AstClassInstanceCreationExpressionNameExprBody : public AstClassInstanceCreationExpression 
    {
    public:
        AstClassInstanceCreationExpressionNameExprBody(AstNew* const ne,
                                                       AstName* const na,
                                                       AstLparen* const lp,
                                                       AstExpressionList* const exli,
                                                       AstRparen* const rp,
                                                       AstClassBody* const clbo,
                                                       File* const srcfile);

        virtual ~AstClassInstanceCreationExpressionNameExprBody();

        AstName* getName() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstName* name_;
    };


    class AstClassInstanceCreationExpressionNameExprNoBody : public AstClassInstanceCreationExpression
    {
    public:
        AstClassInstanceCreationExpressionNameExprNoBody(AstNew* const ne,
                                                         AstName* const na,
                                                         AstLparen* const lp,
                                                         AstExpressionList* const exli,
                                                         AstRparen* const rp,
                                                         File* const srcfile);

        virtual ~AstClassInstanceCreationExpressionNameExprNoBody();

        AstName* getName() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstName* name_;
    };


    class AstClassInstanceCreationExpressionNameIdExprBody : public AstClassInstanceCreationExpression
    {
    public:
        AstClassInstanceCreationExpressionNameIdExprBody(AstName* const na,
                                                         AstDot* const d,
                                                         AstNew* const ne,
                                                         AstIdentifier* const id,
                                                         AstLparen* const lp,
                                                         AstExpressionList* const exli,
                                                         AstRparen* const rp,
                                                         AstClassBody* const clbo,
                                                         File* const srcfile);

        virtual ~AstClassInstanceCreationExpressionNameIdExprBody();

        AstName* getName() const;

        AstDot* getDot() const;

        AstIdentifier* getIdentifier() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstName* name_;
        AstDot* dot_;
        AstIdentifier* identifier_;
    };


    class AstClassInstanceCreationExpressionNameIdExprNoBody : public AstClassInstanceCreationExpression 
    {
    public:
        AstClassInstanceCreationExpressionNameIdExprNoBody(AstName* const na,
                                                           AstDot* const d,
                                                           AstNew* const ne,
                                                           AstIdentifier* const id,
                                                           AstLparen* const lp,
                                                           AstExpressionList* const exli,
                                                           AstRparen* const rp,
                                                           File* const srcfile);

        virtual ~AstClassInstanceCreationExpressionNameIdExprNoBody();

        AstName* getName() const;

        AstDot* getDot() const;

        AstIdentifier* getIdentifier() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstName* name_;
        AstDot* dot_;
        AstIdentifier* identifier_;
    };


    class AstClassInstanceCreationExpressionNameIdNoExprBody : public AstClassInstanceCreationExpression
    {
    public:
        AstClassInstanceCreationExpressionNameIdNoExprBody(AstName* const na,
                                                           AstDot* const d,
                                                           AstNew* const ne,
                                                           AstIdentifier* const id,
                                                           AstLparen* const lp,
                                                           AstRparen* const rp,
                                                           AstClassBody* const clbo,
                                                           File* const srcfile);

        virtual ~AstClassInstanceCreationExpressionNameIdNoExprBody();

        AstName* getName() const;

        AstDot* getDot() const;

        AstIdentifier* getIdentifier() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstName* name_;
        AstDot* dot_;
        AstIdentifier* identifier_;
    };


    class AstClassInstanceCreationExpressionNameIdNoExprNoBody : public AstClassInstanceCreationExpression
    {
    public:
        AstClassInstanceCreationExpressionNameIdNoExprNoBody(AstName* const na,
                                                             AstDot* const d,
                                                             AstNew* const ne,
                                                             AstIdentifier* const id,
                                                             AstLparen* const lp,
                                                             AstRparen* const rp,
                                                             File* const srcfile);

        virtual ~AstClassInstanceCreationExpressionNameIdNoExprNoBody();

        AstName* getName() const;

        AstDot* getDot() const;

        AstIdentifier* getIdentifier() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstName* name_;
        AstDot* dot_;
        AstIdentifier* identifier_;
    };


    class AstClassInstanceCreationExpressionNameNoExprBody : public AstClassInstanceCreationExpression
    {
    public:
        AstClassInstanceCreationExpressionNameNoExprBody(AstNew* const ne,
                                                         AstName* const na,
                                                         AstLparen* const lp,
                                                         AstRparen* const rp,
                                                         AstClassBody* const clbo,
                                                         File* const srcfile);

        virtual ~AstClassInstanceCreationExpressionNameNoExprBody();

        AstName* getName() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstName* name_;
    };


    class AstClassInstanceCreationExpressionNameNoExprNoBody : public AstClassInstanceCreationExpression
    {
    public:
        AstClassInstanceCreationExpressionNameNoExprNoBody(AstNew* const ne,
                                                           AstName* const na,
                                                           AstLparen* const lp,
                                                           AstRparen* const rp,
                                                           File* const srcfile);

        virtual ~AstClassInstanceCreationExpressionNameNoExprNoBody();

        AstName* getName() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstName* name_;
    };


    class AstClassInstanceCreationExpressionPrimaryExprBody : public AstClassInstanceCreationExpression 
    {
    public:
        AstClassInstanceCreationExpressionPrimaryExprBody(AstItem* const pr,
                                                          AstDot* const d,
                                                          AstNew* const ne,
                                                          AstIdentifier* const id,
                                                          AstLparen* const lp,
                                                          AstExpressionList* const exli,
                                                          AstRparen* const rp,
                                                          AstClassBody* const clbo,
                                                          File* const srcfile);

        virtual ~AstClassInstanceCreationExpressionPrimaryExprBody();

        AstItem* getPrimary() const;

        AstDot* getDot() const;

        AstIdentifier* getIdentifier() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstItem* primary_;
        AstDot* dot_;
        AstIdentifier* identifier_;
    };


    class AstClassInstanceCreationExpressionPrimaryExprNoBody : public AstClassInstanceCreationExpression 
    {
    public:
        AstClassInstanceCreationExpressionPrimaryExprNoBody(AstItem* const pr,
                                                            AstDot* const d,
                                                            AstNew* const ne,
                                                            AstIdentifier* const id,
                                                            AstLparen* const lp,
                                                            AstExpressionList* const exli,
                                                            AstRparen* const rp,
                                                            File* const srcfile);

        virtual ~AstClassInstanceCreationExpressionPrimaryExprNoBody();

        AstItem* getPrimary() const;

        AstDot* getDot() const;

        AstIdentifier* getIdentifier() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstItem* primary_;
        AstDot* dot_;
        AstIdentifier* identifier_;
    };


    class AstClassInstanceCreationExpressionPrimaryNoExprBody : public AstClassInstanceCreationExpression 
    {
    public:
        AstClassInstanceCreationExpressionPrimaryNoExprBody(AstItem* const pr,
                                                            AstDot* const d,
                                                            AstNew* const ne,
                                                            AstIdentifier* const id,
                                                            AstLparen* const lp,
                                                            AstRparen* const rp,
                                                            AstClassBody* const clbo,
                                                            File* const srcfile);

        virtual ~AstClassInstanceCreationExpressionPrimaryNoExprBody();

        AstItem* getPrimary() const;

        AstDot* getDot() const;

        AstIdentifier* getIdentifier() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstItem* primary_;
        AstDot* dot_;
        AstIdentifier* identifier_;
    };


    class AstClassInstanceCreationExpressionPrimaryNoExprNoBody : public AstClassInstanceCreationExpression 
    {
    public:
        AstClassInstanceCreationExpressionPrimaryNoExprNoBody(AstItem* const pr,
                                                              AstDot* const d,
                                                              AstNew* const ne,
                                                              AstIdentifier* const id,
                                                              AstLparen* const lp,
                                                              AstRparen* const rp,
                                                              File* const srcfile);

        virtual ~AstClassInstanceCreationExpressionPrimaryNoExprNoBody();

        AstItem* getPrimary() const;

        AstDot* getDot() const;

        AstIdentifier* getIdentifier() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstItem* primary_;
        AstDot* dot_;
        AstIdentifier* identifier_;
    };


    /**
     * class_member_declaration:
     *           CLASS IDENTIFIER class_body
     *         | CLASS IDENTIFIER implements_interface_list class_body
     *         | CLASS IDENTIFIER EXTENDS name class_body
     *         | CLASS IDENTIFIER EXTENDS name implements_interface_list class_body
     *         | modifier_list CLASS IDENTIFIER class_body
     *         | modifier_list CLASS IDENTIFIER implements_interface_list class_body
     *         | modifier_list CLASS IDENTIFIER EXTENDS name class_body
     *         | modifier_list CLASS IDENTIFIER EXTENDS name implements_interface_list class_body
     *         ;
     */
    class AstClassMemberDeclaration : public AstNode 
    {
    public:
        virtual ~AstClassMemberDeclaration();

        AstModifierList* getModifierList() const;

        AstClass* getClass() const;

        AstIdentifier* getIdentifier() const;

        AstExtends* getExtends() const;

        AstName* getName() const;

        AstImplementsInterfaceList* getImplementsInterfaceList() const;

        AstClassBody* getClassBody() const;

    protected:
        AstClassMemberDeclaration(AstModifierList* const moli,
                                  AstClass* const cl,
                                  AstIdentifier* const id,
                                  AstExtends* const ex,
                                  AstName* const na,
                                  AstImplementsInterfaceList* const iminli,
                                  AstClassBody* const clbo,
                                  File* const srcfile);

    private:
        AstModifierList* modifierList_;
        AstClass* class_;
        AstIdentifier* identifier_;
        AstExtends* extends_;
        AstName* name_;
        AstImplementsInterfaceList* implementsInterfaceList_;
        AstClassBody* classBody_;
    };


    class AstClassMemberDeclarationModsExtendsImplements : public AstClassMemberDeclaration 
    {
    public:
        AstClassMemberDeclarationModsExtendsImplements(AstModifierList* const moli,
                                                       AstClass* const cl,
                                                       AstIdentifier* const id,
                                                       AstExtends* const ex,
                                                       AstName* const na,
                                                       AstImplementsInterfaceList* const iminli,
                                                       AstClassBody* const clbo,
                                                       File* const srcfile);

        virtual ~AstClassMemberDeclarationModsExtendsImplements();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstClassMemberDeclarationModsExtendsNoImplements : public AstClassMemberDeclaration 
    {
    public:
        AstClassMemberDeclarationModsExtendsNoImplements(AstModifierList* const moli,
                                                         AstClass* const cl,
                                                         AstIdentifier* const id,
                                                         AstExtends* const ex,
                                                         AstName* const na,
                                                         AstClassBody* const clbo,
                                                         File* const srcfile);

        virtual ~AstClassMemberDeclarationModsExtendsNoImplements();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstClassMemberDeclarationModsNoExtendsImplements : public AstClassMemberDeclaration 
    {
    public:
        AstClassMemberDeclarationModsNoExtendsImplements(AstModifierList* const moli,
                                                         AstClass* const cl,
                                                         AstIdentifier* const id,
                                                         AstImplementsInterfaceList* const iminli,
                                                         AstClassBody* const clbo,
                                                         File* const srcfile);

        virtual ~AstClassMemberDeclarationModsNoExtendsImplements();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstClassMemberDeclarationModsNoExtendsNoImplements : public AstClassMemberDeclaration 
    {
    public:
        AstClassMemberDeclarationModsNoExtendsNoImplements(AstModifierList* const moli,
                                                           AstClass* const cl,
                                                           AstIdentifier* const id,
                                                           AstClassBody* const clbo,
                                                           File* const srcfile);

        virtual ~AstClassMemberDeclarationModsNoExtendsNoImplements();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstClassMemberDeclarationNoModsExtendsImplements : public AstClassMemberDeclaration 
    {
    public:
        AstClassMemberDeclarationNoModsExtendsImplements(AstClass* const cl,
                                                         AstIdentifier* const id,
                                                         AstExtends* const ex,
                                                         AstName* const na,
                                                         AstImplementsInterfaceList* const iminli,
                                                         AstClassBody* const clbo,
                                                         File* const srcfile);

        virtual ~AstClassMemberDeclarationNoModsExtendsImplements();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstClassMemberDeclarationNoModsExtendsNoImplements : public AstClassMemberDeclaration 
    {
    public:
        AstClassMemberDeclarationNoModsExtendsNoImplements(AstClass* const cl,
                                                           AstIdentifier* const id,
                                                           AstExtends* const ex,
                                                           AstName* const na,
                                                           AstClassBody* const clbo,
                                                           File* const srcfile);

        virtual ~AstClassMemberDeclarationNoModsExtendsNoImplements();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstClassMemberDeclarationNoModsNoExtendsImplements : public AstClassMemberDeclaration 
    {
    public:
        AstClassMemberDeclarationNoModsNoExtendsImplements(AstClass* const cl,
                                                           AstIdentifier* const id,
                                                           AstImplementsInterfaceList* const iminli,
                                                           AstClassBody* const clbo,
                                                           File* const srcfile);

        virtual ~AstClassMemberDeclarationNoModsNoExtendsImplements();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstClassMemberDeclarationNoModsNoExtendsNoImplements : public AstClassMemberDeclaration
    {
    public:
        AstClassMemberDeclarationNoModsNoExtendsNoImplements(AstClass* const cl,
                                                             AstIdentifier* const id,
                                                             AstClassBody* const clbo,
                                                             File* const srcfile);

        virtual ~AstClassMemberDeclarationNoModsNoExtendsNoImplements();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    /**
     * ":"
     */
    class AstColon : public AstLeaf 
    {
    public:
        AstColon(char* position, File* const srcfile);

        virtual ~AstColon();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * ","
     */
    class AstComma : public AstLeaf 
    {
    public:
        AstComma(char* position, File* const srcfile);

        virtual ~AstComma();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * ~
     */
    class AstComp : public AstLeaf 
    {
    public:
        AstComp(char* position, File* const srcfile);

        virtual ~AstComp();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * compilation_unit:
     *           package_declaration import_declaration_list type_declaration_list
     *         | package_declaration type_declaration_list
     *         | package_declaration import_declaration_list
     *         | import_declaration_list type_declaration_list
     *         | import_declaration_list
     *         | type_declaration_list
     *         | <<empty>>
     *         | error
     *         ;
     */
    class AstCompilationUnit : public AstNode 
    {
        typedef doctorj::AstNode SUPERCLASS;
        
    public:
        virtual ~AstCompilationUnit();

        /**
         * Sets the trailing whitespace, i.e., the last noncode in the file.
         */
        virtual void setTrailingNoncode(AstNoncode* const nc);

        AstPackageDeclaration* getPackageDeclaration() const;
        
        AstImportDeclarationList* getImportDeclarationList() const;

        AstTypeDeclarationList* getTypeDeclarationList() const;

        virtual AstNoncode* trailingNoncode() const;

    protected:
        AstCompilationUnit(AstPackageDeclaration* const pd, 
                           AstImportDeclarationList* const idl,
                           AstTypeDeclarationList* const tdl,
                           File* const srcfile);

    private:
        AstPackageDeclaration* packageDeclaration_;
        AstImportDeclarationList* importDeclarationList_;
        AstTypeDeclarationList* typeDeclarationList_;
        AstNoncode* lastNoncode_;
    };


    class AstCompilationUnitEmpty : public AstCompilationUnit 
    {
        typedef doctorj::AstCompilationUnit SUPERCLASS;
        
    public:
        AstCompilationUnitEmpty(File* const srcfile);
        
        virtual ~AstCompilationUnitEmpty();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;        
    };

    
    class AstCompilationUnitImp : public AstCompilationUnit 
    {
    public:
        AstCompilationUnitImp(AstImportDeclarationList* const idl, File* const srcfile);

        virtual ~AstCompilationUnitImp();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };

    
    class AstCompilationUnitImpTypes : public AstCompilationUnit 
    {
    public:
        AstCompilationUnitImpTypes(AstImportDeclarationList* const idl, 
                                   AstTypeDeclarationList* const tdl,
                                   File* const srcfile);
        
        virtual ~AstCompilationUnitImpTypes();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };

    
    class AstCompilationUnitPkg : public AstCompilationUnit 
    {
    public:
        AstCompilationUnitPkg(AstPackageDeclaration* const pd, File* const srcfile);

        virtual ~AstCompilationUnitPkg();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };

    
    class AstCompilationUnitPkgImp : public AstCompilationUnit 
    {
    public:
        AstCompilationUnitPkgImp(AstPackageDeclaration* const pd, 
                                 AstImportDeclarationList* const idl,
                                 File* const srcfile);

        virtual ~AstCompilationUnitPkgImp();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };

    
    class AstCompilationUnitPkgImpTypes : public AstCompilationUnit 
    {
    public:
        AstCompilationUnitPkgImpTypes(AstPackageDeclaration* const pd, 
                                      AstImportDeclarationList* const idl,
                                      AstTypeDeclarationList* const tdl,
                                      File* const srcfile);
        
        virtual ~AstCompilationUnitPkgImpTypes();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };

    
    class AstCompilationUnitPkgTypes : public AstCompilationUnit 
    {
    public:
        AstCompilationUnitPkgTypes(AstPackageDeclaration* const pd, 
                                   AstTypeDeclarationList* const tdl,
                                   File* const srcfile);

        virtual ~AstCompilationUnitPkgTypes();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };

    
    class AstCompilationUnitTypes : public AstCompilationUnit 
    {
    public:
        AstCompilationUnitTypes(AstTypeDeclarationList* const tdl, File* const srcfile);

        virtual ~AstCompilationUnitTypes();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };

    
    class AstConditionalAndExpression : public AstNode 
    {
    public:
        AstConditionalAndExpression(AstItem* const le, AstAndand* const a, AstItem* const re, File* const srcfile);

        virtual ~AstConditionalAndExpression();

        virtual void getProcessed(Processor* const proc);

        AstItem* getLhsExpression() const;
        
        AstAndand* getAndand() const;

        AstItem* getRhsExpression() const;

        virtual string type() const;
        
    private:
        AstItem* lhsExpression_;
        AstAndand* andand_;
        AstItem* rhsExpression_;
    };

    
    class AstConditionalArithExpression : public AstNode 
    {        
    public:
        AstConditionalArithExpression(AstItem* const ie, 
                                      AstQuestion* const q,
                                      AstItem* const te,
                                      AstColon* const c,
                                      AstItem* const ee,
                                      File* const srcfile);

        virtual ~AstConditionalArithExpression();

        virtual void getProcessed(Processor* const proc);

        AstItem* getIfExpression() const;
        
        AstQuestion* getQuestion() const;

        AstItem* getThenExpression() const;

        AstColon* getColon() const;

        AstItem* getElseExpression() const;

        virtual string type() const;
        
    private:
        AstItem* ifExpression_;
        AstQuestion* question_;
        AstItem* thenExpression_;
        AstColon* colon_;
        AstItem* elseExpression_;
    };

    
    class AstConditionalOrExpression : public AstNode 
    {
    public:
        AstConditionalOrExpression(AstItem* const le, AstOror* const o, AstItem* const re, File* const srcfile);

        virtual ~AstConditionalOrExpression();

        virtual void getProcessed(Processor* const proc);

        AstItem* getLhsExpression() const;
        
        AstOror* getOror() const;

        AstItem* getRhsExpression() const;

        virtual string type() const;
        
    private:
        AstItem* lhsExpression_;
        AstOror* oror_;
        AstItem* rhsExpression_;
    };


    /**
     * "const"
     */
    class AstConst : public AstLeaf 
    {
    public:
        AstConst(char* position, File* const srcfile);

        virtual ~AstConst();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * constructor_body:
     *           LBRACE explicit_constructor_invocation block_statement_list RBRACE
     *         | LBRACE explicit_constructor_invocation RBRACE
     *         | LBRACE block_statement_list RBRACE
     *         | LBRACE RBRACE
     *         ;
     */
    class AstConstructorBody : public AstNode 
    {
    public:
        virtual ~AstConstructorBody();

        AstLbrace* getLbrace() const;

        AstExplicitConstructorInvocation* getExplicitConstructorInvocation() const;

        AstBlockStatementList* getBlockStatementList() const;

        AstRbrace* getRbrace() const;

    protected:
        AstConstructorBody(AstLbrace* const lb,
                           AstExplicitConstructorInvocation* const excoin,
                           AstBlockStatementList* const blstli,
                           AstRbrace* const rb,
                           File* const srcfile);
        
    private:
        AstLbrace* lbrace_;
        AstExplicitConstructorInvocation* explicitConstructorInvocation_;
        AstBlockStatementList* blockStatementList_;
        AstRbrace* rbrace_;
    };


    class AstConstructorBodyBlock : public AstConstructorBody 
    {
    public:
        AstConstructorBodyBlock(AstLbrace* const lb,
                                AstBlockStatementList* const blstli,
                                AstRbrace* const rb,
                                File* const srcfile);

        virtual ~AstConstructorBodyBlock();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstConstructorBodyEmpty : public AstConstructorBody 
    {
    public:
        AstConstructorBodyEmpty(AstLbrace* const lb, AstRbrace* const rb, File* const srcfile);

        virtual ~AstConstructorBodyEmpty();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstConstructorBodyExplicit : public AstConstructorBody 
    {
    public:
        AstConstructorBodyExplicit(AstLbrace* const lb,
                                   AstExplicitConstructorInvocation* const excoin,
                                   AstRbrace* const rb,
                                   File* const srcfile);

        virtual ~AstConstructorBodyExplicit();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstConstructorBodyExplicitBlock : public AstConstructorBody 
    {
    public:
        AstConstructorBodyExplicitBlock(AstLbrace* const lb,
                                        AstExplicitConstructorInvocation* const excoin,
                                        AstBlockStatementList* const blstli,
                                        AstRbrace* const rb,
                                        File* const srcfile);

        virtual ~AstConstructorBodyExplicitBlock();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    /**
     * constructor_declaration:
     *           constructor_declarator constructor_body
     *         | constructor_declarator throws_name_list constructor_body
     *         | modifier_list constructor_declarator constructor_body
     *         | modifier_list constructor_declarator throws_name_list constructor_body
     *         ;
     */
    class AstConstructorDeclaration : public AstNode 
    {
    public:
        virtual ~AstConstructorDeclaration();

        AstModifierList* getModifierList() const;

        AstConstructorDeclarator* getConstructorDeclarator() const;

        AstThrowsNameList* getThrowsNameList() const;

        AstConstructorBody* getConstructorBody() const;

    protected:
        AstConstructorDeclaration(AstModifierList* const moli,
                                  AstConstructorDeclarator* const code,
                                  AstThrowsNameList* const thnali,
                                  AstConstructorBody* const cobo,
                                  File* const srcfile);

    private:
        AstModifierList* modifierList_;
        AstConstructorDeclarator* constructorDeclarator_;
        AstThrowsNameList* throwsNameList_;
        AstConstructorBody* constructorBody_;
    };


    class AstConstructorDeclarationModsNoThrows : public AstConstructorDeclaration 
    {
    public:
        AstConstructorDeclarationModsNoThrows(AstModifierList* const moli,
                                              AstConstructorDeclarator* const code,
                                              AstConstructorBody* const cobo,
                                              File* const srcfile);

        virtual ~AstConstructorDeclarationModsNoThrows();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstConstructorDeclarationModsThrows : public AstConstructorDeclaration 
    {
    public:
        AstConstructorDeclarationModsThrows(AstModifierList* const moli,
                                            AstConstructorDeclarator* const code,
                                            AstThrowsNameList* const thnali,
                                            AstConstructorBody* const cobo,
                                            File* const srcfile);

        virtual ~AstConstructorDeclarationModsThrows();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstConstructorDeclarationNoModsNoThrows : public AstConstructorDeclaration 
    {
    public:
        AstConstructorDeclarationNoModsNoThrows(AstConstructorDeclarator* const code,
                                                AstConstructorBody* const cobo,
                                                File* const srcfile);

        virtual ~AstConstructorDeclarationNoModsNoThrows();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstConstructorDeclarationNoModsThrows : public AstConstructorDeclaration 
    {
    public:
        AstConstructorDeclarationNoModsThrows(AstConstructorDeclarator* const code,
                                              AstThrowsNameList* const thnali,
                                              AstConstructorBody* const cobo,
                                              File* const srcfile);
        
        virtual ~AstConstructorDeclarationNoModsThrows();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    /**
     * constructor_declarator:
     *           IDENTIFIER LPAREN RPAREN
     *         | IDENTIFIER LPAREN formal_parameter_list RPAREN
     *         ;
     */
    class AstConstructorDeclarator : public AstNode 
    {
    public:
        virtual ~AstConstructorDeclarator();

        AstIdentifier* getIdentifier() const;

        AstLparen* getLparen() const;

        AstFormalParameterList* getFormalParameterList() const;

        AstRparen* getRparen() const;

    protected:
        AstConstructorDeclarator(AstIdentifier* const id,
                                 AstLparen* const lp,
                                 AstFormalParameterList* const fopali,
                                 AstRparen* const rp,
                                 File* const srcfile);

    private:
        AstIdentifier* identifier_;
        AstLparen* lparen_;
        AstFormalParameterList* formalParameterList_;
        AstRparen* rparen_;
    };


    class AstConstructorDeclaratorEmpty : public AstConstructorDeclarator 
    {
    public:
        AstConstructorDeclaratorEmpty(AstIdentifier* const id,
                                      AstLparen* const lp,
                                      AstRparen* const rp,
                                      File* const srcfile);

        virtual ~AstConstructorDeclaratorEmpty();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstConstructorDeclaratorParams : public AstConstructorDeclarator 
    {
    public:
        AstConstructorDeclaratorParams(AstIdentifier* const id,
                                       AstLparen* const lp,
                                       AstFormalParameterList* const fopali,
                                       AstRparen* const rp,
                                       File* const srcfile);

        virtual ~AstConstructorDeclaratorParams();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    /**
     * "continue"
     */
    class AstContinue : public AstLeaf 
    {
    public:
        AstContinue(char* position, File* const srcfile);

        virtual ~AstContinue();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    /**
     * continue_statement:
     *           CONTINUE SEMICOLON
     *         | CONTINUE IDENTIFIER SEMICOLON
     *         ;
     */
    class AstContinueStatement : public AstNode 
    {
    public:
        
        virtual ~AstContinueStatement();

        virtual void getProcessed(Processor* const proc);

        AstContinue* getContinue() const;
        
        AstIdentifier* getIdentifier() const;

        AstSemicolon* getSemicolon() const;

        virtual string type() const;

    protected:
        AstContinueStatement(AstContinue* const c, 
                             AstIdentifier* const i,
                             AstSemicolon* const s,
                             File* const srcfile);
        
    private:
        AstContinue* continue_;
        AstIdentifier* identifier_;
        AstSemicolon* semicolon_;
    };


    class AstContinueStatementId : public AstContinueStatement 
    {
    public:
        AstContinueStatementId(AstContinue* const c, 
                               AstIdentifier* const i,
                               AstSemicolon* const s,
                               File* const srcfile);

        virtual ~AstContinueStatementId();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstContinueStatementNoId : public AstContinueStatement 
    {
    public:
        AstContinueStatementNoId(AstContinue* const c, AstSemicolon* const s, File* const srcfile);

        virtual ~AstContinueStatementNoId();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstCppComment : public AstComment 
    {
    public:
        AstCppComment(char* start, char* end, File* const srcfile);

        virtual ~AstCppComment();

        virtual string type() const;

        /**
         * Invokes process on the Processor for this specific type of
         * AstNode/AstItem/AstLeaf.
         */
        virtual void getProcessed(Processor* const p);
    };


    /**
     * "default"
     */
    class AstDefault : public AstLeaf 
    {
    public:
        AstDefault(char* position, File* const srcfile);

        virtual ~AstDefault();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    class AstDimension : public AstNode 
    {
    public:
        AstDimension(AstLbracket* const item0, AstRbracket* const item1, File* const srcfile);

        virtual ~AstDimension();

        virtual void getProcessed(Processor* const proc);

        AstLbracket* getLbracket() const;
        
        AstRbracket* getRbracket() const;

        virtual string type() const;
        
    private:
        AstLbracket* lbracket_;
        AstRbracket* rbracket_;
    };

    
    class AstDimensionExpression : public AstNode 
    {
    public:
        AstDimensionExpression(AstLbracket* const l, AstItem* const e, AstRbracket* const r, File* const srcfile);

        virtual ~AstDimensionExpression();

        virtual void getProcessed(Processor* const proc);

        AstLbracket* getLbracket() const;
        
        AstItem* getExpression() const;

        AstRbracket* getRbracket() const;

        virtual string type() const;
        
    private:
        AstLbracket* lbracket_;
        AstItem* expression_;
        AstRbracket* rbracket_;
    };

    
    class AstDimensionExpressionList : public AstNode 
    {
    public:
        AstDimensionExpressionList(AstDimensionExpression* const de, File* const srcfile);

        virtual ~AstDimensionExpressionList();

        virtual void getProcessed(Processor* const proc);

        AstDimensionExpression* getDimensionExpression(int index) const;

        int getDimensionExpressionCount() const;

        virtual void add(AstDimensionExpression* const de);

        virtual string type() const;
        
    private:
        vector<AstDimensionExpression*> dimensionExpressions_;
    };

    
    class AstDimensionList : public AstNode 
    {
    public:
        AstDimensionList(AstDimension* const d, File* const srcfile);

        virtual ~AstDimensionList();

        virtual void getProcessed(Processor* const proc);

        AstDimension* getDimension(int index) const;

        int getDimensionCount() const;

        virtual string type() const;

        virtual void add(AstDimension* const d);
        
    private:
        vector<AstDimension*> dimensions_;
    };


    /**
     * "/"
     */
    class AstDiv : public AstLeaf 
    {
    public:
        AstDiv(char* position, File* const srcfile);

        virtual ~AstDiv();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * "/="
     */
    class AstDiveq : public AstAssignmentOperator 
    {
    public:
        AstDiveq(char* position, File* const srcfile);

        virtual ~AstDiveq();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * "do"
     */
    class AstDo : public AstLeaf 
    {
    public:
        AstDo(char* position, File* const srcfile);

        virtual ~AstDo();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    class AstDoStatement : public AstNode 
    {        
    public:
        AstDoStatement(AstDo* const d, 
                       AstItem* const st,
                       AstWhile* const w,
                       AstLparen* const l,
                       AstItem* const e,
                       AstRparen* const r,
                       AstSemicolon* const se,
                       File* const srcfile);

        virtual ~AstDoStatement();

        virtual void getProcessed(Processor* const proc);

        AstDo* getDo() const;
        
        AstItem* getStatement() const;

        AstWhile* getWhile() const;

        AstLparen* getLparen() const;

        AstItem* getExpression() const;

        AstRparen* getRparen() const;

        AstSemicolon* getSemicolon() const;

        virtual string type() const;
        
    private:
        AstDo* do_;
        AstItem* statement_;
        AstWhile* while_;
        AstLparen* lparen_;
        AstItem* expression_;
        AstRparen* rparen_;
        AstSemicolon* semicolon_;
    };


    /**
     * "."
     */
    class AstDot : public AstLeaf 
    {
    public:
        AstDot(char* position, File* const srcfile);

        virtual ~AstDot();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    class AstFloatingPointType : public AstNumericType 
    {
    public:
        AstFloatingPointType(char* position, File* const srcfile);

        virtual ~AstFloatingPointType();
    };


    /**
     * "double"
     */
    class AstDouble : public AstFloatingPointType 
    {
    public:
        AstDouble(char* position, File* const srcfile);

        virtual ~AstDouble();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * "else"
     */
    class AstElse : public AstLeaf 
    {
    public:
        AstElse(char* position, File* const srcfile);

        virtual ~AstElse();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * "="
     */
    class AstEq : public AstAssignmentOperator
    {
    public:
        AstEq(char* position, File* const srcfile);

        virtual ~AstEq();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * "=="
     */
    class AstEqeq : public AstLeaf 
    {
    public:
        AstEqeq(char* position, File* const srcfile);

        virtual ~AstEqeq();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    class AstEqualityExpression : public AstNode 
    {
    public:
        AstEqualityExpression(AstItem* const lhex, AstItem* const rhex, File* const srcfile);

        virtual ~AstEqualityExpression();

        AstItem* getLhsExpression() const;

        AstItem* getRhsExpression() const;

    private:
        AstItem* lhsExpression_;
        AstItem* rhsExpression_;
    };


    class AstEqualityExpressionEq : public AstEqualityExpression 
    {
    public:
        AstEqualityExpressionEq(AstItem* const lhex, AstEqeq* const eq, AstItem* const rhex, File* const srcfile);

        virtual ~AstEqualityExpressionEq();

        AstEqeq* getEqeq() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstEqeq* eqeq_;
    };


    class AstEqualityExpressionNe : public AstEqualityExpression 
    {
    public:
        AstEqualityExpressionNe(AstItem* const lhex, AstNoteq* const no, AstItem* const rhex, File* const srcfile);

        virtual ~AstEqualityExpressionNe();

        AstNoteq* getNoteq() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstNoteq* noteq_;
    };


    class AstExplicitConstructorInvocation : public AstNode 
    {
    public:
        AstExplicitConstructorInvocation(AstLparen* const lp,
                                         AstRparen* const rp,
                                         AstSemicolon* const se,
                                         File* const srcfile);

        virtual ~AstExplicitConstructorInvocation();

        AstLparen* getLparen() const;

        AstRparen* getRparen() const;

        AstSemicolon* getSemicolon() const;

    private:
        AstLparen* lparen_;
        AstRparen* rparen_;
        AstSemicolon* semicolon_;
    };


    class AstExplicitConstructorInvocationPrimarySuper : public AstExplicitConstructorInvocation 
    {
    public:
        AstExplicitConstructorInvocationPrimarySuper(AstItem* const pr,
                                                     AstDot* const d,
                                                     AstSuper* const su,
                                                     AstLparen* const lp,
                                                     AstRparen* const rp,
                                                     AstSemicolon* const se,
                                                     File* const srcfile);

        virtual ~AstExplicitConstructorInvocationPrimarySuper();

        AstItem* getPrimary() const;

        AstDot* getDot() const;

        AstSuper* getSuper() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstItem* primary_;
        AstDot* dot_;
        AstSuper* super_;
    };


    class AstExplicitConstructorInvocationPrimarySuperExpressions : public AstExplicitConstructorInvocation 
    {
    public:
        AstExplicitConstructorInvocationPrimarySuperExpressions(AstItem* const pr,
                                                                AstDot* const d,
                                                                AstSuper* const su,
                                                                AstLparen* const lp,
                                                                AstExpressionList* const exli,
                                                                AstRparen* const rp,
                                                                AstSemicolon* const se,
                                                                File* const srcfile);

        virtual ~AstExplicitConstructorInvocationPrimarySuperExpressions();

        AstItem* getPrimary() const;

        AstDot* getDot() const;

        AstSuper* getSuper() const;

        AstExpressionList* getExpressionList() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstItem* primary_;
        AstDot* dot_;
        AstSuper* super_;
        AstExpressionList* expressionList_;
    };


    class AstExplicitConstructorInvocationPrimaryThis : public AstExplicitConstructorInvocation 
    {
    public:
        AstExplicitConstructorInvocationPrimaryThis(AstItem* const pr,
                                                    AstDot* const d,
                                                    AstThis* const th,
                                                    AstLparen* const lp,
                                                    AstRparen* const rp,
                                                    AstSemicolon* const se,
                                                    File* const srcfile);

        virtual ~AstExplicitConstructorInvocationPrimaryThis();

        AstItem* getPrimary() const;

        AstDot* getDot() const;

        AstThis* getThis() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstItem* primary_;
        AstDot* dot_;
        AstThis* this_;
    };


    class AstExplicitConstructorInvocationPrimaryThisExpressions : public AstExplicitConstructorInvocation 
    {
    public:
        AstExplicitConstructorInvocationPrimaryThisExpressions(AstItem* const pr,
                                                               AstDot* const d,
                                                               AstThis* const th,
                                                               AstLparen* const lp,
                                                               AstExpressionList* const exli,
                                                               AstRparen* const rp,
                                                               AstSemicolon* const se,
                                                               File* const srcfile);

        virtual ~AstExplicitConstructorInvocationPrimaryThisExpressions();

        AstItem* getPrimary() const;

        AstDot* getDot() const;

        AstThis* getThis() const;

        AstExpressionList* getExpressionList() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstItem* primary_;
        AstDot* dot_;
        AstThis* this_;
        AstExpressionList* expressionList_;
    };


    class AstExplicitConstructorInvocationSuper : public AstExplicitConstructorInvocation 
    {
    public:
        AstExplicitConstructorInvocationSuper(AstSuper* const su,
                                              AstLparen* const lp,
                                              AstRparen* const rp,
                                              AstSemicolon* const se,
                                              File* const srcfile);

        virtual ~AstExplicitConstructorInvocationSuper();

        AstSuper* getSuper() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstSuper* super_;
    };


    class AstExplicitConstructorInvocationSuperExpressions : public AstExplicitConstructorInvocation 
    {
    public:
        AstExplicitConstructorInvocationSuperExpressions(AstSuper* const su,
                                                         AstLparen* const lp,
                                                         AstExpressionList* const exli,
                                                         AstRparen* const rp,
                                                         AstSemicolon* const se,
                                                         File* const srcfile);

        virtual ~AstExplicitConstructorInvocationSuperExpressions();

        AstSuper* getSuper() const;

        AstExpressionList* getExpressionList() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstSuper* super_;
        AstExpressionList* expressionList_;
    };


    class AstExplicitConstructorInvocationThis : public AstExplicitConstructorInvocation 
    {
    public:
        AstExplicitConstructorInvocationThis(AstThis* const th,
                                             AstLparen* const lp,
                                             AstRparen* const rp,
                                             AstSemicolon* const se,
                                             File* const srcfile);

        virtual ~AstExplicitConstructorInvocationThis();

        AstThis* getThis() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstThis* this_;
    };


    class AstExplicitConstructorInvocationThisExpressions : public AstExplicitConstructorInvocation 
    {
    public:
        AstExplicitConstructorInvocationThisExpressions(AstThis* const th,
                                                        AstLparen* const lp,
                                                        AstExpressionList* const exli,
                                                        AstRparen* const rp,
                                                        AstSemicolon* const se,
                                                        File* const srcfile);

        virtual ~AstExplicitConstructorInvocationThisExpressions();

        AstThis* getThis() const;

        AstExpressionList* getExpressionList() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstThis* this_;
        AstExpressionList* expressionList_;
    };

    
    class AstExpressionList : public AstNode 
    {
    public:
        AstExpressionList(AstItem* const e, File* const srcfile);

        virtual ~AstExpressionList();

        virtual void getProcessed(Processor* const proc);

        AstItem* getExpression(int index) const;

        int getExpressionCount() const;

        virtual void add(AstItem* const e);

        AstComma* getComma(int index) const;

        int getCommaCount() const;

        virtual void add(AstComma* const c);

        virtual string type() const;
        
    private:
        vector<AstItem*> expressions_;
        vector<AstComma*> commas_;
    };

    
    class AstExpressionStatement : public AstNode 
    {
    public:
        AstExpressionStatement(AstItem* const e, AstSemicolon* const s, File* const srcfile);

        virtual ~AstExpressionStatement();

        virtual void getProcessed(Processor* const proc);

        AstItem* getExpression() const;
        
        AstSemicolon* getSemicolon() const;

        virtual string type() const;
        
    private:
        AstItem* expression_;
        AstSemicolon* semicolon_;
    };


    /**
     * "extends"
     */
    class AstExtends : public AstLeaf 
    {
    public:
        AstExtends(char* position, File* const srcfile);

        virtual ~AstExtends();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    class AstExtendsInterfaceList : public AstNode 
    {
    public:
        AstExtendsInterfaceList(AstExtends* const e, AstNameList* const nl, File* const srcfile);

        virtual ~AstExtendsInterfaceList();

        virtual void getProcessed(Processor* const proc);

        AstExtends* getExtends() const;
        
        AstNameList* getNameList() const;

        virtual string type() const;
        
    private:
        AstExtends* extends_;
        AstNameList* nameList_;
    };


    /**
     * A value token, whose type is FALSE_LITERAL.
     */
    class AstFalseLiteral : public AstBooleanLiteral
    {
    public:
        AstFalseLiteral(char* position, File* const srcfile);

        virtual ~AstFalseLiteral();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);
    };


    class AstFieldAccess : public AstNode 
    {
    public:
        AstFieldAccess(AstIdentifier* const id, File* const srcfile);

        virtual ~AstFieldAccess();

        AstIdentifier* getIdentifier() const;

    private:
        AstIdentifier* identifier_;
    };


    class AstFieldAccessNameSuper : public AstFieldAccess 
    {
    public:
        AstFieldAccessNameSuper(AstName* const na,
                                AstDot* const d0,
                                AstSuper* const su,
                                AstDot* const d1,
                                AstIdentifier* const id,
                                File* const srcfile);

        virtual ~AstFieldAccessNameSuper();

        AstName* getName() const;

        AstDot* getFirstDot() const;

        AstSuper* getSuper() const;

        AstDot* getSecondDot() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstName* name_;
        AstDot* firstDot_;
        AstSuper* super_;
        AstDot* secondDot_;
    };


    class AstFieldAccessPrimary : public AstFieldAccess 
    {
    public:
        AstFieldAccessPrimary(AstItem* const pr,
                              AstDot* const d,
                              AstIdentifier* const id,
                              File* const srcfile);

        virtual ~AstFieldAccessPrimary();

        AstItem* getPrimary() const;

        AstDot* getDot() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstItem* primary_;
        AstDot* dot_;
    };


    class AstFieldAccessSuper : public AstFieldAccess 
    {
    public:
        AstFieldAccessSuper(AstSuper* const su,
                            AstDot* const d,
                            AstIdentifier* const id,
                            File* const srcfile);

        virtual ~AstFieldAccessSuper();

        AstSuper* getSuper() const;

        AstDot* getDot() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstSuper* super_;
        AstDot* dot_;
    };


    /**
     * field_declaration:
     *           modifier_list type variable_declarator_list SEMICOLON
     *         | type variable_declarator_list SEMICOLON
     *         ;
     */
    class AstFieldDeclaration : public AstNode 
    {
    public:
        virtual ~AstFieldDeclaration();

        AstModifierList* getModifierList() const;

        AstItem* getType() const;

        AstVariableDeclaratorList* getVariableDeclaratorList() const;

        AstSemicolon* getSemicolon() const;

    protected:
        AstFieldDeclaration(AstModifierList* const moli,
                            AstItem* const ty,
                            AstVariableDeclaratorList* const vadeli,
                            AstSemicolon* const se,
                            File* const srcfile);
        
    private:
        AstModifierList* modifierList_;
        AstItem* type_;
        AstVariableDeclaratorList* variableDeclaratorList_;
        AstSemicolon* semicolon_;
    };


    class AstFieldDeclarationMods : public AstFieldDeclaration 
    {
    public:
        AstFieldDeclarationMods(AstModifierList* const moli,
                                AstItem* const ty,
                                AstVariableDeclaratorList* const vadeli,
                                AstSemicolon* const se,
                                File* const srcfile);

        virtual ~AstFieldDeclarationMods();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstFieldDeclarationNoMods : public AstFieldDeclaration 
    {
    public:
        AstFieldDeclarationNoMods(AstItem* const ty,
                                  AstVariableDeclaratorList* const vadeli,
                                  AstSemicolon* const se,
                                  File* const srcfile);

        virtual ~AstFieldDeclarationNoMods();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    /**
     * "final"
     */
    class AstFinal : public AstLeaf 
    {
    public:
        AstFinal(char* position, File* const srcfile);

        virtual ~AstFinal();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * "finally"
     */
    class AstFinally : public AstLeaf 
    {
    public:
        AstFinally(char* position, File* const srcfile);

        virtual ~AstFinally();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    class AstFinallyBlock : public AstNode 
    {
    public:
        AstFinallyBlock(AstFinally* const f, AstBlock* const b, File* const srcfile);

        virtual ~AstFinallyBlock();

        virtual void getProcessed(Processor* const proc);

        AstFinally* getFinally() const;
        
        AstBlock* getBlock() const;

        virtual string type() const;
        
    private:
        AstFinally* finally_;
        AstBlock* block_;
    };


    /**
     * "float"
     */
    class AstFloat : public AstFloatingPointType 
    {
    public:
        AstFloat(char* position, File* const srcfile);

        virtual ~AstFloat();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * A value token, whose type is FLOATING_POINT_LITERAL.
     */
    class AstFloatingPointLiteral : public AstLiteral
    {
    public:
        AstFloatingPointLiteral(const string& text, char* position, File* const srcfile);

        virtual ~AstFloatingPointLiteral();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);
    };


    /**
     * "for"
     */
    class AstFor : public AstLeaf 
    {
    public:
        AstFor(char* position, File* const srcfile);

        virtual ~AstFor();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    class AstForStatement : public AstNode 
    {
    public:
        AstForStatement(AstFor* const fo,
                        AstLparen* const lp,
                        AstSemicolon* const se0,
                        AstSemicolon* const se1,
                        AstRparen* const rp,
                        AstItem* const st,
                        File* const srcfile);

        virtual ~AstForStatement();

        AstFor* getFor() const;

        AstLparen* getLparen() const;

        AstSemicolon* getFirstSemicolon() const;

        AstSemicolon* getSecondSemicolon() const;

        AstRparen* getRparen() const;

        AstItem* getStatement() const;

    private:
        AstFor* for_;
        AstLparen* lparen_;
        AstSemicolon* firstSemicolon_;
        AstSemicolon* secondSemicolon_;
        AstRparen* rparen_;
        AstItem* statement_;
    };


    class AstForStatementNoInitConditionNoUpdate : public AstForStatement 
    {
    public:
        AstForStatementNoInitConditionNoUpdate(AstFor* const fo,
                                               AstLparen* const lp,
                                               AstSemicolon* const se0,
                                               AstItem* const ex,
                                               AstSemicolon* const se1,
                                               AstRparen* const rp,
                                               AstItem* const st,
                                               File* const srcfile);

        virtual ~AstForStatementNoInitConditionNoUpdate();

        AstItem* getExpression() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstItem* expression_;
    };


    class AstForStatementNoInitConditionUpdate : public AstForStatement 
    {
    public:
        AstForStatementNoInitConditionUpdate(AstFor* const fo,
                                             AstLparen* const lp,
                                             AstSemicolon* const se0,
                                             AstItem* const ex,
                                             AstSemicolon* const se1,
                                             AstStatementExpressionList* const cstexli,
                                             AstRparen* const rp,
                                             AstItem* const st,
                                             File* const srcfile);

        virtual ~AstForStatementNoInitConditionUpdate();

        AstItem* getExpression() const;

        AstStatementExpressionList* getConditionStatementExpressionList() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstItem* expression_;
        AstStatementExpressionList* conditionStatementExpressionList_;
    };


    class AstForStatementNoInitNoConditionNoUpdate : public AstForStatement 
    {
    public:
        AstForStatementNoInitNoConditionNoUpdate(AstFor* const fo,
                                                 AstLparen* const lp,
                                                 AstSemicolon* const se0,
                                                 AstSemicolon* const se1,
                                                 AstRparen* const rp,
                                                 AstItem* const st,
                                                 File* const srcfile);

        virtual ~AstForStatementNoInitNoConditionNoUpdate();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstForStatementNoInitNoConditionUpdate : public AstForStatement 
    {
    public:
        AstForStatementNoInitNoConditionUpdate(AstFor* const fo,
                                               AstLparen* const lp,
                                               AstSemicolon* const se0,
                                               AstSemicolon* const se1,
                                               AstStatementExpressionList* const cstexli,
                                               AstRparen* const rp,
                                               AstItem* const st,
                                               File* const srcfile);

        virtual ~AstForStatementNoInitNoConditionUpdate();

        AstStatementExpressionList* getConditionStatementExpressionList() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstStatementExpressionList* conditionStatementExpressionList_;
    };


    class AstForStatementStmtsConditionNoUpdate : public AstForStatement 
    {
    public:
        AstForStatementStmtsConditionNoUpdate(AstFor* const fo,
                                              AstLparen* const lp,
                                              AstStatementExpressionList* const istexli,
                                              AstSemicolon* const se0,
                                              AstItem* const ex,
                                              AstSemicolon* const se1,
                                              AstRparen* const rp,
                                              AstItem* const st,
                                              File* const srcfile);

        virtual ~AstForStatementStmtsConditionNoUpdate();

        AstStatementExpressionList* getInitStatementExpressionList() const;

        AstItem* getExpression() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstStatementExpressionList* initStatementExpressionList_;
        AstItem* expression_;
    };


    class AstForStatementStmtsConditionUpdate : public AstForStatement 
    {
    public:
        AstForStatementStmtsConditionUpdate(AstFor* const fo,
                                            AstLparen* const lp,
                                            AstStatementExpressionList* const istexli,
                                            AstSemicolon* const se0,
                                            AstItem* const ex,
                                            AstSemicolon* const se1,
                                            AstStatementExpressionList* const cstexli,
                                            AstRparen* const rp,
                                            AstItem* const st,
                                            File* const srcfile);

        virtual ~AstForStatementStmtsConditionUpdate();

        AstStatementExpressionList* getInitStatementExpressionList() const;

        AstItem* getExpression() const;

        AstStatementExpressionList* getConditionStatementExpressionList() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstStatementExpressionList* initStatementExpressionList_;
        AstItem* expression_;
        AstStatementExpressionList* conditionStatementExpressionList_;
    };


    class AstForStatementStmtsNoConditionNoUpdate : public AstForStatement 
    {
    public:
        AstForStatementStmtsNoConditionNoUpdate(AstFor* const fo,
                                                AstLparen* const lp,
                                                AstStatementExpressionList* const istexli,
                                                AstSemicolon* const se0,
                                                AstSemicolon* const se1,
                                                AstRparen* const rp,
                                                AstItem* const st,
                                                File* const srcfile);

        virtual ~AstForStatementStmtsNoConditionNoUpdate();

        AstStatementExpressionList* getInitStatementExpressionList() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstStatementExpressionList* initStatementExpressionList_;
    };


    class AstForStatementStmtsNoConditionUpdate : public AstForStatement 
    {
    public:
        AstForStatementStmtsNoConditionUpdate(AstFor* const fo,
                                              AstLparen* const lp,
                                              AstStatementExpressionList* const istexli,
                                              AstSemicolon* const se0,
                                              AstSemicolon* const se1,
                                              AstStatementExpressionList* const cstexli,
                                              AstRparen* const rp,
                                              AstItem* const st,
                                              File* const srcfile);

        virtual ~AstForStatementStmtsNoConditionUpdate();

        AstStatementExpressionList* getInitStatementExpressionList() const;

        AstStatementExpressionList* getConditionStatementExpressionList() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstStatementExpressionList* initStatementExpressionList_;
        AstStatementExpressionList* conditionStatementExpressionList_;
    };


    class AstForStatementVarsConditionNoUpdate : public AstForStatement 
    {
    public:
        AstForStatementVarsConditionNoUpdate(AstFor* const fo,
                                             AstLparen* const lp,
                                             AstLocalVariableDeclaration* const lovade,
                                             AstSemicolon* const se0,
                                             AstItem* const ex,
                                             AstSemicolon* const se1,
                                             AstRparen* const rp,
                                             AstItem* const st,
                                             File* const srcfile);

        virtual ~AstForStatementVarsConditionNoUpdate();

        AstLocalVariableDeclaration* getLocalVariableDeclaration() const;

        AstItem* getExpression() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstLocalVariableDeclaration* localVariableDeclaration_;
        AstItem* expression_;
    };


    class AstForStatementVarsConditionUpdate : public AstForStatement 
    {
    public:
        AstForStatementVarsConditionUpdate(AstFor* const fo,
                                           AstLparen* const lp,
                                           AstLocalVariableDeclaration* const lovade,
                                           AstSemicolon* const se0,
                                           AstItem* const ex,
                                           AstSemicolon* const se1,
                                           AstStatementExpressionList* const cstexli,
                                           AstRparen* const rp,
                                           AstItem* const st,
                                           File* const srcfile);

        virtual ~AstForStatementVarsConditionUpdate();

        AstLocalVariableDeclaration* getLocalVariableDeclaration() const;

        AstItem* getExpression() const;

        AstStatementExpressionList* getConditionStatementExpressionList() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstLocalVariableDeclaration* localVariableDeclaration_;
        AstItem* expression_;
        AstStatementExpressionList* conditionStatementExpressionList_;
    };


    class AstForStatementVarsNoConditionNoUpdate : public AstForStatement 
    {
    public:
        AstForStatementVarsNoConditionNoUpdate(AstFor* const fo,
                                               AstLparen* const lp,
                                               AstLocalVariableDeclaration* const lovade,
                                               AstSemicolon* const se0,
                                               AstSemicolon* const se1,
                                               AstRparen* const rp,
                                               AstItem* const st,
                                               File* const srcfile);

        virtual ~AstForStatementVarsNoConditionNoUpdate();

        AstLocalVariableDeclaration* getLocalVariableDeclaration() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstLocalVariableDeclaration* localVariableDeclaration_;
    };


    class AstForStatementVarsNoConditionUpdate : public AstForStatement 
    {
    public:
        AstForStatementVarsNoConditionUpdate(AstFor* const fo,
                                             AstLparen* const lp,
                                             AstLocalVariableDeclaration* const lovade,
                                             AstSemicolon* const se0,
                                             AstSemicolon* const se1,
                                             AstStatementExpressionList* const cstexli,
                                             AstRparen* const rp,
                                             AstItem* const st,
                                             File* const srcfile);

        virtual ~AstForStatementVarsNoConditionUpdate();

        AstLocalVariableDeclaration* getLocalVariableDeclaration() const;

        AstStatementExpressionList* getConditionStatementExpressionList() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstLocalVariableDeclaration* localVariableDeclaration_;
        AstStatementExpressionList* conditionStatementExpressionList_;
    };

    
    /**
     * formal_parameter:
     *           type variable_declarator_id
     *         | FINAL type variable_declarator_id
     *         ;
     */
    class AstFormalParameter : public AstNode 
    {
    public:
        virtual ~AstFormalParameter();

        AstFinal* getFinal() const;

        AstItem* getType() const;
        
        AstVariableDeclaratorId* getVariableDeclaratorId() const;

        virtual string name() const;

        virtual string dataType() const;

    protected:
        AstFormalParameter(AstFinal* const f, 
                           AstItem* const t,
                           AstVariableDeclaratorId* const vdi,
                           File* const srcfile);
        
    private:
        AstFinal* final_;
        AstItem* type_;
        AstVariableDeclaratorId* variableDeclaratorId_;
    };


    class AstFormalParameterFinal : public AstFormalParameter 
    {
    public:
        AstFormalParameterFinal(AstFinal* const f, 
                                AstItem* const t,
                                AstVariableDeclaratorId* const vdi,
                                File* const srcfile);

        virtual ~AstFormalParameterFinal();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };

    
    class AstFormalParameterList : public AstNode 
    {
    public:
        AstFormalParameterList(AstFormalParameter* const fp, File* const srcfile);

        virtual ~AstFormalParameterList();

        virtual void getProcessed(Processor* const proc);

        AstFormalParameter* getFormalParameter(int index) const;

        int getFormalParameterCount() const;

        virtual void add(AstFormalParameter* const fp);

        AstComma* getComma(int index) const;

        int getCommaCount() const;

        virtual void add(AstComma* const c);

        virtual string type() const;
        
    private:
        vector<AstFormalParameter*> formalParameters_;
        vector<AstComma*> commas_;
    };

    
    class AstFormalParameterNonFinal : public AstFormalParameter 
    {
    public:
        AstFormalParameterNonFinal(AstItem* const t, AstVariableDeclaratorId* const vdi, File* const srcfile);

        virtual ~AstFormalParameterNonFinal();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    /**
     * "goto"
     */
    class AstGoto : public AstLeaf 
    {
    public:
        AstGoto(char* position, File* const srcfile);

        virtual ~AstGoto();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * ">"
     */
    class AstGt : public AstLeaf 
    {
    public:
        AstGt(char* position, File* const srcfile);

        virtual ~AstGt();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * ">="
     */
    class AstGteq : public AstLeaf 
    {
    public:
        AstGteq(char* position, File* const srcfile);

        virtual ~AstGteq();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * A value token, whose type is IDENTIFIER.
     */
    class AstIdentifier : public AstValue
    {
    public:
        AstIdentifier(const string& text, char* position, File* const srcfile);

        virtual ~AstIdentifier();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);
    };


    /**
     * "if"
     */
    class AstIf : public AstLeaf 
    {
    public:
        AstIf(char* position, File* const srcfile);

        virtual ~AstIf();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    class AstIfThenElseStatement : public AstNode 
    {        
    public:
        AstIfThenElseStatement(AstIf* const i, 
                               AstLparen* const l,
                               AstItem* const ex,
                               AstRparen* const r,
                               AstItem* const s0,
                               AstElse* const el,
                               AstItem* const s1,
                               File* const srcfile);

        virtual ~AstIfThenElseStatement();

        virtual void getProcessed(Processor* const proc);

        AstIf* getIf() const;
        
        AstLparen* getLparen() const;

        AstItem* getExpression() const;

        AstRparen* getRparen() const;

        AstItem* getFirstStatement() const;

        AstElse* getElse() const;

        AstItem* getSecondStatement() const;

        virtual string type() const;
        
    private:
        AstIf* if_;
        AstLparen* lparen_;
        AstItem* expression_;
        AstRparen* rparen_;
        AstItem* firstStatement_;
        AstElse* else_;
        AstItem* secondStatement_;
    };

    
    class AstIfThenStatement : public AstNode 
    {        
    public:
        AstIfThenStatement(AstIf* const i, 
                           AstLparen* const l,
                           AstItem* const e,
                           AstRparen* const r,
                           AstItem* const s,
                           File* const srcfile);

        virtual ~AstIfThenStatement();

        virtual void getProcessed(Processor* const proc);

        AstIf* getIf() const;
        
        AstLparen* getLparen() const;

        AstItem* getExpression() const;

        AstRparen* getRparen() const;

        AstItem* getStatement() const;

        virtual string type() const;
        
    private:
        AstIf* if_;
        AstLparen* lparen_;
        AstItem* expression_;
        AstRparen* rparen_;
        AstItem* statement_;
    };


    /**
     * "implements"
     */
    class AstImplements : public AstLeaf 
    {
    public:
        AstImplements(char* position, File* const srcfile);

        virtual ~AstImplements();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    class AstImplementsInterfaceList : public AstNode 
    {
    public:
        AstImplementsInterfaceList(AstImplements* const i, AstNameList* const nl, File* const srcfile);

        virtual ~AstImplementsInterfaceList();

        virtual void getProcessed(Processor* const proc);

        AstImplements* getImplements() const;
        
        AstNameList* getNameList() const;

        virtual string type() const;
        
    private:
        AstImplements* implements_;
        AstNameList* nameList_;
    };


    /**
     * "import"
     */
    class AstImport : public AstLeaf 
    {
    public:
        AstImport(char* position, File* const srcfile);

        virtual ~AstImport();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    class AstImportDeclaration : public AstNode 
    {
    public:
        AstImportDeclaration(AstImport* const i, AstName* const n, AstSemicolon* const s, File* const srcfile);

        virtual ~AstImportDeclaration();

        AstImport* getImport() const;
        
        AstName* getName() const;

        AstSemicolon* getSemicolon() const;

        /**
         * Returns what was imported, as a string.
         */
        virtual string importee() const;
        
    private:
        AstImport* import_;
        AstName* name_;
        AstSemicolon* semicolon_;
    };

    
    class AstImportDeclarationList : public AstNode 
    {
    public:
        AstImportDeclarationList(AstImportDeclaration* const id, File* const srcfile);

        virtual ~AstImportDeclarationList();

        virtual void getProcessed(Processor* const proc);

        AstImportDeclaration* getImportDeclaration(int index) const;

        int getImportDeclarationCount() const;

        virtual void add(AstImportDeclaration* const id);

        virtual string type() const;
        
    private:
        vector<AstImportDeclaration*> importDeclarations_;
    };

    
    class AstImportDeclarationOnDemand : public AstImportDeclaration 
    {        
    public:
        AstImportDeclarationOnDemand(AstImport* const i, 
                                     AstName* const n,
                                     AstDot* const d,
                                     AstMult* const m,
                                     AstSemicolon* const s,
                                     File* const srcfile);

        virtual ~AstImportDeclarationOnDemand();

        virtual void getProcessed(Processor* const proc);

        AstDot* getDot() const;

        AstMult* getMult() const;

        virtual string type() const;

        /**
         * Returns what was imported, as a string.
         */
        virtual string importee() const;
        
    private:
        AstImport* import_;
        AstName* name_;
        AstDot* dot_;
        AstMult* mult_;
        AstSemicolon* semicolon_;
    };

    
    class AstImportDeclarationSingle : public AstImportDeclaration 
    {
    public:
        AstImportDeclarationSingle(AstImport* const i, AstName* const n, AstSemicolon* const s, File* const srcfile);

        virtual ~AstImportDeclarationSingle();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    /**
     * "instanceof"
     */
    class AstInstanceof : public AstLeaf 
    {
    public:
        AstInstanceof(char* position, File* const srcfile);

        virtual ~AstInstanceof();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * "int"
     */
    class AstInt : public AstIntegralType 
    {
    public:
        AstInt(char* position, File* const srcfile);

        virtual ~AstInt();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * A value token, whose type is INTEGER_LITERAL.
     */
    class AstIntegerLiteral : public AstLiteral
    {
    public:
        AstIntegerLiteral(const string& text, char* position, File* const srcfile);

        virtual ~AstIntegerLiteral();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);
    };


    /**
     * "interface"
     */
    class AstInterface : public AstLeaf 
    {
    public:
        AstInterface(char* position, File* const srcfile);

        virtual ~AstInterface();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * interface_body:
     *           LBRACE interface_member_declaration_list RBRACE
     *         | LBRACE RBRACE
     *         ;
     */
    class AstInterfaceBody : public AstNode 
    {
    public:
        virtual ~AstInterfaceBody();

        AstLbrace* getLbrace() const;

        AstInterfaceMemberDeclarationList* getInterfaceMemberDeclarationList() const;

        AstRbrace* getRbrace() const;

    protected:
        AstInterfaceBody(AstLbrace* const lb,
                         AstInterfaceMemberDeclarationList* const inmedeli,
                         AstRbrace* const rb,
                         File* const srcfile);

    private:
        AstLbrace* lbrace_;
        AstInterfaceMemberDeclarationList* interfaceMemberDeclarationList_;
        AstRbrace* rbrace_;
    };


    class AstInterfaceBodyDecls : public AstInterfaceBody 
    {
    public:
        AstInterfaceBodyDecls(AstLbrace* const lb,
                              AstInterfaceMemberDeclarationList* const inmedeli,
                              AstRbrace* const rb,
                              File* const srcfile);

        virtual ~AstInterfaceBodyDecls();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
            };


    class AstInterfaceBodyNoDecls : public AstInterfaceBody 
    {
    public:
        AstInterfaceBodyNoDecls(AstLbrace* const lb, AstRbrace* const rb, File* const srcfile);

        virtual ~AstInterfaceBodyNoDecls();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    /**
     * interface_declaration:
     *           INTERFACE IDENTIFIER interface_body
     *         | INTERFACE IDENTIFIER EXTENDS name_list interface_body
     *         | modifier_list INTERFACE IDENTIFIER interface_body
     *         | modifier_list INTERFACE IDENTIFIER EXTENDS name_list interface_body
     *         ;
     */
    class AstInterfaceDeclaration : public AstNode
    {
    public:
        virtual ~AstInterfaceDeclaration();

        AstModifierList* getModifierList() const;

        AstInterface* getInterface() const;

        AstIdentifier* getIdentifier() const;

        AstExtendsInterfaceList* getExtendsInterfaceList() const;

        AstInterfaceBody* getInterfaceBody() const;

    protected:
        AstInterfaceDeclaration(AstModifierList* const moli,
                                AstInterface* const in,
                                AstIdentifier* const id,
                                AstExtendsInterfaceList* const exinli,
                                AstInterfaceBody* const inbo,
                                File* const srcfile);
        
    private:
        AstModifierList* modifierList_;
        AstInterface* interface_;
        AstIdentifier* identifier_;
        AstExtendsInterfaceList* extendsInterfaceList_;
        AstInterfaceBody* interfaceBody_;
    };


    class AstInterfaceDeclarationModsExtends : public AstInterfaceDeclaration
    {
    public:
        AstInterfaceDeclarationModsExtends(AstModifierList* const moli,
                                           AstInterface* const in,
                                           AstIdentifier* const id,
                                           AstExtendsInterfaceList* const exinli,
                                           AstInterfaceBody* const inbo,
                                           File* const srcfile);

        virtual ~AstInterfaceDeclarationModsExtends();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstInterfaceDeclarationModsNoExtends : public AstInterfaceDeclaration
    {
    public:
        AstInterfaceDeclarationModsNoExtends(AstModifierList* const moli,
                                             AstInterface* const in,
                                             AstIdentifier* const id,
                                             AstInterfaceBody* const inbo,
                                             File* const srcfile);

        virtual ~AstInterfaceDeclarationModsNoExtends();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstInterfaceDeclarationNoModsExtends : public AstInterfaceDeclaration 
    {
    public:
        AstInterfaceDeclarationNoModsExtends(AstInterface* const in,
                                             AstIdentifier* const id,
                                             AstExtendsInterfaceList* const exinli,
                                             AstInterfaceBody* const inbo,
                                             File* const srcfile);

        virtual ~AstInterfaceDeclarationNoModsExtends();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstInterfaceDeclarationNoModsNoExtends : public AstInterfaceDeclaration
    {
    public:
        AstInterfaceDeclarationNoModsNoExtends(AstInterface* const in,
                                               AstIdentifier* const id,
                                               AstInterfaceBody* const inbo,
                                               File* const srcfile);

        virtual ~AstInterfaceDeclarationNoModsNoExtends();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstInterfaceMemberDeclarationList : public AstNode 
    {
    public:
        AstInterfaceMemberDeclarationList(AstNode* const imd, 
                                          File* const srcfile);

        virtual ~AstInterfaceMemberDeclarationList();

        virtual void getProcessed(Processor* const proc);

        AstNode* getInterfaceMemberDeclaration(int index) const;

        int getInterfaceMemberDeclarationCount() const;

        virtual void add(AstNode* const imd);

        virtual string type() const;
        
    private:
        vector<AstNode*> interfaceMemberDeclarations_;
    };


    class AstJavadocComment : public AstComment
    {
    public:
        AstJavadocComment(char* start, char* end, File* const srcfile);

        virtual ~AstJavadocComment();

        virtual vector<AstTaggedComment*> taggedComments() const;

        virtual string type() const;

        string description() const;

        char* descriptionStart() const;
    
        char* descriptionEnd() const;

        AstStringLeaf descriptionLeaf() const;

        /**
         * Invokes process on the Processor for this specific type of
         * AstNode/AstItem/AstLeaf.
         */
        virtual void getProcessed(Processor* const p);

    private:
        /**
         * The tagged comments within this Javadoc comment.
         */
        vector<AstTaggedComment*> taggedComments_;
        
        /**
         * The location of the description start.
         */
        char* descriptionStart_;
    
        /**
         * The location of the description end.
         */
        char* descriptionEnd_;
    };

    
    class AstLabeledStatement : public AstNode 
    {
    public:
        AstLabeledStatement(AstIdentifier* const i, AstColon* const c, AstItem* const s, File* const srcfile);

        virtual ~AstLabeledStatement();

        virtual void getProcessed(Processor* const proc);

        AstIdentifier* getIdentifier() const;
        
        AstColon* getColon() const;

        AstItem* getStatement() const;

        virtual string type() const;
        
    private:
        AstIdentifier* identifier_;
        AstColon* colon_;
        AstItem* statement_;
    };


    /**
     * "{"
     */
    class AstLbrace : public AstLeaf 
    {
    public:
        AstLbrace(char* position, File* const srcfile);

        virtual ~AstLbrace();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * "["
     */
    class AstLbracket : public AstLeaf 
    {
    public:
        AstLbracket(char* position, File* const srcfile);

        virtual ~AstLbracket();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    /**
     * local_variable_declaration:
     *           type variable_declarator_list
     *         | FINAL type variable_declarator_list
     *         ;
     */
    class AstLocalVariableDeclaration : public AstNode 
    {
    public:
        virtual ~AstLocalVariableDeclaration();

        AstFinal* getFinal() const;

        AstItem* getType() const;
        
        AstVariableDeclaratorList* getVariableDeclaratorList() const;

    protected:        
        AstLocalVariableDeclaration(AstFinal* const f, 
                                    AstItem* const t, 
                                    AstVariableDeclaratorList* const vdl,
                                    File* const srcfile);
        
    private:
        AstFinal* final_;
        AstItem* type_;
        AstVariableDeclaratorList* variableDeclaratorList_;
    };

    
    class AstLocalVariableDeclarationFinal : public AstLocalVariableDeclaration 
    {
    public:
        AstLocalVariableDeclarationFinal(AstFinal* const f, 
                                         AstItem* const t,
                                         AstVariableDeclaratorList* const vdl,
                                         File* const srcfile);

        virtual ~AstLocalVariableDeclarationFinal();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };

    
    class AstLocalVariableDeclarationNonFinal : public AstLocalVariableDeclaration 
    {
    public:
        AstLocalVariableDeclarationNonFinal(AstItem* const t, 
                                            AstVariableDeclaratorList* const vdl,
                                            File* const srcfile);

        virtual ~AstLocalVariableDeclarationNonFinal();

        /**
         * This method must be defined <em>by every subclass</em>. The
         * implementation should be <code>proc->process(this)</code>.
         */
        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };

    
    class AstLocalVariableDeclarationStatement : public AstNode 
    {
    public:
        AstLocalVariableDeclarationStatement(AstLocalVariableDeclaration* const lvd, 
                                             AstSemicolon* const s,
                                             File* const srcfile);

        virtual ~AstLocalVariableDeclarationStatement();

        virtual void getProcessed(Processor* const proc);

        AstLocalVariableDeclaration* getLocalVariableDeclaration() const;
        
        AstSemicolon* getSemicolon() const;

        virtual string type() const;
        
    private:
        AstLocalVariableDeclaration* localVariableDeclaration_;
        AstSemicolon* semicolon_;
    };


    /**
     * "long"
     */
    class AstLong : public AstIntegralType 
    {
    public:
        AstLong(char* position, File* const srcfile);

        virtual ~AstLong();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * "("
     */
    class AstLparen : public AstLeaf 
    {
    public:
        AstLparen(char* position, File* const srcfile);

        virtual ~AstLparen();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * <<
     */
    class AstLshift : public AstLeaf 
    {
    public:
        AstLshift(char* position, File* const srcfile);

        virtual ~AstLshift();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * <<=
     */
    class AstLshifteq : public AstAssignmentOperator 
    {
    public:
        AstLshifteq(char* position, File* const srcfile);

        virtual ~AstLshifteq();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * <
     */
    class AstLt : public AstLeaf 
    {
    public:
        AstLt(char* position, File* const srcfile);

        virtual ~AstLt();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * <=
     */
    class AstLteq : public AstLeaf 
    {
    public:
        AstLteq(char* position, File* const srcfile);

        virtual ~AstLteq();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    class AstMethodDeclaration : public AstNode 
    {
    public:
        AstMethodDeclaration(AstMethodHeader* const mehe, File* const srcfile);

        virtual ~AstMethodDeclaration();

        AstMethodHeader* getMethodHeader() const;

    private:
        AstMethodHeader* methodHeader_;
    };


    class AstMethodDeclarationBlock : public AstMethodDeclaration 
    {
    public:
        AstMethodDeclarationBlock(AstMethodHeader* const mehe, AstBlock* const bl, File* const srcfile);

        virtual ~AstMethodDeclarationBlock();

        AstBlock* getBlock() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstBlock* block_;
    };


    class AstMethodDeclarationSemicolon : public AstMethodDeclaration 
    {
    public:
        AstMethodDeclarationSemicolon(AstMethodHeader* const mehe, AstSemicolon* const se, File* const srcfile);

        virtual ~AstMethodDeclarationSemicolon();

        AstSemicolon* getSemicolon() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstSemicolon* semicolon_;
    };

    
    /**
     * method_declarator:
     *           IDENTIFIER LPAREN RPAREN
     *         | IDENTIFIER LPAREN formal_parameter_list RPAREN
     *         | method_declarator dimension // deprecated
     * 	       ;
     */
    class AstMethodDeclarator : public AstNode 
    {
    public:
        virtual ~AstMethodDeclarator();

        AstIdentifier* getIdentifier() const;
        
        AstLparen* getLparen() const;

        AstFormalParameterList* getFormalParameterList() const;

        AstRparen* getRparen() const;

        AstDimension* getDimension(int index) const;

        int getDimensionCount() const;

        virtual void add(AstDimension* const d);

    protected:
        
        AstMethodDeclarator(AstIdentifier* const i, 
                            AstLparen* const l,
                            AstFormalParameterList* const fpl,
                            AstRparen* const r,
                            File* const srcfile);
        
    private:
        AstIdentifier* identifier_;
        AstLparen* lparen_;
        AstFormalParameterList* formalParameterList_;
        AstRparen* rparen_;
        vector<AstDimension*> dimensions_;
    };

    
    class AstMethodDeclaratorNoArgs : public AstMethodDeclarator 
    {
    public:
        AstMethodDeclaratorNoArgs(AstIdentifier* const i, AstLparen* const l, AstRparen* const r, File* const srcfile);

        virtual ~AstMethodDeclaratorNoArgs();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };

    
    class AstMethodDeclaratorWArgs : public AstMethodDeclarator 
    {        
    public:
        AstMethodDeclaratorWArgs(AstIdentifier* const i, 
                                 AstLparen* const l,
                                 AstFormalParameterList* const fpl,
                                 AstRparen* const r,
                                 File* const srcfile);
        
        virtual ~AstMethodDeclaratorWArgs();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    /**
     * method_header:
     *           type method_declarator
     *         | type method_declarator throws_name_list
     *         | VOID method_declarator
     *         | VOID method_declarator throws_name_list
     *         | modifier_list type method_declarator
     *         | modifier_list type method_declarator throws_name_list
     *         | modifier_list VOID method_declarator
     *         | modifier_list VOID method_declarator throws_name_list
     *         ;
     */
    class AstMethodHeader : public AstNode 
    {
    public:
       
        virtual ~AstMethodHeader();

        AstModifierList* getModifierList() const;

        AstItem* getType() const;

        AstMethodDeclarator* getMethodDeclarator() const;

        AstThrowsNameList* getThrowsNameList() const;

        virtual string name() const;
        
    protected:
        AstMethodHeader(AstModifierList* const moli,
                        AstItem* const ty,
                        AstMethodDeclarator* const mede,
                        AstThrowsNameList* const thnali,
                        File* const srcfile);

    private:
        AstModifierList* modifierList_;
        AstItem* type_;
        AstMethodDeclarator* methodDeclarator_;
        AstThrowsNameList* throwsNameList_;
    };


    class AstMethodHeaderTypeModsNoThrows : public AstMethodHeader 
    {
    public:
        AstMethodHeaderTypeModsNoThrows(AstModifierList* const moli,
                                        AstItem* const ty,
                                        AstMethodDeclarator* const mede,
                                        File* const srcfile);

        virtual ~AstMethodHeaderTypeModsNoThrows();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstMethodHeaderTypeModsThrows : public AstMethodHeader 
    {
    public:
        AstMethodHeaderTypeModsThrows(AstModifierList* const moli,
                                      AstItem* const ty,
                                      AstMethodDeclarator* const mede,
                                      AstThrowsNameList* const thnali,
                                      File* const srcfile);

        virtual ~AstMethodHeaderTypeModsThrows();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstMethodHeaderTypeNoModsNoThrows : public AstMethodHeader 
    {
    public:
        AstMethodHeaderTypeNoModsNoThrows(AstItem* const ty, AstMethodDeclarator* const mede, File* const srcfile);

        virtual ~AstMethodHeaderTypeNoModsNoThrows();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstMethodHeaderTypeNoModsThrows : public AstMethodHeader 
    {
    public:
        AstMethodHeaderTypeNoModsThrows(AstItem* const ty,
                                        AstMethodDeclarator* const mede,
                                        AstThrowsNameList* const thnali,
                                        File* const srcfile);
        
        virtual ~AstMethodHeaderTypeNoModsThrows();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstMethodHeaderVoidModsNoThrows : public AstMethodHeader 
    {
    public:
        AstMethodHeaderVoidModsNoThrows(AstModifierList* const moli,
                                        AstVoid* const vo,
                                        AstMethodDeclarator* const mede,
                                        File* const srcfile);

        virtual ~AstMethodHeaderVoidModsNoThrows();

        AstVoid* getVoid() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstVoid* void_;
    };


    class AstMethodHeaderVoidModsThrows : public AstMethodHeader 
    {
    public:
        AstMethodHeaderVoidModsThrows(AstModifierList* const moli,
                                      AstVoid* const vo,
                                      AstMethodDeclarator* const mede,
                                      AstThrowsNameList* const thnali,
                                      File* const srcfile);

        virtual ~AstMethodHeaderVoidModsThrows();

        AstVoid* getVoid() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstVoid* void_;
    };


    class AstMethodHeaderVoidNoModsNoThrows : public AstMethodHeader 
    {
    public:
        AstMethodHeaderVoidNoModsNoThrows(AstVoid* const vo, AstMethodDeclarator* const mede, File* const srcfile);

        virtual ~AstMethodHeaderVoidNoModsNoThrows();

        AstVoid* getVoid() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstVoid* void_;
    };


    class AstMethodHeaderVoidNoModsThrows : public AstMethodHeader 
    {
    public:
        AstMethodHeaderVoidNoModsThrows(AstVoid* const vo,
                                        AstMethodDeclarator* const mede,
                                        AstThrowsNameList* const thnali,
                                        File* const srcfile);

        virtual ~AstMethodHeaderVoidNoModsThrows();

        AstVoid* getVoid() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstVoid* void_;
    };


    class AstMethodInvocation : public AstNode 
    {
    public:
        AstMethodInvocation(AstLparen* const lp, AstRparen* const rp, File* const srcfile);

        virtual ~AstMethodInvocation();

        AstLparen* getLparen() const;

        AstRparen* getRparen() const;

    private:
        AstLparen* lparen_;
        AstRparen* rparen_;
    };


    class AstMethodInvocationNameArgs : public AstMethodInvocation 
    {
    public:
        AstMethodInvocationNameArgs(AstName* const na,
                                    AstLparen* const lp,
                                    AstExpressionList* const exli,
                                    AstRparen* const rp,
                                    File* const srcfile);

        virtual ~AstMethodInvocationNameArgs();

        AstName* getName() const;

        AstExpressionList* getExpressionList() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstName* name_;
        AstExpressionList* expressionList_;
    };


    class AstMethodInvocationNameNoArgs : public AstMethodInvocation 
    {
    public:
        AstMethodInvocationNameNoArgs(AstName* const na, AstLparen* const lp, AstRparen* const rp, File* const srcfile);

        virtual ~AstMethodInvocationNameNoArgs();

        AstName* getName() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstName* name_;
    };


    class AstMethodInvocationNameSuperArgs : public AstMethodInvocation 
    {
    public:
        AstMethodInvocationNameSuperArgs(AstName* const na,
                                         AstDot* const d0,
                                         AstSuper* const su,
                                         AstDot* const d1,
                                         AstIdentifier* const id,
                                         AstLparen* const lp,
                                         AstExpressionList* const exli,
                                         AstRparen* const rp,
                                         File* const srcfile);

        virtual ~AstMethodInvocationNameSuperArgs();

        AstName* getName() const;

        AstDot* getFirstDot() const;

        AstSuper* getSuper() const;

        AstDot* getSecondDot() const;

        AstIdentifier* getIdentifier() const;

        AstExpressionList* getExpressionList() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstName* name_;
        AstDot* firstDot_;
        AstSuper* super_;
        AstDot* secondDot_;
        AstIdentifier* identifier_;
        AstExpressionList* expressionList_;
    };


    class AstMethodInvocationNameSuperNoArgs : public AstMethodInvocation 
    {
    public:
        AstMethodInvocationNameSuperNoArgs(AstName* const na,
                                           AstDot* const d0,
                                           AstSuper* const su,
                                           AstDot* const d1,
                                           AstIdentifier* const id,
                                           AstLparen* const lp,
                                           AstRparen* const rp,
                                           File* const srcfile);

        virtual ~AstMethodInvocationNameSuperNoArgs();

        AstName* getName() const;

        AstDot* getFirstDot() const;

        AstSuper* getSuper() const;

        AstDot* getSecondDot() const;
        
        AstIdentifier* getIdentifier() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstName* name_;
        AstDot* firstDot_;
        AstSuper* super_;
        AstDot* secondDot_;
        AstIdentifier* identifier_;
    };


    class AstMethodInvocationPrimary : public AstMethodInvocation 
    {
    public:
        AstMethodInvocationPrimary(AstItem* const pr,
                                   AstDot* const d,
                                   AstIdentifier* const id,
                                   AstLparen* const lp,
                                   AstRparen* const rp,
                                   File* const srcfile);

        virtual ~AstMethodInvocationPrimary();

        AstItem* getPrimary() const;

        AstDot* getDot() const;

        AstIdentifier* getIdentifier() const;

    private:
        AstItem* primary_;
        AstDot* dot_;
        AstIdentifier* identifier_;
    };


    class AstMethodInvocationPrimaryArgs : public AstMethodInvocationPrimary
    {
    public:
        AstMethodInvocationPrimaryArgs(AstItem* const pr,
                                       AstDot* const d,
                                       AstIdentifier* const id,
                                       AstLparen* const lp,
                                       AstExpressionList* const exli,
                                       AstRparen* const rp,
                                       File* const srcfile);

        virtual ~AstMethodInvocationPrimaryArgs();

        AstExpressionList* getExpressionList() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstExpressionList* expressionList_;
    };


    class AstMethodInvocationPrimaryNoArgs : public AstMethodInvocationPrimary
    {
    public:
        AstMethodInvocationPrimaryNoArgs(AstItem* const pr,
                                         AstDot* const d,
                                         AstIdentifier* const id,
                                         AstLparen* const lp,
                                         AstRparen* const rp,
                                         File* const srcfile);

        virtual ~AstMethodInvocationPrimaryNoArgs();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstMethodInvocationSuper : public AstMethodInvocation 
    {
    public:
        AstMethodInvocationSuper(AstSuper* const su,
                                 AstDot* const d,
                                 AstIdentifier* const id,
                                 AstLparen* const lp,
                                 AstRparen* const rp,
                                 File* const srcfile);

        virtual ~AstMethodInvocationSuper();

        AstSuper* getSuper() const;

        AstDot* getDot() const;

        AstIdentifier* getIdentifier() const;

    private:
        AstSuper* super_;
        AstDot* dot_;
        AstIdentifier* identifier_;
    };


    class AstMethodInvocationSuperArgs : public AstMethodInvocationSuper
    {
    public:
        AstMethodInvocationSuperArgs(AstSuper* const su,
                                     AstDot* const d,
                                     AstIdentifier* const id,
                                     AstLparen* const lp,
                                     AstExpressionList* const exli,
                                     AstRparen* const rp,
                                     File* const srcfile);

        virtual ~AstMethodInvocationSuperArgs();

        AstExpressionList* getExpressionList() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstExpressionList* expressionList_;
    };


    class AstMethodInvocationSuperNoArgs : public AstMethodInvocationSuper
    {
    public:
        AstMethodInvocationSuperNoArgs(AstSuper* const su,
                                       AstDot* const d,
                                       AstIdentifier* const id,
                                       AstLparen* const lp,
                                       AstRparen* const rp,
                                       File* const srcfile);

        virtual ~AstMethodInvocationSuperNoArgs();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    /**
     * -
     */
    class AstMinus : public AstLeaf 
    {
    public:
        AstMinus(char* position, File* const srcfile);

        virtual ~AstMinus();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * -=
     */
    class AstMinuseq : public AstAssignmentOperator 
    {
    public:
        AstMinuseq(char* position, File* const srcfile);

        virtual ~AstMinuseq();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * --
     */
    class AstMinusminus : public AstLeaf 
    {
    public:
        AstMinusminus(char* position, File* const srcfile);

        virtual ~AstMinusminus();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * %
     */
    class AstMod : public AstLeaf 
    {
    public:
        AstMod(char* position, File* const srcfile);

        virtual ~AstMod();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * %=
     */
    class AstModeq : public AstAssignmentOperator 
    {
    public:
        AstModeq(char* position, File* const srcfile);

        virtual ~AstModeq();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    class AstModifierList : public AstNode 
    {
    public:
        AstModifierList(AstLeaf* const m, File* const srcfile);

        virtual ~AstModifierList();

        virtual void getProcessed(Processor* const proc);

        AstLeaf* getModifier(int index) const;

        int getModifierCount() const;

        virtual string type() const;

        virtual void add(AstLeaf* const m);

        const vector<AstLeaf*>& getModifiers() const;

    private:
        vector<AstLeaf*> modifiers_;
    };


    /**
     * "*"
     */
    class AstMult : public AstLeaf 
    {
    public:
        AstMult(char* position, File* const srcfile);

        virtual ~AstMult();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * *=
     */
    class AstMulteq : public AstAssignmentOperator 
    {
    public:
        AstMulteq(char* position, File* const srcfile);

        virtual ~AstMulteq();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    class AstMultiplicativeExpression : public AstNode 
    {
    public:
        AstMultiplicativeExpression(AstItem* const lhex, AstItem* const rhex, File* const srcfile);

        virtual ~AstMultiplicativeExpression();

        AstItem* getLhsExpression() const;

        AstItem* getRhsExpression() const;

    private:
        AstItem* lhsExpression_;
        AstItem* rhsExpression_;
    };


    class AstMultiplicativeExpressionDiv : public AstMultiplicativeExpression 
    {
    public:
        AstMultiplicativeExpressionDiv(AstItem* const lhex, AstDiv* const di, AstItem* const rhex, File* const srcfile);

        virtual ~AstMultiplicativeExpressionDiv();

        AstDiv* getDiv() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstDiv* div_;
    };


    class AstMultiplicativeExpressionMod : public AstMultiplicativeExpression 
    {
    public:
        AstMultiplicativeExpressionMod(AstItem* const lhex, AstMod* const mo, AstItem* const rhex, File* const srcfile);

        virtual ~AstMultiplicativeExpressionMod();

        AstMod* getMod() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstMod* mod_;
    };


    class AstMultiplicativeExpressionMult : public AstMultiplicativeExpression 
    {
    public:
        AstMultiplicativeExpressionMult(AstItem* const lhex,
                                        AstMult* const mu,
                                        AstItem* const rhex,
                                        File* const srcfile);

        virtual ~AstMultiplicativeExpressionMult();

        AstMult* getMult() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstMult* mult_;
    };


    class AstName : public AstNode 
    {
    public:
        AstName(AstIdentifier* const id, File* const srcfile);

        virtual ~AstName();

        virtual string fullName() const;

        virtual string type() const;

        virtual void getProcessed(Processor* const p);
    };

    
    class AstNameDotClass : public AstNode 
    {
        typedef doctorj::AstNode SUPERCLASS;
        
    public:
        AstNameDotClass(AstName* const name, AstDot* const item1, AstClass* const cls, File* const srcfile);

        virtual ~AstNameDotClass();

        virtual void getProcessed(Processor* const proc);

        AstName* getName() const;
        
        AstDot* getDot() const;

        AstClass* getClass() const;

        virtual string type() const;
        
    private:
        AstName* name_;
        AstDot* dot_;
        AstClass* class_;
    };

    
    class AstNameDotThis : public AstNode 
    {
        typedef doctorj::AstNode SUPERCLASS;
        
    public:
        AstNameDotThis(AstName* const name, AstDot* const dot, AstThis* const ths, File* const srcfile);

        virtual ~AstNameDotThis();

        virtual void getProcessed(Processor* const proc);

        AstName* getName() const;
        
        AstDot* getDot() const;

        AstThis* getThis() const;

        virtual string type() const;
        
    private:
        AstName* name_;
        AstDot* dot_;
        AstThis* this_;
    };

    
    class AstNameList : public AstNode 
    {
    public:
        AstNameList(AstName* const n, File* const srcfile);

        virtual ~AstNameList();

        virtual void getProcessed(Processor* const proc);

        AstName* getName(int index) const;

        int getNameCount() const;

        virtual void add(AstName* const n);

        AstComma* getComma(int index) const;

        int getCommaCount() const;

        virtual void add(AstComma* const c);

        virtual string type() const;
        
    private:
        vector<AstName*> names_;
        vector<AstComma*> commas_;
    };


    /**
     * "native"
     */
    class AstNative : public AstLeaf 
    {
    public:
        AstNative(char* position, File* const srcfile);

        virtual ~AstNative();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * "new"
     */
    class AstNew : public AstLeaf 
    {
    public:
        AstNew(char* position, File* const srcfile);

        virtual ~AstNew();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    /**
     * Things that are not code (hence the name). It takes a start and end
     * position in the file, and when the components are evaluated, it parses
     * it's block within the file. This is done at evaluation, not
     * initialization (construction), since this is an expensive operation that
     * does not always/often need to be performed, since only some comments are
     * examined.
     *
     * Note that the above statement will/may no longer hold true, once we are
     * spell-checking all comments.
     */
    class AstNoncode : public AstNode
    {
    public:
        typedef doctorj::AstNode SUPERCLASS;
        
        /**
         * Represents noncode: whitespace and comments.
         */
        AstNoncode(char* start, char* end, File* const srcfile);
    
        virtual ~AstNoncode();

        virtual void getProcessed(Processor* const proc);

        virtual string text() const;

        virtual string type() const;       

        vector<AstComment*> comments() const;

        virtual const list<AstItem*>& components() const;

        virtual list<AstItem*>& components();

        virtual char* position() const;

        virtual char* endPosition() const;

    private:
        char* start_;
        char* end_;
        mutable vector<AstComment*>* comments_;
    };


    /**
     * !
     */
    class AstNot : public AstLeaf 
    {
    public:
        AstNot(char* position, File* const srcfile);

        virtual ~AstNot();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * !=
     */
    class AstNoteq : public AstLeaf 
    {
    public:
        AstNoteq(char* position, File* const srcfile);

        virtual ~AstNoteq();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * "null"
     */
    class AstNullLiteral : public AstLiteral 
    {
    public:
        AstNullLiteral(char* position, File* const srcfile);

        virtual ~AstNullLiteral();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * "|"
     */
    class AstOr : public AstLeaf 
    {
    public:
        AstOr(char* position, File* const srcfile);

        virtual ~AstOr();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    class AstOrExpression : public AstNode 
    {
    public:
        AstOrExpression(AstItem* const le, AstOr* const o, AstItem* const re, File* const srcfile);

        virtual ~AstOrExpression();

        virtual void getProcessed(Processor* const proc);

        AstItem* getLhsExpression() const;
        
        AstOr* getOr() const;

        AstItem* getRhsExpression() const;

        virtual string type() const;
        
    private:
        AstItem* lhsExpression_;
        AstOr* or_;
        AstItem* rhsExpression_;
    };


    /**
     * "|="
     */
    class AstOreq : public AstAssignmentOperator 
    {
    public:
        AstOreq(char* position, File* const srcfile);

        virtual ~AstOreq();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * "||"
     */
    class AstOror : public AstLeaf 
    {
    public:
        AstOror(char* position, File* const srcfile);

        virtual ~AstOror();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * "package"
     */
    class AstPackage : public AstLeaf 
    {
    public:
        AstPackage(char* position, File* const srcfile);

        virtual ~AstPackage();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    class AstPackageDeclaration : public AstNode 
    {
    public:
        AstPackageDeclaration(AstPackage* const p, AstName* const n, AstSemicolon* const s, File* const srcfile);

        virtual ~AstPackageDeclaration();

        virtual void getProcessed(Processor* const proc);

        AstPackage* getPackage() const;
        
        AstName* getName() const;

        AstSemicolon* getSemicolon() const;

        virtual string type() const;
        
    private:
        AstPackage* package_;
        AstName* name_;
        AstSemicolon* semicolon_;
    };


    /**
     * +
     */
    class AstPlus : public AstLeaf 
    {
    public:
        AstPlus(char* position, File* const srcfile);

        virtual ~AstPlus();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * +=
     */
    class AstPluseq : public AstAssignmentOperator 
    {
    public:
        AstPluseq(char* position, File* const srcfile);

        virtual ~AstPluseq();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * "++".
     */
    class AstPlusplus : public AstLeaf 
    {
    public:
        AstPlusplus(char* position, File* const srcfile);

        virtual ~AstPlusplus();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    class AstPostdecrementExpression : public AstNode 
    {
    public:
        AstPostdecrementExpression(AstItem* const e, AstMinusminus* const m, File* const srcfile);

        virtual ~AstPostdecrementExpression();

        virtual void getProcessed(Processor* const proc);

        AstItem* getExpression() const;
        
        AstMinusminus* getMinusminus() const;

        virtual string type() const;
        
    private:
        AstItem* expression_;
        AstMinusminus* minusminus_;
    };

    
    class AstPostincrementExpression : public AstNode 
    {
    public:
        AstPostincrementExpression(AstItem* const e, AstPlusplus* const p, File* const srcfile);

        virtual ~AstPostincrementExpression();

        virtual void getProcessed(Processor* const proc);

        AstItem* getExpression() const;
        
        AstPlusplus* getPlusplus() const;

        virtual string type() const;
        
    private:
        AstItem* expression_;
        AstPlusplus* plusplus_;
    };

    
    class AstPredecrementExpression : public AstNode 
    {
    public:
        AstPredecrementExpression(AstMinusminus* const m, AstItem* const e, File* const srcfile);

        virtual ~AstPredecrementExpression();

        virtual void getProcessed(Processor* const proc);

        AstMinusminus* getMinusminus() const;
        
        AstItem* getExpression() const;

        virtual string type() const;
        
    private:
        AstMinusminus* minusminus_;
        AstItem* expression_;
    };

    
    class AstPreincrementExpression : public AstNode 
    {
    public:
        AstPreincrementExpression(AstPlusplus* const p, AstItem* const e, File* const srcfile);

        virtual ~AstPreincrementExpression();

        virtual void getProcessed(Processor* const proc);

        AstPlusplus* getPlusplus() const;
        
        AstItem* getExpression() const;

        virtual string type() const;
        
    private:
        AstPlusplus* plusplus_;
        AstItem* expression_;
    };

    
    class AstPrimaryParenExpression : public AstNode 
    {
    public:
        AstPrimaryParenExpression(AstLparen* const l, AstItem* const e, AstRparen* const r, File* const srcfile);

        virtual ~AstPrimaryParenExpression();

        virtual void getProcessed(Processor* const proc);

        AstLparen* getLparen() const;
        
        AstItem* getExpression() const;

        AstRparen* getRparen() const;

        virtual string type() const;
        
    private:
        AstLparen* lparen_;
        AstItem* expression_;
        AstRparen* rparen_;
    };

    
    class AstPrimitiveDotClass : public AstNode 
    {
    public:
        AstPrimitiveDotClass(AstPrimitiveType* const pt, AstDot* const d, AstClass* const c, File* const srcfile);

        virtual ~AstPrimitiveDotClass();

        virtual void getProcessed(Processor* const proc);

        AstPrimitiveType* getPrimitiveType() const;
        
        AstDot* getDot() const;

        AstClass* getClass() const;

        virtual string type() const;
        
    private:
        AstPrimitiveType* primitiveType_;
        AstDot* dot_;
        AstClass* class_;
    };


    /**
     * "private"
     */
    class AstPrivate : public AstLeaf 
    {
    public:
        AstPrivate(char* position, File* const srcfile);

        virtual ~AstPrivate();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    class AstProject : public AstNode 
    {
    public:
        AstProject(AstCompilationUnit* const cu);

        AstProject(double javaVersion, const vector<char*>& files);

        virtual ~AstProject();

        virtual void getProcessed(Processor* const proc);

        AstCompilationUnit* getCompilationUnit(int index) const;

        int getCompilationUnitCount() const;

        virtual void add(AstCompilationUnit* const cu);

        virtual string type() const;

        /**
         * Processes each of the files, parsing it first.
         */
        virtual void process(const vector<char*>& files);

        /**
         * Processes a single file.
         */
        virtual void process(const string& fname);

        /**
         * Processes a single file.
         */
        virtual void process(File* const file);

        vector<File*> files() const;

    private:
        vector<AstCompilationUnit*> compilationUnits_;
        vector<File*> files_;

        /**
         * The Java version being processed, such as 1.3 or 1.4.
         */
        double javaVersion_;
    };


    /**
     * "protected"
     */
    class AstProtected : public AstLeaf 
    {
    public:
        AstProtected(char* position, File* const srcfile);

        virtual ~AstProtected();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * "public"
     */
    class AstPublic : public AstLeaf 
    {
    public:
        AstPublic(char* position, File* const srcfile);

        virtual ~AstPublic();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * "?"
     */
    class AstQuestion : public AstLeaf 
    {
    public:
        AstQuestion(char* position, File* const srcfile);

        virtual ~AstQuestion();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * "}"
     */
    class AstRbrace : public AstLeaf 
    {
    public:
        AstRbrace(char* position, File* const srcfile);

        virtual ~AstRbrace();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * "]"
     */
    class AstRbracket : public AstLeaf 
    {
    public:
        AstRbracket(char* position, File* const srcfile);

        virtual ~AstRbracket();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    class AstRelationalExpression : public AstNode 
    {
    public:
        AstRelationalExpression(AstItem* const lhex, AstItem* const rhex, File* const srcfile);

        virtual ~AstRelationalExpression();

        AstItem* getLhsExpression() const;

        AstItem* getRhsExpression() const;

    private:
        AstItem* lhsExpression_;
        AstItem* rhsExpression_;
    };


    class AstRelationalExpressionGt : public AstRelationalExpression 
    {
    public:
        AstRelationalExpressionGt(AstItem* const lhex, AstGt* const gt, AstItem* const rhex, File* const srcfile);

        virtual ~AstRelationalExpressionGt();

        AstGt* getGt() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstGt* gt_;
    };


    class AstRelationalExpressionGteq : public AstRelationalExpression 
    {
    public:
        AstRelationalExpressionGteq(AstItem* const lhex, AstGteq* const gt, AstItem* const rhex, File* const srcfile);

        virtual ~AstRelationalExpressionGteq();

        AstGteq* getGteq() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstGteq* gteq_;
    };


    class AstRelationalExpressionInstanceof : public AstRelationalExpression 
    {
    public:
        AstRelationalExpressionInstanceof(AstItem* const lhex, AstInstanceof* const in, AstItem* const rhex, File* const srcfile);

        virtual ~AstRelationalExpressionInstanceof();

        AstInstanceof* getInstanceof() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstInstanceof* instanceof_;
    };


    class AstRelationalExpressionLt : public AstRelationalExpression 
    {
    public:
        AstRelationalExpressionLt(AstItem* const lhex, AstLt* const lt, AstItem* const rhex, File* const srcfile);

        virtual ~AstRelationalExpressionLt();

        AstLt* getLt() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstLt* lt_;
    };


    class AstRelationalExpressionLteq : public AstRelationalExpression 
    {
    public:
        AstRelationalExpressionLteq(AstItem* const lhex, AstLteq* const lt, AstItem* const rhex, File* const srcfile);

        virtual ~AstRelationalExpressionLteq();

        AstLteq* getLteq() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstLteq* lteq_;
    };


    /**
     * "return"
     */
    class AstReturn : public AstLeaf 
    {
    public:
        AstReturn(char* position, File* const srcfile);

        virtual ~AstReturn();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    /**
     * return_statement:
     *           RETURN SEMICOLON
     *         | RETURN expression SEMICOLON
     *         ;
     */
    class AstReturnStatement : public AstNode 
    {
    public:
        virtual ~AstReturnStatement();

        virtual void getProcessed(Processor* const proc);

        AstReturn* getReturn() const;
        
        AstItem* getExpression() const;
        
        AstSemicolon* getSemicolon() const;

        virtual string type() const;

    protected:
        AstReturnStatement(AstReturn* const r, AstItem* const e, AstSemicolon* const s, File* const srcfile);
        
    private:
        AstReturn* return_;
        AstItem* expression_;
        AstSemicolon* semicolon_;
    };

    
    class AstReturnStatementEmpty : public AstReturnStatement 
    {
    public:
        AstReturnStatementEmpty(AstReturn* const r, AstSemicolon* const s, File* const srcfile);

        virtual ~AstReturnStatementEmpty();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };

    
    class AstReturnStatementExpr : public AstReturnStatement 
    {
    public:
        AstReturnStatementExpr(AstReturn* const r, AstItem* const e, AstSemicolon* const s, File* const srcfile);

        virtual ~AstReturnStatementExpr();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    /**
     * ")"
     */
    class AstRparen : public AstLeaf 
    {
    public:
        AstRparen(char* position, File* const srcfile);

        virtual ~AstRparen();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * ">>"
     */
    class AstRshift : public AstLeaf 
    {
    public:
        AstRshift(char* position, File* const srcfile);

        virtual ~AstRshift();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * ">>="
     */
    class AstRshifteq : public AstAssignmentOperator 
    {
    public:
        AstRshifteq(char* position, File* const srcfile);

        virtual ~AstRshifteq();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * ";"
     */
    class AstSemicolon : public AstLeaf 
    {
    public:
        AstSemicolon(char* position, File* const srcfile);

        virtual ~AstSemicolon();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    class AstShiftExpression : public AstNode 
    {
    public:
        AstShiftExpression(AstItem* const lhex, AstItem* const rhex, File* const srcfile);

        virtual ~AstShiftExpression();

        AstItem* getLhsExpression() const;

        AstItem* getRhsExpression() const;

    private:
        AstItem* lhsExpression_;
        AstItem* rhsExpression_;
    };


    class AstShiftExpressionLshift : public AstShiftExpression 
    {
    public:
        AstShiftExpressionLshift(AstItem* const lhex, AstLshift* const ls, AstItem* const rhex, File* const srcfile);

        virtual ~AstShiftExpressionLshift();

        AstLshift* getLshift() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstLshift* lshift_;
    };


    class AstShiftExpressionRshift : public AstShiftExpression 
    {
    public:
        AstShiftExpressionRshift(AstItem* const lhex, AstRshift* const rs, AstItem* const rhex, File* const srcfile);

        virtual ~AstShiftExpressionRshift();

        AstRshift* getRshift() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstRshift* rshift_;
    };


    class AstShiftExpressionUrshift : public AstShiftExpression 
    {
    public:
        AstShiftExpressionUrshift(AstItem* const lhex, AstUrshift* const ur, AstItem* const rhex, File* const srcfile);

        virtual ~AstShiftExpressionUrshift();

        AstUrshift* getUrshift() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstUrshift* urshift_;
    };


    /**
     * "short"
     */
    class AstShort : public AstIntegralType 
    {
    public:
        AstShort(char* position, File* const srcfile);

        virtual ~AstShort();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    class AstStatementExpressionList : public AstNode 
    {
    public:
        AstStatementExpressionList(AstItem* const se, File* const srcfile);

        virtual ~AstStatementExpressionList();

        virtual void getProcessed(Processor* const proc);

        AstItem* getStatementExpression(int index) const;

        int getStatementExpressionCount() const;

        virtual void add(AstItem* const se);

        AstComma* getComma(int index) const;

        int getCommaCount() const;

        virtual void add(AstComma* const c);

        virtual string type() const;
        
    private:
        vector<AstItem*> statementExpressions_;
        vector<AstComma*> commas_;
    };


    /**
     * "static"
     */
    class AstStatic : public AstLeaf 
    {
    public:
        AstStatic(char* position, File* const srcfile);

        virtual ~AstStatic();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    class AstStaticInitializer : public AstNode
    {
    public:
        AstStaticInitializer(AstStatic* const s, AstBlock* const b, File* const srcfile);
        
        virtual ~AstStaticInitializer();

        virtual void getProcessed(Processor* const proc);

        AstStatic* getStatic() const;
        
        AstBlock* getBlock() const;

        virtual string type() const;
        
    private:
        AstStatic* static_;
        AstBlock* block_;
    };


    /**
     * "strictfp"
     */
    class AstStrictfp : public AstLeaf 
    {
    public:
        AstStrictfp(char* position, File* const srcfile);

        virtual ~AstStrictfp();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * Represents a string within a file, treated as a leaf.
     */
    class AstStringLeaf : public AstLeaf 
    {
    public:
        AstStringLeaf(char* start, char* end, File* const srcfile);

        virtual ~AstStringLeaf();

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;

        virtual string str() const;

        /**
         * Returns a pointer to the last character in the string.
         */
        virtual char* endPosition() const;

        /**
         * Returns whether the given string is equal to this one.
         */
        virtual bool operator==(const string& s) const;

        /**
         * Returns whether the given string is not equal to this one.
         */
        virtual bool operator!=(const string& s) const;

        /**
         * @deprecated Only for development.
         */
        virtual string type() const;

    private:
        /**
         * Points to the last character.
         */
        char* end_;
    };


    /**
     * A value token, whose type is STRING_LITERAL.
     */
    class AstStringLiteral : public AstLiteral
    {
    public:
        AstStringLiteral(const string& text, char* position, File* const srcfile);

        virtual ~AstStringLiteral();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);
    };


    /**
     * "super"
     */
    class AstSuper : public AstLeaf 
    {
    public:
        AstSuper(char* position, File* const srcfile);

        virtual ~AstSuper();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * "switch"
     */
    class AstSwitch : public AstLeaf 
    {
    public:
        AstSwitch(char* position, File* const srcfile);

        virtual ~AstSwitch();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * switch_block:
     *           LBRACE switch_block_statement_group_list switch_label_list RBRACE
     *                 { $$ = new AstSwitchBlockGroupsLabels($1, $2, $3, $4, PCF); }
     *         | LBRACE switch_block_statement_group_list RBRACE
     *                 { $$ = new AstSwitchBlockGroups($1, $2, $3, PCF); }
     *         | LBRACE switch_label_list RBRACE
     *                 { $$ = new AstSwitchBlockLabels($1, $2, $3, PCF); }
     *         | LBRACE RBRACE
     *                 { $$ = new AstSwitchBlockEmpty($1, $2, PCF); }
     *         ;
     */
    class AstSwitchBlock : public AstNode 
    {
    public:
        virtual ~AstSwitchBlock();

        AstLbrace* getLbrace() const;

        AstSwitchBlockStatementGroupList* getSwitchBlockStatementGroupList() const;

        AstSwitchLabelList* getSwitchLabelList() const;

        AstRbrace* getRbrace() const;
        
    protected:

        AstSwitchBlock(AstLbrace* const lb,
                       AstSwitchBlockStatementGroupList* const swblstgrli,
                       AstSwitchLabelList* const swlali,
                       AstRbrace* const rb,
                       File* const srcfile);
        
    private:
        AstLbrace* lbrace_;
        AstSwitchBlockStatementGroupList* switchBlockStatementGroupList_;
        AstSwitchLabelList* switchLabelList_;
        AstRbrace* rbrace_;
    };


    class AstSwitchBlockEmpty : public AstSwitchBlock 
    {
    public:
        AstSwitchBlockEmpty(AstLbrace* const lb, AstRbrace* const rb, File* const srcfile);

        virtual ~AstSwitchBlockEmpty();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstSwitchBlockGroups : public AstSwitchBlock 
    {
    public:
        AstSwitchBlockGroups(AstLbrace* const lb,
                             AstSwitchBlockStatementGroupList* const swblstgrli,
                             AstRbrace* const rb,
                             File* const srcfile);

        virtual ~AstSwitchBlockGroups();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstSwitchBlockGroupsLabels : public AstSwitchBlock 
    {
    public:
        AstSwitchBlockGroupsLabels(AstLbrace* const lb,
                                   AstSwitchBlockStatementGroupList* const swblstgrli,
                                   AstSwitchLabelList* const swlali,
                                   AstRbrace* const rb,
                                   File* const srcfile);

        virtual ~AstSwitchBlockGroupsLabels();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstSwitchBlockLabels : public AstSwitchBlock 
    {
    public:
        AstSwitchBlockLabels(AstLbrace* const lb,
                             AstSwitchLabelList* const swlali,
                             AstRbrace* const rb,
                             File* const srcfile);

        virtual ~AstSwitchBlockLabels();

        AstSwitchLabelList* getSwitchLabelList() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstSwitchLabelList* switchLabelList_;
    };

    
    class AstSwitchBlockStatementGroup : public AstNode 
    {
    public:
        AstSwitchBlockStatementGroup(AstSwitchLabelList* const sll, AstBlockStatementList* const bsl, File* const srcfile);

        virtual ~AstSwitchBlockStatementGroup();

        virtual void getProcessed(Processor* const proc);

        AstSwitchLabelList* getSwitchLabelList() const;
        
        AstBlockStatementList* getBlockStatementList() const;

        virtual string type() const;
        
    private:
        AstSwitchLabelList* switchLabelList_;
        AstBlockStatementList* blockStatementList_;
    };

    
    class AstSwitchBlockStatementGroupList : public AstNode 
    {
    public:
        AstSwitchBlockStatementGroupList(AstSwitchBlockStatementGroup* const sbsg, File* const srcfile);

        virtual ~AstSwitchBlockStatementGroupList();

        virtual void getProcessed(Processor* const proc);

        AstSwitchBlockStatementGroup* getSwitchBlockStatementGroup(int index) const;

        int getSwitchBlockStatementGroupCount() const;

        virtual void add(AstSwitchBlockStatementGroup* const sbsg);

        virtual string type() const;
        
    private:
        vector<AstSwitchBlockStatementGroup*> switchBlockStatementGroups_;
    };


    class AstSwitchLabel : public AstNode 
    {
    public:
        AstSwitchLabel(AstColon* const co, File* const srcfile);

        virtual ~AstSwitchLabel();

        AstColon* getColon() const;

    private:
        AstColon* colon_;
    };


    class AstSwitchLabelCase : public AstSwitchLabel 
    {
    public:
        AstSwitchLabelCase(AstCase* const ca, AstItem* const ex, AstColon* const co, File* const srcfile);

        virtual ~AstSwitchLabelCase();

        AstCase* getCase() const;

        AstItem* getExpression() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstCase* case_;
        AstItem* expression_;
    };


    class AstSwitchLabelDefault : public AstSwitchLabel 
    {
    public:
        AstSwitchLabelDefault(AstDefault* const de, AstColon* const co, File* const srcfile);

        virtual ~AstSwitchLabelDefault();

        AstDefault* getDefault() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstDefault* default_;
    };

    
    class AstSwitchLabelList : public AstNode 
    {
    public:
        AstSwitchLabelList(AstSwitchLabel* const sl, File* const srcfile);

        virtual ~AstSwitchLabelList();

        virtual void getProcessed(Processor* const proc);

        AstSwitchLabel* getSwitchLabel(int index) const;

        int getSwitchLabelCount() const;

        virtual void add(AstSwitchLabel* const sl);

        virtual string type() const;
        
    private:
        vector<AstSwitchLabel*> switchLabels_;
    };

    
    class AstSwitchStatement : public AstNode 
    {        
    public:
        AstSwitchStatement(AstSwitch* const s, 
                           AstLparen* const l,
                           AstItem* const e,
                           AstRparen* const r,
                           AstSwitchBlock* const sb,
                           File* const srcfile);

        virtual ~AstSwitchStatement();

        virtual void getProcessed(Processor* const proc);

        AstSwitch* getSwitch() const;
        
        AstLparen* getLparen() const;

        AstItem* getExpression() const;

        AstRparen* getRparen() const;

        AstSwitchBlock* getSwitchBlock() const;

        virtual string type() const;
        
    private:
        AstSwitch* switch_;
        AstLparen* lparen_;
        AstItem* expression_;
        AstRparen* rparen_;
        AstSwitchBlock* switchBlock_;
    };


    /**
     * "synchronized"
     */
    class AstSynchronized : public AstLeaf 
    {
    public:
        AstSynchronized(char* position, File* const srcfile);

        virtual ~AstSynchronized();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    class AstSynchronizedStatement : public AstNode 
    {        
    public:
        AstSynchronizedStatement(AstSynchronized* const s, 
                                 AstLparen* const l,
                                 AstItem* const e,
                                 AstRparen* const r,
                                 AstBlock* const b,
                                 File* const srcfile);

        virtual ~AstSynchronizedStatement();

        virtual void getProcessed(Processor* const proc);

        AstSynchronized* getSynchronized() const;
        
        AstLparen* getLparen() const;

        AstItem* getExpression() const;

        AstRparen* getRparen() const;

        AstBlock* getBlock() const;

        virtual string type() const;
        
    private:
        AstSynchronized* synchronized_;
        AstLparen* lparen_;
        AstItem* expression_;
        AstRparen* rparen_;
        AstBlock* block_;
    };


    /**
     * A comment of the form:
     *
     *     @tag
     */
    class AstTaggedComment : public AstComment
    {
    public:
    
        friend ostream& operator<<(ostream& os, const AstTaggedComment& tc) {
            return tc.print(os);
        }
    
        AstTaggedComment(char* tagStart, char* tagEnd, File* const srcfile);

        virtual ~AstTaggedComment();

        /**
         * See the class comments.
         */
        virtual string tag() const;

        /**
         * Returns an AstStringLeaf wrapping the tag.
         */
        virtual AstStringLeaf tagLeaf() const;

        virtual char* tagStart() const;

        virtual char* tagEnd() const;

        /**
         * @deprecated Only for development.
         */
        virtual string type() const;

        /**
         * @deprecated Only for development.
         */
        virtual ostream& print(ostream& os) const;

        virtual int countTargets() const;

        /**
         * Invokes process on the Processor for this specific type of
         * AstNode/AstItem/AstLeaf.
         */
        virtual void getProcessed(Processor* const p);

    protected:

        /**
         * Passes a different value of start and end of comment to the parent
         * AstComment object. For usage only by subclasses.
         */
        AstTaggedComment(char* start, char* end,
                         char* tagStart, char* tagEnd,
                         File* const srcfile);

    private:
        char* tagStart_;

        char* tagEnd_;
    };


    /**
     * A comment of the form:
     *
     *     @tag target This is the description.
     *
     * Note that the "target" is usually just the second word of the full text. A
     * target can be more complicated, such as:
     *
     *     @see <a href="spec.html#attr">Attribute Specification</a>
     *     @see "The Java Developer's Almanac"
     */
    class AstTaggedDescribedComment : public AstTaggedComment
    {
    public:
        AstTaggedDescribedComment(char* tagStart, char* tagEnd,
                                  char* descriptionStart, char* descriptionEnd, 
                                  File* const srcfile);

        virtual ~AstTaggedDescribedComment();

        /**
         * See the class comments.
         */
        virtual string target() const;

        /**
         * See the class comments.
         */
        virtual string description() const;

        /**
         * @deprecated Only for development.
         */
        virtual string type() const;

        /**
         * @deprecated Only for development.
         */
        virtual ostream& print(ostream& os) const;

        char* descriptionStart() const;
    
        char* descriptionEnd() const;

        char* targetStart() const;
    
        char* targetEnd() const;

        /**
         * Returns an AstStringLeaf wrapping the target.
         */
        virtual AstStringLeaf targetLeaf() const;

        virtual int countTargets() const;

        /**
         * Invokes process on the Processor for this specific type of
         * AstNode/AstItem/AstLeaf.
         */
        virtual void getProcessed(Processor* const p);

    protected:

        /**
         * Parses the target within the description. Deals with HTML (href's)
         * and quoted phrases.
         */
        virtual void parse();

    private:
        char* descriptionStart_;
        char* descriptionEnd_;
        char* targetStart_;
        char* targetEnd_;
        bool isParsed_;
    };


    /**
     * "this"
     */
    class AstThis : public AstLeaf 
    {
    public:
        AstThis(char* position, File* const srcfile);

        virtual ~AstThis();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * "throw"
     */
    class AstThrow : public AstLeaf 
    {
    public:
        AstThrow(char* position, File* const srcfile);

        virtual ~AstThrow();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    class AstThrowStatement : public AstNode 
    {
    public:
        AstThrowStatement(AstThrow* const t, AstItem* const e, AstSemicolon* const s, File* const srcfile);

        virtual ~AstThrowStatement();

        virtual void getProcessed(Processor* const proc);

        AstThrow* getThrow() const;
        
        AstItem* getExpression() const;

        AstSemicolon* getSemicolon() const;

        virtual string type() const;
        
    private:
        AstThrow* throw_;
        AstItem* expression_;
        AstSemicolon* semicolon_;
    };


    /**
     * "throws"
     */
    class AstThrows : public AstLeaf 
    {
    public:
        AstThrows(char* position, File* const srcfile);

        virtual ~AstThrows();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    class AstThrowsNameList : public AstNode 
    {
    public:
        AstThrowsNameList(AstThrows* const t, AstNameList* const nl, File* const srcfile);

        virtual ~AstThrowsNameList();

        virtual void getProcessed(Processor* const proc);

        AstThrows* getThrows() const;
        
        AstNameList* getNameList() const;

        virtual string type() const;
        
    private:
        AstThrows* throws_;
        AstNameList* nameList_;
    };


    /**
     * A token without an associated value, whose type is TOKEN_IDENTIFIER.
     * @deleted - unused
     */
    // class AstTokenClass : public AstLeaf 


    /**
     * "transient"
     */
    class AstTransient : public AstLeaf 
    {
    public:
        AstTransient(char* position, File* const srcfile);

        virtual ~AstTransient();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * A value token, whose type is TRUE_LITERAL.
     */
    class AstTrueLiteral : public AstBooleanLiteral
    {
    public:
        AstTrueLiteral(char* position, File* const srcfile);

        virtual ~AstTrueLiteral();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);
    };


    /**
     * "try"
     */
    class AstTry : public AstLeaf 
    {
    public:
        AstTry(char* position, File* const srcfile);

        virtual ~AstTry();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * try_statement:
     *           TRY block catch_clause_list
     *         | TRY block catch_clause_list finally_block
     *         | TRY block finally_block
     *         ;
     */
    class AstTryStatement : public AstNode 
    {
    public:
        virtual ~AstTryStatement();

        AstCatchClauseList* getCatchClauseList() const;

        AstFinallyBlock* getFinallyBlock() const;

        AstTry* getTry() const;

        AstBlock* getBlock() const;

    protected:
        AstTryStatement(AstTry* const tr,
                        AstBlock* const bl,
                        AstCatchClauseList* const caclli,
                        AstFinallyBlock* const fibl,
                        File* const srcfile);
        
    private:
        AstTry* try_;
        AstBlock* block_;
        AstCatchClauseList* catchClauseList_;
        AstFinallyBlock* finallyBlock_;
    };


    class AstTryStatementCatches : public AstTryStatement 
    {
    public:
        AstTryStatementCatches(AstTry* const tr, AstBlock* const bl, AstCatchClauseList* const caclli, File* const srcfile);
        
        virtual ~AstTryStatementCatches();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstTryStatementCatchesFinally : public AstTryStatement 
    {
    public:
        AstTryStatementCatchesFinally(AstTry* const tr,
                                      AstBlock* const bl,
                                      AstCatchClauseList* const caclli,
                                      AstFinallyBlock* const fibl,
                                      File* const srcfile);

        virtual ~AstTryStatementCatchesFinally();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstTryStatementFinally : public AstTryStatement 
    {
    public:
        AstTryStatementFinally(AstTry* const tr, AstBlock* const bl, AstFinallyBlock* const fibl, File* const srcfile);

        virtual ~AstTryStatementFinally();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };

    
    class AstTypeDeclarationList : public AstNode 
    {
    public:
        AstTypeDeclarationList(AstItem* const td, File* const srcfile);
        
        virtual ~AstTypeDeclarationList();

        virtual void getProcessed(Processor* const proc);

        AstItem* getTypeDeclaration(int index) const;

        int getTypeDeclarationCount() const;

        virtual void add(AstItem* const td);

        virtual string type() const;
        
    private:
        vector<AstItem*> typeDeclarations_;
    };


    class AstUnaryExpression : public AstNode 
    {
    public:
        AstUnaryExpression(AstItem* const ex, File* const srcfile);

        virtual ~AstUnaryExpression();

        AstItem* getExpression() const;

    private:
        AstItem* expression_;
    };


    class AstUnaryExpressionComp : public AstUnaryExpression 
    {
    public:
        AstUnaryExpressionComp(AstComp* const co, AstItem* const ex, File* const srcfile);

        virtual ~AstUnaryExpressionComp();

        AstComp* getComp() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstComp* comp_;
    };


    class AstUnaryExpressionMinus : public AstUnaryExpression 
    {
    public:
        AstUnaryExpressionMinus(AstMinus* const mi, AstItem* const ex, File* const srcfile);

        virtual ~AstUnaryExpressionMinus();

        AstMinus* getMinus() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstMinus* minus_;
    };


    class AstUnaryExpressionNot : public AstUnaryExpression 
    {
    public:
        AstUnaryExpressionNot(AstNot* const no, AstItem* const ex, File* const srcfile);

        virtual ~AstUnaryExpressionNot();

        AstNot* getNot() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstNot* not_;
    };


    class AstUnaryExpressionPlus : public AstUnaryExpression 
    {
    public:
        AstUnaryExpressionPlus(AstPlus* const pl, AstItem* const ex, File* const srcfile);

        virtual ~AstUnaryExpressionPlus();

        AstPlus* getPlus() const;

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;

    private:
        AstPlus* plus_;
    };


    /**
     * ">>>"
     */
    class AstUrshift : public AstLeaf 
    {
    public:
        AstUrshift(char* position, File* const srcfile);

        virtual ~AstUrshift();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * ">>>="
     */
    class AstUrshifteq : public AstAssignmentOperator 
    {
    public:
        AstUrshifteq(char* position, File* const srcfile);

        virtual ~AstUrshifteq();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * variable_declarator:
     *           variable_declarator_id
     *         | variable_declarator_id EQ variable_initializer
     *       ;
     */
    class AstVariableDeclarator : public AstNode 
    {
    public:
        AstVariableDeclarator(AstVariableDeclaratorId* const vadeid, File* const srcfile);

        AstVariableDeclarator(AstVariableDeclaratorId* const vadeid, AstEq* const eq, AstItem* const vain, File* const srcfile);
        
        virtual ~AstVariableDeclarator();

        AstVariableDeclaratorId* getVariableDeclaratorId() const;

        AstEq* getEq() const;

        AstItem* getVariableInitializer() const;

        string name() const;

    private:
        AstVariableDeclaratorId* variableDeclaratorId_;
        AstEq* eq_;
        AstItem* variableInitializer_;
    };


    class AstVariableDeclaratorAssign : public AstVariableDeclarator 
    {
    public:
        AstVariableDeclaratorAssign(AstVariableDeclaratorId* const vadeid, AstEq* const eq, AstItem* const vain, File* const srcfile);

        virtual ~AstVariableDeclaratorAssign();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    /**
     *     variable_declarator_id:
     *	           IDENTIFIER
     *           | IDENTIFIER dimension_list
     *           ;
     */
    class AstVariableDeclaratorId : public AstNode 
    {
    public:
        AstVariableDeclaratorId(AstIdentifier* const id, File* const srcfile);

        AstVariableDeclaratorId(AstIdentifier* const id, AstDimensionList* const dili, File* const srcfile);

        virtual ~AstVariableDeclaratorId();

        AstIdentifier* getIdentifier() const;

        virtual string name() const;

        AstDimensionList* getDimensionList() const;

    private:
        AstIdentifier* identifier_;
        AstDimensionList* dimensionList_;
    };


    class AstVariableDeclaratorIdDimensions : public AstVariableDeclaratorId 
    {
    public:
        AstVariableDeclaratorIdDimensions(AstIdentifier* const id, AstDimensionList* const dili, File* const srcfile);

        virtual ~AstVariableDeclaratorIdDimensions();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };


    class AstVariableDeclaratorIdNoDimensions : public AstVariableDeclaratorId 
    {
    public:
        AstVariableDeclaratorIdNoDimensions(AstIdentifier* const id, File* const srcfile);

        virtual ~AstVariableDeclaratorIdNoDimensions();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };

    
    class AstVariableDeclaratorList : public AstNode 
    {
    public:
        AstVariableDeclaratorList(AstVariableDeclarator* const vd, File* const srcfile);

        virtual ~AstVariableDeclaratorList();

        virtual void getProcessed(Processor* const proc);

        AstVariableDeclarator* getVariableDeclarator(int index) const;

        int getVariableDeclaratorCount() const;

        virtual void add(AstVariableDeclarator* const vd);

        AstComma* getComma(int index) const;

        int getCommaCount() const;

        virtual void add(AstComma* const c);

        virtual string type() const;
        
    private:
        vector<AstVariableDeclarator*> variableDeclarators_;
        vector<AstComma*> commas_;
    };


    class AstVariableDeclaratorNoAssign : public AstVariableDeclarator 
    {
    public:
        AstVariableDeclaratorNoAssign(AstVariableDeclaratorId* const vadeid, File* const srcfile);

        virtual ~AstVariableDeclaratorNoAssign();

        virtual void getProcessed(Processor* const proc);

        virtual string type() const;
    };

    
    class AstVariableInitializerList : public AstNode 
    {
    public:
        AstVariableInitializerList(AstItem* const vi, File* const srcfile);

        virtual ~AstVariableInitializerList();

        virtual void getProcessed(Processor* const proc);

        AstItem* getVariableInitializer(int index) const;

        int getVariableInitializerCount() const;

        virtual void add(AstItem* const vi);

        AstComma* getComma(int index) const;

        int getCommaCount() const;

        virtual void add(AstComma* const c);

        virtual string type() const;
        
    private:
        vector<AstItem*> variableInitializers_;
        vector<AstComma*> commas_;
    };


    /**
     * "void"
     */
    class AstVoid : public AstLeaf 
    {
    public:
        AstVoid(char* position, File* const srcfile);

        virtual ~AstVoid();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    class AstVoidDotClass : public AstNode 
    {
        typedef doctorj::AstNode SUPERCLASS;
        
    public:
        AstVoidDotClass(AstVoid* const v, AstDot* const d, AstClass* const c, File* const srcfile);

        virtual ~AstVoidDotClass();

        virtual void getProcessed(Processor* const proc);

        AstVoid* getVoid() const;
        
        AstDot* getDot() const;

        AstClass* getClass() const;

        virtual string type() const;
        
    private:
        AstVoid* void_;
        AstDot* dot_;
        AstClass* class_;
    };


    /**
     * "volatile"
     */
    class AstVolatile : public AstLeaf 
    {
    public:
        AstVolatile(char* position, File* const srcfile);

        virtual ~AstVolatile();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };


    /**
     * "while"
     */
    class AstWhile : public AstLeaf 
    {
    public:
        AstWhile(char* position, File* const srcfile);

        virtual ~AstWhile();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    class AstWhileStatement : public AstNode 
    {        
    public:
        AstWhileStatement(AstWhile* const w, 
                          AstLparen* const l,
                          AstItem* const e,
                          AstRparen* const r,
                          AstItem* const s,
                          File* const srcfile);

        virtual ~AstWhileStatement();

        virtual void getProcessed(Processor* const proc);

        AstWhile* getWhile() const;
        
        AstLparen* getLparen() const;

        AstItem* getExpression() const;

        AstRparen* getRparen() const;

        AstItem* getStatement() const;

        virtual string type() const;
        
    private:
        AstWhile* while_;
        AstLparen* lparen_;
        AstItem* expression_;
        AstRparen* rparen_;
        AstItem* statement_;
    };


    /**
     * "^"
     */
    class AstXor : public AstLeaf 
    {
    public:
        AstXor(char* position, File* const srcfile);

        virtual ~AstXor();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

    
    class AstXorExpression : public AstNode 
    {
    public:
        AstXorExpression(AstItem* const le, AstXor* const x, AstItem* const re, File* const srcfile);

        virtual ~AstXorExpression();

        virtual void getProcessed(Processor* const proc);

        AstItem* getLhsExpression() const;
        
        AstXor* getXor() const;

        AstItem* getRhsExpression() const;

        virtual string type() const;
        
    private:
        AstItem* lhsExpression_;
        AstXor* xor_;
        AstItem* rhsExpression_;
    };


    /**
     * "^="
     */
    class AstXoreq : public AstAssignmentOperator 
    {
    public:
        AstXoreq(char* position, File* const srcfile);

        virtual ~AstXoreq();

        virtual string type() const;

        virtual void getProcessed(Processor* const proc);

        virtual int length() const;
    };

}

#endif //! AST_h
