#ifndef ItemDocumentationAnalyzer_h
#define ItemDocumentationAnalyzer_h

#ifndef JavadocAnalyzer_h
#include "JavadocAnalyzer.h"
#endif

#ifndef std_string
#define std_string
#include <string>
#endif

#ifndef std_vector
#define std_vector
#include <vector>
#endif

using namespace std;

namespace doctorj
{
    class AstItem;
    class AstJavadocComment;
    class AstModifierList;
    class AstNoncode;
    class AstTaggedComment;
    class ErrorInvalidJavadocTag;
    class Reporter;

    /**
     * Analyzes documentation for Java items, specifically constructors,
     * methods, fields (members), and classes.
     */
    class ItemDocumentationAnalyzer : public JavadocAnalyzer
    {
    public:
        ItemDocumentationAnalyzer(Reporter* const reporter,
                                  AstModifierList* const modifiers);

        virtual ~ItemDocumentationAnalyzer();

        /**
         * Begins the check process. Should be invoked by either the
         * constructors of concrete (and "final") subclasses, or by the client.
         */
        virtual void check();

        /**
         * Returns the noncode preceding this item. This noncode should contain
         * the Javadoc comment, if any.
         */
        virtual AstNoncode* leadingNoncode() = 0;

        /**
         * What type of object we're analyzing.
         */
        virtual string type() const = 0;

        /**
         * The thing that we're analyzing.
         */
        virtual AstItem* getSubject() = 0;

        /**
         * Checks the @deprecated tag.
         */
        virtual void checkDeprecated(AstTaggedComment* const tc);

        /**
         * Checks the @see tag.
         */
        virtual void checkSee(AstTaggedComment* const tc);

        /**
         * Checks the @since tag.
         */
        virtual void checkSince(AstTaggedComment* const tc);

        /**
         * Checks for the summary sentence.
         */
        virtual void checkSummarySentence(AstJavadocComment* const jc);

        /**
         * Checks the spelling.
         */
        virtual void checkSpelling(AstJavadocComment* const jc);
        
    protected:

        /**
         * Not useful for checking code. Only for invoking methods associated
         * with the given documentation analyzer types, not the code itself.
         */
        ItemDocumentationAnalyzer();

        /**
         * Checks the comments in the given noncode block.
         */
        virtual void checkComments(AstNoncode* const nc);

        /**
         * Checks the contents of the Javadoc comment.
         */
        virtual void checkJavadoc(AstJavadocComment* const jc);

        /**
         * Invoked to allow the subclass to handle the given tagged comment.
         */
        virtual bool handleTag(AstTaggedComment* const tc);

        /**
         * Checks to see if the tag is a valid Javadoc tag, and produces an
         * error message.
         */
        virtual void checkUnknownTag(AstTaggedComment* const tc);

        /**
         * Invoked at the end of processing the Javadoc comments, in order to
         * see what code was not documented. A no-op at this level.
         */
        virtual void checkUndocumented();

        /**
         * Complain that the subject is not documented.
         */
        virtual void complainUndocumented();

        /**
         * Complain that the subject is not commented.
         */
        virtual void complainUncommented();

        /**
         * Creates the error for the usage of an invalid tag with an item of
         * this type.
         */
        virtual ErrorInvalidJavadocTag* makeUnknownTagError(AstItem* const item,
                                                            const string& tag) const = 0;

        /**
         * Checks the tag to see if it is misspelled. Returns whether the tag
         * was processed as being a misspelling of a valid tag.
         */
        virtual bool checkTagMisspelling(AstTaggedComment* const tc, const string& tag) const;

    };
}

#endif //! ItemDocumentationAnalyzer_h
