#ifndef CommentSpellCheck_h
#define CommentSpellCheck_h

#ifndef SpellChecker_h
#include "SpellChecker.h"
#endif

#ifndef std_string
#define std_string
#include <string>
#endif

using namespace std;

namespace doctorj
{
    /**
     * Checks description sections for spelling. Skips "<pre>" and "<code>"
     * sections.
     */
    class CommentSpellCheck
    {
    public:
        static CommentSpellCheck* get();

        virtual ~CommentSpellCheck();
        
        /**
         * Checks each word in the description, if it was able to load a
         * dictionary.
         */
        virtual void check(const string& description);

        /**
         * Called when a word has been misspelled.
         */
        virtual void wordMisspelled(const string& word, 
                                    int position, 
                                    const multimap<int, string>& nearMatches);

        /**
         * Adds the given dictionary.
         */
        virtual bool addDictionary(const string& dictName);

        /**
         * Adds the given word.
         */
        virtual void addWord(const string& word);
        
    protected:
        /**
         * Checks each word in the description, if it was able to load a
         * dictionary.
         */
        virtual void checkWord(const string& word, int position);

        /**
         * Actually runs the check on the description, without determining if a
         * dictionary had been loaded. Compare to <code>check</code>.
         */
        virtual void runCheck(const string& desc);

        /**
         * Goes to the next word, skipping the text that isn't checkable (such
         * as links, code and pre sections, and HTML).
         */
        virtual void skipToWord();

        /**
         * Skips a section between HTML tags, such as "<code>".
         */
        virtual void skipSection(const string& section);

        /**
         * Skips whitespace.
         */
        virtual void skipBlanks();

        /**
         * Skips a link, like: {@link }.
         */
        virtual void skipLink();

        /**
         * If the description is at the given string, it is advanced past and
         * true is returned. Else, false is returned. Any leading blanks are
         * skipped.
         */
        virtual bool consume(const string& what);

        /**
         * Goes through the description up through the given string.
         */
        virtual void consumeTo(const string& what);

        /**
         * Checks the current word, assuming that it passes the tests for
         * check-ability.
         */
        virtual void checkCurrentWord();

        /**
         * Goes until it reaches the end of the description, or the first
         * whitespace.
         */
        virtual void skipThroughWord();
        
        static CommentSpellCheck* instance_;

        CommentSpellCheck();

    private:
        NoCaseSpellChecker checker_;

        bool doCheck_;

        /**
         * The current description we're working on.
         */
        string desc_;

        /**
         * The length of the current description.
         */
        int len_;

        /**
         * The current position within the description.
         */
        int pos_;

    };
}

#endif //! CommentSpellCheck_h
