#ifndef Reporter_h
#include "Reporter.h"
#endif

#ifndef AST_h
#include "AST.h"
#endif

#ifndef Error_h
#include "Error.h"
#endif

#ifndef iterext_h
#include "iterext.h"
#endif

#ifndef File_h
#include "File.h"
#endif

#ifndef std_algorithm
#define std_algorithm
#include <algorithm>
#endif

#ifndef Log_h
#include "Log.h"
#endif

using namespace std;
using namespace doctorj;

static bool byLocation(const ReportedError& a, const  ReportedError& b)
{
    return a.startPosition < b.startPosition;
}

Reporter::Reporter(ostream& os) : os_(os), buffered_(true)
{
}

Reporter::~Reporter()
{
    if (errors_.size() > 0) {
        cerr << "WARNING: Reporter has unflushed error messages" << endl;
    }
}

ostream& Reporter::stream()
{
    return os_;
}

void Reporter::setUnbuffered()
{
    buffered_ = false;
}

void Reporter::report(AstItem* const item, const string& errorType, const string& msg)
{
    File* file       = item->sourceFile();

    char* stpos      = item->position();
    int   stln       = file->lineOf(stpos);
    char* start      = file->startPosition(stln);
        
    char* endpos     = item->endPosition();
    int   endln      = file->lineOf(endpos);
    char* end        = file->startPosition(endln);

    if (buffered_) {
        ReportedError re(errorType, msg, file, stpos, stln, start, endpos, endln, end);
        errors_.push_back(re);
    }
    else {
        ReportedError re(errorType, msg, file, stpos, stln, start, endpos, endln, end);
        writeError(re);
    }
}

void Reporter::report(Error const* err)
{
    LOG(2, "");
    AstItem* it      = err->item();
    string   errCode = string("Error<") + err->id() + ">";
    string   msg     = err->message();
    report(it, errCode, msg);
}

void Reporter::flush()
{
    stable_sort(errors_.begin(), errors_.end(), byLocation);

    EACH(vector<ReportedError>, errors_, eit) {
        ReportedError re = *eit;
        writeError(re);
    }
    clearErrors();
}

const vector<ReportedError>& Reporter::getErrors() const
{
    return errors_;
}

void Reporter::clearErrors()
{
    LOG(2, "");
    errors_.clear();
}
