#ifndef App_h
#include "App.h"
#endif

#ifndef AST_h
#include "AST.h"
#endif

#ifndef File_h
#include "File.h"
#endif

#ifndef UserPreferences_h
#include "UserPreferences.h"
#endif

#ifndef ErrorRegistry_h
#include "ErrorRegistry.h"
#endif

#ifndef Yagol_h
#include "Yagol.h"
#endif

#ifndef iterext_h
#include "iterext.h"
#endif

#ifndef Debug_h
#include "Debug.h"
#endif

#ifndef DebugOptions_h
#include "DebugOptions.h"
#endif

#ifndef Platform_h
#include "Platform.h"
#endif

#ifndef ErrorSuppressionOptionCallback_h
#include "ErrorSuppressionOptionCallback.h"
#endif

#ifndef ErrorUnsuppressionOptionCallback_h
#include "ErrorUnsuppressionOptionCallback.h"
#endif

#ifndef ErrorEnableRuleOptionCallback_h
#include "ErrorEnableRuleOptionCallback.h"
#endif

#ifndef ErrorDisableRuleOptionCallback_h
#include "ErrorDisableRuleOptionCallback.h"
#endif

#ifndef Log_h
#include "Log.h"
#endif

#ifndef std_stdlib_h
#define std_stdlib_h
#include <stdlib.h>
#endif

#ifndef std_stdio_h
#define std_stdio_h
#include <stdio.h>
#endif

#ifndef std_fstream
#define std_fstream
#include <fstream>
#endif

#ifndef std_algorithm
#define std_algorithm
#include <algorithm>
#endif

using namespace doctorj;
using namespace yagol;

static DebugOptionRegistrant tkdebug('t', "Tokenizing (scanning)");
static DebugOptionRegistrant prdebug('p', "Java parsing");
static DebugOptionRegistrant ncdebug('n', "Noncode (comments, whitespace)");
static DebugOptionRegistrant jddebug('P', "Javadoc parsing");
static DebugOptionRegistrant bcdebug('b', "Byte code (decompiling)");
static DebugOptionRegistrant ygdebug('y', "Yagol (option processing)");
static DebugOptionRegistrant ytdebug('u', "Utilities");


App::App(const string& name, const string& desc)
        : name_(name), 
     description_(desc),
     showVersion_(false),
     acted_(false)
{
    // anything that looks like a tag is treated as such; we don't expect
    // Java files to begin with '--':
    bool strictArgs = true;

    // if something goes wrong, produce the help page:
    bool helpOnError = true;

//     cout << "APPLI  building options" << endl;

    options_ = new AppOptionSet(strictArgs, 
                                helpOnError, 
                                name_, 
                                description_,
                                string("[options] file..."));

//     cout << "APPLI  adding config files" << endl;
    
    // configuration files
    options_->addConfigFile("~/.doctorjrc");
    options_->addConfigFile(".doctorjrc");

//     cout << "APPLI  adding environment variables" << endl;
    
    // environment variables
    options_->addEnvironmentVariable("DOCTORJOPT");

//     cout << "APPLI  adding option" << endl;
    
    Option* veropt = options_->addOption("version",
                                         "Displays the version and exits.",
                                         &showVersion_);
    veropt->setCommandLineOnly(true);
        
//     cout << "APPLI  adding second option" << endl;
    
    // the codes to be described
    Option* expopt = options_->addOption("explain", 
                                         "Error and warning codes to be explained in detail; 'all' shows all.", 
                                         &codes_);
    expopt->setCommandLineOnly(true);
    
//     cout << "APPLI  getting user preferences" << endl;
    
    UserPreferences* prefs = UserPreferences::get();

//     cout << "APPLI  adding tab width" << endl;   

    options_->addOption("tabwidth",
                        "Number of spaces per tab.",
                        &(prefs->tabWidth));

//     cout << "APPLI  adding debug option" << endl;

    Option* dbgopt = options_->addOption("debug",
                                         "Perform advanced diagnostic/debugging. --debug ? shows options.",
                                         &(prefs->debug));
    dbgopt->setCommandLineOnly(true);

//     cout << "APPLI  getting environment variable" << endl;   

    // EMACS means IDE mode (no context). Do other IDEs export a variable like
    // this?
    if (Platform::getEnv("EMACS")) {
        // we are in emacs
        prefs->contextOutput = false;
    }
    
    options_->addOption("context",
                        "Report using long format. \"--nocontext\" == Emacs/IDE format (file:line: error).",
                        &(prefs->contextOutput));

    options_->addOption("source",
                        "Version of Java source, such as 1.3 or 1.4.",
                        &(prefs->javaVersion));

//     cout << "APPLI  adding suppression callback" << endl;   

    // suppression

    suppressionCB_ = new ErrorSuppressionOptionCallback();

    options_->addOptionCb("suppress",
                          "Errors and warnings to be suppressed; 'all' for all.",
                          suppressionCB_);

//     cout << "APPLI  adding UNsuppression callback" << endl;   

    // unsuppression

    unsuppressionCB_ = new ErrorUnsuppressionOptionCallback();

    options_->addOptionCb("unsuppress",
                          "Errors and warnings to be unsuppressed; 'all' for all.",
                          unsuppressionCB_);

    // Yagol processes options in the order they were added. Thus, enable and
    // disable have to follow suppress and unsuppress

//     cout << "APPLI  adding disable callback" << endl;

    // disable

    disableCB_ = new ErrorDisableRuleOptionCallback();

    Option* disopt = options_->addOptionCb("disable",
                                           "Disable an error rule from being checked.",
                                           disableCB_);
    // Not sure if I want this only on the command line.
    //     disopt->setCommandLineOnly(true);

//     cout << "APPLI  adding enable callback" << endl;

    // enable
    
    enableCB_ = new ErrorEnableRuleOptionCallback();
    
    Option* enbopt = options_->addOptionCb("enable",
                                           "Enable an error rule to be checked.",
                                           enableCB_);
    // Not sure if I want this only on the command line.
    //     enbopt->setCommandLineOnly(true);

//     cout << "APPLI  adding fix option" << endl;

    // fixing
    options_->addOption("fix",
                        "whether to fix errors. See ",
                        &(prefs->fix));

    Option* fbsopt = options_->addOption("fix.backup.suffix",
                                         "The suffix to be used for backup files. Ignored if fix.backup.directory is set.",
                                         &(prefs->backupSuffix));
    fbsopt->setConfigOnly(true);

    Option* fbdopt = options_->addOption("fix.backup.directory",
                                         "The directory that original files will be copied to before changed.",
                                         &(prefs->backupDirectory));
    fbdopt->setConfigOnly(true);

    Option* ostopt = options_->addOption("fix.ostype",
                                         "Operating system type of output files. Used to determine the end of line sequence.",
                                         &(prefs->eolnType));
    ostopt->setConfigOnly(true);

//     cout << "APPLI  complete." << endl;
}

App::~App()
{
    delete suppressionCB_;
    delete unsuppressionCB_;
    delete enableCB_;
    delete disableCB_;
}

void App::buildOptions()
{
    addOptions(options_);
}

void App::addOptions(AppOptionSet* const opts)
{
}

void App::readCommandLine(int argc, char** argv)
{
//     cout << "APPLI  ::readCommandLine()" << endl;
    // process the command line
    try {
        options_->process(argc, argv);
    }
    catch (const yagol::Exception& e) {
        // Something bad happened. Dump the usage statement:
        options_->writeUsage(cout);
    }
//     cout << "APPLI  readCommandLine() done." << endl;
}

void App::displayVersion() 
{
    cout << PACKAGE << ", version " << VERSION;
#ifdef DEBUGGING
    cout << " (debugging enabled)";
#else
    cout << " (debugging disabled)";
#endif

    cout << endl;
    cout << "Written by Jeff Pace (jpace@doctorj.org)." << endl;
    cout << "Released under the Lesser GNU Public License." << endl;
    acted_ = true;
}

void App::handleDebugOptions()
{
    UserPreferences* prefs = UserPreferences::get();
    if (prefs->debug.length() == 1 && prefs->debug[0] == '?') {
        cout << "            Sets debugging flags, if compliled with DEBUGGING. As an" << endl;
        cout << "            alternative, specify a number instead of list of" << endl;
        cout << "            letters." << endl;
        cout << "" << endl;
        DebugOptions::get()->writeAll(cout);
    }
    else if (prefs->debug.length() > 0) {
        DebugOptions* opts = DebugOptions::get();
        opts->set(prefs->debug);
        Log::setEnabled();
    }
}

void App::handleVersion()
{
    if (showVersion_) {
        displayVersion();
    }
}

void App::handleExplainedErrors()
{
    if (codes_.size() == 1 && codes_[0] == "all") {
        ErrorRegistry::describeAll(cout);
        cout << endl;
        acted_ = true;
    }
    else {
        // what do they want to know about?
        EACHC(vector<string>, codes_, cit) {
            string code = *cit;
            try {
                ErrorRegistry::describe(cout, code);
                cout << "-------------------------" << endl;
                cout << endl;
            }
            catch (const NoSuchErrorCodeException& e) {
                cout << "ERROR: no such error code '" << code << "'" << endl;
            }
            acted_ = true;
        }
    }
}

void App::processFiles()
{
    vector<char*> unused = options_->getUnprocessedArgs();

    // last call for option wanna-bees (Yagol doesn't handle single-char options
    // (yet)).

    string dashv("-v");
    vector<char*>::const_iterator dashvpos = find(unused.begin(), unused.end(), dashv);

    if (dashvpos != unused.end()) {
        displayVersion();
        return;
    }

    if (unused.size() > 0) {
        UserPreferences* prefs = UserPreferences::get();
        string backupDirectory = prefs->backupDirectory;
        string backupSuffix = prefs->backupSuffix;
        bool usingBackupDirectory = backupDirectory.length() > 0;
        string eoln;
        string eolnType = StringUtilities::toLower(prefs->eolnType);
        if (eolnType == "unix") {
            eoln = "\n";
        }
        else if (eolnType == "dos") {
            eoln = "\r\n";
        }
        else if (eolnType == "mac") {
            eoln = "\r";
        }
        else {
            eoln = "\n";
        }
        string cwd = Platform::getCurrentDirectory();

        AstProject project(prefs->javaVersion, unused);
        execute(&project);

        if (prefs->fix) {
            vector<File*> files = project.files();
            vector<File*>::iterator it   = files.begin();
            vector<File*>::iterator stop = files.end();
            while (it != stop) {
                File* file = *it;
                if (file->changed()) {
                    cout << "file " << file->name() << " changed" << endl;

                    if (usingBackupDirectory) {
                        string fname = file->name();
                        if (StringUtilities::startsWith(fname, cwd)) {
                            fname = fname.substr(cwd.length());
                        }
                        OutFile outfile(backupDirectory + "/" + fname, file->originalStr(), eoln);
                        outfile.write();
                    }
                    else {
                        Platform::renameFile(file->name(), file->name() + backupSuffix);
                        file->write();
                    }
                }
                else {
                    cout << file->name() << ": no changes" << endl;
                }
                ++it;
            }
        }
    }
    else if (!acted_ && !options_->showedHelp()) {
        // complain only if they didn't do anything meaningful with the options.
        cerr << name_ << ": no input file(s)" << endl;
    }
}

void App::run(int argc, char** argv)
{
    readCommandLine(argc, argv);
    handleVersion();
    handleDebugOptions();
    handleExplainedErrors();

    if (argc <= 1) {
        cerr << name_ << ": no input file(s)" << endl;
    }
    else {
        processFiles();
    }
}

void App::addContextOption()
{
    // ensure that this isn't here already...
    const string ctx("context");
    if (!options_->findFirstByName(ctx)) {
        UserPreferences* prefs = UserPreferences::get();
        options_->addOption(ctx,
                            "show context around reported errors",
                            &(prefs->contextOutput));
    }
}
