/*
 * DiaSCE is a code editor for C and C++.
 * Copyright (C) 2000  Ander Lozano Prez
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Ander Lozano Prez
 * c/Juan de Gardeazabal 4, 1 D
 * 48004 Bilbao
 * Vizcaya
 * Spain
 *
 * ander1@wanadoo.es
 */

#include "main.h"

//*******************************************************************

/*******************************************************************************
 Realiza una compilacion del archivo actual para ver si tiene errores de sintaxis
 *******************************************************************************/
void comp_compilar_actual(void)
{
	gchar *opciones;
	gchar *nombre;
	gchar *comando;
	guint tipo;

	if (edit_actual!=NULL) {
		edit_guardar();
		nombre=edit_actual->nombre;
		tipo=gen_tipo_archivo(nombre);
		if (tipo==1) {
			opciones=pro_opciones_compilacion();
			nombre=pro_nombre_completo_archivo(nombre,TRUE, LOCALE);
			comando=g_strdup_printf("%s %s 2> /dev/stdout\n",opciones,nombre);
			gen_comando(comando,NULL);
			g_print(_("\nDone\n"));
			g_free(nombre);
			g_free(opciones);
			g_free(comando);
		}
	}
}

/*******************************************************************************
 Ejecuta el make
 *******************************************************************************/
gboolean comp_run_make(void)
{
	gchar *make;
	int warnings,errores;
	gboolean ret;
	gchar *nombre_proyecto,*ejecutable;
	gchar *opciones_compilacion;
	xmlNodePtr nodoxml;
	gchar *depuracion,*optimizacion,*electric_fence;

	make=pro_nombre_completo_archivo("Makefile",TRUE, UTF8);
	if (!gen_existe_fichero(make)) {
		gen_comando("./configure 2> /dev/stdout\n",NULL);
	}
	g_free(make);
	edit_guardar_todo();
	gen_limpiar_mensajes();
	nodoxml=buscar_nodo_xml(proyecto.xml->xmlRootNode,"compilar");
	depuracion=xmlGetProp(nodoxml,"depuracion");
	optimizacion=xmlGetProp(nodoxml,"optimizacion");
	electric_fence=xmlGetProp(nodoxml,"electric_fence");
	switch (electric_fence[0]) {
		case '0':
			g_free(electric_fence);
			electric_fence=g_strdup(_("off"));
			break;
		case '1':
			g_free(electric_fence);
			electric_fence=g_strdup(_("on"));
			break;
	}
	if (depuracion[0]=='0') {
		g_free(depuracion);
		depuracion=g_strdup(_("off"));
	}
	if (optimizacion[0]=='0') {
		g_free(optimizacion);
		optimizacion=g_strdup(_("off"));
	}
	opciones_compilacion=g_strdup_printf(_("Debug level: %s --- Optimization level: %s --- Electric-Fence: %s\n\n"),depuracion,optimizacion,electric_fence);
	g_free(depuracion);
	g_free(optimizacion);
	g_free(electric_fence);
	nombre_proyecto=pro_nombre(UTF8);
	ejecutable=pro_ejecutable(UTF8);
	g_print(_("---------------------------- Building %s ----------------------------\n"),nombre_proyecto);
	g_print(opciones_compilacion);
	g_print(_("Checkin build scripts...\n"));
	ret=comp_comando_make(&warnings,&errores);
	g_print(_("\n%s --- %i warning(s) --- %i error(s).\n"),ejecutable,warnings,errores);
	g_free(opciones_compilacion);
	g_free(nombre_proyecto);
	g_free(ejecutable);
	return ret;
}

/*******************************************************************************
 Muestra las salidas del make en la ventana de mensajes
 *******************************************************************************/
gboolean comp_comando_make(int *warnings, int *errores)
{
	FILE *tuberia;
	gchar letra;
	gchar *cadena;
	gchar *temporal;
	guint mas;
	gchar *archivo;
	gboolean primera_compilacion;

	DEBUG_MSG(->comp_comando_make);
	primera_compilacion=TRUE;
	*warnings=0;
	*errores=0;
	tuberia=popen("make 2>/dev/stdout","r");
	temporal=NULL;
	cadena=NULL;
	do {
		gen_refrescar();
		mas=fread(&letra,1,1,tuberia);
		if (mas) {
			if (temporal!=NULL) {
				cadena=g_strdup_printf("%s%c",temporal,letra);
				g_free(temporal);
				temporal=cadena;
			} else {
				cadena=g_strdup_printf("%c",letra);
				temporal=cadena;
			}
			if (letra=='\n') {
				if (comp_es_error(cadena)) {
					g_printerr("\t%s",cadena);
					(*errores)++;
				} else {
					if (comp_es_warning(cadena)) {
						g_printerr("\t%s",cadena);
						(*warnings)++;
					} else {
						archivo=comp_es_compilacion(cadena);
						if (archivo!=NULL) {
							if (primera_compilacion) {
								primera_compilacion=FALSE;
								g_print(_("\nCompiling...\n"));
							}
							g_print("%s\n",archivo);
						} else {
							if (comp_es_lincado(cadena)) {
								g_print(_("\nLinking...\n"));
							} else {
								if (!comp_es_make(cadena)) {
									if (comp_es_casi_error(cadena)) {
										g_print("\t%s",cadena);
									} else {
										g_print("%s",cadena);
									}
								}
							}
						}
						g_free(archivo);
					}
				}
				g_free(cadena);
				temporal=NULL;
				cadena=NULL;
			}
		}
	} while (mas);	
	if (cadena!=NULL) {
		g_free(cadena);
	}

	pclose(tuberia);
	DEBUG_MSG(<-comp_comando_make);
	if ((*errores)!=0) return TRUE;
	return FALSE;
}

/*******************************************************************************
 Indica si una de las lineas devueltas por el make es una linea de warning
 *******************************************************************************/
gboolean comp_es_warning(gchar *salida)
{
	int cont,dos_puntos,cont1,cont2,cont3;
	gboolean es_warning;
	
	cont=0;
	cont1=0;
	cont2=0;
	cont3=0;
	dos_puntos=0;
	while (salida[cont]!='\n') {
		if (salida[cont]==':') {
			dos_puntos++;
			if (dos_puntos==1) cont1=cont;
			if (dos_puntos==2) cont2=cont;
			if (dos_puntos==3) cont3=cont;
		}
		cont++;
	}

	es_warning=FALSE;
	if (dos_puntos==3) {
		es_warning=TRUE;
		for (cont=cont1+1;cont<cont2;cont++) {
			if ((salida[cont]<'0') || (salida[cont]>'9')) es_warning=FALSE;
		}
		if (cont3-cont2!=9)  {
			es_warning=FALSE;
		} else {
			if (salida[cont2+2]!='w') es_warning=FALSE;
		}
		
	};
	return es_warning;
}

/*******************************************************************************
 Indica si una de las salidas del make es una linea de ejecucion del gcc
 para compilar un archivo. En caso de que lo sea devuelve el nombre
 del archivo
 *******************************************************************************/
gchar *comp_es_compilacion(gchar *salida)
{
	gchar *archivo;
	int size,cont;
	gchar compilador[3];
	gchar *lenguaje;
	
	archivo=NULL;
	lenguaje=pro_lenguaje();
	switch (lenguaje[0]) {
		case '0':
			compilador[0]='g';
			compilador[1]='c';
			compilador[2]='c';
			break;
		case '1':
			compilador[0]='c';
			compilador[1]='+';
			compilador[2]='+';
			break;
	}
	g_free(lenguaje);
	for (size=0;salida[size]!='\n';size++);
	if (size<=4) return NULL;
	if ((salida[0]==compilador[0]) && (salida[1]==compilador[1]) && (salida[2]==compilador[2]) && (salida[3]==' ')) {
		for (cont=size;salida[cont-1]!=' ';cont--);
		archivo=g_strndup(salida+cont,size-cont);
		if (!pro_existe_archivo(archivo)) {
			g_free(archivo);
			archivo=NULL;
		}
	}
	return archivo;
}

/*******************************************************************************
 Indica si la linea devuelta por el make es la linea de lincado
 *******************************************************************************/
gboolean comp_es_lincado(gchar *salida)
{
	gchar *archivo;
	int size,cont;
	gchar lincador[3];
	gchar *lenguaje;
	
	archivo=NULL;
	lenguaje=pro_lenguaje();
	switch (lenguaje[0]) {
		case '0':
			lincador[0]='g';
			lincador[1]='c';
			lincador[2]='c';
			break;
		case '1':
			lincador[0]='c';
			lincador[1]='+';
			lincador[2]='+';
			break;
	}
	g_free(lenguaje);
	for (size=0;salida[size]!='\n';size++);
	if (size<=4) return FALSE;
	if ((salida[0]==lincador[0]) && (salida[1]==lincador[1]) && (salida[2]==lincador[2]) && (salida[3]==' ')) {
		for (cont=size;salida[cont-1]!=' ';cont--);
		archivo=g_strndup(salida+cont,size-cont);
		if (!pro_existe_archivo(archivo)) {
			g_free(archivo);
			archivo=NULL;
			return TRUE;
		}
	}
	return FALSE;
}

/*******************************************************************************
 Indica si la linea devuelta por el make es un mensaje del propio make
 *******************************************************************************/
gboolean comp_es_make(gchar *salida)
{
	int size;
	
	for (size=0;salida[size]!='\n';size++);
	if (size<=4) return FALSE;
	if (((salida[0]=='m') || (salida[0]=='M')) && (salida[1]=='a') && (salida[2]=='k') && ((salida[3]=='e') || (salida[3]=='i'))) {
		return TRUE;
	}
	return FALSE;
}

/*******************************************************************************
 Indica si la linea devuelta por el make es una linea de las que preceden
 a la linea que contiene el error o una de las que va a continuacion
 *******************************************************************************/
gboolean comp_es_casi_error(gchar *salida)
{
	int cont,dos_puntos,cont1,cont2;
	gboolean es_casi_error;
	
	cont=0;
	cont1=0;
	cont2=0;
	dos_puntos=0;
	while (salida[cont]!='\n') {
		if (salida[cont]==':') {
			dos_puntos++;
			if (dos_puntos==1) cont1=cont;
			if (dos_puntos==2) cont2=cont;
		}
		cont++;
	}

	es_casi_error=FALSE;
	if (dos_puntos==2) {
		if (salida[cont2+1]=='\n') {
			es_casi_error=TRUE;
		}
		if (salida[cont2+2]=='(') {
			es_casi_error=TRUE;
		}
		if ((salida[cont-1]==')') && (salida[cont-2]=='.')) {
			es_casi_error=TRUE;
		}
	};
	return es_casi_error;
}

/*******************************************************************************
 Indica si la linea devuelta por el make es una linea de error
 *******************************************************************************/
gboolean comp_es_error(gchar *salida)
{
	int cont,dos_puntos,cont1,cont2;
	gboolean es_error;
	
	cont=0;
	cont1=0;
	cont2=0;
	dos_puntos=0;
	while (salida[cont]!='\n') {
		if (salida[cont]==':') {
			dos_puntos++;
			if (dos_puntos==1) cont1=cont;
			if (dos_puntos==2) cont2=cont;
		}
		cont++;
	}

	es_error=FALSE;
	if (dos_puntos>=2) {
		es_error=TRUE;
		if ((salida[cont-1]==')') && (salida[cont-2]=='.')) es_error=FALSE;
		for (cont=cont1+1;cont<cont2;cont++) {
			if ((salida[cont]<'0') || (salida[cont]>'9')) es_error=FALSE;
		}
		if (salida[cont2+1]=='\n') es_error=FALSE;
		if (salida[cont2+2]=='(') es_error=FALSE;
		if ((dos_puntos>2) && (salida[cont2+2]=='w')) es_error=FALSE;
	};
	return es_error;
}

void comp_update_pot_file(void)
{
	gchar *comando;
	gchar *nombre;
	gchar *dir;
	gchar *libs;
	
	
	libs=pro_librerias_lincado();
	if (libs[LGETTEXT]=='0') {
		g_free(libs);
		return;
	}
	g_free(libs);
	
	dir=pro_prodir(LOCALE);
	chdir(dir);
	chdir("po");
	
	nombre=pro_ejecutable(LOCALE);
	comando=g_strdup_printf("make %s.pot 2>/dev/stdout\n",nombre);
	
	gen_comando(comando,_("---------------------------------- Updating .pot file ----------------------------------\n\n"));
	g_print(_("\nDone\n"));
	
	g_free(comando);
	g_free(nombre);
	g_free(dir);
}
