#!/usr/bin/python
# -*- coding: utf-8 -*-
"""Tests for the file system implementation using pyfvde."""

import os
import unittest

from dfvfs.path import fvde_path_spec
from dfvfs.path import os_path_spec
from dfvfs.path import qcow_path_spec
from dfvfs.path import tsk_partition_path_spec
from dfvfs.resolver import context
from dfvfs.resolver import resolver
from dfvfs.vfs import fvde_file_system


class FVDEFileSystemTest(unittest.TestCase):
  """The unit test for the FVDE file system object."""

  _FVDE_PASSWORD = u'fvde-TEST'

  def setUp(self):
    """Sets up the needed objects used throughout the test."""
    self._resolver_context = context.Context()
    test_file = os.path.join(u'test_data', u'fvdetest.qcow2')
    path_spec = os_path_spec.OSPathSpec(location=test_file)
    path_spec = qcow_path_spec.QCOWPathSpec(parent=path_spec)
    path_spec = tsk_partition_path_spec.TSKPartitionPathSpec(
        location=u'/p1', parent=path_spec)
    self._fvde_path_spec = fvde_path_spec.FVDEPathSpec(parent=path_spec)
    resolver.Resolver.key_chain.SetCredential(
        self._fvde_path_spec, u'password', self._FVDE_PASSWORD)

  def testOpenAndClose(self):
    """Test the open and close functionality."""
    file_system = fvde_file_system.FVDEFileSystem(self._resolver_context)
    self.assertIsNotNone(file_system)

    file_system.Open(self._fvde_path_spec)

    file_system.Close()

  def testFileEntryExistsByPathSpec(self):
    """Test the file entry exists by path specification functionality."""
    file_system = fvde_file_system.FVDEFileSystem(self._resolver_context)
    self.assertIsNotNone(file_system)

    file_system.Open(self._fvde_path_spec)

    self.assertTrue(file_system.FileEntryExistsByPathSpec(self._fvde_path_spec))

    file_system.Close()

  def testGetFileEntryByPathSpec(self):
    """Tests the GetFileEntryByPathSpec function."""
    file_system = fvde_file_system.FVDEFileSystem(self._resolver_context)
    self.assertIsNotNone(file_system)

    file_system.Open(self._fvde_path_spec)

    file_entry = file_system.GetFileEntryByPathSpec(self._fvde_path_spec)

    self.assertIsNotNone(file_entry)
    self.assertEqual(file_entry.name, u'')

    file_system.Close()

  def testGetRootFileEntry(self):
    """Test the get root file entry functionality."""
    file_system = fvde_file_system.FVDEFileSystem(self._resolver_context)
    self.assertIsNotNone(file_system)

    file_system.Open(self._fvde_path_spec)

    file_entry = file_system.GetRootFileEntry()

    self.assertIsNotNone(file_entry)
    self.assertEqual(file_entry.name, u'')

    file_system.Close()


if __name__ == '__main__':
  unittest.main()
