# -*- coding: utf-8 -*-
#
# Author: Ingelrest François (Athropos@gmail.com)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import gtk, gtk.glade, media, modules, os, tools, urllib

from gtk        import gdk
from gui        import fileChooser, infoMsgBox, treeview, listview
from tools      import consts, prefs
from media      import playlist
from gettext    import gettext as _
from gobject    import idle_add, TYPE_STRING, TYPE_INT
from gui.window import Window


# Module configuration
MOD_NAME            = _('File Explorer')
MOD_DESC            = _('Navigate through your file system')
MOD_IS_MANDATORY    = True
MOD_IS_CONFIGURABLE = True


# Default preferences
PREFS_DEFAULT_MEDIA_FOLDERS     = {_('Home'): consts.dirUsr}    # List of media folders that can be used as roots for the file explorer
PREFS_DEFAULT_SHOW_HIDDEN_FILES = False                         # True if hidden files should be shown


# The format of a row in the treeview
(
    ROW_PIXBUF,    # Item icon
    ROW_NAME,      # Item name
    ROW_TYPE,      # The type of the item (e.g., directory, file)
    ROW_FULLPATH   # The full path to the item
) = range(4)


# The possible types for an node in the tree
(
    TYPE_DIR,   # A directory
    TYPE_PLIST, # A playlist
    TYPE_FILE,  # A media file
    TYPE_NONE   # A fake item, used to display a '+' in front of a directory when needed
) = range(4)


# The format of a row in the preferences' list (media folders)
(
    ROW_FOLDER_NAME,
    ROW_FOLDER_PATH
) = range(2)


# Data associated with a media folder
(
    MF_PIXBUF,    # The pixbuf to display in the explorer combo box
    MF_FULLPATH   # The full path to the folder
) = range(2)


DND_SOURCES = ('text/uri-list', gtk.TARGET_SAME_APP, 0)


class FileExplorer(modules.Module):
    """
        This explorer lets the user browse the disk from a given root directory (e.g., ~/, /)
        It uses a 'lazy' treeview (i.e., it is populated only when needed)
    """


    def __init__(self, wTree):
        """ Constructor """
        modules.Module.__init__(self)
        modules.register(self, [modules.MSG_EVT_APP_STARTED, modules.MSG_EVT_EXPLORER_CHANGED])

        self.currRoot        = None
        self.popupMenu       = None
        self.cfgWindow       = None
        self.mediaFolders    = prefs.get(__name__, 'media-folders',     PREFS_DEFAULT_MEDIA_FOLDERS)
        self.showHiddenFiles = prefs.get(__name__, 'show-hidden-files', PREFS_DEFAULT_SHOW_HIDDEN_FILES)
        # Create the tree
        txtRdr    = gtk.CellRendererText()
        pixbufRdr = gtk.CellRendererPixbuf()

        columns = (('',   [(pixbufRdr, gdk.Pixbuf), (txtRdr, TYPE_STRING)], True),
                   (None, [(None, TYPE_INT)],                               False),
                   (None, [(None, TYPE_STRING)],                            False))

        self.tree = treeview.TreeView(columns)
        self.tree.setIsDraggableFunc(self.isDraggable)
        # Scroll window
        self.scrolled = gtk.ScrolledWindow()
        self.scrolled.add(self.tree)
        self.scrolled.set_shadow_type(gtk.SHADOW_IN)
        self.scrolled.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        self.scrolled.show()
        # GTK handlers
        self.tree.connect('drag-data-get',           self.onDragDataGet)
        self.tree.connect('treeview-row-expanded',   self.onRowExpanded)
        self.tree.connect('treeview-button-pressed', self.onButtonPressed)


    def toggleHiddenFiles(self):
        """ Show/hide hidden files """
        self.showHiddenFiles = not self.showHiddenFiles
        prefs.set(__name__, 'show-hidden-files', self.showHiddenFiles)
        self.refresh()


    def updateDirIcon(self, parentPath, firstChild):
        """ Update the icon of the first 3 children, starting from firstChild, and then repost the same task while some children are left """
        for index in xrange(firstChild, firstChild+3):
            # Directories are always placed before other files
            child = self.tree.getChild(parentPath, index)
            if child is None or self.tree.getItem(child, ROW_TYPE) != TYPE_DIR:
                return

            directory = self.tree.getItem(child, ROW_FULLPATH)
            if not os.access(directory, os.R_OK | os.X_OK):
                continue

            hasContent, hasMediaContent = False, False
            for (file, fullPath) in tools.listDir(directory, self.showHiddenFiles):
                if os.path.isdir(fullPath):
                    hasContent = True
                elif os.path.isfile(fullPath) and (media.isSupported(file) or playlist.isSupported(file)):
                    hasContent, hasMediaContent = True, True
                    break

            # Append/remove children if needed
            if hasContent and self.tree.getNbChildren(child) == 0:      self.tree.appendRow((consts.icoDir, '', TYPE_NONE, ''), child)
            elif not hasContent and self.tree.getNbChildren(child) > 0: self.tree.removeAllChildren(child)

            # Change the icon based on whether the folder contains some media content
            if hasMediaContent: self.tree.setItem(child, ROW_PIXBUF, consts.icoMediaDir)
            else:               self.tree.setItem(child, ROW_PIXBUF, consts.icoDir)

        idle_add(self.updateDirIcon, parentPath, firstChild+3)


    def playSelection(self, replace):
        """ Replace/extend the tracklist """
        path = [row[ROW_FULLPATH] for row in self.tree.getSelectedRows()]

        if replace: modules.postMsg(modules.MSG_CMD_TRACKLIST_SET, {'files': path, 'playNow': True})
        else:       modules.postMsg(modules.MSG_CMD_TRACKLIST_ADD, {'files': path})


    def showPopupMenu(self, button, time, path):
        """ Show a popup menu """
        if self.popupMenu is None:
            wTree          = tools.loadGladeFile('FileExplorerMenu.glade')
            self.popupMenu = wTree.get_widget('menu-popup')
            # Must be done *before* connecting the handler
            wTree.get_widget('item-hidden').set_active(self.showHiddenFiles)
            # Connect handlers
            self.popupItemAdd  = wTree.get_widget('item-add')
            self.popupItemPlay = wTree.get_widget('item-play')

            self.popupItemAdd.connect('activate', lambda widget: self.playSelection(False))
            self.popupItemPlay.connect('activate', lambda widget: self.playSelection(True))
            wTree.get_widget('item-refresh').connect('activate', lambda widget: self.refresh())
            wTree.get_widget('item-hidden').connect('toggled', lambda widget: self.toggleHiddenFiles())

        self.popupItemAdd.set_sensitive(path is not None and self.tree.getItem(path, ROW_PIXBUF) != consts.icoDir)
        self.popupItemPlay.set_sensitive(path is not None and self.tree.getItem(path, ROW_PIXBUF) != consts.icoDir)
        self.popupMenu.popup(None, None, None, button, time)


    def __cmpRows(self, r1, r2):
        """ Used to sort rows """
        return cmp(r1[ROW_NAME].lower(), r2[ROW_NAME].lower())


    def __getDirContent(self, directory):
        """ Return a tuple of sorted rows (directories, playlists, mediaFiles) for the given directory """
        if not os.path.exists(directory) or not os.path.isdir(directory):
            return ([], [], [])

        playlists   = []
        mediaFiles  = []
        directories = []

        for (file, fullPath) in tools.listDir(directory, self.showHiddenFiles):
            if os.path.isdir(fullPath):
                directories.append([consts.icoDir, file, TYPE_DIR, fullPath])
            elif os.path.isfile(fullPath):
                if media.isSupported(file):      mediaFiles.append([consts.icoMediaFile, file, TYPE_FILE, fullPath])
                elif playlist.isSupported(file): playlists.append([consts.icoMediaFile, file, TYPE_PLIST, fullPath])

        directories.sort(self.__cmpRows)
        playlists.sort(self.__cmpRows)
        mediaFiles.sort(self.__cmpRows)

        return (directories, playlists, mediaFiles)


    def exploreDir(self, directory, parentPath):
        """ List the content of the given directory and append it to the tree as a child of parent """
        directories, playlists, mediaFiles = self.__getDirContent(directory)

        self.tree.appendRows(directories, parentPath)
        self.tree.appendRows(playlists,   parentPath)
        self.tree.appendRows(mediaFiles,  parentPath)

        if len(directories) != 0:
            idle_add(self.updateDirIcon, parentPath, 0)


    def refresh(self, treePath=None):
        """ Refresh the tree, starting from treePath """
        if treePath is None: directory = self.mediaFolders[self.currRoot]
        else:                directory = self.tree.getItem(treePath, ROW_FULLPATH)

        directories, playlists, mediaFiles = self.__getDirContent(directory)

        disk = directories
        disk.extend(playlists)
        disk.extend(mediaFiles)

        # Find if some files have been added/removed
        diskIndex, childIndex = 0, 0
        while diskIndex < len(disk):
            file    = disk[diskIndex]
            rowPath = self.tree.getChild(treePath, childIndex)

            if rowPath is None:
                self.tree.appendRow(file, treePath)
                continue

            cmpResult = self.__cmpRows(self.tree.getRow(rowPath), file)
            if cmpResult < 0:
                self.tree.removeRow(rowPath)
            else:
                if cmpResult > 0:
                    self.tree.insertRowBefore(file, treePath, rowPath)
                diskIndex  += 1
                childIndex += 1

        # Some tree rows may be left: it means that they are no longer on the disk
        while childIndex < self.tree.getNbChildren(treePath):
            self.tree.removeRow(self.tree.getChild(treePath, childIndex))

        # Recursively do the same thing for expanded rows
        for childIndex in xrange(self.tree.getNbChildren(treePath)):
            child = self.tree.getChild(treePath, childIndex)
            if self.tree.row_expanded(child):
                idle_add(self.refresh, child)

        # Update appearance of directories based on their content
        if len(directories) != 0:
            idle_add(self.updateDirIcon, treePath, 0)


    def isDraggable(self):
        """ Determine whether the selected rows can be dragged """
        for row in self.tree.getSelectedRows():
            if row[ROW_PIXBUF] == consts.icoDir:
                return False
        return True


    # --== GTK handlers ==--


    def onButtonPressed(self, tree, event, path):
        """ A mouse button has been pressed """
        if event.button == 3:
            self.showPopupMenu(event.button, event.time, path)
        elif event.button == 1 and event.type == gdk._2BUTTON_PRESS and path is not None:
            if   self.tree.getItem(path, ROW_PIXBUF) != consts.icoDir: self.playSelection(True)
            elif self.tree.row_expanded(path):                         self.tree.collapse_row(path)
            else:                                                      self.tree.expand_row(path, False)


    def onRowExpanded(self, tree, path):
        """ Populate the expanded row if needed """
        child = self.tree.getChild(path, 0)
        if self.tree.getItem(child, ROW_TYPE) == TYPE_NONE:
            self.exploreDir(self.tree.getItem(path, ROW_FULLPATH), path)
            self.tree.removeRow(child)


    def onDragDataGet(self, tree, context, selection, info, time):
        """ Provide information about the data being dragged """
        selection.set('text/uri-list', 8, ' '.join(['file://' + urllib.pathname2url(row[ROW_FULLPATH]) for row in self.tree.getSelectedRows()]))


    def onAppStarted(self):
        """ Add all media folders to the explorer Module """
        for (name, path) in self.mediaFolders.iteritems():
            modules.postMsg(modules.MSG_CMD_EXPLORER_ADD, {'modName': __name__, 'expName': name, 'icon': None, 'usrParam': path, 'widget': self.scrolled})


   # --== Message handler ==--


    def handleMsg(self, msg, params):
        """ Handle messages sent to this module """
        if msg == modules.MSG_EVT_EXPLORER_CHANGED and params['modName'] == __name__:
            self.currRoot = params['expName']
            self.tree.clear()
            self.exploreDir(params['usrParam'], None)
            if self.tree.getCount() != 0:
                self.tree.scroll_to_cell(0)
        elif msg == modules.MSG_EVT_APP_STARTED:
            idle_add(self.onAppStarted)


    # --== Configuration ==--


    def configure(self, parent):
        """ Show the configuration dialog """
        if self.cfgWindow is None:
            self.cfgWindow    = Window('FileExplorer.glade', 'vbox1', __name__, MOD_NAME, 370, 400)
            # Create the list of media folders
            renderer     = gtk.CellRendererText()
            self.cfgList = listview.ListView(((_('Name'), [(renderer, TYPE_STRING)], 0, False), (_('Path'), [(renderer, TYPE_STRING)], 1, True)))
            self.cfgWindow.getWidget('scrolledwindow1').add(self.cfgList)
            # Other widgets
            self.cfgDlgAdd    = None
            self.cfgBtnAdd    = self.cfgWindow.getWidget('btn-add')
            self.cfgBtnRemove = self.cfgWindow.getWidget('btn-remove')
            # Connect handlers
            self.cfgList.connect('key-press-event', self.onCfgKeyboard)
            self.cfgBtnAdd.connect('clicked', self.onAddMediaFolder)
            self.cfgBtnRemove.connect('clicked', lambda btn: self.removeSelectedMediaFolders())
            self.cfgWindow.getWidget('btn-ok').connect('clicked', lambda btn: self.cfgWindow.hide())
            self.cfgWindow.getWidget('btn-cancel').connect('clicked', lambda btn: self.cfgWindow.hide())
            self.cfgWindow.getWidget('btn-help').connect('clicked', self.onHelp)

        if not self.cfgWindow.isVisible():
            self.fillMFList()

        self.cfgWindow.show()


    def onHelp(self, btn):
        """ Display a small help message box """
        infoMsgBox(self.cfgWindow,
                _('This is the list of folders displayed in the main window. You can add/remove folders by using the corresponding buttons'))


    def onAddMediaFolder(self, btn):
        """ Let the user add a new media folder to the list """
        if self.cfgDlgAdd is None:
            wTree              = tools.loadGladeFile('FileExplorer.glade')
            self.cfgDlgAdd     = wTree.get_widget('dlg-add')
            self.cfgDlgBtnAdd  = wTree.get_widget('btn-dlgAdd')
            self.cfgDlgTxtName = wTree.get_widget('txt-dlgName')
            self.cfgDlgTxtPath = wTree.get_widget('txt-dlgPath')
            self.cfgDlgAdd.set_title(MOD_NAME)
            self.cfgDlgAdd.set_transient_for(self.cfgWindow)
            # Connect handlers
            wTree.get_widget('btn-dlgOpen').connect('clicked', self.onDlgAddOpen)
            self.cfgDlgTxtName.connect('changed', self.onDlgAddTxtChanged)
            self.cfgDlgTxtPath.connect('changed', self.onDlgAddTxtChanged)

        self.cfgDlgBtnAdd.set_sensitive(False)
        self.cfgDlgTxtName.set_text('')
        self.cfgDlgTxtPath.set_text('')
        self.cfgDlgTxtName.grab_focus()

        if self.cfgDlgAdd.run() == gtk.RESPONSE_OK:
            name = self.cfgDlgTxtName.get_text()
            path = self.cfgDlgTxtPath.get_text()

            if name not in self.mediaFolders:
                self.mediaFolders[name] = path
                prefs.set(__name__, 'media-folders', self.mediaFolders)
                self.fillMFList()
                modules.postMsg(modules.MSG_CMD_EXPLORER_ADD, {'modName': __name__, 'expName': name, 'icon': None, 'usrParam': path, 'widget': self.scrolled})

        self.cfgDlgAdd.hide()


    def fillMFList(self):
        """ Fill the list of currently used media folders """
        sortedMF = self.mediaFolders.items()
        sortedMF.sort()
        self.cfgList.clear()
        self.cfgList.insertRows(sortedMF)
        self.cfgList.set_cursor(0)
        self.cfgBtnRemove.set_sensitive(self.cfgList.getCount() != 0)


    def onDlgAddOpen(self, btn):
        """ Let the user choose a folder, and fill the corresponding field in the dialog """
        path = fileChooser.openDirectory(self.cfgDlgAdd, _('Choose a folder'))
        if path is not None:
            self.cfgDlgTxtPath.set_text(path)


    def onDlgAddTxtChanged(self, entry):
        """ Enable/disable the OK button based on the content of the text fields """
        self.cfgDlgBtnAdd.set_sensitive(self.cfgDlgTxtName.get_text() != '' and self.cfgDlgTxtPath.get_text() != '')


    def removeSelectedMediaFolders(self):
        """ Remove the selected media folder """
        for row in self.cfgList.getSelectedRows():
            modules.postMsg(modules.MSG_CMD_EXPLORER_REMOVE, {'modName': __name__, 'expName': row[0]})
            del self.mediaFolders[row[0]]
            prefs.set(__name__, 'media-folders', self.mediaFolders)
        self.cfgList.removeSelectedRows()


    def onCfgKeyboard(self, list, event):
        """ Remove the selection if possible """
        if gdk.keyval_name(event.keyval) == 'Delete':
            self.removeSelectedMediaFolders()
