# -*- coding: utf-8 -*-
#
# Author: Ingelrest François (Athropos@gmail.com)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import gobject, modules, os.path, socket

from md5     import md5
from gui     import authentication
from time    import time, sleep
from tools   import consts
from urllib  import quote_plus, urlencode
from urllib2 import urlopen
from modules import ThreadedModule
from gettext import gettext as _


CLI_ID               = 'dbl'  # Identifier of Decibel Audio Player on AudioScrobbler's server
CLI_VER              = '0.1'
PROTO_VER            = '1.2'
AS_SERVER            = 'post.audioscrobbler.com'
CACHE_FILE           = 'audioscrobbler-cache.txt'
MAX_SUBMITTED_TRACKS = 4      # Maximum number of tracks submitted at the same time


# Module configuration
MOD_NAME            = _('AudioScrobbler')
MOD_DESC            = _('Keep your Last.fm profile up to date')
MOD_IS_MANDATORY    = False
MOD_IS_CONFIGURABLE = False


# Session
(
    SESSION_ID,
    NOW_PLAYING_URL,
    SUBMISSION_URL
) = range(3)


# Current track
(
    TRK_STARTED_TIMESTAMP,   # When the track has been started
    TRK_UNPAUSED_TIMESTAMP,  # When the track has been unpaused (if it ever happened)
    TRK_PLAY_TIME,           # The total play time of this track
    TRK_INFO                 # Information about the track
) = range(4)


class AudioScrobbler(ThreadedModule):
    """
        This module implements the Audioscrobbler protocol v1.2
        http://www.audioscrobbler.net/development/protocol/
    """


    def __init__(self, wTree):
        """ Constructor """
        ThreadedModule.__init__(self)
        self.cache          = None
        self.login          = None
        self.passwd         = None
        self.paused         = False
        self.session        = [None, None, None]
        self.isBanned       = False
        self.currTrack      = None
        self.nbHardFailures = 0
        self.lastHandshake  = 0
        self.handshakeDelay = 0
        modules.register(self, [modules.MSG_EVT_NEW_TRACK, modules.MSG_EVT_APP_QUIT, modules.MSG_EVT_MOD_UNLOADED, modules.MSG_EVT_MOD_LOADED,
                                modules.MSG_EVT_PAUSED,    modules.MSG_EVT_UNPAUSED, modules.MSG_EVT_STOPPED,      modules.MSG_EVT_APP_STARTED])


    def loadCache(self):
        """ Load the tracks cached on the disk, if any """
        file = os.path.join(consts.dirCfg, CACHE_FILE)

        if not os.path.exists(file):
            self.cache = []
        else:
            input      = open(file)
            self.cache = [strippedTrack for strippedTrack in [track.strip() for track in input.readlines()] if len(strippedTrack) != 0]
            input.close()


    def saveCache(self):
        """ Save the cache to the disk """
        file   = os.path.join(consts.dirCfg, CACHE_FILE)
        output = open(file, 'w')
        output.writelines('\n'.join(self.cache))
        output.close()


    def addToCache(self):
        """ Add the current track to the cache, if any and that all conditions are OK """
        if self.currTrack is None:
            return

        track      = self.currTrack[TRK_INFO]
        fieldsOk   = track.hasArtist() and track.hasTitle() and track.hasLength()
        durationOk = track.getLengthSec() >= 30 and (self.currTrack[TRK_PLAY_TIME] >= 240 or self.currTrack[TRK_PLAY_TIME] >= track.getLengthSec()/2)

        if not fieldsOk or not durationOk:
            return

        params = (
                    ( 'a[*]', quote_plus(track.getSafeArtist())          ),
                    ( 't[*]', quote_plus(track.getSafeTitle())           ),
                    ( 'i[*]', str(self.currTrack[TRK_STARTED_TIMESTAMP]) ),
                    ( 'o[*]', 'P'                                        ),
                    ( 'r[*]', ''                                         ),
                    ( 'l[*]', track.getSafeLength()                      ),
                    ( 'b[*]', quote_plus(track.getSafeAlbum())           ),
                    ( 'n[*]', track.getSafeNumber()                      ),
                    ( 'm[*]', ''                                         )
                 )

        self.cache.append('&'.join(['%s=%s' % (key, val) for (key, val) in params]))


    def getFromCache(self, howMany):
        """ Return the last howMany tracks from the cache, replace the star with the correct index """
        if len(self.cache) == 0:
            return []

        if howMany > len(self.cache):
            howMany = len(self.cache)

        return [self.cache[-howMany+i].replace('[*]', '[%d]' % i) for i in xrange(howMany)]


    def removeFromCache(self, howMany):
        """ Remove the last howMany tracks from the cache """
        if len(self.cache) != 0:
            if howMany > len(self.cache): self.cache[:] = []
            else:                         self.cache[:] = self.cache[:-howMany]


    def getAuthInfo(self):
        """ Retrieve the login/password of the user """
        if self.login is None: auth = authentication.getAuthInfo('last.fm', _('your Last.fm account'))
        else:                  auth = authentication.getAuthInfo('last.fm', _('your Last.fm account'), self.login, True)

        if auth is None: self.login, self.passwd = None, None
        else:            self.login, self.passwd = auth


    def handshake(self):
        """ Authenticate the user to the submission servers, return True if OK """
        timestamp          = int(time())
        self.session[:]    = [None, None, None]

        # In case of BANNED or BADTIME replies
        if self.isBanned or timestamp - self.lastHandshake < self.handshakeDelay:
            return False

        # Asking for login information must be done in the GTK main loop, because a dialog box might be displayed if needed
        self.gtkExecute(self.getAuthInfo)
        if self.passwd is None:
            return False

        token              = md5('%s%d' % (md5(self.passwd).hexdigest(), timestamp)).hexdigest()
        self.passwd        = None
        request            = 'http://%s/?hs=true&p=%s&c=%s&v=%s&u=%s&t=%d&a=%s' % (AS_SERVER, PROTO_VER, CLI_ID, CLI_VER, self.login, timestamp, token)
        self.lastHandshake = timestamp

        try:
            hardFailure = False
            reply       = urlopen(request).read().strip().split('\n')

            if reply[0] == 'OK':
                self.session[:]     = reply[1:]
                self.handshakeDelay = 0
                self.nbHardFailures = 0

            elif reply[0] == 'BANNED':
                print _('ERROR: This version of %s has been isBanned from AudioScrobbler servers') % consts.appName
                self.isBanned = True

            elif reply[0] == 'BADAUTH':
                print _('ERROR: Bad AudioScrobbler authentication')
                return self.handshake()

            elif reply[0] == 'BADTIME':
                print _('ERROR: AudioScrobbler reported that the current system time is not correct')
                self.banned = True

            else:
                hardFailure = True
                print _('ERROR: Hard failure during AudioScrobbler handshake')

        except:
            hardFailure = True
            print _('ERROR: Exception while performing AudioScrobbler handshake!')

        if hardFailure:
            if   self.handshakeDelay == 0:     self.handshakeDelay  = 1*60         # Start at 1mn
            elif self.handshakeDelay >= 64*60: self.handshakeDelay  = 120*60       # Max 120mn
            else:                              self.handshakeDelay *= 2            # Double the delay

        self.login = None

        return self.session[SESSION_ID] is not None


    def nowPlayingNotification(self, track, firstTry = True):
        """ The Now-Playing notification is a lightweight mechanism for notifying the Audioscrobbler server that a track has started playing """
        if (self.session[SESSION_ID] is None and not self.handshake()) or not track.hasArtist() or not track.hasTitle():
            return

        params = (
                    ( 's', self.session[SESSION_ID]          ),
                    ( 'a', quote_plus(track.getSafeArtist()) ),
                    ( 't', quote_plus(track.getSafeTitle())  ),
                    ( 'b', quote_plus(track.getSafeAlbum())  ),
                    ( 'l', track.getSafeLength()             ),
                    ( 'n', track.getSafeNumber()             ),
                    ( 'm', ''                                )
                 )

        try:
            data  = '&'.join(['%s=%s' % (key, val) for (key, val) in params])
            reply = urlopen(self.session[NOW_PLAYING_URL], data).read().strip().split('\n')

            if reply[0] == 'BADSESSION' and firstTry:
                self.session[:] = [None, None, None]
                self.nowPlayingNotification(track, False)

        except:
            print _('ERROR: Exception while performing AudioScrobbler now-playing notification!')


    def submit(self, firstTry=True):
        """ Submit cached tracks, return True if OK """
        if (self.session[SESSION_ID] is None and not self.handshake()) or len(self.cache) == 0:
            return False

        try:
            hardFailure = False
            data        = 's=%s&%s' % (self.session[SESSION_ID], '&'.join(self.getFromCache(MAX_SUBMITTED_TRACKS)))
            reply       = urlopen(self.session[SUBMISSION_URL], data).read().strip().split('\n')

            if reply[0] == 'OK':
                self.removeFromCache(MAX_SUBMITTED_TRACKS)
                return True

            elif reply[0] == 'BADSESSION' and firstTry:
                self.session[:] = [None, None, None]
                return self.submit(False)

            else:
                hardFailure = True

        except:
            hardFailure = True
            print _('ERROR: Exception while performing AudioScrobbler submission!')

        if hardFailure:
            if self.nbHardFailures < 2: self.nbHardFailures += 1
            else:                       self.handshake()
        else:
            self.nbHardFailures = 0

        return False


    def onTrackEnded(self):
        """ The playback of the current track has stopped """
        if self.currTrack is not None:
            self.currTrack[TRK_PLAY_TIME] += (int(time()) - self.currTrack[TRK_UNPAUSED_TIMESTAMP])
            self.addToCache()

            # Try to submit the whole cache
            submitOk = self.submit()
            while submitOk and len(self.cache) != 0:
                submitOk = self.submit()

            self.currTrack = None


    def onNewTrack(self, track):
        """ A new track has started """
        timestamp = int(time())
        self.onTrackEnded()
        self.nowPlayingNotification(track)
        self.currTrack = [timestamp, timestamp, 0, track]


    # --== Message handler ==--


    def handleMsg(self, msg, params):
        """ Handle messages sent to this module """
        if msg == modules.MSG_EVT_NEW_TRACK:
            self.onNewTrack(params['track'])

        elif msg == modules.MSG_EVT_STOPPED:
            if self.paused:
                self.currTrack[TRK_UNPAUSED_TIMESTAMP] = int(time())
            self.paused = False
            self.onTrackEnded()

        elif msg == modules.MSG_EVT_PAUSED:
            self.currTrack[TRK_PLAY_TIME] += (int(time()) - self.currTrack[TRK_UNPAUSED_TIMESTAMP])
            self.paused = True

        elif msg == modules.MSG_EVT_UNPAUSED:
            self.currTrack[TRK_UNPAUSED_TIMESTAMP] = int(time())
            self.paused = False

        elif msg == modules.MSG_EVT_APP_QUIT or msg == modules.MSG_EVT_MOD_UNLOADED:
            if self.paused:
                self.currTrack[TRK_UNPAUSED_TIMESTAMP] = int(time())
            self.paused = False
            self.addToCache()
            self.saveCache()

        elif msg == modules.MSG_EVT_APP_STARTED or msg == modules.MSG_EVT_MOD_LOADED:
            socket.setdefaulttimeout(10)
            self.loadCache()
