# -*- coding: utf-8 -*-
#
# Author: Ingelrest François (Athropos@gmail.com)
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Library General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

import mp3, ogg, os, os.path, playlist, re, tools

from tools import consts


# The format of the list returned by the readInfo() and getTracks() functions
(
    NFO_NUM,  # Track number
    NFO_TIT,  # Title
    NFO_ART,  # Artist
    NFO_ALB,  # Album
    NFO_LEN,  # Length in seconds
    NFO_FIL   # Full path to the file
) = range(6)


# Supported formats
mFormatsRE = re.compile('^.*\.(mp3|ogg)$', re.IGNORECASE)


def isSupported(file):
    """
        Return True if the given file is a supported format
    """
    return mFormatsRE.match(file)


def getSupportedFormats():
    """
        Return a list of all formats from which tags can be extracted
        This list does not account for the installed gstreamer packages
    """
    return ['*.mp3', '*.ogg']


def readInfo(files):
    """
        Given a list of files, return a list such as:
        [[FILE1, TRACKNUMBER1, TITLE1, ARTIST1, ALBUM1, LENGTH1], [FILE2, TRACKNUMBER2, TITLE2, ARTIST2, ALBUM2, LENGTH2], ...]
    """
    info = []
    for file in files:
        if   file.lower().endswith('.ogg'): info += [ogg.readInfo(file)]
        elif file.lower().endswith('.mp3'): info += [mp3.readInfo(file)]
        else:                               info += [[consts.UNKNOWN_TRACKNUMBER, consts.UNKNOWN_TITLE, consts.UNKNOWN_ARTIST, \
                                                      consts.UNKNOWN_ALBUM, consts.UNKNOWN_LENGTH, file]]
    return info


def __cmpTracks(t1, t2):
    """
        Used to sort tracks
    """
    if t1[NFO_NUM] != t2[NFO_NUM]: return t1[NFO_NUM] - t2[NFO_NUM]
    else:                          return cmp(t1[NFO_TIT].lower(), t2[NFO_TIT].lower())


def getTracks(files):
    """
        Return all the tracks contained in the list of files:
            * For directory, load all media files (playable files and playlists)
            * For a playlist, load the tracks it contains
        The content of playlists is never sorted
    """
    tracks = []

    # Directories
    for directory in [file for file in files if os.path.exists(file) and os.path.isdir(file)]:
        mediaFiles, playlists = [], []
        for (filename, fullPath) in tools.listDir(directory):
            if os.path.isfile(fullPath):
                if isSupported(filename):            mediaFiles.append(fullPath)
                elif playlist.isSupported(filename): playlists.append(fullPath)

        info = readInfo(mediaFiles)
        info.sort(__cmpTracks)
        tracks.extend(info)

        for pl in playlists:
            tracks.extend(readInfo(playlist.load(pl)))

    # Files
    mediaFiles = readInfo([file for file in files if os.path.exists(file) and os.path.isfile(file) and isSupported(file)])
    mediaFiles.sort(__cmpTracks)
    tracks.extend(mediaFiles)

    # Playlists
    for pl in [file for file in files if os.path.exists(file) and os.path.isfile(file) and playlist.isSupported(file)]:
        tracks.extend(readInfo(playlist.load(pl)))

    return tracks
