package com.tildemh.debbuggtk;

import com.tildemh.debbug.*;
import org.gnu.gtk.*;
import org.gnu.gnome.*;
import org.gnu.gtk.event.*;
import org.gnu.glade.*;
import java.util.*;
import java.io.*;
import org.gnu.glib.CustomEvents;

/**
 * When adding comments to bug reports, this will allow the user to choose a
 * pre-written response from a drop down menu. This class provides support for
 * loading, editing and saving these custom responses.
 *
 * <p>This is released under the terms of the GNU Lesser General Public License
 * (LGPL). See the COPYING file for details.
 *
 * @version $Id: ResponseEditor.java,v 1.3 2004/04/21 20:20:05 mh Exp $
 * @author &copy; Mark Howard &lt;mh@debian.org&gt; 2004
 */
public class ResponseEditor implements TreeSelectionListener, TextBufferListener, EntryListener{
	
	private LibGlade glade;

	private Hashtable responses;
	private Vector keyList;

	private ListStore store;
	private DataColumnString dataName = new DataColumnString();
	private VBox selectedSection;
	private Button addButton;
	private Button removeButton;
	private Button cancelButton;
	private Button saveButton;

	private TextBuffer buffer;
	private org.gnu.gtk.Entry entry;
	private TreeSelection selection;

	private int currentIndex; // currently selected item

	private volatile boolean ignoreEntryChange = false;
	private volatile boolean ignoreSelChange = false;

	public static ResponseEditor getInstance(){
		if (instance == null)
			instance = new ResponseEditor();
		return instance;		
	}
	
	private static ResponseEditor instance = null;
	private ResponseEditor(){
		loadResponses();
		store = new ListStore( new DataColumn[] {dataName} );
		fillList();
	}

	private Window app;
	/**
	 */
	public void makeWindow (){

		WindowIcon.setDefaultIcon(DebbugGtk.ICON);
		
		String filename = DebbugGtk.GLADE_XML_DIR+"responseeditor.glade";
		try {
			glade = new LibGlade(filename, this, null);
			app = (Window) glade.getWidget("customResponses");
			app.addListener( DebbugGtk.lifeCycleListener );
			
		} catch (GladeXMLException e) {
			System.err.println("Error parsing glade XML file." + e);
		} catch (FileNotFoundException e) {
			System.err.println("Glade XML file not found.");
		} catch (IOException e) {
			System.err.println("Error reading glade XML file.");
		}

		TreeView view = (TreeView) glade.getWidget("responseTree");
		view.setModel( store );
		TreeViewColumn col = new TreeViewColumn();
		view.appendColumn( col );
		CellRendererText rend = new CellRendererText();
		col.packStart(rend, true);
		col.addAttributeMapping( rend, CellRendererText.Attribute.TEXT, dataName );


		selection = view.getSelection();
		selection.addListener( (TreeSelectionListener) this );

		addButton = (Button) glade.getWidget("addButton");
		removeButton = (Button) glade.getWidget("removeButton");
		cancelButton = (Button) glade.getWidget("cancelButton");
		saveButton = (Button) glade.getWidget("saveButton");

		selectedSection = (VBox) glade.getWidget("selectedSection");
		selectedSection.setSensitive( false );

		buffer = ((TextView) glade.getWidget( "responseText" )).getBuffer();
		buffer.addListener( (TextBufferListener) this );
		entry = (org.gnu.gtk.Entry) glade.getWidget("nameEntry");
		entry.addListener( (EntryListener) this );
	}

	private void fillList(){
		store.clear();
		for( int i = 0; i < keyList.size(); i++){
			TreeIter iter = store.appendRow();
			store.setValue( iter,  dataName, (String) keyList.get(i) );
		}
	}
	
	/**
	 * Returns a list of items to place in the menu
	 */
	public String[] getResponses(){
		String[] ret = new String[ keyList.size() ];
		for(int i = 0; i < ret.length; i++){
			ret[i] = (String) keyList.get( i );
		}
		return ret;
	}

	/**
	 * Called when the openmenu selection is changed. This will modify the
	 * buffer, or open the menu for editing custom responses depending on the
	 * index.
	 */
	public void handleResponse( int id, TextBuffer buffer ){
		if (id >= 0 && id < keyList.size()){
			// we're editing it
			buffer.setText( (String) responses.get( keyList.get(id) ) );
		}else if( id == keyList.size() + 1 ){
			makeWindow();			
		}else{
			(new RuntimeException("Unknown Response id: "+id+"; keyList.size = "+keyList.size())).printStackTrace();
		}
	}

	private String responseFileName = System.getProperty("user.home") + "/.debbug-java/customresponses" ;
	private final String  SEPARATOR = "debbuggtk-customresponses-separator";

	/** Load responses from file */
	private void loadResponses(){
		responses = new Hashtable();
		keyList = new Vector();
		
		File responseFile = new File( responseFileName );
		boolean failure = false;
		if (  responseFile.exists() ){
			try{
			BufferedReader in = new BufferedReader( new FileReader( responseFile ) );
			
			String nextLine = in.readLine();
			while( nextLine != null ){
				String key = nextLine;
				String body = "";
				nextLine = in.readLine();
				String eoln = "";
				while( !nextLine.equals(SEPARATOR)){
					body = body + eoln + nextLine;
					eoln = "\n";
					nextLine = in.readLine();
				}
				nextLine = in.readLine();
				keyList.add(key);
				responses.put( key, body );
			}
			}catch( Exception e ){
				e.printStackTrace();
				failure = true;
			}
		}
		
		if ( ! responseFile.exists() || failure){
			responses.put("Fixed in CVS", "This issue has been fixed in the developers source repository. It will be fixed in the next upstream release packaged for Debian." );			
			keyList.addElement("Fixed in CVS");
			
			responses.put("Forwarded to Bugzilla", "I have forwarded your report to the above bugzilla site. Please add your email address to the bugzilla CC field so that you can quickly get responses from the upstream developers of this program and help them find the source of the problem.");
			keyList.addElement("Forwarded to Bugzilla");
			
			responses.put("Stable only", "The issue you have reported is present only in the stable Debian distribution - it has been fixed in the later developmental branches. The only changes permitted into stable are security fixes and major bug fixes. Your issue does not fall into this category, so there is nothing we can do.\nIf you are adventurous, please try installing a later version of the package from the testing distribution");
			keyList.addElement("Stable only");
		}
	}

	/** Save responses to file */
	private void saveResponses(){
		try{
		PrintWriter out = new PrintWriter( new BufferedWriter( new FileWriter( new File( responseFileName ) ) ) );
		for( int i = 0; i < keyList.size(); i++){
			String name = (String) keyList.get(i);
			out.println( name );
			out.println( (String) responses.get(name) );
			out.println(SEPARATOR);
		}
		out.close();
		}catch(Exception e){
			e.printStackTrace();
			// todo
		}
	}

	private Vector listeners = new Vector();
	public void addListener( ResponseEditorListener listener ){
		listeners.add( listener );
	}

	private void emitChanged(){
		String[] newResponses = getResponses();
		for( int i = 0; i < listeners.size(); i++){
			((ResponseEditorListener) listeners.get(i)).responsesChanged( newResponses );
		}
	}
	
	public void selectionChangedEvent(TreeSelectionEvent event){
		if (ignoreSelChange)
			return;
		synchronized(this){
		ignoreEntryChange = true;
		int index = 0;
		TreeIter iter = store.getFirstIter();
		while( !selection.getSelected( iter ) ){
			iter = iter.getNextIter();
			index++;
		}
		currentIndex = index;
		String key = (String) keyList.get( index );
		String bufferText =  (String) responses.get( (Object) key );
		buffer.setText( bufferText );		
		entry.setText( (String) keyList.get( index ) );
		selectedSection.setSensitive( true );
			ignoreEntryChange = false;
		}
	}
	
	////////////////////////////////////////////////////////////////////////////////
	// Signals from Glade

	/**
	 * Add button is pressed
	 */
	public void addResponse(){
		currentIndex = keyList.size();

		String key = "New entry";
		String suffix = "";
		int i = 0;
		while( keyList.contains( key + suffix) ){
			i++;
			suffix = "_"+i;			
		}
		keyList.add( key + suffix );
		responses.put( key + suffix, "Please type your entry here.");
		
		fillList();
		selection.select( new TreePath(currentIndex+"") );
	}

	/** Remove button is pressed */
	public void removeResponse(){
		int index = 0;
		TreeIter iter = store.getFirstIter();
		while( !selection.getSelected( iter ) ){
			iter = iter.getNextIter();
			index++;
			if (iter == null)
				return;
		}
		if( currentIndex == index ){
			selectedSection.setSensitive( false );
			entry.setText( "" );
			buffer.setText( "" );		
		}
		responses.remove( keyList.get( currentIndex ) );
		keyList.remove( currentIndex );
		fillList();
	}

	/** Cancel button pressed */
	public void cancel(){
		loadResponses();
		app.hide();
	}

	/** save button pressed */
	public void save(){
		saveResponses();
		emitChanged();
		app.hide();
	}

	// name entry has changed
	public synchronized void entryEvent( EntryEvent event ){
		if (ignoreEntryChange){ 
			return;
		}
		
		if (event.isOfType( EntryEvent.Type.CHANGED )){
			ignoreSelChange = true;
			Object text = responses.remove( keyList.get( currentIndex ) );
			keyList.set( currentIndex, entry.getText() );
			responses.put( entry.getText(), text );
			fillList();
			ignoreSelChange = false;
		}
	}

	public void textBufferEvent( TextBufferEvent event ){
		if (event.isOfType( TextBufferEvent.Type.CHANGED )){
			responses.put( keyList.get( currentIndex ), buffer.getText( buffer.getStartIter(), buffer.getEndIter(), false ));
		}
	}

}
