package com.tildemh.debbuggtk;

import com.tildemh.debbug.*;
import org.gnu.gtk.*;
import org.gnu.gtk.event.*;
import org.gnu.gnome.*;
import org.gnu.gnome.event.*;
import org.gnu.gdk.*;
import java.io.*;

/**
 * Application for reporting bugs in Debian packages. This takes the form of a
 * druid which allows you to select which package, see exisiting reports on that
 * package, and compose your report, including system preferences. The final
 * email will then be sent to the gnome mailto: handler to be sent to the Debian
 * bts.
 *
 * <h3>Pages</h3>
 * <dl>
 * <dt>Start Page
 * <dd>Introductory text. Explain difference in Debian and upstream bug reports.
 * 
 * <dt>Choose Package
 * <dd>Select the package to report a bug against. Give warning (dialog) if the
 * package doesn't exist
 *
 * <dt>View Package bug listings
 * <dd>See all outstanding bugs for this package (and source package??). Ability
 * to open with BugViewer (and hence also edit). Text will encourage editing. 
 *
 * <dt>Package Specific
 * <dd>Those /usr/share/bug/package/presubj files (need to read docs for this)
 *
 * <dt>Main report
 * <dd>Title, severity, tags, forwarded, Bug report text. Checkbox for including
 * sys info.
 *
 * <dt>Final screen
 * <dd>Explain how the bts works - emails. Mention gnome mailto: handler.
 * Advertise bugwatcher. Try to get users helping in the bts.
 * 
 * </dl>
 *
 * <p>This is released under the terms of the GNU Lesser General Public License
 * (LGPL). See the COPYING file for details.
 *
 * @version $Id: BugReporter.java,v 1.19 2004/03/16 19:11:46 mh Exp $
 * @author &copy; Mark Howard &lt;mh@debian.org&gt; 2002
 */
public class BugReporter implements DruidPageSetupListener, ButtonListener{

	private org.gnu.gnome.App app;
	private Druid druid;
	private DruidPageEdge 		druidStartPage;
	private DruidPageStandard 	druidSelectPackage;
	private DruidPageStandard	druidViewListing;
	private DruidPageStandard	druidPackageSpecific;
	private DruidPageStandard	druidTitleSeverity;
	private DruidPageStandard	druidTagsOther;
	private DruidPageEdge		druidFinalPage;
	private Button				cancelButton;

	/** The bug report which we're creating */
	private Bug bugReport;
	private String tags;

	private static Pixbuf logo;
	
	/**
	 * Constructs the BugReporter
	 */
	public BugReporter(){
		WindowIcon.setDefaultIcon( DebbugGtk.ICON );
		app = new App("debbuggtk", "BugReporter "+DebbugGtk.VERSION);
		bugReport = BTS.getInstance().getBug(new Integer(-1)); //null ; // TODO: new Bug();
		logo = new Pixbuf(  DebbugGtk.ICON );
		DebbugGtk.newWindow();
		app.addListener( DebbugGtk.lifeCycleListener );
		
		druid = new Druid();
		cancelButton = druid.getCancelButton();
		cancelButton.addListener( (ButtonListener) this );
		druid.setShowHelp( true );
		druidStartPage = new DruidPageEdge( EdgePosition.START );
		druidStartPage.addListener( (DruidPageSetupListener) this );
		druid.appendPage( druidStartPage );

		druidSelectPackage = new BRSelectPackage( this );
		druid.appendPage( druidSelectPackage );

		druidViewListing = new BRViewBugs( this, app );
		druid.appendPage( druidViewListing );

		druidPackageSpecific = new BRPackageSpecific();
		druid.appendPage( druidPackageSpecific );

		druidTitleSeverity = new BRTitleSeverity( this );
		druid.appendPage( druidTitleSeverity );

		druidTagsOther = new BRTagsOther( this );
		druid.appendPage( druidTagsOther );
		
		
		makeEndPage();
		druid.appendPage( druidFinalPage );
		druid.getFinishButton().setLabel("Generate Email");
		druid.getFinishButton().addListener( (ButtonListener) this);


		app.setContent(druid);
		app.showAll();
	}


	/**
	 * Constructs a simple end page for the druid.
	 */
	private void makeEndPage(){
		druidFinalPage = new DruidPageEdge( EdgePosition.FINISH );
		druidFinalPage.setTitle( "Finish" );
		druidFinalPage.setText( "When you press Generate Email, your GNOME email program will open with a formatted email to send to Debian. Please describe the bug you are reporting in the <Describe your problems here> section and send the email.\n\nYou will receive an email notifications from Debian saying they've received your report; this will include a bug number which you must use for further correspondance. At some later time, you will get emails from the package maintainers in response to this bug. You can check the status of the report at any time using the bugviewer tool.\n "
				+"If your report does not get attention, it may be a good idea to contact other people about the issue - perhaps ask on #debian-user on irc, or try the debian-user@lists.debian.org mailing list. In many cases, the developer will either be on holiday, or simply too busy. Please be patient.\n\nDebian is a community effort. Please help by adding comments to bug reports for your favourite packages. The bugwatcher tool is an easy way to do this.\n\nThank you for reporting this issue." );
		druidFinalPage.setLogo( new Pixbuf( DebbugGtk.ICON ) );
	}

	public void druidPageSetupEvent(DruidPageSetupEvent event){
		if (event.getSource().equals( druidStartPage )){
			druidStartPage.setTitle( "Debian Bug Reporter" );
		druidStartPage.setText("Welcome\n\n"
			+ "This tool allows you to report issues with the Debian packaging of programs.\n\n\n"
		    + "***WARNING*** This tool is still under development. reportbug is far superior. Only use this tool if you know about the format of the emails to send to the bug server.");
		}

		// for all pages
		if (event.isOfType( DruidPageSetupEvent.Type.PREPARE )){
			if (event.getSource() instanceof DruidPageEdge)
				((DruidPageEdge) event.getSource()).setLogo(new Pixbuf( DebbugGtk.ICON ) );
			else
				((DruidPageStandard) event.getSource()).setLogo(new Pixbuf( DebbugGtk.ICON ) );
		druid.getCancelButton().setSensitive(false); //setButtonsSensitive(true, true, false, false );
		}else if (event.isOfType( DruidPageSetupEvent.Type.FINISH )){
			System.out.println("Finish event");
		}
	}


	private void generateEmail(){
System.out.println("Generating email");
		StringBuffer msg = new StringBuffer("");
System.out.println("Generating email");
		msg.append( "Package: ");
System.out.println("Generating email");
		msg.append( bugReport.getPackageName() );
System.out.println("Generating email");
		msg.append( "\nVersion: ");
System.out.println("Generating email");
		msg.append( "oops - debbuggtk doesn't know yet" );
System.out.println("Generating email");
		msg.append( "\nSeverity: ");
System.out.println("Generating email");
		msg.append( bugReport.getSeverity().toString() );
System.out.println("Generating email");
		if ( !( null == tags || tags.equals("")) ){
			msg.append( "\nTags: " );
			msg.append( tags );
		}
System.out.println("Generating email");

		msg.append( "\n\n<Please Describe your problems here\n\n" );
System.out.println("Generating email");

		msg.append( Packages.getSystemInfo( bugReport.getPackageName() ) );
System.out.println("Generating email");

		org.gnu.gnome.Url.show("mailto:submit@bugs.debian.org?subject="+bugReport.getTitle() + "&body="+msg.toString());
System.out.println("Generating email");
	}

	/**
	 * ViewBug page has been finished with. Decide what to do next and do it -
	 * BRPackageSpecific, or other.
	 */
	public void viewBugDone(){
		goPkgSpecific( druidTitleSeverity );
	}

	public void goPkgSpecific( DruidPage alternative ){
		// TODO: pkg specific scripts...
		String filename = "/usr/share/bug/"+bugReport.getPackageName()+"/presubj";
		System.out.println("goPkgSpecific - looking for " + filename);
		File info = new File( filename );
		if (info.exists()){
			System.out.println("presubj exists!!!");
			String message = "";
			try{
				try{
					StringBuffer msg = new StringBuffer("");
					BufferedReader br = new BufferedReader( new FileReader( info ) );
					String m = br.readLine();
					while ( m != null){
						System.out.println(m);
						msg.append("\n");
						msg.append( m );
						m = br.readLine() ;
					}
					message = msg.toString();
				}catch(FileNotFoundException e ){
					System.err.println("ERROR: File not found: "+filename);
					e.printStackTrace();
					message = "Error: "+filename+" not found!";
				}catch( IOException e ){
					System.err.println("ERROR: IO error reading " + filename);
					e.printStackTrace();
					message = "IOException while reading " + filename;
				}
			}finally{
				System.out.println("Pkg message is: "+message);
				((BRPackageSpecific) druidPackageSpecific).setDetails( bugReport.getPackageName(), message );
				druid.setPage( druidPackageSpecific );
			}
		}else{
			druid.setPage( alternative );
		}
	}

	public void buttonEvent( ButtonEvent event ){
		if (event.isOfType( ButtonEvent.Type.CLICK )){
			if (event.getSource().equals( cancelButton ) ){
				DebbugGtk.closeWindow();
				app.hide();
			}else{
				System.out.println("About to generate email");
				// generate the email
				generateEmail();
				app.hide();
				DebbugGtk.closeWindow();
			}
		}
	}

	public void pkgSpecificDone(){
		druid.setPage(druidTitleSeverity);
	}

	public void titleSeverityBack(){
		goPkgSpecific( druidViewListing );
	}
		
	public static final Pixbuf getLogo(){
		return logo;
	}

	public Bug getBug(){
		return bugReport;
	}
	
	/**
	 * Initialises GUI libraries and creates the applications.
	 */
	public static void main(String[] args) {
		Program prog = Program.initGnomeUI("debbuggtk", DebbugGtk.VERSION, args);
		BugReporter br = new BugReporter();
		Thread.currentThread().setPriority( Thread.MAX_PRIORITY );

		Gtk.main();
	}
}
