/* dc_gui2 - a GTK+2 GUI for DCTC
 * Copyright (C) 2002 Eric Prevoteau
 *
 * callbacks.c: Copyright (C) Eric Prevoteau <www@a2pb.gotdns.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: callbacks.c,v 1.65 2004/01/21 15:57:32 ericprev Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <math.h>
#include <linux/sem.h>
#include <fcntl.h>
#include <signal.h>
#include <dirent.h>
#include <ctype.h>
#include <string.h>
#include <gnome.h>
#include <libgnomevfs/gnome-vfs.h>

#include "callbacks.h"
#include "interface.h"
#include "support.h"

#include "main.h"
#include "bdb.h"
#include "misc.h"
#include "gui_layout.h"
#include "gui_define.h"
#include "misc_gtk.h"
#include "dctc_process.h"
#include "ls_cache_clist.h"
#include "do_connect.h"
#include "running_hub_clist.h"
#include "public_hub_clist.h"
#include "recent_hub_clist.h"
#include "seen_hub_clist.h"
#include "user_clist.h"
#include "bookmark.h"
#include "init_fnc.h"
#include "unode.h"
#include "unode_clist.h"
#include "flagged_user_clist.h"
#include "user_file_list_clist.h"
#include "gdl_ctree.h"
#include "find_result_clist.h"
#include "shared_dir_clist.h"
#include "uaddr_clist.h"
#include "gtk_helper.h"
#include "network.h"
#include "custom_hublist_tree.h"
#include "bt_support.h"
#include "notes_buffer.h"
#include "global_user.h"
#include "manage_chat.h"

static void build_start_dl_popup(int from_panel);
void on_load_selected_share_lists_button_clicked (GtkButton       *button, gpointer         user_data);

/*****************************************************************/
/* some functions doesn't seem to be called during the list load */
/* especially when we manipulate a store during a foreach        */
/*****************************************************************/
gboolean user_list_locked=FALSE;


/******************************************/
/* list of all file types                 */
/* in the same order as the displayed one */
/******************************************/
static const char *ftype_str[]={ "any", "audio", "compressed", "document", "exe", "picture", "video", "folder", NULL};

gboolean
on_app1_delete_event                   (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
	gui_full_save(widget,NULL);
	save_notes_text_buffer();
	gtk_widget_destroy(widget);
	gtk_main_quit();

  return FALSE;
}

/* ------------------------------------------------------------------------------------------------------------- */
/************/
/* GTK2: ok */
/************/
void
on_connect1_activate                   (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	GtkWidget *w;

	w=get_widget_by_widget_name(main_window,"main_notebook");
	gtk_notebook_set_page(GTK_NOTEBOOK(w),CONNECT_TAB);
}


/* ------------------------------------------------------------------------------------------------------------- */
/************************/
/* end the current dctc */
/************************/
/* GTK2: ok */
/************/
void
on_disconnect1_activate                (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	send_data_to_dctc("/FORCEQUIT\n");
}


/* ------------------------------------------------------------------------------------------------------------- */
/************/
/* GTK2: ok */
/************/
void
on_find1_activate                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	GtkWidget *w;

	w=get_widget_by_widget_name(main_window,"main_notebook");
	gtk_notebook_set_page(GTK_NOTEBOOK(w),FIND_TAB);
}

/* ------------------------------------------------------------------------------------------------------------- */
/************/
/* GTK2: ok */
/************/
void
on_enter_a_password_menu_entry_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	if(current_dctc!=NULL)
	{  /* connected to something */
		enter_passwd_fnc(NULL);
	}
}

/* ------------------------------------------------------------------------------------------------------------- */
/************/
/* GTK2: ok */
/************/
void
on_exit1_activate                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	gui_full_save(main_window,NULL);
	save_notes_text_buffer();
	gtk_main_quit();
}

/* ------------------------------------------------------------------------------------------------------------- */
/************/
/* GTK2: ok */
/************/
void
on_preferences1_activate               (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	GtkWidget *w;

	w=get_widget_by_widget_name(main_window,"main_notebook");
	gtk_notebook_set_page(GTK_NOTEBOOK(w),USER_PREFERENCES_TAB);
   
	fix_pref_window();
}


/* ------------------------------------------------------------------------------------------------------------- */
void
on_about1_activate                     (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	GtkWidget *w;

	w=create_about2();
	gtk_widget_show(w);
}


/* ------------------------------------------------------------------------------------------------------------- */

void
on_main_notebook_switch_page           (GtkNotebook     *notebook,
                                        GtkNotebookPage *page,
                                        guint            page_num,
                                        gpointer         user_data)
{
	switch(page_num)
	{
		case CONNECT_TAB:
					blink_off("connect_page"); 
					break;

		case CHAT_TAB:
					blink_off("chat_page");
					break;
	
		case USER_FILE_LIST_TAB:
					blink_off("user_file_list_page");
					if(user_list_locked==FALSE)
					{
						user_list_locked=TRUE;
						reload_ls_cache_clist(FALSE);
						user_list_locked=FALSE;
					}
					break;

		case FLAGGED_USER_TAB:
					reload_flagged_user_clist(1);
					break;

		case UADDR_TAB:
					reload_unode_clist(1);
					break;

	}
}


void
on_connect_notebook_switch_page        (GtkNotebook     *notebook,
                                        GtkNotebookPage *page,
                                        guint            page_num,
                                        gpointer         user_data)
{
	if(page_num==RUNNING_HUB_TAB)
	{
		/* we enter the page of running process */
		fill_running_hub_clist();
		gtk_widget_show(get_widget_by_widget_name(main_window,"bookmark_button"));
		gtk_widget_hide(get_widget_by_widget_name(main_window,"delete_selected_bookmark_button"));
		gtk_widget_hide(get_widget_by_widget_name(main_window,"start_dctc_selected_hub_button"));
		gtk_widget_show(get_widget_by_widget_name(main_window,"terminate_selected_dctcs_button"));
		gtk_widget_show(get_widget_by_widget_name(main_window,"kill_selected_dctcs_button"));
	}  
	else if(page_num==PUBLIC_HUB_TAB)
	{
		/* we enter the public hub list */
		fill_pub_hub_clist(FALSE);
		gtk_widget_show(get_widget_by_widget_name(main_window,"bookmark_button"));
		gtk_widget_hide(get_widget_by_widget_name(main_window,"delete_selected_bookmark_button"));
		gtk_widget_show(get_widget_by_widget_name(main_window,"start_dctc_selected_hub_button"));
		gtk_widget_hide(get_widget_by_widget_name(main_window,"terminate_selected_dctcs_button"));
		gtk_widget_hide(get_widget_by_widget_name(main_window,"kill_selected_dctcs_button"));
		blink_off("connect_public_label");
	}
	else if(page_num==RECENT_HUB_TAB)
	{
		/* we enter the recent hub list */
		fill_recent_hub_clist();
		gtk_widget_show(get_widget_by_widget_name(main_window,"bookmark_button"));
		gtk_widget_hide(get_widget_by_widget_name(main_window,"delete_selected_bookmark_button"));
		gtk_widget_show(get_widget_by_widget_name(main_window,"start_dctc_selected_hub_button"));
		gtk_widget_hide(get_widget_by_widget_name(main_window,"terminate_selected_dctcs_button"));
		gtk_widget_hide(get_widget_by_widget_name(main_window,"kill_selected_dctcs_button"));
	}
	else if(page_num==FAVORITE_HUB_TAB)
	{
		gtk_widget_hide(get_widget_by_widget_name(main_window,"bookmark_button"));
		gtk_widget_show(get_widget_by_widget_name(main_window,"delete_selected_bookmark_button"));
		gtk_widget_show(get_widget_by_widget_name(main_window,"start_dctc_selected_hub_button"));
		gtk_widget_hide(get_widget_by_widget_name(main_window,"terminate_selected_dctcs_button"));
		gtk_widget_hide(get_widget_by_widget_name(main_window,"kill_selected_dctcs_button"));
	}
	else if(page_num==SEEN_PUBLIC_HUB_TAB)
	{
		fill_seen_hub_clist(FALSE);
		gtk_widget_show(get_widget_by_widget_name(main_window,"bookmark_button"));
		gtk_widget_hide(get_widget_by_widget_name(main_window,"delete_selected_bookmark_button"));
		gtk_widget_show(get_widget_by_widget_name(main_window,"start_dctc_selected_hub_button"));
		gtk_widget_hide(get_widget_by_widget_name(main_window,"terminate_selected_dctcs_button"));
		gtk_widget_hide(get_widget_by_widget_name(main_window,"kill_selected_dctcs_button"));
	}
}

/* ------------------------------------------------------------------------------------------------------------- */
/***********************************************/
/* remove one selected entry from the bookmark */
/***********************************************/
/* GTK 2: ok */
/*************/
static void toggle_autostart_flag_of_one_bm_entry(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	gint mark_num;

	gtk_tree_model_get(model,iter,FHC_MARK_NUM,&mark_num,-1);
	toggle_entry_flag_from_bookmark_by_id(mark_num,AUTOSTART_FLAG,1);      /* toggle without reload */
}

/*****************************************************************/
/* toggle AUTOSTART flag of all selected entries of the bookmark */
/*****************************************************************/
/* GTK 2: ok */
/*************/
void
on_toggle_autostart_item_activate      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	GtkWidget *w;
	GtkTreeSelection *slc;

	w=get_widget_by_widget_name(main_window,"hub_favorite_clist");
	if(w==NULL)
		return;

	slc=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));
	gtk_tree_selection_selected_foreach(slc,toggle_autostart_flag_of_one_bm_entry,NULL);

	reload_bookmark();
}

/***********************************************/
/* remove one selected entry from the bookmark */
/***********************************************/
/* GTK 2: ok */
/*************/
static void set_profile_of_one_bm_entry(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	gint mark_num;

	gtk_tree_model_get(model,iter,FHC_MARK_NUM,&mark_num,-1);
	set_entry_profile_from_bookmark_by_id(	mark_num,
														data /* == profile */,
														 1);     /* alter without reload */
}

/*****************************************************************************************/
/* this function is called when the user clicks on a profile entry of the fav_popup_menu */
/*****************************************************************************************/
/* prof_name is the profile name to use */
/****************************************/
/* GTK 2: ok */
/*************/
static void set_fav_profile(GtkMenuItem *menuitem, gpointer prof_name)
{
	GtkWidget *w;
	GtkTreeSelection *slc;

	w=get_widget_by_widget_name(main_window,"hub_favorite_clist");
	if(w==NULL)
		return;

	slc=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));
	gtk_tree_selection_selected_foreach(slc,set_profile_of_one_bm_entry,prof_name);

	reload_bookmark();
}

/*******************************************************************/
/* rebuild a new favorite_popup menu including the current profile */
/*******************************************************************/
/* GTK 2: ok */
/*************/
static void build_favorite_popup(void)
{
	GtkWidget *mitem;
	static GStringChunk *gsc_entry=NULL;
	gchar *ent;

	if(fav_popup!=NULL)
	{
		gtk_widget_destroy(fav_popup);
	}

	if(gsc_entry!=NULL)
	{
		g_string_chunk_free(gsc_entry);
	}
	gsc_entry=g_string_chunk_new(64);

	/* create the default fav_popup_menu */
	fav_popup=create_fav_popup_menu();

	/* and add it the list of profiles */
	mitem=gtk_menu_item_new_with_label(_("-- use profile"));
	gtk_widget_set_sensitive(mitem,FALSE);
	gtk_menu_append(GTK_MENU(fav_popup),mitem);
	gtk_widget_show(mitem);

	/* first, an empty profile to use the current one */
	mitem=gtk_menu_item_new_with_label(_("Current one"));
	gtk_menu_append(GTK_MENU(fav_popup),mitem);
	ent=g_string_chunk_insert(gsc_entry,"");
	gtk_signal_connect(GTK_OBJECT(mitem),"activate", GTK_SIGNAL_FUNC(set_fav_profile),ent);
	gtk_widget_show(mitem);

	/* and add the list of existing one */
	{
		void *iter;
		iter=gnome_config_init_iterator("/" PROGNAME "/ProfileNames/");
		while(iter!=NULL)
		{
			char *prof_name=NULL;
			char *buf=NULL;

			iter=gnome_config_iterator_next(iter,&prof_name,&buf);
			if(iter!=NULL)
			{
				/* prof_name is the profile name, buf is unused */
				if((prof_name!=NULL)&&(strlen(prof_name)))
				{
					mitem=gtk_menu_item_new_with_label(prof_name);
					gtk_menu_append(GTK_MENU(fav_popup),mitem);
					ent=g_string_chunk_insert(gsc_entry,prof_name);
					gtk_signal_connect(GTK_OBJECT(mitem),"activate", GTK_SIGNAL_FUNC(set_fav_profile),ent);
					gtk_widget_show(mitem);
				}
			}
		}
	}
}

/*******************************************************************/
/* rebuild a new favorite_popup menu including the current profile */
/*******************************************************************/
/* GTK 2: ok */
/*************/
gboolean
on_hub_favorite_clist_button_press_event
                                        (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
	if(event==NULL)
		return TRUE;

	if((event->button==1)&&(event->type==GDK_2BUTTON_PRESS))
	{  /* left click */
		/* double click on a row */
		GtkTreePath *path;

		if(gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(widget),(gint) event->x, (gint) event->y, &path,NULL,NULL,NULL)==TRUE)
		{
			GtkTreeIter iter;
			GtkTreeModel *gtm;
			char *addr=NULL;
			char *profile=NULL;
	
			gtk_tree_model_get_iter (gtm=gtk_tree_view_get_model(GTK_TREE_VIEW(widget)),&iter,path);
	
			gtk_tree_model_get(gtm,&iter,FHC_HUB_ADDR_COL,&addr,-1);
			gtk_tree_model_get(gtm,&iter,FHC_PROFILE,&profile,-1);

			if((addr!=NULL)&&(strlen(addr)!=0))
			{
				if((profile==NULL)||(strlen(profile)==0))
					start_a_new_dctc(addr,0,NULL);      /* start with waiting */
				else
					start_a_new_dctc(addr,0,profile); /* start with waiting */
			}

			if(addr)
				free(addr);
			if(profile)
				free(profile);

			gtk_tree_path_free(path);
		}
		return TRUE;
	}

	if(event->button==3)
	{  /* right-click */
		build_favorite_popup();
		gtk_menu_popup(GTK_MENU(fav_popup),NULL,NULL,NULL,NULL,event->button,event->time);
		return TRUE;
	}

	return FALSE;
}


gboolean
on_hub_favorite_clist_key_press_event  (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{

  return FALSE;
}


gboolean
on_hub_public_clist_button_press_event (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
	if(event==NULL)
		return TRUE;

	if((event->button==1)&&(event->type==GDK_2BUTTON_PRESS))
	{  /* left click */
		/* double click on a row */
		GtkTreePath *path;

		if(gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(widget),(gint) event->x, (gint) event->y, &path,NULL,NULL,NULL)==TRUE)
		{
			GtkTreeIter iter;
			GtkTreeModel *gtm;
			char *addr=NULL;
	
			gtk_tree_model_get_iter (gtm=gtk_tree_view_get_model(GTK_TREE_VIEW(widget)),&iter,path);
	
			gtk_tree_model_get(gtm,&iter,PHC_HUB_ADDR_COL,&addr,-1);

			if((addr!=NULL)&&(strlen(addr)!=0))
			{
				start_a_new_dctc(addr,0,NULL);      /* start with waiting */
			}

			if(addr)
				free(addr);

			gtk_tree_path_free(path);
		}
		return TRUE;
	}

	
  return FALSE;
}


gboolean
on_hub_recent_clist_button_press_event (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
	if(event==NULL)
		return TRUE;

	if((event->button==1)&&(event->type==GDK_2BUTTON_PRESS))
	{  /* left click */
		/* double click on a row */
		GtkTreePath *path;

		if(gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(widget),(gint) event->x, (gint) event->y, &path,NULL,NULL,NULL)==TRUE)
		{
			GtkTreeIter iter;
			GtkTreeModel *gtm;
			char *addr=NULL;
	
			gtk_tree_model_get_iter (gtm=gtk_tree_view_get_model(GTK_TREE_VIEW(widget)),&iter,path);
	
			gtk_tree_model_get(gtm,&iter,REHC_HUB_ADDR_COL,&addr,-1);

			if((addr!=NULL)&&(strlen(addr)!=0))
			{
				start_a_new_dctc(addr,0,NULL);      /* start with waiting */
			}

			if(addr)
				free(addr);

			gtk_tree_path_free(path);
		}
		return TRUE;
	}


  return FALSE;
}

/* ------------------------------------------------------------------------------------------------------------- */
/******************************/
/* clear the recent hubs list */
/******************************/
/* GTK2: ok */
/************/
void
on_clear_recent_list_button_clicked    (GtkButton       *button,
                                        gpointer         user_data)
{
	GString *s;
	char *path;

	s=g_string_new(NULL);
	path=getenv("HOME");
	g_string_sprintf(s,"%s/.dctc/recent",(path!=NULL)?path:".");
	unlink(s->str);
	g_string_free(s,TRUE);
	fill_recent_hub_clist();
}

/* ------------------------------------------------------------------------------------------------------------- */

gboolean
on_seen_hub_clist_button_press_event	(GtkWidget		 *widget,
													 GdkEventButton  *event,
													 gpointer			user_data)
{
	if(event==NULL)
		return TRUE;

	if((event->button==1)&&(event->type==GDK_2BUTTON_PRESS))
	{  /* left click */
		/* double click on a row */
		GtkTreePath *path;

		if(gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(widget),(gint) event->x, (gint) event->y, &path,NULL,NULL,NULL)==TRUE)
		{
			GtkTreeIter iter;
			GtkTreeModel *gtm;
			char *addr=NULL;
	
			gtk_tree_model_get_iter (gtm=gtk_tree_view_get_model(GTK_TREE_VIEW(widget)),&iter,path);
	
			gtk_tree_model_get(gtm,&iter,SHC_HUB_ADDR_COL,&addr,-1);

			if((addr!=NULL)&&(strlen(addr)!=0))
			{
				start_a_new_dctc(addr,0,NULL);      /* start with waiting */
			}

			if(addr)
				free(addr);

			gtk_tree_path_free(path);
		}
		return TRUE;
	}


  return FALSE;
}

/* ------------------------------------------------------------------------------------------------------------- */
/************************************************************************/
/* return the row number of the first selected row, -1 if none selected */
/************************************************************************/
static int get_first_selected_row_of_clist(GtkTreeModel *gtm, GtkTreeSelection *slc, int nb_child)
{
	int i;
	GtkTreeIter iter;

	for(i=0;i<nb_child;i++)
	{
		if(gtk_tree_model_iter_nth_child(gtm,&iter,NULL,i)==TRUE)
		{
			if(gtk_tree_selection_iter_is_selected(slc,&iter)==TRUE)
				return i;
		}
	}
	return -1;
}

/**************************************************************************/
/* check if the given row/column of the clist contains the wanted pattern */
/**************************************************************************/
/* output: 0=no, 1=yes */
/***********************/
static int row_contains_pattern(GtkTreeModel *gtm, GtkTreeIter *iter,int column, GString *pattern)
{
	char *txt;
	int ret;

	gtk_tree_model_get(gtm,iter,column,&txt,-1);

	if(txt==NULL)
		return 0;

	if(my_strcasestr(pattern,txt)!=NULL)
		ret=1;
	else
		ret=0;
	g_free(txt);
	return ret;
}

/***************************************************************************/
/* select a hub matching the given pattern in the currently displayed list */
/****************************************************************************/
/* input: start_position: 0= top of the list, -1= current position          */
/*        direction: 1=to the bottom of the list, -1=to the top of the list */
/*****************************************************************************/
/* NOTE: to avoid multiple characters conversion, everything is done in UTF8 */
/*****************************************************************************/
static void find_hub_in_list_fnc(gint start_position, int direction)
{
	GtkWidget *w;
	const char *pattern;
	int hub_addr=-1,hub_name=-1,hub_desc=-1;
	GtkWidget *wlist;
	GString *gstr_pattern;
	int nb_child;
	GtkTreeView *gtv;
	GtkTreeModel *gtm;
	GtkTreeIter iter;
	GtkTreeSelection *slc;
	GtkTreePath *gtp;

	if((direction!=-1)&&(direction!=1))
	{
		fprintf(stderr,"find_hub_in_list_fnc: invalid direction, only -1 and 1 allowed\n");
		return;
	}

	w=get_widget_by_widget_name(main_window,"hubs_to_select_entry");
	if(w==NULL)
	{
		fprintf(stderr,"no widget: hubs_to_select_entry\n");
		return;
	}

	pattern=gtk_entry_get_text(GTK_ENTRY(w));
	if(strlen(pattern)<=0)
	{
		return;		/* nothing to search */
	}

	switch(gtk_notebook_get_current_page(GTK_NOTEBOOK(get_widget_by_widget_name(main_window,"connect_notebook"))))
	{
		case RUNNING_HUB_TAB:
					wlist=get_widget_by_widget_name(main_window,"running_hub_clist");
					hub_addr=RHC_HUB_ADDR_COL;
					hub_name=RHC_HUB_NAME_COL;
					hub_desc=-1;
					break;

		case FAVORITE_HUB_TAB:
					wlist=get_widget_by_widget_name(main_window,"hub_favorite_clist");
					hub_addr=FHC_HUB_ADDR_COL;
					hub_name=FHC_HUB_NAME_COL;
					hub_desc=FHC_DESC_COL;
					break;
				
		case PUBLIC_HUB_TAB:	
					wlist=get_widget_by_widget_name(main_window,"hub_public_clist");
					hub_addr=PHC_HUB_ADDR_COL;
					hub_name=PHC_HUB_NAME_COL;
					hub_desc=PHC_DESC_COL;
					break;

		case RECENT_HUB_TAB:
					wlist=get_widget_by_widget_name(main_window,"hub_recent_clist");
					hub_addr=REHC_HUB_ADDR_COL;
					hub_name=REHC_HUB_NAME_COL;
					hub_desc=-1;
					break;

		case SEEN_PUBLIC_HUB_TAB:
					wlist=get_widget_by_widget_name(main_window,"seen_hub_clist");
					hub_addr=SHC_HUB_ADDR_COL;
					hub_name=SHC_HUB_NAME_COL;
					hub_desc=SHC_DESC_COL;
					break;

		default:
					return;
	}
	
	gtm=gtk_tree_view_get_model(gtv=GTK_TREE_VIEW(wlist));
	slc=gtk_tree_view_get_selection(gtv);
	nb_child=gtk_tree_model_iter_n_children(gtm,NULL);

	if(nb_child==0)
		return;

	if(start_position==-1)
	{
		start_position=get_first_selected_row_of_clist(gtm,slc,nb_child)+direction;	/* we start at the next position */
	}

	gtk_tree_selection_unselect_all(slc);

	gstr_pattern=g_string_new(pattern);
	/* we will scan the list and try to find a row containing the wanted pattern */
	if(direction>0)
	{
		while(start_position<nb_child)
		{
			if(gtk_tree_model_iter_nth_child(gtm,&iter,NULL,start_position)==TRUE)
			{
				if(  ((hub_addr!=-1)&&(row_contains_pattern(gtm,&iter,hub_addr,gstr_pattern))) ||
				  		((hub_name!=-1)&&(row_contains_pattern(gtm,&iter,hub_name,gstr_pattern))) ||
				  		((hub_desc!=-1)&&(row_contains_pattern(gtm,&iter,hub_desc,gstr_pattern))) )
				{
					gtp=gtk_tree_model_get_path(gtm,&iter);
					gtk_tree_view_scroll_to_cell(gtv,gtp,NULL,TRUE,0.2,0.0);
					//gtk_tree_selection_select_iter(gtm,&iter);
					gtk_tree_view_set_cursor(gtv,gtp,NULL,FALSE);
					gtk_tree_path_free(gtp);
					break;
				}
			}
			start_position++;
		}
	}
	else
	{
		while(start_position>=0)
		{
			if(gtk_tree_model_iter_nth_child(gtm,&iter,NULL,start_position)==TRUE)
			{
				if(  ((hub_addr!=-1)&&(row_contains_pattern(gtm,&iter,hub_addr,gstr_pattern))) ||
				  		((hub_name!=-1)&&(row_contains_pattern(gtm,&iter,hub_name,gstr_pattern))) ||
				  		((hub_desc!=-1)&&(row_contains_pattern(gtm,&iter,hub_desc,gstr_pattern))) )
				{
					gtp=gtk_tree_model_get_path(gtm,&iter);
					gtk_tree_view_scroll_to_cell(gtv,gtp,NULL,TRUE,0.2,0.0);
					//gtk_tree_selection_select_iter(gtm,&iter);
					gtk_tree_view_set_cursor(gtv,gtp,NULL,FALSE);
					gtk_tree_path_free(gtp);
					break;
				}
			}
			start_position--;
		}
	}
	g_string_free(gstr_pattern,TRUE);
}


void
on_hubs_to_select_entry_activate       (GtkEntry        *entry,
                                        gpointer         user_data)
{
	find_hub_in_list_fnc(0,1);    /* from top of the list to the bottom */
}


void
on_find_hub_button_clicked             (GtkButton       *button,
                                        gpointer         user_data)
{
	find_hub_in_list_fnc(0,1);    /* from top of the list to the bottom */
}


void
on_find_prev_hub_button_clicked        (GtkButton       *button,
                                        gpointer         user_data)
{
	find_hub_in_list_fnc(-1,-1);     /* from current position to the top of the list */
}


void
on_find_next_hub_button_clicked        (GtkButton       *button,
                                        gpointer         user_data)
{
	find_hub_in_list_fnc(-1,1);      /* from current position to the botton of the list */
}

/* ------------------------------------------------------------------------------------------------------------- */
/****************************************/
/* refresh hub list currently displayed */
/****************************************/
/* GTK 2: ok */
/*************/
void
on_refresh_sitelist_button_clicked     (GtkButton       *button,
                                        gpointer         user_data)
{
	switch(gtk_notebook_get_current_page(GTK_NOTEBOOK(get_widget_by_widget_name(main_window,"connect_notebook"))))
	{
		case RUNNING_HUB_TAB:
					fill_running_hub_clist();
					break;

		case PUBLIC_HUB_TAB: 
					fill_pub_hub_clist(TRUE);
					break;

		case RECENT_HUB_TAB:
					fill_recent_hub_clist();
					break;

		case SEEN_PUBLIC_HUB_TAB:
					fill_seen_hub_clist(TRUE);
					break;
	}
}

/* ------------------------------------------------------------------------------------------------------------- */
typedef struct
{
	int hubname_col;
	int description_col;
	int address_col;
} BM_TEMP;

/******************************************/
/* add one selected entry to the bookmark */
/******************************************/
/* GTK 2: ok */
/*************/
static void add_one_bm_entry(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	BM_TEMP *col=data;
	char *hubname=NULL;
	char *description=NULL;
	char *address=NULL;

	if(col->hubname_col!=-1)
	{
		gtk_tree_model_get(model,iter,col->hubname_col,&hubname,-1);
	}

	if(hubname==NULL)
		hubname=strdup("No name");

	if(col->description_col!=-1)
	{
		gtk_tree_model_get(model,iter,col->description_col,&description,-1);
	}

	if(description==NULL)
		description=strdup("No description");

	if(col->address_col!=-1)
	{
		gtk_tree_model_get(model,iter,col->address_col,&address,-1);
	}
	
	if(address==NULL)
		address=strdup("No address");

	add_entry_to_bookmark(-1,hubname,description,address,NULL,NULL,1);		/* add without reload */

	if(hubname)
		free(hubname);
	if(description)
		free(description);
	if(address)
		free(address);
}

/*************************************************************************************/
/* add the selected entries of the given clist (GtkTreeView) to the bookmark address */
/*************************************************************************************/
/* GTK 2: ok */
/*************/
static void add_bm_entries(char *clist_name,int hubname_col,int description_col, int address_col)
{
	GtkWidget *w;
	BM_TEMP col;
	GtkTreeSelection *slc;

	w=get_widget_by_widget_name(main_window,clist_name);
	if(w==NULL)
		return;

	col.hubname_col=hubname_col;
	col.description_col=description_col;
	col.address_col=address_col;

	slc=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));
	gtk_tree_selection_selected_foreach(slc,add_one_bm_entry,&col);

	reload_bookmark();
}

/**********************************************************************************/
/* add the selected entries of the currently display page to the bookmark address */
/**********************************************************************************/
/* GTK 2: in progress */
/*************/
void
on_bookmark_button_clicked             (GtkButton       *button,
                                        gpointer         user_data)
{
	switch(gtk_notebook_get_current_page(GTK_NOTEBOOK(get_widget_by_widget_name(main_window,"connect_notebook"))))
	{
		case RUNNING_HUB_TAB:
					add_bm_entries("running_hub_clist",RHC_HUB_NAME,-1,RHC_HUB_ADDR_COL);
					break;

		case PUBLIC_HUB_TAB:
					add_bm_entries("hub_public_clist",PHC_HUB_NAME,PHC_DESC,PHC_HUB_ADDR_COL);
					break;

		case RECENT_HUB_TAB:
					add_bm_entries("hub_recent_clist",REHC_HUB_NAME,-1,REHC_HUB_ADDR_COL);
					break;

		case SEEN_PUBLIC_HUB_TAB:
					add_bm_entries("seen_hub_clist",SHC_HUB_NAME,SHC_DESC,SHC_HUB_ADDR_COL);
					break;
	}
}

/* ------------------------------------------------------------------------------------------------------------- */
/***********************************************/
/* remove one selected entry from the bookmark */
/***********************************************/
/* GTK 2: ok */
/*************/
static void delete_one_bm_entry(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	gint mark_num;

	gtk_tree_model_get(model,iter,FHC_MARK_NUM,&mark_num,-1);
	delete_entry_from_bookmark_by_id(mark_num,1);     /* delete without reload */
}

/***********************************************/
/* delete all selected entries of the bookmark */
/***********************************************/
/* GTK 2: ok */
/*************/
void
on_delete_selected_bookmark_button_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *w;
	GtkTreeSelection *slc;

	w=get_widget_by_widget_name(main_window,"hub_favorite_clist");
	if(w==NULL)
		return;

	slc=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));
	gtk_tree_selection_selected_foreach(slc,delete_one_bm_entry,NULL);
	reload_bookmark();
}


/* ------------------------------------------------------------------------------------------------------------- */
typedef struct
{
	int hub_addr_col;
	int profile_col;
} STRT_TEMP;

/*************************************************************************************/
/* add the selected entries of the given clist (GtkTreeView) to the bookmark address */
/*************************************************************************************/
/* GTK 2: ok */
/*************/
static void start_one_dctc_selected_list_entry(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	STRT_TEMP *st=data;
	char *addr=NULL;
	char *profile=NULL;
	
	gtk_tree_model_get(model,iter,st->hub_addr_col,&addr,-1);

	if(st->profile_col!=-1)
	{
		gtk_tree_model_get(model,iter,st->profile_col,&profile,-1);
	}

	if((addr!=NULL)&&(strlen(addr)!=0))
	{
		if((profile==NULL)||(strlen(profile)==0))
			start_a_new_dctc(addr,1,NULL);      /* start without waiting */
		else
			start_a_new_dctc(addr,1,profile); /* start without waiting */
	}

	if(addr)
		free(addr);
	if(profile)
		free(profile);
}

/********************************************************/
/* start all selected entries of currently display page */
/********************************************************/
/* GTK 2: ok */
/*************/
void
on_start_dctc_selected_hub_button_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	const char *clist_name;
	STRT_TEMP st;
	GtkWidget *w;
	GtkTreeSelection *slc;


	switch(gtk_notebook_get_current_page(GTK_NOTEBOOK(get_widget_by_widget_name(main_window,"connect_notebook"))))
	{
		case FAVORITE_HUB_TAB:
					clist_name="hub_favorite_clist";
					st.hub_addr_col=FHC_HUB_ADDR_COL;
					st.profile_col=FHC_PROFILE;
					break;

		case PUBLIC_HUB_TAB:
					clist_name="hub_public_clist";
					st.hub_addr_col=PHC_HUB_ADDR_COL;
					st.profile_col=-1;
					break;

		case RECENT_HUB_TAB:
					clist_name="hub_recent_clist";
					st.hub_addr_col=REHC_HUB_ADDR_COL;
					st.profile_col=-1;
					break;

		case SEEN_PUBLIC_HUB_TAB:
					clist_name="seen_hub_clist";
					st.hub_addr_col=SHC_HUB_ADDR_COL;
					st.profile_col=-1;
					break;

		default:
					return;
	}

	w=get_widget_by_widget_name(main_window,clist_name);
	if(w==NULL)
		return;

	slc=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));
	gtk_tree_selection_selected_foreach(slc,start_one_dctc_selected_list_entry,&st);
}

/* ------------------------------------------------------------------------------------------------------------- */

void
on_start_dummy_client_button_clicked   (GtkButton       *button,
                                        gpointer         user_data)
{
	start_a_new_dctc("dummy_client",0,NULL);
}


void
on_connect_entry_activate              (GtkEntry        *entry,
                                        gpointer         user_data)
{
	GtkWidget *w;
	const char *name;

	w=get_widget_by_widget_name(main_window,"connect_entry");
	if(w==NULL)
		return;

	name=gtk_entry_get_text(GTK_ENTRY(w));

	if(name&&(strlen(name)>3))
		start_a_new_dctc(name,0,NULL);
}


void
on_do_connection_button_clicked        (GtkButton       *button,
                                        gpointer         user_data)
{
	on_connect_entry_activate(NULL,NULL);
}


void
on_do_reconnect_now_button_clicked     (GtkButton       *button,
                                        gpointer         user_data)
{
	send_data_to_dctc("/RECON\n");
}

static void go_to_another_hub_clist_selected(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	int *col_num=data;
	char *hub_addr;

	if(*col_num==-1)
		return;

	gtk_tree_model_get(model,iter,*col_num,&hub_addr,-1);
	if(hub_addr!=NULL)
	{
		if(strlen(hub_addr)>3)
			go_to_another_hub(hub_addr);
		free(hub_addr);
	}
	*col_num=-1;
}

void
on_do_goto_button_clicked              (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *w;

	w=get_widget_by_widget_name(main_window,"connect_entry");
	if(w!=NULL)
	{
		const char *name;
		name=gtk_entry_get_text(GTK_ENTRY(w));

		if(name&&(strlen(name)>3))	 /* if a valid name exist, use it */
			go_to_another_hub(name);
		else
		{
			const char *list_name;
			int col_num;
			GtkWidget *w;
			GtkTreeSelection *gts;

			/* else use the selected site in the currently visible hub list */
			switch(gtk_notebook_get_current_page(GTK_NOTEBOOK(get_widget_by_widget_name(main_window,"connect_notebook"))))
			{
				case RUNNING_HUB_TAB:
							list_name="running_hub_clist";
							col_num=RHC_HUB_ADDR_COL;
							break;

				case PUBLIC_HUB_TAB:
							list_name="hub_public_clist";
							col_num=PHC_HUB_ADDR_COL;
							break;

				case RECENT_HUB_TAB:
							list_name="hub_recent_clist";
							col_num=REHC_HUB_ADDR_COL;
							break;

				case SEEN_PUBLIC_HUB_TAB:
							list_name="seen_hub_clist";
							col_num=SHC_HUB_ADDR_COL;
							break;

				default: return;
			}

			w=get_widget_by_widget_name(main_window,list_name);
			if(w==NULL)
				return;

			gts=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));
			gtk_tree_selection_selected_foreach(gts,go_to_another_hub_clist_selected,&col_num);
		}
	}
}

/* ------------------------------------------------------------------------------------------------------------- */
/***************************************************************/
/* send a /QUIT\n to the DCTC having the given socket filename */
/***************************************************************/
/* GTK 2: ok */
/*************/
static void send_quit_command_to_client(char *unx_sock_name)
{
	GString *cur_udp_path;
	struct sockaddr_un name;

	if(local_udp_socket==-1)
		return;

	cur_udp_path=g_string_new(dctc_dir->str);
	g_string_sprintfa(cur_udp_path,"/%s.udp",unx_sock_name);

	name.sun_family=AF_UNIX;
	strcpy(name.sun_path,cur_udp_path->str);
	if(sendto(local_udp_socket,"/FORCEQUIT\n",strlen("/FORCEQUIT\n"),MSG_NOSIGNAL|MSG_DONTWAIT,(void*)&name,sizeof(struct sockaddr_un))!=strlen("/FORCEQUIT\n"))
	{
		perror("send_quit_command_to_client");
	}
	
	g_string_free(cur_udp_path,TRUE); 
}

static void soft_end_one_dctc_entry(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	GArray *ga=data;
	gchar *unx_sock_name;

	gtk_tree_model_get(model,iter,RHC_PROCESS_ID_COL,&unx_sock_name,-1);
	if(unx_sock_name)
	{
		if(strlen(unx_sock_name))
		{
			glong proc_pid;

			send_quit_command_to_client(unx_sock_name);
			gtk_tree_model_get(model,iter,RHC_PROCESS_PID_COL,&proc_pid,-1);
			g_array_append_val(ga,proc_pid);
		}

		free(unx_sock_name);
	}
}

/******************************************************/
/* terminate selected DCTC with more or less violence */
/******************************************************/
/* GTK 2: ok */
/*************/
static void terminate_selected_dctc_entries(int kill_it)
{
	GtkWidget *w;
	GtkTreeSelection *slc;
	GArray *ga=NULL;

	w=get_widget_by_widget_name(main_window,"running_hub_clist");
	if(w==NULL)
		return;

	ga=g_array_new(FALSE,FALSE,sizeof(glong));

	slc=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));
	gtk_tree_selection_selected_foreach(slc,soft_end_one_dctc_entry,ga);

	if((kill_it)&&(ga->len!=0))
	{
		int i;
		sleep(3);		/* wait a little before killing everybody */
		for(i=0;i<ga->len;i++)
		{
			pid_t num;

			num=g_array_index(ga,glong,i);

			kill(num,SIGKILL);
		}
	}

	if(ga->len!=0)
	{
		if(kill_it)
		{
			sleep(1);
		}
		else
		{
			sleep(2);
		}
		on_refresh_sitelist_button_clicked(NULL,NULL);
	}
	g_array_free(ga,TRUE);
}

/******************************************************/
/* try to end softly all selected running DCTC client */
/******************************************************/
/* GTK 2: ok */
/*************/
void
on_terminate_selected_dctcs_button_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	terminate_selected_dctc_entries(0);
}


/******************************************************/
/* try to end hardly all selected running DCTC client */
/******************************************************/
/* GTK 2: ok */
/*************/
void
on_kill_selected_dctcs_button_clicked  (GtkButton       *button,
                                        gpointer         user_data)
{
	terminate_selected_dctc_entries(1);
}
/* ------------------------------------------------------------------------------------------------------------- */


void
on_dctc_cmd_entry_activate             (GtkEntry        *entry,
                                        gpointer         user_data)
{
	GString *str;
	const char *param;
	
	param=gtk_entry_get_text(entry);
	if(param==NULL)
		return;
	if(strlen(param)<1)
		return;

	str=g_string_new(param);
	if(str->str[str->len-1]!='\n')
	{	  
		str=g_string_append_c(str,'\n');
	}

	send_data_to_dctc(str->str);
	g_string_free(str,TRUE);
}

/* ------------------------------------------------------------------------------------------------------------- */
/********************************************************/
/* convert all \n in \r except the last one (last byte) */
/********************************************************/
static void translate_char(GString *str,int start_pos)
{
	int i;

	/* convert \n into \r */
	for(i=start_pos;i<(str->len-1);i++)
	{
		if(str->str[i]=='\n')
			str->str[i]='\r';
	}

	/* convert | into &#124; */
	i=start_pos;
	while(i<str->len)
	{
		if(str->str[i]=='|')
		{
			str->str[i]='&';
			i++;
			g_string_insert(str,i,"#124;");
			i+=5;
		}
		else
			i++;
	}
}

/* ------------------------------------------------------------------------------------------------------------- */
/******************************************************************/
/* complete the nick at the cursor position in the given editable */
/******************************************************************/
/* GTK2: ok */
/************/
void do_nick_completion(GtkEditable *w)
{
	gchar *str;
	char *beg;
	int ln;

	int cur_pos=gtk_editable_get_position(w);

	if(cur_pos<2)		/* at least 2 characters to perform a completion */
		return;

	str=gtk_editable_get_chars(w,0,cur_pos);
	beg=strrchr(str,' ');
	if(beg==NULL)
		beg=str;
	else
		beg++;

	if((ln=strlen(beg))>1)
	{
		/* now, beg is the beginning of a string which can be the beginning of a nickname */
		GString *comp;

		comp=complete_a_nick(beg);

		if(comp!=NULL)
		{
			if(ln!=comp->len)
			{
				gtk_editable_insert_text(w,comp->str+ln,comp->len-ln,&cur_pos);
			}
			gtk_editable_insert_text(w," ",1,&cur_pos);
			gtk_editable_set_position(w,cur_pos);
			g_string_free(comp,TRUE);
		}
	}
	
	g_free(str);
}

/* ------------------------------------------------------------------------------------------------------------- */
/****************************/
/* discard editable content */
/****************************/
/* GTK2: ok */
/************/
static void wipe_gtk_text(char *widget_name)
{
   GtkWidget *w;
	GtkTextBuffer *gtb;
	GtkTextIter p0,p1;

   w=get_widget_by_widget_name(main_window,widget_name);
   if(w==NULL)
      return;

	gtb=gtk_text_view_get_buffer(GTK_TEXT_VIEW(w));
	gtk_text_buffer_get_start_iter(gtb,&p0);
	gtk_text_buffer_get_end_iter(gtb,&p1);
	gtk_text_buffer_delete(gtb,&p0,&p1);
}


/* ------------------------------------------------------------------------------------------------------------- */
static gunichar *unichar_inf=NULL;
static gunichar *unichar_sup=NULL;

static gboolean find_a_char(gunichar sh, gpointer data)
{
	gunichar *cmp=data;

	return sh==*cmp;
}

/***********************************************************/
/* flag the user at the cursor position in the public chat */
/***********************************************************/
/* GTK2: not tested */
/********************/
void
on_pubchat_flag_selected_user_button_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *w;
	GtkTextBuffer *gtb;
	GtkTextIter cursor,ending;
	GtkTextMark *cursor_mark;
	gchar *utf8_nickname;
	gchar *nickname;
	
	if(unichar_inf==NULL)
	{
		unichar_inf=g_utf8_to_ucs4_fast("<",1,NULL);
		unichar_sup=g_utf8_to_ucs4_fast(">",1,NULL);
	}

	w=get_widget_by_widget_name(main_window,"chat_output");
	gtb=gtk_text_view_get_buffer(GTK_TEXT_VIEW(w));
	
	cursor_mark=gtk_text_buffer_get_insert(gtb);
	gtk_text_buffer_get_iter_at_mark(gtb,&cursor,cursor_mark);
	if(gtk_text_iter_backward_find_char(&cursor,find_a_char,unichar_inf,NULL)==FALSE)
		return;
	if(gtk_text_iter_forward_char(&cursor)==FALSE)
		return;
	ending=cursor;
	if(gtk_text_iter_forward_find_char(&ending,find_a_char,unichar_sup,NULL)==FALSE)
		return;
	
	utf8_nickname=gtk_text_buffer_get_text(gtb,&cursor,&ending,FALSE);
	nickname=g_locale_from_utf8(utf8_nickname,-1,NULL,NULL,NULL);
	if((strchr(nickname,'<')==NULL)&&(strchr(nickname,'\n')==NULL))
		auto_flag_user(nickname);
	g_free(utf8_nickname);
	g_free(nickname);
}
/* ------------------------------------------------------------------------------------------------------------- */


gboolean
on_find_result_button_press_event      (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
	if(event==NULL)
		return TRUE;

	if((event->button==1)&&(event->type==GDK_2BUTTON_PRESS))
	{  /* left click */
		/* double click on a row */
		GtkTreePath *path;

		if(gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(widget),(gint) event->x, (gint) event->y, &path,NULL,NULL,NULL)==TRUE)
		{
			GtkTreeIter iter;
			GtkTreeModel *gtm;
			GString *dl;
			char *filename;
			gulong file_size;
			GLOB_USER *gu;
	
			gtk_tree_model_get_iter (gtm=gtk_tree_view_get_model(GTK_TREE_VIEW(widget)),&iter,path);
			gtk_tree_model_get(gtm,&iter,FRC_USER_POINTER,&gu,FRC_FULL_PATH,&filename,FRC_SIZE_AS_VAL,&file_size,-1);
	
			dl=g_string_new("/DL |");						  /* adapted to new /DL */
			g_string_sprintfa(dl,"%s||%s|%lu\n",gu->unfmt_nick,filename,file_size);			 /* let DCTC generate the filename itself */
			send_data_to_gdl_dctc(dl->str);
			g_string_free(dl,TRUE);
			free(filename);
		}
		return TRUE;
	}
	else if(event->button==3)
	{  /* right-click */
		build_start_dl_popup(FIND_TAB);
		gtk_menu_popup(GTK_MENU(start_dl_popup),NULL,NULL,NULL,NULL,event->button,event->time);
		return TRUE;
	}


	return FALSE;
}

/************************************************************************/
/* this function add generate a "cmd user" from the given find_cl_entry */
/************************************************************************/
/* cmd is a GString "" and it must be restored to this at the end */
/******************************************************************/
static void do_dl_from_find_result(GtkTreeModel *gtm, GtkTreePath *path, GtkTreeIter *iter, gpointer cmd)
{  
	int ln;
	GString *dl=cmd;
	char *filename;
	gulong file_size;
	GLOB_USER *gu;
													 
	ln=dl->len;

	gtk_tree_model_get(gtm,iter,FRC_USER_POINTER,&gu,FRC_FULL_PATH,&filename,FRC_SIZE_AS_VAL,&file_size,-1);
	g_string_sprintf(dl,"/DL |%s|%s|%s|%lu\n",gu->unfmt_nick,"",filename,file_size);
	send_data_to_gdl_dctc(dl->str);
	
	g_string_truncate(dl,ln);	  /* restore original string lenght */
	free(filename);
}

static void start_download_from_find_result_clist(void)
{  
	GString *dl;
	dl=g_string_new("");

	generic_selected_find_result_calls(do_dl_from_find_result,dl);
	g_string_free(dl,TRUE);
}	  

gboolean
on_find_result_key_press_event			(GtkWidget		 *widget,
													 GdkEventKey	  *event,
													 gpointer			user_data)
{
	if(event==NULL)
		return FALSE;
					
	switch(event->keyval)
	{
		case GDK_d: /* 'd' pressed */
		case GDK_D:
						start_download_from_find_result_clist();
						break;
	}
	
	return FALSE;
}


/* ------------------------------------------------------------------------------------------------------------- */

#define MAX_FIND_HISTORY_SIZE 10

/************************************************************************/
/* this function add a new search pattern to the find_history combo box */
/* the number of entry in the history is limited to 10 pattern. If the  */
/* given pattern already exists, it is moved to the first place in the  */
/* history.                                                             */
/************************************************************************/
static void add_pattern_to_find_history(const char *pattern)
{
	GtkWidget *w;
	static GList *current_history=NULL;
	gchar *tmp_str;
	int i;

	w=get_widget_by_widget_name(main_window,"find_history_combo");
	if(w==NULL)
		return;

	if(utf8_mode==TRUE)
		tmp_str=g_strdup(pattern);
	else
		tmp_str=g_locale_to_utf8(pattern,-1,NULL,NULL,NULL);

	if(tmp_str==NULL)
	{
		fprintf(stderr,"Out of memory: history not updated.\n");
		return;
	}

	current_history=g_list_prepend(current_history,tmp_str);

	/* remove duplicate entry (if exists) */
	i=1;
	while(i<g_list_length(current_history))
	{
		gpointer str_to_test;

		str_to_test=g_list_nth_data(current_history,i);
		if(!strcasecmp(tmp_str,str_to_test))
		{
			g_free(str_to_test);
			current_history=g_list_remove(current_history,str_to_test);
		}
		else
			i++;
	}

	/* keep only the first Nth entry of the history */
	while(g_list_length(current_history)>MAX_FIND_HISTORY_SIZE)
	{
		gpointer str_to_delete;

		str_to_delete=g_list_nth_data(current_history,MAX_FIND_HISTORY_SIZE);
		current_history=g_list_remove(current_history,str_to_delete);
		g_free(str_to_delete);
	}

	gtk_combo_set_popdown_strings (GTK_COMBO (w), current_history);
}


/***************************************/
/* handle activation of the find entry */
/***************************************/
/* GTK2: in progress */
/*********************/
void
on_find_entry_activate                 (GtkEntry        *entry,
                                        gpointer         user_data)
{
	gchar *param;
	unsigned int ftype1,ftype2;
	GString *req1,*req2;
	GString *req;				/* common part of the request */
	GtkWidget *w;

	param=get_gtk_entry(entry);
	if(param==NULL)
		return;

	if(strlen(param)<1)
	{
		g_free(param);
		return;
	}

	/* convert filetype_entry into a number */
	ftype1=gtk_entry_to_number("filetype_entry",ftype_str);
	req1=g_string_new("/SRCH ");
	g_string_sprintfa(req1,"|%s|%u",param,ftype1+1);

	/* same thing for fileentry_entry1 */
	ftype2=gtk_entry_to_number_wo_default("filetype_entry1",ftype_str);
	req2=g_string_new("/SRCH ");
	g_string_sprintfa(req2,"|%s|%u",param,ftype2+1);

	/* build the common part of the request */
	req=g_string_new("");

	w=get_widget_by_widget_name(main_window,"size_checkbutton");
	if(w==NULL)
	{
		abrt:
		g_free(param);
		g_string_free(req,TRUE);
		return;
	}
	
	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w))==TRUE)
	{
		static const char *sizetype_str[3];
		static const char *scale_str[5];
		static int unset=1;
		double val;
		double scale;

		if(unset)
		{
			sizetype_str[0]=_("at least");
			sizetype_str[1]=_("at most");
			sizetype_str[2]=NULL;
			scale_str[0]=_("Bytes");
			scale_str[1]=_("KBytes");
			scale_str[2]=_("MBytes");
			scale_str[3]=_("GBytes");
			scale_str[4]=NULL;
			unset=0;
		}

		/* get the size type */
		switch(gtk_entry_to_number("sizetype_entry",sizetype_str))
		{
			case 0:
						req=g_string_append(req,"|L|");
						break;

			case 1:
						req=g_string_append(req,"|M|");
						break;
		}

		switch(gtk_entry_to_number("size_unit_entry",scale_str))
		{
			default:
			case 0:	scale=1.0;
						break;
			case 1:	scale=1024.0;
						break;
			case 2:	scale=1024.0*1024.0;
						break;
			case 3:	scale=1024.0*1024.0*1024.0;
						break;
		}

		w=get_widget_by_widget_name(main_window,"size_entry");
		if(w==NULL)
			goto abrt;

		val=atof(gtk_entry_get_text(GTK_ENTRY(w)));

		g_string_sprintfa(req,"%f",scale*val);
	}

	w=get_widget_by_widget_name(main_window,"find_result");
	if(w!=NULL)
	{
		gtk_list_store_clear(GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(w))));
	}

	req=g_string_append_c(req,'\n');

	/* send the first string */
	req1=g_string_append(req1,req->str);
	send_data_to_dctc(req1->str);

	if(last_search[0]!=NULL)
	{
		g_string_free(last_search[0],TRUE);
	}
	last_search[0]=req1;

	/* if the second search criteria exists, start a search */
	if((ftype2!=-1)&&(ftype2!=ftype1))
	{
		req2=g_string_append(req2,req->str);
		send_data_to_dctc(req2->str);

		if(last_search[1]!=NULL)
		{
			g_string_free(last_search[1],TRUE);
		}
		last_search[1]=req2;
	}
	else
	{
		if(last_search[1]!=NULL)
		{
			g_string_free(last_search[1],TRUE);
			last_search[1]=NULL;
		}
		g_string_free(req2,TRUE);
	}

	g_string_free(req,TRUE);

	/* keep a pre-sliced search pattern */
	if(last_started_search!=NULL)
	{
		g_strfreev(last_started_search);
		last_started_search=NULL;
	}

	last_started_search=g_strsplit(param,"$",0);

	/* we must add the new search pattern to the history at the end of the function */
	/* because when the popdown list is modified, it seems to alter the entry (not its value but its memory position) */
	add_pattern_to_find_history(param);
	g_free(param);
}

/************************************************/
/* simulate activation of the find_entry widget */
/************************************************/
/* GTK2: ok */
/************/
void
on_find_button_clicked                 (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *w;

	w=get_widget_by_widget_name(main_window,"find_entry");
	if(w!=NULL)
	{
		on_find_entry_activate(GTK_ENTRY(w),NULL);
	}
}
/* ------------------------------------------------------------------------------------------------------------- */


void
on_show_search_user_side_button_clicked
													 (GtkButton		 *button,
													 gpointer			user_data)
{
	gtk_widget_show(get_widget_by_widget_name(main_window,"user_search_vbox"));
	gtk_widget_show(get_widget_by_widget_name(main_window,"hide_search_user_button"));
	gtk_widget_hide(get_widget_by_widget_name(main_window,"show_search_user_side_button"));
}


void
on_hide_search_user_button_clicked     (GtkButton       *button,
                                        gpointer         user_data)
{
	gtk_widget_hide(get_widget_by_widget_name(main_window,"user_search_vbox"));
	gtk_widget_hide(get_widget_by_widget_name(main_window,"hide_search_user_button"));
	gtk_widget_show(get_widget_by_widget_name(main_window,"show_search_user_side_button"));
}


gboolean
on_locate_user_clist_button_press_event
                                        (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{

  return FALSE;
}

/* ------------------------------------------------------------------------------------------------------------- */
/**********************************************/
/* activate entry to start a search of a user */
/**********************************************/
/* GTK2: to test */
/*****************/
void
on_locate_user_entry_activate          (GtkEntry        *entry,
                                        gpointer         user_data)
{
	gchar *param;

	param=get_gtk_entry(entry);
	if(param==NULL)
		return;

	if(strlen(param)>0)
	{
		GString *req;

		req=g_string_new("/LOCATEUSER ");
		g_string_sprintfa(req,"%s\n",param);
		send_data_to_dctc(req->str);
		g_string_free(req,TRUE);
	}
	g_free(param);
}


/************************************************/
/* simulate activation of the locate_user_entry */
/************************************************/
/* GTK2: to test */
/*****************/
void
on_search_user_button_clicked          (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *w;

	w=get_widget_by_widget_name(main_window,"locate_user_entry");
	if(w!=NULL)
	{
		on_locate_user_entry_activate(GTK_ENTRY(w),NULL);
	}
}


/* ------------------------------------------------------------------------------------------------------------- */
/****************************/
/* Add an autoscan to a GDL */
/****************************/
/* GTK2: to test */
/*****************/
void
on_add_as_to_gdl_button_clicked        (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *w1;
	const char *gid;
	gchar *pattern;

	w1=get_widget_by_widget_name(main_window,"as_gid_label");

	gid=gtk_label_get_text(GTK_LABEL(w1));
	if(strlen(gid)==0)
		return;

	pattern=get_gtk_entry_by_name(main_window,"gdl_as_pattern_entry");
	if(strlen(pattern)!=0)
	{
		GString *str;
		unsigned int ftype;

		ftype=gtk_entry_to_number("gdl_as_filetype_entry",ftype_str);

		str=g_string_new("");
		g_string_sprintf(str,"/GDLAS+ %s|%u|%s\n",gid,ftype+1,pattern);

		send_data_to_gdl_dctc(str->str);
		g_string_free(str,TRUE);
	}
	g_free(pattern);
}

/* ------------------------------------------------------------------------------------------------------------- */
/******************************/
/* rename the GDL file at end */
/******************************/
/* GTK2: to test */
/*****************/
void
on_do_gdl_rename_button_clicked		  (GtkButton		 *button,
													 gpointer			user_data)
{
	GtkWidget *w1;
	const char *gid;
	char *fullname;

	w1=get_widget_by_widget_name(main_window,"gdl_rename_gid_label");
	if(w1==NULL)
		return;

	gid=gtk_label_get_text(GTK_LABEL(w1));
	if(strlen(gid)==0)
		return;

	fullname=get_gtk_entry_by_name(main_window,"gdl_rename_filename_entry");
	if(strlen(fullname)!=0)
	{
		GString *str;
		char *t;

		t=strrchr(fullname,'/');
		if(t!=NULL)
		{
			*t++='\0';

			str=g_string_new("");
			g_string_sprintf(str,"/GDLRENAME %s|%s|%s\n",gid,t,fullname);
			send_data_to_gdl_dctc(str->str);
			g_string_free(str,TRUE);
		}
	}
	g_free(fullname);

}


/*************************************/
/* remove rename the GDL file at end */
/*************************************/
/* GTK2: to test */
/*****************/
void
on_remove_current_gdl_rename_button_clicked
													 (GtkButton		 *button,
													 gpointer			user_data)
{
	GtkWidget *w1;
	const char *gid;
	GString *str;

	w1=get_widget_by_widget_name(main_window,"gdl_rename_gid_label");

	if(w1==NULL)
		return;

	gid=gtk_label_get_text(GTK_LABEL(w1));
	if(strlen(gid)==0)
		return;

	str=g_string_new("");
	g_string_sprintf(str,"/GDLNORENAME %s\n",gid);
	send_data_to_gdl_dctc(str->str);
	g_string_free(str,TRUE);

}
/* ------------------------------------------------------------------------------------------------------------- */
/***************************************/
/* set a script to start at end of GDL */
/***************************************/
/* GTK2: to test */
/*****************/
void
on_do_gdl_script_button_clicked		  (GtkButton		 *button,
													 gpointer			user_data)
{
	GtkWidget *w1;
	const char *gid;
	char *fullname;

	w1=get_widget_by_widget_name(main_window,"gdl_script_gid_label");

	if(w1==NULL)
		return;

	gid=gtk_label_get_text(GTK_LABEL(w1));
	if(strlen(gid)==0)
		return;

	fullname=get_gtk_entry_by_name(main_window,"gdl_endname_filename_entry");
	if(strlen(fullname)!=0)
	{
		GString *str;

		str=g_string_new("");
		g_string_sprintf(str,"/GDLSCRIPT %s|%s\n",gid,fullname);
		send_data_to_gdl_dctc(str->str);
		g_string_free(str,TRUE);
	}
	g_free(fullname);
}

/*********************************************/
/* discard the script to start at end of GDL */
/*********************************************/
/* GTK2: to test */
/*****************/
void
on_remove_current_gdl_script_button_clicked
													 (GtkButton		 *button,
													 gpointer			user_data)
{
	GtkWidget *w1;
	const char *gid;
	GString *str;

	w1=get_widget_by_widget_name(main_window,"gdl_script_gid_label");

	if(w1==NULL)
		return;

	gid=gtk_label_get_text(GTK_LABEL(w1));
	if(strlen(gid)==0)
		return;

	str=g_string_new("");
	g_string_sprintf(str,"/GDLNOSCRIPT %s\n",gid);
	send_data_to_gdl_dctc(str->str);
	g_string_free(str,TRUE);
}

/* ------------------------------------------------------------------------------------------------------------- */

static unsigned long int base_gdl_id=0;

/****************************************************************************************/
/* this function is called when the user wants to create one GDL for all selected files */
/****************************************************************************************/
/* this function is not used anymore, it remains here for historic reason */
/**************************************************************************/
static void create_one_gdl_for_all_files(char *clist_name, int nick_col, int fname_col, int fsize_col)
{
}


/****************************************************************************************/
/* this function is called when the user wants to create one GDL for all selected files */
/****************************************************************************************/
/* this function is not used anymore, it remains here for historic reason */
/**************************************************************************/
static void create_one_gdl_for_all_files_with_autoscan(char *clist_name, int nick_col, int fname_col, int fsize_col)
{
}

/*****************************************************************************************/
/* this function is called when the user wants to create one GDL for each selected files */
/*****************************************************************************************/
/* this function is not used anymore, it remains here for historic reason */
/**************************************************************************/
static void create_one_gdl_per_files(char *clist_name, int nick_col, int fname_col, int fsize_col)
{
}

/************************************************************************/
/* this function add generate a "cmd user" from the given find_cl_entry */
/************************************************************************/
/* cmd is a GString "/GDLADD gdl_id|" and it must be restored to this */
/* at the end                                                         */
/**********************************************************************/
static void do_add_gdl_source_from_find_result(GtkTreeModel *gtm, GtkTreePath *path, GtkTreeIter *iter, gpointer cmd)
{
	int ln;
	GString *dl=cmd;
	char *filename;
	gulong file_size;
	GLOB_USER *gu;

	ln=dl->len;

	gtk_tree_model_get(gtm,iter,FRC_USER_POINTER,&gu,FRC_FULL_PATH,&filename,FRC_SIZE_AS_VAL,&file_size,-1);
	
	g_string_sprintfa(dl,"%s|%s|%lu\n",gu->unfmt_nick,filename,file_size);
	send_data_to_gdl_dctc(dl->str);

	g_string_truncate(dl,ln);		/* restore original string lenght */
	free(filename);
}

static void add_source_to_gdl_from_find_result_clist(unsigned long int gdl_id)
{
	GString *dl;
	dl=g_string_new("");
	g_string_sprintf(dl,"/GDLADD %lu|",gdl_id);	/* it is the common part of the string */

	generic_selected_find_result_calls(do_add_gdl_source_from_find_result,dl);
	g_string_free(dl,TRUE);
}

/* ------------------------------------------------------------------------------------------------------------- */
/*****************************************************************************/
/* this function add generate a "cmd user" from the given user_file_cl_entry */
/*****************************************************************************/
/* cmd is a GString "/GDLADD gdl_id|" and it must be restored to this */
/* at the end                                                         */
/**********************************************************************/
static void do_add_gdl_source_from_user_file_list(GtkTreeModel *gtm, GtkTreePath *path, GtkTreeIter *iter, gpointer cmd)
{
	int ln;
	GString *dl=cmd;
	GtkTreeIter nick_iter;
	guint type;
	char *nickname;
	gulong fsize;
	char *fname;

	gtk_tree_model_get(gtm,iter,UFLC_TYPE,&type,-1);
	if(type!=UFL_FILE_ENTRY)
		return;

	if(get_nick_entry_for_user_file_list_from_ufce(gtm,iter,&nick_iter)==FALSE)
	{
		fprintf(stderr,"do_add_gdl_source_from_user_file_list: no USER_FILE_CL_ENTRY found\n");
		return;
	}

	ln=dl->len;

	gtk_tree_model_get(gtm,&nick_iter,UFLC_STR,&nickname,-1);
	gtk_tree_model_get(gtm,iter,UFLC_STR,&fname,UFLC_SIZE_AS_VAL,&fsize,-1);
	g_string_sprintfa(dl,"%s|%s|%lu\n",nickname,fname,fsize);
	send_data_to_gdl_dctc(dl->str);

	g_string_truncate(dl,ln);		/* restore original string lenght */
	free(nickname);
	free(fname);
}

static void add_source_to_gdl_from_user_file_list_clist(unsigned long int gdl_id)
{
	GString *dl;
	dl=g_string_new("");
	g_string_sprintf(dl,"/GDLADD %lu|",gdl_id);	/* it is the common part of the string */

	generic_selected_user_file_list_calls(do_add_gdl_source_from_user_file_list,dl);
	g_string_free(dl,TRUE);
}

/******************************************************************************************/
/* this function is called when the user clicks on a GDL entry of the start_dl_popup_menu */
/******************************************************************************************/
/* user_data is the GDL ID */
/***************************/
static void add_selected_entries_to_this_gdl(GtkMenuItem *menuitem, gpointer user_data)
{
	unsigned long gdl_id=(unsigned long)user_data;
	GtkWidget *w;

	w=get_widget_by_widget_name(main_window,"main_notebook");
	if(w==NULL)
		return;

	switch(gtk_notebook_get_current_page(GTK_NOTEBOOK(w)))
	{
		case FIND_TAB:		/* find */
					add_source_to_gdl_from_find_result_clist(gdl_id);
					break;

		case USER_FILE_LIST_TAB:	/* user file list */
					add_source_to_gdl_from_user_file_list_clist(gdl_id);
					break;

		default:
					break;
	}
}

/* ------------------------------------------------------------------------------------------------------------- */

static void
on_expand_all_ctree             (GtkMenuItem     *menuitem,
                                 gpointer         user_data)
{
   GtkWidget *w;

   w=get_widget_by_widget_name(main_window,(char*)user_data);
   if(w!=NULL)
   {
		gtk_tree_view_expand_all(GTK_TREE_VIEW(w));
   }
}
      
/* ------------------------------------------------------------------------------------------------------------- */
static void
on_shrink_all_ctree             (GtkMenuItem     *menuitem,
                                 gpointer         user_data)
{
   GtkWidget *w;

   w=get_widget_by_widget_name(main_window,(char*)user_data);
   if(w!=NULL)
   {
		gtk_tree_view_collapse_all(GTK_TREE_VIEW(w));
   }
}
 
/* ------------------------------------------------------------------------------------------------------------- */
typedef struct
{
	int with_file_size;			/* if ==0, file_size is not set */
	unsigned long file_size;	/* else if ==1 file_size contains the searched size.  */
										/* else >1 there is more than one searched size */
	GtkWidget *bad_size;			/* this widget is the menu containing GDL having a size different of file_size */
	int nb_bad_size;				/* number of entry in bad_size menu */
} DL_POPUP_XTRA_PARAM;

/*********************************************************/
/* add the given cnode to the end of start_dl_popup_menu */
/*********************************************************/
/* GTK2: to test */
/*****************/
static void add_gdl_entry_to_popup(GtkTreeModel *gtm,GtkTreeStore *gts, GtkTreeIter *iter,DL_POPUP_XTRA_PARAM *dpxp)
{
	guint ct_type;

	gtk_tree_model_get(gtm,iter,GCC_TYPE,&ct_type,-1);
	if(ct_type==GDL_ROOT)
	{
		GtkWidget *mitem;
		GString *str;
		gulong gdl_id,gdl_size;
		char *local_filename;

		gtk_tree_model_get(gtm,iter,GCC_ID,&gdl_id,GCC_FULL_PATH_COL,&local_filename,GCC_VAL1,&gdl_size,-1);
		/* we use FULL_PATH_COL instead of STR1 because we must have an UTF8 string */
		str=g_string_new("");
#ifndef NO_PRINTF_LOCALE
		g_string_sprintf(str,"%s (%'lu)",local_filename,gdl_size);		/* NO_PRINTF_LOCAL support added */
#else
		g_string_sprintf(str,"%s (%lu)",local_filename,gdl_size);
#endif

		free(local_filename);

		mitem=gtk_menu_item_new_with_label(str->str);
		gtk_signal_connect(GTK_OBJECT(mitem),"activate", GTK_SIGNAL_FUNC(add_selected_entries_to_this_gdl),(void*)(gdl_id));

		if(dpxp->with_file_size==0)
		{
			/* no size => no submenu => everything in the main menu */
			gtk_menu_append(GTK_MENU(start_dl_popup),mitem);
		}
		else
		{
			if(dpxp->file_size==gdl_size)
			{	/* a matching size => in the main menu */
				gtk_menu_append(GTK_MENU(start_dl_popup),mitem);
			}
			else
			{	/* a not matching size => in the submenu */
				gtk_menu_append(GTK_MENU(dpxp->bad_size),mitem);
				dpxp->nb_bad_size++;
			}
		}
		gtk_widget_show(mitem);
		g_string_free(str,TRUE);
	}
}


/* ------------------------------------------------------------------------------------------------------------- */
/*****************************************************************/
/* search the start of the last directory of the given directory */
/*****************************************************************/
/* dir can be "", if not, dir always ends with a '/' */
/* output: pointer on the last directory name        */
/*****************************************************/
static char *find_start_of_last_level(char *dir)
{
	int ln;

	ln=strlen(dir);
	if(ln==0)
		return dir;

	ln-=2;		/* skip the last character which is always a '/' */
	while(ln>=0)
	{
		if(dir[ln]=='\\')
			return dir+ln+1;
		ln--;
	}
	return dir;
}

static void create_one_gdl_for_offspring(GtkTreeModel *model,GtkTreeIter *iter,GtkTreeIter *start_node)
{
	GtkTreeIter nickiter;
	char *nickname;
	char *s_dirname;		/* full file path */
	char *e_filename;		/* the path of the start directory */
	gulong file_size;
	unsigned long int new_gid=rand();
	char *t;
	int to_ignore;

	if(get_nick_entry_for_user_file_list_from_ufce(model,iter,&nickiter)==FALSE)	/* searching from start_node provides the same result as from iter */
		return;																								/* because iter is a descendant of start_node but nickiter can be NULL, iter not */

	gtk_tree_model_get(model,&nickiter,UFLC_STR,&nickname,-1);
	gtk_tree_model_get(model,iter,UFLC_STR,&e_filename,UFLC_SIZE_AS_VAL,&file_size,-1);
	if(start_node!=NULL)
	{
		gtk_tree_model_get(model,start_node,UFLC_STR,&s_dirname,-1);
	}
	else
	{
		s_dirname=strdup(e_filename);
	}

	/* we keep the last level of the start directory */
	t=find_start_of_last_level(s_dirname);
	if(t==NULL)
		to_ignore=0;
	else
		to_ignore=t-(s_dirname);		/* part of the path to discard at the beginning of the file path */

	{
		GString *end_base_name;
		GString *end_dir_name;
		GString *cmd;

		end_dir_name=g_string_new(e_filename+to_ignore);
		t=strrchr(end_dir_name->str,'\\');
		if(t==NULL)
		{
			end_base_name=g_string_new(end_dir_name->str);
			end_dir_name=g_string_truncate(end_dir_name,0);
		}
		else
		{
			int i;
			end_base_name=g_string_new(t+1);
			end_dir_name=g_string_truncate(end_dir_name,t-end_dir_name->str);
			for(i=0;i<end_dir_name->len;i++)
			{
				if(end_dir_name->str[i]=='\\')
					end_dir_name->str[i]='/';
			}
		}

		cmd=g_string_new("");
		g_string_sprintf(cmd,"/GDLNEW %lu|%lu|%s\n",new_gid,file_size,end_base_name->str);
		send_data_to_gdl_dctc(cmd->str);

		if(end_dir_name->len!=0)
		{
			g_string_sprintf(cmd,"/GDLRENAME %lu|%s|%s\n",new_gid,end_base_name->str,end_dir_name->str);
			send_data_to_gdl_dctc(cmd->str);
		}

		g_string_sprintf(cmd,"/GDLADD %lu|%s|%s|%lu\n",new_gid,nickname,e_filename,file_size);
		send_data_to_gdl_dctc(cmd->str);

		g_string_free(end_dir_name,TRUE);
		g_string_free(end_base_name,TRUE);
		g_string_free(cmd,TRUE);
	}

	free(nickname);
	free(s_dirname);
	free(e_filename);
}

/*************************************************************************/
/* walk in the tree from a selected node and for each leaf, create a GDL */
/***************************************************************************************/
/* NOTE: path is not valid if data is !=NULL (the recursive function does not build it */
/***************************************************************************************/
static void create_offspring_gdl(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	guint ent_type;

	gtk_tree_model_get(model,iter,UFLC_TYPE,&ent_type,-1);
	switch((USER_FILE_ENTRY_TYPE)ent_type)
	{
		case UFL_NICK_ROOT_ENTRY:
									{	/* go into the deeper level but ignore the nickname */
										gboolean valid;
										GtkTreeIter child;

										valid=gtk_tree_model_iter_children(model,&child,iter);
										while(valid)
										{
											create_offspring_gdl(model,NULL,&child,NULL);
											valid=gtk_tree_model_iter_next(model,&child);
										}
									}
									break;

		case UFL_DIR_ENTRY:
									{
										gboolean valid;
										GtkTreeIter child;

										valid=gtk_tree_model_iter_children(model,&child,iter);
										while(valid)
										{
											create_offspring_gdl(model,NULL,&child,(data!=NULL)?data:iter);
											valid=gtk_tree_model_iter_next(model,&child);
										}
									}
									break;

		case UFL_FILE_ENTRY:
									create_one_gdl_for_offspring(model,iter,(GtkTreeIter*)data);
									break;
	}
}

/*****************************************************************************************************************/
/* this function is called when the user wants to create one GDL for each selected files of a recursive download */
/*****************************************************************************************************************/
/* GTK2: ok */
/************/
static void add_recursive_to_gdl_with_create(GtkMenuItem *menuitem, gpointer user_data)
{
	GtkWidget *w;

	w=get_widget_by_widget_name(main_window,"main_notebook");
	if(w==NULL)
		return;

	switch(gtk_notebook_get_current_page(GTK_NOTEBOOK(w)))
	{
		case USER_FILE_LIST_TAB:	/* user file list */
					/* 1) for each selected entry, check if it does not have a selected parent */
					/*    algo: for each selected entry, we unselect all offsprings and also all leafs */
					keep_selected_lowest_user_file_list_nodes();
					/* 2) for each remaining selected entry (which is always a directory), we scan its leafs and build GDLs */
					generic_selected_user_file_list_calls(create_offspring_gdl,NULL);
					break;

		default:
					break;
	}
}

/* ------------------------------------------------------------------------------------------------------------- */

typedef struct 
{
	GString *dl;
	unsigned long gdl_id;
	unsigned long biggest_size;
	char *biggest_filename;		/* this value must be freed at end */
} CREATE_GDL_FOR_FILES_PARAM;

static void do_find_biggest_selected_from_find_result(GtkTreeModel *gtm, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	CREATE_GDL_FOR_FILES_PARAM *cgf=data;
	gulong file_size;
	char *filename;

	gtk_tree_model_get(gtm,iter,FRC_FULL_PATH,&filename,FRC_SIZE_AS_VAL,&file_size,-1);

	if(file_size>cgf->biggest_size)
	{
		if(cgf->biggest_filename)
			free(cgf->biggest_filename);
		cgf->biggest_filename=filename;		/* the caller will free the string */
		cgf->biggest_size=file_size;
	}
	else
		free(filename);
}

static void do_find_biggest_selected_from_user_file_list(GtkTreeModel *gtm, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	CREATE_GDL_FOR_FILES_PARAM *cgf=data;
	guint type;
	gulong fsize;
	char *fname;

	gtk_tree_model_get(gtm,iter,UFLC_TYPE,&type,-1);
	if(type!=UFL_FILE_ENTRY)
		return;

	gtk_tree_model_get(gtm,iter,UFLC_STR,&fname,UFLC_SIZE_AS_VAL,&fsize,-1);

	if(fsize>cgf->biggest_size)
	{
		if(cgf->biggest_filename)
			free(cgf->biggest_filename);
		cgf->biggest_filename=fname;
		cgf->biggest_size=fsize;
	}
	else
		free(fname);
}

/****************************************************************************************/
/* this function is called when the user wants to create one GDL for all selected files */
/****************************************************************************************/
/* autoscan_pattern can be either NULL or a string with the format "filetype|pattern" */
/**************************************************************************************/
static void do_create_one_gdl_for_all_files_from_find_result(GtkTreeModel *gtm, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	CREATE_GDL_FOR_FILES_PARAM *cgf=data;
	gulong file_size;
	char *filename;
	GLOB_USER *gu;

	gtk_tree_model_get(gtm,iter,FRC_USER_POINTER,&gu,FRC_FULL_PATH,&filename,FRC_SIZE_AS_VAL,&file_size,-1);

	g_string_sprintf(cgf->dl,"/GDLADD %lu|%s|%s|%lu\n",cgf->gdl_id,gu->unfmt_nick,filename,file_size);
	send_data_to_gdl_dctc(cgf->dl->str);
	free(filename);
}

/****************************************************************************************/
/* this function is called when the user wants to create one GDL for all selected files */
/****************************************************************************************/
static void do_create_one_gdl_for_all_files_from_user_file_list(GtkTreeModel *gtm, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	CREATE_GDL_FOR_FILES_PARAM *cgf=data;
	GtkTreeIter nickiter;
	char *nickname;
	char *filename;
	gulong file_size;
	guint type;

	gtk_tree_model_get(gtm,iter,UFLC_TYPE,&type,-1);
	if(type!=UFL_FILE_ENTRY)
		return;

	if(get_nick_entry_for_user_file_list_from_ufce(gtm,iter,&nickiter)==FALSE)
		return;

	gtk_tree_model_get(gtm,iter,UFLC_STR,&filename,UFLC_SIZE_AS_VAL,&file_size,-1);
	gtk_tree_model_get(gtm,&nickiter,UFLC_STR,&nickname,-1);

	g_string_sprintf(cgf->dl,"/GDLADD %lu|%s|%s|%lu\n",cgf->gdl_id,nickname,filename,file_size);
	send_data_to_gdl_dctc(cgf->dl->str);

	free(nickname);
	free(filename);
}

/***************************************************************************************/
/* create a new GDL from the given gdl_id, the remote filename and the remote filesize */
/***************************************************************************************/
static void	do_gdl_new_from_info(unsigned long int gdl_id, const char *remote_fname, unsigned long remote_fsize)
{
	GString *str;
	const char *lname;

	lname=strrchr(remote_fname,'\\');
	if(lname==NULL)
		lname=remote_fname;
	else
		lname++;

	str=g_string_new("");
	g_string_sprintf(str,"/GDLNEW %lu|%lu|%s\n",gdl_id,remote_fsize,lname);
	send_data_to_gdl_dctc(str->str);
	g_string_free(str,TRUE);
}

/********************************************************************************************************/
/* this function is called when the user wants to create one GDL for each selected files (find version) */
/********************************************************************************************************/
static void do_create_one_gdl_per_files_from_find_result(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	GString *dl=data;
	unsigned long int increment=123456789;	/* this value is added to gdl_id each time */
	char *filename;
	gulong file_size;
	GLOB_USER *gu;

	if(base_gdl_id==0)
		base_gdl_id=time(NULL);

	gtk_tree_model_get(model,iter,FRC_USER_POINTER,&gu,FRC_FULL_PATH,&filename,FRC_SIZE_AS_VAL,&file_size,-1);

	do_gdl_new_from_info(base_gdl_id,filename,file_size);
	g_string_sprintf(dl,"/GDLADD %lu|%s|%s|%lu\n",base_gdl_id,gu->unfmt_nick,filename,file_size);
	send_data_to_gdl_dctc(dl->str);
	base_gdl_id+=increment;		/* to limit the risk of having 2 GDL with the same number, should be safe enough */
	g_string_truncate(dl,0);

	free(filename);
}

/******************************************************************************************************************/
/* this function is called when the user wants to create one GDL for each selected files (user file list version) */
/******************************************************************************************************************/
static void do_create_one_gdl_per_files_from_user_file_list(GtkTreeModel *gtm, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	GString *dl=data;
	unsigned long int increment=123456789;	/* this value is added to gdl_id each time */
	guint type;
	GtkTreeIter nickiter;
	char *nickname;
	char *filename;
	gulong file_size;

	gtk_tree_model_get(gtm,iter,UFLC_TYPE,&type,-1);
	if(type!=UFL_FILE_ENTRY)
		return;

	if(get_nick_entry_for_user_file_list_from_ufce(gtm,iter,&nickiter)==FALSE)
		return;

	gtk_tree_model_get(gtm,iter,UFLC_STR,&filename,UFLC_SIZE_AS_VAL,&file_size,-1);
	gtk_tree_model_get(gtm,&nickiter,UFLC_STR,&nickname,-1);

	if(base_gdl_id==0)
		base_gdl_id=time(NULL);

	do_gdl_new_from_info(base_gdl_id,filename,file_size);
	g_string_sprintf(dl,"/GDLADD %lu|%s|%s|%lu\n",base_gdl_id,nickname,filename,file_size);
	send_data_to_gdl_dctc(dl->str);
	base_gdl_id+=increment;		/* to limit the risk of having 2 GDL with the same number, should be safe enough */
	g_string_truncate(dl,0);

	free(nickname);
	free(filename);
}

/*********************************************************************************************/
/* this function is called when the user wants to create one GDL for each/all selected files */
/*********************************************************************************************/
static void add_to_gdl_with_create(GtkMenuItem *menuitem, gpointer user_data)
{
	GtkWidget *w;
	void (*fnc)(char *clist_name, int nick_col,int fname_col,int fsize_col)=user_data;

	w=get_widget_by_widget_name(main_window,"main_notebook");
	if(w==NULL)
		return;

	switch(gtk_notebook_get_current_page(GTK_NOTEBOOK(w)))
	{
		case FIND_TAB:		/* find */
					if(fnc==create_one_gdl_per_files)
					{
						GString *dl;
						dl=g_string_new("");

						generic_selected_find_result_calls(do_create_one_gdl_per_files_from_find_result,dl);
						g_string_free(dl,TRUE);
					}
					else
					{
						CREATE_GDL_FOR_FILES_PARAM cgf;
						GString *autoscan=NULL;

						cgf.dl=g_string_new("");
						cgf.gdl_id=0;

						if(fnc==create_one_gdl_for_all_files_with_autoscan)
						{
							GtkWidget *w3;

							w3=get_widget_by_widget_name(main_window,"find_history_combo");
							if(w3!=NULL)
							{
								const char *pattern;

								pattern=gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(w3)->entry));
								if(strlen(pattern)!=0)
								{
									unsigned int ftype;
									ftype=gtk_entry_to_number("filetype_entry",ftype_str);
							
									autoscan=g_string_new("");
									g_string_sprintf(autoscan,"%u|%s",ftype+1,pattern);
								}
							}
						}
						cgf.biggest_size=0;
						cgf.biggest_filename=NULL;

						/* search the entry having the biggest file size */
						generic_selected_find_result_calls(do_find_biggest_selected_from_find_result,&cgf);
						if(cgf.biggest_filename!=NULL)
						{
							if(base_gdl_id==0)
								base_gdl_id=time(NULL);

							/* create a new GDL, using information of the biggest file */
							do_gdl_new_from_info(base_gdl_id, cgf.biggest_filename, cgf.biggest_size);
							cgf.gdl_id=base_gdl_id;
							base_gdl_id+=12345678;

							generic_selected_find_result_calls(do_create_one_gdl_for_all_files_from_find_result,&cgf);

							/* add the autoscan pattern if exist */
							if(autoscan!=NULL)
							{
								g_string_sprintf(cgf.dl,"/GDLAS+ %lu|%s\n",cgf.gdl_id,autoscan->str);
								send_data_to_gdl_dctc(cgf.dl->str);
							}
							free(cgf.biggest_filename);
							cgf.biggest_filename=NULL;
						}

						g_string_free(cgf.dl,TRUE);
						if(autoscan)
							g_string_free(autoscan,TRUE);
					}
					break;

		case USER_FILE_LIST_TAB:	/* user file list */
					if(fnc==create_one_gdl_per_files)
					{
						GString *dl;
						dl=g_string_new("");

						generic_selected_user_file_list_calls(do_create_one_gdl_per_files_from_user_file_list,dl);
						g_string_free(dl,TRUE);
					}
					else
					{
						CREATE_GDL_FOR_FILES_PARAM cgf;

						cgf.dl=g_string_new("");
						cgf.gdl_id=0;
						cgf.biggest_size=0;
						cgf.biggest_filename=NULL;

						/* search the entry having the biggest file size */
						generic_selected_user_file_list_calls(do_find_biggest_selected_from_user_file_list,&cgf);
						if(cgf.biggest_filename!=NULL)
						{
							if(base_gdl_id==0)
								base_gdl_id=time(NULL);

							/* create a new GDL, using information of the biggest file */
							do_gdl_new_from_info(base_gdl_id, cgf.biggest_filename, cgf.biggest_size);
							cgf.gdl_id=base_gdl_id;
							base_gdl_id+=12345678;

							generic_selected_user_file_list_calls(do_create_one_gdl_for_all_files_from_user_file_list,&cgf);
							free(cgf.biggest_filename);
							cgf.biggest_filename=NULL;
						}

						g_string_free(cgf.dl,TRUE);
					}
					break;

		default:
					break;
	}
}

/********************************************************************************/
/* fill DL_POPUP_XTRA_PARAM file_size according to selected rows of find_result */
/********************************************************************************/
static void check_find_size(GtkTreeModel *gtm, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	DL_POPUP_XTRA_PARAM *dpxp=data;
	gulong file_size;
	gtk_tree_model_get(gtm,iter,FRC_SIZE_AS_VAL,&file_size,-1);

	if(dpxp->with_file_size==0)
	{
		dpxp->with_file_size++;
		dpxp->file_size=file_size;
	}
	else
	{
		if(dpxp->file_size!=file_size)
			dpxp->with_file_size++;
	}
}

/*************************************************************************************/
/* fill DL_POPUP_XTRA_PARAM file_size according to selected rows of "user_file_list" */
/*************************************************************************************/
static void check_share_list_size(GtkTreeModel *gtm, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	DL_POPUP_XTRA_PARAM *dpxp=data;
	
	guint type;
	gulong fsize;

	gtk_tree_model_get(gtm,iter,UFLC_TYPE,&type,UFLC_SIZE_AS_VAL,&fsize,-1);
	if(type==UFL_FILE_ENTRY)
	{
		if(dpxp->with_file_size==0)
		{
			dpxp->with_file_size++;
			dpxp->file_size=fsize;
		}
		else
		{
			if(dpxp->file_size!=fsize)
				dpxp->with_file_size++;
		}
	}
}


/****************************************************************/
/* rebuild a new start_dl_popup_menu with currently running GDL */
/****************************************************************/
/* GTK2: to test */
/*****************/
static void build_start_dl_popup(int from_panel)
{
	GtkWidget *w;
	DL_POPUP_XTRA_PARAM dpxp;

	if(start_dl_popup!=NULL)
	{
		gtk_widget_destroy(start_dl_popup);
	}

	/* create the default start_dl_popup_menu */
	start_dl_popup=create_start_dl_popup_menu();

	/* and add it the list of GDL */
	w=get_widget_by_widget_name(main_window,"gdl_ctree");
	if(w!=NULL)
	{
		GtkTreeModel *gtm;
		GtkTreeStore *gts;
		GtkWidget *mitem;

		if(from_panel==USER_FILE_LIST_TAB)
		{
			mitem=gtk_menu_item_new_with_label(_("Download Selected directories"));
			gtk_menu_append(GTK_MENU(start_dl_popup),mitem);
			gtk_signal_connect(GTK_OBJECT(mitem),"activate", GTK_SIGNAL_FUNC(add_recursive_to_gdl_with_create),NULL);
			gtk_widget_show(mitem);

			mitem=gtk_menu_item_new_with_label(_("Shrink all"));
			gtk_menu_append(GTK_MENU(start_dl_popup),mitem);
			gtk_signal_connect(GTK_OBJECT(mitem),"activate", GTK_SIGNAL_FUNC(on_shrink_all_ctree),"user_file_list_clist");
			gtk_widget_show(mitem);

			mitem=gtk_menu_item_new_with_label(_("Expand all"));
			gtk_menu_append(GTK_MENU(start_dl_popup),mitem);
			gtk_signal_connect(GTK_OBJECT(mitem),"activate", GTK_SIGNAL_FUNC(on_expand_all_ctree),"user_file_list_clist");
			gtk_widget_show(mitem);
		}

		mitem=gtk_menu_item_new_with_label(_("-- create GDLs"));
		gtk_widget_set_sensitive(mitem,FALSE);
		gtk_menu_append(GTK_MENU(start_dl_popup),mitem);
		gtk_widget_show(mitem);

		mitem=gtk_menu_item_new_with_label(_("Create a new GDL for each selected files"));
		gtk_menu_append(GTK_MENU(start_dl_popup),mitem);
		gtk_signal_connect(GTK_OBJECT(mitem),"activate", GTK_SIGNAL_FUNC(add_to_gdl_with_create),create_one_gdl_per_files);
		gtk_widget_show(mitem);

		mitem=gtk_menu_item_new_with_label(_("Create one new GDL for all selected files"));
		gtk_menu_append(GTK_MENU(start_dl_popup),mitem);
		gtk_signal_connect(GTK_OBJECT(mitem),"activate", GTK_SIGNAL_FUNC(add_to_gdl_with_create),create_one_gdl_for_all_files);
		gtk_widget_show(mitem);

		if(from_panel==FIND_TAB)
		{
			mitem=gtk_menu_item_new_with_label(_("Create one new GDL for all selected files + add autoscan"));
			gtk_menu_append(GTK_MENU(start_dl_popup),mitem);
			gtk_signal_connect(GTK_OBJECT(mitem),"activate", GTK_SIGNAL_FUNC(add_to_gdl_with_create),create_one_gdl_for_all_files_with_autoscan);
			gtk_widget_show(mitem);
		}

		dpxp.with_file_size=0;
		dpxp.file_size=0;

		switch(from_panel)
		{
			case FIND_TAB:	generic_selected_find_result_calls(check_find_size,&dpxp);
								break;
			case USER_FILE_LIST_TAB:
								generic_selected_user_file_list_calls(check_share_list_size,&dpxp);
								break;
		}

		if(dpxp.with_file_size==1)
		{
			/* we have one size only */
			dpxp.bad_size=gtk_menu_new();
			dpxp.nb_bad_size=0;
		}
		else
		{
			dpxp.with_file_size=0;		/* file size is irrevelant */
			dpxp.bad_size=NULL;
			dpxp.nb_bad_size=0;
		}


		if(dpxp.with_file_size==0)
			mitem=gtk_menu_item_new_with_label(_("-- active GDLs"));
		else
			mitem=gtk_menu_item_new_with_label(_("-- active GDLs with the same size"));
		gtk_widget_set_sensitive(mitem,FALSE);
		gtk_menu_append(GTK_MENU(start_dl_popup),mitem);
		gtk_widget_show(mitem);

		gts=GTK_TREE_STORE(gtm=gtk_tree_view_get_model(GTK_TREE_VIEW(w)));
		{
			GtkTreeIter iter;
			gboolean valid;

			valid=gtk_tree_model_get_iter_first(gtm,&iter);
			while(valid)
			{
				add_gdl_entry_to_popup(gtm,gts,&iter,&dpxp);
				valid=gtk_tree_model_iter_next(gtm,&iter);
			}
		}

		if(dpxp.with_file_size==1)
		{
			if(dpxp.nb_bad_size==0)
				gtk_widget_destroy(dpxp.bad_size);
			else
			{
				mitem=gtk_menu_item_new_with_label(_("-- active GDLs with a different size"));
				gtk_menu_append(GTK_MENU(start_dl_popup),mitem);
				gtk_widget_show(mitem);
				gtk_menu_item_set_submenu(GTK_MENU_ITEM(mitem),dpxp.bad_size);
			}
		}
	}
}

/*****************/
/* GTK2: to test */
/*****************/
gboolean
on_user_file_list_clist_button_press_event
													 (GtkWidget		 *widget,
													 GdkEventButton  *event,
													 gpointer			user_data)
{
	if(event==NULL)
		return TRUE;

	if((event->button==1)&&(event->type==GDK_2BUTTON_PRESS))
	{  /* left click */
		/* double click on a row */
		GtkTreePath *path;

		if(gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(widget),(gint) event->x, (gint) event->y, &path,NULL,NULL,NULL)==TRUE)
		{
			GtkTreeIter iter;
			GtkTreeModel *gtm;
			guint type;

			gtk_tree_model_get_iter (gtm=gtk_tree_view_get_model(GTK_TREE_VIEW(widget)),&iter,path);

			gtk_tree_model_get(gtm,&iter,UFLC_TYPE,&type,-1);

			if(type==UFL_FILE_ENTRY)
			{
				GtkTreeIter nick_iter;

				if(get_nick_entry_for_user_file_list_from_ufce(gtm,&iter,&nick_iter))
				{
					gulong fsize;
					char *nickname;
					char *fname;
					GString *cm;

					gtk_tree_model_get(gtm,&nick_iter,UFLC_STR,&nickname,-1);
					gtk_tree_model_get(gtm,&iter,UFLC_STR,&fname,UFLC_SIZE_AS_VAL,&fsize,-1);
					cm=g_string_new("/DL |");										  /* adapted to new /DL */
					g_string_sprintfa(cm,"%s||%s|%lu\n",nickname,fname,fsize);

					send_data_to_gdl_dctc(cm->str);
					g_string_free(cm,TRUE);
					free(fname);
					free(nickname);
				}
			}
		}
		return TRUE;
	}
	else if(event->button==3)
	{  /* right-click */
		build_start_dl_popup(USER_FILE_LIST_TAB);
		gtk_menu_popup(GTK_MENU(start_dl_popup),NULL,NULL,NULL,NULL,event->button,event->time);
		return TRUE;
	}

	return FALSE;
}


gboolean
on_user_file_list_clist_key_press_event
													 (GtkWidget		 *widget,
													 GdkEventKey	  *event,
                                        gpointer         user_data)
{

  return FALSE;
}


/* ------------------------------------------------------------------------------------------------------------- */
/***********************************/
/* clear the user share list clist */
/***********************************/
/* GTK2: to test */
/*****************/
void
on_clear_user_file_list_clicked        (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *w;
	
	w=get_widget_by_widget_name(main_window,"user_file_list_clist");
	if(w!=NULL)
	{
		GtkTreeStore *gts;

		if(user_list_locked==FALSE)
		{
			user_list_locked=TRUE;
			gts=GTK_TREE_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(w)));
			gtk_tree_store_clear(gts);
			user_list_locked=FALSE;
		}
	}
}


/* ------------------------------------------------------------------------------------------------------------- */
/**************************************************************************************/
/* scan the given tree model (it is a TREE STORE) from the given tree path            */
/* the scan is recursive. The direction of the scan is -1 (to go to previous rows)    */
/* or 1 (to go to next rows). For each encountered row, the function fnc is called    */
/* NOTE: if direction is 1, the function is called for the root node before its       */
/* children. If direction is -1, the function is called for root node children before */
/* the root node itself.                                                              */
/* if the function fnc returns TRUE, the scan is aborted.                             */
/**************************************************************************************/
/* GTK2: to test */
/*****************/
void gtk_tree_store_scan(GtkTreeModel *gtm, GtkTreePath *path, int direction, void *data, gboolean (*fnc)(GtkTreeModel *, GtkTreePath *, GtkTreeIter *,void *))
{
	gboolean ret;
	int i;
	GtkTreeIter iter;

	while(1)
	{
		/* call the function */
		if(gtk_tree_model_get_iter(gtm,&iter,path)==FALSE)
		{
			fprintf(stderr,"WARNING: during gtk_tree_store_scan, an invalid path was created\n");
			break;
		}

		ret=(*fnc)(gtm,path,&iter,data);
		if(ret==TRUE)
			break;

		if(direction==1)
		{

			/* move to the next node */
			if(gtk_tree_model_iter_has_child(gtm,&iter)==TRUE)
			{
				/* the node has child, let's go deeper in the tree */
				gtk_tree_path_down(path);
			}
			else
			{
				/* there is no child, go to the next sibbling */
				dir_down:
				if(gtk_tree_model_iter_next(gtm,&iter)==TRUE)
				{
					/* there is a sibbling */
					gtk_tree_path_next(path);
				}
				else
				{
					/* no child, no sibbling, go to the sibbling of the parent */
					if(gtk_tree_path_up(path)==FALSE)
					{
						/* looks weird but there is no problem to go to ancestor of the root node */
						/* that's why, the depth is also tested */
						break;
					}

					if(gtk_tree_path_get_depth(path)==0)
					{
						/* and there is no parent ... it is the end of the world :) */
						/* => the end of the tree scan (last row reached) */
						break;
					}

					/* adjust the path position to the parent */
					if(gtk_tree_model_get_iter(gtm,&iter,path)==FALSE)
					{
						fprintf(stderr,"WARNING: during gtk_tree_store_scan, an invalid path was created. (2)\n");
						break;
					}

					/* and retry the sibbling */
					goto dir_down;
				}
			}
		}
		else
		{
			/* move to the previous node */
			if(gtk_tree_path_prev(path)==TRUE)
			{
				/* there is a previous node but the previous node of a node is not its previous sibbling */
				/* it is the last children of the last children ... of its previous sibbling */
				take_last:
				if(gtk_tree_model_get_iter(gtm,&iter,path)==FALSE)
				{
					fprintf(stderr,"WARNING: during gtk_tree_store_scan, an invalid path was created. (3)\n");
					break;
				}

				if(gtk_tree_model_iter_has_child(gtm,&iter)==TRUE)
				{
					/* damn, the previous node has children, take the last children */
					int nb_children;

					nb_children=gtk_tree_model_iter_n_children(gtm,&iter);

					/* go to the last children of the current node */
					gtk_tree_path_down(path);
					for(i=0;i<(nb_children-1);i++)
						gtk_tree_path_next(path);

					/* well, we now have the last child */
					if(gtk_tree_model_get_iter(gtm,&iter,path)==FALSE)
					{
						fprintf(stderr,"WARNING: during gtk_tree_store_scan, an invalid path was created. (4)\n");
						break;
					}
					goto take_last;
				}
			}
			else
			{
				/* the node has no previous node, take its parent */
				if(gtk_tree_path_up(path)==FALSE)
				{
					/* looks weird but there is no problem to go to ancestor of the root node */
					/* that's why, the depth is also tested */
					break;
				}

				if(gtk_tree_path_get_depth(path)==0)
				{
					/* and there is no parent ... it is the end of the world :) */
					/* => the end of the tree scan (first row reached) */
					break;
				}
			}
		}
	}
}

typedef struct
{
	GtkTreeView *gtv;
	GString *str;
	int has_dir;
	int has_file;
	int ignore;		/* if !=0, the first row is ignored */
} TMP_UFL_SEARCH;

static void my_gtk_tree_view_expand_to_path(GtkTreeView *gtv, GtkTreePath *path)
{
	GtkTreePath *parent;

	parent=gtk_tree_path_copy(path);
	if(gtk_tree_path_up(parent))
		my_gtk_tree_view_expand_to_path(gtv,parent);
	gtk_tree_path_free(parent);

	gtk_tree_view_expand_row(gtv,path,FALSE);
}

/********************************************************/
/* compare the given gtm[iter] with the searched string */
/********************************************************/
/* output: FALSE= continue; TRUE= abort */
/****************************************/
gboolean search_for_tus(GtkTreeModel *gtm, GtkTreePath *treepath, GtkTreeIter *iter, void *data)
{
	TMP_UFL_SEARCH *tus=data;
	int ret=FALSE;
	guint type;
	char *str_val;

	if(tus->ignore!=0)
	{
		tus->ignore=0;
		return ret;
	}

	gtk_tree_model_get(gtm,iter,UFLC_TYPE,&type,UFLC_TREE_COL,&str_val,-1);

	if(str_val!=NULL)
	{
		if( ((tus->has_dir)&&(type==UFL_DIR_ENTRY)) ||
		 	((tus->has_file)&&(type==UFL_FILE_ENTRY)) )
		{
			if(my_strcasestr(tus->str,str_val)!=NULL)
			{
				ret=TRUE;
			}
		}
	
		free(str_val);
	}

	if(ret==TRUE)
	{
		/* we have a row, select it, focus it and stop */
#if 0
		gtk_tree_view_expand_to_path(tus->gtv,treepath);		/* GTK2.2 at least */
#else
		my_gtk_tree_view_expand_to_path(tus->gtv,treepath);		/* and my version for GTK2.0 */
#endif
		gtk_tree_view_set_cursor(tus->gtv,treepath,NULL,FALSE);
		gtk_tree_view_scroll_to_cell(tus->gtv,treepath,NULL,TRUE,0.2,0.0);
	}
	return ret;	/* don't stop */
}

static void user_file_list_find(const char *pattern,int has_dir, int has_file,int start_position,int direction)
{
	GtkWidget *w;
	GtkTreeStore *gts;
	GtkTreeView *gtv;
	GtkTreeModel *gtm;
	GtkTreeSelection *slc;
	GtkTreePath *path;
	TMP_UFL_SEARCH tus;

	if((direction!=-1)&&(direction!=1))
	{
		fprintf(stderr,"user_file_list_find: invalid direction, only -1 and 1 allowed\n");
		return;
	}

	w=get_widget_by_widget_name(main_window,"user_file_list_clist");
	gts=GTK_TREE_STORE(gtm=gtk_tree_view_get_model(gtv=GTK_TREE_VIEW(w)));
	slc=gtk_tree_view_get_selection(gtv);

	tus.gtv=gtv;
	tus.ignore=0;
	tus.str=g_string_new(pattern);
	tus.has_dir=has_dir;
	tus.has_file=has_file;

	if(start_position==-1)
	{
		gtk_tree_view_get_cursor(gtv,&path,NULL);
		if(path==NULL)
			path=gtk_tree_path_new_first();
		else
			tus.ignore=1;
	}
	else
	{
		path=gtk_tree_path_new_first();
	}

	gtk_tree_selection_unselect_all(slc);

	gtk_tree_store_scan(gtm, path, direction, &tus, search_for_tus);

	gtk_tree_path_free(path);
	g_string_free(tus.str,TRUE);
}

/*********************************************/
/* prepare search for a pattern with options */
/*********************************************/
/* GTK2: to test */
/*****************/
void do_user_file_list_search(int start_position, int direction)
{
	const char *pattern;

	pattern=gtk_entry_get_text(GTK_ENTRY(get_widget_by_widget_name(main_window,"user_file_list_search_entry")));
	if((pattern!=NULL)&&(strlen(pattern)))
	{
		int has_dir;
		int has_file;
	
		has_dir=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(get_widget_by_widget_name(main_window,"user_file_list_search_as_directory_checkbutton")));
		has_file=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(get_widget_by_widget_name(main_window,"user_file_list_search_as_file_checkbutton")));

		user_file_list_find(pattern,has_dir,has_file,start_position,direction);
	}
}

/******************************************************/
/* search the first file/directory matching a pattern */
/******************************************************/
/* GTK2: to test */
/*****************/
void
on_user_file_list_search_entry_activate
													 (GtkEntry		  *entry,
													 gpointer			user_data)
{
	do_user_file_list_search(0,1);   /* search from the first list and from top to bottom of the list */
}


/* ------------------------------------------------------------------------------------------------------------- */
/******************************************************/
/* search the first file/directory matching a pattern */
/******************************************************/
/* GTK2: to test */
/*****************/
void
on_user_file_list_search_button_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	do_user_file_list_search(0,1);   /* search from the first list and from top to bottom of the list */
}


/* ------------------------------------------------------------------------------------------------------------- */
/*****************************************************/
/* search the next file/directory matching a pattern */
/*****************************************************/
/* GTK2: to test */
/*****************/
void
on_user_file_list_find_previous_button_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	do_user_file_list_search(-1,-1); /* search from the currently selected row and from bottom to top of the list */
}


/* ------------------------------------------------------------------------------------------------------------- */
/*********************************************************/
/* search the previous file/directory matching a pattern */
/*********************************************************/
/* GTK2: to test */
/*****************/
void
on_user_file_list_find_next_button_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	do_user_file_list_search(-1,1); /* search from the currently selected row and from bottom to top of the list */
}

/* ------------------------------------------------------------------------------------------------------------- */
gboolean
on_cached_user_list_clist_button_press_event
                                        (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
	if(event==NULL)
		return TRUE;

	if((event->button==1)&&(event->type==GDK_2BUTTON_PRESS))
	{  /* left click */
		/* double click on a row */
		on_load_selected_share_lists_button_clicked(NULL,NULL);
		return TRUE;
	}

  return FALSE;
}


/* ------------------------------------------------------------------------------------------------------------- */
/********************************************/
/* manage reloading of user share list list */
/********************************************/
/* GTK 2: ok */
/*************/
void
on_reload_cached_user_list_clicked     (GtkButton       *button,
                                        gpointer         user_data)
{
	if(user_list_locked==FALSE)
	{
		user_list_locked=TRUE;
		reload_ls_cache_clist(TRUE);
		user_list_locked=FALSE;
	}
}


/* ------------------------------------------------------------------------------------------------------------- */
/**************************************/
/* manage loading of user share lists */
/**************************************/
/* GTK 2: ok */
/*************/
static void load_one_share_list(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	GLOB_USER *gu;

	gtk_tree_model_get(model,iter,CULC_USER_POINTER,&gu,-1);

	/* create a dummy string used to simulate the LSCCH incoming message */
	put_nick_ls_into_user_file_list_clist(gu->unfmt_nick);
}

void
on_load_selected_share_lists_button_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *w;
	GtkTreeSelection *slc;

	w=get_widget_by_widget_name(main_window,"cached_user_list_clist");
	if(w==NULL)
		return;

	if(user_list_locked==FALSE)
	{
		user_list_locked=TRUE;
		slc=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));
		gtk_tree_selection_selected_foreach(slc,load_one_share_list,NULL);
		user_list_locked=FALSE;
	}
}

/* ------------------------------------------------------------------------------------------------------------- */
/***************************************/
/* manage deletion of user share lists */
/***************************************/
/* GTK 2: ok */
/*************/
static void delete_one_share_list(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	GString *str=data;
	GLOB_USER *gu;

	gtk_tree_model_get(model,iter,CULC_USER_POINTER,&gu,-1);

	g_string_sprintf(str,"%s/ls_cache/%s",dctc_main_dir->str,gu->unfmt_nick);
	unlink(str->str);
}

void
on_delete_selected_share_lists_button_clicked
													 (GtkButton		 *button,
													 gpointer			user_data)
{
	GtkWidget *w;
	GtkTreeSelection *slc;
	GString *str;

	w=get_widget_by_widget_name(main_window,"cached_user_list_clist");
	if(w==NULL)
		return;

	if(user_list_locked==FALSE)
	{
		user_list_locked=TRUE;
		slc=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));

		str=g_string_new("");
		gtk_tree_selection_selected_foreach(slc,delete_one_share_list,str);
		g_string_free(str,TRUE);

		reload_ls_cache_clist(TRUE);
		user_list_locked=FALSE;
	}
}
/* ------------------------------------------------------------------------------------------------------------- */


/*******************************************/
/* stop blinking of the private chat label */
/*******************************************/
/* GTK2: ok */
/************/
void
on_chat_notebook_switch_page           (GtkNotebook     *notebook,
                                        GtkNotebookPage *page,
                                        guint            page_num,
                                        gpointer         user_data)
{
	if(page_num!=0)
		blink_off(lbl_chat[page_num-1]);		/* private chat */
}

/* ------------------------------------------------------------------------------------------------------------- */
/************************/
/* close a private chat */
/************************/
/* GTK2: to test */
/*****************/
void
on_close_pchat_button_clicked          (GtkButton       *button,
                                        gpointer         user_data)
{
	int i;
	GtkWidget *w;

	w=get_widget_by_widget_name(main_window,"chat_notebook");
	if(w)
	{
		GtkWidget *lbl;

		i=gtk_notebook_get_current_page(GTK_NOTEBOOK(w));
		if(i!=0)		/* the first page is the public chat */
		{
			i--;		/* ignore the public chat */
			lbl=get_widget_by_widget_name(main_window,lbl_chat[i]);
			if(lbl!=NULL)
				gtk_label_set(GTK_LABEL(lbl),_("empty"));
			g_string_assign(gstr_chat[i],"");		/* wipe nickname */
	
			wipe_gtk_text(chat_text[i]);
		}
	}
}


/* ------------------------------------------------------------------------------------------------------------- */
/***************************/
/* close all private chats */
/***************************/
/* GTK2: to test */
/*****************/
void
on_close_all_pchat_button_clicked		(GtkButton		 *button,
													 gpointer			user_data)
{
	int i;
	for(i=0;i<9;i++)	  /* there is 9 private chats */
	{
		GtkWidget *lbl;

		lbl=get_widget_by_widget_name(main_window,lbl_chat[i]);
		if(lbl!=NULL)
			gtk_label_set(GTK_LABEL(lbl),_("empty"));
		g_string_assign(gstr_chat[i],"");		/* wipe nickname */

		wipe_gtk_text(chat_text[i]);
		blink_off(lbl_chat[i]);
	}
}


/* ------------------------------------------------------------------------------------------------------------- */
/**********************************************/
/* send a message to the current private chat */
/**********************************************/
/* GTK2: to test */
/*****************/
void
on_pchat_entry_activate					 (GtkEntry		  *entry,
													 gpointer			user_data)
{
	int i;
	GtkWidget *w;

	w=get_widget_by_widget_name(main_window,"chat_notebook");
	if(w)
	{
		const char *nick;
		gchar *msg;

		msg=get_gtk_entry(entry);
		if(msg!=NULL)
		{
			if(strlen(msg)>0)
			{
				GString *out;

				i=gtk_notebook_get_current_page(GTK_NOTEBOOK(w));
				if(i==0)
				{
					/* it is for the public chat */
					out=g_string_new("/CHAT ");
					g_string_sprintfa(out,"%s\n",msg);
					translate_char(out,6);
				}
				else
				{
					i--;	/* ignore the public chat */
					nick=gstr_chat[i]->str;
	
					out=g_string_new("/PRIV ");
		  			g_string_sprintfa(out,"|%s|%s\n",nick,msg);
					translate_char(out,strlen(nick)+2+6);  /* translate the message */
				}
				send_data_to_dctc(out->str);
				g_string_free(out,TRUE);
			}
			g_free(msg);
		}
		gtk_entry_set_position(entry,0);
		gtk_entry_set_text(entry,"");
	}
}

/********************************************************/
/* complete the partial nickname at the cursor position */
/********************************************************/
gboolean
on_pchat_entry_key_press_event			(GtkWidget		 *widget,
													 GdkEventKey	  *event,
													 gpointer			user_data)
{
	if(event==NULL)
		return FALSE;

	switch(event->keyval)
	{
		case GDK_Tab:  /* tab pressed */
						do_nick_completion(GTK_EDITABLE(widget));
						return TRUE;
	}

	return FALSE;
}

/* ------------------------------------------------------------------------------------------------------------- */
/**********************************/
/* empty the private chat content */
/**********************************/
/* GTK2: to test */
/*****************/
void
on_clear_private_chat_button_clicked	(GtkButton		 *button,
                                        gpointer         user_data)
{
	GtkWidget *w;
	w=get_widget_by_widget_name(main_window,"chat_notebook");
	if(w)
	{
		int i;

		i=gtk_notebook_get_current_page(GTK_NOTEBOOK(w));
		if(i==0)
		{
			/* it is the public chat */
			wipe_gtk_text("chat_output");
		}
		else
		{
			i--;	/* ignore the public chat */
			wipe_gtk_text(chat_text[i]);
		}
	}
}


/* ------------------------------------------------------------------------------------------------------------- */
/******************************/
/* flag the user of this chat */
/******************************/
/* GTK2: to test */
/*****************/
void
on_flag_pchat_user_button_clicked		(GtkButton		 *button,
													 gpointer			user_data)
{
	GtkWidget *w;
	w=get_widget_by_widget_name(main_window,"chat_notebook");
	if(w)
	{
		int i;
	
		i=gtk_notebook_get_current_page(GTK_NOTEBOOK(w));
		if(i!=0)
		{
			i--;
			auto_flag_user(gstr_chat[i]->str);
		}
	}
}

gboolean
on_shared_dir_clist_key_press_event    (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{

  return FALSE;
}

/* ------------------------------------------------------------------------------------------------------------- */
/********************************************************/
/* root widget for the add share directory fileselector */
/********************************************************/
static GtkWidget *add_share_dir_fselect=NULL;

void
on_add_shared_dir_button_clicked       (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *dl_dir_ent;

	add_share_dir_fselect=create_shared_dir_fileselection();

	dl_dir_ent=get_widget_by_widget_name(main_window,"dl_dir_entry");
	if(dl_dir_ent!=NULL)
	{
		const char *t;

		t=gtk_entry_get_text(GTK_ENTRY(dl_dir_ent));
		gtk_file_selection_set_filename(GTK_FILE_SELECTION(add_share_dir_fselect),t);
	}
	gtk_widget_show(add_share_dir_fselect);
}

/* ------------------------------------------------------------------------------------------------------------- */
void
on_remove_shared_dir_button_clicked    (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *w;
	GtkTreeModel *gtm;
	GtkListStore *gls;
	GtkTreeSelection *slc;

	int i;

	w=get_widget_by_widget_name(main_window,"shared_dir_clist");
	gls=GTK_LIST_STORE(gtm=gtk_tree_view_get_model(GTK_TREE_VIEW(w)));
	slc=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));

	i=gtk_tree_model_iter_n_children(gtm,NULL)-1;
	while(i>=0)
	{
		GtkTreeIter iter;
		if(gtk_tree_model_iter_nth_child(gtm,&iter,NULL,i)==TRUE)
		{
			if(gtk_tree_selection_iter_is_selected(slc,&iter)==TRUE)
				gtk_list_store_remove(gls,&iter);
		}
		i--;
	}

	/* refresh the hidden entry */
	sync_share_list_with_hidden_entry();
}


/* ------------------------------------------------------------------------------------------------------------- */
/*******************************************************/
/* root widget for the download directory fileselector */
/*******************************************************/
static GtkWidget *browse_dl_dir=NULL;

void
on_do_browse_dl_dir_button_clicked     (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *dl_dir_ent;

	browse_dl_dir=create_dl_dir_fileselection();

	dl_dir_ent=get_widget_by_widget_name(main_window,"dl_dir_entry");
	if(dl_dir_ent!=NULL)
	{
		const char *t;

		t=gtk_entry_get_text(GTK_ENTRY(dl_dir_ent));
		gtk_file_selection_set_filename(GTK_FILE_SELECTION(browse_dl_dir),t);
	}
	gtk_widget_show(browse_dl_dir);
}

/*****************************************************/
/* root widget for the vshare directory fileselector */
/*****************************************************/
static GtkWidget *vshare_dir_fselect=NULL;


void
on_do_browse_vshare_dir_button_clicked (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *dl_dir_ent;

	vshare_dir_fselect=create_vshare_dir_fileselection();

	/* no need to perform UTF8 -> locale conversion because set_filename requires UTF8 */
	dl_dir_ent=get_widget_by_widget_name(main_window,"vshare_dir_entry");
	if(dl_dir_ent!=NULL)
	{
		const char *t;

		t=gtk_entry_get_text(GTK_ENTRY(dl_dir_ent));
		gtk_file_selection_set_filename(GTK_FILE_SELECTION(vshare_dir_fselect),t);
	}
	gtk_widget_show(vshare_dir_fselect);
}


/* ------------------------------------------------------------------------------------------------------------- */

/*************************************************/
/* check if the given string is inside the array */
/*************************************************/
/* output: 1=yes, 0=no */
/***********************/
static int in_array(char *str,GPtrArray *array)
{
	int i;

	if(array==NULL)
		return 0;
	for(i=0;i<array->len;i++)
	{
		char *t;
		t=g_ptr_array_index(array,i);
		if(!strcmp(str,t))
			return 1;
	}
	return 0;
}

void
on_apply_pref_clicked                  (GtkButton       *button,
                                        gpointer         user_data)
{
	const char *var_var;
	const char *text_var;

	struct
	{
		char *widget_name;
		char *var_name;
		char *cmd_name;
	} lnk[]={{"user_description_entry","user_desc","/DESC "},
				{"cnx_type_entry","cnx_type","/CNX "},
				{"e_mail_entry","email","/EMAIL "},
				{"incoming_port_number_entry","com_port","/PORT "},
				{"xfer_host_ip_entry","hostip","/IP "},
				{"dl_dir_entry","dl_path","/LPATH "},
				{"unodeport_entry","unode_port","/UNODEPORT "},
				{"min_gdl_wake_up_delay_entry","min_gdl_wake_up_delay","/DFLAG min_gdl_wake_up_delay "},
				{"vshare_dir_entry","vshare_dir","/VSHARE "},
				{"lmule_temp_entry","gdl_met_dir","/GDLMETDIR "},
				{NULL,NULL,NULL}};
	int i;
	GString *out;
	gfloat nflt;
	char *t;
	const char *ct;

	if(current_dctc==NULL)
		gnome_app_error(GNOME_APP(main_window),_("You are not connected to a running DCTC,\nit is not necessary to press this button."));

	var_var=get_var("nickname");
	text_var=gtk_entry_get_text(GTK_ENTRY(get_widget_by_widget_name(main_window,"nickname_entry")));
	
	if(var_var&&text_var&&(strcmp(var_var,text_var)))
	{
		gnome_app_error(GNOME_APP(main_window),_("Due to a direct connect limitation, it is not\npossible to change your nickname on a running client."));
		
	}

	out=g_string_new("");
	i=0;

	/* first, the strings, easy */
	while(lnk[i].widget_name!=NULL)
	{
		gchar *var_textvar;

		var_var=get_var(lnk[i].var_name);
		var_textvar=get_gtk_entry_by_name(main_window,lnk[i].widget_name);
	
		if( (var_var&&var_textvar&&(strcmp(var_var,var_textvar))) ||
			 ((var_var==NULL)&&(var_textvar!=NULL)) ||
			 ((var_var!=NULL)&&(var_textvar==NULL)) 
			)
		{
			g_string_sprintf(out,"%s%s\n",lnk[i].cmd_name,var_textvar);
			send_data_to_dctc(out->str);
			send_data_to_gdl_dctc(out->str);
		}
		g_free(var_textvar);
		i++;
	}

	/* the #download slot slider */
	var_var=get_var("dl_slot");
	nflt=gtk_range_get_adjustment(GTK_RANGE(get_widget_by_widget_name(main_window,"sim_dl_hscale")))->value;
	if(var_var==NULL)
		var_var="0";
	if(strtod(var_var,NULL)!=nflt)
	{
		g_string_sprintf(out,"/SLOT %u\n",(unsigned)nflt);
		send_data_to_dctc(out->str);
		send_data_to_gdl_dctc(out->str);
	}

	/* the reconnection delay slider */
	var_var=get_var("recon_delay");
	nflt=gtk_range_get_adjustment(GTK_RANGE(get_widget_by_widget_name(main_window,"reconnect_delay_scale")))->value;
	if(var_var==NULL)
		var_var="0";
	if(strtod(var_var,NULL)!=nflt)
	{
		g_string_sprintf(out,"/RECOND %u\n",(unsigned)nflt);
		send_data_to_dctc(out->str);
		send_data_to_gdl_dctc(out->str);
	}

	/* the rebuild database delay slider */
	var_var=get_var("auto_rebuild_delay");
	nflt=gtk_range_get_adjustment(GTK_RANGE(get_widget_by_widget_name(main_window,"rebuild_delay_scale")))->value;
	if(var_var==NULL)
		var_var="0";
	if(strtod(var_var,NULL)!=nflt)
	{
		g_string_sprintf(out,"/REBUILD %u\n",60*(unsigned)nflt);
		send_data_to_dctc(out->str);
		send_data_to_gdl_dctc(out->str);
	}

	{
		int i;
		struct
		{
			const char *toggle_widget_name;
			char *cmd_if_true;
			char *cmd_if_false;
		} tbl_toggled_cmd[]={
									{"active_mode_radio_button",		"/ACTIVE\n",		"/PASSIVE\n"}, 		/* the firewall flag */
									{"enable_upload_checkbutton",		"/DLON\n",			"/DLOFF\n"}, 			/* the dl flag */
									{"ddl_checkbutton",					"/DDL\n",			"/NODDL\n"},			/* the DDL flag */
									{"grabip_checkbutton",				"/GBANIP\n",		"/NOGBANIP\n"},		/* the GRABIP flag */
									{"use_done_dir_checkbutton",		"/DONE\n",			"/UNDONE\n"},			/* the when_done flag */
									{"follow_forcemove_checkbutton",	"/FOLLOWFORCE\n",	"/UNFOLLOWFORCE\n"},	/* the followforcemove flag */
									{"abort_upload_checkbutton",		"/ABORTLEAVED\n",	"/NOABORTLEAVED\n"},	/* the abort upload flag */
									{"hide_kick_checkbutton",			"/HIDE_KICK\n",	"/SHOW_KICK\n"},		/* the hide kick message flag */
									{"lazykc_checkbutton",				"/LAZYKC\n",		"/NOLAZYKC\n"},		/* the lazy key check flag */
									{"incoming_wake_up_checkbutton",	"/DFLAG with_incoming_wake_up 1\n",		"/DFLAG with_incoming_wake_up 0\n"},/* the incoming wake up flag */
									{"sr_wake_up_checkbutton",			"/DFLAG with_sr_wake_up 1\n",				"/DFLAG with_sr_wake_up 0\n"},		/* the incoming wake up flag */
									{"dynipcheckbutton",					"/DFLAG dynamic_ip 1\n",					"/DFLAG dynamic_ip 0\n"},				/* the dynamic ip flag */
									{"sharelist_dl_checkbutton",		"/DFLAG sharelist_dl 1\n",					"/DFLAG sharelist_dl 0\n"},				/* the sharelist dl with slot flag */
									{"fake_dcpp_checkbutton",		"/DFLAG fake_dcpp_client 1\n",					"/DFLAG fake_dcpp_client 0\n"},				/* the fake DC++ client flag */
									{NULL,NULL,NULL}
									};

		i=0;
		while(tbl_toggled_cmd[i].toggle_widget_name!=NULL)
		{
			if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(get_widget_by_widget_name(main_window,tbl_toggled_cmd[i].toggle_widget_name)))==TRUE)
			{
				send_data_to_dctc(tbl_toggled_cmd[i].cmd_if_true);
				send_data_to_gdl_dctc(tbl_toggled_cmd[i].cmd_if_true);
			}
			else
			{
				send_data_to_dctc(tbl_toggled_cmd[i].cmd_if_false);
				send_data_to_gdl_dctc(tbl_toggled_cmd[i].cmd_if_false);
			}
			i++;
		}
	}

	/* set xbl values */
	{
		char buf[512];
		struct
		{
			char *cmd_name;
			char *check_widget;
			char *val_widget;
		} sema_bl[]={	{"/UBL","ubl_checkbutton","ubl_entry"},
							{"/DBL","dbl_checkbutton","dbl_entry"},
							{"/GBL","gbl_checkbutton","gbl_entry"},
							{NULL,NULL,NULL}};

		i=0;
		while(sema_bl[i].cmd_name!=NULL)
		{
			if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(get_widget_by_widget_name(main_window,sema_bl[i].check_widget)))==FALSE)
			{
				sprintf(buf,"%s %d\n",sema_bl[i].cmd_name,SEMVMX);
			}
			else
			{
				sprintf(buf,"%s %s\n",sema_bl[i].cmd_name,gtk_entry_get_text(GTK_ENTRY(get_widget_by_widget_name(main_window,sema_bl[i].val_widget))));
			}
			send_data_to_dctc(buf);
			send_data_to_gdl_dctc(buf);

			i++;
		}
	}

	{
		char buf[512];
		sprintf(buf,"/MAXRUNGDLSRC %u\n",(unsigned int)gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(get_widget_by_widget_name(main_window,"maxrunspinbutton"))));
		send_data_to_dctc(buf);
		send_data_to_gdl_dctc(buf);
		sprintf(buf,"/GDLASOFFAFT %u\n",(unsigned int)gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(get_widget_by_widget_name(main_window,"maxasoffspinbutton"))));
		send_data_to_dctc(buf);
		send_data_to_gdl_dctc(buf);
		sprintf(buf,"/MAXUDL %u\n",(unsigned int)gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(get_widget_by_widget_name(main_window,"maxudl_spinbutton"))));
		send_data_to_dctc(buf);
		send_data_to_gdl_dctc(buf);
		sprintf(buf,"/DFLAG min_delay_between_search %u\n",(unsigned int)gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(get_widget_by_widget_name(main_window,"min_delay_between_search_spinbutton"))));
		send_data_to_dctc(buf);
		send_data_to_gdl_dctc(buf);
		sprintf(buf,"/GDLMETPOLL %u\n",(unsigned int)gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(get_widget_by_widget_name(main_window,"lmule_scan_spinbutton"))));
		send_data_to_dctc(buf);
		send_data_to_gdl_dctc(buf);

		if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(get_widget_by_widget_name(main_window,"limit_as_port_checkbutton")))==TRUE)
		{
			sprintf(buf,"/GDLASPORTS %u,%u\n",
					(unsigned int)gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(get_widget_by_widget_name(main_window,"low_as_port_spinbutton"))),
					(unsigned int)gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(get_widget_by_widget_name(main_window,"up_as_port_spinbutton")))
						);
			send_data_to_dctc(buf);
			send_data_to_gdl_dctc(buf);
		}
		else
		{
			send_data_to_dctc("/GDLASPORTS 0,0\n");
			send_data_to_gdl_dctc("/GDLASPORTS 0,0\n");
		}
	}


	/* the shared files list. the harder. We must remove no more shared directories and add newly shared ones */
	{
		GStringChunk *sc;
		GPtrArray *local,*remote,*to_add,*to_remove;

		sc=g_string_chunk_new(128);
		local=g_ptr_array_new();
		remote=g_ptr_array_new();
		to_add=g_ptr_array_new();
		to_remove=g_ptr_array_new();

		/* fill the local array using shared_dir_clist */
		{
			gchar *shared_dir_lst;
			gchar **fld;

			shared_dir_lst=get_gtk_entry(GTK_ENTRY(get_widget_by_widget_name(main_window,"hidden_shared_dir_entry")));
			fld=g_strsplit(shared_dir_lst,"|",0);
			i=0;
			while(fld[i]!=NULL)
			{
				g_ptr_array_add(local,g_string_chunk_insert(sc,fld[i]));
				i++;
			}
			g_free(shared_dir_lst);
			g_strfreev(fld);
		}

		/* fill the remote array using ul_path var */
		ct=get_var("ul_path");
		if(ct!=NULL)
		{
			gchar **fld;
			fld=g_strsplit(ct,"|",0);
			i=0;
			while(fld[i]!=NULL)
			{
				g_ptr_array_add(remote,g_string_chunk_insert(sc,fld[i]));
				i++;
			}
			g_strfreev(fld);
		}

		/* now, all strings of local which do not exist in remote are put into to_add */
		for(i=0;i<local->len;i++)
		{
			t=g_ptr_array_index(local,i);
			if(!in_array(t,remote))
			{
				g_ptr_array_add(to_add,t);
			}
		}

		/* and, all strings of remote which do not exist in all are put into to_remove */
		for(i=0;i<remote->len;i++)
		{
			t=g_ptr_array_index(remote,i);
			if(!in_array(t,local))
			{
				g_ptr_array_add(to_remove,t);
			}
		}

		/* send command to add new shared dir */
		for(i=0;i<to_add->len;i++)
		{
			t=g_ptr_array_index(to_add,i);
			g_string_sprintf(out,"/SHARE %s\n",t);
			send_data_to_dctc(out->str);
			send_data_to_gdl_dctc(out->str);
		}
		
		/* send command to remove shared dir */
		for(i=0;i<to_remove->len;i++)
		{
			t=g_ptr_array_index(to_remove,i);
			g_string_sprintf(out,"/UNSHARE %s\n",t);
			send_data_to_dctc(out->str);
			send_data_to_gdl_dctc(out->str);
		}
		
		/* it is not necessary to free string of to_add and to_remove because they are the same string */
		/* as in local and remote (same ptr) */
		g_ptr_array_free(local,TRUE);
		g_ptr_array_free(remote,TRUE);
		g_ptr_array_free(to_add,TRUE);
		g_ptr_array_free(to_remove,TRUE);
		g_string_chunk_free(sc);
	}

	/* and now, the offset */
	ct=get_var("offset");
	if(ct==NULL)
		ct="0";

	{
		double val;
		double scale[4]={1.0,1024.0, 1024.0*1024.0, 1024.0*1024.0*1024.0};
		char *unit_name[4]={_("Bytes"), _("KBytes"), _("MBytes"), _("GBytes")};
		int i=0;
		char buf[512];
		const char *u;
		const char *v;

		val=strtod(ct,NULL);

		while(i<3)
		{
			double res,ent;

			res=modf(val/scale[i+1],&ent);
			if(res!=0.0)
				break;
			i++;
		}
		sprintf(buf,"%.0f",val/scale[i]);

		u=gtk_entry_get_text(GTK_ENTRY(get_widget_by_widget_name(main_window,"size_offset_unit_entry")));
		v=gtk_entry_get_text(GTK_ENTRY(get_widget_by_widget_name(main_window,"size_offset_entry")));
		if((strcmp(u,unit_name[i]))||(strcmp(v,buf)))
		{
			for(i=0;i<4;i++)
			{
				if(!strcmp(u,unit_name[i]))
					break;
			}
			if(i==4)
				i=0;
			g_string_sprintf(out,"/OFFSET %f\n",strtod(v,NULL)*scale[i]);
			send_data_to_dctc(out->str);
			send_data_to_gdl_dctc(out->str);
		}
	}

	{
		g_string_sprintf(out,"/TOS %u,%u,%u,%u\n",
				tos_string_to_num(gtk_entry_get_text(GTK_ENTRY(get_widget_by_widget_name(main_window,"hub_tos_entry")))),
				tos_string_to_num(gtk_entry_get_text(GTK_ENTRY(get_widget_by_widget_name(main_window,"udp_tos_entry")))),
				tos_string_to_num(gtk_entry_get_text(GTK_ENTRY(get_widget_by_widget_name(main_window,"dl_tos_entry")))),
				tos_string_to_num(gtk_entry_get_text(GTK_ENTRY(get_widget_by_widget_name(main_window,"ul_tos_entry")))));

		send_data_to_dctc(out->str);
		send_data_to_gdl_dctc(out->str);
	}

	g_string_free(out,TRUE);

	send_data_to_dctc("/VARS\n");

	/* update periodic task rate */
	set_periodic_function_call_rate(&running_client_list_refresh_rate,&running_client_list_refresh_rate_gta,
												running_client_list_periodic_refresh,NULL,
												1000*gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(get_widget_by_widget_name(main_window,"running_client_list_refresh_rate_spinbutton"))) );
	set_periodic_function_call_rate(&favorite_client_autostart_check_rate,&favorite_client_autostart_check_rate_gta,
												favorite_client_periodic_autostart,NULL,
												1000*gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(get_widget_by_widget_name(main_window,"favorite_client_autostart_check_rate_spinbutton"))) );
}


/* ------------------------------------------------------------------------------------------------------------- */
void
on_reload_pref_clicked                 (GtkButton       *button,
                                        gpointer         user_data)
{
	fix_pref_window();                  /* if there is no client */
	send_data_to_dctc("/VARS\n");       /* and also if there is a client */
}

/* ------------------------------------------------------------------------------------------------------------- */
/*******************************************************************/
/* check if the given string contains only alphanumeric characters */
/*******************************************************************/
/* output: 1=yes, 0=no */
/***********************/
static int is_all_alnum(const char *str)
{
	if(str==NULL)
		return 0;

	while(*str!='\0')
	{
		if(!isalnum(*str))
			return 0;
		str++;
	}
	return 1;
}

void
on_save_default_profile_button_clicked (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *w;
	const char *profile_name;

	w=get_widget_by_widget_name(main_window,"profile_name_combo_entry");
	if(w==NULL)
		return;

	profile_name=gtk_entry_get_text(GTK_ENTRY(w));
	if((profile_name==NULL)||(strlen(profile_name)==0))
	{
		gnome_app_error(GNOME_APP(main_window),_("You must give profile name first"));
	}
	else
	{
		if(is_all_alnum(profile_name))
		{
			gchar *pname;

			pname=g_strconcat("Profile/",profile_name,"/",NULL);
			gui_full_save(get_widget_by_widget_name(main_window,"user_pref_notebook"),pname);
			g_free(pname);
			gconf_client_suggest_sync(engine,NULL);

			pname=g_strconcat("/" PROGNAME "/ProfileNames/",profile_name,NULL);
			gnome_config_set_string(pname,"User Profile");
			g_free(pname);

			gnome_config_sync();
	
			load_profile_name_combo();
		}
		else
		{
			gnome_app_error(GNOME_APP(main_window),_("Invalid profile name, only alphanumeric characters allowed"));
		}
	}
}


void
on_load_default_profil_button_clicked  (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *w;
	const char *profile_name;

	w=get_widget_by_widget_name(main_window,"profile_name_combo_entry");
	if(w==NULL)
		return;

	profile_name=gtk_entry_get_text(GTK_ENTRY(w));
	if((profile_name==NULL)||(strlen(profile_name)==0))
	{
		gnome_app_error(GNOME_APP(main_window),_("You must give profile name first"));
	}
	else
	{
		if(is_all_alnum(profile_name))
		{
			gchar *pname;
			char *pval;

			pname=g_strconcat("/" PROGNAME "/ProfileNames/",profile_name,NULL);
			pval=gnome_config_get_string(pname);
			g_free(pname);

			if(pval==NULL)
			{
				gnome_app_error(GNOME_APP(main_window),_("No profile has this name, give another one"));
			}
			else
			{
				pname=g_strconcat("Profile/",profile_name,"/",NULL);
				gui_full_restore(get_widget_by_widget_name(main_window,"user_pref_notebook"),pname);
				g_free(pname);
				sync_hidden_entry_with_share_list();
			}
		}
		else
		{
			gnome_app_error(GNOME_APP(main_window),_("Invalid profile name, only alphanumeric characters allowed"));
		}
	}
}


gboolean
on_uaddr_clist_button_press_event      (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
	if(event==NULL)
		return TRUE;

	if(event->button==3)
	{  /* right-click */
		gtk_menu_popup(GTK_MENU(uaddr_popup),NULL,NULL,NULL,NULL,event->button,event->time);
		return TRUE;
	}

	return FALSE;
}


/* ------------------------------------------------------------------------------------------------------------- */
typedef struct
{
	int first_row;
	GString nick;
	GtkTreeView *gtv;
	GtkTreeSelection *slc;
} TMP_FND_UADDR;

/**************************************************/
/* compare a given row with the searched nickname */
/**************************************************/
/* GTK2: to test */
/*****************/
static gboolean match_one_uaddr_row(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	TMP_FND_UADDR *tfu=data;
	GLOB_USER *gu;

	gtk_tree_model_get(model,iter,UAC_USER_POINTER,&gu,-1);
	if(my_strcasestr(&(tfu->nick),gu->unfmt_nick)!=NULL)
	{
		gtk_tree_selection_select_iter(tfu->slc,iter);
		if(tfu->first_row==TRUE)
		{
			GtkTreePath *tree_path;

			tfu->first_row=FALSE;

			tree_path=gtk_tree_model_get_path(model,iter);
			gtk_tree_view_scroll_to_cell(tfu->gtv,tree_path,NULL,TRUE,0.0,0.0);
			gtk_tree_path_free(tree_path);
		}
	}
	
	return FALSE;	/* don't stop */
}

/*********************************************/
/* select all UADDR rows matching a nickname */
/*********************************************/
/* GTK2: to test */
/*****************/
void
on_find_uaddr_by_nick_button_clicked   (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *w;
	char *nick;

	w=get_widget_by_widget_name(main_window,"uaddr_clist");
	if(w==NULL)
		return;

	/* get the searched nickname */
	nick=get_gtk_entry_by_name(main_window,"uaddr_nick_entry");
	if(strlen(nick)!=0)
	{
		TMP_FND_UADDR tfu;
		GtkTreeModel *gtm;

		gtm=gtk_tree_view_get_model(tfu.gtv=GTK_TREE_VIEW(w));

		tfu.slc=gtk_tree_view_get_selection(tfu.gtv);
		gtk_tree_selection_unselect_all(tfu.slc);
		tfu.nick.str=(char*)nick;						/* create a GString-like structure */
		tfu.nick.len=strlen(tfu.nick.str);
		tfu.first_row=TRUE;
		gtk_tree_model_foreach(gtm,match_one_uaddr_row,&tfu);
	}
	g_free(nick);
}

/* ------------------------------------------------------------------------------------------------------------- */
/*********************************************/
/* add a new (Nick,ipport) to the UADDR list */
/*********************************************/
/* GTK2: to test */
/*****************/
void
on_add_new_uaddr_full_clicked          (GtkButton       *button,
                                        gpointer         user_data)
{
	char *nick,*addr;
	GString *str;

	nick=get_gtk_entry_by_name(main_window,"uaddr_nick_entry");
	addr=get_gtk_entry_by_name(main_window,"uaddr_hostipport_entry");
	if((strlen(nick)!=0)&&(strlen(addr)!=0))
	{
		str=g_string_new("");
		if(strchr(addr,':')!=NULL)
			g_string_sprintf(str,"/UADDRUADD %s|%s\n",nick,addr);
		else
			g_string_sprintf(str,"/UADDRUADD %s|%s:412\n",nick,addr);
	
		send_data_to_dctc(str->str);
		g_string_free(str,TRUE);
	
		send_data_to_dctc("/UADDRLST\n");
	}
	g_free(nick);
	g_free(addr);
}


/* ------------------------------------------------------------------------------------------------------------- */
/****************************************/
/* add a new (ipport) to the UADDR list */
/****************************************/
/* GTK2: to test */
/*****************/
void
on_add_new_uaddr_host_clicked			 (GtkButton		 *button,
													 gpointer			user_data)
{
	char *addr;

	addr=get_gtk_entry_by_name(main_window,"uaddr_hostipport_entry");
	if(strlen(addr)!=0)
	{
		GString *str;

		str=g_string_new("");
		if(strchr(addr,':')!=NULL)
			g_string_sprintf(str,"/UADDRIPADD %s\n",addr);
		else
			g_string_sprintf(str,"/UADDRIPADD %s:412\n",addr);
	
		send_data_to_dctc(str->str);
		g_string_free(str,TRUE);

		send_data_to_dctc("/UADDRLST\n");
	}
	g_free(addr);
}

/* ------------------------------------------------------------------------------------------------------------- */
typedef struct
{
	int first_row;
	char *ip;
	int len_ip;
	GtkTreeView *gtv;
	GtkTreeSelection *slc;
} TMP_FND_1UADDR;

/***************************************/
/* compare a given row with the ipport */
/***************************************/
/* GTK2: to test */
/*****************/
static gboolean match_one_uaddr_row_by_uaddr(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	char *ipport;
	TMP_FND_1UADDR *tf1=data;

	gtk_tree_model_get(model,iter,UAC_ADDR_PORT_COL,&ipport,-1);
	if(!strncmp(ipport,tf1->ip,tf1->len_ip))
	{
		gtk_tree_selection_select_iter(tf1->slc,iter);
		if(tf1->first_row==TRUE)
		{
			GtkTreePath *tree_path;

			tf1->first_row=FALSE;

			tree_path=gtk_tree_model_get_path(model,iter);
			gtk_tree_view_scroll_to_cell(tf1->gtv,tree_path,NULL,TRUE,0.0,0.0);
			gtk_tree_path_free(tree_path);
		}
	}
	
	free(ipport);
	return FALSE;	/* don't stop */
}

/*********************************************/
/* select all UADDR rows matching an ip:port */
/*********************************************/
/* GTK2: to test */
/*****************/
void
on_find_uaddr_by_addr_button_clicked	(GtkButton		 *button,
													 gpointer			user_data)
{
	GtkWidget *w;
	TMP_FND_1UADDR tf1;

	w=get_widget_by_widget_name(main_window,"uaddr_clist");
	if(w==NULL)
		return;

	/* get the searched nickname */
	tf1.ip=get_gtk_entry_by_name(main_window,"uaddr_hostipport_entry");
	if((tf1.len_ip=strlen(tf1.ip))!=0)
	{
		GtkTreeModel *gtm;

		gtm=gtk_tree_view_get_model(tf1.gtv=GTK_TREE_VIEW(w));

		tf1.slc=gtk_tree_view_get_selection(tf1.gtv);
		gtk_tree_selection_unselect_all(tf1.slc);
		tf1.first_row=TRUE;
		gtk_tree_model_foreach(gtm,match_one_uaddr_row_by_uaddr,&tf1);
	}
	g_free(tf1.ip);
}

/* ------------------------------------------------------------------------------------------------------------- */
/*********************************************/
/* add a new unode address to the unode list */
/*********************************************/
/* GTK2: to test */
/*****************/
void
on_unode_addr_entry_activate			  (GtkEntry		  *entry,
													 gpointer			user_data)
{
	UNODE_DATA und;
	const char *param;

	param=gtk_entry_get_text(GTK_ENTRY(entry));
	if(param==NULL)
		return;

	if(strlen(param)<1)
		return;

	memset(&und,0,sizeof(und));
	und.entry_enabled=1;
	und.dynamic_ip=1;
	strncpy_max(und.host_unresolved_addr,param,MIN(128,strlen(param)+1));
	
	/* add a new entry to the unode database */
	{
		G_LOCK(unode_lmp);
		if(!lmp_lock_and_map(unode_lmp))
		{
			int exact=-1; 
			int free_entry=-1;
			int i;
			UNODE_DATA *ulm=unode_lmp->mapped_addr;

			for(i=1;i<unode_lmp->nb_records;i++)	/* skip the first record */
			{
				UNODE_DATA *udt=&(ulm[i]);
	
				if(udt->entry_enabled)
				{
					if(!strcmp(udt->host_unresolved_addr,und.host_unresolved_addr))
					{
						exact=i;
						break;
					}
				}
				else
				{ 
					if(free_entry==-1)
						free_entry=i;
				}
			}

			if((exact==-1)&&(free_entry!=-1))
				exact=free_entry;

			if(exact!=-1)
			{
				memcpy(&(ulm[exact]),&und,sizeof(UNODE_DATA));
			}
			else
			{
				lmp_append_record(unode_lmp,&und);
			}
			lmp_unmap_and_unlock(unode_lmp);
		}
		G_UNLOCK(unode_lmp);
	}
	reload_unode_clist(0);

	gtk_entry_set_position(entry,0);
	gtk_entry_set_text(entry,"");

}

/* ------------------------------------------------------------------------------------------------------------- */
/*************************/
/* add a new UNODE entry */
/*************************/
/* GTK2: to test */
/*****************/
void
on_addr_unode_button_clicked			  (GtkButton		 *button,
													 gpointer			user_data)
{
	GtkWidget *w;

	w=get_widget_by_widget_name(main_window,"unode_addr_entry");
	if(w!=NULL)
	{
		on_unode_addr_entry_activate(GTK_ENTRY(w),NULL);
	}
}

/* ------------------------------------------------------------------------------------------------------------- */
typedef struct
{
	int changed;
	UNODE_DATA *ulm;
	int nb_unode_records;
} TMP_UNODE_DEL;

/***************************************/
/* manage deletion of user share lists */
/***************************************/
/* GTK 2: ok */
/*************/
static void delete_one_unode_entry(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	TMP_UNODE_DEL *tud=data;
	char *unode;
	int i;

	gtk_tree_model_get(model,iter,UNC_UNODE_ADDR_COL,&unode,-1);

	for(i=1;i<tud->nb_unode_records;i++)	/* skip the first record */
	{
		UNODE_DATA *udt=&(tud->ulm[i]);

		if(udt->entry_enabled)
		{
			if(!strcmp(udt->host_unresolved_addr,unode))
			{
				udt->entry_enabled=0;
				break;
			}
		}
	}
	tud->changed=TRUE;
	free(unode);
}

/*************************/
/* delete an UNODE entry */
/*************************/
/* GTK2: to test */
/*****************/
void
on_del_selected_unode_button_clicked	(GtkButton		 *button,
													 gpointer			user_data)
{
	GtkWidget *w;
	w=get_widget_by_widget_name(main_window,"unode_clist");
	if(w)
	{
		GtkTreeSelection *slc;
		GtkTreeModel *gtm;
		GtkTreeView *gtv;
		TMP_UNODE_DEL tud;

		tud.changed=FALSE;

		gtm=gtk_tree_view_get_model(gtv=GTK_TREE_VIEW(w));

		slc=gtk_tree_view_get_selection(gtv);
		G_LOCK(unode_lmp);
		if(!lmp_lock_and_map(unode_lmp))
		{
			tud.ulm=unode_lmp->mapped_addr;
			tud.nb_unode_records=unode_lmp->nb_records;
			gtk_tree_selection_selected_foreach(slc,delete_one_unode_entry,&tud);
			lmp_unmap_and_unlock(unode_lmp);
		}
		G_UNLOCK(unode_lmp);

		if(tud.changed==TRUE)
			reload_unode_clist(0);
	}
}

/* ------------------------------------------------------------------------------------------------------------- */
/*************************/
/* Reload the UADDR list */
/*************************/
/* GTK2: to test */
/*****************/
void
on_reload_unode_address_list_button_clicked
													 (GtkButton		 *button,
													 gpointer			user_data)
{
	reload_unode_clist(0);
}

/* ------------------------------------------------------------------------------------------------------------- */
static struct
			{
				const char *widget_name;
				const char *option_name;
				const char *autoflag_widget_name;
			} opt_list[]={
								{"user_flag_ignore_pmsg_togglebutton","IGNORE_PMSG","autoflag_user_flag_ignore_pmsg_togglebutton"},
								{"user_flag_ignore_pubmsg_togglebutton","IGNORE_PUBMSG","autoflag_user_flag_ignore_pubmsg_togglebutton"},
								{"user_flag_ignore_srch_togglebutton","IGNORE_SRCH","autoflag_user_flag_ignore_srch_togglebutton"},
								{"user_flag_ignore_sr_togglebutton","IGNORE_SREP","autoflag_user_flag_ignore_sr_togglebutton"},
								{"user_flag_no_xfer_togglebutton","IGNORE_XFER","autoflag_user_flag_no_xfer_togglebutton"},
								{"user_flag_ignore_dl_limit_togglebutton","IGNORE_SLOT_LIMIT","autoflag_user_flag_ignore_dl_limit_togglebutton"},
								{NULL,NULL}};


/**************************************************************************/
/* update the given nickname in flagged user list (this includes removal) */
/**************************************************************************/
/* GTK2: to test */
/*****************/
static void update_flag_user_list1(const char *nickname)
{
	char *data_val=NULL;
	int data_len=0;
	int ret;
	GtkWidget *w;
	char buf[8192];
	GtkTreeIter iter;
	int valid_iter;
	GtkTreeModel *gtm;
	GtkListStore *gls;
	GLOB_USER *gu;

	w=get_widget_by_widget_name(main_window,"flagged_user_clist");
	if(w==NULL)
		return;

	ret=get_key_data(unwanted_user,nickname,strlen(nickname),(void*)&data_val,&data_len);

	/* find the row of this user */
	gls=GTK_LIST_STORE(gtm=gtk_tree_view_get_model(GTK_TREE_VIEW(w)));

	gu=gu_get_user(nickname);
	if(gu==NULL)
		valid_iter=FALSE;
	else
		valid_iter=gu_user_get_iter_for_model(gu,gtm,&iter);
	
	if(ret!=0)
	{
		/* user not found, delete its row */
		if(valid_iter)
		{
			gu_unref_from_iter(gu,gtm,&iter);
			gtk_list_store_remove(gls,&iter);
		}
	}
	else
	{  /* user found, add/update its row */
		strncpy_max(buf,data_val,MIN(data_len+1,sizeof(buf)));

		if(valid_iter==FALSE)
		{
			if(gu==NULL)
			{
				gu=gu_add_user(nickname,TRUE,NULL);
			}

			gtk_list_store_append(gls,&iter);
			gtk_list_store_set(gls,&iter,FUC_USER_POINTER,gu,FUC_FLAGS_COL,buf,-1);
			gu_ref_from_iter(gu,gtm,&iter);
		}
		else
			gtk_list_store_set(gls,&iter,FUC_FLAGS_COL,buf,-1);

		if(data_val!=NULL)
			free(data_val);
	}
}

/**************************************/
/* add/update flags of a flagged user */
/**************************************/
/* GTK2: to test */
/*****************/
void
on_add_modify_flag_user_button_clicked (GtkButton		 *button,
													 gpointer			user_data)
{
	GtkWidget *w;
	w=get_widget_by_widget_name(main_window,"flagged_user_label");
	if(w)			 
	{
		const char *lbl;

		lbl=gtk_label_get_text(GTK_LABEL(w));

		if((lbl!=NULL)&&(strlen(lbl)))
		{
			GString *flags;
			int i;

			flags=g_string_new("");

			i=0;
			while(opt_list[i].widget_name!=NULL)
			{
				w=get_widget_by_widget_name(main_window,opt_list[i].widget_name);
				if(w!=NULL)
				{
					if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w))==TRUE)
					{
						if(flags->len!=0)
							flags=g_string_append_c(flags,'|');
						flags=g_string_append(flags,opt_list[i].option_name);
					}
				}
				i++;
			}

			if(flags->len!=0)
				set_key_data(unwanted_user,lbl,strlen(lbl),flags->str,flags->len);
			else
				del_key_data(unwanted_user,lbl,strlen(lbl));
			g_string_free(flags,TRUE);

			update_flag_user_list1(lbl);
		}
	}
}

/* ------------------------------------------------------------------------------------------------------------- */
/*************************/
/* delete a flagged user */
/*************************/
/* GTK2: to test */
/*****************/
void
on_delete_flag_user_button_clicked	  (GtkButton		 *button,
													 gpointer			user_data)
{
	GtkWidget *w;
	w=get_widget_by_widget_name(main_window,"flagged_user_label");
	if(w)
	{
		const char *lbl;

		lbl=gtk_label_get_text(GTK_LABEL(w));

		if((lbl!=NULL)&&(strlen(lbl)))
		{
			del_key_data(unwanted_user,lbl,strlen(lbl));
			update_flag_user_list1(lbl);
		}
	}

}

/* ------------------------------------------------------------------------------------------------------------- */
/*****************************/
/* reload flagged user clist */
/*****************************/
/* GTK2: to test */
/*****************/
void
on_reload_flagged_user_list_button_clicked
													 (GtkButton		 *button,
													 gpointer			user_data)
{
	reload_flagged_user_clist(0);    /* force reload ... always */
}

/* ------------------------------------------------------------------------------------------------------------- */
/*****************/
/* GTK2: to test */
/*****************/
gboolean
on_user_clist_button_press_event		 (GtkWidget		 *widget,
													 GdkEventButton  *event,
													 gpointer			user_data)
{
	if(event==NULL)
		return TRUE;

	if((event->button==1)&&(event->type==GDK_2BUTTON_PRESS))
	{  /* left click */
		/* double click on a row */
		foreach_selected_entry_send_cmd_with_glob_user("user_clist",UCC_LST_USER_POINTER,"/LS");
		return TRUE;
	}

	if(event->button==3)
	{  /* right-click */
		gtk_menu_popup(GTK_MENU(user_popup),NULL,NULL,NULL,NULL,event->button,event->time);
		return TRUE;
	}

	return FALSE;
}


/* ------------------------------------------------------------------------------------------------------------- */
/************/
/* GTK2: ok */
/************/
void
on_away_togglebutton_toggled			  (GtkToggleButton *togglebutton,
													 gpointer			user_data)
{
	GtkWidget *w;

	w=get_widget_by_widget_name(main_window,"away_togglebutton");
	if(w!=NULL)
	{
		if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w))==TRUE)
			send_data_to_dctc("/AWAY\n");
		else
			send_data_to_dctc("/HERE\n");
	}
}

/* ------------------------------------------------------------------------------------------------------------- */

/****************************************************/
/* used to store the name of unattached GDL entries */
/****************************************************/
#ifdef UGL_POP
static GStringChunk *gdl_popup_gsc=NULL;

/******************************************************************************************/
/* this function is called when the user clicks on a GDL entry of the start_dl_popup_menu */
/******************************************************************************************/
/* user_data is the name of the GDL */
/************************************/
static void active_add_unattached_gdl(GtkMenuItem *menuitem, gpointer user_data)
{
	GString *str;

	str=g_string_new("/GDLATTACH ");
	str=g_string_append(str,user_data);
	str=g_string_append_c(str,'\n');
	send_data_to_gdl_dctc(str->str);
	g_string_free(str,TRUE);
}

static void try_to_add_to_gdl_popup(const char *dir1, const char *dir2)
{
	GString *str;
	int fd;

	str=g_string_new("");
	g_string_sprintf(str,"%s/%s/.lock",dir1,dir2);

	fd=open(str->str,O_RDWR);
	if(fd!=-1)
	{
		if(lockf(fd,F_TEST,1)==0)
		{
			/* .lock file exists and is lockable, so nobody uses this entry */
			GtkWidget *mitem;

			if(gdl_popup_gsc==NULL)
				gdl_popup_gsc=g_string_chunk_new(48);

			mitem=gtk_menu_item_new_with_label(dir2);
			gtk_signal_connect(GTK_OBJECT(mitem),"activate", 
							GTK_SIGNAL_FUNC(active_add_unattached_gdl),
								g_string_chunk_insert_const(gdl_popup_gsc,dir2));
			gtk_menu_append(GTK_MENU(gdl_popup),mitem);
			gtk_widget_show(mitem);
			
		}
		close(fd);
	}
	g_string_free(str,TRUE);
}
#endif

/**********************************************************/
/* rebuild a new gdl_popup_menu with currently unused GDL */
/**********************************************************/
static void build_gdl_popup(void)
{
#ifdef UGL_POP
	DIR *dir;
	struct dirent *obj;
	const char *t;
	GtkWidget *mitem;

	if(gdl_popup!=NULL)
	{
		gtk_widget_destroy(gdl_popup);
	}
	if(gdl_popup_gsc!=NULL)
	{
		g_string_chunk_free(gdl_popup_gsc);
		gdl_popup_gsc=NULL;
	}

	/* create the default gdl_popup_menu */
	gdl_popup=create_gdl_popup_menu();

	mitem=gtk_menu_item_new_with_label(_("-- unattached GDLs"));
	gtk_widget_set_sensitive(mitem,FALSE);
	gtk_menu_append(GTK_MENU(gdl_popup),mitem);
	gtk_widget_show(mitem);

	t=get_var("dl_path");
	if((t!=NULL)&&(strlen(t)))
	{
		GString *st1;

		st1=g_string_new(t);
		st1=g_string_append(st1,"/GDL");
		dir=opendir(st1->str);
		if(dir!=NULL)
		{
			while((obj=readdir(dir))!=NULL)
			{
				try_to_add_to_gdl_popup(st1->str,obj->d_name);
			}
			closedir(dir);
		}
		g_string_free(st1,TRUE);
	}
#endif
}

/*****************/
/* GTK2: to test */
/*****************/
gboolean
on_gdl_ctree_button_press_event        (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
	if(event==NULL)
		return TRUE;

	if(event->button==3)
	{  /* right-click */					  
		build_gdl_popup();					 
		gtk_menu_popup(GTK_MENU(gdl_popup),NULL,NULL,NULL,NULL,event->button,event->time);
		return TRUE;
	}

	return FALSE;
}

/* ------------------------------------------------------------------------------------------------------------- */
/*****************/
/* GTK2: to test */
/*****************/
gboolean
on_gdl_ctree_key_press_event			  (GtkWidget		 *widget,
													 GdkEventKey	  *event,
													 gpointer			user_data)
{
	if(event==NULL)
		return FALSE;

	switch(event->keyval)
	{
		case GDK_d: /* 'd' pressed */
		case GDK_D:
						kill_selected_gdl_entry();
						break;
	}
	
	return FALSE;
}


/* ------------------------------------------------------------------------------------------------------------- */
/*****************/
/* GTK2: to test */
/*****************/
gboolean
on_upload_clist_button_press_event	  (GtkWidget		 *widget,
													 GdkEventButton  *event,
													 gpointer			user_data)
{
	if(event==NULL)
		return TRUE;

	if(event->button==3)
	{  /* right-click */
		gtk_menu_popup(GTK_MENU(ul_popup),NULL,NULL,NULL,NULL,event->button,event->time);
		return TRUE;
	}

	return FALSE;
}


/* ------------------------------------------------------------------------------------------------------------- */
/*****************/
/* GTK2: to test */
/*****************/
gboolean
on_upload_clist_key_press_event		  (GtkWidget		 *widget,
													 GdkEventKey	  *event,
													 gpointer			user_data)
{
	if(event==NULL)
		return FALSE;

	switch(event->keyval)
	{
		case GDK_d: /* 'd' pressed */
		case GDK_D:
						kill_selected_entry("/KILL","upload_clist",ULC_ID_COL);
						break;
	}

	return FALSE;
}

/* ------------------------------------------------------------------------------------------------------------- */
/*****************/
/* GTK2: to test */
/*****************/
gboolean
on_queue_clist_button_press_event		(GtkWidget		 *widget,
													 GdkEventButton  *event,
													 gpointer			user_data)
{
	if(event==NULL)
		return TRUE;

	if(event->button==3)
	{  /* right-click */
		gtk_menu_popup(GTK_MENU(q_popup),NULL,NULL,NULL,NULL,event->button,event->time);
		return TRUE;
	}

	return FALSE;
}

/* ------------------------------------------------------------------------------------------------------------- */
/*****************/
/* GTK2: to test */
/*****************/
gboolean
on_queue_clist_key_press_event			(GtkWidget		 *widget,
													 GdkEventKey	  *event,
													 gpointer			user_data)
{
	if(event==NULL)
		return FALSE;

	switch(event->keyval)
	{
		case GDK_d: /* 'd' pressed */	 
		case GDK_D:				 
						kill_selected_entry("/KILLKBN","queue_clist",QUC_ID_COL);
						break;
	}

  return FALSE;
}


/* ------------------------------------------------------------------------------------------------------------- */
/*****************/
/* GTK2: to test */
/*****************/
gboolean
on_done_clist_button_press_event		 (GtkWidget		 *widget,
													 GdkEventButton  *event,
													 gpointer			user_data)
{
	if(event==NULL)
		return TRUE;
	
#if 0
	if(event->button==3)
	{  /* right-click */
		gtk_menu_popup(GTK_MENU(done_popup),NULL,NULL,NULL,NULL,event->button,event->time);
		return TRUE;
	}
#endif

	return FALSE;
}


/* ------------------------------------------------------------------------------------------------------------- */
/*****************/
/* GTK2: to test */
/*****************/
gboolean
on_download_clist_button_press_event	(GtkWidget		 *widget,
													 GdkEventButton  *event,
													 gpointer			user_data)
{
	if(event==NULL)
		return TRUE;
	
	if(event->button==3)
	{  /* right-click */
		gtk_menu_popup(GTK_MENU(dl_popup),NULL,NULL,NULL,NULL,event->button,event->time);
		return TRUE;
	}

	return FALSE;
}


/* ------------------------------------------------------------------------------------------------------------- */
/*****************/
/* GTK2: to test */
/*****************/
gboolean
on_download_clist_key_press_event		(GtkWidget		 *widget,
													 GdkEventKey	  *event,
													 gpointer			user_data)
{
	if(event==NULL)
		return FALSE;

	switch(event->keyval)
	{
		case GDK_d: /* 'd' pressed */
		case GDK_D:
						kill_selected_entry("/KILL","download_clist",DLC_ID_COL);
						break;
	}

	return FALSE;
}
/* ------------------------------------------------------------------------------------------------------------- */

/*****************/
/* GTK2: to test */
/*****************/
void
on_clear_error_messages_button_clicked (GtkButton		 *button,
													 gpointer			user_data)
{
	wipe_gtk_text("error_messages_text");
}


void
on_cancel_dl_activate						(GtkMenuItem	  *menuitem,
													 gpointer			user_data)
{
	kill_selected_entry("/KILL","download_clist",DLC_ID_COL);
}


/* ------------------------------------------------------------------------------------------------------------- */
void
on_dl_ul_done_view_file_list_activate  (GtkMenuItem	  *menuitem,
													 gpointer			user_data)
{
	GtkWidget *w;

	w=get_widget_by_widget_name(main_window,"xfer_notebook");
	if(w==NULL)
		return;
		
	switch(gtk_notebook_get_current_page(GTK_NOTEBOOK(w)))
	{
		case XFER_DL_TAB:
						foreach_selected_entry_send_cmd_with_glob_user("download_clist",DLC_USER_POINTER,"/LS");
						break;

		case XFER_UL_TAB:
						foreach_selected_entry_send_cmd_with_glob_user("upload_clist",ULC_USER_POINTER,"/LS");
						break;
	}
}

/* ------------------------------------------------------------------------------------------------------------- */
/*****************************************************************/
/* open chat for one selected entry from a list containing users */
/******************************************************************/
/* GTK 2: ok */
/*************/
static void open_one_private_chat_with_a_selected_user(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	int column=GPOINTER_TO_INT(data);
	char *pchat_label;
	GtkWidget *w;
	int pchat_num;
	GLOB_USER *gu;

	gtk_tree_model_get(model,iter,column,&gu,-1);
	if(gu==NULL)
		return;

	w=get_widget_by_widget_name(main_window,"main_notebook");
	gtk_notebook_set_page(GTK_NOTEBOOK(w),CHAT_TAB);	 /* switch to private chat panel */

	if(get_pchat_text_widget_from_nick(gu,TRUE,&pchat_label,&pchat_num)==NULL)
	{
		GString *out;
 
		out=g_string_new("");

		g_string_sprintfa(out,_("WARNING: all private chats are busy, close unused one.\nUnable to create a private chat for %s.\n"),gu->fmt_nick);
		
		gnome_app_error(GNOME_APP(main_window),out->str);
		g_string_free(out,TRUE);
	}
}

/********************************************************************/
/* scan the given list to open private chat with the selected users */
/********************************************************************/
/* GTK 2: ok */
/*************/
void open_a_private_chat_with_selected_user_in_list(char *clist_name, int clist_column)
{
	GtkWidget *w;
	GtkTreeSelection *slc;

	/* get the nickname of the remote side of the private chat */
	w=get_widget_by_widget_name(main_window,clist_name);
	if(w==NULL)
		return;

	slc=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));
	gtk_tree_selection_selected_foreach(slc,open_one_private_chat_with_a_selected_user,GINT_TO_POINTER(clist_column));
}

/* ------------------------------------------------------------------------------------------------------------- */


void
on_dl_ul_done_open_private_chat_activate
													 (GtkMenuItem	  *menuitem,
													 gpointer			user_data)
{
	GtkWidget *w;

	w=get_widget_by_widget_name(main_window,"xfer_notebook");
	if(w==NULL)
		return;

	switch(gtk_notebook_get_current_page(GTK_NOTEBOOK(w)))
	{
		case XFER_DL_TAB:
						open_a_private_chat_with_selected_user_in_list("download_clist",DLC_USER_POINTER);
						break;

		case XFER_UL_TAB:
						open_a_private_chat_with_selected_user_in_list("upload_clist",ULC_USER_POINTER);
						break;
	}
}

/* ------------------------------------------------------------------------------------------------------------- */
/********************************************************/
/* flag one selected entry from a list containing users */
/********************************************************/
/* GTK 2: to test */
/******************/
static void flag_on_select_user(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	int column=GPOINTER_TO_INT(data);
	GLOB_USER *gu;

	gtk_tree_model_get(model,iter,column,&gu,-1);
	if(gu!=NULL)
		auto_flag_user(gu->unfmt_nick);
}


static void flag_a_user(char *clist_name, int clist_column)
{													
	GtkWidget *w;
	GtkTreeSelection *slc;

	/* get the nickname of the remote side of the private chat */
	w=get_widget_by_widget_name(main_window,clist_name);
	if(w==NULL)
		return;

	slc=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));
	gtk_tree_selection_selected_foreach(slc,flag_on_select_user,GINT_TO_POINTER(clist_column));
}

/*************************************/
/* flag a user of the download clist */
/*************************************/
/* GTK2: to test */
/*****************/
void
on_flag_user2_activate					  (GtkMenuItem	  *menuitem,
													 gpointer			user_data)
{
	flag_a_user("download_clist",DLC_USER_POINTER);
}


/* ------------------------------------------------------------------------------------------------------------- */
/*************************************************/
/* load the locate_user entry and start a search */
/*************************************************/
/* GTK 2: to test */
/******************/
static void input_this_in_locate_user_entry(const char *content)
{
	GtkWidget *w;
	w=get_widget_by_widget_name(main_window,"locate_user_entry");
	if(w==NULL)
		return;

	gtk_entry_set_text(GTK_ENTRY(w),content);
	gtk_button_clicked(GTK_BUTTON(get_widget_by_widget_name(main_window,"search_user_button")));

	/* and switch to the find page */
	on_show_search_user_side_button_clicked(NULL,NULL);
	gtk_notebook_set_page(GTK_NOTEBOOK(get_widget_by_widget_name(main_window,"main_notebook")),FIND_TAB);
}

/********************************************************/
/* flag one selected entry from a list containing users */
/********************************************************/
/* GTK 2: to test */
/******************/
static void search_a_select_user(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	int column=GPOINTER_TO_INT(data);
	GLOB_USER *gu;

	gtk_tree_model_get(model,iter,column,&gu,-1);
	if(gu!=NULL)
		input_this_in_locate_user_entry(gu->unfmt_nick);
}


/**************************************************************/
/* start a search for a user in a given clist, in a given row */
/**************************************************************/
/* GTK2: to test */
/*****************/
static void search_for_selected_user_in_clist(const char *clist_name, int clist_column)
{
	GtkWidget *w;
	GtkTreeSelection *slc;

	/* get the nickname of the remote side of the private chat */
	w=get_widget_by_widget_name(main_window,clist_name);
	if(w==NULL)
		return;

	slc=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));
	gtk_tree_selection_selected_foreach(slc,search_a_select_user,GINT_TO_POINTER(clist_column));
}

/***************************************/
/* search a user of the download clist */
/***************************************/
/* GTK2: to test */
/*****************/
void
on_search_user_of_dl_clist_activate    (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	search_for_selected_user_in_clist("download_clist",DLC_USER_POINTER);
}


/* ------------------------------------------------------------------------------------------------------------- */
/*****************/
/* GTK2: to test */
/*****************/
void
on_cancel_ul_activate                  (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	kill_selected_entry("/KILL","upload_clist",ULC_ID_COL);
}

/* ------------------------------------------------------------------------------------------------------------- */
/*****************/
/* GTK2: to test */
/*****************/
void
on_unqueue_xfer_activate               (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	kill_selected_entry("/KILLKBN","queue_clist",QUC_ID_COL);
}


/*********************************************/
/* /LS of all selected user of the user list */
/*********************************************/
/* GTK2: to test */
/*****************/
void
on_view_file_list_activate             (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	foreach_selected_entry_send_cmd_with_glob_user("user_clist",UCC_LST_USER_POINTER,"/LS");
}

/* ------------------------------------------------------------------------------------------------------------- */
/***********************************************/
/* open a private chat with all selected users */
/***********************************************/
/* GTK 2: ok */
/*************/
void
on_open_private_chat_activate			 (GtkMenuItem	  *menuitem,
                                        gpointer         user_data)
{
	open_a_private_chat_with_selected_user_in_list("user_clist",UCC_LST_USER_POINTER);
}
/* ------------------------------------------------------------------------------------------------------------- */


void
on_kick_user_activate                  (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	foreach_selected_entry_send_cmd_with_glob_user("user_clist",UCC_LST_USER_POINTER,"/KICK");
}

/* ------------------------------------------------------------------------------------------------------------- */
void
on_flag_user_activate						(GtkMenuItem	  *menuitem,
													 gpointer			user_data)
{
	flag_a_user("user_clist",UCC_LST_USER_POINTER);
}

/* ------------------------------------------------------------------------------------------------------------- */
/*****************************/
/* start a search for a user */
/*****************************/
/* GTK2: to test */
/*****************/
void
on_search_user_of_user_clist_activate  (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	search_for_selected_user_in_clist("user_clist",UCC_LST_USER_POINTER);
}

/* ------------------------------------------------------------------------------------------------------------- */
/**************************************************/
/* wake GDL of all selected user of the user list */
/**************************************************/
/* GTK2: to test */
/*****************/
void
on_wake_this_user_gdl_source_on_user_clist_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	foreach_selected_entry_send_cmd_to_gdl_with_glob_user("user_clist",UCC_LST_USER_POINTER,"/WAKEUGDL");
}
/* ------------------------------------------------------------------------------------------------------------- */


void
on_dl_dir_select_ok_button_clicked     (GtkButton       *button,
                                        gpointer         user_data)
{
	const gchar *selected_filename;

	selected_filename=gtk_file_selection_get_filename(GTK_FILE_SELECTION(browse_dl_dir));
	if(selected_filename!=NULL)
	{
		char *path;

		path=strdup(selected_filename);
		if(path!=NULL)
		{
			struct stat st;

			/* verify if the given name is a valid directory */
			if(stat(path,&st)==-1)
			{
				err:
				gnome_app_error(GNOME_APP(main_window),_("invalid dir name"));
			}
			else
			{
				if(!S_ISDIR(st.st_mode))
				{	/* it exists but it is not a directory, remove the filename and retry */
					char *t;
					t=strrchr(path,'/');
					if(t==path)
						goto err;
					t[1]='\0';		/* keep the trailing / */

					if(stat(path,&st)==-1)
						goto err;
					if(!S_ISDIR(st.st_mode))
						goto err;
				}
				printf("'%s'\n",path);
				
				{
					GtkWidget *dl_dir_ent;

					dl_dir_ent=get_widget_by_widget_name(main_window,"dl_dir_entry");
					if(dl_dir_ent!=NULL)
					{
						gtk_entry_set_text(GTK_ENTRY(dl_dir_ent),path);
					}
				}
			}
			free(path);
		}
	}
	gtk_widget_destroy(browse_dl_dir);
	browse_dl_dir=NULL;
}


void
on_dl_dir_cancel_button_clicked        (GtkButton       *button,
                                        gpointer         user_data)
{
	gtk_widget_destroy(browse_dl_dir);
	browse_dl_dir=NULL;
}


void
on_shared_dir_fs_ok_button_clicked     (GtkButton       *button,
                                        gpointer         user_data)
{
	const gchar *selected_filename;

	selected_filename=gtk_file_selection_get_filename(GTK_FILE_SELECTION(add_share_dir_fselect));
	if(selected_filename!=NULL)
	{
		char *path;

		path=strdup(selected_filename);
		if(path!=NULL)
		{
			struct stat st;

			/* verify if the given name is a valid directory */
			if(stat(path,&st)==-1)
			{
				err:
				gnome_app_error(GNOME_APP(main_window),_("invalid dir name"));
			}
			else
			{
				GtkWidget *w;

				if(!S_ISDIR(st.st_mode))
				{	/* it exists but it is not a directory, remove the filename and retry */
					char *t;
					t=strrchr(path,'/');
					if(t==path)
						goto err;
					t[1]='\0';		/* keep the trailing / */

					if(stat(path,&st)==-1)
						goto err;
					if(!S_ISDIR(st.st_mode))
						goto err;
				}
				
				w=get_widget_by_widget_name(main_window,"shared_dir_clist");
				if(w!=NULL)
				{
					GtkListStore *gls;
					GtkTreeIter iter;

					gls=GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(w)));
					gtk_list_store_append(gls,&iter);
					gtk_list_store_set(gls,&iter,SDC_DIRECTORY_COL,path,-1);
				}
			}
			free(path);
		}
	}
}

void
on_shared_dir_fs_cancel_button_clicked (GtkButton       *button,
                                        gpointer         user_data)
{
	gtk_widget_destroy(add_share_dir_fselect);
	add_share_dir_fselect=NULL;
	sync_share_list_with_hidden_entry();
}


/* ------------------------------------------------------------------------------------------------------------- */
/*****************************************************************************/
/* this function add generate a "cmd user" from the given user_file_cl_entry */
/*****************************************************************************/
/* cmd is a GString "" and it must be restored to this at the end */
/******************************************************************/
static void do_dl_from_user_file_list(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer cmd)
{
	int ln;
	GString *dl=cmd;
	GtkTreeIter parent;
	char *nickname;
	char *filename;
	gulong file_size;
	guint entry_type;

	gtk_tree_model_get(model,iter,UFLC_TYPE,&entry_type,-1);
	if(entry_type!=UFL_FILE_ENTRY)
		return;

	if(get_nick_entry_for_user_file_list_from_ufce(model,iter,&parent)==FALSE)
		return;
	
	ln=dl->len;

	gtk_tree_model_get(model,iter,UFLC_STR,&filename,UFLC_SIZE_AS_VAL,&file_size,-1);
	gtk_tree_model_get(model,&parent,UFLC_STR,&nickname,-1);
	g_string_sprintf(dl,"/DL |%s||%s|%lu\n",nickname,filename,file_size);
	free(nickname);
	free(filename);
	send_data_to_gdl_dctc(dl->str);

	g_string_truncate(dl,ln);		/* restore original string lenght */
}

static void start_download_from_user_file_list_ctree(void)
{
	GString *dl;
	dl=g_string_new("");

	generic_selected_user_file_list_calls(do_dl_from_user_file_list,dl);
	g_string_free(dl,TRUE);
}

void
on_download_selected_files1_activate   (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	GtkWidget *w;

	w=get_widget_by_widget_name(main_window,"main_notebook");
	if(w==NULL)
		return;

	switch(gtk_notebook_get_current_page(GTK_NOTEBOOK(w)))
	{
		case FIND_TAB:		/* find */
					start_download_from_find_result_clist();
					break;

		case USER_FILE_LIST_TAB:	/* user file list */
					start_download_from_user_file_list_ctree();
					break;

		default:
					break;
	}
}


/* ------------------------------------------------------------------------------------------------------------- */
void
on_unselect_files1_activate            (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	GtkWidget *w;

	w=get_widget_by_widget_name(main_window,"main_notebook");
	if(w==NULL)
		return;

	switch(gtk_notebook_get_current_page(GTK_NOTEBOOK(w)))
	{
		case FIND_TAB:		/* find */
					w=get_widget_by_widget_name(main_window,"find_result");
					break;

		case USER_FILE_LIST_TAB:	/* user file list */
					w=get_widget_by_widget_name(main_window,"user_file_list_clist");
					break;

		default:
					w=NULL;
					break;
	}

	if(w!=NULL)
	{
		GtkTreeSelection *slc;

		slc=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));
		gtk_tree_selection_unselect_all(slc);
	}
}

/* ------------------------------------------------------------------------------------------------------------- */
void get_user_file_list_from_user_file_list(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	GString *str=data;
	guint type;
	char *nickname;

	gtk_tree_model_get(model,iter,UFLC_TYPE,&type,-1);
	if(type!=UFL_NICK_ROOT_ENTRY)
		return;
	
	gtk_tree_model_get(model,iter,UFLC_STR,&nickname,-1);
	g_string_sprintf(str,"/LS %s\n",nickname);
	send_data_to_gdl_dctc(str->str);
	free(nickname);
}


void
on_get_file_list_of_selected_users1_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	GtkWidget *w;

	w=get_widget_by_widget_name(main_window,"main_notebook");
	if(w==NULL)
		return;

	switch(gtk_notebook_get_current_page(GTK_NOTEBOOK(w)))
	{
		case FIND_TAB:		/* find */
					foreach_selected_entry_send_cmd_with_glob_user("find_result",FRC_USER_POINTER,"/LS");
					return;

		case USER_FILE_LIST_TAB:	/* user file list */
					{
						GString *str;

						str=g_string_new("");
						generic_selected_user_file_list_calls(get_user_file_list_from_user_file_list,str);
						g_string_free(str,TRUE);
					}
					return;

		default:
					break;
	}
}

/* ------------------------------------------------------------------------------------------------------------- */
void flag_from_user_file_list(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	guint type;
	char *nickname;

	gtk_tree_model_get(model,iter,UFLC_TYPE,&type,-1);
	if(type!=UFL_NICK_ROOT_ENTRY)
		return;
	
	gtk_tree_model_get(model,iter,UFLC_STR,&nickname,-1);
	auto_flag_user(nickname);
	free(nickname);
}


void
on_flag_user3_activate                 (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	GtkWidget *w;

	w=get_widget_by_widget_name(main_window,"main_notebook");
	if(w==NULL)
		return;

	switch(gtk_notebook_get_current_page(GTK_NOTEBOOK(w)))
	{
		case FIND_TAB:		/* find */
					flag_a_user("find_result",FRC_USER_POINTER);
					return;

		case USER_FILE_LIST_TAB:	/* user file list */
					generic_selected_user_file_list_calls(flag_from_user_file_list,NULL);
					return;

		default:
					break;
	}
}


/* ------------------------------------------------------------------------------------------------------------- */
void wake_user_from_user_file_list(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	GString *str=data;
	guint type;
	char *nickname;

	gtk_tree_model_get(model,iter,UFLC_TYPE,&type,-1);
	if(type!=UFL_NICK_ROOT_ENTRY)
		return;
	
	gtk_tree_model_get(model,iter,UFLC_STR,&nickname,-1);
	g_string_sprintf(str,"/WAKEUGDL %s\n",nickname);
	send_data_to_gdl_dctc(str->str);
	free(nickname);
}

void
on_wake_this_user_gdl_source_of_start_dl_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	GtkWidget *w;

	w=get_widget_by_widget_name(main_window,"main_notebook");
	if(w==NULL)
		return;

	switch(gtk_notebook_get_current_page(GTK_NOTEBOOK(w)))
	{
		case FIND_TAB:		/* find */
					foreach_selected_entry_send_cmd_to_gdl_with_glob_user("find_result",FRC_USER_POINTER,"/WAKEUGDL");
					return;

		case USER_FILE_LIST_TAB:	/* user file list */
					{
						GString *str;

						str=g_string_new("");
						generic_selected_user_file_list_calls(wake_user_from_user_file_list,str);
						g_string_free(str,TRUE);
					}
					return;

		default:
					break;
	}
}


/* ------------------------------------------------------------------------------------------------------------- */
void
on_contact_selected_running_client1_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{

}


void
on_kill_selected_unning_clients1_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{

}

/* ------------------------------------------------------------------------------------------------------------- */
/*****************/
/* GTK2: to test */
/*****************/
void
on_stop_and_detach1_activate           (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	detach_selected_gdl_entry();
}

/* ------------------------------------------------------------------------------------------------------------- */
/*****************/
/* GTK2: to test */
/*****************/
void
on_cancel_gdl_activate                 (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	kill_selected_gdl_entry();
}

/* ------------------------------------------------------------------------------------------------------------- */
/*****************/
/* GTK2: to test */
/*****************/
void
on_attach_autoscan_pattern_to_selected_gdl1_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	create_autoscan_window_for_selected_gdl_entry();
}

/* ------------------------------------------------------------------------------------------------------------- */
typedef struct
{
	const char *pattern;
	int pattern_length;
} FORBIDDEN_PATTERN;

static FORBIDDEN_PATTERN fpat[]={
										{"sharereactor",sizeof("sharereactor")-1},
										{"divx",sizeof("divx")-1},
										{"xvid",sizeof("xvid")-1},
										{NULL,0}
									};

/******************************************************************************************/
/* test if part (at offset 0) is a forbidden pattern                                      */
/* if yes, TRUE is returned and *fnd_pattern_len is the length of the encountered pattern */
/******************************************************************************************/
static gboolean is_an_invalid_pattern(const char *part, int *fnd_pattern_len)
{
	int i;

	i=0;

	while(fpat[i].pattern!=NULL)
	{
		if( (!strncasecmp(part,fpat[i].pattern,fpat[i].pattern_length)) &&
				( (part[fpat[i].pattern_length]=='$') || (part[fpat[i].pattern_length]=='\0') ) )
		{
			*fnd_pattern_len=fpat[i].pattern_length;
			return TRUE;
		}

		i++;
	}
	return FALSE;
}

/* search and discard some known parts of the pattern that will prevent successful search */
static void filter_autoscan(GString *str)
{
	int i;

	i=0;
	while(i<str->len)
	{
		int len;

		if(is_an_invalid_pattern(str->str+i,&len))
		{
			if(str->str[i+len]=='$')
				len++;
			g_string_erase(str,i,len);
		}
		else
		{
			char *t;

			t=strchr(str->str+i,'$');
			if(t==NULL)
				i=str->len;
			else
				i=i+ (t-(str->str+i)) +1;		/* +1 to skip the $ */
		}
	}
}

/* ------------------------------------------------------------------------------------------------------------- */
static void generate_and_assign_as_pattern_from_filename_and_gdl_id(gulong gdl_id, const char *local_filename)
{
	char *t;
	GString *str;

	str=g_string_new(local_filename);

	/* remove everything before the last '/' (and the '/' itself) */
	t=strrchr(str->str,'/');
	if(t!=NULL)
	{
		g_string_erase(str,0,(t+1)-str->str);
	}

	if(str->len>10)
	{
		int i;
		int nb=0;

		/* replace non alpha-num character by '$' */
		for(i=0;i<str->len;i++)
		{
			if(!isalnum(str->str[i]))
			{
				str->str[i]='$';
				nb++;
			}
		}

		/* replace multiple consecutive '$' by one '$' */
		i=0;
		while(i<(str->len-1))
		{
			if((str->str[i]=='$') && (str->str[i+1]=='$'))
			{
				g_string_erase(str,i,1);
				nb--;
			}
			else
				i++;
		}

		filter_autoscan(str);

		if((str->len/nb)>2)	/* the pattern should not at least look like "$$$$$$" */
		{
			GString *cmd;

			cmd=g_string_new("");
			g_string_sprintf(cmd,"/GDLAS+ %lu|%d|%s\n",gdl_id,1,str->str);
			send_data_to_gdl_dctc(cmd->str);
			g_string_free(cmd,TRUE);
		}
	}
	g_string_free(str,TRUE);
}

/*****************/
/* GTK2: to test */
/*****************/
static void generate_as_pattern_from_root_name(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	char *local_filename;
	gulong gdl_id;

	gtk_tree_model_get(model,iter,GCC_ID,&gdl_id,GCC_STR1,&local_filename,-1);
	generate_and_assign_as_pattern_from_filename_and_gdl_id(gdl_id, local_filename);
	free(local_filename);
}

/*****************/
/* GTK2: to test */
/*****************/
void
on_generate_autoscan_pattern_for_selected_gdl1_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	static void (*fnc[GDL_NUMBER_OF_GDL_CT_TYPE])={generate_as_pattern_from_root_name,NULL,NULL,NULL,NULL,NULL};
	
	generic_selected_gdl_entry_list_calls((void*)fnc,NULL);
}

/* ------------------------------------------------------------------------------------------------------------- */
/*****************/
/* GTK2: to test */
/*****************/
void
on_rename_selected_gdl_activate		  (GtkMenuItem	  *menuitem,
													 gpointer			user_data)
{
	create_rename_window_for_selected_gdl_entry();
}

/* ------------------------------------------------------------------------------------------------------------- */
/*****************/
/* GTK2: to test */
/*****************/
void
on_script_selected_gdl_activate		  (GtkMenuItem	  *menuitem,
													 gpointer			user_data)
{
	create_script_window_for_selected_gdl_entry();
}

/* ------------------------------------------------------------------------------------------------------------- */
/********************************************************/
/* send a command (with nickname) from a GDL active row */
/********************************************************/
/* GTK2: to test */
/*****************/
static void gdl_ctree_send_cmd_from_active(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	gchar *cmd;
	GLOB_USER *gu;

	gtk_tree_model_get(model,iter,GCC_MISC_POINTER,&gu,-1);
	cmd=g_strconcat((char*)data,gu->unfmt_nick,"\n",NULL);
	send_data_to_gdl_dctc(cmd);
	g_free(cmd);
}

/* ------------------------------------------------------------------------------------------------------------- */
/*****************/
/* GTK2: to test */
/*****************/
void
on_gdl_view_file_list_activate			(GtkMenuItem	  *menuitem,
													 gpointer			user_data)
{
	static void (*fnc[GDL_NUMBER_OF_GDL_CT_TYPE])={NULL,NULL,gdl_ctree_send_cmd_from_active,NULL,NULL,NULL};
	generic_selected_gdl_entry_list_calls((void*)fnc,"/LS ");
}

/* ------------------------------------------------------------------------------------------------------------- */
/*********************************************************/
/* call a function (with nickname) from a GDL active row */
/*********************************************************/
/* GTK2: to test */
/*****************/
static void gdl_ctree_call_fnc_from_active(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	gchar *nickname;
	void (*fnc)(const char *)=data;
	GLOB_USER *gu;

	gtk_tree_model_get(model,iter,GCC_MISC_POINTER,&gu,-1);

	nickname=g_strdup(gu->unfmt_nick);
	(*fnc)(nickname);
	g_free(nickname);
}

/*****************/
/* GTK2: to test */
/*****************/
void
on_flag_user_from_gdl_activate			(GtkMenuItem	  *menuitem,
													 gpointer			user_data)
{
	static void (*fnc[GDL_NUMBER_OF_GDL_CT_TYPE])={NULL,NULL,gdl_ctree_call_fnc_from_active,NULL,NULL,NULL};
	generic_selected_gdl_entry_list_calls((void*)fnc,auto_flag_user);
}

/* ------------------------------------------------------------------------------------------------------------- */
/********************************************************/
/* flag one selected entry from a list containing users */
/********************************************************/
/* GTK2: to test */
/*****************/
void
on_search_user_of_gdl_clist_activate	(GtkMenuItem	  *menuitem,
													 gpointer			user_data)
{
	static void (*fnc[GDL_NUMBER_OF_GDL_CT_TYPE])={NULL,NULL,gdl_ctree_call_fnc_from_active,NULL,NULL,NULL};
	generic_selected_gdl_entry_list_calls((void*)fnc,input_this_in_locate_user_entry);
}

/* ------------------------------------------------------------------------------------------------------------- */
/****************************************************/
/* wake user source of the selected GDL source rows */
/****************************************************/
/*****************/
/* GTK2: to test */
/*****************/
void
on_wake_this_user_gdl_source_on_gdl_clist_activate
													 (GtkMenuItem	  *menuitem,
													 gpointer			user_data)
{
	static void (*fnc[GDL_NUMBER_OF_GDL_CT_TYPE])={NULL,NULL,gdl_ctree_send_cmd_from_active,NULL,NULL,NULL};
	generic_selected_gdl_entry_list_calls((void*)fnc,"/WAKEUGDL ");
}

/* ------------------------------------------------------------------------------------------------------------- */
/*****************/
/* GTK2: to test */
/*****************/
void
on_expand_all_gdl_activate				 (GtkMenuItem	  *menuitem,
													 gpointer			user_data)
{
	GtkWidget *w;
	w=get_widget_by_widget_name(main_window,"gdl_ctree");
	if(w!=NULL)
		gtk_tree_view_expand_all(GTK_TREE_VIEW(w));
}

/* ------------------------------------------------------------------------------------------------------------- */
/*****************/
/* GTK2: to test */
/*****************/
void
on_shrink_all_gdl_activate				 (GtkMenuItem	  *menuitem,
													 gpointer			user_data)
{
	GtkWidget *w;
	w=get_widget_by_widget_name(main_window,"gdl_ctree");
	if(w!=NULL)
		gtk_tree_view_collapse_all(GTK_TREE_VIEW(w));
}

/* ------------------------------------------------------------------------------------------------------------- */

static void try_to_attach_this_gdl(const char *dir1, const char *dir2)
{
	GString *str;
	int fd;

	str=g_string_new("");
	g_string_sprintf(str,"%s/%s/.lock",dir1,dir2);

	fd=open(str->str,O_RDWR);
	if(fd!=-1)
	{
		if(lockf(fd,F_TEST,1)==0)
		{
			/* .lock file exists and is lockable, so nobody uses this entry */

			/* create and send the GDL attachment query */
			g_string_sprintf(str,"/GDLATTACH %s\n",dir2);
			send_data_to_gdl_dctc(str->str);
		}
		close(fd);
	}
	g_string_free(str,TRUE);
}

/*****************/
/* GTK2: to test */
/*****************/
void
on_attach_all_unattached_gdl_activate  (GtkMenuItem	  *menuitem,
													 gpointer			user_data)
{
	DIR *dir; 
	struct dirent *obj;
	const char *t;
	
	t=get_var("dl_path");
	if((t!=NULL)&&(strlen(t)))
	{
		GString *st1;

		st1=g_string_new(t);
		st1=g_string_append(st1,"/GDL");
		dir=opendir(st1->str);
		{
			while((obj=readdir(dir))!=NULL)
			{
				try_to_attach_this_gdl(st1->str,obj->d_name);
			}
			closedir(dir);			
		}				  
		g_string_free(st1,TRUE);
	}
}

/* ------------------------------------------------------------------------------------------------------------- */
/*************************************************/
/* /LS on one selected entry from the UADDR list */
/*************************************************/
/* GTK 2: to test */
/******************/
static void ls_one_selected_uaddr_entry(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	char *nickname;
	GString *tmp_str=data;

	gtk_tree_model_get(model,iter,UAC_NICK_COL,&nickname,-1);
	if(strlen(nickname)!=0)
	{
		g_string_sprintf(tmp_str,"/LS %s\n",nickname);
		send_data_to_gdl_dctc(tmp_str->str);
	}

	free(nickname);
}


/*************************************/
/* /LS on all selected UADDR entries */
/*************************************/
/* GTK2: to test */
/*****************/
void
on_uaddr_view_file_list_activate       (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	GtkWidget *w;
	GtkTreeSelection *slc;
	GString *tmp_str;

	w=get_widget_by_widget_name(main_window,"uaddr_clist");
	if(w==NULL)
		return;

	slc=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));
	tmp_str=g_string_new("");
	gtk_tree_selection_selected_foreach(slc,ls_one_selected_uaddr_entry,tmp_str);
	g_string_free(tmp_str,TRUE);
	send_data_to_dctc("/UADDRLST\n");
}

/* ------------------------------------------------------------------------------------------------------------- */
/*************************************************/
/* delete one selected entry from the UADDR list */
/*************************************************/
/* GTK 2: to test */
/******************/
static void delete_one_selected_uaddr_entry(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
	char *nickname;
	char *ipport;
	GString *tmp_str=data;

	gtk_tree_model_get(model,iter,UAC_NICK_COL,&nickname,UAC_ADDR_PORT_COL,&ipport,-1);
	if(strlen(nickname)==0)
	{
		g_string_sprintf(tmp_str,"/UADDRIPDEL %s\n",ipport);
	}
	else
	{
		g_string_sprintf(tmp_str,"/UADDRUDEL %s\n",nickname);
	}

	send_data_to_dctc(tmp_str->str);
	free(nickname);
	free(ipport);
}

/*************************************/
/* delete all selected UADDR entries */
/*************************************/
/* GTK2: to test */
/*****************/
void
on_delete_uaddr_entries_activate		 (GtkMenuItem	  *menuitem,
													 gpointer			user_data)
{
	GString *tmp_str;

	tmp_str=g_string_new("");
	generic_select_uaddr_entry_list_calls(delete_one_selected_uaddr_entry,tmp_str);
	g_string_free(tmp_str,TRUE);

	send_data_to_dctc("/UADDRLST\n");
}


/* ------------------------------------------------------------------------------------------------------------- */
/***************************************/
/* ask for a refresh of the UADDR list */
/***************************************/
/* GTK2: to test */
/*****************/
void
on_refresh_uaddr_list_activate			(GtkMenuItem	  *menuitem,
													 gpointer			user_data)
{
	send_data_to_dctc("/UADDRLST\n");
}

/* ------------------------------------------------------------------------------------------------------------- */
/******************************/
/* unselect all UADDR entries */
/******************************/
/* GTK2: to test */
/*****************/
void
on_unselect_uaddr_activate				 (GtkMenuItem	  *menuitem,
													 gpointer			user_data)
{
	GtkWidget *w;
	GtkTreeSelection *slc;

	w=get_widget_by_widget_name(main_window,"uaddr_clist");
	if(w==NULL)
		return;
	slc=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));
	gtk_tree_selection_unselect_all(slc);
}

/* ------------------------------------------------------------------------------------------------------------- */

void
on_vshare_dir_select_ok_button_clicked (GtkButton		 *button,
													 gpointer			user_data)
{
	const gchar *selected_filename;

	selected_filename=gtk_file_selection_get_filename(GTK_FILE_SELECTION(vshare_dir_fselect));
	if(selected_filename!=NULL)
	{
		char *path;

		path=strdup(selected_filename);
		if(path!=NULL)
		{
			struct stat st;

			/* verify if the given name is a valid directory */
			if(stat(path,&st)==-1)
			{
				err:
				gnome_app_error(GNOME_APP(main_window),_("invalid dir name"));
			}
			else
			{
				if(!S_ISDIR(st.st_mode))
				{	/* it exists but it is not a directory, remove the filename and retry */
					char *t;
					t=strrchr(path,'/');
					if(t==path)
						goto err;
					t[1]='\0';		/* keep the trailing / */

					if(stat(path,&st)==-1)
						goto err;
					if(!S_ISDIR(st.st_mode))
						goto err;
				}
				printf("'%s'\n",path);
				
				{
					GtkWidget *dl_dir_ent;

					dl_dir_ent=get_widget_by_widget_name(main_window,"vshare_dir_entry");
					if(dl_dir_ent!=NULL)
					{
						gtk_entry_set_text(GTK_ENTRY(dl_dir_ent),path);
					}
				}
			}
			free(path);
		}
	}
	gtk_widget_destroy(vshare_dir_fselect);
	vshare_dir_fselect=NULL;
}


void
on_vshare_dir_cancel_button_clicked	 (GtkButton		 *button,
													 gpointer			user_data)
{
	gtk_widget_destroy(vshare_dir_fselect);
	vshare_dir_fselect=NULL;
}



gboolean
on_running_hub_clist_button_press_event
													 (GtkWidget		 *widget,
													 GdkEventButton  *event,
													 gpointer			user_data)
{
	if(event==NULL)
		return TRUE;

	if((event->button==1)&&(event->type==GDK_2BUTTON_PRESS))
	{  /* left click */
		/* double click on a row */
		GtkTreePath *path;

		if(gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(widget),(gint) event->x, (gint) event->y, &path,NULL,NULL,NULL)==TRUE)
		{
			char *col;
			GtkTreeIter iter;
			GtkTreeModel *gtm;
	
			gtk_tree_model_get_iter (gtm=gtk_tree_view_get_model(GTK_TREE_VIEW(widget)),&iter,path);
			gtk_tree_model_get(gtm,&iter,RHC_PROCESS_ID_COL,&col,-1);
			connect_to_a_running_dctc(col);
			free(col);
			gtk_tree_path_free(path);
		}
		return TRUE;
	}
	return FALSE;
}


/************************************************************/
/* check if the given flag list contains the following flag */
/************************************************************/
/* output: 1= yes, the flag is here */
/*         0= no                    */
/************************************/
int flag_in_flag_list(char *data_val,const char *flag)
{
	int data_len;
	int ret;
	char *pos;
	int flag_len;

	ret=0;
	data_len=strlen(data_val);
	flag_len=strlen(flag);
	pos=data_val;

	while(flag_len<data_len)
	{
		if(!strncmp(flag,pos,flag_len))
		{  /* the beginning of flag match */
			if(pos[flag_len]=='|')	  /* and the current flag immediatly ends with a pipe */
			{
				ret=1;
				break;
			}
		}
		pos++;
		data_len--;
	}

	/* check the last flag if not yet found, last chance */
	if((ret==0)&&(flag_len==data_len))
	{
		if(!strncmp(flag,pos,flag_len))
			ret=1;
	}

	return ret;
}


void
on_flagged_user_clist_row_activated    (GtkTreeView     *treeview,
                                        GtkTreePath     *path,
                                        GtkTreeViewColumn *column,
                                        gpointer         user_data)
{
	GtkWidget *w;
	w=get_widget_by_widget_name(main_window,"flagged_user_label");
	if(w)
	{
		GtkTreeModel *gtm;
		GtkTreeIter iter;

		gtm=gtk_tree_view_get_model(treeview);

		if(gtk_tree_model_get_iter(gtm,&iter,path))
		{
			char *flag_list;
			GLOB_USER *gu;

			int i=0;

			gtk_tree_model_get(gtm,&iter,FUC_USER_POINTER,&gu,FUC_FLAGS_COL,&flag_list,-1);

			gtk_label_set(GTK_LABEL(w),gu->fmt_nick);

			while(opt_list[i].widget_name!=NULL)
			{
				w=get_widget_by_widget_name(main_window,opt_list[i].widget_name);
				if(w!=NULL)
				{
					if(flag_in_flag_list(flag_list,opt_list[i].option_name))
						gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w),TRUE);
					else
						gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w),FALSE);
				}
				i++;
			}

			free(flag_list);
		}
	}
}


void
on_utf8_mode_checkbutton_toggled       (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	utf8_mode=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(togglebutton));
}

/*********************************************/
/* replace %XX by a byte having the value XX */
/*********************************************/
static void decode_encoded_name(GString *fname)
{
	int i;

	i=0;
	while(i<(fname->len-3))
	{
		if((fname->str[i]=='%')&&(isxdigit(fname->str[i+1]))&&(isxdigit(fname->str[i+2])))
		{
			fname->str[i]=ascii2_to_bin1(fname->str+i+1);		/* replace the % by the value of its 2 following bytes */

			g_string_erase(fname,i+1,2);		/* and discard the 2 following bytes */
			i++;
		}
		else
		{
#define HTML_CONV(html_code,real_code)			{html_code,sizeof(html_code)-1,real_code,sizeof(real_code)-1}
			int j;
			int skip=1;
			struct
			{
				const char *html_pattern;
				int len_html_pattern;
				const char *real_pattern;
				int len_real_pattern;
			}	html_escape[]={
									HTML_CONV("&amp;","&"),
									HTML_CONV("&lt;","<"),
									HTML_CONV("&gt;",">"),
									HTML_CONV("&quot;","\""),
									{NULL,0,NULL,0}};
			j=0;
			while(html_escape[j].html_pattern!=NULL)
			{
				if(!strncmp(fname->str+i,html_escape[j].html_pattern,html_escape[j].len_html_pattern))
				{
					g_string_erase(fname,i,html_escape[j].len_html_pattern);
					g_string_insert(fname,i,html_escape[j].real_pattern);
					skip=html_escape[j].len_real_pattern;
					break;
				}
				j++;
			}
			i+=skip;
		}
	}
}

/******************************************************/
/* convert a string containing an ed2k URL into a GDL */
/******************************************************/
void process_ed2k_file_url(const char *url)
{
	gchar **fields;

	fields=g_strsplit(url,"|",0);
	if(size_of_null_array((void**)fields)!=6)
	{
		gnome_app_error(GNOME_APP(main_window),_("invalid URL, its format must be 'edk2://|file|filename|filesize|CRC|'"));
	}
	else
	{
		GString *cmd;
		/* 
			fields[0]="ed2k://"
			fields[1]="file"
			fields[2]=the name of the file
			fields[3]=the size of the file
			fields[4]=the CRC of the file
			fields[5]=""
		*/

		if(base_gdl_id==0)
			base_gdl_id=time(NULL);

		cmd=g_string_new("");

		/* create a new GDL */
		{
			GString *decoded_fname;

			decoded_fname=g_string_new(fields[2]);
			decode_encoded_name(decoded_fname);
			g_string_sprintf(cmd,"/GDLNEW %lu|%s|%s\n",base_gdl_id,fields[3],decoded_fname->str);
			send_data_to_gdl_dctc(cmd->str);

			g_string_free(decoded_fname,TRUE);
		}

		/* assign it a CRC */
		g_string_sprintf(cmd,"/GDLCRC %lu|%s\n",base_gdl_id,fields[4]);
		send_data_to_gdl_dctc(cmd->str);

		generate_and_assign_as_pattern_from_filename_and_gdl_id(base_gdl_id,fields[2]);
		g_string_free(cmd,TRUE);

		base_gdl_id+=12345678;    /* to limit the risk of having 2 GDL with the same number, should be safe enough */
	}
	g_strfreev(fields);
}

void
on_ed2k_url_entry_activate             (GtkEntry        *entry,
                                        gpointer         user_data)
{
	const char *url;

	url=gtk_entry_get_text(entry);

	if((url!=NULL)&&(strlen(url)))
	{
		if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(get_widget_by_widget_name(main_window,"ed2k_url_radiobutton")))==TRUE)
		{
			/* the given URL is an ed2k */
			if(strncmp(url,"ed2k://|file|",strlen("ed2k://|file|")))
			{
				gnome_app_error(GNOME_APP(main_window),_("invalid URL, it must start with 'edk2://|file|'"));
			}
			else
			{
				process_ed2k_file_url(url);
			}
		}
		else
		{
			/* the given URL is a bittorrent URL */
			char *encoded_url;

			if(strpbrk(url," []")!=NULL)
				encoded_url=gnome_vfs_escape_host_and_path_string(url);
			else
				encoded_url=strdup(url);

#ifndef WITH_BITTORRENT
			{
				gboolean flag;
				GError *err;
		
				flag=gconf_client_get_bool(engine,"/apps/" PROGNAME "/Flags/NoBittorrentWarning",&err);
				if(err!=NULL)
				{
					flag=FALSE;
					g_error_free(err);
				}

				if(flag==FALSE)
				{
#ifndef WITH_PYTHON
					gnome_app_error(GNOME_APP(main_window),_("Warning: bittorrent and Python were not available when this application was compiled.\n"
																  		"If they have been installed in an unusual place, starting a bittorrent may fail."));
#else
					gnome_app_error(GNOME_APP(main_window),_("Warning: bittorrent was not available when this application was compiled.\n"
																  		"If it has been installed in an unusual place, starting a bittorrent may fail."));
#endif
					gconf_client_set_bool(engine,"/apps/" PROGNAME "/Flags/NoBittorrentWarning",TRUE,NULL);
				}
			}
#endif
			process_bittorrent_url(encoded_url);
			free(encoded_url);
		}
	}
}


void
on_create_gdl_from_ed2k_url_button_activate
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *w;

	w=get_widget_by_widget_name(main_window,"ed2k_url_entry");
	if(w!=NULL)
	{
		on_ed2k_url_entry_activate(GTK_ENTRY(w),NULL);
	}
}


void
on_close_connection_with_the_client1_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
   if(current_dctc!=NULL)
   {  /* close the connection to the current dctc if exists */
      close_dctc_com(&current_dctc);
   }
}

static GtkWidget *profile_wizard_window=NULL;

void
on_profile_wizard1_activate            (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	on_close_connection_with_the_client1_activate(NULL,NULL);

	if(profile_wizard_window==NULL)
	{
		profile_wizard_window=create_profile_wizard_window();
		gtk_widget_show(profile_wizard_window);
	}
}


void
on_profile_druid_cancel                (GnomeDruid      *gnomedruid,
                                        gpointer         user_data)
{
	gtk_widget_destroy(profile_wizard_window);
	profile_wizard_window=NULL;
}


gboolean
on_druidpageid_next                    (GnomeDruidPage  *gnomedruidpage,
                                        GtkWidget       *widget,
                                        gpointer         user_data)
{
	const char *str;

	str=gtk_entry_get_text(GTK_ENTRY(get_widget_by_widget_name(profile_wizard_window,"druid_nickname_entry")));

	/* check the nickname */
	if((str==NULL)||(strlen(str)==0))
	{
		gnome_app_error(GNOME_APP(main_window),_("You must enter a nickname."));
		gnome_druid_set_page(GNOME_DRUID(get_widget_by_widget_name(profile_wizard_window,"profile_druid")),gnomedruidpage);
		return TRUE;		/* we have set the druid page */
	}

	if((strchr(str,'$')!=NULL)||(strchr(str,'|')!=NULL)||(strchr(str,' ')!=NULL))
	{
		gnome_app_error(GNOME_APP(main_window),_("'$' (dollar), ' ' (space) and '|' (pipe) are forbidden in the nickname."));
		gnome_druid_set_page(GNOME_DRUID(get_widget_by_widget_name(profile_wizard_window,"profile_druid")),gnomedruidpage);
		return TRUE;		/* we have set the druid page */
	}

	/* check the e-mail */
	str=gtk_entry_get_text(GTK_ENTRY(get_widget_by_widget_name(profile_wizard_window,"druid_e_mail_entry")));
	if((strchr(str,'$')!=NULL)||(strchr(str,'|')!=NULL)||(strchr(str,' ')!=NULL))
	{
		gnome_app_error(GNOME_APP(main_window),_("'$' (dollar), ' ' (space) and '|' (pipe) are forbidden in the e-mail."));
		gnome_druid_set_page(GNOME_DRUID(get_widget_by_widget_name(profile_wizard_window,"profile_druid")),gnomedruidpage);
		return TRUE;		/* we have set the druid page */
	}

	/* check the description */
	str=gtk_entry_get_text(GTK_ENTRY(get_widget_by_widget_name(profile_wizard_window,"druid_user_description_entry")));
	if((strchr(str,'$')!=NULL)||(strchr(str,'|')!=NULL))
	{
		gnome_app_error(GNOME_APP(main_window),_("'$' (dollar) and '|' are forbidden in the description."));
		gnome_druid_set_page(GNOME_DRUID(get_widget_by_widget_name(profile_wizard_window,"profile_druid")),gnomedruidpage);
		return TRUE;		/* we have set the druid page */
	}

	return FALSE;
}


gboolean
on_druidpage_share_next                (GnomeDruidPage  *gnomedruidpage,
                                        GtkWidget       *widget,
                                        gpointer         user_data)
{
	const char *str;
	struct stat st;

	str=gtk_entry_get_text(GTK_ENTRY(get_widget_by_widget_name(profile_wizard_window,"druid_dl_dir_entry")));

	/* check the download directory */
	if((str==NULL)||(strlen(str)==0))
	{
		gnome_app_error(GNOME_APP(main_window),_("You must have a download directory."));
		gnome_druid_set_page(GNOME_DRUID(get_widget_by_widget_name(profile_wizard_window,"profile_druid")),gnomedruidpage);
		return TRUE;		/* we have set the druid page */
	}

	if(stat(str,&st))
	{
		gnome_app_error(GNOME_APP(main_window),_("Your download directory must exist."));
		gnome_druid_set_page(GNOME_DRUID(get_widget_by_widget_name(profile_wizard_window,"profile_druid")),gnomedruidpage);
		return TRUE;		/* we have set the druid page */
	}

	if(!S_ISDIR(st.st_mode))
	{
		gnome_app_error(GNOME_APP(main_window),_("The path you gave as download directory is not a directory."));
		gnome_druid_set_page(GNOME_DRUID(get_widget_by_widget_name(profile_wizard_window,"profile_druid")),gnomedruidpage);
		return TRUE;		/* we have set the druid page */
	}

	str=gtk_entry_get_text(GTK_ENTRY(get_widget_by_widget_name(profile_wizard_window,"druid_hidden_shared_dir_entry")));

	/* check the download directory */
	if((str==NULL)||(strlen(str)==0))
	{
		gnome_app_error(GNOME_APP(main_window),_("You must have a shared directory."));
		gnome_druid_set_page(GNOME_DRUID(get_widget_by_widget_name(profile_wizard_window,"profile_druid")),gnomedruidpage);
		return TRUE;		/* we have set the druid page */
	}

	if(stat(str,&st))
	{
		gnome_app_error(GNOME_APP(main_window),_("Your shared directory must exist."));
		gnome_druid_set_page(GNOME_DRUID(get_widget_by_widget_name(profile_wizard_window,"profile_druid")),gnomedruidpage);
		return TRUE;		/* we have set the druid page */
	}

	if(!S_ISDIR(st.st_mode))
	{
		gnome_app_error(GNOME_APP(main_window),_("The path you gave as shared directory is not a directory."));
		gnome_druid_set_page(GNOME_DRUID(get_widget_by_widget_name(profile_wizard_window,"profile_druid")),gnomedruidpage);
		return TRUE;		/* we have set the druid page */
	}

	return FALSE;
}


gboolean
on_druidpage_profname_next             (GnomeDruidPage  *gnomedruidpage,
                                        GtkWidget       *widget,
                                        gpointer         user_data)
{
	const char *str;
	int i;

	str=gtk_entry_get_text(GTK_ENTRY(get_widget_by_widget_name(profile_wizard_window,"druid_profile_name_entry")));

	/* check the nickname */
	if((str==NULL)||(strlen(str)==0))
	{
		gnome_app_error(GNOME_APP(main_window),_("You must enter a profile name."));
		gnome_druid_set_page(GNOME_DRUID(get_widget_by_widget_name(profile_wizard_window,"profile_druid")),gnomedruidpage);
		return TRUE;		/* we have set the druid page */
	}

	i=0;
	while(str[i]!='\0')
	{
		if(!isalnum(str[i]))
		{
			gnome_app_error(GNOME_APP(main_window),_("The profile name must only contain alphanumeric characters."));
			gnome_druid_set_page(GNOME_DRUID(get_widget_by_widget_name(profile_wizard_window,"profile_druid")),gnomedruidpage);
			return TRUE;		/* we have set the druid page */
		}
		i++;
	}

	return FALSE;
}

void
on_druidpagefinish1_finish             (GnomeDruidPage  *gnomedruidpage,
                                        GtkWidget       *widget,
                                        gpointer         user_data)
{
	const char *str;
	int i;
	static const char *druid2prof[][2]={
									{"druid_nickname_entry","nickname_entry"},
									{"druid_e_mail_entry","e_mail_entry"},
									{"druid_user_description_entry","user_description_entry"},
									{"druid_cnx_type_entry","cnx_type_entry"},
									{"druid_dl_dir_entry","dl_dir_entry"},
									{"druid_hidden_shared_dir_entry","hidden_shared_dir_entry"},
									{"druid_profile_name_entry","profile_name_combo_entry"},
									{NULL,NULL}};

	/* we have reached the final step. We must set all profile flags to their default values */
	/* we must copy all entered values to their final place and we must save the profile */
	load_default_profile_parameters();

	/* copy the text entries */
	i=0;
	while(druid2prof[i][0]!=NULL)
	{
		str=gtk_entry_get_text(GTK_ENTRY(get_widget_by_widget_name(profile_wizard_window,druid2prof[i][0])));
		gtk_entry_set_text(GTK_ENTRY(get_widget_by_widget_name(main_window,druid2prof[i][1])),str);
		i++;
	}

	sync_hidden_entry_with_share_list();

	/* don't forget to set connection mode (active/passive) */
	{
		GString *host_ip=get_default_host_ip();
		if(host_ip!=NULL)
		{
			unsigned int v1,v2,v3,v4;

			if(sscanf(host_ip->str,"%u.%u.%u.%u",&v1,&v2,&v3,&v4)!=4)
				goto unknown_connection_mode;

			/* RFC1918: private address space:
					10.0.0.0        -   10.255.255.255  (10/8 prefix)
     				172.16.0.0      -   172.31.255.255  (172.16/12 prefix)
     				192.168.0.0     -   192.168.255.255 (192.168/16 prefix)
			*/
			if( (v1==10)||
				 ((v1==172) && (v2>=16) && (v2<=31)) ||
				 ((v1==192) && (v2==168))
				)
				goto passive_connection_mode;

			/* the only case where the active mode can be set when we are in the private address space */
			/* is when the hub and all other users are also in this space */
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(get_widget_by_widget_name(main_window,"active_mode_radio_button")),TRUE);
			g_string_free(host_ip,TRUE);
		}
		else
		{
			unknown_connection_mode:
			passive_connection_mode:
			/* if the IP is not available, set the passive mode, it will work in any case */
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(get_widget_by_widget_name(main_window,"passive_mode_radio_button")),TRUE);
		}
	}

	/* and save the profile */
	gtk_button_clicked(GTK_BUTTON(get_widget_by_widget_name(main_window,"save_default_profile_button")));

	gtk_widget_destroy(profile_wizard_window);
	profile_wizard_window=NULL;
}


/* --------------------------------------- */
static GtkWidget *druid_browse_dl_dir=NULL;

void
on_druid_do_browse_dl_dir_button_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *dl_dir_ent;

	druid_browse_dl_dir=create_druid_dl_dir_fileselection();

	dl_dir_ent=get_widget_by_widget_name(profile_wizard_window,"druid_dl_dir_entry");
	if(dl_dir_ent!=NULL)
	{
		const char *t;

		t=gtk_entry_get_text(GTK_ENTRY(dl_dir_ent));
		gtk_file_selection_set_filename(GTK_FILE_SELECTION(druid_browse_dl_dir),t);
	}
	gtk_widget_show(druid_browse_dl_dir);
}


void
on_druid_dl_dir_select_ok_button_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	const gchar *selected_filename;

	selected_filename=gtk_file_selection_get_filename(GTK_FILE_SELECTION(druid_browse_dl_dir));
	if(selected_filename!=NULL)
	{
		char *path;

		path=strdup(selected_filename);
		if(path!=NULL)
		{
			struct stat st;

			/* verify if the given name is a valid directory */
			if(stat(path,&st)==-1)
			{
				err:
				gnome_app_error(GNOME_APP(main_window),_("invalid dir name"));
			}
			else
			{
				if(!S_ISDIR(st.st_mode))
				{	/* it exists but it is not a directory, remove the filename and retry */
					char *t;
					t=strrchr(path,'/');
					if(t==path)
						goto err;
					t[1]='\0';		/* keep the trailing / */

					if(stat(path,&st)==-1)
						goto err;
					if(!S_ISDIR(st.st_mode))
						goto err;
				}
				
				{
					GtkWidget *dl_dir_ent;

					dl_dir_ent=get_widget_by_widget_name(profile_wizard_window,"druid_dl_dir_entry");
					if(dl_dir_ent!=NULL)
					{
						gtk_entry_set_text(GTK_ENTRY(dl_dir_ent),path);
					}
				}
			}
			free(path);
		}
	}
	gtk_widget_destroy(druid_browse_dl_dir);
	druid_browse_dl_dir=NULL;
}


void
on_druid_dl_dir_cancel_button_clicked  (GtkButton       *button,
                                        gpointer         user_data)
{
	gtk_widget_destroy(druid_browse_dl_dir);
	druid_browse_dl_dir=NULL;
}

/* --------------------------------------- */
static GtkWidget *druid_add_shared_dir=NULL;
void
on_druid_add_shared_dir_button_clicked (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *dl_dir_ent;

	druid_add_shared_dir=create_druid_shared_dir_fileselection();

	dl_dir_ent=get_widget_by_widget_name(profile_wizard_window,"druid_hidden_shared_dir_entry");
	if(dl_dir_ent!=NULL)
	{
		const char *t;

		t=gtk_entry_get_text(GTK_ENTRY(dl_dir_ent));
		gtk_file_selection_set_filename(GTK_FILE_SELECTION(druid_add_shared_dir),t);
	}
	gtk_widget_show(druid_add_shared_dir);
}

void
on_druid_shared_dir_fs_ok_button_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	const gchar *selected_filename;

	selected_filename=gtk_file_selection_get_filename(GTK_FILE_SELECTION(druid_add_shared_dir));
	if(selected_filename!=NULL)
	{
		char *path;

		path=strdup(selected_filename);
		if(path!=NULL)
		{
			struct stat st;

			/* verify if the given name is a valid directory */
			if(stat(path,&st)==-1)
			{
				err:
				gnome_app_error(GNOME_APP(main_window),_("invalid dir name"));
			}
			else
			{
				if(!S_ISDIR(st.st_mode))
				{	/* it exists but it is not a directory, remove the filename and retry */
					char *t;
					t=strrchr(path,'/');
					if(t==path)
						goto err;
					t[1]='\0';		/* keep the trailing / */

					if(stat(path,&st)==-1)
						goto err;
					if(!S_ISDIR(st.st_mode))
						goto err;
				}
				
				{
					GtkWidget *dl_dir_ent;

					dl_dir_ent=get_widget_by_widget_name(profile_wizard_window,"druid_hidden_shared_dir_entry");
					if(dl_dir_ent!=NULL)
					{
						gtk_entry_set_text(GTK_ENTRY(dl_dir_ent),path);
					}
				}
			}
			free(path);
		}
	}
	gtk_widget_destroy(druid_add_shared_dir);
	druid_add_shared_dir=NULL;

}


void
on_druid_shared_dir_fs_cancel_button_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	gtk_widget_destroy(druid_add_shared_dir);
	druid_add_shared_dir=NULL;
}


/* --------------------------------------- */
gboolean
on_profile_wizard_window_delete_event  (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
	profile_wizard_window=NULL;

	return FALSE;
}


void
on_add_hublist_entry_activate          (GtkEntry        *entry,
                                        gpointer         user_data)
{
	const char *url;

	url=gtk_entry_get_text(entry);

	if(url!=NULL)
	{
		if(strncmp(url,"http://",strlen("http://")))
		{
			gnome_app_error(GNOME_APP(main_window),_("invalid URL, it must start with 'http://'"));
		}
		else
		{
			GtkWidget *w;

			w=get_widget_by_widget_name(main_window,"custom_hublist_treeview");
			if(w!=NULL)
			{
      		GtkTreeIter iter;
				GtkListStore *gls=GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(w)));

      		gtk_list_store_append(gls,&iter);
      		gtk_list_store_set(gls,&iter,CHT_URL_COL,url,-1);

				sync_custom_hublist_treeview_with_hidden_planar_hublist();

				gtk_entry_set_position(entry,0);
				gtk_entry_set_text(entry,"");
			}
		}
	}	
}


void
on_remove_hublist_button_clicked       (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *w;
	GtkTreeModel *gtm;
	GtkListStore *gls;
	GtkTreeSelection *slc;

	int i;

	w=get_widget_by_widget_name(main_window,"custom_hublist_treeview");
	gls=GTK_LIST_STORE(gtm=gtk_tree_view_get_model(GTK_TREE_VIEW(w)));
	slc=gtk_tree_view_get_selection(GTK_TREE_VIEW(w));

	i=gtk_tree_model_iter_n_children(gtm,NULL)-1;
	while(i>=0)
	{
		GtkTreeIter iter;
		if(gtk_tree_model_iter_nth_child(gtm,&iter,NULL,i)==TRUE)
		{
			if(gtk_tree_selection_iter_is_selected(slc,&iter)==TRUE)
				gtk_list_store_remove(gls,&iter);
		}
		i--;
	}

	/* refresh the hidden entry */
	sync_custom_hublist_treeview_with_hidden_planar_hublist();
}


void
on_add_hublist_button_clicked          (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *w;

	w=get_widget_by_widget_name(main_window,"add_hublist_entry");
	if(w!=NULL)
	{
		on_add_hublist_entry_activate(GTK_ENTRY(w),NULL);
	}
}


void
on_get_uinfo_checkbutton_toggled       (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	GtkWidget *w;

	w=get_widget_by_widget_name(main_window,"user_clist");
	if(w)
	{
		gboolean visible;
		GtkTreeView *gtv=GTK_TREE_VIEW(w);

		visible=gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(get_widget_by_widget_name(main_window,"get_uinfo_checkbutton")));

		gtk_tree_view_column_set_visible(gtk_tree_view_get_column(gtv,UCC_TVC_TYPE_COL),visible);
		gtk_tree_view_column_set_visible(gtk_tree_view_get_column(gtv,UCC_TVC_SIZE_COL),visible);
		gtk_tree_view_column_set_visible(gtk_tree_view_get_column(gtv,UCC_TVC_EMAIL_COL),visible);
		gtk_tree_view_column_set_visible(gtk_tree_view_get_column(gtv,UCC_TVC_DESCRIPTION_COL),visible);
	}
}

static GtkWidget *sound_file_selection=NULL;

typedef enum
{
	NO_SOUND=-1,
	END_OF_DL_SOUND=0,
	PRIV_MSG_SOUND,
	HUBLIST_SOUND,
}  SND_FIC_TYPE;

static SND_FIC_TYPE sound_file_selection_type=NO_SOUND;

void
on_end_of_dl_sound_button_clicked      (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *w;
	const char *pattern;

	w=get_widget_by_widget_name(main_window,"end_of_dl_sound_entry");

	if(sound_file_selection==NULL)
	{
		sound_file_selection=create_sound_file_fileselection();
		gtk_widget_show(sound_file_selection);
	}

	pattern=gtk_entry_get_text(GTK_ENTRY(w));
	if(pattern)
	{
		gtk_file_selection_set_filename(GTK_FILE_SELECTION(sound_file_selection),pattern);
	}
	sound_file_selection_type=END_OF_DL_SOUND;
}


void
on_privmsg_sound_button_clicked        (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *w;
	const char *pattern;

	w=get_widget_by_widget_name(main_window,"privmsg_sound_entry");

	if(sound_file_selection==NULL)
	{
		sound_file_selection=create_sound_file_fileselection();
		gtk_widget_show(sound_file_selection);
	}

	pattern=gtk_entry_get_text(GTK_ENTRY(w));
	if(pattern)
	{
		gtk_file_selection_set_filename(GTK_FILE_SELECTION(sound_file_selection),pattern);
	}
	sound_file_selection_type=PRIV_MSG_SOUND;
}


void
on_ok_snd_file_fs_button_clicked       (GtkButton       *button,
                                        gpointer         user_data)
{
	const gchar *selected_filename;

	selected_filename=gtk_file_selection_get_filename(GTK_FILE_SELECTION(sound_file_selection));
	if(selected_filename!=NULL)
	{
		GtkWidget *w;
		switch(sound_file_selection_type)
		{
			case END_OF_DL_SOUND: w=get_widget_by_widget_name(main_window,"end_of_dl_sound_entry");
								break;
			case PRIV_MSG_SOUND:	w=get_widget_by_widget_name(main_window,"privmsg_sound_entry");
								break;
			case HUBLIST_SOUND:	w=get_widget_by_widget_name(main_window,"privmsg_sound_entry");
								break;
			default:	w=NULL;
								break;
		}
		if(w!=NULL)
			gtk_entry_set_text(GTK_ENTRY(w),selected_filename);
	}

	gtk_widget_destroy(sound_file_selection);
	sound_file_selection=NULL;
}


void
on_cancel_snd_file_fs_button_clicked   (GtkButton       *button,
                                        gpointer         user_data)
{
	gtk_widget_destroy(sound_file_selection);
	sound_file_selection=NULL;
}


void
on_hublist_sound_button_clicked        (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *w;
	const char *pattern;

	w=get_widget_by_widget_name(main_window,"hublist_sound_entry");

	if(sound_file_selection==NULL)
	{
		sound_file_selection=create_sound_file_fileselection();
		gtk_widget_show(sound_file_selection);
	}

	pattern=gtk_entry_get_text(GTK_ENTRY(w));
	if(pattern)
	{
		gtk_file_selection_set_filename(GTK_FILE_SELECTION(sound_file_selection),pattern);
	}
	sound_file_selection_type=HUBLIST_SOUND;
}


void
on_copy_search_result_button_clicked   (GtkButton       *button,
                                        gpointer         user_data)
{
	copy_find_result_store_into_the_copy();
}


void
on_paste_search_result_button_clicked  (GtkButton       *button,
                                        gpointer         user_data)
{
	copy_the_copy_into_find_result_store();
}


void
on_apply_filter_button_clicked         (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *w;
	w=get_widget_by_widget_name(main_window,"search_result_filter_combo_entry");
	on_search_result_filter_param_entry_activate(GTK_ENTRY(w),NULL);
}

static FILTER_ID filter_string_to_filter_id(const char *filter_name)
{
	if(!strcmp(filter_name,_("Size <")))
		return FI_SIZE_BELOW;
	if(!strcmp(filter_name,_("Size >")))
		return FI_SIZE_ABOVE;
	if(!strcmp(filter_name,_("Containing the word")))
		return FI_FNAME_WITH;
	if(!strcmp(filter_name,_("Not containing the word")))
		return FI_FNAME_WITHOUT;
	return FI_UNKNOW;
}


void
on_search_result_filter_param_entry_activate
                                        (GtkEntry        *entry,
                                        gpointer         user_data)
{
	const char *param;
	const char *filter;
	GtkWidget *w;
	FILTER_ID filter_id;

	w=get_widget_by_widget_name(main_window,"search_result_filter_param_entry");
	param=gtk_entry_get_text(GTK_ENTRY(w));
	if((param==NULL)||(strlen(param)==0))
		return;

	filter=gtk_entry_get_text(GTK_ENTRY(get_widget_by_widget_name(main_window,"search_result_filter_combo_entry")));
	if((filter==NULL)||(strlen(filter)==0))		/* should never fail... but who knows */
		return;

	filter_id=filter_string_to_filter_id(filter);

	switch(filter_id)
	{
		case FI_UNKNOW:
						fprintf(stderr,"Fatal error: on_search_result_filter_param_entry_activate : unknown filter name '%s'.\n",filter);
						break;

		case FI_SIZE_BELOW:
		case FI_SIZE_ABOVE:
						{
							gulong fsize;
							fsize=strtoul(param,NULL,10);
							filter_find_result_store(filter_id,&fsize);
						}
						break;

		case FI_FNAME_WITH:
		case FI_FNAME_WITHOUT:
						{
							GString fake_gstring;

							fake_gstring.str=(void*)param;
							fake_gstring.len=strlen(param);
							filter_find_result_store(filter_id,&fake_gstring);
						}
						break;
	}
}


void
on_lmule_default_temp_dir_button_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *w;
	GString *str;
	char *path;
	gchar *utf8_path;
	struct stat st;

	w=get_widget_by_widget_name(main_window,"lmule_temp_entry");

	path=getenv("HOME");
	str=g_string_new((path!=NULL)?path:".");
	g_string_append(str,"/.lmule/Temp");
	if(stat(str->str,&st)!=0)
	{
		/* .lmule directory does not exist, let's try .xMule one */
		g_string_assign(str,(path!=NULL)?path:".");
		g_string_append(str,"/.xMule/Temp");
	}
	utf8_path=g_locale_to_utf8(str->str,-1,NULL,NULL,NULL);

	gtk_entry_set_text(GTK_ENTRY(w),utf8_path);

	g_free(utf8_path);
	g_string_free(str,TRUE);

}


void
on_check_ed2k_handler_activate         (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	check_url_handler();
}


void
on_save_gui_prefs_button_clicked       (GtkButton       *button,
                                        gpointer         user_data)
{
	gui_full_save(get_widget_by_widget_name(main_window,"gui_pref_notebook"),"GUIPrefs/");
	gconf_client_suggest_sync(engine,NULL);
}


void
on_reload_gui_prefs_button_clicked     (GtkButton       *button,
                                        gpointer         user_data)
{
	gui_full_restore(get_widget_by_widget_name(main_window,"gui_pref_notebook"),"GUIPrefs/");
}


void
on_clear_ed2k_url_entry_button_clicked (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkWidget *w;

	w=get_widget_by_widget_name(main_window,"ed2k_url_entry");
	if(w!=NULL)
	{
		gtk_entry_set_position(GTK_ENTRY(w),0);
		gtk_entry_set_text(GTK_ENTRY(w),"");
	}
}


void
on_notes_textview_insert_at_cursor     (GtkTextView     *textview,
                                        gchar           *arg1,
                                        gpointer         user_data)
{
	set_notes_buffer_to_dirty();
}


void
on_save_notes_button_clicked           (GtkButton       *button,
                                        gpointer         user_data)
{
	save_notes_text_buffer();
}


void
on_clear_notes_button_clicked          (GtkButton       *button,
                                        gpointer         user_data)
{
	clear_notes_text_buffer();
}


void
on_reload_notes_button_clicked         (GtkButton       *button,
                                        gpointer         user_data)
{
	load_notes_text_buffer();
}

/* input: data: GConfEntry * */
static void remove_profile_subentry(gpointer data, gpointer user_data)
{
	gconf_client_unset(engine,((GConfEntry *)data)->key,NULL);
}

gboolean
on_delete_profile_button_button_press_event
                                        (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
	if(event==NULL)
		return TRUE;

	if((event->button==1)&&(event->type==GDK_2BUTTON_PRESS))
	{  /* left click */
		/* double click on a row */
		GtkWidget *w;
		const char *profile_name;

		w=get_widget_by_widget_name(main_window,"profile_name_combo_entry");
		if(w==NULL)
			return TRUE;

		profile_name=gtk_entry_get_text(GTK_ENTRY(w));
		if((profile_name==NULL)||(strlen(profile_name)==0))
		{
			gnome_app_error(GNOME_APP(main_window),_("You must give profile name first"));
		}
		else
		{
			if(is_all_alnum(profile_name))
			{
				char *pval;
				GString *str;
				GString *str2;

				str=g_string_new("");
				str2=g_string_new("");
				g_string_sprintf(str,"/" PROGNAME "/ProfileNames/%s",profile_name);
			
				pval=gnome_config_get_string(str->str);
				if(pval!=NULL)
				{
					GSList *gsl;

					/* clear the main profile key */
					gnome_config_clean_key(str->str);

					/* create the string to search */
					g_string_sprintf(str,"/apps/" PROGNAME "/Profile/%s",profile_name);

					gsl=gconf_client_all_entries(engine,str->str,NULL);
					if(gsl)
					{
						g_slist_foreach(gsl,remove_profile_subentry,NULL);
						g_slist_free(gsl);
					}

					gnome_config_sync();
					gconf_client_suggest_sync(engine,NULL);
	
					load_profile_name_combo();
				}
				else
				{
					g_string_sprintf(str,_("No profile named %s exists."),profile_name);
					gnome_app_error(GNOME_APP(main_window),str->str);
				}
				g_string_free(str,TRUE);
				g_string_free(str2,TRUE);
			}
			else
			{
				gnome_app_error(GNOME_APP(main_window),_("Invalid profile name, only alphanumeric characters allowed"));
			}
		}
		return TRUE;
	}

	return FALSE;
}


void
on_check_torrent_handler_registration_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{

}


void
on_disable_tooltips_checkbutton_toggled
                                        (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	GtkWidget *w;

	w=get_widget_by_widget_name(main_window,"tooltips");

	if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(togglebutton)))
		gtk_tooltips_disable(GTK_TOOLTIPS(w));
	else
		gtk_tooltips_enable(GTK_TOOLTIPS(w));
}

