/***************************************************************************
                          dchubsearch.cpp  -  description
                             -------------------
    begin                : Fri Mar 15 2002
    copyright            : (C) 2002-2004 by Mathias Kster
    email                : mathen@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/** qt includes */
#include <qapplication.h>
#include <qlineedit.h>
#include <qtextedit.h>
#include <qradiobutton.h>
#include <qlistview.h>
#include <qcheckbox.h>
#include <qlcdnumber.h>
#include <qcombobox.h>
#include <qregexp.h>
#include <qspinbox.h>
#include <qpushbutton.h>
#include <qmessagebox.h>
#include <qstringlist.h>
#include <qtabwidget.h>
#include <qbuttongroup.h>
#include <qpopupmenu.h>
#include <qcursor.h>
#include <qfiledialog.h>
#include <qclipboard.h>
#include <qprogressbar.h>
#include <qvaluevector.h>
//#include <qdragobject.h>
#include <qtoolbutton.h>
#include <qinputdialog.h>

/** */
#include <dclib/dcos.h>
#include <dclib/cutils.h>

#ifndef WIN32
#include <stdlib.h>
#endif

#include <dcconfig.h>
#include <dcclient.h>
#include <dcconnectionmanager.h>
#include <dchublistmanager.h>
#include <dcfiletool.h>
#include <dctransferview.h>
#include <dcmenuhandler.h>
#include <dcwidget.h>
#include <dciconloader.h>

#include "dchubsearch.h"

#include <dclib/core/cdir.h>
#include <dclib/core/cxml.h>
#include <dclib/chttp.h>

DCHubSearch * g_pHubSearch = 0;

/** */
DCHubSearch::DCHubSearch(QWidget* parent, const char *name, int wflags) : DCDialogHubSearch(parent, name, wflags)
{
	int idx;

	// set default icon
	setIcon( g_pIconLoader->GetPixmap(eiFIND) );

	m_pWorkspace = (QWorkspace*)parent;

	// set the ColumnWidthMode to manual, we will take care of this task
	for( idx = 0; idx < ListView_SEARCHRESULT->columns(); idx++ )
	{
		ListView_SEARCHRESULT->setColumnWidthMode( idx, QListView::Manual );
	}

	for( idx = 0; idx < ListView_SEARCHRESULTUSER->columns(); idx++ )
	{
		ListView_SEARCHRESULTUSER->setColumnWidthMode( idx, QListView::Manual );
	}

	m_ehSearchType = ehstNONE;

	m_pMessageList      = new CList<CObject>();
	m_pSearchResultList = new CList<CObject>();
	m_pSearchQueryList  = 0;

	InitDocument();

	g_pHubSearch = this;
}

/** */
DCHubSearch::~DCHubSearch()
{
	g_pHubSearch = NULL;

	ListView_SEARCH->removeEventFilter(this);

	m_ehSearchType = ehstNONE;

	m_Timer.stop();

	SocketCallbackThread.Lock();

	// cleanup all lists
	if ( m_pMessageList )
	{
		delete m_pMessageList;
		m_pMessageList = 0;
	}

	if ( m_pSearchResultList )
	{
		delete m_pSearchResultList;
		m_pSearchResultList = 0;
	}

	if ( m_pSearchQueryList )
	{
		delete m_pSearchQueryList;
		m_pSearchQueryList = 0;
	}

	SocketCallbackThread.UnLock();
}

/** */
void DCHubSearch::InitDocument()
{
	StringMap * map;

	// restore settings
	if ( g_pConfig->GetMap("SEARCHVIEW",map) == TRUE )
	{
		if ( ((*map)["WIDTH"].toInt() > 0) && ((*map)["HEIGHT"].toInt() > 0) )
		{
			setGeometry( (*map)["X"].toInt(), (*map)["Y"].toInt(), (*map)["WIDTH"].toInt(), (*map)["HEIGHT"].toInt() );
		}
	}

	connect( LineEdit_INCLUDE, SIGNAL(returnPressed()), this, SLOT(slotTextFilterResults()) );
	connect( LineEdit_EXCLUDE, SIGNAL(returnPressed()), this, SLOT(slotTextFilterResults()) );
	connect( PushButton_SEARCH, SIGNAL(clicked()), this, SLOT(slotSearchReturnPressed()) );
	connect( PushButton_ADDQUEUE, SIGNAL(clicked()), this, SLOT(slotAddSearchQueue()) );
	connect( PushButton_REFRESHCONNECTEDHUBS, SIGNAL(clicked()), this, SLOT(slotRefreshConnectedHubs()) );
	connect( PushButton_APPLYTEXTFILTER, SIGNAL(clicked()), this, SLOT(slotTextFilterResults()) );
	connect( PushButton_RESETTEXTFILTER, SIGNAL(clicked()), this, SLOT(slotTextFilterReset()) );
 	connect( ToolButton_RESETFILTER, SIGNAL(clicked()), this, SLOT(slotReset()) );
	connect( ToolButton_RESETFILTER, SIGNAL(clicked()), this, SLOT(slotReset()) );
	connect( SpinBox_FREESLOTS, SIGNAL(valueChanged(int)), this, SLOT(slotChangedFreeSlots(int)) );
	connect( CheckBox_SEARCHFILEONLY, SIGNAL(toggled(bool)), this, SLOT(slotToggledSearchFileOnly(bool)) );
	connect( TabWidget_HUBSEARCH,SIGNAL(currentChanged(QWidget*)), this, SLOT(slotTabWidgetCurrentChange(QWidget*)) );
	connect( ListView_SEARCHRESULT,SIGNAL(doubleClicked(QListViewItem*)), this, SLOT(slotDoubleClickedSearchResult(QListViewItem*)) );
	connect( ListView_SEARCHRESULT,SIGNAL(contextMenuRequested( QListViewItem *, const QPoint &, int )), this, SLOT(slotRightButtonClickedSearchResult(QListViewItem*, const QPoint &, int )) );
	connect( ListView_SEARCH,SIGNAL(contextMenuRequested( QListViewItem *, const QPoint &, int )), this, SLOT(slotRightButtonClickedSearch(QListViewItem*, const QPoint &, int )) );
	
	m_bSearchState = FALSE;

	connect( &m_Timer, SIGNAL(timeout()), this, SLOT(timerDone()) );
	m_Timer.start( 500, TRUE );

	ListView_SEARCH->installEventFilter(this);
	Combobox_SEARCH->installEventFilter(this);
}

/** */
void DCHubSearch::DeInitDocument()
{
	StringMap * map;

	// save search view settings
	g_pConfig->GetMap("SEARCHVIEW",map);

	(*map)["X"]         = QString().setNum(x());
	(*map)["Y"]         = QString().setNum(y());
	(*map)["WIDTH"]     = QString().setNum(width());
	(*map)["HEIGHT"]    = QString().setNum(height());
	(*map)["VISIBLE"]   = QString().setNum(isVisible());
	(*map)["MAXIMIZED"] = QString().setNum(isMaximized());
	(*map)["MINIMIZED"] = QString().setNum(isMinimized());
	(*map)["ENABLED"]   = QString().setNum(isEnabled());
}

/** current tab widget change slot */
void DCHubSearch::slotTabWidgetCurrentChange(QWidget*)
{
	ResizeListViewColumn();
}

/** overridden so that the columns are resized on show() */
void DCHubSearch::show()
{
	QWidget::show();

	ResizeListViewColumn();
}

/** resize event handler */
void DCHubSearch::resizeEvent( QResizeEvent * )
{
	ResizeListViewColumn();
}

/** resize the ListView columns */
void DCHubSearch::ResizeListViewColumn()
{
	int width;

	if ( ListView_SEARCHRESULT->isVisible() )
	{
		width = ListView_SEARCHRESULT->width();

		ListView_SEARCHRESULT->setColumnWidth( 0, ((width*7)/32) );
		ListView_SEARCHRESULT->setColumnWidth( 1, ((width*3)/32) );
		ListView_SEARCHRESULT->setColumnWidth( 2, ((width*4)/32) );
		ListView_SEARCHRESULT->setColumnWidth( 3, ((width*4)/32) );
		ListView_SEARCHRESULT->setColumnWidth( 4, ((width*1)/32) );
		ListView_SEARCHRESULT->setColumnWidth( 5, ((width*4)/32) );
		ListView_SEARCHRESULT->setColumnWidth( 6, ((width*6)/32) );
		ListView_SEARCHRESULT->setColumnWidth( 7, ((width*3)/32) );
	}

	if ( ListView_SEARCHRESULTUSER->isVisible() )
	{
		width = ListView_SEARCHRESULTUSER->width();

		ListView_SEARCHRESULTUSER->setColumnWidth( 0, ((width)/2) );
		ListView_SEARCHRESULTUSER->setColumnWidth( 1, ((width)/2) );
	}
}

/** event filter */
bool DCHubSearch::eventFilter( QObject * object, QEvent * event )
{
	if ((event->type() == QEvent::KeyPress)&&((QListView*)object==ListView_SEARCH))
	{
		QKeyEvent * e = (QKeyEvent*)event;

		if ( e->key() == Key_Delete )
		{
			RemoveSelectedSearch(NULL);
		}
	}
	else if ((event->type() == QEvent::KeyPress)&&((QComboBox*)object==Combobox_SEARCH))
	{
		QKeyEvent * e = (QKeyEvent*)event;
		if((e->key() == Key_Enter) || (e->key() == Key_Return ))
		{
			slotSearchReturnPressed();
		}
	}

	return QWidget::eventFilter( object, event );    // standard event processing
}

/** */
int DCHubSearch::selectedItems( QListView * list, QPtrList<QListViewItem> & lst )
{
	QListViewItemIterator it( (QListView *)list );

	for ( ; it.current(); it++ )
	{
		if ( it.current()->isSelected() )
		{
			lst.append(it.current());
		}
	}

	return lst.count();
}

/** */
void DCHubSearch::SetSearchView( bool enabled )
{
	if ( enabled )
	{
		PushButton_SEARCH->setText(tr("Start"));
		PushButton_SEARCH->setEnabled(TRUE);
		m_bSearchState = FALSE;
	}
	else
	{
		PushButton_SEARCH->setText(tr("Stop"));
		m_bSearchState = TRUE;
	}

	//LineEdit_SEARCH->setEnabled(enabled);
	Combobox_SEARCH->setEnabled(enabled);
	ButtonGroup_HUBS->setEnabled(enabled);
	SpinBox_MAXTHREADS->setEnabled(enabled);
	CheckBox_SEARCHUSER->setEnabled(enabled);
	PushButton_ADDQUEUE->setEnabled(enabled);
	GroupBox_FILTER->setEnabled(enabled);
	CheckBox_ENABLETAG->setEnabled(enabled);
	CheckBox_MULTISEARCH->setEnabled(enabled);
}

/** */
int DCHubSearch::DC_CallBack( CObject * Object )
{
	SocketCallbackThread.Lock();

	int err = -1;

	if ( m_ehSearchType != ehstNONE )
	{
		if ( (Object != 0) && (m_pMessageList != 0) )
		{
			m_pMessageList->Add(Object);
			err = 0;
		}
        }

	SocketCallbackThread.UnLock();

	return err;
}

/** */
void DCHubSearch::timerDone()
{
	int i,t;

	if ( (m_bSearchState == FALSE) && (m_ehSearchType != ehstNONE) )
	{
		SetSearchView( FALSE );
	}
	else if ( (m_bSearchState == TRUE) && (m_ehSearchType == ehstNONE) )
	{
		SetSearchView( TRUE );
	}

	if ( (m_ehSearchType != ehstNONE) && (m_ehSearchType != ehstEXTERNAL) )
	{
		if ( m_ehSearchType != ehstNONE )
		{
			ShowResults(FALSE);
		}

		qApp->processEvents();

		if ( m_pHubServerList != 0 )
		{
			LineEdit_LOGHUBS->setText( QString().setNum(m_nCurrentHub)+"/"+QString().setNum(m_pHubServerList->Count())+" ("+QString().setNum(m_nError)+")" );

			i=0;
			if (m_pHubServerList->Count()>0)
				i = ((m_nCurrentHub*100)/m_pHubServerList->Count());
			ProgressBar_LOGHUBS->setProgress(i);

			i = time(0)-m_tStartTime;

			if ( m_nCurrentHub > 0 )
				t = ((i*m_pHubServerList->Count())/m_nCurrentHub);
			else
				t = 0;

			LineEdit_LOGTIME->setText( (CUtils::GetTimeString(i) + "/" + CUtils::GetTimeString(t)).Data() );
		}
	}

	// restart timer
	m_Timer.start( 500, TRUE );
}

/** */
void DCHubSearch::ShowResults( bool bClearList )
{
	CDCMessage *DCMsg;
	CObject * Object;
	bool bupdate;

	if ( bClearList == TRUE )
	{
		ListView_SEARCHRESULT->clear();
	}

	if ( SocketCallbackThread.TryLock() == FALSE )
	{
		return;
	}

	bupdate = FALSE;

	if ( m_pMessageList != 0 )
	{
		while( (Object = m_pMessageList->Next(0)) != 0 )
		{
			m_pMessageList->Remove(Object);

			DCMsg = (CDCMessage*) Object;

			switch ( DCMsg->m_eType )
			{
				case DC_MESSAGE_SEARCHRESULT:
				{
					bool b;
					b = DC_SearchResult( (CMessageSearchResult *) Object );

					if( b == TRUE )
					{
						Object  = 0;
						bupdate = TRUE;
						ListView_SEARCHRESULT->setUpdatesEnabled(FALSE);
					}

					break;
				}

				case DC_MESSAGE_SEARCHRESULT_USER:
				{
					CMessageSearchResultUser * msg = (CMessageSearchResultUser*) Object;

					QListViewItem *item = new QListViewItem(ListView_SEARCHRESULTUSER);
					item->setText(0,msg->m_sNick.Data());
					item->setText(1,msg->m_sHubName.Data());
					break;
				}

				case DC_MESSAGE_LOG:
				{
					DC_LogMessage( (CMessageLog *) Object );

					break;
				}

				default:
				{
					break;
				}
			}

			if ( Object )
				delete Object;
		}
	}

	SocketCallbackThread.UnLock();

	if ( bupdate == TRUE )
	{
		ListView_SEARCHRESULT->setUpdatesEnabled(TRUE);
		ListView_SEARCHRESULT->triggerUpdate();
	}
}

/** search result */
bool DCHubSearch::DC_SearchResult( CMessageSearchResult * MessageSearchResult )
{
	bool res = FALSE;
	ulonglong size;

	if ( m_ehSearchType == ehstNONE )
	{
		return res;
	}

	if ( SpinBox_MAXRESULT->value() > 0 )
	{
		if ( ListView_SEARCHRESULT->childCount() >= SpinBox_MAXRESULT->value() )
		{
			return res;
		}
	}

	size = CString(LineEdit_SEARCHSIZE->text().ascii()).asULL();

	if ( ComboBox_SEARCHUNIT->currentItem() == 1 )
	{
		size *= 1024;
	}
	else if ( ComboBox_SEARCHUNIT->currentItem() == 2 )
	{
		size *= 1024*1024;
	}
	else if ( ComboBox_SEARCHUNIT->currentItem() == 3 )
	{
		size *= 1024*1024*1024;
	}

	if ( ComboBox_SEARCHLIMIT->currentItem() == 0 )
	{
		if ( MessageSearchResult->m_nSize < size )
		{
			return res;
		}
	}
	else if ( ComboBox_SEARCHLIMIT->currentItem() == 1 )
	{
		if ( MessageSearchResult->m_nSize > size )
		{
			return res;
		}
	}
	else if ( ComboBox_SEARCHLIMIT->currentItem() == 2 )
	{
		if ( MessageSearchResult->m_nSize != size )
		{
			return res;
		}
	}

	ShowSearchResult( MessageSearchResult, ListView_SEARCHRESULT );

	res = TRUE;

	m_pSearchResultList->Add(MessageSearchResult);

	LCDNumber_RESULTS->display(ListView_SEARCHRESULT->childCount());

	return res;
}

/** */
QListViewItem * DCHubSearch::ShowSearchResult( CMessageSearchResult * MessageSearchResult, QListView * parent )
{
	DC_ListResult *item = new DC_ListResult(parent);

	AddSearchResult( MessageSearchResult, item );

	return item;
}

/** */
QListViewItem * DCHubSearch::ShowSearchResult( CMessageSearchResult * MessageSearchResult, DC_ListResult * parent )
{
	DC_ListResult *item = new DC_ListResult(parent);

	AddSearchResult( MessageSearchResult, item );

	return item;
}

/** */
bool DCHubSearch::AddSearchResult( CMessageSearchResult * MessageSearchResult, DC_ListResult * item )
{
	bool res = FALSE;
	bool visible = TRUE;
	int i;
	QString s;
	CString spath,sname;
	DCFileTool FileTool;

	if ( MessageSearchResult->m_nFreeSlot < (unsigned int)SpinBox_FREESLOTS->value() )
	{
		visible = FALSE;
	}

	s = MessageSearchResult->m_sFile.Data();

	if (CheckBox_SEARCHFILEONLY->isChecked())
	{
	        // Checks if s matches the search pattern
		QString search_pattern;
		//for(i=0;i<=LineEdit_SEARCH->text().contains(" ");i++)
		for(i=0;i<=Combobox_SEARCH->currentText().contains(" ");i++)
		{
		        //search_pattern = LineEdit_SEARCH->text().section(" ",i,i);
		        search_pattern = Combobox_SEARCH->currentText().section(" ",i,i);

			if ( s.find(search_pattern,0,FALSE) == -1 )
			{
				visible = FALSE;
			}
		}
	}

	// split filename and path
	s = MessageSearchResult->m_sFile.Data();

	CDir().SplitPathFile(s.ascii(),spath,sname);

	item->myvalue = MessageSearchResult->m_nSize;
	item->mycol   = 1;

	item->setText(0,sname.Data());
	item->setText(1,CUtils::GetSizeString(MessageSearchResult->m_nSize,g_pConfig->GetUnit()).Data());
	item->setText(2,MessageSearchResult->m_sHash.Data());
	item->setText(3,MessageSearchResult->m_sNick.Data());
	item->setText(4,QString().setNum(MessageSearchResult->m_nFreeSlot)+'/'+QString().setNum(MessageSearchResult->m_nMaxSlot));
	item->setText(5,MessageSearchResult->m_sHubName.Data());
	item->setText(6,spath.Data());
	item->setText(7,MessageSearchResult->m_sHubHost.Data());

	item->setVisible(visible);
	item->setEnabled(visible);

	return res;
}

/** */
void DCHubSearch::GroupSearchResults( eGroupSearchResultType type )
{
	CStringList sl;
	DC_ListResult * item;
	CMessageSearchResult * msg = 0;
	CString s,st;

	ListView_SEARCHRESULT->setUpdatesEnabled(FALSE);
	ListView_SEARCHRESULT->clear();

	ListView_SEARCHRESULT->setRootIsDecorated((type != egsrtNONE));

	while ( (msg=(CMessageSearchResult *)m_pSearchResultList->Next((CObject*)msg)) != 0 )
	{
		item = 0;

		switch(type)
		{
			case egsrtFILE:
				CDir().SplitPathFile(msg->m_sFile,st,s);
				break;
			case egsrtSIZE:
				s = CString().setNum(msg->m_nSize);
				break;
			case egsrtNICK:
				s = msg->m_sNick;
				break;
			case egsrtHASH:
				s = msg->m_sHash;
				break;
			case egsrtSLOTS_FREE:
				s = CString().setNum(msg->m_nFreeSlot);
				break;
			case egsrtHUB:
				s = msg->m_sHubName;
				break;
			case egsrtPATH:
				CDir().SplitPathFile(msg->m_sFile,s,st);
				break;
			case egsrtHOST:
				s = msg->m_sHubHost;
				break;
			case egsrtNONE:
			default:
				s = "";
				break;
		}

		if ( (s != "") && (sl.Get(s,(CObject*&)item) == 0) )
		{
			ShowSearchResult(msg,item);
		}
		else
		{
			item = (DC_ListResult *)ShowSearchResult(msg,ListView_SEARCHRESULT);

			if ( s != "" )
				sl.Add( s, (CObject*)item );
		}
	}

	item = 0;
	s = "";

	while( sl.Next( s, (CObject*&)item ) != 0 )
	{
		sl.Remove(s);
		item = 0;
	}

	ListView_SEARCHRESULT->setUpdatesEnabled(TRUE);
	ListView_SEARCHRESULT->triggerUpdate();
}

/** */
void DCHubSearch::DC_LogMessage( CMessageLog * MessageLog )
{
	bool bscroll;

	if ( TextEdit_LOG->verticalScrollBar()->maxValue() == TextEdit_LOG->verticalScrollBar()->value() )
	{
		bscroll = TRUE;
	}
	else
	{
		bscroll = FALSE;
	}

	TextEdit_LOG->append(MessageLog->sMessage.Data());

	if ( bscroll )
	{
		TextEdit_LOG->scrollToBottom();
		TextEdit_LOG->moveCursor( QTextEdit::MoveEnd, FALSE );
	}
}

/** */
CDCMessage * DCHubSearch::GetSearchObject()
{
	CMessageSearchUser * MessageSearchUser = 0;
	CMessageSearchFile * MessageSearchFile = 0;

	//if ( LineEdit_SEARCH->text() == "" )
	if ( Combobox_SEARCH->currentText() == "")
	{
		return 0;
	}

	// set searchmode
	if ( CheckBox_SEARCHUSER->isChecked() )
	{
		MessageSearchUser = new CMessageSearchUser();
		MessageSearchUser->m_eType = DC_MESSAGE_SEARCH_USER;
		//MessageSearchUser->m_sNick = LineEdit_SEARCH->text().ascii();
		MessageSearchUser->m_sNick = Combobox_SEARCH->currentText().ascii();

		return MessageSearchUser;
	}

	MessageSearchFile = new CMessageSearchFile();
	MessageSearchFile->m_eType = DC_MESSAGE_SEARCH_FILE;

	//MessageSearchFile->m_sString = LineEdit_SEARCH->text().ascii();
	MessageSearchFile->m_sString = Combobox_SEARCH->currentText().ascii();

	if ( g_pConfig->GetMode() == ecmPASSIVE )
	{
		MessageSearchFile->m_bLocal = TRUE;

		if ( (RadioButton_CONNECTEDSINGLEHUB->isChecked() == TRUE ) ||
		     (RadioButton_CONNECTEDHUBS->isChecked() == TRUE) )
		{ 
			MessageSearchFile->m_sSource = g_pConfig->GetNick();
		}
		else
		{ 
			MessageSearchFile->m_sSource = g_pConfig->GetSearchNick();
		}
	}
	else
	{
		MessageSearchFile->m_bLocal  = FALSE;
		MessageSearchFile->m_sSource = g_pConfig->GetUDPHostString();
	}

	if ( MessageSearchFile->m_sSource == "" )
	{
		delete MessageSearchFile;
		return 0;
	}

	// for exact size searches set size to 0
	if ( ComboBox_SEARCHLIMIT->currentItem() == 2 )
	{
		MessageSearchFile->m_nSize = 0;
	}
	else
	{
		MessageSearchFile->m_nSize = CString(LineEdit_SEARCHSIZE->text().ascii()).asULL();
	}

	if ( ComboBox_SEARCHUNIT->currentItem() == 1 )
	{
		MessageSearchFile->m_nSize *= 1024;
	}
	else if ( ComboBox_SEARCHUNIT->currentItem() == 2 )
	{
		MessageSearchFile->m_nSize *= 1024*1024;
	}
	else if ( ComboBox_SEARCHUNIT->currentItem() == 3 )
	{
		MessageSearchFile->m_nSize *= 1024*1024*1024;
	}

	if ( MessageSearchFile->m_nSize == 0 )
	{
		MessageSearchFile->m_bSizeLimit = FALSE;
	}

	if ( (ComboBox_SEARCHLIMIT->currentItem() == 0) || (ComboBox_SEARCHLIMIT->currentItem() == 2) || (MessageSearchFile->m_nSize == 0) )
	{
		MessageSearchFile->m_bSizeAtMost = FALSE;
	}
	else
	{
		MessageSearchFile->m_bSizeAtMost = TRUE;
	}

	MessageSearchFile->m_eFileType = eFileTypes(ComboBox_SEARCHTYPE->currentItem()+1);

	return MessageSearchFile;
}

/** */
void DCHubSearch::slotRefreshConnectedHubs()
{
	CString *ps;
	CStringList * StringList;

	ComboBox_CONNECTEDHUBS->clear();

	if ( (StringList = g_pConnectionManager->GetConnectedHubServerList()) != 0 )
	{
		ps = 0;
		while( StringList->Next((CObject*&)ps) )
		{
			ComboBox_CONNECTEDHUBS->insertItem(ps->Data());
		}

		delete StringList;
	}
}

/** */
void DCHubSearch::UpdateHidden( bool SearchFileOnly, int FreeSlots, bool TextFilter )
{
	QListViewItemIterator iter( ListView_SEARCHRESULT );

	ListView_SEARCHRESULT->setUpdatesEnabled(FALSE);

	for ( ; iter.current(); iter++ )
	{
		bool visible = TRUE;
		QString path_file;

		path_file = iter.current()->text(6) + iter.current()->text(0);

		if ( SearchFileOnly )
		{
			// Checks if iter.current() path+filename matches the search pattern
			QString search_pattern;
			//for(int j=0; j<=LineEdit_SEARCH->text().contains(" "); j++)
			for(int j=0; j<=Combobox_SEARCH->currentText().contains(" "); j++)
                        {
				//search_pattern = LineEdit_SEARCH->text().section(" ",j,j);
				search_pattern = Combobox_SEARCH->currentText().section(" ",j,j);
				if ( path_file.find(search_pattern,0,FALSE) == -1 )
                                {
					visible = FALSE;
                                }
			}
		}

		// filtering with include
		if (TextFilter && (LineEdit_INCLUDE->text().isEmpty()==FALSE))
		{
			QString search_pattern;
			for(int i=0; search_pattern = LineEdit_INCLUDE->text().section(" ",i,i,QString::SectionSkipEmpty); i++)
			{
				if ( path_file.find(search_pattern,0,FALSE) == -1 )
				{
					visible = FALSE;
				}
			}
		}
		// filtering with exclude
		if (TextFilter && (LineEdit_EXCLUDE->text().isEmpty()==FALSE))
		{
			QString search_pattern;
			for(int i=0; search_pattern = LineEdit_EXCLUDE->text().section(" ",i,i,QString::SectionSkipEmpty); i++)
			{
				if ( path_file.find(search_pattern,0,FALSE) != -1 )
				{
					visible = FALSE;
				}
			}
		}
		
		if ( visible == TRUE )
		{
			if ( iter.current()->text(4).section("/",0,0).toInt() < FreeSlots )
			{
				visible = FALSE;
			}
		}

		iter.current()->setVisible(visible);
		iter.current()->setEnabled(visible);
	}

	ListView_SEARCHRESULT->setUpdatesEnabled(TRUE);
	ListView_SEARCHRESULT->triggerUpdate();
}

/** */
void DCHubSearch::slotTextFilterResults()
{
	UpdateHidden( CheckBox_SEARCHFILEONLY->isOn(), SpinBox_FREESLOTS->value(), TRUE );
}

/** */
void DCHubSearch::slotTextFilterReset()
{
	UpdateHidden( CheckBox_SEARCHFILEONLY->isOn(), SpinBox_FREESLOTS->value(), FALSE );
}

/** */
void DCHubSearch::slotToggledSearchFileOnly( bool chkstate )
{
	UpdateHidden( chkstate, SpinBox_FREESLOTS->value() );
}

/** */
void DCHubSearch::slotChangedFreeSlots( int free_slots )
{
	UpdateHidden( CheckBox_SEARCHFILEONLY->isOn(), free_slots );
}

/** */
void DCHubSearch::slotSearchReturnPressed()
{
	if ( m_ehSearchType == ehstNONE )
	{	
		startSearch();
	}
	else
	{
		StopSearch();
		PushButton_SEARCH->setEnabled(FALSE);
	}
}

/** */
void DCHubSearch::startSearch()
{
	CObject * Object;
	eHubSearchHubs hubs;
	CStringList * hublist = 0;
	bool multi;

	if ( m_ehSearchType == ehstEXTERNAL )
	{
		QMessageBox::critical( this, tr("Hub Search Error"),
			QString( tr("External search is already running!") ));
		return;
	}

	// clear messagelist
	SocketCallbackThread.Lock();

	while( (Object = m_pMessageList->Next(0)) != 0 )
		m_pMessageList->Del(Object);

	m_pSearchResultList->Clear();

	SocketCallbackThread.UnLock();

	// clear searchresults & log
	ListView_SEARCHRESULT->clear();
	ListView_SEARCHRESULTUSER->clear();
	TextEdit_LOG->clear();

	// reset counter
	LCDNumber_RESULTS->display(0);
	ProgressBar_LOGHUBS->setProgress(0);

	// reset
	LineEdit_LOGTIME->setText("");
	LineEdit_LOGHUBS->setText("0/0");

	// clear the list on new search if only one search in the list
	if ( (m_pSearchQueryList != 0) && (m_pSearchQueryList->Count() == 1) )
	{
		m_pSearchQueryList->Clear();
		ListView_SEARCH->clear();
	}

	// add a new search into empty list
	if ( (m_pSearchQueryList == 0) || (m_pSearchQueryList->Count() == 0) )
	{
		slotAddSearchQueue();
	}
	
	if ( (m_pSearchQueryList == 0) || (m_pSearchQueryList->Count() == 0) )
	{
		QMessageBox::critical( this, tr("Hub Search Error"),
				       QString( tr("Please add a search!") ));
		return;
	}

//	g_pConfig->AddSearchHistory( LineEdit_SEARCH->text().ascii() );

	multi = CheckBox_MULTISEARCH->isChecked();

	if ( (g_pConfig->GetMode() == ecmPASSIVE) && (multi) )
	{
		QMessageBox::critical( this, tr("Hub Search Error"),
			QString(tr("Multi Search only work in active mode!") ));
		return;
	}

	// set searchtype
	if ( RadioButton_CONNECTEDSINGLEHUB->isChecked() )
	{
		hubs = ehshCONNECTEDSINGLE;
	}
	else if ( RadioButton_CONNECTEDHUBS->isChecked() )
	{
		hubs = ehshCONNECTEDALL;
	}
	else if ( RadioButton_FILTEREDHUBS->isChecked() )
	{
		hublist = g_pHubListManager->GetFilteredHubList();

		hubs = ehshPUBLIC;
	}
	else if ( RadioButton_AVAILABLEHUBS->isChecked() )
	{
		hubs = ehshPUBLIC;
	}
	else
	{
		hubs = ehshBOOKMARK;
	}

	m_nMaxThreads = SpinBox_MAXTHREADS->value();
	m_bEnableTag  = CheckBox_ENABLETAG->isChecked();

	if ( hubs == ehshCONNECTEDSINGLE )
	{
		CString hubname;

		hubname = ComboBox_CONNECTEDHUBS->currentText().ascii();

		if ( hubname == "" )
		{
			QMessageBox::critical( this, tr("Hub Search Error"),
				QString(tr("Please select a connected hub.")));

			return;
		}
		else if ( StartSearch(hubs,multi,m_pSearchQueryList,hublist,hubname) != 0 )
		{
			QMessageBox::critical( this, tr("Hub Search Error"),
				QString(tr("No connected hubs found.")));

			return;
		}
	}
	else if ( hubs == ehshCONNECTEDALL )
	{
		if ( StartSearch(hubs,multi,m_pSearchQueryList,hublist) != 0 )
		{
			QMessageBox::critical( this, tr("Hub Search Error"),
				QString(tr("No connected hubs found.")));

			return;
		}
	}
	else
	{
		if ( hubs == ehshPUBLIC )
		{
			if ( StartSearch(hubs,multi,m_pSearchQueryList,hublist) != 0 )
			{
				delete hublist;

				QMessageBox::critical( this, tr("Hub Search Error"),
					QString(tr("No hubs found.")));

				return;
			}
		}
		else
		{
			if ( StartSearch(hubs,multi,m_pSearchQueryList,hublist) != 0 )
			{
				QMessageBox::critical( this, tr("Hub Search Error"),
					QString(tr("No hubs found.")));

				return;
			}
		}
	}

	// update view
	SetSearchView( FALSE );

	return;
}

/** */
void DCHubSearch::slotAddSearchQueue()
{
	CDCMessage * msg, *msg1;
	
	CMessageSearchUser * MessageSearchUser = 0;
	CMessageSearchFile * MessageSearchFile = 0;

	if ( m_pSearchQueryList == 0 )
	{
		m_pSearchQueryList = new CList<CObject>();
	}

	if ( (msg = GetSearchObject()) == 0 )
	{
		return;
	}

	if ( msg->m_eType == DC_MESSAGE_SEARCH_USER )
	{
		MessageSearchUser = (CMessageSearchUser *)msg;
	}
	else
	{
		MessageSearchFile = (CMessageSearchFile *)msg;
	}

	msg1 = 0;

	while ( (msg1 = (CDCMessage*)m_pSearchQueryList->Next((CObject*)msg1)) != 0 )
	{
		if ( msg1->m_eType == msg->m_eType )
		{
			//TODO: compare if search already in the list
			if ( MessageSearchUser )
			{
				if ( ((CMessageSearchUser*)msg1)->m_sNick == MessageSearchUser->m_sNick )
				{
					return;
				}
			}
			else
			{
				if ( ((CMessageSearchFile*)msg1)->m_sString == MessageSearchFile->m_sString )
				{
					return;
				}
			}
		}
	}

	DC_QSearchListViewItem *item = new DC_QSearchListViewItem(ListView_SEARCH);

	item->p_data = msg;

	if ( MessageSearchUser )
	{
		item->setText(0,tr("USER"));
		item->setText(1,MessageSearchUser->m_sNick.Data());
	}
	else
	{
		item->setText(0,tr("FILE"));
		item->setText(1,MessageSearchFile->m_sString.Data());
	}

	m_pSearchQueryList->Add(msg);
}

/** */
void DCHubSearch::slotReset()
{
	//LineEdit_SEARCH->setText("");
	Combobox_SEARCH->clearEdit();
	LineEdit_SEARCHSIZE->setText("0");
	ComboBox_SEARCHUNIT->setCurrentItem(0);
	ComboBox_SEARCHLIMIT->setCurrentItem(0);
	ComboBox_SEARCHTYPE->setCurrentItem(0);
	SpinBox_FREESLOTS->setValue(0);
	ListView_SEARCH->clear();

	if ( m_pSearchQueryList != 0 )
	{
		m_pSearchQueryList->Clear();
	}
}

/** */
void DCHubSearch::slotDoubleClickedSearchResult( QListViewItem * /*item*/ )
{
	QListViewItem * curitem;
	ulonglong size;
	QString localname;
	DCFileTool FileTool;

	curitem   = ListView_SEARCHRESULT->currentItem();
	size      = ((DC_QListViewItem*)curitem)->myvalue;
	localname = curitem->text(0);

	// add file as new source
	FileTool.AddFileSource( curitem->text(3).ascii(),
				curitem->text(5).ascii(),
				curitem->text(7).ascii(),
				(curitem->text(6)+curitem->text(0)).ascii(),
				localname.ascii(),
				"",
				"",
				eltFILE,
				size );

	return;
/*
	int i;
	CString s;
	CByteArray cba;
	QByteArray qba;

	DC_ListResult *item1 = (DC_ListResult*)item;

	QStoredDrag *sd = new QStoredDrag( "dchubsearch/searchresult", this );

	s = item1->text(0).ascii();
	if ( s.Data() != 0 )
		cba.Append( s.Data(), s.Length() );
	cba.Append("\0",1);

	s = s.setNum(item1->myvalue);
	if ( s.Data() != 0 )
		cba.Append( s.Data(), s.Length() );
	cba.Append("\0",1);

	for(i=2;i<=6;i++)
	{
		s = item1->text(i).ascii();
		if ( s.Data() != 0 )
			cba.Append( s.Data(), s.Length() );
		cba.Append("\0",1);
	}

	qba.assign((const char*)cba.Data(),(unsigned int)cba.Size());
	sd->setEncodedData(qba);

	QDragObject *d = sd;
	d->dragCopy();
*/
}

/** */
void DCHubSearch::slotRightButtonClickedSearchResult( QListViewItem * /*item*/, const QPoint &, int column )
{
	DCFileTool filetool;
	CString s;
	int id;
	ulonglong size;
        QPtrList<QListViewItem> selitems;
        QListViewItem * curitem, * item;
	bool isadmin;
	CXml xml;

	// user have select items ?
	selectedItems( ListView_SEARCHRESULT, selitems );

	// check if admin for admin-menu
	for ( isadmin = FALSE, curitem = selitems.first(); curitem; curitem = selitems.next() )
	{
		if ( g_pConnectionManager->IsAdmin(curitem->text(5).ascii(),curitem->text(7).ascii()) == FALSE )
		{
			isadmin = FALSE;
			break;
		}
		else
		{
			isadmin = TRUE;
		}
	}
	
	QPopupMenu *m,*msort;

	m = new QPopupMenu(this);

	DCMenuHandler::InsertMenu( m, emiDOWNLOAD, (selitems.count() > 0) );
	DCMenuHandler::InsertMenu( m, emiDOWNLOAD_TO, (selitems.count() > 0) );
	DCMenuHandler::InsertMenu( m, emiDOWNLOAD_AS, (selitems.count() > 0) );
	DCMenuHandler::InsertMenu( m, emiDOWNLOAD_IN, (selitems.count() > 0) );
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	DCMenuHandler::InsertMenu( m, emiBROWSE_USER_FILES, (selitems.count() > 0) );
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	DCMenuHandler::InsertMenu( m, emiCOPY_COLUMN_TO_CLIPBOARD, (selitems.count() > 0) );
	DCMenuHandler::InsertMenu( m, emiCOPY_ROW_TO_CLIPBOARD, (selitems.count() > 0) );
	DCMenuHandler::InsertMenu( m, emiCOPYDCFILELINK, (selitems.count() == 1) );
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	DCMenuHandler::InsertMenu( m, emiCONNECT_TO_HUB, (selitems.count() > 0) );
	DCMenuHandler::InsertMenu( m, emiADD_BOOKMARK, (selitems.count() > 0) );
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );

	// insert submenu
	msort = DCMenuHandler::InsertMenu( m, emisGROUP, TRUE );
	DCMenuHandler::InsertMenu( msort, emiDISABLE_GROUP, TRUE );
	DCMenuHandler::InsertMenu( msort, emiGROUP_BY_FILE, TRUE );
	DCMenuHandler::InsertMenu( msort, emiGROUP_BY_SIZE, TRUE );
	DCMenuHandler::InsertMenu( msort, emiGROUP_BY_NICK, TRUE );
	DCMenuHandler::InsertMenu( msort, emiGROUP_BY_HASH, TRUE );
	DCMenuHandler::InsertMenu( msort, emiGROUP_BY_SLOTS_FREE, TRUE );
	DCMenuHandler::InsertMenu( msort, emiGROUP_BY_HUB, TRUE );
	DCMenuHandler::InsertMenu( msort, emiGROUP_BY_PATH, TRUE );
	DCMenuHandler::InsertMenu( msort, emiGROUP_BY_HOST, TRUE );

	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	DCMenuHandler::InsertMenu( m, emiKICK, isadmin && (selitems.count() > 0) );
	DCMenuHandler::InsertMenu( m, emiFORCE_MOVE, isadmin && (selitems.count() > 0) );
	DCMenuHandler::InsertMenu( m, emiSEPARATOR );
	DCMenuHandler::InsertMenu( m, emiLOAD );
	DCMenuHandler::InsertMenu( m, emiSAVE, (m_pSearchResultList->Count() > 0) );

	id = m->exec(QCursor::pos());

	delete m;

	if ( id == emiCOPYDCFILELINK )
	{
		QClipboard *cb = QApplication::clipboard();
		s = "";

                for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
                {
			s += "DCFILE://";
			s += xml.ToUTF8(curitem->text(7).ascii()) + "?";
			s += "file=" + CHttp::Encode((curitem->text(6) + curitem->text(0)).ascii()) + "&";
			s += "size=" + CString().setNum(((DC_QListViewItem*)curitem)->myvalue) + "&";
			s += "nick=" + CHttp::Encode(curitem->text(3).ascii()) + "&";
			s += "hub="  + CHttp::Encode(curitem->text(5).ascii());
			
			if ( curitem->text(2) != "" )
			{
				s += "&";
				s += "hash="  + CHttp::Encode(curitem->text(2).ascii());
			}
		}

		cb->setText(s.Data());
	}
	else if ( (id == emiDOWNLOAD) || (id == emiDOWNLOAD_AS) || (id == emiDOWNLOAD_TO) )
	{
		QString localrootpath = "";
		QString localname = "";

		// select downloadfolder for all selected files
		if ( id == emiDOWNLOAD_TO )
		{
			localrootpath = QFileDialog::getExistingDirectory( "", this, "bdf", tr("Select download folder"), TRUE );

			if ( localrootpath == "" )
				return;
		}

                for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
                {
			size = ((DC_QListViewItem *)curitem)->myvalue;
			localname = curitem->text(0);

			if ( id == emiDOWNLOAD_AS )
			{
				localrootpath = QFileDialog::getSaveFileName( "", "", this, "bdf", tr("Select file for")+" "+localname );

				if ( localrootpath == "" )
					return;

				QFileInfo fi(localrootpath);
				localrootpath = fi.dirPath();
				localname     = fi.fileName();

				if ( (localrootpath=="") || (localname=="") )
					return;
			}

			// add transfer to the waitlist
			filetool.CheckFile( curitem->text(3).ascii(), curitem->text(5).ascii(), curitem->text(7).ascii(),
						(curitem->text(6)+curitem->text(0)).ascii(), localname.ascii(), "", localrootpath.ascii(), eltFILE,
						size );
		}

		// redraw the list
		ListView_SEARCHRESULT->triggerUpdate();
	}
	else if ( id == emiDOWNLOAD_IN )
	{
		QString localrootpath;
		QString localname;
		QString localpath;

		if ( !(curitem = selitems.first()) )
			return;

		size = ((DC_QListViewItem *)curitem)->myvalue;

		// all files need equal size
                for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
                {
			if ( size != ((DC_QListViewItem *)curitem)->myvalue )
				return;
		}

		if ( filetool.SelectFileSource( size, localname, localrootpath, localpath ) == FALSE )
		{
			return;
		}

                for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
                {
			// add transfer to the waitlist
			filetool.CheckFile( curitem->text(3).ascii(), curitem->text(5).ascii(), curitem->text(7).ascii(),
					  (curitem->text(6)+curitem->text(0)).ascii(), localname.ascii(), localpath.ascii(), localrootpath.ascii(), eltFILE,
					  size, TRUE);
		}

		// redraw the list
		ListView_SEARCHRESULT->triggerUpdate();
	}
	else if ( id == emiBROWSE_USER_FILES)
	{
		/** add transfer to the waitlist */
                for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
                {
			CString hubname(curitem->text(5).ascii());
			CString hubhost(curitem->text(7).ascii());

			if ( g_pConnectionManager->IsHubOnline(hubname,hubhost) == ehsNONE )
			{
				switch( QMessageBox::warning( this, tr("Filelist download"),
							      tr("Not connected to required hub!"),
							      tr("Connect"),
							      tr("Cancel"), 0, 0, 1 ) )
				{
					case 0:
						 g_pConnectionManager->Connect(hubname.Data(),hubhost.Data());
						 break;
					case 1:
						break;
				}
			}

			g_pTransferView->DLM_QueueAdd( curitem->text(3).ascii(), curitem->text(5).ascii(), curitem->text(7).ascii(),
						DC_USER_FILELIST_HE3, DC_USER_FILELIST_HE3, "", "", eltBUFFER,
						0, 0 );
		}
	}
	else if ( id == emiCOPY_COLUMN_TO_CLIPBOARD )
	{
		QString s="";
		QClipboard *cb = QApplication::clipboard();
		cb->setText("");

                for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
                {
			s += curitem->text(column) + "\n";
		}

		cb->setText(s);
	}
	else if ( id == emiCOPY_ROW_TO_CLIPBOARD )
	{
		int idx;
		QString s="";
		QClipboard *cb = QApplication::clipboard();
		cb->setText("");

                for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
                {
			for( idx = 0; idx < ListView_SEARCHRESULT->columns(); idx++ )
			{
				s += curitem->text(idx) + " ";
			}

			s += "\n";
		}

		cb->setText(s);
	}
	else if ( id == emiCONNECT_TO_HUB )
	{
		// connect to the hub
                for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
                {
			g_pConnectionManager->Connect( curitem->text(5).ascii(), curitem->text(7).ascii() );
		}
	}
	else if ( id == emiADD_BOOKMARK )
	{
		// bookmark this hub
                for ( curitem = selitems.first(); curitem; curitem = selitems.next() )
                {
			g_pHubListManager->AddBookmark( curitem->text(5), curitem->text(7), "" );
		}
	}
	else if ( id == emiDISABLE_GROUP )
	{
		GroupSearchResults( egsrtNONE );
	}
	else if ( id == emiGROUP_BY_FILE )
	{
		GroupSearchResults( egsrtFILE );
	}
	else if ( id == emiGROUP_BY_SIZE )
	{
		GroupSearchResults( egsrtSIZE );
	}
	else if ( id == emiGROUP_BY_HASH )
	{
		GroupSearchResults( egsrtHASH );
	}
	else if ( id == emiGROUP_BY_NICK )
	{
		GroupSearchResults( egsrtNICK );
	}
	else if ( id == emiGROUP_BY_SLOTS_FREE )
	{
		GroupSearchResults( egsrtSLOTS_FREE );
	}
	else if ( id == emiGROUP_BY_HUB )
	{
		GroupSearchResults( egsrtHUB );
	}
	else if ( id == emiGROUP_BY_PATH )
	{
		GroupSearchResults( egsrtPATH );
	}
	else if ( id == emiGROUP_BY_HOST )
	{
		GroupSearchResults( egsrtHOST );
	}
	else if ( id == emiKICK )
	{
		// TODO: need a patch to kick a user only one time
		QString message;

		if ( ! DCClient::GetOPKickMessage( message ) )
		{
			return;
		}

		// kick all user
		while( (item = selitems.first()) )
		{
			// remove this item from the list
			selitems.remove(item);

			// kick this user
			g_pConnectionManager->OPKick( item->text(5), item->text(7), item->text(3), message );

			// search for same entry and remove it
			for ( curitem = selitems.first(); curitem; )
			{
				if ( (curitem->text(3) == item->text(3)) &&
				     (curitem->text(5) == item->text(5)) &&
				     (curitem->text(7) == item->text(7)) )
				{
					selitems.remove(curitem);
					curitem = selitems.current();
				}
				else
				{
					curitem = selitems.next();
				}
			}
		}
	}
	else if ( id == emiFORCE_MOVE )
	{
		// TODO: need a patch to move a user only one time
		QString message, host;

		if ( ! DCClient::GetOPForceMoveMessage( message, host ) )
		{
			return;
		}

		// force move all user
		while( (item = selitems.first()) )
		{
			// remove this item from the list
			selitems.remove(item);

			// force move this user
			g_pConnectionManager->OPForceMove( item->text(5), item->text(7), item->text(3), message, host );

			// search for same entry and remove it
			for ( curitem = selitems.first(); curitem; )
			{
				if ( (curitem->text(3) == item->text(3)) &&
				     (curitem->text(5) == item->text(5)) &&
				     (curitem->text(7) == item->text(7)) )
				{
					selitems.remove(curitem);
					curitem = selitems.current();
				}
				else
				{
					curitem = selitems.next();
				}
			}
		}
	}
	else if ( id == emiSAVE )
	{
		CString s;
		CMessageSearchResult * msr = 0;

		QString filename = QFileDialog::getSaveFileName(
			"",
			"",
			this,
			tr("save file dialog"),
			tr("Choose a filename to save under") );

		if ( filename != "" )
		{
			QFile file(filename);

			if ( file.open( IO_WriteOnly ) )
			{
				while ( (msr=(CMessageSearchResult*)m_pSearchResultList->Next((CObject*)msr)) != 0 )
				{
					s  = "$SR ";
					s += msr->m_sNick + " ";
					s += msr->m_sFile + 0x05;
					s += CString().setNum(msr->m_nSize) + " ";
					s += CString().setNum(msr->m_nFreeSlot) + "/";
					s += CString().setNum(msr->m_nMaxSlot) + 0x05;
					s += msr->m_sHubName + " (";
					s += msr->m_sHubHost + ")" + 0x05 + "|";
					s += "\n";
					file.writeBlock(s.Data(),s.Length());
				}

				file.close();
			}
		}
	}
	else if ( id == emiLOAD )
	{
		QString s;
		CString m;
		int p;
		CObject * object;
		CMessageHandler MessageHandler;

		QString filename = QFileDialog::getOpenFileName(
			"",
			"",
			this,
			tr("open file dialog"),
			tr("Choose a file to open") );

		if ( filename != "" )
		{
			QFile file(filename);
			if ( file.open( IO_ReadOnly ) )
			{
				m_pSearchResultList->Clear();

				ListView_SEARCHRESULT->setUpdatesEnabled(FALSE);
				ListView_SEARCHRESULT->clear();

				while( file.readLine(s,4096) != -1 )
				{
					m = s.ascii();
					p = 0;

					if ( MessageHandler.Parse(&m,p,object=0) == DC_MESSAGE_SEARCHRESULT )
					{
						m_pSearchResultList->Add(object);
					}
				}

				file.close();

				GroupSearchResults( egsrtNONE );

				ListView_SEARCHRESULT->setUpdatesEnabled(TRUE);
				ListView_SEARCHRESULT->triggerUpdate();
				LCDNumber_RESULTS->display(ListView_SEARCHRESULT->childCount());
			}
		}
	}
}

/** */
void DCHubSearch::slotRightButtonClickedSearch( QListViewItem *, const QPoint &, int )
{
	// TODO: handle delete,edit searches (multiselection support)
	int id;
        QPtrList<QListViewItem> selitems;

	// user have select items ?
	selectedItems( ListView_SEARCH, selitems );

	QPopupMenu *m;

	m = new QPopupMenu(this);

	DCMenuHandler::InsertMenu( m, emiREMOVE, selitems.count() > 0 );
	DCMenuHandler::InsertMenu( m, emiCLEAR );

	id = m->exec(QCursor::pos());

	delete m;

	if ( id == emiREMOVE )
	{
		RemoveSelectedSearch(&selitems);
	}
	else if ( id == emiCLEAR )
	{
		if ( m_pSearchQueryList )
			m_pSearchQueryList->Clear();
		ListView_SEARCH->clear();
	}
}

/** */
void DCHubSearch::RemoveSelectedSearch( QPtrList<QListViewItem> * list )
{
        QPtrList<QListViewItem> selitems;
        DC_QSearchListViewItem * curitem;

	if ( !list )
	{
		if ( selectedItems( ListView_SEARCH, selitems ) == 0 )
		{
			return;
		}

		list = &selitems;
	}

	for ( curitem = (DC_QSearchListViewItem *)list->first(); curitem; curitem = (DC_QSearchListViewItem *)list->next() )
	{
		if ( m_pSearchQueryList )
			m_pSearchQueryList->Del((CObject*)curitem->p_data);
		ListView_SEARCH->takeItem(curitem);
		delete curitem;
	}
}
