/******************************************************************************
 *                    Internetting Cooperating Programmers
 * ----------------------------------------------------------------------------
 *
 *  ____    PROJECT
 * |  _ \  __ _ _ __   ___ ___ _ __ 
 * | | | |/ _` | '_ \ / __/ _ \ '__|
 * | |_| | (_| | | | | (_|  __/ |   
 * |____/ \__,_|_| |_|\___\___|_|   the IRC bot
 *
 * All files in this archive are subject to the GNU General Public License.
 *
 * $Source: /cvsroot/dancer/dancer/src/user.h,v $
 * $Revision: 1.1.1.1 $
 * $Date: 2000/11/13 02:42:50 $
 * $Author: holsta $
 * $State: dancer.c $
 * $Locker:  $
 *
 * ---------------------------------------------------------------------------
 *****************************************************************************/

#ifndef USER_H
#define USER_H

#include <sys/types.h>
#include <netinet/in.h>

#include "dancer.h"
#include "list.h"
#include "language.h"

/*
 * In general:
 *
 *  "user"   refers to a registered user
 *  "guest"  -           user on the channel
 *  "client" -           dcc chat connection
 */

/* I line = unrestricted, i line = restricted */
enum {
  PREFIX_NONE  = 0,      /* I line with ident */
  PREFIX_HAT   = '^',    /* I line with OTHER type ident (see RFC1413) */
  PREFIX_TILDE = '~',    /* I line without ident */
  PREFIX_PLUS  = '+',    /* i line with ident */
  PREFIX_EQUAL = '=',    /* i line with OTHER type ident */
  PREFIX_DASH  = '-',    /* i line without ident */
};

/* perform the actual delete after this time: */
#define USER_DELETE_TIMEOUT SECINMONTH 

typedef struct Msgstruct {
  struct Header h;
  char *nick;  /* Recipient */
  char *msg;   /* Message */
} itemmsg;

typedef struct UserFlags {
  unsigned char autoop;     /* autoop this user if autoop is enabled */
  unsigned char banprotect; /* banprotect this user */
  unsigned char linkbot;    /* linked bot */
  unsigned char regularbot; /* regular bot */
  unsigned char delete;     /* this user should get deleted */
  unsigned char stealth;    /* commands avoid spy link */
} uFlags;

typedef enum {
  FLAG_AUTOOP = 'A', /* automatically op this user if AUTOOP is enabled */
  FLAG_BANPROTECT, /* unban bans matching this user */
  FLAG_LINKBOT,    /* this is a bot that is capable of linking to us */
  FLAG_REGULARBOT, /* this is a bot that can't link with us */
  FLAG_DELETE,     /* delete this user after the specified number of timeout
		      days */
  FLAG_STEALTH,    /* commands doesn't go through spy link but are logged */
  FLAG_MAXFLAGS
} UserFlags;

typedef struct Userstruct {  /* Registered users */
  struct Header h;
  char     *nick;           /* Registered nick */
  itemlist *domainhead;     /* List of accounts */
  char     *passwd;         /* Password (crypted) */
  time_t   passwdtime;      /* Time of last change */

  char     *stat_bywhom;    /* who changed this user's stat latest */
  int      stat_lastdone;   /* last status change - what was done */
  time_t   stat_lastchange; /* last status change - when was it done */

  long     newsid;          /* the ID of the last news item showed to this
                               user */
  long     autonews;        /* for the news stuff */

  /* the user can change these himself: */
  char     *realname;       /* Realname/email */
  char     *comment;        /* Comment */
  LangNum  language;        /* prefered language */

  time_t   modified;        /* ANY modification */
  time_t   created;         /* time of first add */
  int      level;           /* User level */
  uFlags   flags;           /* User attributes */
  long     checksum;        /* the LOADED checksum */

  long     totaljoins;      /* number of joins, *not* part of checksum */
#ifdef HAVE_LIBFPL
  void *label;              /* list of labels FOR NON-JOINED USERS */
#endif
} itemuser;


#define USER_ADDED   1
#define USER_LOWERED 2
#define USER_RAISED  3

typedef struct Identstruct {  /* Identification data */
  struct Header h;
  struct Userstruct *user;
  struct Gueststruct *guest;
  struct Clientstruct *client;
#ifdef HAVE_LIBFPL
  void *label;     /* list of labels */
#endif
  int    level;
  bool   passed;            /* Password valid or invalid */
  bool   illegalname;       /* host name contains illegal character '*' */

  char   *host;             /* user@machine.domain (complete _with_ prefix) */
  char   *userdomain;       /* user@*.domain OR
			       user@127.0.0.* OR (if hostisp)
			       *@machine.domain */

  char   *phost;            /* parsed host 'machine.domain' */
  char   *pname;            /* parsed name 'user' */
  char   *puserdomain;      /* parsed domain '*.domain' */
  ulong  signature;         /* Hashed 2-signature */
  char   nick[NICKLEN+1];
} itemident;

#ifdef BAGDER
typedef struct itemfreecl { /* general file transfer client info */
  struct Header h;
  itemclient *client; /* point out the itemclient struct for this */
} itemfreecl;
#endif

typedef struct Gueststruct {  /* People on the channel */
  struct Header h;
  itemident *ident;           /* Identification */

  long     posts;             /* public posts counter */
  time_t   posttime;          /* timestamp of last posting */
  long     postsame;          /* counter of postings written the
                                 same or following second */
  long     ctcp;              /* counter of public CTCP message sent
                                 within the same 20 secs period */
  time_t   ctcptime;          /* timestamp of last CTCP */
  time_t   nicktime;          /* timestamp of last nick change */
  long     nickchanges;       /* counter of nick changes */
  long     warnings;          /* number of misbehaves, when this
                                 reached MAX_WARNINGS, even a high
                                 level user gets kicked */
  long     msgs;              /* number of messages stored */
  long     joins;             /* number of joins */
  time_t   jointime;          /* time of join */

  ulong    hashpost;          /* hash number of the written line */
  long     hashsame;          /* number of identical lines written */
  time_t   hashtime;          /* time of the first of the identical lines */

  time_t   kicktime;          /* Time of sticky kick */
  char    *kickmsg;           /* Sticky kick message */

  time_t   op_this_person;    /* [AUTOOP] When this time has come, the person
				 should get opped if it isn't already */

  struct {
    unsigned int chanop : 1;  /* Is channel operator */
    unsigned int voice  : 1;  /* Has user mode +v */
    unsigned int split  : 1;  /* This user has split */
    unsigned int kick   : 1;  /* Marked for kick */
    unsigned int kicked : 1;  /* Nick is sent to server for KICK */
    unsigned int bot    : 1;  /* linked or regular, this is a bot */
    unsigned int ircop  : 1;  /* ircop when he joined (we don't know if he
                                 gets oper while joined) */
#ifdef CONFMODE
    unsigned int confoper: 1; /* Is conference operator */
#endif
    unsigned int splitop : 1; /* Guest had +o when he split */
  } flags;
#ifdef BAGDER
  itemfreecl *freecl; /* "free client" which is a ftp-downloads, dcc sends and
                         more, should all be abandonned if the user leaves */
#endif
  long	   bans;			/* number of bans done */
  long	   kicks;			/* number of kicks done */
} itemguest;

typedef struct Clientstruct {  /* DCC chat/send clients */
  struct Header h;
  itemident *ident;         /* Identification */
  int      socket;
  int      status;

#define CL_NONE 0
#define CL_CONNECTED 1
#define CL_CONNECTING 2
#define CL_LISTENING 3
#define CL_LINKED 4

  time_t   buildtime;       /* Time when connection was established */
  time_t   lasttime;        /* Time it was used last */
  ulong    flags;           /* What kind of multicasts to listen to */
#define WALLCAST (1<<0)
#define SPYCAST (1<<1)
#define DEBUGCAST (1<<2)
#define REPORTCAST (1<<3)
#define CHATCAST (1<<4)

  bool     chatecho;        /* FALSE makes chat messages not get sent to the
			       sender, thus preventing the echo-mode feeling.
			       Default is TRUE for compatility. */

  char     linksort;        /* OUT_CHAT, OUT_SEND or OUT_GET */
  struct sockaddr_in addr;

  /* FILE TRANSFER STUFF: */

  int      file;            /* fd in use when sending or getting files */
  char     fileflags;       /* define file (transfer) status */
#define CLF_FILECOMPLETE 1  /* File is closed and stored */

  int      fileread;        /* number of bytes read from the file */
  int      filesent;        /* number of bytes sent */
  int      fileacked;       /* number of bytes ACKed */

  int      filebufn;        /* amount of data in the buffer */
  char     *filebuffer;     /* allocated memory area for transfers */
#define CLF_FILEBUFFER 4096 /* the bigger the faster */
} itemclient;

typedef struct Linkmsgstruct {
  short flags;
  short type;
  char *msg;
} itemlnmsg;

typedef struct Linkstruct { /* Link bots */
  struct Header h;
  itemuser *user;
  char *name;        /* Nickname (doesn't have to be unique) */
  int bottype;       /* Bot type */
  int protoversion;  /* Protocol version */
  bool passed;       /* Authenticated or not */
  bool new;          /* Newly added? */
  ulong linkvalue;   /* For authentication */
  ulong tmpvalue;
  time_t buildtime;
  time_t lasttime;
  time_t commontime;
  struct sockaddr_in addr;

  unsigned long mymsgid;     /* Last msgid sent to this link */
  unsigned long yourmsgid;   /* Last msgid recieved from this link */
  struct Msgstruct RemoteHead;

  /* The following values are for the reliable message parsing */
  float rtt_rtt;
  float rtt_srtt;
  float rtt_rttdev;
  short rtt_nrexmt;
  short rtt_currto;
  short rtt_nxtrto;
  struct timeval time_start;
  struct timeval time_stop;
  time_t expires;
} itemlink;

typedef struct Splitstruct {  /* List of split servers */
  struct Header h;
  time_t splittime;
  char *servers;
  itemguest *wholeft; /* the people who left */
  bool heal;  /* set to TRUE whenever someone has joined again from this server */
} itemsplit;


int DoCountJoins(itemguest *);
/* Netsplit */
void AddSplitter(itemguest *, char *);
itemguest *SplitterReturns(char *, char *);
void ScanSplitServers(bool);
/* Users */
void FreeUser(void *);
void UserCleanup(void);
void UserInit(void);
void UserReload(char *name);
void SavePasswd(void);
bool UserLoad(char *);
itemuser *AddUser(char *);
time_t LatestUserUpdate(void);
itemuser *FindUser(char *nick, char *userhost);
itemuser *FindUserByNick(char *);
/* Guests */
bool AddGuest(char *, char *, bool, bool, bool, itemguest **);
void RemoveGuest(itemguest *);
void DeleteGuests(void);
void NewGuest_Who(char *);
itemguest *FindNick(char *);
itemguest *FindHost(char *host);
itemguest *FindSplitNick(char *);
void ChangeGuest(char *, char *);
/* Clients */
itemclient *AddClient(char *, char *, itemuser *, int);
void RemoveClient(itemclient *);
void DeleteClients(void);
itemclient *FindClientByNick(char *);
char IsPrefix(char *name);
char *GetPrefix(char prefix);
char *WriteUserHead(char *, size_t, itemuser *);
bool UserSave(void);
char *MakePassword(char *);
bool CheckPassword(char *, char *);
itemlink *AddLink(char *, itemuser *, ulong, ushort);
itemlink *RemoveLink(itemlink *);
itemlink *FindLink(ulong, ushort);
itemlink *FindLinkByNick(char *);

char *VerboseFlags(uFlags *flags);
bool SplitHost(char *userhost, /* user@machine.host.domain */
               char *parsedhost,
               char *parseddomain,
               char *parseduser);
int BadUsername(char *name);
int ID2User(int id, itemuser **user, itemguest **guest, itemclient **client);
int User2ID(char *nick, char *flags);
itemident *ID2Ident(int id);
#ifdef HAVE_LIBFPL
/* itemuserlabel **ID2Label(int id); */
#endif
bool IsAutoOp(itemguest *g);
void CheckForAutoOp(void);
void NickReport(char *from, char *nick);
bool HostISP(char *checkthis);
bool DontSiteBan(char *checkthis);

#endif /* USER_H */
