/* Song.m - this file is part of Cynthiune
 *
 * Copyright (C) 2002, 2003 Wolfgang Sourdeau
 *
 * Author: Wolfgang Sourdeau <wolfgang@contre.com>
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#import <Foundation/Foundation.h>

#import "Format.h"
#import "FormatTester.h"
#import "Song.h"

#if defined (NeXT_PDO) || defined (__APPLE__)
#define _(X) NSLocalizedString (X, nil)
#define NSDebugLog
#endif

@implementation Song : NSObject

+ (Song *) songWithFilename: (NSString *) aFilename
{
  return [[self alloc] initWithFilename: aFilename];
}

- (id) init
{
  if ((self = [super init]))
    {
      filename = nil;
      formatClass = nil;
      _infosRead = NO;
    }

  return self;
}

- (id) initWithFilename: (NSString *) aFilename
{
  self = [self init];
  [self setFilename: aFilename];

  return self;
}

- (id) openStreamForSong
{
  id <Format> stream;
  FormatTester *formatTester;

  stream = nil;

  if (filename
      && [[NSFileManager defaultManager] fileExistsAtPath: filename])
    {
      if (!formatClass)
        {
          formatTester = [FormatTester formatTester];
          formatClass = [formatTester classForFile: filename];
        }

      if (formatClass)
        {
          stream = [[formatClass alloc] init];

          if (![stream streamOpen: filename])
            {
              [stream release];
              stream = nil;
            }
        }
    }

  return stream;
}

- (void) readInfos
{
  id <Format> stream;

  if (!_infosRead)
    {
      stream = [self openStreamForSong];

      if (stream)
        {
          channels = [stream readChannels];
          rate = [stream readRate];
          duration = [stream readDuration];

          title = [stream readTitle];
          [title retain];
          genre = [stream readGenre];
          [genre retain];
          artist = [stream readArtist];
          [artist retain];
          album = [stream readAlbum];
          [album retain];
          trackNumber = [stream readTrackNumber];
          [trackNumber retain];

          [stream streamClose];
          [stream release];
        }
      else
        {
          title = [NSString stringWithFormat: _(@"[not found: %@]"),
                            filename];
          [title retain];
        }

      _infosRead = YES;
    }
}

- (void) setFilename: (NSString *) aFilename
{
  if (filename)
    [filename release];
  filename = aFilename;
  if (filename)
    [filename retain];
}

- (NSString *) filename
{
  NSString *string;

  if (filename)
    string = [NSString stringWithString: [filename lastPathComponent]];
  else
    string = nil;

  return string;
}

- (NSString *) fullFilename
{
  NSString *string;

  if (filename)
    string = [NSString stringWithString: filename];
  else
    string = nil;

  return string;
}

- (unsigned int) channels
{
  [self readInfos];

  return channels;
}

- (unsigned long) rate
{
  [self readInfos];

  return rate;
}

- (unsigned int) duration
{
  return duration;
}

- (NSString *) title
{
  [self readInfos];

  return [NSString stringWithString: title];
}

- (NSString *) artist
{
  [self readInfos];

  return [NSString stringWithString: artist];
}

- (NSString *) album
{
  [self readInfos];

  return [NSString stringWithString: album];
}

- (NSString *) genre
{
  [self readInfos];

  return [NSString stringWithString: genre];
}

- (NSString *) trackNumber
{
  [self readInfos];

  return [NSString stringWithString: trackNumber];
}

- (NSString *) displayRepresentation
{
  NSString *displayString;

  displayString = [[NSString alloc] init];
  [self readInfos];

  if ([trackNumber length])
    displayString = [displayString stringByAppendingFormat: @"%s.",
                                   [trackNumber cString]];
  if ([trackNumber length] && [artist length])
    displayString = [displayString stringByAppendingString: @" "];
  if ([artist length])
    displayString = [displayString stringByAppendingFormat: @"%s",
                                   [artist cString]];
  if ([artist length] && [title length])
    displayString = [displayString stringByAppendingString: @" - "];
  if ([title length])
    displayString = [displayString stringByAppendingFormat: @"%s",
                                   [title cString]];
  if ([title length] && [album length])
    displayString = [displayString stringByAppendingString: @" "];
  if ([album length])
    displayString = [displayString stringByAppendingFormat: @"(%s)",
                                   [album cString]];

  if (![displayString length])
    displayString = [displayString stringByAppendingFormat: @"[%s]",
                                   [[self filename] cString]];

  return displayString;
}

- (BOOL) isSeekable
{
  BOOL result;
  id <Format> stream = [self openStreamForSong];

  if (stream)
    {
      result = [stream isSeekable];
      [stream streamClose];
      [stream release];
    }
  else
    result = NO;

  return result;
}

@end
