/* PlaylistController.m - this file is part of Cynthiune
 *
 * Copyright (C) 2002-2004  Wolfgang Sourdeau
 *
 * Author: Wolfgang Sourdeau <wolfgang@contre.com>
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#import <AppKit/AppKit.h>

#import "FormatTester.h"
#import "Preference.h"
#import "GeneralPreference.h"
#import "InfoDisplayController.h"
#import "Player.h"
#import "Playlist.h"
#import "PlaylistController.h"
#import "Song.h"
#import "TableViewController.h"

#if defined (NeXT_PDO) || defined (__APPLE__)
#define _(X) NSLocalizedString (X, nil)
#define NSDebugLog
#endif

@implementation PlaylistController : NSObject

- (id) init
{
  if ((self = [super init]))
    {
      player = nil;
      playlist = [[Playlist alloc] init];
      [playlist loadFromDefaults];
      [player setDelegate: self];
      playlistFilename = nil;
      timer = nil;
      timerShouldBeReset = NO;
    }

  return self;
}

- (void) _initButtonImages
{
  [previousButton setImage: [NSImage imageNamed: @"previous"]];
  [previousButton setAlternateImage: [NSImage imageNamed: @"previous-pushed"]];
  [playButton setImage: [NSImage imageNamed: @"play"]];
  [playButton setAlternateImage: [NSImage imageNamed: @"play-pushed"]];
  [pauseButton setImage: [NSImage imageNamed: @"pause"]];
  [pauseButton setAlternateImage: [NSImage imageNamed: @"pause-pushed"]];
  [stopButton setImage: [NSImage imageNamed: @"stop-pushed"]];
  [stopButton setAlternateImage: [NSImage imageNamed: @"stop-pushed"]];
  [nextButton setImage: [NSImage imageNamed: @"next"]];
  [nextButton setAlternateImage: [NSImage imageNamed: @"next-pushed"]];
  [ejectButton setImage: [NSImage imageNamed: @"eject"]];
  [ejectButton setAlternateImage: [NSImage imageNamed: @"eject-pushed"]];
  [muteButton setImage: [NSImage imageNamed: @"nomute"]];
  [muteButton setAlternateImage: [NSImage imageNamed: @"mute"]];
  [repeatButton setImage: [NSImage imageNamed: @"repeat-off"]];
  [repeatButton setAlternateImage: [NSImage imageNamed: @"repeat-on"]];
  [shuffleButton setImage: [NSImage imageNamed: @"shuffle-off"]];
  [shuffleButton setAlternateImage: [NSImage imageNamed: @"shuffle-on"]];
}

- (void) _initPlaylistPopups
{
  float delta;
  NSRect frameBefore, frameAfter;

  frameBefore = [addPopup frame];
  [addPopup removeAllItems];
  [addPopup addItemWithTitle: _(@"Add")];
  [addPopup addItemWithTitle: _(@"+ dir")];
  [addPopup selectItemAtIndex: 0];
  [addPopup sizeToFit];
  frameAfter = [addPopup frame];
  delta = frameAfter.size.width - frameBefore.size.width;

  frameBefore = [removePopup frame];
  [removePopup removeAllItems];
  [removePopup addItemWithTitle: _(@"Remove")];
  [removePopup addItemWithTitle: _(@"- all")];
  [removePopup selectItemAtIndex: 0];
  [removePopup sizeToFit];
  frameAfter = [removePopup frame];
  frameAfter.origin.x += delta;
  [removePopup setFrame: frameAfter];
  delta += frameAfter.size.width - frameBefore.size.width;

  frameBefore = [savePopup frame];
  [savePopup removeAllItems];
  [savePopup addItemWithTitle: _(@"Save")];
  [savePopup addItemWithTitle: _(@"... as...")];
  [savePopup selectItemAtIndex: 0];
  [savePopup sizeToFit];
  frameAfter = [savePopup frame];
  frameAfter.origin.x += delta;
  [savePopup setFrame: frameAfter];
}

- (void) awakeFromNib
{
  [tableViewController setDelegate: self];
  [tableViewController setPlaylist: playlist];

  [infoDisplayController hide];

  [progressSlider setEnabled: NO];
  [progressSlider setMinValue: 0];
  [progressSlider setIntValue: 0];

  [self _initButtonImages];
  [self _initPlaylistPopups];

  [pauseButton setEnabled: NO];
  [stopButton setEnabled: NO];
}

- (void) toggleMute: (id) sender
{
  if (player)
    [player setMuted: [muteButton state]];
}

- (void) toggleShuffle: (id) sender
{
  [playlist setShuffle: ![playlist shuffle]];
}

- (void) toggleRepeat: (id) sender
{
  [playlist setRepeat: ![playlist repeat]];
}

- (void) _initializePlayer
{
  Class playerClass;

  playerClass = [[GeneralPreference instance] preferredPlayerClass];
  player = [[playerClass alloc] init];
  [player setDelegate: self];
  [player setMuted: [muteButton state]];
}

- (void) _setPlayerSong: (Song *) aSong
{
  [player setSong: aSong];
  [progressSlider setIntValue: 0];
  [progressSlider setMaxValue: [aSong duration]];
  [progressSlider setEnabled: [aSong isSeekable]];
}

- (void) _updatePlayer: (BOOL) force
{
  Song *currentSong;
  int currentSongNumber;

  if (!player)
    NSLog (@"player uninitialized");
  currentSongNumber = [playlist currentSongNumber];
  currentSong = [playlist songAtIndex: currentSongNumber];
  if (force || currentSong != [player song])
    [self _setPlayerSong: currentSong];
  currentSong = [player song];
  if ([player isRunning] && currentSong)
    {
      [infoDisplayController updateInfoFieldsFromSong: currentSong];
      [infoDisplayController setTimerFromSeconds: 0];
    }
  [tableViewController updateView];
}

- (BOOL) _fileIsAcceptable: (NSString *) fileName
{
  BOOL answer;
  NSDictionary *fileAttributes;
  NSFileManager *aMgr;

  aMgr = [NSFileManager defaultManager];

  fileAttributes = [aMgr fileAttributesAtPath: fileName traverseLink: NO];
  while ([aMgr fileExistsAtPath: fileName]
         && [[fileAttributes fileType]
              isEqualToString: NSFileTypeSymbolicLink])
    {
      fileName = [aMgr pathContentOfSymbolicLinkAtPath: fileName];
      fileAttributes = [aMgr fileAttributesAtPath: fileName traverseLink: NO];
    }

  answer = ([aMgr fileExistsAtPath: fileName]
            && [[fileAttributes fileType] isEqualToString: NSFileTypeRegular]);

  return answer;
}

- (void) _addFileIfAcceptable: (NSString *) filename
{
  FormatTester *formatTester;
  Song *newSong;

  if ([self _fileIsAcceptable: filename])
    {
      if ([filename hasSuffix: @"pls"]
          || [filename hasSuffix: @"m3u"])
        [playlist loadFromFile: filename];
      else
        {
          formatTester = [FormatTester formatTester];
          if ([formatTester fileType: filename] > -1)
            {
              newSong = [Song songWithFilename: filename];
              [playlist appendSong: newSong];
            }
        }
    }
}

- (void) _addSongsFromArray: (NSArray *) songArray
                    withDir: (NSString *) aDirectory
{
  NSString *filename, *songFilename;
  int max, count;

  max = [songArray count];
  for (count = 0; count < max; count++)
    {
      songFilename = [songArray objectAtIndex: count];
      filename = (aDirectory)
        ? [aDirectory stringByAppendingPathComponent: songFilename]
        : songFilename;
      [self _addFileIfAcceptable: filename];
    }
}

- (void) _addSongsFromOPanel: (NSOpenPanel*) oPanel
{
  [self _addSongsFromArray: [oPanel filenames] withDir: nil];
  [tableViewController updateView];
  [playlist saveToDefaults];
}

- (void) _oPanelDidEnd: (NSOpenPanel *) oPanel
            returnCode: (int) result
           contextInfo: (void *) contextInfo
{
  if (result == NSOKButton)
    {
      [[NSUserDefaults standardUserDefaults] setObject: [oPanel directory]
                                             forKey:
                                             @"NSDefaultOpenDirectory"];
      [self _addSongsFromOPanel: oPanel];
    }
}

- (void) _addDirSongsFromOPanel: (NSOpenPanel *) oPanel
{
  int count, max;
  NSArray *dirs;
  NSDirectoryEnumerator *dirEnum;
  NSString *dir;

  dirs = [oPanel filenames];
  max = [dirs count];

  for (count = 0; count < max; count++)
    {
      dir = [dirs objectAtIndex: count];
#ifdef GNUSTEP
      dirEnum = [[NSDirectoryEnumerator alloc]
                  initWithDirectoryPath: dir
                  recurseIntoSubdirectories: YES
                  followSymlinks: NO
                  justContents: NO];
#else
      dirEnum = [[NSFileManager defaultManager] enumeratorAtPath: dir];
#endif
      [self _addSongsFromArray: [dirEnum allObjects]
            withDir: dir];
    }

  [tableViewController updateView];
  [playlist saveToDefaults];
}

- (void) _dirOPanelDidEnd: (NSOpenPanel *) oPanel
               returnCode: (int) result
              contextInfo: (void *) contextInfo
{
  if (result == NSOKButton)
    {
      [[NSUserDefaults standardUserDefaults] setObject: [oPanel directory]
                                             forKey:
                                             @"NSDefaultOpenDirectory"];
      [self _addDirSongsFromOPanel: oPanel];
    }
}

- (void) _runOpenPanelWithDidEndSelector: (SEL) selector
{
  NSOpenPanel *oPanel;
  NSString *openDir;
  NSMutableArray *extensions;
  FormatTester *formatTester;

  openDir = [[NSUserDefaults standardUserDefaults]
              stringForKey: @"NSDefaultOpenDirectory"];

  oPanel = [NSOpenPanel openPanel];
  [oPanel setAllowsMultipleSelection: YES];
  [oPanel setCanChooseDirectories: NO];
  [oPanel setTitle: _(@"Add music files...")];

  formatTester = [FormatTester formatTester];
  extensions = [NSMutableArray arrayWithObjects: @"m3u", @"pls", nil];
  [extensions addObjectsFromArray: [formatTester acceptedFileExtensions]];
  [oPanel beginSheetForDirectory: openDir
          file: nil
          types: extensions
          modalForWindow: [NSApp mainWindow]
          modalDelegate: self
          didEndSelector: selector
          contextInfo: nil];
}

- (void) _runDirOpenPanel
{
  NSString *openDir;
  NSOpenPanel *oPanel;

  openDir = [[NSUserDefaults standardUserDefaults]
              stringForKey: @"NSDefaultOpenDirectory"];

  oPanel = [NSOpenPanel openPanel];
  [oPanel setAllowsMultipleSelection: YES];
  [oPanel setCanChooseDirectories: YES];
  [oPanel setCanChooseFiles: NO];
  [oPanel setTitle: _(@"Add a music folder...")];
  [oPanel beginSheetForDirectory: openDir
          file: nil
          types: nil
          modalForWindow: [NSApp mainWindow]
          modalDelegate: self
          didEndSelector: @selector(_dirOPanelDidEnd:returnCode:contextInfo:)
          contextInfo: nil];
}

- (void) addSong: (id) sender
{
  NSString *title;

  title = [sender titleOfSelectedItem];

  if ([title isEqualToString: _(@"Add")])
    [self _runOpenPanelWithDidEndSelector: @selector(_oPanelDidEnd:returnCode:contextInfo:)];
  else if ([title isEqualToString: _(@"+ dir")])
    [self _runDirOpenPanel];
}

- (void) removeSong: (id) sender
{
  Song *currentSong;
  int currentSongNumber;
  NSString *title;
  NSArray *selection;

  title = [sender titleOfSelectedItem];

  if ([title isEqualToString: _(@"Remove")])
    {
      selection = [tableViewController getSelection];
      if ([selection count])
        {
          currentSong = [player song];
          currentSongNumber = [playlist songIndex: ([player isRunning])
                                        ? currentSong
                                        : [selection objectAtIndex: 0]];
          [playlist selectSongNumber: currentSongNumber];
          if ([selection containsObject: currentSong])
            [playlist nextSong];
          [playlist removeArrayOfSongs: selection];
        }
      [playlist saveToDefaults];
    }
  else if ([title isEqualToString: _(@"- all")])
    {
      [playlist removeAll];
      [playlist saveToDefaults];
    }

  [self _updatePlayer: NO];
}

- (void) _sPanelDidEnd: (NSSavePanel *) sPanel
            returnCode: (int) result
           contextInfo: (NSString *) extension
{
  NSString *saveDir, *filename;

  if (result == NSOKButton)
    {
      saveDir = [sPanel directory];
      [[NSUserDefaults standardUserDefaults] setObject: saveDir
                                             forKey:
                                               @"DefaultPlaylistDirectory"];

      if (playlistFilename)
        [playlistFilename release];
      playlistFilename = [[[sPanel filename] lastPathComponent]
                           stringByDeletingPathExtension];
      [playlistFilename retain];

      filename = [playlistFilename stringByAppendingPathExtension: extension];
      [playlist
        saveToFile: [saveDir stringByAppendingPathComponent: filename]];
    }
}

- (void) _runSaveListPanelWithExtension: (NSString *) extension
                             andSaveDir: (NSString *) saveDir
{
  NSSavePanel *sPanel;

  sPanel = [NSSavePanel savePanel];
  [sPanel setTitle: _(@"Save playlist as...")];
  [sPanel setRequiredFileType: extension];
  [sPanel setExtensionHidden: YES];
  [sPanel beginSheetForDirectory: saveDir
          file: playlistFilename
          modalForWindow: [NSApp mainWindow]
          modalDelegate: self
          didEndSelector: @selector (_sPanelDidEnd:returnCode:contextInfo:)
          contextInfo: extension];
}

- (void) saveList: (id) sender
{
  NSString *extension, *filename, *saveDir;

  extension = [[GeneralPreference instance] preferredPlaylistFormat];
  saveDir = [[NSUserDefaults standardUserDefaults]
              stringForKey: @"DefaultPlaylistDirectory"];
  if (!saveDir || [saveDir isEqualToString: @""])
    saveDir = [[NSUserDefaults standardUserDefaults]
                stringForKey: @"NSDefaultOpenDirectory"];

  if (!playlistFilename
      || [[sender titleOfSelectedItem] isEqualToString: _(@"... as...")])
    [self _runSaveListPanelWithExtension: extension
          andSaveDir: saveDir];
  else
    {
      filename = [playlistFilename stringByAppendingPathExtension: extension];
      [playlist
        saveToFile: [saveDir stringByAppendingPathComponent: filename]];
    }
}

- (void) addSongFromNSApp: (id) sender
{
  [self _runOpenPanelWithDidEndSelector:
          @selector(_oPanelDidEnd:returnCode:contextInfo:)];
}

- (BOOL) openSongFromNSApp: (Song *) aSong
{
  BOOL result;
  FormatTester *formatTester;

  formatTester = [FormatTester formatTester];
  result = YES;

  if ([formatTester fileType: [aSong fullFilename]])
    {
      [playlist appendSong: aSong];
      [tableViewController updateView];
    }
  else
    result = NO;

  return result;
}

- (void) _realStartPlaying
{
  if ([playlist count])
    {
      if ([playlist currentSongNumber] < 0)
        [playlist firstSong];
      if ([playlist shuffle])
        [playlist reShuffle];
      if (!player)
        [self _initializePlayer];
      [self _updatePlayer: NO];
      [player startPlayLoop];
    }
}

- (void) _startOPanelDidEnd: (NSOpenPanel *) oPanel
                 returnCode: (int) result
                contextInfo: (void *) contextInfo
{
  if (result == NSOKButton)
    {
      [[NSUserDefaults standardUserDefaults] setObject: [oPanel directory]
                                             forKey:
                                             @"NSDefaultOpenDirectory"];
      [self _addSongsFromOPanel: oPanel];
      [self _realStartPlaying];
    }
}

- (void) startPlaying: (id) sender
{
  if (![playlist count])
    [self _runOpenPanelWithDidEndSelector: @selector(_startOPanelDidEnd:returnCode:contextInfo:)];
  [self _realStartPlaying];
}

- (void) pausePlayer: (id) sender
{
  if (player && [player isRunning])
    [player setPaused: ![player paused]];
}

- (void) prevSong: (id) sender
{
  int currentSongNumber;

  currentSongNumber = [playlist currentSongNumber];
  [playlist prevSong];
  if (player
      && currentSongNumber != [playlist currentSongNumber])
    [self _updatePlayer: NO];
  else
    [tableViewController updateView];
}

- (void) nextSong: (id) sender
{
  int currentSongNumber;

  currentSongNumber = [playlist currentSongNumber];
  [playlist nextSong];
  if (player
      && currentSongNumber != [playlist currentSongNumber])
    [self _updatePlayer: NO];
  else
    [tableViewController updateView];
}

- (void) stopPlaying: (id) sender
{
  [player stopPlayLoop];
  [player release];
  player = nil;
}

- (void) _ejectOPanelDidEnd: (NSOpenPanel *) oPanel
                 returnCode: (int) result
                contextInfo: (void *) contextInfo
{
  if (result == NSOKButton)
    {
      [[NSUserDefaults standardUserDefaults] setObject: [oPanel directory]
                                             forKey:
                                             @"NSDefaultOpenDirectory"];
      if (player && [player isRunning])
        [player stopPlayLoop];
      [playlist removeAll];
      [self _addSongsFromOPanel: oPanel];

      if ([playlist count])
        {
          [playlist firstSong];
          if ([playlist shuffle])
            [playlist reShuffle];
          if (!player)
            [self _initializePlayer];
          [self _updatePlayer: NO];
          [player startPlayLoop];
        }
    }
}

- (void) eject: (id) sender
{
  [self _runOpenPanelWithDidEndSelector: @selector(_ejectOPanelDidEnd:returnCode:contextInfo:)];
}

/* when the timer field is used to display the total amound of seconds in the
   selection... */
- (void) _resetTimeDisplay: (NSTimer *) aTimer
{
  BOOL *passedOnce;

  passedOnce = [[aTimer userInfo] pointerValue];
  if (!*passedOnce)
    *passedOnce = YES;
  else
    {
      [timer invalidate];
      timer = nil;
      [infoDisplayController resetTimerField];
      timerShouldBeReset = NO;
    }
}

- (void) _updateDisplayResetTimer
{
  static BOOL passedOnce;

  passedOnce = NO;
  if (timer)
    [timer invalidate];
  timer = [NSTimer scheduledTimerWithTimeInterval: 1
                   target: self
                   selector: @selector (_resetTimeDisplay:)
                   userInfo: [NSValue valueWithPointer: &passedOnce]
                   repeats: YES];
  [timer fire];
}

- (void) _updateSelectionTimeDisplay
{
  NSArray *selection;
  unsigned int totalSeconds, count, max;

  totalSeconds = 0;
  selection = [tableViewController getSelection];
  max = [selection count];

  for (count = 0; count < max; count++)
    totalSeconds += [[selection objectAtIndex: count] duration];

  [infoDisplayController setTimerFromTotalSeconds: totalSeconds];
  timerShouldBeReset = YES;

  if (![player isRunning])
    [self _updateDisplayResetTimer];
}

/* tableview delegate */
- (void) tableDoubleClick: (NSNotification*) aNotification
{
  if (!player)
    [self _initializePlayer];
  [self _updatePlayer: YES];
  if (![player isRunning])
    [self startPlaying: self];
}

- (void) tableSelectionDidChange: (NSNotification*) aNotification
{
  if (![player isRunning])
    [playlist selectSongNumber: [tableViewController getSelectedRow]];
}

- (void) tableSelectionIsChanging: (NSNotification*) aNotification
{
  [self _updateSelectionTimeDisplay];
}

- (void) songCursorChange: (id) sender
{
  unsigned int time;

  time = [sender intValue];
  [infoDisplayController setTimerFromSeconds: time];
  [player seek: time];
}

- (void) _updateTimeDisplay
{
  unsigned int time;

  if (timerShouldBeReset)
    {
      [infoDisplayController resetTimerField];
      timerShouldBeReset = NO;
    }
  time = [player timer];
  [infoDisplayController setTimerFromSeconds: time];
  [progressSlider setIntValue: time];
}

- (void) _startTimeTimer
{
  if (timer)
    [timer invalidate];
  timer = [NSTimer scheduledTimerWithTimeInterval: 1
                   target: self
                   selector: @selector (_updateTimeDisplay)
                   userInfo: nil
                   repeats: YES];
  [timer fire];
}

- (void) _stopTimeTimer
{
  [timer invalidate];
  timer = nil;
}

/* as a player delegate... */
- (void) playerPlaying: (NSNotification*) aNotification
{
  int currentSongNumber;
  Song *currentSong;

  currentSongNumber = [playlist currentSongNumber];
  currentSong = [playlist songAtIndex: currentSongNumber];
  [self _startTimeTimer];
  [infoDisplayController show];
  [infoDisplayController updateInfoFieldsFromSong: currentSong];
  [infoDisplayController setTimerFromSeconds: 0];
  [progressSlider setMaxValue: [currentSong duration]];

  [playButton setImage: [NSImage imageNamed: @"play-pushed"]];
  [playButton setEnabled: NO];

  [pauseButton setImage: [NSImage imageNamed: @"pause"]];
  [pauseButton setEnabled: YES];

  [stopButton setImage: [NSImage imageNamed: @"stop"]];
  [stopButton setAlternateImage: [NSImage imageNamed: @"stop-pushed"]];
  [stopButton setEnabled: YES];

  [tableViewController selectRow: [playlist songIndex: currentSong]];
}

- (void) playerPaused: (NSNotification*) aNotification
{
  [self _stopTimeTimer];
  [pauseButton setImage: [NSImage imageNamed: @"pause-pushed"]];
}

- (void) playerResumed: (NSNotification*) aNotification
{
  [self _startTimeTimer];
  [pauseButton setImage: [NSImage imageNamed: @"pause"]];
}

- (void) playerStopped: (NSNotification*) aNotification
{
  [self _stopTimeTimer];

  [playButton setImage: [NSImage imageNamed: @"play"]];
  [playButton setAlternateImage: [NSImage imageNamed: @"play-pushed"]];
  [playButton setEnabled: YES];

  [pauseButton setImage: [NSImage imageNamed: @"pause"]];
  [pauseButton setEnabled: NO];

  [stopButton setImage: [NSImage imageNamed: @"stop-pushed"]];
  [stopButton setEnabled: NO];
  [progressSlider setEnabled: NO];
  [progressSlider setMaxValue: 0];

  [infoDisplayController hide];
  [tableViewController updateView];

  [player release];
  player = nil;
}

- (void) playerSongEnded: (NSNotification*) aNotification
{
  int currentSongNumber;

  currentSongNumber = [playlist currentSongNumber];
  [playlist nextSong];
  if ([playlist repeat])
    [self _updatePlayer: YES];
  else
    {
      if (currentSongNumber == [playlist currentSongNumber])
        {
          [playlist firstSong];
          [player stopPlayLoop];
          [player release];
          player = nil;
        }
      else
        [self _updatePlayer: NO];
    }
}

/* Services */
- (id) validRequestorForSendType: (NSString *)sendType
                      returnType: (NSString *)returnType
{
  return ((sendType
           && [sendType isEqual: NSFilenamesPboardType]
           && !returnType)
          ? self
          : nil);
}

- (BOOL) writeSelectionToPasteboard: (NSPasteboard*) pboard
                              types: (NSArray*) types
{
  NSArray *declaredType, *filenames;
  BOOL answer;

  if ([types containsObject: NSFilenamesPboardType])
    {
      declaredType = [NSArray arrayWithObject: NSFilenamesPboardType];
      [pboard declareTypes: declaredType owner: self];
      filenames = [tableViewController getSelectionAsFilenames];
      answer = [pboard setPropertyList: filenames
                       forType: NSFilenamesPboardType];
    }
  else
    answer = NO;

  return answer;
}

@end
