/* InfoDisplayController.m - this file is part of Cynthiune
 *
 * Copyright (C) 2002-2004  Wolfgang Sourdeau
 *
 * Author: Wolfgang Sourdeau <wolfgang@contre.com>
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#import <AppKit/AppKit.h>

#import "FormatTester.h"
#import "InfoDisplayController.h"
#import "Song.h"
#import "utils.h"

#if defined (NeXT_PDO) || defined (__APPLE__)
#define _(X) NSLocalizedString (X, nil)
#define NSDebugLog
#endif

@implementation InfoDisplayController : NSObject

- (void) awakeFromNib
{
  NSFont *myFont;

  myFont = [NSFont boldSystemFontOfSize: 12.0];
  [songTitleField setFont: myFont];
  myFont = [NSFont systemFontOfSize: 12.0];
  [songTimerField setFont: myFont];
  [songNumberField setFont: myFont];
  [songArtistField setFont: myFont];
  [songAlbumField setFont: myFont];

  [self hide];
}

- (InfoDisplayController *) init
{
  if ((self = [super init]))
    {
      reverseTimer = NO;
      totalTime = 0;
    }

  return self;
}

- (void) hideTextFields
{
  [songTitleField setSelectable: NO];
  [songNumberField setSelectable: NO];
  [songArtistField setSelectable: NO];
  [songAlbumField setSelectable: NO];

  [songTitleField setStringValue: @""];
  [songTimerField setStringValue: @""];
  [songNumberField setStringValue: @""];
  [songArtistField setStringValue: @""];
  [songAlbumField setStringValue: @""];
#ifdef __MACOSX__
  [songTimerBtn setEnabled: NO];
#else
  [songTimerField setTarget: nil];
  [songTimerField setAction: NULL];
#endif
}

- (void) showTextFields
{
  [songTitleField setSelectable: YES];
  [songNumberField setSelectable: YES];
  [songArtistField setSelectable: YES];
  [songAlbumField setSelectable: YES];
#ifdef __MACOSX__
  [songTimerBtn setEnabled: YES];
#else
  [songTimerField setTarget: self];
  [songTimerField setAction: @selector(changeTimeDisplay:)];
#endif
}

- (void) show
{
  [splashImage setImage: [NSImage imageNamed: @"cynthiune-splash-faded"]];
  [self showTextFields];
}

- (void) hide
{
  [splashImage setImage: [NSImage imageNamed: @"cynthiune-splash"]];
  [self hideTextFields];
}

- (void) updateInfoFieldsFromSong: (Song *) aSong
{
  NSString *title;

  title = [aSong title];
  if ([title isEqualToString: @""])
    title = [NSString stringWithFormat: @"[%@]", [aSong filename]];
  [songArtistField setStringValue: [aSong artist]];
  [songAlbumField setStringValue: [aSong album]];
  [songTitleField setStringValue: title];
  [songNumberField setStringValue: [aSong trackNumber]];
  totalTime = [aSong duration];
}

- (void) setTimerFromSeconds: (unsigned int) seconds
{
  int realSeconds;

  realSeconds = ((reverseTimer)
                 ? totalTime - seconds
                 : seconds);
  [songTimerField setStringValue: timeStringFromSeconds (realSeconds)];
}

- (void) setTimerFromTotalSeconds: (unsigned int) seconds
{
  NSFont *myFont;

  myFont = [NSFont boldSystemFontOfSize: 12.0];
  [songTimerField setFont: myFont];
  [songTimerField setStringValue: timeStringFromSeconds (seconds)];
}

- (void) resetTimerField
{
  NSFont *myFont;

  myFont = [NSFont systemFontOfSize: 12.0];
  [songTimerField setFont: myFont];
  [songTimerField setStringValue: @""];
}

- (void) changeTimeDisplay: (id) sender
{
  reverseTimer = !reverseTimer;
}

- (BOOL) timerIsReversed
{
  return reverseTimer;
}

@end
