/* MP3.m - this file is part of Cynthiune
 *
 * Copyright (C) 2002, 2003, 2004 Wolfgang Sourdeau
 *
 * portions are Copyright (C) 2000-2003 Robert Leslie
 *                        (C) 2001-2002 Bertrand Petit
 *
 * Author: Wolfgang Sourdeau <wolfgang@contre.com>
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#import <Foundation/Foundation.h>

#import <sys/types.h>
#import <sys/stat.h>
#import <string.h>
#import <mad.h>
#import <id3tag.h>

#import <CynthiuneBundle.h>
#import <Format.h>

#import "xing.h"
#import "MP3.h"

#if defined (NeXT_PDO) || defined (__APPLE__)
#define _(X) NSLocalizedString (X, nil)
#define NSDebugLog
#endif

@implementation MP3 : NSObject

+ (NSArray *) bundleClasses
{
  return [NSArray arrayWithObject: [self class]];
}

- (id) init
{
  if ((self = [super init]))
    {
      mf = NULL;
      id3File = NULL;
      id3Tag = NULL;

      channels = 0;
      rate = 0;
      duration = 0;
      size = 0;
      iRemain = 0;
      oRemain = 0;

      lastError = MAD_ERROR_NONE;
      opened = NO;
    }

  return self;
}

- (void) calcInputRemain
{
  if (stream.next_frame)
    {
      iRemain = stream.bufend - stream.next_frame;
      memcpy (iBuffer, stream.next_frame, iRemain);
    }
  else
    iRemain = 0;
}

- (void) readStreamMetaData
{
  int iRBytes;

  mad_stream_init (&stream);
  mad_frame_init (&frame);
  mad_synth_init (&synth);

  while (rate == 0)
    {
      [self calcInputRemain];
      iRBytes = fread (iBuffer + iRemain, sizeof (char),
                       IBUFFER_SIZE - iRemain, mf);
      if (iRBytes > 0)
        {
          mad_stream_buffer (&stream, iBuffer, iRBytes + iRemain);

          stream.error = 0;

          if (mad_frame_decode (&frame, &stream))
            {
              if (MAD_RECOVERABLE (stream.error)
                  || stream.error == MAD_ERROR_BUFLEN)
                continue;
              else
                {
                  NSLog (@"%s: unrecoverable frame level error (%s)",
                         __FILE__, mad_stream_errorstr (&stream));
                  break;
                }
            }

          channels = MAD_NCHANNELS(&frame.header);
          rate = frame.header.samplerate;

          if (xing_parse (&xing, stream.anc_ptr, stream.anc_bitlen) == 0)
            {
              duration = (int) (xing.bytes * 8
                                / (frame.header.bitrate * channels));
            }
          else
            {
              mad_timer_multiply (&frame.header.duration,
                                  size / (stream.next_frame
                                          - stream.this_frame));
              duration = mad_timer_count (frame.header.duration,
                                          MAD_UNITS_SECONDS);
            }
        }
      else
        {
          NSLog (@"%s: empty file?", __FILE__);
          break;
        }
    }

  mad_stream_finish (&stream);
  mad_frame_finish (&frame);
  mad_synth_finish (&synth);
  memset (iBuffer, 0, IBUFFER_SIZE);
}

- (void) streamInit
{
  [self readStreamMetaData];

  xing_init (&xing);
  mad_stream_init (&stream);
  mad_frame_init (&frame);
  mad_synth_init (&synth);
}

- (void) id3Init: (NSString *) fileName
{
#if defined(GNUSTEP) || defined (NeXT_PDO)
  id3File = id3_file_open ([fileName cString], ID3_FILE_MODE_READONLY);
#else
  id3File = id3_file_open ([fileName UTF8String], ID3_FILE_MODE_READONLY);
#endif
  if (id3File)
    id3Tag = id3_file_tag (id3File);
}

- (void) _readSize: (const char *) filename;
{
  struct stat stat_val;

  stat (filename, &stat_val);
  size = stat_val.st_size;
}

- (BOOL) streamOpen: (NSString *) fileName
{
  const char *cfilename;

  if (!opened)
    {
#if defined (GNUSTEP) || defined (NeXT_PDO)
      cfilename = [fileName cString];
#else
      cfilename = [fileName UTF8String];
#endif
      mf = fopen (cfilename, "r");
      [self _readSize: cfilename];

      if (mf)
        {
          [self streamInit];
          [self id3Init: fileName];
          opened = YES;
        }
      else
        NSLog (@"%s: no handle...", __FILE__);
    }
  else
    NSLog (@"Attempted to open opened stream");

  return (mf != NULL);
}

+ (int) _seekOffset: (FILE *) _f
{
  char testchar;
  int counter;

  counter = -1;
  testchar = 0;

  while (!testchar)
    {
      fread (&testchar, 1, 1, _f);
      counter++;
    }
  fseek (_f, counter, SEEK_SET);

  return counter;
}

+ (BOOL) _testMP3Header: (char *) buffer
{
  u_int16_t *header;
  BOOL result;

  header = (u_int16_t *) buffer;
#if (BYTE_ORDER == __LITTLE_ENDIAN)
  if ((*header & 0xfeff) == 0xfaff
      || (*header & 0xfeff) == 0xfcff
      || (*header & 0xf8ff) == 0xf0ff
      || (*header & 0xf8ff) == 0xe0ff)
    result = YES;
#else
  if ((*header & 0xfffe) == 0xfffa
      || (*header & 0xfffe) == 0xfffc
      || (*header & 0xfff8) == 0xfff0
      || (*header & 0xfff8) == 0xffe0)
    result = YES;
#endif
  else
    result = NO;

  return result;
}

+ (BOOL) _testRIFFHeader: (char *) buffer
                  inFile: (FILE *) _f
              withOffset: (int) offset
{
  char tag;
  BOOL result;

  if (!strncmp (buffer, "WAVE", 4))
    {
      fseek (_f, 20 + offset, SEEK_SET);
      fread (&tag, 1, 1, _f);
      result = (tag == 80 || tag == 85) ? YES : NO;
    }
  else
    result = NO;

  return result;
}

+ (BOOL) streamTestOpen: (NSString *) fileName
{
  FILE *_f;
  char buffer[4];
  BOOL result;
  int offset;

#if defined (GNUSTEP) || defined (NeXT_PDO)
  _f = fopen ([fileName cString], "r");
#else
  _f = fopen ([fileName UTF8String], "r");
#endif

  if (_f)
    {
      offset = [self _seekOffset: _f];

      fread (buffer, 1, 4, _f);
      if (!strncmp (buffer, "RIFF", 4))
        {
          fseek (_f, 8 + offset, SEEK_SET);
          fread (buffer, 1, 4, _f);
          result = [self _testRIFFHeader: buffer
                         inFile: _f
                         withOffset: offset];
        }
      else
        result = ([self _testMP3Header: buffer]
                  || !strncmp (buffer, "ID3", 3));

      fclose (_f);
    }
  else
    result = NO;

  return result;
}

// - (NSString *) errorText
// {
//   return @"[error unimplemented]";
// }

- (signed long) _audioLinearDither: (mad_fixed_t) sample
                        withDither: (audioDither *) dither
{
  unsigned int scalebits;
  mad_fixed_t output, mask, random;

  /* noise shape */
  sample += dither->error[0] - dither->error[1] + dither->error[2];

  dither->error[2] = dither->error[1];
  dither->error[1] = dither->error[0] / 2;

  /* bias */
  output = sample + (1L << (MAD_F_FRACBITS - 16));

  scalebits = MAD_F_FRACBITS - 15;
  mask = (1L << scalebits) - 1;

  /* dither */
  random = (dither->random * 0x0019660dL + 0x3c6ef35fL) & 0xffffffffL;;
  output += (random & mask) - (dither->random & mask);

  dither->random = random;

  /* clip */
  if (output > (MAD_F_ONE - 1))
    {
      output = MAD_F_ONE - 1;

      if (sample > MAD_F_ONE - 1)
        sample = MAD_F_ONE - 1;
    }
  else if (output < -MAD_F_ONE)
    {
      output = -MAD_F_ONE;

      if (sample < -MAD_F_ONE)
        sample = -MAD_F_ONE;
    }

  /* quantize */
  output &= ~mask;

  /* error feedback */
  dither->error[0] = sample - output;

  /* scale */
  return output >> scalebits;
}

- (void) fillPCMBuffer: (unsigned char*) buffer
             withStart: (int) start
              andLimit: (int) limit
            bufferSize: (int) aSize
{
  int i;
  unsigned char *oBPtr;
  register signed int sample;

  oBPtr = buffer;

  for (i = start; i < limit ; i++)
    {
      sample = [self _audioLinearDither: synth.pcm.samples[0][i]
                     withDither: &leftDither];
      *oBPtr++ = sample >> 0;
      *oBPtr++ = sample >> 8;
      if (channels == 2)
        {
          sample = [self _audioLinearDither: synth.pcm.samples[1][i]
                         withDither: &rightDither];
          *oBPtr++ = sample >> 0;
          *oBPtr++ = sample >> 8;
        }
      if (oBPtr > (buffer+aSize))
        NSLog (@"buffer overrun: %d (max = %d)", (oBPtr - buffer), aSize);
    }
}

- (int) translateBufferToPCM: (unsigned char*) buffer
                    withSize: (int) bufferSize
{
  int start, limit, mult, delta;

  mult = 2 * channels;
  if (oRemain)
    {
      start = synth.pcm.length - oRemain;
      limit = synth.pcm.length;
    }
  else
    {
      start = 0;
      limit = (bufferSize / mult);
      if (synth.pcm.length < limit)
        limit = synth.pcm.length;
    }

  delta = (limit - start) * mult - bufferSize;
  if (delta > 0)
    limit -= delta / mult;

  [self fillPCMBuffer: buffer
        withStart: start
        andLimit: limit
        bufferSize: bufferSize];

  oRemain = synth.pcm.length - limit;

  return ((limit - start) * mult);
}

/* FIXME: we should put some error handling here... */
- (int) readNextChunk: (unsigned char *) buffer
	     withSize: (unsigned int) bufferSize
{
  int iRBytes;

  if (!oRemain)
    {
      do
        {
          stream.error = 0;
          [self calcInputRemain];
          iRBytes = fread (iBuffer + iRemain, sizeof (char),
                           IBUFFER_SIZE - iRemain, mf);
          if (iRBytes <= 0 || feof (mf))
            return 0;

          mad_stream_buffer (&stream, iBuffer, iRBytes + iRemain);

          if (mad_frame_decode (&frame, &stream)
              && !MAD_RECOVERABLE (stream.error)
              && stream.error != MAD_ERROR_BUFLEN)
            {
              NSLog (@"%s: unrecoverable frame level error (%s)",
                     __FILE__, mad_stream_errorstr (&stream));
              return 0;
            }

          lastError = stream.error;

          mad_synth_frame (&synth, &frame);
        }
      while (MAD_RECOVERABLE (stream.error));
    }

  return [self translateBufferToPCM: buffer
               withSize: bufferSize];
}

- (int) lastError
{
  return lastError;
}

- (unsigned int) readChannels
{
  return channels;
}

- (unsigned long) readRate
{
  return rate;
}

- (unsigned int) readDuration
{
  return duration;
}

- (NSString *) readComment: (char*) comString
{
  NSString *comment;
  Id3Frame *id3Frame;
  Id3Field *field;
  const id3_ucs4_t *string;
  char *cString;

  comment = @"";
  id3Frame = id3_tag_findframe (id3Tag, comString, 0);
  if (id3Frame)
    {
      field = id3_frame_field (id3Frame, 1);
      if (field)
        {
          string = id3_field_getstrings (field, 0);
          if (string)
            {
              if (comString == ID3_FRAME_GENRE)
                string = id3_genre_name (string);
              cString = id3_ucs4_latin1duplicate (string);
              comment = [NSString stringWithCString: cString];
              free (cString);
            }
        }
    }

  return comment;
}

- (NSString *) readTitle
{
  return [self readComment: ID3_FRAME_TITLE];
}

- (NSString *) readGenre
{
  return [self readComment: ID3_FRAME_GENRE];
}

- (NSString *) readArtist
{
  return [self readComment: ID3_FRAME_ARTIST];
}

- (NSString *) readAlbum
{
  return [self readComment: ID3_FRAME_ALBUM];
}

- (NSString *) readTrackNumber
{
  return [self readComment: ID3_FRAME_TRACK];
}

- (void) streamClose
{
  if (opened)
    {
      mad_synth_finish (&synth);
      mad_frame_finish (&frame);
      mad_stream_finish (&stream);

      if (id3Tag)
        {
          id3_tag_delete (id3Tag);
          id3Tag = NULL;
        }

      if (id3File)
        {
          id3_file_close (id3File);
          id3File = NULL;
        }

      if (mf)
        {
          fclose (mf);
          mf = NULL;
        }

      opened = NO;
    }
  else
    NSLog (@"Attempted to close closed stream");
}

// Player Protocol
+ (NSArray *) acceptedFileExtensions
{
  return [NSArray arrayWithObjects: @"mp2", @"mp3", @"mpa", @"mpga", @"mpega",
                  @"MP2", @"MP3", @"MPA",@"MPGA", @"MPEGA", nil];
}

/* FIXME: this might not be true */
- (BOOL) isSeekable
{
  return YES;
}

- (void) seek: (unsigned int) aPos
{
  unsigned long filePos;

  if (size)
    {
      filePos = (size * aPos) / duration;
      fseek (mf, filePos, SEEK_SET);
    }
  else
    NSLog (@"size not computed?");
}

@end
