#!/usr/bin/ruby -w
# -*- ruby -*-

require './regexp'
require './log'


# Represents .cvsignore files.

class IgnoredPatterns < Hash
  include Loggable

  def initialize
    @ignorename = ".cvsignore"
    @dirsread = Array.new
  end

  def read(dir)
    # from the CVS default settings -- ignoring overrides

    log dir

    return if @dirsread.include?(dir)
    @dirsread.push(dir)

    log "reading #{dir}"

    pats = %w{
                  CVS
                  *~
                  .cvsignore
                  *.o
                  *$
                  *.BAK
                  *.Z
                  *.a
                  *.bak
                  *.elc
                  *.exe
                  *.ln
                  *.obj
                  *.olb
                  *.old
                  *.orig
                  *.rej
                  *.so
                  .
                  ..
                  .del-*
                  .make.state
                  .nse_depinfo
                  CVS.adm
                  RCS
                  RCSLOG
                  SCCS
                  TAGS
                  _$*
                  core
                  cvslog.*
                  tags
              }
    
    # can't put these guys in the qw() list:
    ['.#*', '#*', ',*'].each { |p| pats.push(p) }

    # read the repository-wide cvsignore file, if it exists and is local.
    #cvsroot = ENV["CVSROOT"]
    #if cvsroot
    #  cri = cvsroot + "/CVSROOT/cvsignore"
    #  repo = read_ignore_file_named(cri)
    #  pats.push(*repo) unless repo.length == 0
    #end

    # read ~/<ignore>
    homedir = ENV["HOME"]       # unix
    unless homedir              # windows
      homedir  = ENV["HOMEDRIVE"]
      homepath = ENV["HOMEPATH"]
      if homepath then
        if homedir then
          homedir += homepath
        else
          homedir = homepath
        end
      end
    end
    
    global = read_ignore_file(homedir)
    pats.push(*global) unless global.length == 0

    # read <ignore> in the current directory
    local = read_ignore_file(dir)
    pats.push(*local) unless local.length == 0

    # prepend the current directory to the patterns, contending with the fact
    # that the directory might actually be a valid regular expression.

    # wildcard if the pattern is a directory
    pats = pats.collect do |p|
      p += "/*" if File.directory?(dir + "/" + p)
      p
    end

    # make a regular expression for each one, to be the entire string (^...$)

    qdir = Regexp.quote(dir)
    self[dir] = pats.collect do |p| 
      p = qdir + "/" + Regexp.unixre_to_string(p)
      re = Regexp.new("^" + p + "$")
      # log "storing ignored pattern re " + re.source + ", dir " + dir
      re
    end
  end

    # Reads the ignore file from the given directory, using the default ignore
    # name.

  def read_ignore_file(dir)
    pats = Array.new

    if dir
      cifile = dir + "/" + @ignorename
      if File.exists?(cifile)
        log "reading #{cifile}"
        IO.foreach(cifile) do |line|
          line.chomp!
          line.gsub!(/\+/, '\\+')
          pats.push(*line.split) if line.split.size > 0
        end
      else
        log "no ignore file in " + dir
      end
    end

    log "patterns: #{pats}"

    pats
  end

    # Reads the given ignore file, if it exists.

  def read_ignore_file_named(fname)
    pats = Array.new

    if File.exists?(fname)
      IO.foreach(fname) do |line|
        line.chomp!
        line.gsub!(/\+/, '\\+')
        pats.push(line)
      end
    else
      log "no such file: " + fname
    end

    pats
  end

  # Returns if the file is ignored. Checks the name as both "./name" and "name".

  def is_ignored?(name)
    # log "name = #{name}"
    if name.index("./") == 0
      withpref, nopref = name, name.sub!("./", "")
    else
      withpref, nopref = "./" + name, name
    end
    
    [ withpref, nopref ].each do |name|
      dir = name
      # log "dirs = " + keys.join(", ")
      while dir = File.dirname(dir)
        if include?(dir)
          regexps = self[dir]
          regexps.each do |re|
            # log "matching " + name + " against " + re.source
            # stop as soon as we find out it is ignored
            return true if re.match(name)
          end
        else
          # log "dir " + dir + " is not included"
        end
        break if dir == "."     # else we'll cycle continuously
      end
    end
    
    return false              # it's not ignored
  end

end
