#include <assert.h>
#include <errno.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <poll.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <unistd.h>

#include "acceptor.h"
#include "log.h"
#include "serverpool.h"
#include "state.pb.h"
#include "util.h"

using namespace std;

extern ServerPool *servers;

int create_server_socket(const sockaddr_in6 &addr, SocketType socket_type)
{
	int server_sock;
	if (socket_type == TCP_SOCKET) {
		server_sock = socket(PF_INET6, SOCK_STREAM, IPPROTO_TCP);
	} else {
		assert(socket_type == UDP_SOCKET);
		server_sock = socket(PF_INET6, SOCK_DGRAM, IPPROTO_UDP);
	}
	if (server_sock == -1) {
		log_perror("socket");
		exit(1);
	}

	int one = 1;
	if (setsockopt(server_sock, SOL_SOCKET, SO_REUSEADDR, &one, sizeof(one)) == -1) {
		log_perror("setsockopt(SO_REUSEADDR)");
		exit(1);
	}

	// We want dual-stack sockets. (Sorry, OpenBSD and Windows XP...)
	int zero = 0;
	if (setsockopt(server_sock, IPPROTO_IPV6, IPV6_V6ONLY, &zero, sizeof(zero)) == -1) {
		log_perror("setsockopt(IPV6_V6ONLY)");
		exit(1);
	}

	// Set as non-blocking, so the acceptor thread can notice that we want to shut it down.
	if (ioctl(server_sock, FIONBIO, &one) == -1) {
		log_perror("ioctl(FIONBIO)");
		exit(1);
	}

	if (bind(server_sock, reinterpret_cast<const sockaddr *>(&addr), sizeof(addr)) == -1) {
		log_perror("bind");
		exit(1);
	}

	if (socket_type == TCP_SOCKET) {
		if (listen(server_sock, 128) == -1) {
			log_perror("listen");
			exit(1);
		}
	}

	return server_sock;
}

sockaddr_in6 create_any_address(int port)
{
	sockaddr_in6 sin6;
	memset(&sin6, 0, sizeof(sin6));
	sin6.sin6_family = AF_INET6;
	sin6.sin6_port = htons(port);
	return sin6;
}

sockaddr_in6 extract_address_from_acceptor_proto(const AcceptorProto &proto)
{
	sockaddr_in6 sin6;
	memset(&sin6, 0, sizeof(sin6));
	sin6.sin6_family = AF_INET6;

	if (!proto.addr().empty()) {
		int ret = inet_pton(AF_INET6, proto.addr().c_str(), &sin6.sin6_addr);
		assert(ret == 1);
	}

	sin6.sin6_port = htons(proto.port());
	return sin6;
}
	
Acceptor::Acceptor(int server_sock, const sockaddr_in6 &addr)
	: server_sock(server_sock),
	  addr(addr)
{
}

Acceptor::Acceptor(const AcceptorProto &serialized)
	: server_sock(serialized.server_sock()),
	  addr(extract_address_from_acceptor_proto(serialized))
{
}

AcceptorProto Acceptor::serialize() const
{
	char buf[INET6_ADDRSTRLEN];
	inet_ntop(addr.sin6_family, &addr.sin6_addr, buf, sizeof(buf));

	AcceptorProto serialized;
	serialized.set_server_sock(server_sock);
	serialized.set_addr(buf);
	serialized.set_port(ntohs(addr.sin6_port));
	return serialized;
}

void Acceptor::close_socket()
{
	safe_close(server_sock);
}

void Acceptor::do_work()
{
	while (!should_stop()) {
		if (!wait_for_activity(server_sock, POLLIN, NULL)) {
			continue;
		}

		sockaddr_in6 addr;
		socklen_t addrlen = sizeof(addr);

		// Get a new socket.
		int sock = accept(server_sock, reinterpret_cast<sockaddr *>(&addr), &addrlen);
		if (sock == -1 && errno == EINTR) {
			continue;
		}
		if (sock == -1) {
			log_perror("accept");
			usleep(100000);
			continue;
		}

		// Set the socket as nonblocking.
		int one = 1;
		if (ioctl(sock, FIONBIO, &one) == -1) {
			log_perror("ioctl(FIONBIO)");
			exit(1);
		}

		// Enable TCP_CORK for maximum throughput. In the rare case that the
		// stream stops entirely, this will cause a small delay (~200 ms)
		// before the last part is sent out, but that should be fine.
		if (setsockopt(sock, SOL_TCP, TCP_CORK, &one, sizeof(one)) == -1) {
			log_perror("setsockopt(TCP_CORK)");
			// Can still continue.
		}

		// Pick a server, round-robin, and hand over the socket to it.
		servers->add_client(sock);
	}
}
